/*  
 * LX200 Telescope Control library
 * Author:  Ken Shouse (ken@kshouse.engine.swri.edu).
 * Modified by: Carlos Guirao (cguirao@eso.org)
 * Modified by: Carlos Guirao 10.4.2000, using select() 
 * Modified by: Carlos Guirao 25.5.2000, adding SetFormat(),SetPrecision()
 */

#include <stdio.h>
#include <errno.h>
#include <time.h> 
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <termios.h>
#include <math.h>

#include "lx200.h"

static int	lx200PortFD;
static int	lx200ConnectFlag = 0;
static int	lx200LongFormat;

#define NULL_PTR(x) (x *)0
#ifndef FD_SET                          /* If it is not the Berkeley select */
        typedef int lx200fds;
#       define FD_SET(n, p)     (*(p) |= (1 << (n)) )
#else
        typedef fd_set lx200fds;
#endif

static int readn(int fd, char *ptr, int nbytes, int sec);
static int writen(int fd, char *ptr, int nbytes);
static int lx200stat(int fd,int sec,int usec);
static void checkFormat();

/*
 * Change test_main() to main() so you can use it in standalone mode
 * Compile: make CCOPTIONS=-g lx200.o
 * Link:    gcc lx200.o -o lx200
 * Debug:   gdb ./lx200
 */
test_main()
{
	int	i;
	float	startRA,startDec;
	float	endRA, endDec;

	InitiatePortLX200("/dev/ttyS0");

	printf("Start...\n");
	fflush(stdout);

	SetFormat(1);
	SetRate(SLEW);

	startRA = GetRA();
	startDec = GetDec();
	printf("Starting RA = %f\tDec = %f\n",startRA,startDec);

	i = SlewToCoords(startRA+1.0,startDec+24.0); 
	while( !CheckCoords(startRA+1.0,startDec+24.0) );
	MySleep(2.0);
	printf("SlewToCoords returned %d\n",i);
	endRA = GetRA();
	endDec = GetDec();
	printf("Midpoint RA = %f\tDec = %f\n",endRA,endDec);

	i = SlewToCoords(startRA,startDec); 
	while( !CheckCoords(startRA,startDec) );
	MySleep(10.0);
	printf("SlewToCoords returned %d\n",i);
	startRA = GetRA();
	startDec = GetDec();
	printf("Ending RA = %f\tDec = %f\n",startRA,startDec);

	printf("end\n");
	fflush(stdout);

	DisconnectLX200();
}

void MySleep(float numSecs)
{
	clock_t	startTime;

	startTime = clock();
	while( (clock() - startTime) <
		(int) (numSecs*((float) CLOCKS_PER_SEC)) );
}

int CheckConnectLX200(void)
{
   return lx200ConnectFlag;
}

/* Set precision ON/OFF with the command #:P# 
 * precision is "LOW " or "HIGH"
 */
void SetPrecision (char * precision)
{
	char	returnStr[128];
	int numRead;

	for (;;) {
       		writen(lx200PortFD,"#:P#",4);
        	numRead = read (lx200PortFD,returnStr,14);
		returnStr[numRead] = '\0';
		/* printf("Precision: %d read %s\n",numRead,returnStr);  */
		if (!strncmp(returnStr,precision,4)) break; 
	}
}

/* Check format 
 * When the long format is active, requests are of the form 
 * HH:MM:SS or sDD*MM:SS
 * otherwise they are HH:MM.T or sDD*MM
 * returns 0: sort format, 1 long format, 2 error
 */
static void checkFormat()
{
	char	returnStr[128];
	int numRead;

	/* 
	 * If format long, the command #:GR# 
	 * returns 9 chars, otherwise 8 
	 */
	writen(lx200PortFD,"#:GR#",5);
	numRead = readn(lx200PortFD,returnStr,9,1);
	returnStr[numRead] = '\0';
	/* printf("GetRA: %d read %s\n",numRead,returnStr); */
	switch (numRead) {
	case 8: lx200LongFormat=0; break;
	case 9: lx200LongFormat=1; break;
	default: fprintf(stderr,"Error determining format!\n");
		 DisconnectLX200();
	}
}

/* Set long format ON/OFF with the command #:U# 
 * When the long format is active, requests are of the form 
 * HH:MM:SS or sDD*MM:SS
 * otherwise they are HH:MM.T or sDD*MM
 * format=1 ON,format=O OFF
 */
void SetFormat (int format)
{
	char	returnStr[128];
	int numRead, expected;

	/* 
	 * If requested format already return
	 * otherwise toggle format
	 */
	/* printf("SetFormat: %d\n",format); */
	if ( (format == 1 && lx200LongFormat == 1) || 
	     (format == 0 && lx200LongFormat == 0) ) return;
            
	/* otherwise toggle format */
	writen(lx200PortFD,"#:U#",4);
	checkFormat();
}

int ConnectLX200(void)
{
	char	returnStr[128];
        int numRead;

	if(lx200ConnectFlag == 0) return 0; 

	writen(lx200PortFD,"#:GL#",5);
	numRead=readn(lx200PortFD,returnStr,9,1);
	returnStr[numRead] = '\0';
	/* printf("LocatTime: %d read %s\n",numRead,returnStr); */
	if (numRead == 9) {
		checkFormat();
		/* SetPrecision("LOW "); */
		return 1;

	}
	else { /* Disconnect and return error */
		/* simulates an IO error if there is no answer from LX200 */
		DisconnectLX200();
		errno = EIO;	
		return 0;
	}
}
	
void InitiatePortLX200(char *port)
{
	struct termios		tty;
	char	tempChar;


	/* printf("Connect port=%s\n",port); */

	if(lx200ConnectFlag != 0)
		return;

	/* first, do the connect */
	lx200PortFD = open(port,O_RDWR);
	if(lx200PortFD == -1)
		return;

	tcgetattr(lx200PortFD,&tty);
	cfsetospeed(&tty, (speed_t) B9600);
	cfsetispeed(&tty, (speed_t) B9600);
	tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8;
	tty.c_iflag =  IGNBRK;
	tty.c_lflag = 0;
	tty.c_oflag = 0;
	tty.c_cflag |= CLOCAL | CREAD;
	tty.c_cc[VMIN] = 1;
	tty.c_cc[VTIME] = 5;
	tty.c_iflag &= ~(IXON|IXOFF|IXANY);
	tty.c_cflag &= ~(PARENB | PARODD);
	tcsetattr(lx200PortFD, TCSANOW, &tty);

	/* now, flush the input (read) buffer */
	tcflush(lx200PortFD,TCIOFLUSH);

	/* and finally, send a # to terminate any bullshit */
	writen(lx200PortFD,"#",1);
 
	lx200ConnectFlag = 1;
}

void SetRate(int newRate)
{
	if(newRate == SLEW) 
		writen(lx200PortFD,"#:RS#",5);
	else if(newRate == FIND) 
		writen(lx200PortFD,"#:RM#",5);
	else if(newRate == CENTER) 
		writen(lx200PortFD,"#:RC#",5);
	else if(newRate == GUIDE) 
		writen(lx200PortFD,"#:RG#",5);
}

void StartSlew(int direction)
{
	if(direction == NORTH)
		writen(lx200PortFD,"#:Mn#",5);
	else if(direction == EAST)
		writen(lx200PortFD,"#:Me#",5);
	else if(direction == SOUTH)
		writen(lx200PortFD,"#:Ms#",5);
	else if(direction == WEST)
		writen(lx200PortFD,"#:Mw#",5);
}

void StopSlew(int direction)
{
	if(direction == NORTH)
		writen(lx200PortFD,"#:Qn#",5);
	else if(direction == EAST)
		writen(lx200PortFD,"#:Qe#",5);
	else if(direction == SOUTH)
		writen(lx200PortFD,"#:Qs#",5);
	else if(direction == WEST)
		writen(lx200PortFD,"#:Qw#",5);
}

void DisconnectLX200(void)
{
	/* printf("DisconnectLX200\n"); */
	if(lx200ConnectFlag == 1)
		close(lx200PortFD);
	lx200ConnectFlag = 0;
}

void GetRAlx200(char *returnStr)
{
	int	numRead;

	writen(lx200PortFD,"#:GR#",5);

	/* LongFormat read 9, othewise read 8 */
        if (lx200LongFormat) {
		numRead = readn(lx200PortFD,returnStr,9,1);
		if (numRead != 9) {
			DisconnectLX200();
			return;
			}
		returnStr[8] = '\0';
       	}
	else {
		numRead = readn(lx200PortFD,returnStr,8,1);
		if (numRead != 8) {
			DisconnectLX200();
			return;
			}
		returnStr[7] = '\0';
       	}

	/* printf("GetRAlx200: %d read %s\n",numRead,returnStr); */
}

float GetRA(void)
{
	char	returnStr[10];
	float	raHrs,raMin,raSec;
	float	returnVal;
	int	numRead;

	writen(lx200PortFD,"#:GR#",5);

	numRead = readn(lx200PortFD,returnStr,9,1);
	if (numRead != 9) { 
		DisconnectLX200(); 
		return 0;
		}
	returnStr[8] = '\0';
	printf("GetRA: %d read %s\n",numRead,returnStr); 

	sscanf(returnStr,"%f:%f:%f#",&raHrs,&raMin,&raSec);

	returnVal = raHrs + raMin/60.0 + raSec/600.0;
	return returnVal;
}

void GetDeclx200(char *returnStr)
{
	float	returnVal;
	int	numRead;

	writen(lx200PortFD,"#:GD#",5);

	/* LongFormat read 10, othewise read 7 */
        if (lx200LongFormat) {
		numRead = readn(lx200PortFD,returnStr,10,1);
		if (numRead != 10) {
			DisconnectLX200();
			return;
			}
		returnStr[9] = '\0';
       	}
	else {
		numRead = readn(lx200PortFD,returnStr,7,1);
		if (numRead != 7) {
			DisconnectLX200();
			return;
			}
		returnStr[6] = '\0';
       	}

	returnStr[3] = ':';
	/* printf("GetDeclx200:%d read %s\n",numRead,returnStr);  */
}

float GetDec(void)
{
	char	returnStr[10];
	float	decDeg,decMin,decSec;
	float	returnVal;
	int	numRead;

	writen(lx200PortFD,"#:GD#",5);

	numRead = readn(lx200PortFD,returnStr,10,1);
	if (numRead != 10) { 
		DisconnectLX200(); 
		return 0;
		}
	returnStr[9] = '\0';
	printf("GetDec:%d read %s\n",numRead,returnStr);
	returnStr[3] = returnStr[6] = '\0';

	sscanf(returnStr,"%f",&decDeg);
	sscanf(returnStr+4,"%f#",&decMin);
	sscanf(returnStr+7,"%f#",&decSec);

	if(decDeg >= 0.0)
		returnVal = decDeg + decMin/60.0 + decSec/600.0;
	else
		returnVal = decDeg - decMin/60.0 - decSec/600.0;

	return returnVal;
}

void SyncObjectCoords()
{
	char	inputStr[2];
	writen(lx200PortFD,"#:CM#",5);

	/* Keep reading until getting a '#' */
        for (;;) {
		if ( readn(lx200PortFD,inputStr,1,1) ) {
			if (inputStr[0] == '#') break;
		}
		else { DisconnectLX200(); return; }
	}
}

int Slew2Coords(char *newRA, char *newDec)
{
	char	outputStr[32], inputStr[2048];
	int	returnVal;

	sprintf(outputStr,"#:Sr %s",newRA);
	if (lx200LongFormat) {
	  outputStr[13] = '#';
 	  outputStr[14] = '\0';
	  writen(lx200PortFD,outputStr,14);
          }
	else {
	  outputStr[12] = '#';
 	  outputStr[13] = '\0';
	  writen(lx200PortFD,outputStr,13);
        }
          
	/* printf("Slew2Coords: sendRA=%s\n",outputStr); */

	if (readn(lx200PortFD,inputStr,1,1) != 1 ) {
		DisconnectLX200(); 
		return -1;
	}

	if(inputStr[0] != '1') {
		fprintf(stderr,"Error setting object RA in Slew2Coords!\n");
		fprintf(stderr,"Tried to send command string %s\n",outputStr);
	}

	sprintf(outputStr,"#:Sd %s",newDec);
	if (outputStr[5] == ' ' && outputStr[6] == ' ') {
		outputStr[5] = '+';
		outputStr[6] = '0';
	}
	else if (outputStr[5] == ' ' && outputStr[6] == '-') {
		outputStr[5] = '-';
		outputStr[6] = '0';
	}
	else if (outputStr[5] == ' ' && outputStr[6] != '-') {
		outputStr[5] = '+';
	}
	outputStr[8] = 0xdf;

	if (lx200LongFormat) {
	  outputStr[14] = '#';
	  outputStr[15] = '\0'; 
	  writen(lx200PortFD,outputStr,15);
 	  }
	else {
	  outputStr[11] = '#';
	  outputStr[12] = '\0'; 
	  writen(lx200PortFD,outputStr,12);
 	  }
	/* printf("Slew2Coords: sendDec=%s\n",outputStr); */

        if (readn(lx200PortFD,inputStr,1,1) != 1 ) {
                DisconnectLX200();
                return -1;
        }

	if(inputStr[0] != '1') {
		fprintf(stderr,"Error setting object Dec in Slew2Coords!\n");
		fprintf(stderr,"Tried to send command string %s\n",outputStr);
	}

	writen(lx200PortFD,"#:MS#",5);
        if (readn(lx200PortFD,inputStr,1,5) != 1 ) {
                DisconnectLX200();
                return -1;
        }

	switch(inputStr[0]) {
	case '0':
		return 0;
		break;
	case '1':
		returnVal = 1;
		break;
	case '2':
		returnVal = 2;
		break;
	default:
		fprintf(stderr,"Unspected return from command #:MS#\n");
                return -1;
		break;
	}

	/* Keep reading until getting a '#' */
        for (;;) {
		if ( readn(lx200PortFD,inputStr,1,1) ) {
			if (inputStr[0] == '#') break;
		}
		else return -1;
	}
	return returnVal;
}

int SlewToCoords(float newRA, float newDec)
{
	int	raHrs,decDeg,decMin;
	float	raMin;
	char	outputStr[32], inputStr[2048];
	int	returnVal;

	raHrs = (int) newRA;
	raMin = (newRA - (float) raHrs)*60.0;
	sprintf(outputStr,"#:Sr %02d:%04.1f#\n",raHrs,raMin);
        printf("SlewToCoords: sendRA=%s\n",outputStr);
	writen(lx200PortFD,outputStr,13);
        if (readn(lx200PortFD,inputStr,1,1) != 1 ) exit(0); 
	if(inputStr[0] != '1')
	{
		fprintf(stderr,"Error setting object RA in SlewToCoords!\n");
		fprintf(stderr,"Tried to send command string %s\n",outputStr);
		fprintf(stderr,"Exiting...\n");
		exit(0);  
	}

	if(newDec >= 0.0)
	{
		decDeg = (int) newDec;
		decMin = (int) ((newDec - (float) decDeg)*60.0);
	}
	else
	{
		decDeg = (int) (-newDec);
		decMin = (int) (((-newDec) - (float) decDeg)*60.0);
		decDeg *= -1;
	}
	sprintf(outputStr,"#:Sd +%02d%c%02d#\n",decDeg,0xdf,decMin);
        printf("SlewToCoords: sendRA=%s\n",outputStr);
	writen(lx200PortFD,outputStr,12);
        if (readn(lx200PortFD,inputStr,1,1) != 1 ) exit(0);
	if(inputStr[0] != '1')
	{
		fprintf(stderr,"Error setting object Dec in SlewToCoords!\n");
		fprintf(stderr,"Tried to send command string %s\n",outputStr);
		fprintf(stderr,"Exiting...\n");
		exit(0); 
	}

        writen(lx200PortFD,"#:MS#",5);
        if (readn(lx200PortFD,inputStr,1,1) != 1 ) {
                DisconnectLX200();
		exit(0); 
        }

        switch(inputStr[0]) {
        case '0':
                return 0;
                break;
        case '1':
                returnVal = 1;
                break;
        case '2':
                returnVal = 2;
                break;
        default:
                fprintf(stderr,"Unspected return from command #:MS#\n");
                return -1;
                break;
        }

        /* Keep reading until getting a '#' */
        for (;;) {
                if ( readn(lx200PortFD,inputStr,1,1) ) {
                        if (inputStr[0] == '#') break;
		}
                else return -1;
        }
        return returnVal;
}


int	CheckCoords(float desRA, float desDec)
{
	float	raError,decError;

	raError = GetRA() - desRA;
	decError = GetDec() - desDec;

	/* Skysensor do not slew with secons precision */
	/*if( fabs(raError) > 1.0/(15.0*30.0) || fabs(decError) > 1.0/30.0)*/
	if( fabs(raError) > 0.1 || fabs(decError) > 0.1)
		return 0;
	else
		return 1;
}

static int writen(fd, ptr, nbytes)
int fd;
char *ptr;
int nbytes;
{
	int nleft, nwritten;
	nleft = nbytes;
	while (nleft > 0) {
		nwritten = write (fd, ptr, nleft);
		if (nwritten <=0 ) break;
		nleft -= nwritten;
		ptr += nwritten;
		}
	return (nbytes - nleft);
}

static int readn(fd, ptr, nbytes, sec)
int fd;
char *ptr;
int nbytes;
int sec;
{
	int stat;
	int nleft, nread;
	nleft = nbytes;
	while (nleft > 0) {
		stat = lx200stat(fd,sec,0);
	 	if (stat <=  0 ) break;
		nread  = read (fd, ptr, nleft);
		/* printf("readn: %d read\n", nread); */
		if (nread <= 0)  break;
		nleft -= nread;
		ptr += nread;
		}
	return (nbytes - nleft);
}

/*
 * Examines the read status of a file descriptor.
 * The timeout (sec, usec) specifies a maximum interval to
 * wait for data to be available in the descriptor.
 * To effect a poll, the timeout (sec, usec) should be 0.
 * Returns non-negative value on data available.
 * 0 indicates that the time limit referred by timeout expired.
 * On failure, it returns -1 and errno is set to indicate the
 * error.
 */
static int lx200stat(fd,sec,usec)
register int fd, sec, usec;
{
        int ret;
        int width;
        struct timeval timeout;
        lx200fds readfds;

        memset((char *)&readfds,0,sizeof(readfds));
        FD_SET(fd, &readfds);
        width = fd+1;
        timeout.tv_sec = sec;
        timeout.tv_usec = usec;
        ret = select(width,&readfds,NULL_PTR(lx200fds),NULL_PTR(lx200fds),&timeout);
        return(ret);
}
