/*  
 * Handles the history and log file of lx200 telescope control 
 * Author: Carlos Guirao (cguirao@eso.org)
 */

#include <stdio.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/Label.h>
#include <Xm/CascadeB.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "P_.h"
#include "astro.h"
#include "circum.h"
#include "preferences.h"
#include "lx200hist.h"

extern Widget toplevel_w;
extern Colormap xe_cm;
extern FILE *fopenh P_((char *name, char *how));

extern Obj objx;

extern Widget fw P_((int fid)); 
extern void get_something P_((Widget w, char *resource, XtArgVal value));
extern void get_xmstring P_((Widget w, char *resource, char **txtp));
extern void query P_((Widget tw, char *msg, char *label1, char *label2,
    char *label3, void (*func1)(), void (*func2)(), void (*func3)()));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txtp));
extern void xe_msg P_((char *msg, int app_modal));
extern void watch_cursor P_((int want));
extern void wtip P_((Widget w, char *tip));

static void plt_manage P_((void));
static void plt_create_form P_((void));
static void plt_turn_off P_((void));
static void plt_turn_on P_((char *how));
static void plt_turn_off P_((void));
static void plt_try_append P_((void));
static void plt_try_overwrite P_((void));
static void plt_try_cancel P_((void));
static void plt_try_turn_on P_((void));
static void plt_ok_cb P_((Widget w, XtPointer client, XtPointer call));
static void plt_cancel_cb P_((Widget w, XtPointer client, XtPointer call));
static void plt_help_cb P_((Widget w, XtPointer client, XtPointer call));


/* N.B. if over ten, will need a different mnemonic system */
#define	MAXHIST	20	/* max history items */

/* a collection of PB widgets and the histories they represent.
 * the first nhist entries of hist[] are in use.
 */
typedef struct {
    Widget w;
    Lx200History h;
} Hist;
static Hist hist[MAXHIST];
static int nhist;

static Widget delone_w;		/* PB set to enable deleting one item */
static Widget delall_w;		/* PB set to delete all items */
static int delone_on;		/* set while delete-one mode is on */
static Widget plotform_w;
static Widget filename_w;
#define DEF_PLTFN       "./ephem.log"     /* default plot file name */
static FILE *plt_fp;            /* the plot file; == 0 means don't plot */


void lxh_add_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_log_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_log_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_delall_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_delone_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_pick_cb P_((Widget w, XtPointer client, XtPointer call));
static void delone_setup P_((int enable));
static void delete_one P_((int n));
static void delete_all_confirmed P_((void));
static void add_one P_((void));
static void setup_pb P_((Hist *hp));

/* field ids
 * N.B. must be in same order as they appear in mm_field_map[].
 */
enum {
    JD_FID, UD_FID, UT_FID, LST_FID,TZN_FID, TZONE_FID, LD_FID, LT_FID, DT_FID,
    DIP_FID, DAWN_FID, DUSK_FID, LON_FID, LSTM_FID, GAP, STPSZ_FID, NSTEP_FID,
    PAUSE_FID, SITE_FID, LAT_FID, LONG_FID, ELEV_FID, TEMP_FID, PRES_FID,
    EPOCH_FID
};

/* called to create the history cascade and pulldown off the given menubar.
 * also gets the current settings and adds it as the first item in the list.
 */
void
lxh_create (mb_w)
Widget mb_w;
{
	Widget cb_w, pd_w;
	Widget w;
	Arg args[20];
	int i;
	int n;

	n = 0;
	pd_w = XmCreatePulldownMenu (mb_w, "HPD", args, n);

	n = 0;
	XtSetArg (args[n], XmNsubMenuId, pd_w); n++;
	XtSetArg (args[n], XmNmnemonic, 'H'); n++; 
	cb_w = XmCreateCascadeButton (mb_w, "History", args, n);
	wtip (cb_w, "Manage a collection of target logs");
	XtManageChild (cb_w);

	/* create the Add PB */
	n = 0;
	w = XmCreatePushButton (pd_w, "Add", args, n);
	XtAddCallback (w, XmNactivateCallback, lxh_add_cb, NULL);
	wtip (w, "Add all the current target specifics to the history collection");
	XtManageChild (w);

	/* create the Log PB */
	n = 0;
	w = XmCreatePushButton (pd_w, "Log File", args, n);
	XtAddCallback (w, XmNactivateCallback, lxh_log_cb, NULL);
	wtip (w, "Log to a file the history collection");
	XtManageChild (w);

	n = 0;
	w = XmCreateSeparator (pd_w, "LSep1", args, n);
	XtManageChild (w);

	/* create the Delete All PB -- make insensitive until an item is added*/
	n = 0;
	delall_w = XmCreatePushButton (pd_w, "DelAll", args, n);
	XtAddCallback (delall_w, XmNactivateCallback, lxh_delall_cb, NULL);
	set_xmstring (delall_w, XmNlabelString, "Delete All");
	XtSetSensitive (delall_w, False);
	wtip (delall_w, "Delete the entire history collection");
	XtManageChild (delall_w);

	/* create the Delete One PB -- make insensitive until an item is added*/
	n = 0;
	XtSetArg (args[n], XmNindicatorOn, False); n++;
	delone_w = XmCreatePushButton (pd_w, "DelOne", args, n);
	XtAddCallback (delone_w, XmNactivateCallback, lxh_delone_cb, NULL);
	set_xmstring (delone_w, XmNlabelString, "Delete One");
	XtSetSensitive (delone_w, False);
	wtip (delone_w, "Delete any one history collection entry: Press once, then select item");
	XtManageChild (delone_w);

	n = 0;
	w = XmCreateSeparator (pd_w, "LSep2", args, n);
	XtManageChild (w);

        n = 0;
        w = XmCreateLabel (pd_w, "Name             UTC-Date   UTC-Time      RA         Dec      RA(offset)  Dec(offset)  Mag  SP   Exp  Airmass", args, n);
	XtManageChild (w);

	/* create the activation PBs. not managed until defined */

	for (i = 0; i < MAXHIST; i++) {
	    n = 0;
	    w = XmCreatePushButton (pd_w, "H", args, n);
	    XtAddCallback (w, XmNactivateCallback, lxh_pick_cb, (XtPointer)i);
	    hist[i].w = w;
	    wtip (w, "Reinstate this entry if normal color, delete if reverse color");
	}

	/* init the list with the current settings */
	/* add_one(); */
}

/* called when the Add button is hit in the history pulldown */
/* ARGSUSED */
/*
static void
*/
void
lxh_add_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* turn off delete-1 */
	delone_setup (0);

	/* check for room */
	if (nhist == MAXHIST) {
	    char msg[128];

	    (void) sprintf (msg, "No room for additional history -- max is %d",
								    MAXHIST);
	    xe_msg (msg, 1);
	    return;
	}

	/* do it */
	add_one();

}

/* called when the Delete all button is hit in the history pulldown */
/* ARGSUSED */
static void
lxh_delall_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* turn off delete-1 */
	delone_setup (0);

	/* confirm */
	if (nhist > 0)
	    query (toplevel_w, "Delete all Sky view history entries?",
				"Yes -- delete all", "No -- delete none", NULL,
					    delete_all_confirmed, NULL, NULL);
	else
	    xe_msg ("No history entries to delete", 1);
}

/* called when the Delete one PB is activated */
/* ARGSUSED */
static void
lxh_delone_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* complain if turning it on and nothing to delete */
	if (!delone_on && nhist == 0) {
	    xe_msg ("No history entries to delete", 1);
	    return;
	}

	delone_setup (!delone_on);
}

/* called when a history entry is activated.
 * client is an index into hist[].
 */
/* ARGSUSED */
static void
lxh_pick_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int n = (int)client;

	if (n < 0 || n >= nhist) {
	    printf ("Bad history number: %d\n", n);
	    return;
	}


	if (delone_on) {
	    delete_one (n);
	    delone_setup (0);
	} else {
	    watch_cursor (1);
	    lxh_goto (&hist[n].h);
	    watch_cursor (0);
	}
}

/* enable or disable deleting one item */
static void
delone_setup (enable)
int enable;
{
	static unsigned long fg, bg;
	Arg args[2];
	int i;

	/* get initial colors once */
	if (fg == bg) {
	    XtSetArg (args[0], XmNforeground, &fg);
	    XtSetArg (args[1], XmNbackground, &bg);
	    XtGetValues (delone_w, args, 2);
	}

	/* toggle colors depending on current mode, if new state */
	if (enable == delone_on)
	    return;
	delone_on = enable;
	if (enable) {
	    XtSetArg (args[0], XmNforeground, bg);
	    XtSetArg (args[1], XmNbackground, fg);
	} else {
	    XtSetArg (args[0], XmNforeground, fg);
	    XtSetArg (args[1], XmNbackground, bg);
	}
	for (i = 0; i < MAXHIST; i++)
	    XtSetValues (hist[i].w, args, 2);
	XtSetValues (delone_w, args, 2);
}

/* delete item n */
static void
delete_one (n)
int n;
{
	char *title;
	int i;

	/* copy all down one */
	for (i = n; i < nhist-1; i++) {
	    get_xmstring (hist[i+1].w, XmNlabelString, &title);
	    set_xmstring (hist[i].w,   XmNlabelString,  title);
	    XtFree (title);
	    hist[i].h = hist[i+1].h;
	}

	/* unuse the last */
	XtUnmanageChild (hist[--nhist].w);

	/* if no more entries, disable the delete buttons */
	if (nhist == 0) {
	    XtSetSensitive (delone_w, False);
	    XtSetSensitive (delall_w, False);
	}
}

/* delete all history entries */
static void
delete_all_confirmed()
{
	int i;

	for (i = 0; i < nhist; i++)
	    XtUnmanageChild (hist[i].w);
	
	nhist = 0;

	/* now that there are no entries, disable the delete buttons */
	XtSetSensitive (delone_w, False);
	XtSetSensitive (delall_w, False);
}

/* get current settings and assign to next position */
static void
add_one()
{
	lxh_get (&hist[nhist].h); 
	setup_pb (&hist[nhist]);
	nhist++;

	/* now that there is at least one entry, enable the delete buttons */
	XtSetSensitive (delone_w, True);
	XtSetSensitive (delall_w, True);
}

/* setup the widget for the given hist entry */
static void
setup_pb (hp)
Hist *hp;
{
	char buf[128];
        char out_ra[64];
        char out_dec[64];
        char out_offra[64];
        char out_offdec[64];
	char *text;

	Lx200History *shp = &hp->h;

	shp->lx_exposure = lxh_get_exp ();
	shp->lx_airmass = lxh_get_airmass ();

        get_xmstring (fw(UD_FID), XmNlabelString, &text);
	strcpy (shp->lx_utc_date, text);
        get_xmstring (fw(UT_FID), XmNlabelString, &text);
	strcpy (shp->lx_utc_time, text);

        fs_ra(out_ra,shp->lx_RA);
        fs_prdec(out_dec,shp->lx_dec);
        fs_ra(out_offra,shp->lx_offRA);
        fs_prdec(out_offdec,shp->lx_offdec);

	if (shp->lx_class) 
	  (void) sprintf (buf,"%-14s %10s %10s %s %s %s %s  %2.2f  %.*s %5.0f   %2.2f",
		shp->lx_name,
		shp->lx_utc_date,
		shp->lx_utc_time,
		out_ra,out_dec,
		out_offra,out_offdec,
		shp->lx_mag/MAGSCALE, 
		(int)sizeof(shp->lx_spect),shp->lx_spect, 
		shp->lx_exposure,
		shp->lx_airmass
		);
	else
	  (void) sprintf (buf,"%-14s %10s %10s %s %s %s %s    --  --    --   %2.2f",
		shp->lx_name,
		shp->lx_utc_date,
		shp->lx_utc_time,
		out_ra,out_dec,
		out_offra,out_offdec,
		shp->lx_airmass
		);

	set_xmstring (hp->w, XmNlabelString, buf);
	XtManageChild (hp->w);
}

/* ARGSUSED */
static void
lxh_log_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	plt_manage ();
}

/* called when the plot menu is activated via the main menu pulldown.
 * if never called before, create and manage all the widgets as a child of a
 * form. otherwise, just toggle whether the form is managed.
 */
static void
plt_manage ()
{
        if (!plotform_w)
            plt_create_form();

        if (XtIsManaged(plotform_w))
            XtUnmanageChild (plotform_w);
        else
            XtManageChild (plotform_w);
}

/* create the print dialog */
static void
plt_create_form()
{
	XmString str;
        Widget w;
        Widget f_w, rc_w;
        char *deffn;
        Arg args[20];
        int n;

        /* create the main dialog */

        n = 0;
        XtSetArg (args[n], XmNautoUnmanage, False); n++;
        XtSetArg (args[n], XmNdefaultPosition, False); n++;
        XtSetArg (args[n], XmNallowShellResize, True); n++;
        XtSetArg (args[n], XmNcolormap, xe_cm); n++;
        XtSetArg (args[n], XmNmarginHeight, 10); n++;
        XtSetArg (args[n], XmNmarginWidth, 10); n++;
        XtSetArg (args[n], XmNverticalSpacing, 10); n++;
        plotform_w = XmCreateFormDialog (toplevel_w, "Log", args, n);
        set_something (plotform_w, XmNcolormap, (XtArgVal)xe_cm);
        XtAddCallback (plotform_w, XmNhelpCallback, plt_help_cb, 0);

        n = 0;
        XtSetArg (args[n], XmNtitle, "xephem Log To File"); n++;
        XtSetValues (XtParent(plotform_w), args, n);

        /* make a RowColumn to hold everything */

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNspacing, 8); n++;
        XtSetArg (args[n], XmNisAligned, False); n++;
        XtSetArg (args[n], XmNadjustMargin, False); n++;
        rc_w = XmCreateRowColumn(plotform_w, "PlotRC", args, n);
        XtManageChild (rc_w);

        /* create filename text area and its label */

        n = 0;
        str = XmStringCreate("File name:", XmSTRING_DEFAULT_CHARSET);
        XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
        XtSetArg (args[n], XmNlabelString, str); n++;
        w = XmCreateLabel (rc_w, "PlotFnL", args, n);
        XmStringFree (str);
        XtManageChild (w);

        n = 0;
        filename_w = XmCreateText (rc_w, "Filename", args, n);
        deffn = XmTextGetString (filename_w);
        if (strlen(deffn) == 0)
            XmTextSetString (filename_w, DEF_PLTFN);
        XtFree (deffn);
        wtip (filename_w, "Enter name of file to write");
        XtManageChild (filename_w);

        /* add another separator */

        n = 0;
        w = XmCreateSeparator (rc_w, "Sep2", args, n);
        XtManageChild (w);

        /* add the command buttons across the bottom */

        n = 0;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        f_w = XmCreateForm (rc_w, "LogF", args, n);
        XtManageChild (f_w);

            n = 0;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 1); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 3); n++;
            w = XmCreatePushButton (f_w, "Ok", args, n);
            XtAddCallback (w, XmNactivateCallback, plt_ok_cb, 0);
            wtip (w, "Write log file and close this dialog");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 4); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 6); n++;
            w = XmCreatePushButton (f_w, "Cancel", args, n);
            XtAddCallback (w, XmNactivateCallback, plt_cancel_cb, 0);
            wtip (w, "Close this dialog and do nothing");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 7); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 9); n++;
            w = XmCreatePushButton (f_w, "Help", args, n);
            XtAddCallback (w, XmNactivateCallback, plt_help_cb, 0);
            wtip (w, "Additional information");
            XtManageChild (w);
}


/* called when the Ok button is hit in the print dialog */
/* ARGSUSED */
static void
plt_ok_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        char *txt = XmTextGetString (filename_w);
        if (existsh (txt) == 0) {
            char *buf;
            buf = XtMalloc (strlen(txt)+100);
            (void) sprintf (buf, "%s exists: Append or Overwrite?", txt);
            query (toplevel_w, buf, "Append", "Overwrite", "Cancel",
                            plt_try_append, plt_try_overwrite, plt_try_cancel);
            XtFree (buf);
        } else {
            plt_turn_on("w");
            plt_turn_off();
	    }
        XtFree (txt);
	if (XtIsManaged(plotform_w))
          XtUnmanageChild (plotform_w);
}

/* called by Cancel */
/* ARGSUSED */
static void
plt_cancel_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (XtIsManaged(plotform_w))
           XtUnmanageChild (plotform_w);
}

/* called by Help */
/* ARGSUSED */
static void
plt_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        static char *msg[] = {
       "Save to file (and give filename)",
       "Then press Ok.",
	};

        hlp_dialog ("LogFile", msg, sizeof(msg)/sizeof(msg[0]));
}


/* turn on plotting.
 * establish a file to use (and thereby set plt_fp, the plotting_is_on flag).
 */
static void
plt_turn_on (how)
char *how;      /* fopen how argument */
{
        char *txt;

        /* plotting is on if file opens ok */
        txt = XmTextGetString (filename_w);
        plt_fp = fopenh (txt, how);
        if (!plt_fp) {
            char *buf;
            buf = XtMalloc (strlen(txt)+100);
            (void) sprintf (buf, "Can not open %s: %s", txt, syserrstr());
            xe_msg (buf, 1);
            XtFree (buf);
        }
        XtFree (txt);

        if (plt_fp) {
	    int i;

	    /*
	     * If overwriting or writing for the first time
  	     * write then a header with a brief description
	     */
	    if (!strcmp(how,"w")) 
        	(void) fprintf (plt_fp, "Name             UTC-Date   UTC-Time      RA         Dec      RA(offset)  Dec(offset)  Mag  SP   Exp  Airmass       FITS\n");
		
	    for (i = 0; i < nhist; i++) {
		get_xmstring (hist[i].w, XmNlabelString, &txt);
                if (txt[0] != '\0')
                	(void) fprintf (plt_fp, "%s\n", txt);
            	XtFree (txt);
	    }
        }
}

/* called from the query routine when want to append to an existing plot file.*/
static void
plt_try_append()
{
        plt_turn_on("a");
        plt_turn_off();
}

/* called from the query routine when want to overwrite to an existing plot
 * file.
 */
static void
plt_try_overwrite()
{
        plt_turn_on("w");
        plt_turn_off();
}

/* called from the query routine when want decided not to make a plot file.  */
static void
plt_try_cancel()
{
	if (XtIsManaged(plotform_w))
           XtUnmanageChild (plotform_w);
}

static void
plt_turn_off ()
{
        if (plt_fp) {
            (void) fclose (plt_fp);
            plt_fp = 0;
        }
}
