/*
 * Stuff to control the MEADE LX200 telescope
 * Author: Carlos Guirao. cguirao@eso.org
 */
#include <stdio.h>
#include <errno.h>
#include <math.h>
#include <signal.h>
#if defined(__STDC__)
#include <stdlib.h>
#endif


#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/CascadeB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Scale.h>
#include <Xm/SelectioB.h>
#include <Xm/TextF.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>

#include "P_.h"
#include "astro.h"
#include "circum.h"
#include "preferences.h"
#include "skylist.h"
#include "lx200.h"
#include "lx200hist.h"

extern Widget toplevel_w;
extern XtAppContext xe_app;
extern Colormap xe_cm;

extern Now *mm_get_now P_((void));
extern char *getXRes P_((char *name, char *def));
extern char *syserrstr P_((void));
extern char *obj_description P_((Obj *op));
extern double atod P_((char *buf));
extern void fs_prdec P_((char out[], double jd));
extern void fs_ra P_((char buf[], double ra));
extern void get_something P_((Widget w, char *resource, XtArgVal value));
extern void get_xmstring P_((Widget w, char *resource, char **txtp));
extern void prompt_map_cb P_((Widget w, XtPointer client, XtPointer call));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txtp));
extern void xe_msg P_((char *msg, int app_modal));
extern void wtip P_((Widget w, char *tip));
extern void f_prdec P_((Widget w, double a));
extern void f_ra P_((Widget w, double ra));
extern void f_scansex P_((double o, char bp[], double *np));

extern void nairmass P_((double iha, double idec, double ilat, float exposure, float *airm));

/* Defaults for ETC and others */
#define DEF_ETC_MAG     "0.08"       /* default magnitude for ETC */
#define DEF_ETC_EXP     "10"         /* default exposure for ETC */
#define DEF_DEVICE      "/dev/ttyS0" /* default device port to telescope */
#define DEF_MAX_DEC     "80"         /* default device port to telescope */
#define DEF_SPECTRAL    "K2"         /* default spectral type */
#define DEF_INTERVAL    "2000"       /* 2 sec. to refresh Get Coordinates */

/* info about each field for a given type of object.
 * N.B. the prompt field is used to decide whether a button should be created
 *   to let this field be changed. Some prompts are determined at runtime
 *   though based on preference. For these, we need at least a dummy.
 *   the runtime setting is done in obj_change_cb().
 * N.B. the altp entries must match the values of the PREF_XX enums.
 */
typedef struct {
    int id;             /* field id */
    char *label;        /* label */
    char *prompt;       /* prompt if changeable, or 0 if it can't be */
    char *tip;          /* widget tip */
    char *altp[3];      /* alternate prompts for fields effected by prefs */
    Widget pb_w;        /* PushButton if it can be changed, else Label */
} Field;

enum {
    SLEW_RA, SLEW_DEC, OFF_RA, OFF_DEC
};

enum {
    PREF_COM1, PREF_COM2, PREF_COM3, PREF_COM4, PREF_COM5
};

static int port_lx200;

static char *ports[] = {
    "/dev/ttyS0","/dev/ttyS1","/dev/ttyS2","/dev/ttyS3", NULL
};

/*
 * Reusing F_SIZE for the exposure time calculator 
 */
static Field fixed_fields[] = {
    {F_RA,   "RA:",  "RA (h:m:s):", "Enter a new right ascension"},
    {F_DEC,  "Dec:", "Dec (d:m:s):", "Enter a new declination"},
    {F_RA,   "OFF_RA:",  "RA (h:m:s):", "Enter an offset for right ascension"},
    {F_DEC,  "OFF_Dec",  "Dec (d:m:s):", "Enter a offset for declination"},
    {O_NAME,  "Name:",  "Object name", "Name of Object"},
    {F_CLASS, "Class:", "Class name", "Database type classification"},
    {F_MAG,   "Mag:",   "Magnitude:", "Nominal magnitude"},
    {F_SPECT, "Spect:", "Spectral Type:", "Spectral type"},
    {E_INC,  "Exposure:", "Exposure Time:", "Calculated Exposure Time in seconds"},
    {0,  "Airmass:", "Airmass:", "Airmass at current location"},
    {0, "Airmass:", "Airmass:", "Median averaged airmass at given exposure"},
    {F_SIZE, "Interval:", "Interval:", "Interval to Get Coordinates from LX200"}
};

static Widget list_w;	/* object list dialog */

static void sv_create_list_w P_((void));
static void svc_pu_speed_cb P_((Widget w, XtPointer client, XtPointer call));
static void svc_pu_dir_cb P_((Widget w, XtPointer client, XEvent *ev,
    Boolean *continue_to_dispatch));
static void get_coord_cb P_((Widget w, XtPointer client, XtPointer call));
static void slew_cb P_((Widget w, XtPointer client, XtPointer call));
static void offset_cb P_((Widget w, XtPointer client, XtPointer call));
static void slew_offset_cb P_((Widget w, XtPointer client, XtPointer call));
static void obj_change_cb P_((Widget w, XtPointer client, XtPointer call));
static void obj_ctl_cb P_((Widget w, XtPointer client, XtPointer call));
static void obj_set_button P_((Widget w, int id));
static void sd_option_cb P_((Widget w, XtPointer client, XtPointer call));
static void prompt_ok_cb P_((Widget w, XtPointer client, XtPointer call));
static void lxh_help_cb P_((Widget w, XtPointer client, XtPointer call));
static void commu_cb P_((Widget w, XtPointer client, XtPointer call));
static void etc_control_cb P_((Widget w, XtPointer client, XtPointer call));
static void interval_control_cb P_((Widget w, XtPointer client, XtPointer call));
static void set_port_cb P_((Widget w, XtPointer client, XtPointer call));
static void prompt P_((Field *fp));
static void lx200_timeout_handler P_((XtPointer client, XtIntervalId *id));
static void set_alarm();
static void restore_alarm();
static void ignore_alarm();
static void compute_etc();
static void compute_interval();
static void etc_help_cb();
static void interval_help_cb();


/* stuff for the lx200 control dialog */

#define DIR_ROWS        3               /* rows in the direction matrix */
#define DIR_COLS        3               /* columns in the direction matrix */
#define DEFAULT_SPEED   GUIDE		/* Default slew speed when starting */

static int lx200_interval; 	        /* Interval between Get Coordinates */
static XtIntervalId lx200_interval_id;  /* set while waiting in pause loop */
static int lx200Connect = 0;
static int lx200HighPrecision = 2;	/* 1/0/2 if xephem High/Low/Undef prec*/
static int speed_mode = DEFAULT_SPEED;
static int dir_north_south = 0;
static int dir_east_west = 0;
static Widget lx200_w;		/* overall telescope control dialog */
static Widget dir_w[DIR_ROWS*DIR_COLS];	/* pushbtns in the DIR matrix */
static Widget lx200_ra_w;
static Widget lx200_dec_w;
static Widget off_ra_w;
static Widget off_dec_w;
static Widget ra_w;
static Widget dec_w;
static Widget offset_ra_w;
static Widget offset_dec_w;
static Widget name_w;
static Widget mag_w;
static Widget spect_w;
static Widget type_w;
static Widget exp_w;
static Widget etclabel_w;
static Widget airmass_w;
static Widget mairmass_w;
static Widget etc_w;
static Widget etc_mag_w;
static Widget etc_spect_w;
static Widget etc_exp_w;
static Widget interval_w;
static Widget get_interval_w;



static Pixel fg_yellow_pix;  	     /* used to Mark LX200 in yellow */
Pixel fg_lx200_p;

struct Btns {   /* to streamline creation of control buttons */
    int id;
    char *name;
    char *tip;
};

typedef struct {
            char *label;        /* toggle button label */
            char *name;         /* instance name */
            char *tip;          /* tip text */
} DirOpt;

enum {
    DIR_RESET, DIR_LR, DIR_TB, DIR_ROT
};

static struct Btns dopts[] = {
            {DIR_RESET, "Reset",    "Reset pad to default"},
            {DIR_LR,  	"Flip L/R", "Flip pad left-to-right"},
            {DIR_TB,    "Flip T/B", "Flip pad top-to-bottom"},
            {DIR_ROT,   "Rotate",   "Rotate the pad"},
};

static struct Btns speedbtns[] = {
    {SLEW, "Slew",    "Slewing telescope at speed Slew"},
    {FIND, "Find",    "Slewing telescope at speed Find"},
    {CENTER, "Center","Slewing telescope at speed Center"},
    {GUIDE, "Guide",  "Slewing telescope at speed Guide"},
};

struct Dir {  
    int id;
    char *name;
};

enum {
    DIR_NORTH_WEST,
    DIR_NORTH,
    DIR_NORTH_EAST,
    DIR_WEST, 
    DIR_STOP, 
    DIR_EAST,
    DIR_SOUTH_WEST,
    DIR_SOUTH,
    DIR_SOUTH_EAST
};

static char dir_name[][6] = {
    "NW", 
    "North",
    "NE",
    "West",
    "",
    "East",
    "SW",
    "South",
    "SE"
};

static struct Dir dirs[] = {
    {DIR_NORTH_WEST, 	dir_name[0]}, 
    {DIR_NORTH, 	dir_name[1]},
    {DIR_NORTH_EAST, 	dir_name[2]},
    {DIR_WEST,		dir_name[3]},
    {DIR_STOP,		dir_name[4]},
    {DIR_EAST,		dir_name[5]},
    {DIR_SOUTH_WEST, 	dir_name[6]},
    {DIR_SOUTH, 	dir_name[7]},
    {DIR_SOUTH_EAST,	dir_name[8]},
};

static void svc_create_lx200_w P_((void)); 
static void etc_create_w P_((void)); 
static void interval_create_w P_((void)); 

/* working storage - set when Reset, made permanent in database with Ok/Apply
 */
static Obj objx;                        /* local working copy */
static Obj objy;                        /* local working copy */
static Obj objz;                        /* local working copy */
static int objidx = OBJX;               /* which one is on-screen now */
#define objp    (objidx == OBJX ? &objx : objidx == OBJY ? &objy : &objz) 

/* bottom control panel buttons */
enum {
    SYNC_LX200, IDENT_LX200, IDENT_SLEW, LOG_INTO_FILE, LX200_CLOSE,  ETC_HELP, ETC_CLOSE, 
    INTERVAL_HELP, INTERVAL_CLOSE
};

static struct Btns skybtns[] = {
    {IDENT_LX200, "Mark LX200",
            "Mark LX200 coordinates on Sky View, but only if visible"},
    {IDENT_SLEW, "Mark Target",
            "Mark target coordinates on Sky View, but only if visible"},
    {LOG_INTO_FILE, "Add History",
            "Add all the target specifics to the history collection"},
};

static struct Btns skybtns1[] = {
    {SYNC_LX200, "Sync LX200",
            "Set LX200 coordinates to Target coordinates"},
    {LX200_CLOSE, "Close",
            "Close this dialog"},
};

static struct Btns etcbtns[] = {
    {ETC_CLOSE, "Close",
            "Close this dialog"},
    {ETC_HELP, "Help",
            "Display detailed help information"},
};

static struct Btns intervalbtns[] = {
    {INTERVAL_CLOSE, "Close",
            "Close this dialog"},
    {INTERVAL_HELP, "Help",
            "Display detailed help information"},
};

int
svc_ismanaged()
{
	return (lx200_w && XtIsManaged(lx200_w));
}

void 
svc_unmanage()
{
	if (lx200_w)
	    XtUnmanageChild (lx200_w);
}

void 
svc_manage()
{
	if (lx200_w)
	    XtManageChild (lx200_w);
}


/* callback from LX200... control button (from Control in skyviewmenu). */
/* ARGSUSED */
void
svc_control_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!lx200_w)
	    svc_create_lx200_w(); 

	if (XtIsManaged(lx200_w)) {
	    DisconnectLX200();
	    lx200Connect = 0;
	    XtUnmanageChild (lx200_w);
	    }
	else 
	    XtManageChild (lx200_w);
}


/* create the telescope control dialog */
static void
svc_create_lx200_w()
{
	Widget w;
	Widget mb_w;
	Widget pd_w;
	Widget cb_w;
	Widget rc_w;
	Widget nrb_w;
	Widget drb_w;
	Widget text_ra_w;
	Widget text_dec_w;
	Widget slew_coor_w;
	Widget text_slew_w;
	Widget labelt_w, labelf_w, labele_w, target_w;
	Widget ctlf_w, offf_w, slewf_w, offsetf_w;
	Widget skyf_w, skyg_w;
	Widget cascade;
	Widget pull_right;
	Widget tb1, tb2, tb3, tb4, tb5;
	Arg args[20];
	EventMask mask;
	XmString str;
	Field *fp;
	int n;
	int i;
	char *xr_device;

	/* initialize local objects */

        objx.o_type = objy.o_type = objz.o_type = FIXED;
	objx.f_epoch = objy.f_epoch = objz.f_epoch = (float)J2000;
        objx.f_RA = objy.f_RA = objz.f_RA = 0.;
        objx.f_dec = objy.f_dec = objz.f_dec =  0.;
	compute_interval();

	
	/* Only for test purposes */
	/*
        lx200Connect = 1;
	set_alarm();
	*/

	/* create form */

	n = 0;
	XtSetArg(args[n], XmNdefaultPosition, False); n++;
	XtSetArg(args[n], XmNautoUnmanage, False); n++; 
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	lx200_w = XmCreateFormDialog (toplevel_w, "LX200CTRL", args, n);
	set_something (lx200_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (lx200_w, XmNmapCallback, prompt_map_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "LX200"); n++;
	XtSetValues (XtParent(lx200_w), args, n);

        /* make the menubar */

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        mb_w = XmCreateMenuBar (lx200_w, "MB", args, n);
        XtManageChild (mb_w);

  	    /* make the Control pulldown */
 	    n = 0;
            pd_w = XmCreatePulldownMenu (mb_w, "ControlLX", args, n);

            n = 0;
            XtSetArg (args[n], XmNsubMenuId, pd_w);  n++;
            XtSetArg (args[n], XmNmnemonic, 'C'); n++;
            cb_w = XmCreateCascadeButton (mb_w, "ControlCB", args, n);
            set_xmstring (cb_w, XmNlabelString, "Control");
            XtManageChild (cb_w);

	    /* create the LX200 communications port: 4 options */
	    n = 0;
	    XtSetArg (args[n], XmNradioBehavior, True); n++;
	    pull_right = XmCreatePulldownMenu (pd_w, "LX200Port",args,n);

                xr_device  = getXRes ("Device",DEF_DEVICE);
                if (!strcmp(xr_device,DEF_DEVICE))
                   port_lx200 = PREF_COM1;
                else {
                   port_lx200 = PREF_COM5;
		   ports[PREF_COM5] = xr_device;
		   }

		n = 0;
		XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
                if (port_lx200 == PREF_COM1)
	           XtSetArg (args[n], XmNset, True); n++;
		tb1 = XmCreateToggleButton (pull_right, "COM1", args, n);
		XtManageChild (tb1);
		XtAddCallback (tb1, XmNvalueChangedCallback, set_port_cb,
							(XtPointer)PREF_COM1);
		set_xmstring (tb1, XmNlabelString, "/dev/ttyS0 (COM1)");

		n = 0;
		XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		tb2 = XmCreateToggleButton (pull_right, "COM2", args, n);
		XtAddCallback (tb2, XmNvalueChangedCallback, set_port_cb,
							(XtPointer)PREF_COM2);
		XtManageChild (tb2);
		set_xmstring (tb2, XmNlabelString, "/dev/ttyS1 (COM2)");

		n = 0;
		XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		tb3 = XmCreateToggleButton (pull_right, "COM3", args, n);
		XtAddCallback (tb3, XmNvalueChangedCallback, set_port_cb,
							(XtPointer)PREF_COM3);
		XtManageChild (tb3);
		set_xmstring (tb3, XmNlabelString, "/dev/ttyS2 (COM3)");

		n = 0;
		XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
		tb4 = XmCreateToggleButton (pull_right, "COM4", args, n);
		XtAddCallback (tb4, XmNvalueChangedCallback, set_port_cb,
							(XtPointer)PREF_COM4);
		XtManageChild (tb4);
		set_xmstring (tb4, XmNlabelString, "/dev/ttyS3 (COM4)");

                if (port_lx200 == PREF_COM5) {
		  n = 0;
		  XtSetArg (args[n], XmNvisibleWhenOff, True); n++;
	          XtSetArg (args[n], XmNset, True); n++;
		  tb5 = XmCreateToggleButton (pull_right, "COM5", args, n);
		  XtManageChild (tb5);
		  XtAddCallback (tb5, XmNvalueChangedCallback, set_port_cb,
							(XtPointer)PREF_COM5);
		  set_xmstring (tb5, XmNlabelString, xr_device);
		  }

              n = 0;
              XtSetArg (args[n], XmNsubMenuId, pull_right);  n++;
              cascade= XmCreateCascadeButton(pd_w,"LXS",args,n);
              XtManageChild (cascade);
              set_xmstring (cascade, XmNlabelString, "LX200 Port");
              wtip (cascade, "Select the LX200 port");

	      /* create the Exposure Time Calculator */
              n = 0;
              w = XmCreatePushButton (pd_w, "ETCS", args, n);
	      set_xmstring (w, XmNlabelString, "ETC Setup");
              XtAddCallback (w, XmNactivateCallback, etc_control_cb, NULL);
              wtip (w, "Exposure Time Calculator");
              XtManageChild (w);

	      /* create the Interval Modifier */
              n = 0;
              w = XmCreatePushButton (pd_w, "INTS", args, n);
	      set_xmstring (w, XmNlabelString, "Interval Setup");
              XtAddCallback (w, XmNactivateCallback, interval_control_cb, NULL);
              wtip (w, "Modify interval to Get Coordinates from LX200");
              XtManageChild (w);

              n = 0;
              w = XmCreateSeparator (pd_w, "Sep", args, n);
              XtManageChild (w);

              n = 0;
              w = XmCreatePushButton (pd_w, "Close", args, n);
              XtAddCallback (w, XmNactivateCallback, obj_ctl_cb,
						(XtPointer)LX200_CLOSE);
              wtip (w, "Close this display");
              XtManageChild (w);

            /* make the History pulldown */
	    lxh_create(mb_w);

            /* make the Help pulldown */
            n = 0;
            pd_w = XmCreatePulldownMenu (mb_w, "HPD", args, n);

            n = 0;
            XtSetArg (args[n], XmNsubMenuId, pd_w); n++;
            XtSetArg (args[n], XmNmnemonic, 'H'); n++;
            cb_w = XmCreateCascadeButton (mb_w, "Help", args, n);
            XtManageChild (cb_w);
            set_something (mb_w, XmNmenuHelpWidget, (XtArgVal)cb_w);

            n = 0;
            w = XmCreatePushButton (pd_w, "Help", args, n);
            XtAddCallback (w, XmNactivateCallback, lxh_help_cb, 0);
            XtManageChild (w);

        /* round or square Radio box */
        n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNtopOffset, 30); n++; 
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;

        XtSetArg (args[n], XmNleftOffset, 40); n++; 
        XtSetArg (args[n], XmNspacing, 10); n++; 
        XtSetArg (args[n], XmNorientation, XmVERTICAL); n++;
        XtSetArg (args[n], XmNnumColumns, 4); n++;
        nrb_w = XmCreateRadioBox (lx200_w, "SPEED", args, n);
        XtManageChild (nrb_w);

            for (i = 0; i < XtNumber(speedbtns); i++) {
              n = 0;
	      if (speedbtns[i].id==DEFAULT_SPEED) 
		{XtSetArg (args[n], XmNset, True); n++; }
              w = XmCreateToggleButton (nrb_w, speedbtns[i].name, args, n);
	      XtAddCallback (w, XmNvalueChangedCallback,svc_pu_speed_cb,
                                                    (XtPointer)speedbtns[i].id);
              wtip (w, speedbtns[i].tip);
              XtManageChild (w);
            }
	get_something (w, XmNforeground, (XtArgVal)&fg_yellow_pix);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, nrb_w); n++; 
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (lx200_w, "Sep", args, n);
	XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNleftOffset, 20); n++; 
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 40); n++;
        drb_w = XmCreateForm (lx200_w, "DirOptF", args, n); 
        XtManageChild (drb_w);

        for (i = 0; i < XtNumber(dopts); i++) {
            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, i*10 ); n++; 
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 8 + i*10); n++; 
            w = XmCreateToggleButton (drb_w, dopts[i].name, args, n);
            XtAddCallback (w, XmNvalueChangedCallback, sd_option_cb,
						(XtPointer)dopts[i].id);
            XtManageChild (w);
            wtip (w, dopts[i].tip);
	    }

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, drb_w); n++;
        XtSetArg (args[n], XmNleftOffset, 100); n++; 
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
/*
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++; 
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_WIDGET); n++;
*/
        XtSetArg (args[n], XmNnumColumns, DIR_COLS); n++; 
        XtSetArg (args[n], XmNisAligned, False); n++;
        XtSetArg (args[n], XmNpacking, XmPACK_COLUMN); n++;
        XtSetArg (args[n], XmNorientation, XmHORIZONTAL); n++;
	XtSetArg (args[n], XmNentryAlignment, XmALIGNMENT_CENTER); n++;
        XtSetArg (args[n], XmNadjustMargin, False); n++;
        XtSetArg (args[n], XmNspacing, 0); n++;
        rc_w = XmCreateRowColumn (lx200_w, "DirRC", args, n);
        XtManageChild (rc_w);

        for (i = 0; i < DIR_ROWS*DIR_COLS; i++) {
            n = 0;
            XtSetArg (args[n], XmNrecomputeSize, False); n++;
            XtSetArg (args[n], XmNmarginTop, 10); n++;
            XtSetArg (args[n], XmNmarginBottom, 10); n++;
            XtSetArg (args[n], XmNshadowThickness, 2); n++;
            dir_w[i] = XmCreatePushButton (rc_w, dirs[i].name, args, n);
	    mask = ButtonReleaseMask | ButtonPressMask;
	    /* If center then ignore */
	    if ( i != 4) 
            XtAddEventHandler (dir_w[i], mask, False, svc_pu_dir_cb,
						(XtPointer)&dirs[i].id);
            XtManageChild (dir_w[i]);
        }

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, rc_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (lx200_w, "Sep", args, n);
	XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 30); n++;
        labelt_w = XmCreateForm (lx200_w, "LabelT", args, n);
        XtManageChild (labelt_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
	    w = XmCreateLabel (labelt_w, "TName", args, n);
	    set_xmstring (w, XmNlabelString, "Object Name");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 30); n++;
	    w = XmCreateLabel (labelt_w, "TType", args, n);
	    set_xmstring (w, XmNlabelString, "Type");
            XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, labelt_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 30); n++;
        target_w = XmCreateForm (lx200_w, "Target", args, n);
        XtManageChild (target_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
            name_w = XmCreatePushButton (target_w, "-----", args, n);
            fp = &fixed_fields[4];
            fp->pb_w = name_w;
            wtip (name_w, fp->tip);
            XtManageChild (name_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 30); n++;
            type_w = XmCreatePushButton (target_w, "-----", args, n);
            fp = &fixed_fields[5];
            fp->pb_w = type_w;
            wtip (type_w, fp->tip);
            XtManageChild (type_w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, target_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 30); n++;
        labele_w = XmCreateForm (lx200_w, "LabelE", args, n);
        XtManageChild (labele_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 6); n++;
	    w = XmCreateLabel (labele_w, "TMag", args, n);
	    set_xmstring (w, XmNlabelString, "Mag");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 6); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 12); n++;
	    w = XmCreateLabel (labele_w, "TSpect", args, n);
	    set_xmstring (w, XmNlabelString, "SP");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 12); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 18); n++;
	    w = XmCreateLabel (labele_w, "TETC", args, n);
	    set_xmstring (w, XmNlabelString, "Exp");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 18); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 24); n++;
	    w = XmCreateLabel (labele_w, "AIRMASS", args, n);
	    set_xmstring (w, XmNlabelString, "Air");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 24); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 30); n++;
	    w = XmCreateLabel (labele_w, "MAIRMASS", args, n);
	    set_xmstring (w, XmNlabelString, "Air-M");
            XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, labele_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 30); n++;
        etclabel_w = XmCreateForm (lx200_w, "ETCLABEL", args, n);
        XtManageChild (etclabel_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 6); n++;
            mag_w = XmCreatePushButton (etclabel_w, "--", args, n);
	    fp = &fixed_fields[6];
            fp->pb_w = mag_w;
            wtip (mag_w, fp->tip);
            XtManageChild (mag_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 6); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 12); n++;
            spect_w = XmCreatePushButton (etclabel_w, "--", args, n);
	    fp = &fixed_fields[7];
            fp->pb_w = spect_w;
            wtip (spect_w, fp->tip);
            XtManageChild (spect_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 12); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 18); n++;
            exp_w = XmCreatePushButton (etclabel_w, "--", args, n);
	    fp = &fixed_fields[8];
            fp->pb_w = exp_w;
            wtip (exp_w, fp->tip);
            XtManageChild (exp_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 18); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 24); n++;
            airmass_w = XmCreatePushButton (etclabel_w, "--", args, n);
	    fp = &fixed_fields[9];
            fp->pb_w = airmass_w;
            wtip (airmass_w, fp->tip);
            XtManageChild (airmass_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 24); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 30); n++;
            mairmass_w = XmCreatePushButton (etclabel_w, "--", args, n);
	    fp = &fixed_fields[10];
            fp->pb_w = mairmass_w;
            wtip (mairmass_w, fp->tip);
            XtManageChild (mairmass_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, etclabel_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (lx200_w, "Sep", args, n);
	XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 25); n++;
        labelf_w = XmCreateForm (lx200_w, "LabelF", args, n);
        XtManageChild (labelf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
	    w = XmCreateLabel (labelf_w, "LCMD", args, n);
	    set_xmstring (w, XmNlabelString, "Command");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
	    w = XmCreateLabel (labelf_w, "LAR", args, n);
	    set_xmstring (w, XmNlabelString, "RA");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 25); n++;
	    w = XmCreateLabel (labelf_w, "LDEC", args, n);
	    set_xmstring (w, XmNlabelString, "Dec");
            XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, labelf_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 25); n++;
        ctlf_w = XmCreateForm (lx200_w, "GetF", args, n);
        XtManageChild (ctlf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
            w = XmCreatePushButton (ctlf_w, "LX200", args, n);
            XtAddCallback (w, XmNactivateCallback, get_coord_cb,NULL);
            wtip (w, "Get coordinates from LX200");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
            lx200_ra_w = XmCreatePushButton (ctlf_w, "0:00:00.00", args, n);
            wtip (lx200_ra_w, "Right ascension from LX200");
            XtManageChild (lx200_ra_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 25); n++;
            lx200_dec_w = XmCreatePushButton (ctlf_w, "0:00:00.0", args, n);
            wtip (lx200_dec_w, "Declination from LX200");
            XtManageChild (lx200_dec_w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, ctlf_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 25); n++;
        offf_w = XmCreateForm (lx200_w, "GetOffF", args, n);
        XtManageChild (offf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
            w = XmCreatePushButton (offf_w, "Offset", args, n);
            XtAddCallback (w, XmNactivateCallback, offset_cb,NULL);
            wtip (w, "Calculates offset between values in LX200 and Slew windows");
            XtManageChild (w);
            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
            off_ra_w = XmCreatePushButton (offf_w, "0:00:00.00", args, n);
            wtip (off_ra_w, "Offset in Right ascension between LX200 and Slew");
            XtManageChild (off_ra_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 25); n++;
            off_dec_w = XmCreatePushButton (offf_w, "0:00:00.0", args, n);
            wtip (off_dec_w, "Offset in declination between LX200 and Slew");
            XtManageChild (off_dec_w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, offf_w); n++; 
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 25); n++;
        slewf_w = XmCreateForm (lx200_w, "SlewF", args, n);
        XtManageChild (slewf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
            w = XmCreatePushButton (slewf_w, "Target", args, n);
            XtAddCallback (w, XmNactivateCallback, slew_cb,NULL);
            wtip (w, "Slew LX200 to target coordinates");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
            ra_w = XmCreatePushButton (slewf_w, "0:00:00.00", args, n);
	    fp = &fixed_fields[0];
	    fp->pb_w = ra_w;
            XtAddCallback (ra_w, XmNactivateCallback, obj_change_cb,
                                                      (XtPointer)fp);
            wtip (ra_w, fp->tip);
            XtManageChild (ra_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 25); n++;
            dec_w = XmCreatePushButton (slewf_w, "0:00:00.0", args, n);
	    fp = &fixed_fields[1];
	    fp->pb_w = dec_w;
            XtAddCallback (dec_w, XmNactivateCallback, obj_change_cb,
                                                       (XtPointer)fp);
            wtip (dec_w, fp->tip);
            XtManageChild (dec_w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, slewf_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 25); n++;
        offsetf_w = XmCreateForm (lx200_w, "OffsetF", args, n);
        XtManageChild (offsetf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
            w = XmCreatePushButton (offsetf_w, "Target+", args, n);
            XtAddCallback (w, XmNactivateCallback, slew_offset_cb, NULL);
            wtip (w, "Slew LX200 to target coordinates + offset");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 15); n++;
            offset_ra_w = XmCreatePushButton (offsetf_w, "0:00:00.00", args, n);
	    fp = &fixed_fields[2];
	    fp->pb_w = offset_ra_w;
            XtAddCallback (offset_ra_w, XmNactivateCallback, obj_change_cb,
                                                      (XtPointer)fp);
            wtip (offset_ra_w, fp->tip);
            XtManageChild (offset_ra_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 15); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 25); n++;
            offset_dec_w = XmCreatePushButton (offsetf_w, "0:00:00.0", args, n);
	    fp = &fixed_fields[3];
	    fp->pb_w = offset_dec_w;
            XtAddCallback (offset_dec_w, XmNactivateCallback, obj_change_cb,
                                                       (XtPointer)fp);
            wtip (offset_dec_w, fp->tip);
            XtManageChild (offset_dec_w);


	/* separator */
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, offsetf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (lx200_w, "Sep", args, n);
	XtManageChild (w);

        /* make a form in a frame to hold the bottom control buttons */

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 15); n++;
        skyf_w = XmCreateForm (lx200_w, "SkyF", args, n);
        XtManageChild (skyf_w);
            for (i = 0; i < XtNumber(skybtns); i++) {
                n = 0;
                XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
                XtSetArg (args[n], XmNtopOffset, 5); n++; 
                XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            /*    XtSetArg (args[n], XmNbottomOffset, 5); n++; */
                XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNleftPosition, 0 + i*5); n++;
                XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNrightPosition, 5 + i*5); n++;
                w = XmCreatePushButton (skyf_w, skybtns[i].name, args, n);
                XtAddCallback (w, XmNactivateCallback, obj_ctl_cb,
                                                    (XtPointer)skybtns[i].id);
                if (skybtns[i].tip)
                    wtip (w, skybtns[i].tip);
                XtManageChild (w);
            }

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, skyf_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 20); n++;
        skyg_w = XmCreateForm (lx200_w, "SkyG", args, n);
        XtManageChild (skyg_w);
            for (i = 0; i < XtNumber(skybtns1); i++) {
                n = 0;
                XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
              /*  XtSetArg (args[n], XmNtopOffset, 5); n++; */
                XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
              /*  XtSetArg (args[n], XmNbottomOffset, 5); n++; */
                XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNleftPosition, 0 + i*10); n++;
                XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNrightPosition, 10 + i*10); n++;
                w = XmCreatePushButton (skyg_w, skybtns1[i].name, args, n);
                XtAddCallback (w, XmNactivateCallback, obj_ctl_cb,
                                                    (XtPointer)skybtns1[i].id);
                if (skybtns1[i].tip)
                    wtip (w, skybtns1[i].tip);
                XtManageChild (w);
            }

}

/*
 * Checking connection with LX200. Return 0 if OK, else 1 
 */
static int
check_lx200()
{
	char msg[128];
	ignore_alarm();
        if(CheckConnectLX200()) {
		lx200Connect = 1;
		set_alarm();
	}
	else	{
		InitiatePortLX200(ports[port_lx200]);
        	if(ConnectLX200()) {
			lx200Connect = 1;
			set_alarm();
		}
		else  	{
			lx200Connect = 0;
			(void) sprintf (msg, "Error connecting to LX200 on port %s\n%s\n", ports[port_lx200],strerror(errno));
			xe_msg (msg, 1); 
		}
	}
	return(lx200Connect);
}

/* called on receipt of a 
 * ButtonPress/ButonRelease while the cursor is over the dirs_w[i] windows.
 */
/* ARGSUSED */
static void
svc_pu_dir_cb (w, client, ev, continue_to_dispatch)
Widget w;
XtPointer client;
XEvent *ev;
Boolean *continue_to_dispatch;
{
	int code = *(int *)(client);
	int i;
        int evt = ev->type;
        int b1p, b1r;

        /* what happened? */
        b1p = evt == ButtonPress   && ev->xbutton.button == Button1;
        b1r = evt == ButtonRelease && ev->xbutton.button == Button1;

        /* do we care? */
        if (!b1p && !b1r) 
            return;

	if (b1r) code = DIR_STOP;
	if (check_lx200()) {
	  if (dir_north_south) StopSlew(dir_north_south);
	  if (dir_east_west) StopSlew(dir_east_west);

          switch (code) {
          case DIR_NORTH:
		dir_north_south = NORTH;
		dir_east_west = 0;
		break;
          case DIR_NORTH_EAST:
		dir_north_south = NORTH;
		dir_east_west = EAST;
		break;
          case DIR_NORTH_WEST:
		dir_north_south = NORTH;
		dir_east_west = WEST;
		break;
          case DIR_SOUTH:
		dir_north_south = SOUTH;
		dir_east_west = 0;
		break;
          case DIR_SOUTH_EAST:
		dir_north_south = SOUTH;
		dir_east_west = EAST;
		break;
          case DIR_SOUTH_WEST:
		dir_north_south = SOUTH;
		dir_east_west = WEST;
		break;
          case DIR_EAST:
		dir_north_south = 0;
		dir_east_west = EAST;
		break;
          case DIR_WEST:
		dir_north_south = 0;
		dir_east_west = WEST;
		break;
          case DIR_STOP:
		dir_north_south = 0;
		dir_east_west = 0;
		break;
	  default:
		printf ("svc_pu_dir_db(): code=%d\n", code);
		exit (1);
		break;
	  }
	  ignore_alarm();
          SetRate(speed_mode);
	  if (dir_north_south) StartSlew(dir_north_south);
	  if (dir_east_west) StartSlew(dir_east_west);
	  restore_alarm();
	}
}

/* called when any of the speed pushbuttons are activated.
 * client is a code to indicate which.
 */
/* ARGSUSED */
static void
svc_pu_speed_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call; 
{ 
	speed_mode = (int)client; 
}

/* called from skyviewmenu.c when the Slew to coordinates 
 *  pushbutton is activated.
 */
void
point_slew (Obj *op)
{
	if (!lx200_w)
	    svc_create_lx200_w(); 

	if (!etc_w)
	    etc_create_w(); 

        f_ra (ra_w,op->s_ra);
        f_prdec (dec_w,op->s_dec);

/*	slew_cb(NULL,NULL,NULL); */
	/*
	* Set coordinates for default object 
        */
        if (op->f_class) {
	  (void) strcpy(objx.o_name, op->o_name);
          f_double (mag_w, "%g", get_mag(op));
	  }
	else {
	  (void) strcpy(objx.o_name, "-----");
          f_string (mag_w, "--");
	  }
	objx.f_class = op->f_class;
	objx.f_mag = op->f_mag;
	objx.o_type = op->o_type;
	objx.f_epoch = op->f_epoch;
        objx.f_RA = op->s_ra;
        objx.f_dec = op->s_dec;

        f_string (name_w, objx.o_name);
        f_string (type_w, obj_description(op));
        if (is_type(op, FIXEDM) && op->f_class != 'G' && op->f_spect[0]) {
	    objx.f_spect[0] = op->f_spect[0];
	    objx.f_spect[1] = op->f_spect[1];
            }
        else { objx.f_spect[0] ='-';  objx.f_spect[1] ='-';}
        f_string (spect_w, objx.f_spect);

        if (op->f_class) 
	  compute_etc();
	else
          f_string (exp_w, "--");
}	


/* called when the Slew to coordinates pushbutton is activated.
 */
static void
slew_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        double ra, dec;
        double tmp_ra, tmp_dec;
        char *txt_ra, *txt_dec;
	double maxdec;
	char *max_dec;
        double old = 0.0;
        char out_ra[64];
        char out_dec[64];

	/* printf("slew_cb\n"); */
        if (check_lx200()) { 
           get_xmstring (ra_w, XmNlabelString, &txt_ra);
           get_xmstring (dec_w, XmNlabelString, &txt_dec);
           f_scansex (old,txt_ra,&tmp_ra);
           f_scansex (old,txt_dec,&tmp_dec);
           ra = hrrad(tmp_ra);
           dec = degrad(tmp_dec);
           XtFree(txt_ra);
           XtFree(txt_dec);
           fs_ra(out_ra,ra);
           fs_prdec(out_dec,dec);


           /* Check if target above technical horizont */
           /* In an equatorial mount anything above this horizont */
           /* will make the instrument atached to the telescope */
           /* to hit the telescope hook */
           /* printf("slew_cb: dec=%f\n",dec); */
           max_dec  = getXRes ("MaxDec",DEF_MAX_DEC);
           f_scansex (old,max_dec,&maxdec);
           if (fabs(tmp_dec) > fabs(maxdec)) {
                char msg[128];
                (void) sprintf (msg, "Error: Target Dec %2.2f is above MaxDec of %2.2f!\n", fabs(tmp_dec), fabs(maxdec));
                xe_msg (msg, 1);
		return;
		}
	   /* printf("slew_cb: ra=%s dec=%s\n",out_ra,out_dec); */

	   ignore_alarm();
	   /* Toggle telescope to the same xephem precision */
	   if (out_ra[5] == '.' && lx200HighPrecision != 0) {
		SetFormat(0);
		lx200HighPrecision = 0;
		}
	   if (out_ra[5] == ':' && lx200HighPrecision != 1) {
		SetFormat(1);
		lx200HighPrecision = 1;
		}
           Slew2Coords(out_ra,out_dec);
	   restore_alarm();

	   /*
	    * Set coordinates for default object 
            */
           objx.f_RA = ra;
           objx.f_dec = dec;
	} 
}


/* called when the Get Coordinates pushbutton is activated.
 */
static void
get_coord_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	double ra, dec;
	double tmp_ra, tmp_dec;
        char in_ra[64];
        char in_dec[64];
        double old = 0.0;

/* Only for test purposes */
/*
        char *txt_ra, *txt_dec;
        get_xmstring (lx200_ra_w, XmNlabelString, &txt_ra);
        get_xmstring (lx200_dec_w, XmNlabelString, &txt_dec);
        f_scansex (old,txt_ra,&tmp_ra);
        f_scansex (old,txt_dec,&tmp_dec);
        ra = hrrad(tmp_ra);
        dec = degrad(tmp_dec);
	ra++;
	dec++;
        XtFree(txt_ra);
        XtFree(txt_dec);
        f_ra(lx200_ra_w,ra);
        f_prdec(lx200_dec_w,dec);
        XmUpdateDisplay (lx200_ra_w);
        XmUpdateDisplay (lx200_dec_w);
        objy.f_RA = ra;
        objy.f_dec = dec;
*/

	if (check_lx200()) {
	   ignore_alarm();
	   GetRAlx200(in_ra);
	   GetDeclx200(in_dec);
	   restore_alarm();
	   /* printf("get_coord_cb: ra=%s dec=%s\n",in_ra,in_dec); */
           f_scansex (old,in_ra,&tmp_ra);
           f_scansex (old,in_dec,&tmp_dec);
           ra = hrrad(tmp_ra);
           dec = degrad(tmp_dec);

           f_ra (lx200_ra_w,ra);
           f_prdec (lx200_dec_w,dec);

           XmUpdateDisplay (lx200_ra_w);
           XmUpdateDisplay (lx200_dec_w);

           objy.f_RA = ra;
           objy.f_dec = dec;
     	}
        offset_cb (w, client, call);
}

/* called when the slew+ pushbuttons are activated.
 */
static void
slew_offset_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        double ra, dec;         
        double off_ra, off_dec;

        char *txt_ra, *txt_dec;
        double old = 0.0;
	double tmp_ra, tmp_dec;

        char out_ra[64];
        char out_dec[64];

        get_xmstring (offset_ra_w, XmNlabelString, &txt_ra);
        get_xmstring (offset_dec_w, XmNlabelString, &txt_dec);
        f_scansex (old,txt_ra,&tmp_ra);
        f_scansex (old,txt_dec,&tmp_dec);
	off_ra = hrrad(tmp_ra);
	off_dec = degrad(tmp_dec);
        XtFree(txt_ra);
        XtFree(txt_dec);

        ra = objx.f_RA + off_ra;
        dec = objx.f_dec + off_dec;

        fs_ra(out_ra,ra);
        fs_prdec(out_dec,dec);

        if (check_lx200()) {
		/* printf("slew_cb: ra=%s dec=%s\n",out_ra,out_dec); */
		ignore_alarm();

		/* Toggle telescope to the same xephem precision */
		if (out_ra[5] == '.' && lx200HighPrecision != 0) {
				SetFormat(0);
				lx200HighPrecision = 0;
				}
		if (out_ra[5] == ':' && lx200HighPrecision != 1) {
				SetFormat(1);
				lx200HighPrecision = 1;
				}
		Slew2Coords(out_ra,out_dec);
		restore_alarm();
	}
}

/* called when the Offset pushbuttons are activated.
 */
static void
offset_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        objz.f_RA =  objy.f_RA  - objx.f_RA;
        objz.f_dec = objy.f_dec - objx.f_dec;

        f_ra (off_ra_w,objz.f_RA);
        f_prdec (off_dec_w,objz.f_dec);

        XmUpdateDisplay (off_ra_w);
        XmUpdateDisplay (off_dec_w);
}

/* set label for the given label widget according to the info in the
 * given field of the current working object.
 */
static void
obj_set_button (w, id)
Widget w;
int id;
{
        static char me[] = "obj_set_button";

        switch (id) {
        case O_NAME:
            f_string (w, objp->o_name);
            break;
        case F_CLASS:
            f_string (w, obj_description(objp));
            break;
        case F_MAG:
            f_double (w, "%g", get_mag(objp));
	    compute_etc();
            break;
        case F_SPECT:
            f_string (w, objp->f_spect);
            break; 
        case F_SIZE:
            f_double (w, "%g", objp->f_size);
	    compute_interval();
            break;
        case E_INC:
            f_double (w, "%g", objp->e_inc);
	    compute_etc();
            break;
        case F_RA:
            f_ra (w, objp->f_RA);
            break;
        case F_DEC:
            f_prdec (w, objp->f_dec);
            break;
        default:
            printf ("%s: bad parabolic id: %d\n", me, id);
            exit (1);
        }
}



/* user typed OK to a prompt for fp. get his new value and use it.
 * get fp from userData.
 */
/* ARGSUSED */
static void
prompt_ok_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        XmSelectionBoxCallbackStruct *s = (XmSelectionBoxCallbackStruct *)call;
        Field *fp;
        char *text;

        switch (s->reason) {
        case XmCR_OK:
            get_xmstring(w, XmNtextString, &text);
            get_something (w, XmNuserData, (XtArgVal)&fp);
            (void) db_set_field (text, fp->id,
                            (PrefDateFormat) pref_get(PREF_DATE_FORMAT), objp);
            obj_set_button (fp->pb_w, fp->id);
            XtFree (text);

            break;
        }

        XtUnmanageChild (w);
}


/* put up a prompt dialog near the cursor to ask about fp.
 * make it app modal since we don't want the object type to change under us.
 */
static void
prompt (fp)
Field *fp;
{
        static Widget dw;
        Widget w;

        if (!dw) {
            XmString title;
            Arg args[20];
            int n;

            title = XmStringCreate("xephem Object Prompt",
                                                    XmSTRING_DEFAULT_CHARSET);
            n = 0;
            XtSetArg(args[n], XmNdefaultPosition, False);  n++;
            XtSetArg(args[n], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);  n++;

            XtSetArg(args[n], XmNdialogTitle, title);  n++;
            XtSetArg (args[n], XmNcolormap, xe_cm); n++;
            dw = XmCreatePromptDialog(toplevel_w, "ObjPrompt", args, n);
            set_something (dw, XmNcolormap, (XtArgVal)xe_cm);
            XtAddCallback (dw, XmNmapCallback, prompt_map_cb, NULL);
            XtAddCallback (dw, XmNokCallback, prompt_ok_cb, NULL);
            XtAddCallback (dw, XmNcancelCallback, prompt_ok_cb, NULL);
            XmStringFree (title);

            w = XmSelectionBoxGetChild (dw, XmDIALOG_HELP_BUTTON);
            XtUnmanageChild (w);
        }

        set_something (dw, XmNuserData, (XtArgVal)fp);
        set_xmstring (dw, XmNselectionLabelString, fp->prompt);

        /* preload with current string -- and skip leading blanks */
        if (pref_get (PREF_PRE_FILL) == PREF_PREFILL) {
            char *txt, *txt0;

            get_xmstring (fp->pb_w, XmNlabelString, &txt0);
            for (txt = txt0; *txt == ' '; txt++)
                continue;
	   set_xmstring(dw, XmNtextString, txt);
            XtFree (txt0);
        } else
            set_xmstring(dw, XmNtextString, "");

        XtManageChild (dw);

#if XmVersion >= 1001
        w = XmSelectionBoxGetChild (dw, XmDIALOG_TEXT);
        XmProcessTraversal (w, XmTRAVERSE_CURRENT);
        XmProcessTraversal (w, XmTRAVERSE_CURRENT); /* yes, twice!! */
#endif
}

/* callback from any of the obj field info buttons being activated.
 * client is the Field pointer for this item.
 */
/* ARGSUSED */
static void
obj_change_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        Field *fp = (Field *)client;
	if (strcmp(fp->label,"RA:") && strcmp(fp->label,"Dec:")) 
            objidx = OBJZ;               
    	else 
            objidx = OBJX;               

        prompt (fp);
}

/* callback from any of the bottom control buttons.
 * id is in client.
 */
/* ARGSUSED */
static void
obj_ctl_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        int id = (int) client;

        switch (id) {
        case SYNC_LX200:
            if (check_lx200()) {
              /* printf("Sync Coordinates to those from target"); */
              ignore_alarm();
              SyncObjectCoords();
              restore_alarm();
            } break;
        case IDENT_LX200:
            objidx = OBJY;               
            if (objp->o_type != UNDEFOBJ) {
                if (obj_cir (mm_get_now(), objp) < 0) {
                    printf ("obj_ctl_cb: can't get obj_cir for IDENT\n"); 
                    exit (1);
                }
		/* yellow color to Mark LX200 in the skymap */
		fg_lx200_p = fg_yellow_pix;
                sv_id (objp);
		fg_lx200_p = 0;
            } break;
        case IDENT_SLEW:
            objidx = OBJX;               
            if (objp->o_type != UNDEFOBJ) {
                if (obj_cir (mm_get_now(), objp) < 0) {
                    printf ("obj_ctl_cb: can't get obj_cir for IDENT\n");
                    exit (1);
                }
                sv_id (objp);
            }
            break;
        case LOG_INTO_FILE:
	    lxh_add_cb (w, client, call);
            break;
        case LX200_CLOSE:
	    ignore_alarm();
	    DisconnectLX200();
	    restore_alarm();
	    lx200Connect = 0;
            XtUnmanageChild (lx200_w);
            break;
        case ETC_CLOSE:
            XtUnmanageChild (etc_w);
            break;
        case ETC_HELP:
            etc_help_cb();
            break;
        case INTERVAL_CLOSE:
            XtUnmanageChild (interval_w);
            break;
        case INTERVAL_HELP:
            interval_help_cb();
            break;
        }
}

/* callback from any of the direcction control buttons.
 * id is in client.
 */
/* ARGSUSED */
static void
sd_option_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	int tmp_id;
	int i;
	char *tmp_name;
        int id = (int) client;
	/* printf("DIR_OPTION=%d\n",id); */

        switch (id) {
        case DIR_RESET:
	    for (i = 0; i < DIR_ROWS*DIR_COLS; i++) {
	       dirs[i].name=dir_name[i]; dirs[i].id=i;
	       }
            break;
        case DIR_LR:
	    tmp_name=dirs[0].name; tmp_id=dirs[0].id;
	    dirs[0].name=dirs[2].name; dirs[0].id=dirs[2].id;
	    dirs[2].name=tmp_name; dirs[2].id=tmp_id;
	    tmp_name=dirs[3].name; tmp_id=dirs[3].id;
	    dirs[3].name=dirs[5].name; dirs[3].id=dirs[5].id;
	    dirs[5].name=tmp_name; dirs[5].id=tmp_id;
	    tmp_name=dirs[6].name; tmp_id=dirs[6].id;
	    dirs[6].name=dirs[8].name; dirs[6].id=dirs[8].id;
	    dirs[8].name=tmp_name; dirs[8].id=tmp_id;
            break;
        case DIR_TB:
	    tmp_name=dirs[0].name; tmp_id=dirs[0].id;
	    dirs[0].name=dirs[6].name; dirs[0].id=dirs[6].id;
	    dirs[6].name=tmp_name; dirs[6].id=tmp_id;
	    tmp_name=dirs[1].name; tmp_id=dirs[1].id;
	    dirs[1].name=dirs[7].name; dirs[1].id=dirs[7].id;
	    dirs[7].name=tmp_name; dirs[7].id=tmp_id;
	    tmp_name=dirs[2].name; tmp_id=dirs[2].id;
	    dirs[2].name=dirs[8].name; dirs[2].id=dirs[8].id;
	    dirs[8].name=tmp_name; dirs[8].id=tmp_id;
            break;
        case DIR_ROT:
	    tmp_name=dirs[0].name; tmp_id=dirs[0].id;
	    dirs[0].name=dirs[1].name; dirs[0].id=dirs[1].id;
	    dirs[1].name=dirs[2].name; dirs[1].id=dirs[2].id;
	    dirs[2].name=dirs[5].name; dirs[2].id=dirs[5].id;
	    dirs[5].name=dirs[8].name; dirs[5].id=dirs[8].id;
	    dirs[8].name=dirs[7].name; dirs[8].id=dirs[7].id;
	    dirs[7].name=dirs[6].name; dirs[7].id=dirs[6].id;
	    dirs[6].name=dirs[3].name; dirs[6].id=dirs[3].id;
	    dirs[3].name=tmp_name; dirs[3].id=tmp_id;
            break;
        }

	for (i = 0; i < DIR_ROWS*DIR_COLS; i++) 
	    set_xmstring (dir_w[i], XmNlabelString, dirs[i].name);
}

/* 
 * Timeout: get coordinates from lx200 
 */
static void 
lx200_timeout_handler (client, id)
XtPointer client;
XtIntervalId *id;
{
	/* printf("timeout\n"); */
	if (svc_ismanaged() && lx200Connect) {
		get_coord_cb (NULL,NULL,NULL); 
	     	}
	else 	{
		ignore_alarm();
	    	if (lx200Connect) DisconnectLX200();
		lx200Connect = 0;
		}
}

/*
 * Set an alarm to call the lx200_timeout_handler
 */
static void 
set_alarm()
{
	lx200_interval_id = XtAppAddTimeOut (xe_app, lx200_interval, 
					     lx200_timeout_handler, 0);
}

/*
 * Ignore alarm
 */
static sigset_t oset, nset;
static void 
ignore_alarm()
{
	if (lx200_interval_id != 0) {
		XtRemoveTimeOut (lx200_interval_id);
		lx200_interval_id = 0;
	}
}

/*
 * Restore alarm
 */
static void 
restore_alarm()
{
	if (svc_ismanaged() && lx200Connect) 
		lx200_interval_id = XtAppAddTimeOut (xe_app, lx200_interval, 
					     lx200_timeout_handler, 0);
	else ignore_alarm();
}

/* callback from the Control Interval Help button
 */
static void
interval_help_cb ()
{
  static char *msg[] = {
    "This control dialog allows you to change the interval or frecuency used",
    "to Get Coordinates from LX200. Unit is in miliseconds.",
    "",
    "The default value of 2000 miliseconds can be changed in the XEphem file:",
    "with the entry: XEphem.Interval"
    };
  hlp_dialog ("LX200_INTERVAL", msg, XtNumber(msg));
}

/* callback from the Control ETC Help button
 */
static void
etc_help_cb ()
{
  static char *msg[] = {
    "The Exposure Time Calculator (ETC) calculates the exposure time for any",
    "magnitude based in the exposure time of a referenced magnitude."
    };
  hlp_dialog ("LX200_ETC", msg, XtNumber(msg));
}

/* callback from the Data table Help button
 */
static void
/* ARGSUSED */
lxh_help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
  static char *msg[] = {
  "The LX200 Telescope Control Panel (TCP) allows the control with XEphem of ",
  "any Meade/LX200 telescope connected to the computer through a serial port.",
  "The TCP was created by C. Guirao (cguirao@eso.org) on top of a C library",
  "originally created by Ken Shouse (ken@kshouse.engine.swri.edu).",
  };
  hlp_dialog ("LX200", msg, XtNumber(msg));
}

/* callback from the communication button
 */
static void
/* ARGSUSED */
commu_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
}

/* callback from any Select LX200 port buttons
 */
static void
/* ARGSUSED */
set_port_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        port_lx200 = (int) client;
	/* printf("set_port_cb, port=%d\n",port_lx200); */
	DisconnectLX200();
	lx200Connect = 0;
}

/* called to reinstate a SvHistory */
void
lxh_goto (hp)
Lx200History *hp;
{
        char out_ra[64];
        char out_dec[64];

        objx.f_RA = hp->lx_RA;
        objx.f_dec = hp->lx_dec;
        objx.f_mag = hp->lx_mag;
        objx.f_class = hp->lx_class;
	objx.f_spect[0] = hp->lx_spect[0] ;
	objx.f_spect[1] = hp->lx_spect[1];
	strcpy(objx.o_name,hp->lx_name);

        fs_ra(out_ra,hp->lx_offRA);
        fs_prdec(out_dec,hp->lx_offdec);
        set_xmstring (offset_ra_w, XmNlabelString, out_ra);
        set_xmstring (offset_dec_w, XmNlabelString, out_dec);

        f_string (name_w, objx.o_name);
        f_string (type_w, obj_description(&objx));
        f_string (spect_w, objx.f_spect);
        f_ra (ra_w,objx.f_RA);
        f_prdec (dec_w,objx.f_dec);                                              
	if (objx.f_class) {
          f_double (mag_w, "%g", get_mag(&objx));
  	  compute_etc();
	  }
	else {
          f_string (mag_w, "--");
          f_string (exp_w, "--");
	  }
}	


/* called by the history mechanism when it needs to know the current settings.
 */
void
lxh_get (hp)
Lx200History *hp;
{
        hp->lx_RA = objx.f_RA;
        hp->lx_dec = objx.f_dec;
        hp->lx_offRA = objz.f_RA;
        hp->lx_offdec = objz.f_dec;
        hp->lx_mag = objx.f_mag;
        hp->lx_class = objx.f_class;
	hp->lx_spect[0] = objx.f_spect[0];
	hp->lx_spect[1] = objx.f_spect[1];
	strcpy(hp->lx_name,objx.o_name);
}

/* called by the history mechanism when it needs to know the airmass */
float
lxh_get_airmass ()
{
	char *text;
	float airmass;

        get_xmstring (airmass_w, XmNlabelString, &text);
	airmass = atod (text);
        XtFree(text);
	return(airmass);
}

/* called by the history mechanism when it needs to know the exposure time */
float
lxh_get_exp ()
{
	char *text;
	float exposure;

        get_xmstring (exp_w, XmNlabelString, &text);
	exposure = atod (text);
        XtFree(text);
	return(exposure);
}

/* callback from ETC control button (from Control in LX200). */
void
etc_control_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!etc_w)
	    etc_create_w(); 
	if (XtIsManaged(etc_w)) {
	    XtUnmanageChild (etc_w);
	    }
	else 
	    XtManageChild (etc_w);
}

/* callback from Interval control button (from Control in LX200). */
void
interval_control_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (!interval_w)
	    interval_create_w(); 
	if (XtIsManaged(interval_w)) {
	    XtUnmanageChild (interval_w);
	    }
	else 
	    XtManageChild (interval_w);
}

/* create the interval control dialog */
static void
interval_create_w()
{
	Widget w;
	Widget label_w;
	Widget ctlf_w;
	char *xr_interval;
        Field *fp;
	Arg args[20];
	int n;
	int i;

	/* create form */
	n = 0;
	XtSetArg(args[n], XmNdefaultPosition, False); n++;
	XtSetArg(args[n], XmNautoUnmanage, False); n++; 
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	interval_w = XmCreateFormDialog (toplevel_w, "Interval", args, n);
	set_something (interval_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (interval_w, XmNmapCallback, prompt_map_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "Interval Setup"); n++;
	XtSetValues (XtParent(interval_w), args, n);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        label_w = XmCreateForm (interval_w, "LabelE", args, n);
        XtManageChild (label_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 10); n++;
	    w = XmCreateLabel (label_w, "EINTERVAL", args, n);
	    set_xmstring (w, XmNlabelString, "Interval in miliseconds:");
            XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, label_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        ctlf_w = XmCreateForm (interval_w, "GetI", args, n);
        XtManageChild (ctlf_w);


            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 10); n++;
            xr_interval = getXRes ("Interval",DEF_INTERVAL);
            lx200_interval = atoi (xr_interval);
            get_interval_w = XmCreatePushButton (ctlf_w, xr_interval, args, n);

	    fp = &fixed_fields[11];
            fp->pb_w = get_interval_w;
            XtAddCallback (get_interval_w, XmNactivateCallback, obj_change_cb,
                                                       (XtPointer)fp);
            wtip (get_interval_w, fp->tip);
            XtManageChild (get_interval_w);

	/* separator */
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, ctlf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (interval_w, "ISep", args, n);
	XtManageChild (w);

        /* make a form in a frame to hold the bottom control buttons */
        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        ctlf_w = XmCreateForm (interval_w, "INTERVALF", args, n);
        XtManageChild (ctlf_w);

            for (i = 0; i < XtNumber(intervalbtns); i++) {
                n = 0;
                XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
                XtSetArg (args[n], XmNtopOffset, 5); n++; 
                XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
                XtSetArg (args[n], XmNbottomOffset, 5); n++; 
                XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNleftPosition, 0 + i*5); n++;
                XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNrightPosition, 5 + i*5); n++;
                w = XmCreatePushButton (ctlf_w, intervalbtns[i].name, args, n);
                XtAddCallback (w, XmNactivateCallback, obj_ctl_cb,
                                                 (XtPointer)intervalbtns[i].id);
                if (intervalbtns[i].tip)
                    wtip (w, intervalbtns[i].tip);
                XtManageChild (w);
            }
	compute_interval();
}

static void
compute_interval()
{
	char *text;

	ignore_alarm();
	if (!interval_w) 
            text = getXRes ("Interval",DEF_INTERVAL);
	else
	    get_xmstring (get_interval_w, XmNlabelString, &text);
        lx200_interval = atoi (text);
	restore_alarm();
}

/* create the ETC control dialog */
static void
etc_create_w()
{
	Widget w;
	Widget label_w;
	Widget ctlf_w;
	char *xr_mag, *xr_exp;
        Field *fp;
	Arg args[20];
	int n;
	int i;

	/* create form */
	n = 0;
	XtSetArg(args[n], XmNdefaultPosition, False); n++;
	XtSetArg(args[n], XmNautoUnmanage, False); n++; 
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	etc_w = XmCreateFormDialog (toplevel_w, "Etc", args, n);
	set_something (etc_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (etc_w, XmNmapCallback, prompt_map_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "Exposure Time Calculator"); n++;
	XtSetValues (XtParent(etc_w), args, n);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
/*        XtSetArg (args[n], XmNfractionBase, 15); n++; */
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        label_w = XmCreateForm (etc_w, "LabelE", args, n);
        XtManageChild (label_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
	    w = XmCreateLabel (label_w, "EMAG", args, n);
	    set_xmstring (w, XmNlabelString, "Magnitude");
            XtManageChild (w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 10); n++;
	    w = XmCreateLabel (label_w, "EEXP", args, n);
	    set_xmstring (w, XmNlabelString, "Exposure");
            XtManageChild (w);

        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, label_w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        ctlf_w = XmCreateForm (etc_w, "GetE", args, n);
        XtManageChild (ctlf_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 0); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 5); n++;
	    xr_mag  = getXRes ("Magnitude",DEF_ETC_MAG);
            etc_mag_w = XmCreatePushButton (ctlf_w, xr_mag, args, n);

	    fp = &fixed_fields[6];
            fp->pb_w = etc_mag_w;
            XtAddCallback (etc_mag_w, XmNactivateCallback, obj_change_cb,
                                                       (XtPointer)fp);
            wtip (etc_mag_w, "Set magnitude for calibration star");
            XtManageChild (etc_mag_w);

            n = 0;
            XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
            XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNleftPosition, 5); n++;
            XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
            XtSetArg (args[n], XmNrightPosition, 10); n++;
	    xr_exp  = getXRes ("Exposure",DEF_ETC_EXP);
            etc_exp_w = XmCreatePushButton (ctlf_w, xr_exp, args, n);
	    fp = &fixed_fields[8];
            fp->pb_w = etc_exp_w;
            XtAddCallback (etc_exp_w, XmNactivateCallback, obj_change_cb,
                                                       (XtPointer)fp);
            wtip (etc_exp_w, "Set exposure time in seconds");
            XtManageChild (etc_exp_w);

	/* separator */
	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, ctlf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (etc_w, "Sep", args, n);
	XtManageChild (w);

        /* make a form in a frame to hold the bottom control buttons */
        n = 0;
        XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
        XtSetArg (args[n], XmNtopWidget, w); n++;
        XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
        XtSetArg (args[n], XmNfractionBase, 10); n++;
        ctlf_w = XmCreateForm (etc_w, "ETCF", args, n);
        XtManageChild (ctlf_w);

            for (i = 0; i < XtNumber(etcbtns); i++) {
                n = 0;
                XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
                XtSetArg (args[n], XmNtopOffset, 5); n++; 
                XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
                XtSetArg (args[n], XmNbottomOffset, 5); n++; 
                XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNleftPosition, 0 + i*5); n++;
                XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
                XtSetArg (args[n], XmNrightPosition, 5 + i*5); n++;
                w = XmCreatePushButton (ctlf_w, etcbtns[i].name, args, n);
                XtAddCallback (w, XmNactivateCallback, obj_ctl_cb,
                                                    (XtPointer)etcbtns[i].id);
                if (etcbtns[i].tip)
                    wtip (w, etcbtns[i].tip);
                XtManageChild (w);
            }
	compute_etc();
}

static void
compute_etc()
{
	double ha, lst;
        float airm, mairm;
        Now *np;

	char *text;
	float in;
	short magnitude;	
	float ref_exposure, exposure;

	get_xmstring (etc_mag_w, XmNlabelString, &text);
        in = atod (text);
	magnitude = floor((in)*MAGSCALE + 0.5);
	XtFree(text);

	get_xmstring (etc_exp_w, XmNlabelString, &text);
        ref_exposure = atod (text);
	XtFree(text);

	/* Update exposure in current object */
	exposure = ref_exposure * pow (2.5,(double)(objx.f_mag - magnitude)/100.0);
	f_double (exp_w, "%5.0f", exposure);

        /* work with a copy of Now */
        np = mm_get_now();
        now_lst(np,&lst);
	ha = (lst-radhr(objx.f_RA))*15.0;

        /* Calculater airmass & median airmasss in current object */
	nairmass(ha,(double)raddeg(objx.f_dec),raddeg(lat),0.,&airm);
        f_double (airmass_w, "%1.3f", airm);
	nairmass(ha,(double)raddeg(objx.f_dec),raddeg(lat),exposure,&mairm);
        f_double (mairmass_w, "%1.3f", mairm);
/*
	printf("lst=%lf RA=%f\n",lst,radhr(objx.f_RA));
	printf("DEC=%f\n",raddeg(objx.f_dec));
	printf("slew_cb: ha=%lf lat=%lf\n",ha,raddeg(lat));
	printf("airmass=%f\n",airm);
	printf("median airmass=%f\n",mairm);
*/
}
