#!/usr/bin/env python
# PURPOSE:	pet.py: (P)ython (e)ngine for (t)rendPlotter
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- June 2012
#		1.0.1	-- bug fixes with get_stat: data clipping and FIXED average,
#			   thresholds after clipping, thresh1 == thresh2, 
#			   negative average and PERcent thresholds;
#			   bug fix in write_stat in case of empty data set;
#			   enable plot symbol 21, enforce no line for opslog data,
#			   add option -v, reading of date file safer (2012-07-09)
#		1.1	-- new optional configuration SFORMAT in PLOT_NAME (2012-10-01)
#		1.1.1	-- left-over debug output removed (2012-12-05)
#		1.1.2	-- check that PLOT_NAME definitions in delta config are consistent with main config (2013-03-07)
#		1.1.3	-- better date marking for FULL plots that have long time range (2013-05-13)
#		1.1.4   -- updated python path for SL63 (2014-03-11)
#		1.2	-- suppress FIT_RULE in close-up plot if configured;
#			   usage of argparse module, remove usage of has_key() method (2014-06-04)
#		1.3	-- opslog support de-commissioned (2014-10-31)
#		1.4	-- supports KPI plots (2018-02-12)
#		1.4.1	-- usage of astropy.io instead of asciidata (2018-03-09)
# ---------------------------------------------------------------------------------------------------------------------
#		2.0	-- usage of Highcharts for close-up plots (2019-03-25)
#		2.0.1	-- do not plot EVENTs that are outside MJD range of plot (2019-03-28)
#		2.0.2	-- suppress PNG for close-up plot if INTERACTIVE!=STATIC (2019-05-15)
#		2.0.3	-- bug fix with avgmethod in write_stat (2019-05-17)
#
# ARGUMENTS:	<REPORT_NAME>
# OPTIONS:	-v, --version  			show program's version number and exit
#		-h, --help  			show help message and exit
#		--closeup=CLOSEUP		closeup plot [optional]; default=0 [main plot]
#		--trending_type=TRENDING_TYPE	either 'HEALTH' or 'HISTORY' [optional]; default='HEALTH'
#		--start=YMSTART			<YEAR>_<MONTH> of start in case of HISTORY [optional]
#		--date_descr=DATEDESCR		date description text [optional]
#		--date_range=DATERANGE		date range text [optional]
#		--mjd_min=MJD_MIN		minimum MJD-OBS for MJD axis [optional]; default=0
#		--mjd_max=MJD_MAX		maximum MJD-OBS for MJD axis [optional]; default=0
#		--time_stamp=MJD_TIMESTAMP	time stamp for marking current MJD [optional]
#		--tool_config=TOOL_CONFIG	non-standard tool configuration [optional]
#		--tp_filename=TP_FILENAME	file name of calling trendPlotter script [optional]; default=trendPlotter
#
# COMMENTS:	pet.py is called from trendPlotter v4.x to create plots.
# =====================================================================================

TOOL_VERSION="2.0.3"
_version_ = "2.0.3"

# =====================================================================================
# 0. initialization
# 0.1 import modules
# =====================================================================================

import sys				# interaction with Python interpreter
import os				# operating system services
import logging				# output of logging messages
import math				# mathematical functions
import numpy				# methods for numerical arrays
from astropy.io import ascii		# read/write ASCII data
import time				# to get plot date
import pylab				# matplotlib interface
import argparse				# for parsing command line arguments and options

# =====================================================================================
# 0.2 constants
# =====================================================================================

# DFO environment variables
dfo_instrument = os.environ.get('DFO_INSTRUMENT')
dfo_bin_dir = os.environ.get('DFO_BIN_DIR')
dfo_config_dir = os.environ.get('DFO_CONFIG_DIR')

# default trendPlotter config dir
config_dir = dfo_config_dir + '/trendPlotter'

# possible report types
report_types = ('REPORT1', 'REPORT2', 'REPORT4', 'REPORT6', 'REPORT8', 'REPORT10', 'CUSTOM') 

# possible data sources
# v1.3: OPSLOG not supported any longer
data_sources = ('QC1DB', 'LOCAL')

# recreate trendPlotter original figure size: 782 x 542 px
fig_dpi = 100		# 100 dpi
fig_size = (7.82, 5.42)	# 7.82 x 5.42 in
			# x: 7.82 * 100 = 782
			# y: 5.42 * 100 = 542

# conversion scale from MIDAS (in mm) to axes coordinates (relative plot coordinates)
xscale = 0.00525
yscale = 0.00752

# convert MIDAS colour codes into Matplotlib codes
col_mid_to_py = {'0': 'w', '1': 'k', '2': 'r', '3': 'g', '4': 'b', '5': 'y', '6': 'm', '7': 'c', '8': 'w'}

# convert MIDAS colour codes into Highcharts codes
col_mid_to_hc = {'0': 'white', '1': 'black', '2': 'red', '3': 'green', '4': 'blue', '5': '#cccc00', '6': 'magenta', '7': 'cyan', '8': 'white'}

# Highcharts color codes for outliers
col_to_out = {'white': 'black', 'black': 'red', 'red': 'blue', 'green': 'red', 'blue': 'red', '#cccc00': 'red', 'magenta': 'red', 'cyan': 'red'}

# Highcharts symbol names for outliers
sym_to_out = {'circle': 'triangle', 'square': 'triangle', 'diamond': 'triangle', 'triangle': 'triangle-down', 'triangle-down': 'triangle'}
# Highcharts color codes for last data point
col_to_last = {'white': 'black', 'black': 'cyan', 'red': 'green', 'green': 'red', 'blue': 'red', '#cccc00': 'red', 'magenta': 'cyan', 'cyan': 'blue'}

# convert MIDAS plot symbols into Matplotlib symbols
sym_mid_to_py = {'0': ' ', '1': '.', '2': 'h', '3': 's', '4': '^', '5': '+', '6': 'x', '7': '*', '8': '*',
		'9': 's', '10': 's', '11': 'd', '12': '_', '13': '|', '14': '>', '15': '^', '16': '<',
		'17': 'v', '18': 'o', '19': 's', '20': '^', '21': 'd'}

# convert MIDAS plot symbols into Highcharts symbols
# only good for: 17 (triangle down), 18 (circle), 19 (square), 20 (triangle), 21 (diamond)
sym_mid_to_hc = {'0': 'circle', '1': 'circle', '2': 'circle', '3': 'square', '4': 'triangle', '5': 'square', '6': 'diamond', '7': 'circle', '8': 'circle',
		'9': 'square', '10': 'square', '11': 'diamond', '12': 'circle', '13': 'circle', '14': 'triangle', '15': 'triangle', '16': 'triangle-down',
		'17': 'triangle-down', '18': 'circle', '19': 'square', '20': 'triangle', '21': 'diamond'}

# Matplotlib marker edge width for MIDAS symbols
marker_edge_width = {'0': 1, '1': 1, '2': 1, '3': 1, '4': 1, '5': 1, '6': 1, '7': 0, '8': 1,
		'9': 1, '10': 1, '11': 1, '12': 1, '13': 1, '14': 1, '15': 1, '16': 1,
		'17': 1, '18': 0, '19': 0, '20': 0, '21': 0}

# Highcharts: open symbols
open_sym_hc = {'0': True, '1': True, '2': True, '3': True, '4': True, '5': True, '6': True, '7': False, '8': True,
		'9': True, '10': True, '11': True, '12': True, '13': True, '14': True, '15': True, '16': True,
		'17': True, '18': False, '19': False, '20': False, '21': False}

# convert MIDAS colour codes into HTML code
col_mid_to_html = {'0': '#FFFFFF', '1': '#000000', '2': '#FF3333', '3': '#339933', '4': '#3333FF', '5': '#CCCC33',
		'6': '#DD00DD', '7': '#33CCCC', '8': '#FFFFFF'}
# values in trendPlotter V2.7.x:
#col_mid_to_html = {'0': '#FFFFFF', '1': '#000000', '2': '#FF3333', '3': '#33FF33', '4': '#3333FF', '5': '#FFFF33',
#		'6': '#FF33FF', '7': '#33FFFF', '8': '#FFFFFF'}

# convert MIDAS plot symbols into HTML code
sym_mid_to_html = {'1': '.', '2': 'o', '3': '&#9633;', '4': '&Delta;', '5': '+', '6': 'x', '7': '&lowast;', 
		'8': '<font size=2>*</font>', '9': '<font size=1>[+ square]</font>', '10': '<font size=1>[x square]</font>',
		'11': '&loz;', '12': '&ndash;', '13': '|', '14': '&rarr;', '15': '&uarr;', '16': '&larr;', '17': '&darr;',
		'18': '<font size=4>&#149;</font>', '19': '&#9632;', '20': '&#9650;', '21': '<font size=4>&#9830;</font>'}

# =====================================================================================
# 0.3 definitions, classes, and functions from qclib.py
# =====================================================================================

# change some default plotting parameters
from matplotlib import rcParams
rcParams['font.size'] = 8
rcParams['lines.linewidth'] = 0.75
rcParams['axes.linewidth'] = 0.75
rcParams['legend.borderpad'] = 0.3
rcParams['legend.borderaxespad'] = 0.3
rcParams['legend.handletextpad'] = 0.4
rcParams['legend.labelspacing'] = 0.0
rcParams['patch.linewidth'] = 0.75

def set_logging(level='info', format=''):
	"set logging level and output format"

	if format == '':
		logFMT = '%(module)15s [%(levelname)7s] %(message)s'
	else:
		logFMT = format
	loglevel = {'debug': logging.DEBUG, 'info': logging.INFO, 'warning': logging.WARNING, 'error': logging.ERROR}
	logging.basicConfig(level=loglevel.get(level.lower(), logging.INFO), format=logFMT, stream=sys.stdout)

class ConfigFile:
	"""representation of ASCII configuration file
	format: <TAG> <VALUE1> <VALUE2> ... """

	def __init__(self, file_name='', mult_tags=[]):
		conf_file = open(file_name, 'r')
		all_the_file=conf_file.read().splitlines()
		conf_file.close()

		self.content = {}
		for line in all_the_file:
			words = line.split()
			if len(words) == 0:
				continue
			if words[0][0] == '#':
				continue
			tag = words[0]
			if not tag in mult_tags:
				# only first value is taken
				if len(words) == 1:
					self.content[tag] = ''
				else:
					if words[1] == '#':
						self.content[tag] = ''
					else:
						self.content[tag] = os.path.expandvars(words[1])
			else:
				if not tag in self.content:
					self.content[tag] = []
				sublist = []
				for word in words[1:]:
					if word[0] == '#':
						continue
					sublist.append(os.path.expandvars(word))
				self.content[tag].append(sublist)

def arr_median(arr):
	"get median of a numpy array"

	flatarr = arr.ravel()
	indarr = flatarr.argsort()
	index = indarr.shape[0] // 2
	if indarr.shape[0] % 2 != 0:
		return flatarr[indarr[index]]
	else:
		return (flatarr[indarr[index]] + flatarr[indarr[index-1]]) / 2.

# =====================================================================================
# 0.4 specific class and function definitions
# =====================================================================================

class tp_options:
	"definition of command line options"

	def __init__(self, vers='0.0'):
		self.parser = argparse.ArgumentParser(description='pet.py is called from trendPlotter v3.x to create plots')
		#self.parser.add_argument('-v', '--version', action='version', version='%(prog)s ' + vers)
		#self.parser.add_argument('-v', '--version', action='version', version=vers)
		self.parser.add_argument('-v', '--version', dest='version', action='store_true', default=False,
				help='print version number and exit')
		self.parser.add_argument('report', nargs='?', metavar='REPORT_NAME', 
				help='name of report (Health Check plot)')
		self.parser.add_argument('--closeup', dest='closeup', metavar='CLOSEUP', type=int, default=0,
				help='closeup plot [optional]; default=0 [main plot]')
		self.parser.add_argument('--trending_type', dest='type', metavar='TRENDING_TYPE', default='HEALTH',
				help='trending type, either "HEALTH" or "HISTORY" [optional]; default="HEALTH"')
		self.parser.add_argument('--date_descr', dest='date_descr', metavar='DATEDESCR', default='',
				help='date description text [optional]')
		self.parser.add_argument('--date_range', dest='date_range', metavar='DATERANGE', 
				default='Test run: data range not specified',
				help='date range text [optional]')
		self.parser.add_argument('--mjd_min', dest='mjd_min', metavar='MJD_MIN', default=0,
				help='minimum MJD-OBS for MJD axis [optional]; default=0')
		self.parser.add_argument('--mjd_max', dest='mjd_max', metavar='MJD_MAX', default=0,
				help='maximum MJD-OBS for MJD axis [optional]; default=0')
		self.parser.add_argument('--tp_filename', dest='tp_filename', metavar='TP_FILENAME', default='trendPlotter',
				help='file name of calling trendPlotter script [optional]; default=trendPlotter')
		self.parser.add_argument('--tool_config', dest='tool_config', metavar='TOOL_CONFIG', default='',
				help='non-standard tool configuration [optional]')
		self.parser.add_argument('--start', dest='ymstart', metavar='YMSTART', default='HEALTH',
				help='<YEAR>_<MONTH> of start in case of HISTORY [optional]')
		self.parser.add_argument('--time_stamp', dest='time_stamp', metavar='MJD_TIMESTAMP', default=0.0,
				help='time stamp for marking current MJD [optional]')

	def get(self):
		options = self.parser.parse_args()
		return options

class tp_config:
	"""representation of trendPlotter configuration
	format: <TAG> <VALUE1> <VALUE2> ... 
	this is a dedicated implementation that scans the configuration file only for one grep tag"""

	def __init__(self, file_name='', grep_tag='#REPORT_TYPE'):
		conf_file = open(file_name, 'r')
		all_the_file=conf_file.read().splitlines()
		conf_file.close()

		self.content = {}
		for line in all_the_file:
			words = line.split()
			if len(words) == 0:
				continue
			if len(words) == 1:
				continue
			if words[0] == grep_tag:
				tag = words[1]
				if not tag in self.content:
					self.content[tag] = []
				sublist = []
				for word in words[2:]:
					if word[0] == '#':
						continue
					sublist.append(word)
				self.content[tag].append(sublist)

def list_to_str(list):
	"merge a list of character strings into a single string"

	merged = ''
	for word in list:
		merged = merged + word + ' '
	#for word in list:
	#	merged = merged + word + ' '
	#	if word[-2:] == '&&':
	#		break
	merged = merged.replace('&&', '')
	if len(merged) > 0:
		return merged[0:-1]
	else:
		return merged

def get_stat(data, avg_meth, thresh_def, sformat):
	"do statistics on data set"

	thresh1, thresh2 = 0, 0
	tot_num = len(data)
	if len(data) > 0:
		if avg_meth == 'MEAN':
			avg = data.mean()
		elif avg_meth == 'MEDIAN':
			avg = arr_median(data)
		elif avg_meth[0:5] == 'FIXED':
			strval = avg_meth.replace('FIXED=', '')
			avg = float(strval)
		else:
			avg = -999
	else:
		avg = -999
	
	if thresh_def[0:3] == 'PER' and avg_meth in ['MEAN', 'MEDIAN'] and tot_num > 0:
		strval = thresh_def.replace('PER=', '')
		per_thresh = float(strval)
		# we relax boundaries by 1% in order to avoid running into rounding problems in case of small tot_num
		if avg >= 0.0:
			thresh1 = 0.99*avg*(1.0-per_thresh/100.0)
			thresh2 = 1.01*avg*(1.0+per_thresh/100.0)
		else:
			thresh1 = 0.99*avg*(1.0+per_thresh/100.0)
			thresh2 = 1.01*avg*(1.0-per_thresh/100.0)
		# clip data
		data1 = data[data>=thresh1]
		data2 = data1[data1<=thresh2]
		if len(data2) > 0:
			if avg_meth == 'MEAN':
				avg = data2.mean()
			else:
				avg = arr_median(data2)
		# determine thresholds again after clipping (new with v1.0.1: without relaxing)
		if avg >= 0.0:
			thresh1 = avg*(1.0-per_thresh/100.0)
			thresh2 = avg*(1.0+per_thresh/100.0)
		else:
			thresh1 = avg*(1.0+per_thresh/100.0)
			thresh2 = avg*(1.0-per_thresh/100.0)
		#thresh1 = 0.99*avg*(1.0-per_thresh/100.0)
		#thresh2 = 1.01*avg*(1.0+per_thresh/100.0)
		#thval = '&plusmn; ' + str((thresh2-thresh1)/2.0)
		fmt_strg = '&plusmn; '+sformat
		thval = fmt_strg % ((thresh2-thresh1)/2.0)
	elif thresh_def[0:3] == 'OFF' and avg_meth in ['MEAN', 'MEDIAN'] and tot_num > 0:
		strval = thresh_def.replace('OFF=', '')
		off_thresh = float(strval)
		thval = '&plusmn; ' + strval
		# we relax boundaries by 1% in order to avoid running into rounding problems in case of small tot_num
		thresh1 = 0.99*avg-off_thresh
		thresh2 = 1.01*avg+off_thresh
		# clip data
		data1 = data[data>=thresh1]
		data2 = data1[data1<=thresh2]
		if len(data2) > 0:
			if avg_meth == 'MEAN':
				avg = data2.mean()
			else:
				avg = arr_median(data2)
		# determine thresholds again after clipping (important: without relaxing)
		thresh1 = avg-off_thresh
		thresh2 = avg+off_thresh
	elif thresh_def[0:3] == 'VAL':
		strval = thresh_def.replace('VAL=', '')
		exec 'thresh1, thresh2 = ' + strval
		thval = strval
	else:
		thresh1, thresh2 = 0, 0
		thval = '&nbsp;'
	
	return avg, [thresh1, thresh2], thval, tot_num
	#return avg, (thresh1, thresh2), thval, tot_num

def write_stat(parset, p_index, plt_name, average, thresholds, thval, tot_num, sformat):
	"""write statistics results into text file for scoreQC
	and into HTML code for later usage"""

	# get y parameter name in QC1 database for later usage
	if parset[1] == 'QC1DB':
		yparam = parset[4]
	else:
		local_name = parset[4]
		for dataset in plot_config['LOCAL_DATASET']:
			if dataset[0] == local_name:
				yparam = dataset[3]
				break
		else:
			yparam = ''

	# averaging method
	if symbols[parset[0]][4] == 'NONE':
		avgmethod = 'none'
	else:
		#avgmethod = symbols[parset[0]][4].rstrip('=.0123456789')
		avgmethod = symbols[parset[0]][4].rstrip('=-.0123456789')

	# edit method names for output
	if symbols[parset[0]][5][0:3] == 'PER' and avgmethod in ['MEAN', 'MEDIAN']:
		thmethod = symbols[parset[0]][5].replace('=', '') + '%'
		thlabel = symbols[parset[0]][5].replace('PER=', '') + '%'
	elif symbols[parset[0]][5][0:3] == 'OFF' and avgmethod in ['MEAN', 'MEDIAN']:
		thmethod = 'OFF'
		thlabel = thmethod
	elif symbols[parset[0]][5][0:3] == 'VAL':
		thmethod = 'VAL'
		thlabel = '<a title=fixed_thresholds>%s</a>' % thmethod
	else:
		thmethod = 'none'
		thlabel = 'none'

	stat_text_filename = 'stat_text'
	stat_text_file = open(stat_text_filename, 'a')
	if average == -999 and thval == '&nbsp;':
		stat_text_file.write('%i %s %s %s %s %s %s %s %i %s\n' % 
			(p_index, plt_name, parset[1], avgmethod, '-999.0', 'none', 'none', 'none', tot_num, yparam))
	elif average != -999 and thval == '&nbsp;':
		fmt_strg = '%i %s %s %s '+sformat+' %s %s %s %i %s\n'
		stat_text_file.write(fmt_strg % 
			(p_index, plt_name, parset[1], avgmethod, average, 'none', 'none', 'none', tot_num, yparam))
	elif average == -999 and thval != '&nbsp;':
		stat_text_file.write('%i %s %s %s %s %s %s %s %i %s\n' % 
			(p_index, plt_name, parset[1], avgmethod, '-999.0', thmethod, thresholds[0], thresholds[1], tot_num, yparam))
	else:
		fmt_strg = '%i %s %s %s '+sformat+' %s %s %s %i %s\n' 
		stat_text_file.write(fmt_strg %
			(p_index, plt_name, parset[1], avgmethod, average, thmethod, thresholds[0], thresholds[1], tot_num, yparam))
	stat_text_file.close()
	
	html_file = open('stat_results', 'a')
	html_file.write('<tr bgcolor="#FFFFFF" valign=top align=left>\n')
	html_file.write('<td align=center><font size=1><b>%i</b></td>\n' % p_index)
	html_file.write('<td align=center><font color=%s>\n' % col_mid_to_html[symbols[parset[0]][2]])
	html_file.write('%s</td><td><font size=1>%s</td>\n' % (sym_mid_to_html[symbols[parset[0]][0]], parset[1]))
	html_file.write('<td><font size=1>%s</td>\n' % avgmethod)
	if avgmethod == 'none':
		html_file.write('<td align=right><font size=1><b>&nbsp;</b></td>\n')
	elif average == -999:
		html_file.write('<td align=right><font size=1><b>-999.0</b></td>\n')
	else:
		fmt_strg = '<td align=right><font size=1><b>'+sformat+'</b></td>\n'
		html_file.write(fmt_strg % average)
	if len(symbols[parset[0]]) > 7:
		units_orig = symbols[parset[0]][7]
		units = units_orig.replace('SPACE', ' ')
	else:
		units_orig = '&nbsp;'
		units = '&nbsp;'
	html_file.write('<td><font size=1>%s</td>\n' % units)
	html_file.write('<td><font size=1>%s</td><td nowrap align=right><font size=1><b>%s</td>\n' % (thlabel, thval))
	html_file.write('<td align=right><font size=1>%i</td>\n' % tot_num)
	html_file.write('<td><font size=1>%s</td>\n' % yparam)
	# append links for downloading 
	if os.path.exists('downloads_'+parset[0]):
		download_file = open('downloads_'+parset[0], 'r')
		download_txt = download_file.readlines()
		download_file.close()
		html_file.writelines(download_txt)
	# find remark (description) for parset
	if len(parset) > 7 and 'REMARK' in plot_config:
		remark_tag = parset[7]
		for line in plot_config['REMARK']:
			if line[0] == remark_tag:
				remark = list_to_str(line[1:])
				break
		else:
			remark = '&nbsp;'
	else:
		remark = '&nbsp;'
	html_file.write('<td><font size=1>%s</td></tr>\n' % remark)
	html_file.close()
	
	# output into list_sym_<parset>, this is used for tool tips
	sym_file = open('list_sym_'+parset[0], 'w')
	sym_file.write('SYMBOLS %s %s %s %s\n' % (parset[0], sym_mid_to_html[symbols[parset[0]][0]].replace(' ', 'SPACE'), 
		col_mid_to_html[symbols[parset[0]][2]], units_orig) )
	sym_file.close()

	return

# =====================================================================================
# 0.5 class and function definitions for using Highcharts
# =====================================================================================

class hc_plot:
	"""defining a HighCharts plots, write to HTML file"""

	def __init__(self, html_name):
		"set default values for plot"

		self.html_name = html_name
		self.title = ''
		self.subtitle = ''
		self.xname, self.yname = '', ''
		self.xdata, self.ydata = [], []
		self.outliers = []
		self.thresh = []
		self.average = []
		self.aliens = False
		self.mark_last = []
		self.n_last = []
		self.pnames = []
		self.symcol = []
		self.symline = []
		self.symname = []
		self.symopen = []
		self.dates = []
		self.ref_values = []
		self.events = []
		self.lines = []
		self.marker_radius = 4

	def header(self):
		header = ['<!DOCTYPE HTML>\n', '<html>\n', '<head>\n', 
				'<meta http-equiv="Content-Type" content="text/html; charset=utf-8">\n',
				'<meta name="viewport" content="width=device-width, initial-scale=1">\n', 
				'<title>%s</title>\n' % self.title,
				'<style type="text/css">\n',
				'#container {\n',
				'        min-width: 310px;\n',
				'        max-width: 800px;\n',
				'        height: 400px;\n',
				'        margin: 0 auto\n', '}\n', '</style>\n',
				'</head>\n',
				'<body>\n',
				'<script src="http://www.eso.org/observing/dfo/quality/ALL/jscript/highcharts.js"></script>\n',
				'<script src="http://www.eso.org/observing/dfo/quality/ALL/jscript/annotations.js"></script>\n',
				'<script src="http://www.eso.org/observing/dfo/quality/ALL/jscript/exporting.js"></script>\n',
				'<script src="http://www.eso.org/observing/dfo/quality/ALL/jscript/export-data.js"></script>\n',
#				'<script src="https://code.highcharts.com/highcharts.js"></script>\n',
#				'<script src="https://code.highcharts.com/modules/annotations.js"></script>\n',
#				'<script src="https://code.highcharts.com/modules/exporting.js"></script>\n',
#				'<script src="https://code.highcharts.com/modules/export-data.js"></script>\n',
				'<div id="container"></div>\n' ]
		return header

	def footer(self):
		footer = ['</body>\n', '</html>\n']
		return footer

	def plot_basics(self):
		year, month, day, hour, minute, sec = time.gmtime()[0:6]
		today = str('%04i-%02i-%02iT%02i:%02i:%02i' % (year, month, day, hour, minute, sec))
		created_by = 'created on ' + today + ', powered by QC: www.eso.org/HC'

		basics = [	"chart: {plotBorderColor: '#ccd6eb', plotBorderWidth: 1, zoomType: 'xy'},\n",
				"credits: {enabled: true, position: {x: 10, align: 'left'}, text: '%s', href: 'http://www.eso.org/HC'},\n" % created_by,
#				"annotations: [{labels: [{align: 'left', verticalAlign: 'bottom', distance: 0, backgroundColor: 'white', borderColor: 'white', point: {x: -20, y: 335}, text: 'powered by QC: www.eso.org/HC', style: {color: 'gray'}}]},\n{labels: [{align: 'right', verticalAlign: 'bottom', distance: 0, backgroundColor: 'white', borderColor: 'white', point: {x: 600, y: 335}, text: '%s', style: {color: 'gray'}}]}],\n" % created_by,
				"exporting: {buttons: {contextButton: {menuItems: ['printChart', 'separator', 'downloadPNG', 'downloadPDF', 'separator', 'viewData', 'downloadCSV']}}},\n",
				"title: {text: '%s'},\n" % self.title,
				"subtitle: {text: '%s'},\n" % self.subtitle ]
		# definiton of x axis
		plot_lines = "plotLines: ["
		if self.dates != [] or self.events != []:
			for dd in self.dates:
				plot_lines = plot_lines + "{value: %f, color: 'gray', dashStyle: 'Dash', width: 1, label: {text: '%s', rotation: 270, align: 'right', textAlign: 'right', x: -5, y: 2, style: {color: 'gray'}}},\n" % (dd[0], dd[1])
			# events
			lab_ypos = {False: -5, True: -20}
			lab_sel = True
			for event in self.events:
				lab_sel = not lab_sel
				plot_lines = plot_lines + "{value: %f, color: '%s', dashStyle: 'DashDot', width: 1, zIndex: 6, label: {text: '%s', rotation: 0, align: 'left', textAlign: 'left', verticalAlign: 'bottom', y: %i, style: {color: '%s'}}},\n" % (event[0], event[1], event[2], lab_ypos[lab_sel], event[1])
		plot_lines = plot_lines + "]},\n"
		basics.append("xAxis: {min: %f, max: %f, lineColor: '#ccd6eb', title: {text: '%s'},\n" % (self.xmin, self.xmax, self.xname) + plot_lines)

		# definition of y axis
		minmax = "min: %f, max: %f, lineColor: '#ccd6eb', title: {text: '%s'}" % (self.ymin, self.ymax, self.yname)
		if self.thresh == []:
			th_lines = ""
		else:
			if self.thresh[0] == self.ymin and self.thresh[1] == self.ymax:
				th_lines = "{value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Lower threshold: %s', align: 'center', style: {color: 'gray'}}},\n {value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Upper threshold: %s', y:12, align: 'center', style: {color: 'gray'}}},\n" % (self.thresh[0], self.thresh[2], self.thresh[1], self.thresh[3])
			elif self.thresh[0] == self.ymin:
				th_lines = "{value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Lower threshold: %s', align: 'center', style: {color: 'gray'}}},\n {value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Upper threshold: %s', align: 'center', style: {color: 'gray'}}},\n" % (self.thresh[0], self.thresh[2], self.thresh[1], self.thresh[3])
			elif self.thresh[1] == self.ymax:
				th_lines = "{value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Lower threshold: %s', y: 12, align: 'center', style: {color: 'gray'}}},\n {value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Upper threshold: %s', y:12, align: 'center', style: {color: 'gray'}}},\n" % (self.thresh[0], self.thresh[2], self.thresh[1], self.thresh[3])
			else:
				th_lines = "{value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Lower threshold: %s', y: 12, align: 'center', style: {color: 'gray'}}},\n {value: %f, color: 'gray', dashStyle: 'Dot', width:1, label: {text: 'Upper threshold: %s', align: 'center', style: {color: 'gray'}}},\n" % (self.thresh[0], self.thresh[2], self.thresh[1], self.thresh[3])
		
		avg_lines = ""
		if self.average != []:
			for avg in self.average:
				avg_lines = avg_lines + "{value: %f, color: 'black', dashStyle: 'Solid', width: 1, label: {text: '%s: %f', align: 'left', style: {color: 'black'}}},\n" % (avg[0], avg[1], avg[0])
		
		basics.append("yAxis: {" + minmax + ", gridLineWidth: 0, tickWidth: 1,\nplotLines: [" + th_lines + avg_lines + "]},\n")

		basics.append("legend: {layout: 'vertical', align: 'right', verticalAlign: 'middle'},\n")
# TBD: whether to use fixed min/max for y axis
# Check: in case of aliens, do not use fixed min/max for y axis
#				"yAxis: {title: {text: '%s'}},\n" % (self.ymin, self.ymax, self.yname),
#				"legend: {layout: 'vertical', align: 'right', verticalAlign: 'middle'},\n" ]
		return basics

	def plot_options(self):
		options = [	"plotOptions: {\n",
				"series: {label: {connectorAllowed: false}},\n",
				"line: {marker: {enabled: true}}\n",
				"},\n" ]
		return options

	def plot_responsive(self):
		responsive = [	"responsive: {\n",
				"rules: [{condition: {maxWidth: 500},\n",
				"chartOptions: {legend: {layout: 'horizontal', align: 'center', verticalAlign: 'bottom'}}\n",
				"}]}\n" ]
		return responsive

	def series(self, sdata):
		series = [ "series: [\n" ]
		for idx in range(len(self.pnames)):
			if self.symopen[idx]:
				series.append("{name: '%s', type: 'scatter', color: '%s', lineWidth: %i, marker: {symbol: '%s', radius: %i, lineWidth: 1, lineColor: '%s', fillColor: 'white'}, data: [\n" % (self.pnames[idx], self.symcol[idx], self.symline[idx], self.symname[idx], self.marker_radius-1, self.symcol[idx]))
			else:
				series.append("{name: '%s', type: 'scatter', color: '%s', lineWidth: %i, marker: {symbol: '%s', radius: %i}, data: [\n" % (self.pnames[idx], self.symcol[idx], self.symline[idx], self.symname[idx], self.marker_radius))
			if self.mark_last[idx]:
				if len(sdata[idx]) > 0:
					for j in range(len(sdata[idx]) - self.n_last[idx]):
						pair = sdata[idx][j]
						if numpy.in1d(pair[1], self.outliers[idx]):
							series.append("{marker: {symbol: '%s', lineColor: '%s', fillColor: '%s'}, color:'%s', x: %f, y: %f},\n" % (sym_to_out[self.symname[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], pair[0], pair[1]))
						else:
							series.append("[%f, %f],\n" % (pair[0], pair[1]))
					for j in range(len(sdata[idx]) - self.n_last[idx], len(sdata[idx])):
						pair = sdata[idx][j]
						if numpy.in1d(pair[1], self.outliers[idx]):
							series.append("{marker: {symbol: '%s', lineColor: '%s', fillColor: '%s'}, color:'%s', x: %f, y: %f},\n" % (sym_to_out[self.symname[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], pair[0], pair[1]))
						else:
							series.append("{marker: {radius: %i, fillColor: '%s', lineColor: '%s', lineWidth: 2}, x: %f, y: %f},\n" % (self.marker_radius, self.symcol[idx], col_to_last[self.symcol[idx]], pair[0], pair[1]))
			else:
				first_point = True
				for pair in sdata[idx]:
					if numpy.in1d(pair[1], self.outliers[idx]):
						if first_point:
							# stupid trick to avoid failure for GIRAFFE BIASFULL plot 6; unclear why necessary
							series.append("[%f, %f],\n" % (pair[0], pair[1]))
						series.append("{marker: {symbol: '%s', lineColor: '%s', fillColor: '%s'}, color:'%s', x: %f, y: %f},\n" % (sym_to_out[self.symname[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], col_to_out[self.symcol[idx]], pair[0], pair[1]))
					else:
						series.append("[%f, %f],\n" % (pair[0], pair[1]))
					first_point = False
			series.append("]},\n")
		# KPI only: add line(s) for reference value(s)
		if self.ref_values != []:
			for line in self.ref_values:
				val, v_start, v_end = line[0], line[1], line[2]
				series.append("{name: 'KPI', type: 'line', color: 'green', marker: {enabled: false}, showInLegend: false, data: [[%f, %f], [%f, %f]]},\n" % (v_start, val, v_end, val))
		# user-defined lines
		if self.lines != []:
			idx = 0
			for line in self.lines:
				idx = idx + 1
				x1, y1, x2, y2, colour = line[0], line[1], line[2], line[3], line[4]
				series.append("{name: 'LINE%i', type: 'line', color: '%s', dashStyle: 'Dot', width: 1, marker: {enabled: false}, showInLegend: false, data: [[%f, %f], [%f, %f]]},\n" % (idx, colour, x1, y1, x2, y2))
		series.append("],\n")

		return series

	def script(self, content):
		script = ['<script type="text/javascript">\n', "Highcharts.chart('container', {\n"] + content + ['});\n', '</script>\n' ]
		return script

	def add_xdata(self, xarray):
		self.xdata.append(xarray)

	def add_ydata(self, yarray):
		self.ydata.append(yarray)
	
	def add_pname(self, name):
		self.pnames.append(name)
	
	def add_symcol(self, color):
		self.symcol.append(color)

	def add_symline(self, line_yn):
		if line_yn == 'YES':
			self.symline.append(1)
		else:
			self.symline.append(0)

	def add_symname(self, symbol):
		self.symname.append(symbol)

	def add_symopen(self, val):
		self.symopen.append(val)

	def add_outliers(self, yout):
		self.outliers.append(yout)

	def add_mark_last(self, mark, nlast):
		self.mark_last.append(mark)
		self.n_last.append(nlast)
	
	def add_avg(self, val, meth):
		self.average.append((val, meth))
	
	def add_ref_values(self, val, mjd1, mjd2):
		self.ref_values.append((val, mjd1, mjd2))

	def add_events(self, mjd, colour, text, short_text):
		self.events.append((float(mjd), colour, text, short_text))

	def add_lines(self, x1, y1, x2, y2, colour):
		self.lines.append((x1, y1, x2, y2, colour))

	def make_sdata(self):
		sdata = []
		for i in range(len(self.xdata)):
			ser = []
			xs = self.xdata[i]
			ys = self.ydata[i]
			for j in range(len(xs)):
				ser.append([xs[j], ys[j]])
			sdata.append(ser)
		return sdata

	def write(self):
		sdata = self.make_sdata()
		for ser in sdata:
			if len(ser) > 200:
				# smaller marker radius for large data sets
				self.marker_radius = 2
				break
		html_file = open(self.html_name, 'w')
		html_file.writelines(self.header())
		#self.series(sdata)
		#script_content = self.plot_basics() + self.plot_options() + self.series() + self.plot_responsive()
		script_content = self.plot_basics() + self.plot_options()
		script_content = script_content + self.series(sdata) + self.plot_responsive()
		html_file.writelines(self.script(script_content))
		html_file.writelines(self.footer())
		html_file.close()

# =====================================================================================
# 1. main function: create a sub plot
# =====================================================================================

def create_subplot(p_index=1, rect=[], stat_out=True):

	# find all parset definitions for this sub plot
	plt_name = plot_name[p_index][0]
	parset_name = []
	for line in plot_config['PARSET_NAME']:
		if line[0] == plt_name:
			parset_name.append(line[1:])
	
	logging.info('Plot '+str(p_index)+': '+plt_name)

	ax = pylab.axes(rect)
	ax.xaxis.set_major_formatter(pylab.ScalarFormatter(useOffset=False))
	ax.yaxis.set_major_formatter(pylab.ScalarFormatter(useOffset=False))

	# save data limits and statistics of each parset
	xminarr, xmaxarr, yminarr, ymaxarr, statarr = [], [], [], [], []

	# count empty parsets, for having '[no data]' label at right place
	cnt_empty = 0

	# plot label
	if len(plot_name[p_index]) > 7:
		label = plot_name[p_index][7].replace('NONE', '')
	else:
		label = ''
	pylab.figtext(rect[0]+2*xscale, rect[1]+rect[3]-3.5*yscale, label)
	
	# x/y labels
	xlabel = plot_name[p_index][2].replace('SPACE', ' ')
	if xlabel == 'NONE':
		xlabel = ''
	ylabel = plot_name[p_index][4].replace('SPACE', ' ')
	if ylabel == 'NONE':
		ylabel = ''
	
	# tick marks
	xaxis = plot_name[p_index][5]
	yaxis = plot_name[p_index][6]
	exec 'xlimits = (' + xaxis +')'
	exec 'ylimits = (' + yaxis +')'

	if len(ylimits) >= 2:
		ymin, ymax = ylimits[0], ylimits[1]
	else:
		ymin, ymax = 0, 0

	# format of tick mark numbers
	xformat = plot_name[p_index][1]
	yformat = plot_name[p_index][3]

	# check for SCORE_ENABLED
	if len(plot_name[p_index]) > 8:
		if not plot_name[p_index][8] in ['YES', 'NO']:
			logging.warning('NO_MARK_LAST has value "%s", should be "YES" or "NO". scoreHC might give wrong results.' % plot_name[p_index][8])

	# no mark last: overwrite mark_last for this sub plot
	if len(plot_name[p_index]) > 9:
		no_mark_last = plot_name[p_index][9]
	else:
		no_mark_last = 'NO'
	if not no_mark_last in ['YES', 'NO']:
		logging.warning('NO_MARK_LAST has value "%s", should be "YES" or "NO". Forced to "NO".' % no_mark_last)
		no_mark_last = 'NO'
	
	# format for statistics output
	if len(plot_name[p_index]) > 10:
		if plot_name[p_index][10] in ['YES', 'NO', 'STATIC', 'INTER']:
			logging.warning('SFORMAT has value "%s". Forced to "AUTO".' % plot_name[p_index][10])
			plot_name[p_index][10] = 'AUTO'
		if plot_name[p_index][10] == 'AUTO':
			sformat = '%.3g'
		elif plot_name[p_index][10] == 'NONE':
			sformat = '%g'
		else:
			sformat = '%'+plot_name[p_index][10]
	else:
		sformat = '%.3g'

	# loop over parsets
	for parset in parset_name:
		if not parset[1] in data_sources:
			logging.warning('parset %s has source %s. Ignoring ...' % (parset[0], parset[1]))
			continue

		# get x parameter name in QC1 database for later usage
		if parset[1] == 'QC1DB':
			xparam = parset[3]
		else:
			local_name = parset[3]
			for dataset in plot_config['LOCAL_DATASET']:
				if dataset[0] == local_name:
					xparam = dataset[1]
					break
			else:
				xparam = ''

		# get symbol definition
		ycolor = col_mid_to_py[symbols[parset[0]][2]]
		ysymbol = sym_mid_to_py[symbols[parset[0]][0]]
		sym_edge = marker_edge_width[symbols[parset[0]][0]]
		draw_line = symbols[parset[0]][3]
		if draw_line == 'YES':
			ywidth = 1
		else:
			ywidth = 0
		if options.closeup == 0:
			# main plot, symbol size from configuration
			ysize = 8.0 * float(symbols[parset[0]][1])
		else:
			# close-up plot: fixed to 0.8 in trendPlotter for MIDAS
			# now scaled with actual configuration which is scaled to "standard" value of 0.5
			ysize = 8.0 * float(symbols[parset[0]][1]) / 0.5 * 0.8

		# get data
		data_file = 'input_'+parset[0]
		if os.path.exists(data_file) and os.path.getsize(data_file) > 0:
			try:
				parset_data = numpy.loadtxt(data_file)
				if len(parset_data.shape) == 1:
					# only one data point
					xdat, ydat, tdat = numpy.split(parset_data, 3)
				else:
					orig_xdat = parset_data[:,0]
					orig_ydat = parset_data[:,1]
					orig_tdat = parset_data[:,2]
					tsort = orig_tdat.argsort()
					xdat = orig_xdat[tsort]
					ydat = orig_ydat[tsort]
					tdat = orig_tdat[tsort]
			except:
				logging.warning('An error occured while reading parset '+parset[0])
				sys.stderr.write('An error occured while reading parset '+parset[0]+'\n')
				xdat, ydat, tdat = [], [], []
		else:
			logging.info(data_file + ' not found or empty')
			xdat, ydat, tdat = [], [], []

		# apply compute rule
		if len(symbols[parset[0]]) > 8:
			rule_name = symbols[parset[0]][8]
		else:
			rule_name = 'NONE'
		if rule_name != 'NONE' and len(ydat) > 0:
			if 'COMPUTE_RULE' in plot_config:
				for line in plot_config['COMPUTE_RULE']:
					if line[0] == rule_name:
						compute_rule = list_to_str(line[1:])
						break
				else:
					logging.warning('COMPUTE_RULE '+rule_name+' specified but not defined')
					compute_rule = 'pass'
			else:
				logging.warning('COMPUTE_RULE '+rule_name+' specified but not defined')
				compute_rule = 'pass'

			try:
				exec compute_rule
			except:
				logging.warning('An error occured while executing COMPUTE_RULE '+rule_name+'. Please check.')
				sys.stderr.write('An error occured while executing COMPUTE_RULE '+rule_name+'. Please check.\n')

		# get statistics
		average, thresholds, thval, tot_num = get_stat(ydat, symbols[parset[0]][4], symbols[parset[0]][5], sformat)
		statarr.append((average, thresholds[0], thresholds[1], thval))

		# write statistics into file, create html output
		if stat_out:
			write_stat(parset, p_index, plt_name, average, thresholds, thval, tot_num, sformat)

		# empty data set: continue with next one
		if len(ydat) == 0:
			# check for option to turn off "[no data]" label
			if 'MARK_NO_DATA' in plot_config:
				if plot_config['MARK_NO_DATA'] == 'NO':
					continue
			cnt_empty = cnt_empty + 1
			pylab.figtext(rect[0]+2*xscale, rect[1]+rect[3]-3.5*(cnt_empty+1)*yscale, '[no data]', color=ycolor)
			continue

		# store min/max values for plot scaling
		xminarr.append(xdat.min())
		xmaxarr.append(xdat.max())
		yminarr.append(ydat.min())
		ymaxarr.append(ydat.max())

		# NOTE: plot symbols '+' and 'x' appear slightly shifted when plotted on top of 'o'

		# handle outliers
		if symbols[parset[0]][6] == 'YES':
			if thval != '&nbsp;':
				# thresholds defined: plot only data within thresholds
				# NOTE: plot symbols '+' and 'x' appear slightly shifted when plotted on top of 'o'
				xdat1 = xdat[ydat>=thresholds[0]]
				ydat1 = ydat[ydat>=thresholds[0]]
				xdat2 = xdat1[ydat1<=thresholds[1]]
				ydat2 = ydat1[ydat1<=thresholds[1]]
				if ywidth > 0:
					# plot line first, if configured
					pylab.plot(xdat, ydat, color=ycolor, markersize=0, linestyle='-', linewidth=ywidth)
				if sym_edge > 0:
					# plot open symbols
					pylab.plot(xdat2, ydat2, marker=ysymbol, markersize=ysize, markeredgewidth=sym_edge,
							markerfacecolor='w', markeredgecolor=ycolor, linewidth=0)
				else:
					# plot filled symbols (without marker edge)
					pylab.plot(xdat2, ydat2, marker=ysymbol, markersize=ysize, markeredgewidth=0, 
							markerfacecolor=ycolor, markeredgecolor=ycolor, linewidth=0)
		
				# plot outliers
				xdat1 = xdat[ydat<thresholds[0]]
				ydat1 = ydat[ydat<thresholds[0]]
				xdat2 = xdat[ydat>thresholds[1]]
				ydat2 = ydat[ydat>thresholds[1]]
				if ycolor == 'r':
					ocolor = 'b'
				else:
					ocolor = 'r'
				pylab.plot(xdat1, ydat1, color=ocolor, marker='+', markersize=ysize, 
						markeredgewidth=marker_edge_width['5'], linewidth=0)
				pylab.plot(xdat1, ydat1, color=ocolor, marker='x', markersize=ysize, 
						markeredgewidth=marker_edge_width['6'], linewidth=0)
				pylab.plot(xdat2, ydat2, color=ocolor, marker='+', markersize=ysize, 
						markeredgewidth=marker_edge_width['5'], linewidth=0)
				pylab.plot(xdat2, ydat2, color=ocolor, marker='x', markersize=ysize, 
						markeredgewidth=marker_edge_width['6'], linewidth=0)
			else:
				if sym_edge > 0:
					# plot open symbols
					pylab.plot(xdat, ydat, marker=ysymbol, markersize=ysize, markeredgewidth=sym_edge,
							markerfacecolor='w', markeredgecolor=ycolor, linestyle='-', 
							linewidth=ywidth)
				else:
					# plot filled symbols (without marker edge)
					pylab.plot(xdat, ydat, marker=ysymbol, markersize=ysize, markeredgewidth=sym_edge,
							markerfacecolor=ycolor, markeredgecolor=ycolor, 
							linestyle='-', linewidth=ywidth, color=ycolor)
		else:
			if sym_edge > 0:
				# plot open symbols
				pylab.plot(xdat, ydat, marker=ysymbol, markersize=ysize, markeredgewidth=sym_edge,
						markerfacecolor='w', markeredgecolor=ycolor, linestyle='-', 
						linewidth=ywidth, color=ycolor)
			else:
				# plot filled symbols (without marker edge)
				pylab.plot(xdat, ydat, marker=ysymbol, markersize=ysize, markeredgewidth=sym_edge,
						markerfacecolor=ycolor, markeredgecolor=ycolor, 
						linestyle='-', linewidth=ywidth, color=ycolor)

		# mark last data point
		if options.type == 'HEALTH' and mark_last == 'YES' and no_mark_last != 'YES':
			tmax = max(tdat) - 1
			xdat1 = xdat[tdat>tmax]
			ydat1 = ydat[tdat>tmax]
			edgecolor = {'k': 'c', 'r': 'g', 'b': 'r', 'g': 'r', 'm': 'c', 'c': 'b', 'y': 'r'}[ycolor]
			pylab.plot(xdat1, ydat1, marker='o', markersize=1.5*ysize, markerfacecolor=ycolor,
					markeredgewidth=0.5*ysize, markeredgecolor=edgecolor, linewidth=0)

		# mark aliens (data points outside y range of the subplot marked by arrows)
		if mark_aliens == 'YES' and ymin != ymax:
			ydelt = ymax - ymin
			ylength = 0.14 * ydelt * 0.2 / rect[3] # normalized length of arrow
			xdat1 = xdat[ydat>ymax]
			xdat2 = xdat[ydat<ymin]
			for idx in range(len(xdat1)):
				pylab.plot(xdat1[idx], [0.96*ydelt+ymin], marker='^', markersize=5, markeredgecolor='r',
						markerfacecolor='r')
				pylab.plot([xdat1[idx],xdat1[idx]], [0.96*ydelt+ymin-ylength,0.96*ydelt+ymin], marker='.', 
						color='r', markersize=2.5, linewidth=1)
			for idx in range(len(xdat2)):
				pylab.plot(xdat2[idx], [0.04*ydelt+ymin], marker='v', markersize=5, markeredgecolor='r',
						markerfacecolor='r')
				pylab.plot([xdat2[idx],xdat2[idx]], [0.04*ydelt+ymin+ylength,0.04*ydelt+ymin], marker='.', 
						color='r', markersize=2.5, linewidth=1)

		# apply fit rule
		if len(symbols[parset[0]]) > 9:
			if options.closeup == 0:
				# main plot: always apply FIT_RULE
				rule_name = symbols[parset[0]][9]
			else:
				# close-up plot: check if CLOSEUP_FIT == 'NO'
				if len(symbols[parset[0]]) > 11:
					if symbols[parset[0]][11] == 'NO':
						rule_name = 'NONE'
					else:
						rule_name = symbols[parset[0]][9]
				else:
					rule_name = symbols[parset[0]][9]
		else:
			rule_name = 'NONE'
		if rule_name != 'NONE':
			if 'FIT_RULE' in plot_config:
				for line in plot_config['FIT_RULE']:
					if line[0] == rule_name:
						fit_rule = list_to_str(line[1:])
						break
				else:
					logging.warning('FIT_RULE '+rule_name+' specified but not defined')
					fit_rule = 'pass'
			else:
				logging.warning('FIT_RULE '+rule_name+' specified but not defined')
				fit_rule = 'pass'

			try:
				exec fit_rule
			except:
				logging.warning('An error occured while executing FIT_RULE '+rule_name+'. Please check.')
				sys.stderr.write('An error occured while executing FIT_RULE '+rule_name+'. Please check.\n')

	# define tick marks and limits of plot
	if len(xlimits) == 4:
		if xlimits[2] >= xlimits[3] and xlimits[2] > 0 and xlimits[3] > 0:
			majorLocator = pylab.MultipleLocator(xlimits[2])
			minorLocator = pylab.MultipleLocator(xlimits[3])
			ax.xaxis.set_major_locator(majorLocator)
			ax.xaxis.set_minor_locator(minorLocator)
		xmin, xmax = xlimits[0], xlimits[1]
	elif len(xlimits) == 2:
		if xparam == 'mjd_obs' and float(options.mjd_min) < float(options.mjd_max):
			majorLocator = pylab.MultipleLocator(xlimits[0])
			minorLocator = pylab.MultipleLocator(xlimits[1])
			ax.xaxis.set_major_locator(majorLocator)
			ax.xaxis.set_minor_locator(minorLocator)
			xmin = float(options.mjd_min)
			xmax = float(options.mjd_max)
		else:
			xmin, xmax = xlimits[0], xlimits[1]
	else:
		xmin, xmax = min(xminarr), max(xmaxarr)
		xmin = xmin - 0.05 * xdelt
		xmax = xmax + 0.05 * xdelt
	if len(ylimits) == 4:
		if ylimits[2] >= ylimits[3] and ylimits[2] > 0 and ylimits[3] > 0:
			majorLocator = pylab.MultipleLocator(ylimits[2])
			minorLocator = pylab.MultipleLocator(ylimits[3])
			ax.yaxis.set_major_locator(majorLocator)
			ax.yaxis.set_minor_locator(minorLocator)
		ymin, ymax = ylimits[0], ylimits[1]
	elif len(ylimits) == 2:
		ymin, ymax = ylimits[0], ylimits[1]
	else:
		ymin, ymax = min(yminarr), max(ymaxarr)
		ydelt = ymax - ymin
		ymin = ymin - 0.05 * ydelt
		ymax = ymax + 0.05 * ydelt

	# plot average and thresholds
	for stat in statarr:
		if stat[0] != -999:
			pylab.plot([xmin,xmax], [stat[0],stat[0]], 'k-')
		if stat[3] != '&nbsp;':
			pylab.plot([xmin,xmax], [stat[1],stat[1]], 'k:')
			pylab.plot([xmin,xmax], [stat[2],stat[2]], 'k:')
	
	# plot lines 
	if 'LINE' in plot_config:
		for line in plot_config['LINE']:
			if line[0] == plt_name:
				x1 = float(line[1])
				y1 = float(line[2])
				x2 = float(line[3])
				y2 = float(line[4])
				colour = col_mid_to_py[line[5]]
				pylab.plot([x1, x2], [y1, y2], color=colour, linestyle=':')

	# plot events
	if 'EVENT' in plot_config:
		lab_ypos = {False: 0.2, True: 0.3}
		lab_sel = True
		for line in plot_config['EVENT']:
			if line[0] == plt_name:
				mjd = float(line[1])
				if mjd < float(options.mjd_min) or mjd > float(options.mjd_max):
					continue
				lab_sel =  not lab_sel
				colour = col_mid_to_py[line[2]]
				short_text = line[3]
				text = list_to_str(line[4:])
				pylab.plot([mjd, mjd], [ymin, ymax], color=colour, linestyle='-')
				dx = 0.02 * (xmax - xmin)
				#dy = 0.2 * (ymax - ymin)
				dy = lab_ypos[lab_sel] * (ymax - ymin)
				pylab.text(mjd + dx, ymax-dy, short_text, color=colour, horizontalalignment='left', verticalalignment='bottom')

	# plot reference values (KPIs)
	if time_range == 'KPI':
		ref_range = []
		ref_val = []
		if 'REF_VAL' in plot_config:
			for line in plot_config['REF_VAL']:
				if line[0] == plt_name:
					ref_range.append(line[1])
					ref_val.append(line[2])
		if ref_range == [] or ref_val == []:
			logging.warning('No REF_VAL configured for KPI plot.')
			logging.warning('Please check if this is intended.')
		if ref_range != [] and ref_val != []:
			for idx in range(len(ref_range)):
				if ref_range[idx] == 'FULL':
					val = float(ref_val[idx])
					v_start = xmin
					v_end = xmax
					pylab.plot([v_start, v_end], [val, val], color='g', linestyle='-', linewidth=1.5)
				elif ref_range[idx][0] == '>':
					val = float(ref_val[idx])
					v_start = float(ref_range[idx][1:])
					v_end = xmax
					pylab.plot([v_start, v_end], [val, val], color='g', linestyle='-', linewidth=1.5)
				elif ref_range[idx][0] == '<':
					val = float(ref_val[idx])
					v_start = xmin
					v_end = float(ref_range[idx][1:])
					pylab.plot([v_start, v_end], [val, val], color='g', linestyle='-', linewidth=1.5)
				elif ',' in ref_range[idx]:
					try:
						val = float(ref_val[idx])
						exec 'v_range = ['+ ref_range[idx] + ']'
						v_start = v_range[0]
						v_end = v_range[1]
						pylab.plot([v_start, v_end], [val, val], color='g', linestyle='-', linewidth=1.5)
					except:
						pass

	# plot date lines
	if options.closeup == 0:
		# main plot
		if len(format_def[p_index]) > 6:
			plot_date_label = format_def[p_index][6]
		else:
			plot_date_label = 'NO'
		if len(format_def[p_index]) > 7:
			dlabel_size = float(format_def[p_index][7])
		else:
			dlabel_size = 1.4
	else:
		# close-up plot
		if len(format_def[1]) > 6:
			plot_date_label = format_def[1][6]
		else:
			plot_date_label = 'NO'
		if len(format_def[1]) > 7:
			dlabel_size = float(format_def[1][7])
		else:
			dlabel_size = 1.4

	if mark_dates == 'YES' and xparam == 'mjd_obs':
		ydelt = ymax - ymin
		idx = 0
		for mjd in date_mjd:
			if mjd > xmin and mjd < xmax:
				if time_range != 'FULL' and time_range != 'KPI':
					mark_this_date = True
				elif options.closeup != 0:
					if date_months[idx] in ['01', '07']:
						# always mark two dates per year in close-up plots
						mark_this_date = True
					else:
						mark_this_date = False
				else:
					# not a close-up plot: marking depends on time range and report type
					delta_mjd = xmax - xmin
					if delta_mjd < 2920 and date_months[idx] in ['01', '07']:
						# less than 8 years
						mark_this_date = True
					elif report_type in ['REPORT1', 'REPORT2', 'REPORT4'] and delta_mjd < 4380 and date_months[idx] in ['01', '07']:
						# less than 12 years
						mark_this_date = True
					elif date_months[idx] in ['01']:
						mark_this_date = True
					else:
						mark_this_date = False
				if mark_this_date:
					pylab.plot([mjd,mjd], [ymin,ymax], 'k:')
					if plot_date_label == 'YES':
						pylab.text(mjd, ymax+0.03*ydelt, date_labels[idx], family='monospace', 
								size=int(8*dlabel_size/1.4), rotation=90, 
								horizontalalignment='center', verticalalignment='bottom')
			idx = idx + 1

	# time stamp for monitoring plots
	mjd_mark = float(options.time_stamp)
	if mjd_mark > 0.0:
		pylab.plot([mjd_mark, mjd_mark], [ymin, ymax], 'm-')
		pylab.text(mjd_mark-0.02*(xmax-xmin), ymax-0.05*(ymax-ymin), 'now', 
				color='m', size=8, verticalalignment='top', horizontalalignment='right')

	# do labels and axis scaling
	if xlabel == '' and not stat_out and xparam == 'mjd_obs':
		# for close-up plot (stat_out=false): always label time axis
		xlabel = 'MJD_OBS'
	pylab.xlabel(xlabel)
	pylab.ylabel(ylabel)

	pylab.xlim(xmin, xmax)
	if xformat == 'NONE' and stat_out:
		# only suppress tick labels for main plot, not for close-up plots (which have stat_out=false)
		locs, labels = pylab.yticks()
		for idx in range(len(labels)):
			labels[idx] = ''
		pylab.xticks(locs, labels)
		pylab.xlim(xmin, xmax)
	elif xformat != 'AUTO' and xformat != 'NONE':
		fmt = '%'+xformat
		locs, labels = pylab.xticks()
		for idx in range(len(labels)):
			labels[idx] = fmt % locs[idx]
		pylab.xticks(locs, labels)
		pylab.xlim(xmin, xmax)
	
	pylab.ylim(ymin, ymax)
	if yformat == 'NONE' and stat_out:
		# only suppress tick labels for main plot, not for close-up plots (which have stat_out=false)
		locs, labels = pylab.yticks()
		for idx in range(len(labels)):
			labels[idx] = ''
		pylab.yticks(locs, labels)
		pylab.ylim(ymin, ymax)
	elif yformat != 'AUTO' and yformat != 'NONE':
		fmt = '%'+yformat
		locs, labels = pylab.yticks()
		for idx in range(len(labels)):
			labels[idx] = fmt % locs[idx]
		pylab.yticks(locs, labels)
		pylab.ylim(ymin, ymax)
	
	return

# =====================================================================================
# 1.2 create HTML for Highcharts
# =====================================================================================

def create_hc(p_index):

	#page_name = 'trend_report_' + report_name + '_HC_' + str(p_index) + '.html'
	page_name = 'trend_report.html'
	cplot = hc_plot(page_name)

	plt_name = plot_name[p_index][0]
	logging.info('Plot '+str(p_index)+': '+plt_name)

	xlabel = plot_name[p_index][2].replace('SPACE', ' ')
	if xlabel == 'NONE':
		xlabel = ''
	ylabel = plot_name[p_index][4].replace('SPACE', ' ')
	if ylabel == 'NONE':
		ylabel = ''

	# tick marks
	xaxis = plot_name[p_index][5]
	yaxis = plot_name[p_index][6]
	exec 'xlimits = (' + xaxis +')'
	exec 'ylimits = (' + yaxis +')'

	cplot.xname = xlabel
	cplot.yname = ylabel
	cplot.title = label + ', subplot %i' % p_index
	cplot.subtitle = date_range_text.replace('*', '')

	# find all parset definitions for this sub plot
	parset_name = []
	for line in plot_config['PARSET_NAME']:
		if line[0] == plt_name:
			parset_name.append(line[1:])
	
	# format for statistics output
	if len(plot_name[p_index]) > 10:
		if plot_name[p_index][10] == 'AUTO':
			sformat = '%.3g'
		elif plot_name[p_index][10] == 'NONE':
			sformat = '%g'
		else:
			sformat = '%'+plot_name[p_index][10]
	else:
		sformat = '%.3g'

	# no mark last: overwrite mark_last for this sub plot
	if len(plot_name[p_index]) > 9:
		no_mark_last = plot_name[p_index][9]
	else:
		no_mark_last = 'NO'

	# loop over parsets
	data_min_max = []
	for parset in parset_name:
		if not parset[1] in data_sources:
			logging.warning('parset %s has source %s. Ignoring ...' % (parset[0], parset[1]))
			continue

		# get x parameter name in QC1 database for later usage
		if parset[1] == 'QC1DB':
			xparam = parset[3]
		else:
			local_name = parset[3]
			for dataset in plot_config['LOCAL_DATASET']:
				if dataset[0] == local_name:
					xparam = dataset[1]
					break
			else:
				xparam = ''

		# always label time axis
		if cplot.xname == '' and xparam == 'mjd_obs':
			cplot.xname = 'MJD_OBS'

		# get data
		data_file = 'input_'+parset[0]
		if os.path.exists(data_file) and os.path.getsize(data_file) > 0:
			try:
				parset_data = numpy.loadtxt(data_file)
				if len(parset_data.shape) == 1:
					# only one data point
					xdat, ydat, tdat = numpy.split(parset_data, 3)
				else:
					orig_xdat = parset_data[:,0]
					orig_ydat = parset_data[:,1]
					orig_tdat = parset_data[:,2]
					tsort = orig_tdat.argsort()
					xdat = orig_xdat[tsort]
					ydat = orig_ydat[tsort]
					tdat = orig_tdat[tsort]
			except:
				logging.warning('An error occured while reading parset '+parset[0])
				sys.stderr.write('An error occured while reading parset '+parset[0]+'\n')
				xdat, ydat, tdat = [], [], []
		else:
			logging.info(data_file + ' not found or empty')
			xdat, ydat, tdat = [], [], []

		# apply compute rule
		if len(symbols[parset[0]]) > 8:
			rule_name = symbols[parset[0]][8]
		else:
			rule_name = 'NONE'
		if rule_name != 'NONE' and len(ydat) > 0:
			if 'COMPUTE_RULE' in plot_config:
				for line in plot_config['COMPUTE_RULE']:
					if line[0] == rule_name:
						compute_rule = list_to_str(line[1:])
						break
				else:
					logging.warning('COMPUTE_RULE '+rule_name+' specified but not defined')
					compute_rule = 'pass'
			else:
				logging.warning('COMPUTE_RULE '+rule_name+' specified but not defined')
				compute_rule = 'pass'

			try:
				exec compute_rule
			except:
				logging.warning('An error occured while executing COMPUTE_RULE '+rule_name+'. Please check.')
				sys.stderr.write('An error occured while executing COMPUTE_RULE '+rule_name+'. Please check.\n')

		# get statistics
		average, thresholds, thval, tot_num = get_stat(ydat, symbols[parset[0]][4], symbols[parset[0]][5], sformat)

		# save average
		if average != -999:
			cplot.add_avg(average, symbols[parset[0]][4])

		# empty data set: continue with next one
		if len(ydat) == 0:
			continue

		# check for mark last
		if options.type == 'HEALTH' and mark_last == 'YES' and no_mark_last != 'YES':
			tmax = max(tdat) - 1
			nlast = len(xdat[tdat>tmax])
			cplot.add_mark_last(True, nlast)
		else:
			cplot.add_mark_last(False, 0)
		# unclear if useful. For now always set to false:
		#cplot.add_mark_last(False)

		# check for aliens
		if mark_aliens == 'YES':
			if len(data_min_max) == 0:
				data_min_max = [ydat.min(), ydat.max()]
			if len(ylimits) == 2 or len(ylimits) == 4:
				ymin, ymax = ylimits[0], ylimits[1]
				#if (ydat>ymax).any() or (ydat<ymin).any():
				#	cplot.aliens = True
				if (ydat>ymax).any():
					cplot.aliens = True
					if ydat.max() > data_min_max[1]:
						data_min_max[1] =  ydat.max()
				elif (ydat<ymin).any():
					cplot.aliens = True
					if ydat.min() < data_min_max[0]:
						data_min_max[0] =  ydat.min()
				else:
					cplot.aliens = cplot.aliens or False
			else:
				cplot.aliens = cplot.aliens or False

		# handle outliers
		if symbols[parset[0]][6] == 'YES' and thval != '&nbsp;':
			ydat1 = ydat[ydat<thresholds[0]]
			ydat2 = ydat[ydat>thresholds[1]]
			yout = numpy.append(ydat1, ydat2)
		else:
			yout = []

		# save thresholds
		if thval != '&nbsp;':
			if symbols[parset[0]][5][0:3] != 'VAL':
				thstr = ['%f' % thresholds[0], '%f' % thresholds[1]]
			else:
				thstr = symbols[parset[0]][5].replace('VAL=', '').replace(',', ' ').split()
			cplot.thresh = thresholds + thstr

		# add data points
		cplot.add_xdata(xdat)
		cplot.add_ydata(ydat)
		cplot.add_outliers(yout)

		cplot.add_pname(parset[0])

		# add symbol definitions
		cplot.add_symcol(col_mid_to_hc[symbols[parset[0]][2]])
		cplot.add_symname(sym_mid_to_hc[symbols[parset[0]][0]])
		cplot.add_symline(symbols[parset[0]][3])
		cplot.add_symopen(open_sym_hc[symbols[parset[0]][0]])

	# define tick marks and limits of plot
	if len(xlimits) == 4:
		xmin, xmax = xlimits[0], xlimits[1]
	elif len(xlimits) == 2:
		if xparam == 'mjd_obs' and float(options.mjd_min) < float(options.mjd_max):
			xmin = float(options.mjd_min)
			xmax = float(options.mjd_max)
		else:
			xmin, xmax = xlimits[0], xlimits[1]
	else:
		xmin, xmax = min(xminarr), max(xmaxarr)
		xmin = xmin - 0.05 * xdelt
		xmax = xmax + 0.05 * xdelt
	if len(ylimits) == 4:
		ymin, ymax = ylimits[0], ylimits[1]
	elif len(ylimits) == 2:
		ymin, ymax = ylimits[0], ylimits[1]
	else:
		ymin, ymax = min(yminarr), max(ymaxarr)
		ydelt = ymax - ymin
		ymin = ymin - 0.05 * ydelt
		ymax = ymax + 0.05 * ydelt

	cplot.xmin = xmin
	cplot.xmax = xmax
	cplot.ymin = ymin
	cplot.ymax = ymax
	if cplot.aliens and data_min_max[0] < ymin:
		cplot.ymin = data_min_max[0]
	if cplot.aliens and data_min_max[1] > ymax:
		cplot.ymax = data_min_max[1]

	# define date labels
	if xparam == 'mjd_obs' and float(options.mjd_min) < float(options.mjd_max) and len(date_labels) > 0:
		# always plot date labels with MJD as x axis
		dates = []
		idx = 0
		if float(options.mjd_max) - float(options.mjd_min) < 300:
			for mjd in date_mjd:
				if mjd < float(options.mjd_max) and mjd > float(options.mjd_min):
					dates.append((mjd, date_labels[idx]))
				idx = idx + 1
		elif float(options.mjd_max) - float(options.mjd_min) < 450:
			for mjd in date_mjd:
				if mjd < float(options.mjd_max) and mjd > float(options.mjd_min) and date_labels[idx][5:7] in ['01', '04', '07', '10']:
					dates.append((mjd, date_labels[idx]))
				idx = idx + 1
		elif float(options.mjd_max) - float(options.mjd_min) < 4000:
			for mjd in date_mjd:
				if mjd < float(options.mjd_max) and mjd > float(options.mjd_min) and date_labels[idx][5:7] in ['01', '07']:
					dates.append((mjd, date_labels[idx]))
				idx = idx + 1
		else:
			for mjd in date_mjd:
				if mjd < float(options.mjd_max) and mjd > float(options.mjd_min) and date_labels[idx][5:7] in ['01']:
					dates.append((mjd, date_labels[idx]))
				idx = idx + 1
		cplot.dates = dates
	
	# define reference values (KPIs)
	if time_range == 'KPI':
		ref_range = []
		ref_val = []
		if 'REF_VAL' in plot_config:
			for line in plot_config['REF_VAL']:
				if line[0] == plt_name:
					ref_range.append(line[1])
					ref_val.append(line[2])
		if ref_range != [] and ref_val != []:
			for idx in range(len(ref_range)):
				if ref_range[idx] == 'FULL':
					val = float(ref_val[idx])
					v_start = xmin
					v_end = xmax
					cplot.add_ref_values(val, v_start, v_end)
				elif ref_range[idx][0] == '>':
					val = float(ref_val[idx])
					v_start = float(ref_range[idx][1:])
					v_end = xmax
					cplot.add_ref_values(val, v_start, v_end)
				elif ref_range[idx][0] == '<':
					val = float(ref_val[idx])
					v_start = xmin
					v_end = float(ref_range[idx][1:])
					cplot.add_ref_values(val, v_start, v_end)
				elif ',' in ref_range[idx]:
					try:
						val = float(ref_val[idx])
						exec 'v_range = ['+ ref_range[idx] + ']'
						v_start = v_range[0]
						v_end = v_range[1]
						cplot.add_ref_values(val, v_start, v_end)
					except:
						pass

	# define events
	if 'EVENT' in plot_config:
		for line in plot_config['EVENT']:
			if line[0] == plt_name:
				mjd = line[1]
				if float(mjd) < float(options.mjd_min) or float(mjd) > float(options.mjd_max):
					continue
				colour = col_mid_to_hc[line[2]]
				short_text = line[3]
				text = list_to_str(line[4:])
				cplot.add_events(mjd, colour, text, short_text)

	# define line definitions
	if 'LINE' in plot_config:
		for line in plot_config['LINE']:
			if line[0] == plt_name:
				x1 = float(line[1])
				y1 = float(line[2])
				x2 = float(line[3])
				y2 = float(line[4])
				colour = col_mid_to_hc[line[5]]
				cplot.add_lines(x1, y1, x2, y2, colour)
	
	cplot.write()
	return

# =====================================================================================
# 2. parse command line options and configuration
# 2.1 parse arguments and options
# =====================================================================================

cmd_line_options = tp_options(_version_)
options = cmd_line_options.get()

if options.version:
	print _version_
	sys.exit(0)

report_name = options.report
date_descr = options.date_descr
date_range_text = options.date_range
if options.tool_config != "":
	tool_config_file = options.tool_config
else:
	tool_config_file = dfo_config_dir + '/trendPlotter/config.trendPlotter'

# =====================================================================================
# 2.2 parse configuration
# =====================================================================================

# set logging level
set_logging(level='info')

# version number of trendPlotter script
tp_file = open(dfo_bin_dir+'/'+options.tp_filename, 'r')
all_the_file = tp_file.read().splitlines()
tp_file.close()
for line in all_the_file:
	if line[0:12] == 'TOOL_VERSION':
		tp_version = line.replace('TOOL_VERSION=', '').replace('"', '')
		break

# get tool configuration
config_dir = os.path.dirname(tool_config_file)
tool_config = ConfigFile(tool_config_file)

# get plot configuration
plot_config_file = 'config.tp_' + report_name
delta_config_file = 'delta.tp_' + report_name

if not os.path.exists(config_dir+'/'+plot_config_file):
	logging.error('config file '+plot_config_file+' not found. Exit.')
	sys.exit(0)
else:
	all_config = ConfigFile(config_dir+'/'+plot_config_file, 
			mult_tags=['REPORT_NAME', 'PLOT_NAME', 'PARSET_NAME', 'SYMBOLS', 'LOCAL_DATASET', 'CONDITION',
				'ADD_TEXT', 'REMARK', 'COMPUTE_RULE', 'FIT_RULE', 'EVENT', 'LINE', 'REF_VAL'])
	plot_config = all_config.content

if report_name != plot_config['REPORT_NAME'][0][0]:
	logging.error('given report name '+report_name+' does not match with name in '+plot_config_file+'. Exit.')
	sys.exit(0)

report_type = plot_config['REPORT_NAME'][0][1]
if report_type not in report_types:
	logging.error('<REPORT_TYPE> '+report_type+' not allowed. Exit.')
	sys.stderr.write('<REPORT_TYPE> '+report_type+' not allowed. Exit.\n')
	sys.exit(0)

# check for delta configuration
if options.type == 'HISTORY':
	if os.path.exists(config_dir+'/'+delta_config_file):
		all_delta = ConfigFile(config_dir+'/'+delta_config_file, mult_tags=['PLOT_NAME', 'SYMBOLS'])
		delta_config = all_delta.content
	else:
		delta_config = {}
else:
	delta_config = {}

# define some variables
time_range = plot_config['REPORT_NAME'][0][2]
mark_dates = plot_config['REPORT_NAME'][0][3]
mark_aliens = plot_config['REPORT_NAME'][0][4]

tag_add_text = plot_config['REPORT_NAME'][0][5]
add_text = ''
if tag_add_text != 'NONE':
	if 'ADD_TEXT' in plot_config:
		for line in plot_config['ADD_TEXT']:
			if line[0] == tag_add_text:
				add_text = list_to_str(line[1:])
				break
		else:
			logging.warning('ADD_TEXT '+tag_add_text+' specified but not defined')
	else:
		logging.warning('ADD_TEXT '+tag_add_text+' specified but not defined')

if len(plot_config['REPORT_NAME'][0]) > 7:
	mark_last = plot_config['REPORT_NAME'][0][7]
else:
	mark_last = 'NO'

# plot configuration: sub plots
plot_name = {}
for line in plot_config['PLOT_NAME']:
	plot_name[int(line[0])] = line[1:]

# update if delta configuration exists
year_tag = options.ymstart[0:4] + '_ALL'
if 'PLOT_NAME' in delta_config:
	# check all history
	for line in delta_config['PLOT_NAME']:
		if line[-1] == 'HISTORY_ALL':
			if int(line[0]) in plot_name:
				plot_name[int(line[0])] = line[1:-1]
			else:
				logging.warning('delta config file '+delta_config_file+' has PLOT_NAME definition for')
				logging.warning('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.')
				sys.stderr.write('delta config file '+delta_config_file+' has PLOT_NAME definition for ')
				sys.stderr.write('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.\n')
	# check for years
	for line in delta_config['PLOT_NAME']:
		if line[-1] == year_tag:
			if int(line[0]) in plot_name:
				plot_name[int(line[0])] = line[1:-1]
			else:
				logging.warning('delta config file '+delta_config_file+' has PLOT_NAME definition for')
				logging.warning('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.')
				sys.stderr.write('delta config file '+delta_config_file+' has PLOT_NAME definition for ')
				sys.stderr.write('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.\n')
	# check for year-month
	for line in delta_config['PLOT_NAME']:
		if line[-1] == options.ymstart:
			if int(line[0]) in plot_name:
				plot_name[int(line[0])] = line[1:-1]
			else:
				logging.warning('delta config file '+delta_config_file+' has PLOT_NAME definition for')
				logging.warning('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.')
				sys.stderr.write('delta config file '+delta_config_file+' has PLOT_NAME definition for ')
				sys.stderr.write('P_INDEX '+line[0]+' which is not found in '+plot_config_file+'. Skipping.\n')

# plot configuration: symbol definitions
symbols = {}
for line in plot_config['SYMBOLS']:
	symbols[line[0]] = line[1:]

# if KPI: no statistics, no thresholds
if time_range == 'KPI':
	for parset in symbols:
		if symbols[parset][4] != 'NONE':
			symbols[parset][4] = 'NONE'
			logging.warning('statistics defined for parset ' + parset)
			logging.warning('forced to NONE')
		if symbols[parset][5] != 'NONE':
			logging.warning('thresholds defined for parset ' + parset)
			logging.warning('forced to NONE')
			symbols[parset][5] = 'NONE'
			symbols[parset][6] = 'NO'

# update if delta configuration exists
year_tag = options.ymstart[0:4] + '_ALL'
if 'SYMBOLS' in delta_config:
	# check all history
	for line in delta_config['SYMBOLS']:
		if line[-1] == 'HISTORY_ALL':
			symbols[line[0]] = line[1:-1]
	# check for years
	for line in delta_config['SYMBOLS']:
		if line[-1] == year_tag:
			symbols[line[0]] = line[1:-1]
	# check for year-month
	for line in delta_config['SYMBOLS']:
		if line[-1] == options.ymstart:
			symbols[line[0]] = line[1:-1]

# get format configuration for report type
if report_type == 'CUSTOM' and options.closeup == 0:
	format_def_file = config_dir + '/' + report_name + '.fdf'
else:
	format_def_file = dfo_bin_dir+'/'+options.tp_filename

if not os.path.exists(format_def_file):
	logging.error('format definition file '+format_def_file+' not found. Exit.')
	sys.stderr.write('format definition file '+format_def_file+' not found. Exit.\n')
	sys.exit(0)

all_format_def = tp_config(format_def_file, '#REPORT_TYPE')
format_def = {}
if options.closeup == 0:
	# main plot
	for line in all_format_def.content[report_type]:
		format_def[int(line[0])] = line[1:]
else:
	# close-up plot
	format_def[1] = all_format_def.content['REPORT1'][0][1:]

# read date1.dat: this is the temporary version that should be used for plotting;
# it is correct for both normal and monitoring plots
if mark_dates == 'YES':
	date_file = 'date1.dat'
	try:
		dates = ascii.read(date_file)
		date_labels = dates['col1']
		date_mjd = dates['col2']
		date_months = []
		for date_lab in date_labels:
			date_months.append(date_lab[5:7])
	except:
		date_labels = []
		date_mjd = []
		date_months = []

# plot label
if 'PLOT_LABEL' in tool_config.content:
	label = add_text + ' ' + date_descr
else:
	label = dfo_instrument + ': ' + add_text + ' ' + date_descr

# =====================================================================================
# 3. create html using Highcharts
# =====================================================================================

if options.closeup > 0:
	p_index = int(options.closeup)
	if len(plot_name[p_index]) > 11:
		interactive = plot_name[p_index][11]
	else:
		interactive = 'INTER'
	if interactive == 'INTER':
		create_hc(p_index = p_index)
		# do not create PNG for close-up plot
		sys.exit(0)

# =====================================================================================
# 4. create PNGs: loop over sub plots or create closeup plot
# =====================================================================================

fig = pylab.figure(1, figsize=fig_size, dpi=fig_dpi)

# invisible main axes plot
main_ax = pylab.axes([0.001,0.001,0.998,0.998])
pylab.axis('off')

if options.closeup == 0:
	# create main plot: loop over all sub plots
	p_index_arr = plot_name.keys()
	p_index_arr.sort()
	# write statistics
	stat_output = True
else:
	# create close-up plot: only one sub plot
	p_index_arr = [options.closeup]
	# do not write statistics
	stat_output = False

# main loop
for p_index in p_index_arr:
	if options.closeup == 0:
		# main plot: index for format definition is identical to p_index of sub plot
		fmt_idx = p_index
	else:
		# close-up plot: uses format definition from REPORT1 which has only one p_index
		fmt_idx = 1
	xsize = abs(int(format_def[fmt_idx][0])) * xscale
	ysize = abs(int(format_def[fmt_idx][1])) * yscale
	xoff = int(format_def[fmt_idx][2]) * xscale
	yoff = int(format_def[fmt_idx][3]) * yscale + 0.02
	rect = [xoff, yoff, xsize, ysize]
	create_subplot(p_index, rect, stat_output)

	# add p_index label
	pylab.sca(main_ax)
	xlab = int(format_def[fmt_idx][4]) * xscale
	ylab = int(format_def[fmt_idx][5]) * yscale
	pylab.plot([xoff+xlab], [yoff+ylab], color='b', marker='o', markersize=12)
	pylab.text(xoff+xlab, yoff+ylab, p_index, color=(1,1,0.63), family='sans-serif', size=9,
			horizontalalignment='center', verticalalignment='center')

# add labels and signature
main_ax.broken_barh([(0,1)], (120*yscale,1-120*yscale), linewidths=0, facecolors='0.93')
main_ax.broken_barh([(0,1)], (0,0.04), linewidths=0, facecolors='0.93')

# figure title
if options.closeup != 0:
	# add 'close-up' in case of close-up plot
	tmp_str = date_descr.replace(')', ', close-up)')
	date_descr = tmp_str

pylab.figtext(1*xscale, 126*yscale, label, size=14, horizontalalignment='left', verticalalignment='bottom')

# additional label on sub plots
if options.closeup != 0:
	pylab.figtext(0.5, 118*yscale, '*** not interactive ***', size=12, horizontalalignment='center', verticalalignment='top', color='r')

# data range
if 'DATE_RANGE' in tool_config.content:
	date_range = tool_config.content['DATE_RANGE']
else:
	date_range = 'YES'

if date_range == 'YES':
	pylab.figtext(1*xscale, 121*yscale, date_range_text, size=12, horizontalalignment='left', verticalalignment='bottom')

# signature
pylab.figtext(1*xscale, 1*yscale, 'powered by QC: www.eso.org/HC', size=8, horizontalalignment='left', verticalalignment='bottom')
year, month, day, hour, minute, sec = time.gmtime()[0:6]
today = str('%04i-%02i-%02iT%02i:%02i:%02i' % (year, month, day, hour, minute, sec))
pylab.figtext(0.995, 1*yscale, 'created by trendPlotter v' + tp_version + ' on ' + today, size=8,
		                horizontalalignment='right', verticalalignment='bottom')
pylab.xlim(0,1)
pylab.ylim(0,1)

# save png image
outpng = 'trend_report.png'
pylab.savefig(outpng, dpi=fig_dpi, orientation='portrait')

sys.exit(0)
 
