#!/usr/bin/env python
# PURPOSE:	qc_detlin.py: instrument-independent basic QC plots on DETLIN frames (from detector monitoring)
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- January 2009
#		1.0.1	-- quadratic term to non-linearity fit added (2009-02-17)
#		1.1	-- new handling of options and configuration import (2009-03-20)
#		1.1.1	-- QC1 parameters now distributed among different pipeline products (2009-06-05)
#			   [For this version, catg_coeff_map needs to be defined in configuration.]
#		1.1.2	-- parameters plotindex, plottag, ccol, crow, xstat, ystat added (2009-08-14)
#		1.2	-- upgrade to numpy V1.3 and matplotlib V0.99 (2009-10-29)
#		1.2.1	-- command line options pscan and oscan [JP] (2010-11-04) 
#		1.2.2	-- plot nominal gain in plot 3 (2011-05-27)
#		1.2.3	-- remove usage of has_key dictionary attribute (2014-03-12)
#		1.2.4	-- usage of argparse module; option --ext_list instead of -e (2014-04-03)
#		1.2.5	-- annotation for plot 4 updated (2015-02-02)
#		1.2.6	-- enable third order polynomial for plot 5 (2016-02-19)
#
# PARAMETERS:	-a <AB>	[ --ext_list=<EXT_LIST> ]
#		[ --plotindex=<IDX> ] [ --plottag=<TAG> ]
#		[ --ccol=<CCOL> ] [ --crow=<CROW> ]
#		[ --xstat=<XSTAT> ] [ --ystat=<YSTAT> ]
#		[ --pscan=<PSCAN> ] [ --oscan=<OSCAN> ]
#
# OPTIONS:	--version | -h | --help
#
# COMMENTS:	procedure can be called standalone but requires configuration file

_version_ = "1.2.6"

# =====================================================================================
# 0. initialization: import modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 1. function for drawing plots
# =====================================================================================

def draw_plots(AB, rawHDUs, proHDUs, ext_list=(), ccol=-1, crow=-1, xrange_stat=(), yrange_stat=(), pscan=-1, oscan=-1,
		plot_tag='X', plot_index=0, figno=1):

	# =============================================================================
	# 1.1 import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB

	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.error('configuration file could not be found. Exit.')
		sys.exit(1)
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	masterfile = proHDUs[catg_master][0].header['PIPEFILE']

	# function parameters overwrite configuration values
	if crow < 0:
		crow = centrow
	if ccol < 0:
		ccol = centcol
	if pscan >= 0:
		prescan = pscan
	if oscan >= 0:
		overscan = oscan
	
	if len(xrange_stat) == 2:
		stat_xrange = xrange_stat
	if len(yrange_stat) == 2:
		stat_yrange = yrange_stat

	# =============================================================================
	# 1.3 create plots per extension

	# Plotting the results using matplotlib
	# Plot display = 2 x 3 subplots
	# 
	# Plot Order   = 1 2
	#                3 4
	#		 5 6
	#
	#
	# Plot 1: cuts of lamp-on frame with highest exposure time
	# Plot 2: histogram of lamp-on frame
	# Plot 3: fit to gain measurements
	# Plot 4: gain deviations
	# Plot 5: linearity
	# Plot 6: deviations from linearity

	# create figure
	fig = pylab.figure(figno, figsize=paper_size('a4'))

	# which detectors to plot?
	if ext_list == ():
		ext_list = allow_ext

	# loop over extensions
	for ext in ext_list:
		logging.info('plotting for ext ' + str(ext))

		fig.clear()

		# header of figure
		fig.text(0.01, 0.99, ins_name + ': ' + plot_type, horizontalalignment='left', verticalalignment='top', fontsize=14)
		fig.text(0.5, 0.99, AB.content['AB_NAME'] + '[det=' + str(ext) + ']', 
				horizontalalignment='center', verticalalignment='top', fontsize=14)
		fig.text(0.99, 0.99, AB.content['DATE'], horizontalalignment='right', verticalalignment='top', fontsize=14)
		fig.text(0.01, 0.96, 'Lamp-on frame, gain, and linearity', 
				horizontalalignment='left', verticalalignment='top', fontsize=12)
		paramstr = ''
		for param in parameters:
			if param[1] in rawHDUs[1][0].header:
				paramstr = paramstr + ' ' + param[0] + ' = ' + str(rawHDUs[1][0].header[param[1]])
		fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=12)

		# footer
		fig.text(0.01, 0.01, 'Product: ' + masterfile + '[ext=' + str(ext) + ']', 
				horizontalalignment='left', verticalalignment='bottom', fontsize=8)
		
		# figure date
		year, month, day, hour, minute, sec = time.localtime()[0:6]
		today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, minute, sec))
		fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=8)

		# find lamp-on frame with highest exposure time
		max_exp = 0.0
		idx_exp = -1
		for idx in range(0, len(rawHDUs)):
			if rawHDUs[idx][0].header[lamp_on_key] == lamp_on_keyval:
				if rawHDUs[idx][0].header[exp_key] > max_exp:
					idx_exp = idx
					max_exp = rawHDUs[idx][0].header[exp_key]
		if idx_exp == -1:
			logging.error('No lamp-on frame found. Exit.')
			return

		# normalise raw data
		raw_image = norm_raw(rawHDUs[idx_exp], ext)
	
		# chop pre and overscan
		if overscan > 0:
			raw_image = raw_image[:,prescan:-overscan]
		else:
			raw_image = raw_image[:,prescan:]
	
		# chop images for histograms
		raw_hst = raw_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]

		# get tables
		lin_tab = proHDUs[catg_lin_info][max(ext,1)].data
		gain_tab = proHDUs[catg_gain_info][max(ext,1)].data

		# Plot 1
		logging.info('plot 1: cut in X/Y')

		pylab.subplot(321)

		plot1 = LinePlot()
		plot1.add_line(raw_image, 'row', '@'+str(crow), 'raw @Y='+str(crow), col_raw1)
		plot1.add_line(raw_image, 'col', '@'+str(ccol), 'raw @X='+str(ccol), col_col)
		plot1.draw(yrange=plot1_yrange)
		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('X/Y', size=8)
		pylab.ylabel('ADU', size=8)
		pylab.title('cuts in X and Y', size=10)
		pylab.legend()

		# Plot 2
		logging.info('plot 2: raw histogram')

		pylab.subplot(322)

		plot2 = HistoPlot(logplot=histo_log, stepsize=histo_step, binrange=histo_range)
		plot2.add_histo(raw_hst, False, 'raw', col_raw1)
		plot2.draw()

		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('counts',size=8)
		pylab.ylabel('log frequency',size=8)
		pylab.title('histogram [x='+str(stat_xrange[0])+':'+str(stat_xrange[1]) +
				'; y='+str(stat_yrange[0])+':'+str(stat_yrange[1])+']', size=10)
		pylab.legend()

		# Plots 3-4: gain

		# fitted gain and nominal value
		qc_gain = proHDUs[catg_gain_info][ext].header[qc_gain_key]
		if nom_gain_key in rawHDUs[0][ext].header:
			nom_gain = rawHDUs[0][ext].header[nom_gain_key]
		else:
			nom_gain = 0

		# Plot 3

		logging.info('plot 3: gain fit')

		pylab.subplot(323)
		plot3_xrange = (0.0, 1.1*gain_tab.field('X_FIT_CORR').max())
		pylab.plot(gain_tab.field('X_FIT_CORR'), gain_tab.field('Y_FIT'), 'o', markerfacecolor='w', markeredgecolor='r',
				markersize=3)
		pylab.plot([0.0,plot3_xrange[1]], [0.0,1.0/qc_gain*plot3_xrange[1]], 'b-')
		if nom_gain > 0:
			pylab.plot([0.0,plot3_xrange[1]], [0.0,1.0/nom_gain*plot3_xrange[1]], 'g-')

		pylab.text(plot3_xrange[1]/10., 0.83/min(qc_gain,nom_gain)*plot3_xrange[1], '1/gain = dY/dX', horizontalalignment='left',
				color='b', size=8)
		pylab.text(plot3_xrange[1]/10., 0.91/min(qc_gain,nom_gain)*plot3_xrange[1], '1/nominal_gain', horizontalalignment='left',
				color='g', size=8)
		pylab.xlim(plot3_xrange[0], plot3_xrange[1])
		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('ON1 + ON2 - OFF1 - OFF2 [ADU]', size=8)
		pylab.ylabel('sigma(ON)^2 - sigma(OFF)^2 [ADU^2/e-]', size=8)

		# Plot 4

		logging.info('plot 4: gain vs. flux')

		pylab.subplot(324)
		plot4_xrange = (0.0, 1.1*gain_tab.field('ADU').max())
		try:
			# plot GAIN column
			pylab.plot(gain_tab.field('ADU'), gain_tab.field('GAIN'), 'o', markerfacecolor='w', markeredgecolor='r',
					markersize=3)
		except:
			try:
				# try without GAIN column
				gain_values = gain_tab.field('X_FIT') / gain_tab.field('Y_FIT')
				pylab.plot(gain_tab.field('ADU'), gain_values, 'o', markerfacecolor='w', markeredgecolor='r',
						markersize=3)
			except:
				logging.warning('individual gain values not plotted')

		if nom_gain > 0:
			pylab.plot([0.0, plot4_xrange[1]], [nom_gain,nom_gain], 'g-', linewidth=0.5)
			pylab.text(0.9*plot4_xrange[1], nom_gain, 'nominal gain = %5.3f' % nom_gain, horizontalalignment='right',
					color='g', size=8)
		pylab.plot([0.0, plot4_xrange[1]], [qc_gain,qc_gain], 'b-', linewidth=0.5)
		pylab.text(0.1*plot4_xrange[1], qc_gain, 'QC gain = %5.3f' % qc_gain, horizontalalignment='left',
				color='b', size=8)
		pylab.xlim(plot4_xrange[0], plot4_xrange[1])
		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('1/2 * (ON1 + ON2 - OFF1 - OFF2) [ADU]', size=8)
		pylab.ylabel('gain [e-/ADU]', size=8)
		#pylab.title('gain vs. flux', size=10)

		# Plot 5-6: linearity

		# Plot 5: median of lamp-on frames and linear fit

		logging.info('plot 5: linearity')

		pylab.subplot(325)
		plot5_xrange = (0.0, 1.1*lin_tab.field(exp_tab_col).max())
		pylab.plot(lin_tab.field(exp_tab_col), lin_tab.field('MED'), 'o', markerfacecolor='w', markeredgecolor='r',
				markersize=3)

		# using median linearity coefficients
		a = proHDUs[catg_coeff_map][ext].header['HIERARCH ESO QC LIN COEF0']
		b = proHDUs[catg_coeff_map][ext].header['HIERARCH ESO QC LIN COEF1']
		c = proHDUs[catg_coeff_map][ext].header['HIERARCH ESO QC LIN COEF2']
		t = pylab.arange(0.0, plot5_xrange[1]+plot5_xrange[1]/100.0, plot5_xrange[1]/100.0)
		if 'HIERARCH ESO QC LIN COEF3' in proHDUs[catg_coeff_map][ext].header:
			d = proHDUs[catg_coeff_map][ext].header['HIERARCH ESO QC LIN COEF3']
			funct = a + b * t + c * t*t + d * t*t*t
		else:
			funct = a + b * t + c * t*t
		pylab.plot([0.0, plot5_xrange[1]], [a, a+b*plot5_xrange[1]], 'b-')
		pylab.plot(t, funct, 'g-')
		
		pylab.xlim(plot5_xrange[0], plot5_xrange[1])
		plot5_yrange = pylab.ylim()
		if 'HIERARCH ESO QC LIN COEF3' in proHDUs[catg_coeff_map][ext].header:
			pylab.text(plot5_xrange[1]/20., 0.85*(plot5_yrange[1]-plot5_yrange[0])+plot5_yrange[0],
					'f(t) = %6.2f + %6.2f * t + %6.2f * t^2 + %6.2f *t^3' % (a,b,c,d),
					horizontalalignment='left', color='g', size=8)
		else:
			pylab.text(plot5_xrange[1]/20., 0.85*(plot5_yrange[1]-plot5_yrange[0])+plot5_yrange[0], 
					'f(t) = %6.2f + %6.2f * t + %6.2f * t^2' % (a,b,c), 
					horizontalalignment='left', color='g', size=8)
		pylab.text(plot5_xrange[1]/20., 0.72*(plot5_yrange[1]-plot5_yrange[0])+plot5_yrange[0], 
				'f(t) = %6.2f + %6.2f * t' % (a,b), 
				horizontalalignment='left', color='b', size=8)

		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('exptime [sec]', size=8)
		pylab.ylabel('median ON [ADU]', size=8)

		# Plot 6: residuals

		logging.info('plot 6: linearity residuals')

		pylab.subplot(326)
		plot6_xrange = (0.0, 1.1*lin_tab.field('MED').max())
		pylab.plot(lin_tab.field('MED'), lin_tab.field('MED_DIT'), 'o', markerfacecolor='w', markeredgecolor='r',
				markersize=3)

		pylab.xlim(plot6_xrange[0], plot6_xrange[1])
		pylab.xticks(size=8)
		pylab.yticks(size=8)
		pylab.xlabel('median ON [ADU]', size=8)
		pylab.ylabel('median ON / exptime [ADU/sec]', size=8)

		# additional information
		fig.text(0.91, 0.80, 'lamp-on frame:', horizontalalignment='left', verticalalignment='top', fontsize=8)
		fig.text(0.91, 0.78, '%5.1f sec' % max_exp, horizontalalignment='left', verticalalignment='top', fontsize=8)

		fig.text(0.91, 0.52, 'gain', horizontalalignment='left', verticalalignment='top', fontsize=8)
		fig.text(0.91, 0.50, 'computation', horizontalalignment='left', verticalalignment='top', fontsize=8)

		fig.text(0.91, 0.27, 'non-linearity', horizontalalignment='left', verticalalignment='top', fontsize=8)
		fig.text(0.91, 0.25, 'correction:', horizontalalignment='left', verticalalignment='top', fontsize=8)
		if 'HIERARCH ESO QC LIN EFF' in proHDUs[catg_lin_info][ext].header:
			lin_eff = proHDUs[catg_lin_info][ext].header['HIERARCH ESO QC LIN EFF']
			fig.text(0.91, 0.23, '%-7.4f @' % lin_eff, horizontalalignment='left', verticalalignment='top', fontsize=8)
			ref_level = '0'
			for idx in range(1,100):
				param_key = 'HIERARCH ESO PRO REC1 PARAM%i NAME' % idx
				if param_key in proHDUs[catg_master][0].header:
					if proHDUs[catg_master][0].header[param_key] == 'ref_level':
						ref_level = proHDUs[catg_master][0].header['HIERARCH ESO PRO REC1 PARAM%i VALUE' % idx]
						break
				else:
					break
			fig.text(0.91, 0.21, '%s ADU' % ref_level, horizontalalignment='left', verticalalignment='top', fontsize=8)
		else:
			fig.text(0.91, 0.23, 'n/a', horizontalalignment='left', verticalalignment='top', fontsize=8)

		# save figures
		outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
		pylab.savefig(outpng, dpi=150, orientation='portrait')
		logging.info('plot saved as ' + outpng)

		plot_index = plot_index + 1

	# return plot_index for usage in other procedures
	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Creates a basic QC plot on detector monitoring data.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('--ext_list', metavar='EXT_LIST', dest='ext_list', default='()',
			help='''list of extentions in raw/product frames, e.g. --ext_list="(1,2,3,4)"''')
	parser.add_argument('--plotindex', metavar='IDX', dest='plot_index', default='0',
			help='''plot index used for png output;
			output is written to <PIPEFILE_BASENAME>_<TAG><IDX>.png''') 
	parser.add_argument('--plottag', metavar='TAG', dest='plot_tag', default='X',
			help='''plot tag used for png output;
			output is written to <PIPEFILE_BASENAME>_<TAG><IDX>.png''')
	parser.add_argument('--ccol', metavar='CCOL', dest='ccol', default='-1',
			help='''central column; overwrites centcol definition in configuration''')
	parser.add_argument('--crow', metavar='CROW', dest='crow', default='-1',
			help='''central row; overwrites centrow definition in configuration [optional]''')
	parser.add_argument('--xstat', metavar='XSTAT', dest='xstat', default='()',
			help='''x area for histogram statistics, e.g. --xstat="(0,1024)";
			overwrites value of stat_xrange in configuration''')
	parser.add_argument('--ystat', metavar='YSTAT', dest='ystat', default='()',
			help='''y area for histogram statistics, e.g. --ystat="(100,400)";
			overwrites value of stat_yrange in configuration''')
	parser.add_argument('--pscan', metavar='PSCAN', dest='pscan', default='-1',
			help='''PRE-scan; overwrites prescan definition in configuration''')
	parser.add_argument('--oscan', metavar='OSCAN', dest='oscan', default='-1',
			help='''OVER-scan; overwrites overscan definition in configuration''')

	# parse arguments/options
	args = parser.parse_args()

	# convert args.xstat and args.ystat (char strings) into Python tuples/lists
	exec 'ext_list = ' + args.ext_list
	exec 'xstat = ' + args.xstat
	exec 'ystat = ' + args.ystat

	# set logging level
	set_logging()

	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')

	# reading fits files
	# get list with HDUs of raw files

	logging.info('reading raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('reading product frames')
	proHDUs = AB.get_pro()

	# draw plots
	draw_plots(AB, rawHDUs, proHDUs, ext_list=ext_list, ccol=int(args.ccol), crow=int(args.crow),
			xrange_stat=xstat, yrange_stat=ystat, pscan=int(args.pscan), oscan=int(args.oscan),
			plot_tag=str(args.plot_tag), plot_index=int(args.plot_index))

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

