#!/usr/bin/env python
# PURPOSE:	qc_mstplt.py: instrument-independent basic QC plots on master products (master bias, etc.)
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	0.9	-- November/December 2007
#		1.0	-- July 2008
#		1.0.1	-- avoid error when header key in 'parameters' does not exist (2008-08-07)
#		1.0.2	-- fixed stepsize in plot 9, needed to allow pre-defined bin range in plot 3 (2008-11-24)
#		1.1	-- new handling of options and configuration import (2009-03-19)
#		1.2	-- updated layout of figure header and footer;
#			   options plot_tag, plot_index, and fig_size introduced (2009-08-05)
#		1.2.1	-- crow and ccol as parameters for draw_plots (2009-08-13)
#		1.3	-- upgrade to numpy V1.3 and matplotlib V0.99 (2009-10-29)
#		1.3.1	-- introduction of general configuration file config_gen (2010-02-17)
#		1.4	-- map detector -> extension in products (2014-01-15)
#		1.4.1	-- bug fix with outpng name; improved mapping detector -> extension (2014-01-24)
#		1.4.2	-- usage of argparse module: basic_parser instead of basic_options (2014-03-31)
#		2.0	-- plot 6 configurable: displays either row/col average or 2D image (2014-05-21)
#		2.0.1	-- xcentrange/ycentrange as function parameters (2014-12-17)
#		2.1	-- enable defaults for centcol, etc.; introduce print_extname in config;
#			   introduce pro_catg as function parameter (2015-11-09)
#		2.1.1	-- avoid to create an axis twice for plot 6 (2018-06-12)
#
# PARAMETERS:	-a <AB>	[ --raw_exts=<RAW_EXT_LIST> ] [ --pro_exts=<PRO_EXT_LIST> ]
#
# OPTIONS:	--version | -h | --help
#

_version_ = "2.1.1"

# =====================================================================================
# 0. initialization: import modules 
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 1. function for drawing plots
# =====================================================================================

def draw_plots(AB, rawHDUs, proHDUs, pro_catg='', raw_exts=(), pro_exts=(), crow=-1, ccol=-1, xcentrange=(), ycentrange=(),
		plot_tag='X', plot_index=0, fig_size='a4'):

	# =============================================================================
	# 1.1 import configuration

	# import general configuration file, if present
	try:
		exec 'from config_gen import * '
	except:
		pass
	else:
		logging.info('configuration imported from config_gen')

	# import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB
	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.error('configuration file could not be found. Exit.')
		sys.exit(1)
	else:
		# save xcentrange/ycentrange before import
		xcr = xcentrange
		ycr = ycentrange
		# import
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)
		# copy back
		if len(xcr) == 2:
			xcentrange = xcr
		if len(ycr) == 2:
			ycentrange = ycr

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	if pro_catg != '':
		catg_master = pro_catg
	masterfile = proHDUs[catg_master][0].header['PIPEFILE']

	# central row, column (first row, column is 0)
	if crow < 0:
		crow = centrow
	if ccol < 0:
		ccol = centcol

	# check for configuration of plot 6
	try:
		dummy = len(plot6_type)
	except:
		plot6_type = 'col_row_avg'

	# check for print_extname in config file
	try:
		dummy = print_extname
	except:
		print_extname = False

	# =============================================================================
	# 1.3 find reference frame

	ref_keys = ['HIERARCH ESO PRO CATG']
	key_values = [catg_master]
	for key in ref_match_keys:
		if key in proHDUs[catg_master][0].header:
			ref_keys.append(key)
			key_values.append(proHDUs[catg_master][0].header[key])

	ref_frame = find_fits(ref_keys, key_values, ref_dir)

	if ref_frame == '':
		logging.warning('no reference frame found')
	else:
		ref_HDU = pyfits.open(ref_dir + '/' + ref_frame)
		if proHDUs[catg_master][0].header['PIPEFILE'] == ref_HDU[0].header['PIPEFILE']:
			ref_frame = ''
			logging.warning('reference and master identical')
		else:
			logging.info('reference frame = ' + ref_frame)

	# =============================================================================
	# 1.4 create plots per detector

	# Plotting the results using matplotlib
	# Plot display = 3 x 3 subplots
	# 
	# Plot Order   = 1 2 3
	#                4 5 6
	#		 7 8 9
	#
	#
	# Plot 1: Row cut
	# Plot 2: Column cut
	# Plot 3: Histogram (raw/master)
	# Plot 4: Enlargement (rows)
	# Plot 5: Enlargement (columns)
	# Plot 6: Averages of rows and columns
	# Plot 7: comparison to reference (row cut)
	# Plot 8: comparison to reference (col cut)
	# Plot 9: comparison to reference (histogram)

	# which detectors to plot?
	if raw_exts == ():
		det_list = allow_ext
	else:
		det_list = raw_exts

	# map extension in raw frame to extension in product
	# default: it is assumed that detector 1 is in extension 1 of raw fits, detector 2 in extension 2, etc.
	# and that this is the same for pipeline products;
	# this is not true if products contain e.g. error maps in addition
	if pro_exts == ():
		try:
			pro_ext = raw_to_pro_ext
		except:
			pro_ext = allow_ext
	else:
		pro_ext = pro_exts

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes(fig_size)
	fig = pylab.figure(20, figsize=paper_size(fig_size))

	for det_idx in range(len(det_list)):
		det = det_list[det_idx]
		logging.info('plotting for det ' + str(det))

		fig.clear()

		# header of figure
		fig.text(0.01, 0.99, ins_name + ': ' + plot_type, 
				horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
		if print_extname and 'EXTNAME' in rawHDUs[0][det].header:
			fig.text(0.5, 0.99, AB.ab_name + '[' + rawHDUs[0][det].header['EXTNAME'] + ']',
					horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
		else:
			fig.text(0.5, 0.99, AB.ab_name + '[det=' + str(det) + ']', 
					horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
		fig.text(0.99, 0.99, AB.date, horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
		fig.text(0.01, 0.96, 'Raw and product frames', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
		paramstr = ''
		if len(parameters) == 0:
			if 'RAW_MATCH_KEY' in AB.content:
				for raw_match_key in AB.content['RAW_MATCH_KEY']:
					paramstr = paramstr + ' ' + string.split(string.replace(raw_match_key[0], '=', ' '))[-1]
		else:
			for param in parameters:
				if param[1] in proHDUs[catg_master][0].header:
					paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])

		if fig_size == 'a6':
			fig.text(0.99, 0.955, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)
		else:
			fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

		# bottom
		if ref_frame == '':
			fig.text(0.01, 0.03, 'RefFrame: none', 
					horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		else:
			fig.text(0.01, 0.03, 'RefFrame: ' + ref_frame, 
					horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		fig.text(0.01, 0.01, 'Product: ' + masterfile + '[ext=' + str(pro_ext[det_idx]) + ']', 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		# figure date
		year, month, day, hour, min, sec = time.localtime()[0:6]
		today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
		fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

		# normalise raw data
		raw1_image = norm_raw(rawHDUs[0], det)
		raw2_image = norm_raw(rawHDUs[-1], det)
	
		# chop pre and overscan
		if overscan > 0:
			raw1_image = raw1_image[:,prescan:-overscan]
			raw2_image = raw2_image[:,prescan:-overscan]
		else:
			raw1_image = raw1_image[:,prescan:]
			raw2_image = raw2_image[:,prescan:]
	
		# scale master and reference
		mst_image = norm_mst(proHDUs[catg_master], pro_ext[det_idx])
		if ref_frame != '':
			ref_image = norm_mst(ref_HDU, pro_ext[det_idx])
			# comparison of master with reference
			if comp_method == 'sub':
				comp_image = mst_image - ref_image
			else:
				comp_image = im_divide(mst_image, ref_image)

		# determine crow, etc. in case of default values
		if ccol < 0:
			ccol = mst_image.shape[1] // 2
		if crow < 0:
			crow = mst_image.shape[0] // 2
		if len(xcentrange) == 0:
			xdelt = mst_image.shape[1] // 10
			xcentrange = (ccol-xdelt, ccol+xdelt)
		if len(ycentrange) == 0:
			ydelt = mst_image.shape[0] // 10
			ycentrange = (crow-ydelt, crow+ydelt)
		if len(stat_xrange) == 0:
			stat_xrange = (0, mst_image.shape[1])
		if len(stat_yrange) == 0:
			stat_yrange = (0, mst_image.shape[0])
	
		# chop images for histograms
		raw1_hst = raw1_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
		raw2_hst = raw2_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
		mst_hst = mst_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
		if ref_frame != '':
			ref_hst = ref_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
			comp_hst = comp_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]

		# Plot 1
		logging.info('plot 1: cut in X')

		if ref_frame == '':
			pylab.subplot(231)
		else:
			pylab.subplot(331)

		plot1 = LinePlot()
		for pdata in plot1_data:
			if pdata == 'first_raw':
				plot1.add_line(raw1_image, 'row', '@'+str(crow), 'first raw', col_raw1)
			if pdata == 'last_raw':
				plot1.add_line(raw2_image, 'row', '@'+str(crow), 'last raw', col_raw2)
			if pdata == 'master':
				plot1.add_line(mst_image, 'row', '@'+str(crow), 'master', col_mst)
			if pdata == 'reference' and ref_frame != '':
				plot1.add_line(ref_image, 'row', '@'+str(crow), 'reference', col_ref)
		plot1.draw(yrange=plot1_yrange)
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.title('row @ Y = ' + str(crow), size=fs_sub)
		pylab.legend()

		# Plot 2
		if ref_frame == '':
			pylab.subplot(232)
		else:
			pylab.subplot(332)

		logging.info('plot 2: cut in Y')

		plot2 = LinePlot()
		for pdata in plot2_data:
			if pdata == 'first_raw':
				plot2.add_line(raw1_image, 'col', '@'+str(ccol), 'first raw', col_raw1)
			if pdata == 'last_raw':
				plot2.add_line(raw2_image, 'col', '@'+str(ccol), 'last raw', col_raw2)
			if pdata == 'master':
				plot2.add_line(mst_image, 'col', '@'+str(ccol), 'master', col_mst)
			if pdata == 'reference' and ref_frame != '':
				plot2.add_line(ref_image, 'col', '@'+str(ccol), 'reference', col_ref)
		plot2.draw(yrange=plot2_yrange)
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('Y', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.title('column @ X = ' + str(ccol), size=fs_sub)
		pylab.legend()

		# Plot 3
		logging.info('plot 3: raw/master histograms')

		if ref_frame == '':
			pylab.subplot(233)
		else:
			pylab.subplot(333)

		plot3 = HistoPlot(logplot=histo_log, stepsize=histo_step, binrange=histo_range)
		for pdata in plot3_data:
			if pdata[0] == 'master' and pdata[1] == 'nofit':
				plot3.add_histo(mst_hst, False, 'mst', col_mst)
			if pdata[0] == 'master' and pdata[1] == 'fit':
				plot3.add_histo(mst_hst, True, 'fit', col_histo)
			if pdata[0] == 'first_raw' and pdata[1] == 'nofit':
				plot3.add_histo(raw1_hst, False, 'raw', col_raw1)
			if pdata[0] == 'first_raw' and pdata[1] == 'fit':
				plot3.add_histo(raw1_hst, True, 'fit', col_histo)
			if pdata[0] == 'last_raw' and pdata[1] == 'nofit':
				plot3.add_histo(raw2_hst, False, 'raw', col_raw2)
			if pdata[0] == 'last_raw' and pdata[1] == 'fit':
				plot3.add_histo(raw2_hst, True, 'fit', col_histo)
		plot3.draw()

		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		if plot6_type == 'col_row_avg' or ref_frame == '':
			pylab.xlabel('counts',size=fs_foot)
		pylab.ylabel('log frequency',size=fs_foot)
		pylab.title('histograms',size=fs_sub)
		pylab.legend()

		# Plot 4
		logging.info('plot 4: cut in X detail')

		if ref_frame == '':
			pylab.subplot(234)
		else:
			pylab.subplot(334)

		plot4 = LinePlot()
		for pdata in plot4_data:
			if pdata == 'first_raw':
				plot4.add_line(raw1_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), 'first raw', col_raw1)
			if pdata == 'last_raw':
				plot4.add_line(raw2_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), 'last raw', col_raw2)
			if pdata == 'master':
				plot4.add_line(mst_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), 'master', col_mst)
			if pdata == 'reference' and ref_frame != '':
				plot4.add_line(ref_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), 'reference', col_ref)
		plot4.draw(xrange=xcentrange, yrange=plot4_yrange, xarray=numpy.arange(xcentrange[0], xcentrange[1]))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.legend()

		# Plot 5
		logging.info('plot 5: cut in Y detail')

		if ref_frame == '':
			pylab.subplot(235)
		else:
			pylab.subplot(335)

		plot5 = LinePlot()
		for pdata in plot5_data:
			if pdata == 'first_raw':
				plot5.add_line(raw1_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), 'first raw', col_raw1)
			if pdata == 'last_raw':
				plot5.add_line(raw2_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), 'last raw', col_raw2)
			if pdata == 'master':
				plot5.add_line(mst_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), 'master', col_mst)
			if pdata == 'reference' and ref_frame != '':
				plot5.add_line(ref_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), 'reference', col_ref)
		plot5.draw(xrange=ycentrange, yrange=plot5_yrange, xarray=numpy.arange(ycentrange[0], ycentrange[1]))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('Y', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.legend()

		# plot 6
		if plot6_type == 'image':
			logging.info('plot 6: image')
			if ref_frame == '':
				plot6 = ImagePlot(image_map=(3,2))
			else:
				plot6 = ImagePlot(image_map=(3,3))
			if plot6_data == 'first_raw':
				plot6_image = raw1_image
				plot6_title = 'first raw'
			elif plot6_data == 'last_raw':
				plot6_image = raw2_image
				plot6_title = 'last raw'
			elif plot6_data == 'reference':
				if ref_frame != '':
					plot6_image = comp_image
				else:
					plot6_image = numpy.zeros((100,100), dtype=numpy.float32)
				plot6_title = 'mst vs. ref'
			else:
				plot6_image = mst_image
				plot6_title = 'master'
			if plot6_area == 'centre':
				plot6.add_image(plot6_image, 6, plot6_title, cuts=plot6_cuts,  xrange=xcentrange, yrange=ycentrange)
			else:
				plot6.add_image(plot6_image, 6, plot6_title, cuts=plot6_cuts)
			plot6.draw(aspect='equal')
		else:
			logging.info('plot 6: avg of rows/cols')

			if ref_frame == '':
				pylab.subplot(236)
			else:
				pylab.subplot(336)

			plot6 = LinePlot()
			plot6.add_line(mst_image, 'row', 'avg', 'avg rows', col_row)
			plot6.add_line(mst_image, 'col', 'avg', 'avg cols', col_col)
			plot6.draw()
			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('X / Y', size=fs_foot)
			pylab.ylabel('ADU', size=fs_foot)
			pylab.legend()

		if ref_frame == '':
			logging.info('no reference frame available, comparison plots not created')
		else:
			# Plot 7
			logging.info('plot 7: comparison to reference, cut in X')

			pylab.subplot(337)

			plot7 = LinePlot()
			plot7.add_line(comp_image, 'row', '@'+str(crow), 'mst vs. ref', col_ref)
			plot7.draw()
	
			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('X', size=fs_foot)
			pylab.ylabel('ADU', size=fs_foot)
			pylab.legend()
	
			# Plot 8
			logging.info('plot 8: comparison to reference, cut in Y')

			pylab.subplot(338)

			plot8 = LinePlot()
			plot8.add_line(comp_image, 'col', '@'+str(ccol), 'mst vs. ref', col_ref)
			plot8.draw()
	
			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('Y', size=fs_foot)
			pylab.ylabel('ADU', size=fs_foot)
			pylab.legend()
	
			# Plot 9
			logging.info('plot 9: mst vs. ref histogram')

			pylab.subplot(339)

			#plot9 = HistoPlot(logplot=histo_log, stepsize=histo_step, binrange=histo_range)
			#plot9 = HistoPlot(logplot=histo_log, stepsize=histo_step, binrange=20.0)
			plot9 = HistoPlot(logplot=histo_log, stepsize=0, binrange=20.0)
			plot9.add_histo(comp_hst, False, 'mst vs. ref', col_ref)
			#plot9.add_histo(comp_image, True, 'fit', (0.00,1.00,0.00))
			plot9.draw()

			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('counts',size=fs_foot)
			pylab.ylabel('log frequency',size=fs_foot)
			pylab.legend()

		# save figure
		#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
		#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'_0123456789.fits'), plot_tag, plot_index)
		outpng = '%s%s%02i.png' % (string.rstrip(masterfile,'0123456789.fits'), plot_tag, plot_index)
		pylab.savefig(outpng, dpi=150, orientation='portrait')
		logging.info('plot saved as ' + outpng)
		plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Creates instrument-independent basic QC plots on master products.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('--raw_exts', metavar='RAW_EXTs', dest='raw_exts', default='()',
			help='''list of extentions in raw frames, e.g. --raw_exts="(1,2,3)"''')
	parser.add_argument('--pro_exts', metavar='PRO_EXTs', dest='pro_exts', default='()',
			help='''map det -> ext of product, e.g. --pro_exts="(1,3,5)"''')

	# parse arguments/options
	args = parser.parse_args()

	# convert into Python list
	exec 'raw_exts = ' + args.raw_exts
	exec 'pro_exts = ' + args.pro_exts
	
	# set logging level
	set_logging()
	
	logging.info('started')

	# parse AB
	# AB.content: dictionary with string content of AB 

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')
	
	# reading fits files
	# get list with HDUs of raw files

	logging.info('reading raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('reading product frames')
	proHDUs = AB.get_pro()

	# draw plots
	draw_plots(AB, rawHDUs, proHDUs, raw_exts=raw_exts, pro_exts=pro_exts, plot_tag='X', plot_index=0, fig_size='a4')

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

