#ifndef amdlib_H
#define amdlib_H
/************************************************************************
 * E.S.O. - VLT project
 *
 * "@(#) $Id: amdlib.h,v 1.21 2006/01/11 15:19:56 vltsccm Exp $"
 *
 * who       when      what
 * --------  --------  ----------------------------------------------
 * gduvert   25/10/05  - added Flat field structure in p2vm to enable correct
 *                       computation of sigma2 at science data level.
 * gduvert   30/08/05  - replaced visError by sigma2Vis since this is really
 *                       a variance. Noted that pistonOPD is of the same
 *                       unit as lambda (NOT metres as in OI-FITS).
 * gduvert   24/08/05  - added dateObs (string) to vis, vis2 and vis3 struct.
 *                     - supressed unused IAU references.
 *                     - previous dateObs length was wrong
 *                     - added timeTag to RAW_DATA since it must be used both
 *                       by the algorithm to detect null frames and in OI_VIS
 *                       tables.
 * fmillour  30/05/05  - added amdlibPHOTOMETRY structure
 * gduvert   28/05/05  - dit time vs. exposure time 
 * gduvert   22/04/05  - changed several function argument list
 * gduvert   26/10/04  - add detector saturation
 *                     - add wavelength offset, in pixel, for the Interf column
 *                     - delete amdlibBAD_PIXEL_MAP in 
 *                       amdlibPerformSpectralCalibration function
 *                     - amdlibFindIndexOfMaximum: change *data from double to 
 *                       float
 * gzins     29/01/03  created (adapted from amoql library written by Etienne
 *                     Le-Coarer and Bram Acke)
 */

/************************************************************************
 *
 *----------------------------------------------------------------------
 */

/* The following piece of code alternates the linkage type to C for all
 * functions declared within the braces, which is necessary to use the
 * functions in C++-code.
 */

#ifdef __cplusplus
extern "C" { 
#endif

/** Definition of boolean */
typedef enum 
{
    amdlibFALSE = 0,
    amdlibTRUE
} amdlibBOOLEAN;


#include "fitsio.h"
#include "sfftw.h" /* fast FFT program */
#include "srfftw.h"

/*
 * This diagram shows the detector arrays (512x512 pixels) with a possible
 * sub-windowing configuration.
 *
 *      +---------------------------------------------+
 *      |                                             |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      |                                             |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      |                                             |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  |   |    |   |    |   |    |   |    |   |  |
 *      |  +---+    +---+    +---+    +---+    +---+  |
 *      +---------------------------------------------+
 *
 * The following naming convention for the variables has been adopted :
 *   - nbCols    : specifies the number of colunms defined in the current
 *                 detector configuration (5 in the above example) 
 *   - nbRows    : specifies the number of spectral windows read on the
 *                 detector (3 in the above example) 
 *   - nx        : the number of pixels in column (in X direction)
 *   - ny        : the number of pixels in rows (in Y direction)
 *    
 */

#define amdlibMax(A,B) ((A)>(B)?(A):(B))
#define amdlibMin(A,B) ((A)<(B)?(A):(B))

/* Definition of the structure corresponding to the binary tables included in
 * the FITS files produced by the AMBER instrument (see document
 * VLT-SPE-ESO-15000-2764) */

#define amdlibNTEL      3
#define amdlibNBASELINE amdlibNTEL*(amdlibNTEL-1)/2

#define amdlibDET_SIZE_X  512
#define amdlibDET_SIZE_Y  512

#define amdlibBAD_PIXEL_FLAG  0
#define amdlibGOOD_PIXEL_FLAG 1

/* When a pixel is marked as bad, its value is set to the following value,
 * when generating calibrated images */
#define amdlibBAD_PIXEL_VALUE 0

#define amdlibMAX_NB_ROWS       3
#define amdlibMAX_NB_COLS       5
#define amdlibNB_PHOTO_CHANNELS 3
#define amdlibPHOTO1_CHANNEL    0
#define amdlibPHOTO2_CHANNEL    1
#define amdlibPHOTO3_CHANNEL    2
#define amdlibINTERF_CHANNEL    3

#define amdlibMAX_NB_SPEC 15
#define amdlibMAX_NB_PEAKS 10

#define amdlibFIRST_FRAME  1
#define amdlibALL_FRAMES  -1

#define amdlibDETECTOR_SATURATION 8000.0
/* Blanking value used when result is undefined */
#define amdlibBLANKING_VALUE 1.0e30

/* IMAGING_DETECTOR :
 * This table describes the setup and characteristics of the 2-dimensional
 * detector used to record photometric and visibility data */
typedef struct
{
    /** Origine of the data */
    char origin[16];
    /** Instrument name */
    char instrument[16];
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /** HDU creation date (YYY-MM-DDThh:mm:ss UTC) */
    char date[32];
    /** Data dictionary name */
    char detDictionaryId[16];
    /** DCS software version string */
    char detId[16];
    /** Number of detectors */
    int nbDetectors;
    /** Number of regions */
    int nbRegions;
    /** Maximum number of polynomial coefficients for DMC and DMP */
    int maxCoef;
    /** Number of dimensions of detector array */
    int numDim;
    /** Maximun number of telescopes for instrument */
    int maxTel;
} amdlibIMAGING_DETECTOR;

/* IMAGING_DATA :
 * This table contains the detector pixel data for an exposure */
typedef struct
{
    /** Origin of the data */
    char origin[16];
    /** Instrument name */
    char instrument[16];
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /** HDU creation date (YYY-MM-DDThh:mm:ss UTC) */
    char date[32];
    /** Data dictionary name */
    char detDictionaryId[16];
    /** DCS software version string */
    char detId[16];
    /** Number of regions */
    int nbRegions;
    /** Maximun number of telescopes for instrument */
    int maxTel;
    /** Maximum number of entries in OPTICAL_TRAIN table */
    int maxIns;
    /** Maximum number of steps in an observational cycle */
    int maxStep;
} amdlibIMAGING_DATA;

/* REGION : */
#define amdlibMAX_NB_AXIS 3
typedef struct
{
    /* The first part gives informations on the regions. These informations are
     * stored in the IMAGING_DETECTOR binary table */

    /*** Be careful, here you can replace 2 by amdlibMAX_NB_AXIS ***/

    /** Region number that is being described */
    int regionNumber;
    /** Descriptive name of the region; e.g. I1, I2, PA, PB or QL */
    char regionName[16];
    /** Detector that is on the region */
    int detectorNumber;
    /** Entrance ports of the interferometer that contribute to this region */
    int ports[amdlibNTEL];
    /** Correlation type, 0=background, 1=photometric and 2=interferometric */
    int correlation;
    /** Corner of this region on the detector as low value of x, low value of
     ** y, inclusive */
    int corner[2];
    /** Detector conversion factor from electron to ADU (e-/ADU) */
    double gain;
    /** Coordinates at the reference pixels */
    double crVal[2];
    /** Reference pixels on 'x' and 'y' axes for this region */
    double crPix[2];     
    /** Type de coordinates for the 'x' and 'y' axes for this region. Standard
     ** WCS conventions are used */
    char cType[2][8];
    /** Transformation applied after DMPn/DMCn transformations. Standard
     ** WCS conventions are used */
    double cd[2][2];
    /** Distorsion correction array */
    double ***dmp;
    /** Distorsion correction array */
    double ***dmc;

    /* The second part contains the data which are stored in the IMAGING_DATA
     * binary table */
    /** Dimension of each axis */
    int dimAxis[amdlibMAX_NB_AXIS];
    /** Data */
    float *data;
} amdlibREGION;

/* Position and dimension of spectral band (1 row) */
typedef struct
{
    /* Start pixel (in Y direction) of the row */
    int startPixel;
    /* Number of spectral channels */
    int nbSpecChannels;
} amdlibROW_INFO;

/* Position and dimension of photometric or interferometric channel 
 * (1 column) */
typedef struct
{
    /* Start pixel (in X direction) of the column */
    int startPixel;
    /* Number of pixels */
    int nbPixels;
} amdlibCOL_INFO;


/* AMBER_WAVEDATA :
 * This table contains the effective wavelengths/bandwidths for each spectral
 * channel for the interferometric data, and the wavelength offset for each
 * photometric column */
#define amdlibNB_SPECTRAL_CHANNELS   amdlibDET_SIZE_Y
#define amdlibOFFSETY_NOT_CALIBRATED 999.0
typedef struct
{
    /** Origine of the data */
    char origin[16];
    /** Instrument name */
    char instrument[16];
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /** HDU creation date (YYY-MM-DDThh:mm:ss UTC) */
    char date[32];
    /** Instrument dictionary name */
    char insDictionaryId[16];
    /** INS software version string */
    char insId[16];
    /** INS MODE string */
    char insMode[64];

    /* Number of wavelengths */
    int nbWlen;

    /* Arrays containing the wavelengths and bandwidth (microns) corresponding
     * to the interferometric data */
    float wlen[amdlibNB_SPECTRAL_CHANNELS];
    float bandwidth[amdlibNB_SPECTRAL_CHANNELS];

    /* Wavelength offset, in pixel, for each photometric column */
    float photoOffset[3];
    float interfOffset; /* Wavelength offset, in pixel, for the Interf column */
} amdlibAMBER_WAVEDATA;

/* Phase difference */ 
typedef enum
{
    amdlibNO_DELAY = 0,   /* No phase difference */
    amdlibDELAY1_2,       /* Phase difference between beams 1 and 2 */
    amdlibDELAY12_3       /* Phase difference between beams 1 and 3 or between
                           * beams 2 and 3 */
} amdlibPHASE;

/* Frame type **DO NOT CHANGE ORDER, JUST ADD AT END!!!!!** */
typedef enum 
{
    /* */
    amdlibDARK_FRAME = 0,
    /* for P2VM */
    amdlibTEL1_FRAME,
    amdlibTEL2_FRAME,
    amdlibTEL3_FRAME,
    amdlibTEL12_FRAME,
    amdlibTEL23_FRAME,
    amdlibTEL13_FRAME,
    amdlibTEL12Q_FRAME,
    amdlibTEL23Q_FRAME,
    amdlibTEL13Q_FRAME,
    amdlibTEL123_FRAME,
    /* for Spectral Calibration*/
    amdlibTEL1_SC_FRAME,
    amdlibTEL2_SC_FRAME,
    amdlibTEL3_SC_FRAME,
    /* others */
    amdlibUNKNOWN_FRAME
} amdlibFRAME_TYPE;

/* Simple definition of keyword */
#define amdlibNB_INS_CFG_KEYW 1024
#define amdlibKEYW_LINE_LEN 80
typedef struct 
{
    char name   [amdlibKEYW_LINE_LEN+1];
    char value  [amdlibKEYW_LINE_LEN+1];
    char comment[amdlibKEYW_LINE_LEN+1];
} amdlibKEYWORD;
typedef char amdlibKEYW_LINE[amdlibKEYW_LINE_LEN+1];

/* DPR keyword */
typedef struct
{
    char catg[32];
    char type[32];
    char tech[32];
} amdlibDPR;

#define amdlibNB_OF_SHUTTERS  3
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;

    /** Id of the P2VM which can be used to reduce this data */
    int p2vmId;

    /** Time for this exposure (Modified Julian Date) */
    double time;
    /** Dit time in seconds */
    float expTime;
    /* Number of spectral windows read (up to 3) */
    int nbRows;
    /* Number of columns */
    int nbCols;
    /* Number of frames */
    int nbFrames;
      
    /* Array of keywords corresponding to the instrument configuration at the
     * time of the data acquisition */
    int           nbKeywords;
    amdlibKEYWORD insCfg[amdlibNB_INS_CFG_KEYW];

    /* Wave data */
    amdlibAMBER_WAVEDATA   waveData;

    /* Imaging data and Imaging Detector informations */
    amdlibIMAGING_DETECTOR imagingDetector;
    amdlibIMAGING_DATA     imagingData;
   
    /* Number of regions = nbRows * nbCols */
    int                    nbRegions; 

    /* List of regions */
    amdlibREGION           *region;

    /* Time each Frame obtained (given by DCS in IMAGING_DATA */
    double *timeTag;
    
    /** Bias for each frame */
    float *globalBias;
    float *rmsBias;

    /* True if file only contains detector data; i.e. does not contains
     * ICS sub-system keywords nor wave data binary table */
    amdlibBOOLEAN detDataOnly;
    
    /* True if the data has been loaded */ 
    amdlibBOOLEAN dataLoaded;
    
    /* True if the data has been already calibrated */
    amdlibBOOLEAN dataCalibrated;

    /* True if the raw data has [a number of] saturated pixels */ 
    amdlibBOOLEAN dataIsSaturated;
    
    /* Phase difference */
    amdlibPHASE delay;

    /* Shutter positions */
    int    shutterPos[amdlibNB_OF_SHUTTERS];

    /* Frame type */
    amdlibFRAME_TYPE frameType;

    /* DPR keyword */
    amdlibDPR dpr;

} amdlibRAW_DATA;


/** CALIBRATION DATA */
typedef struct
{
    /* Flat field map */
    float data[amdlibDET_SIZE_X * amdlibDET_SIZE_Y];
} amdlibFLAT_FIELD_MAP;

typedef struct
{
    /* Bad pixel map */
    float data[amdlibDET_SIZE_X * amdlibDET_SIZE_Y];
} amdlibBAD_PIXEL_MAP;

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;

    /* Dit time in seconds. Bias MUST have same dit as frame
     * to which it is applied */
    float expTime;

    /* Number of spectral windows and columns */ 
    int nbRows;      
    int nbCols;      

    /* Imaging Detector informations */
    amdlibIMAGING_DETECTOR imagingDetector;
      
    /* Number of regions = nbRows * nbCols */
    int                    nbRegions; 

    /* List of regions */
    amdlibREGION           *region;

    /* Time each Frame obtained (given by DCS in IMAGING_DATA */
    double *timeTag;

} amdlibPIXEL_BIAS_DATA;

/** SIENCE DATA */

/* Science data corresponding to 1 row */
typedef struct 
{   
    /* Array contening the interferometric data for this row */
    double *intf;

    /* Arrays contening the photometric data for this row and corresp variance*/
    double *photo1;
    double *photo2;
    double *photo3;
    double *Sigma2Photo1;
    double *Sigma2Photo2;
    double *Sigma2Photo3;
  
    /* integrated values and corresponding SNR */
    float snrPhoto1;
    float snrPhoto2;
    float snrPhoto3;
    float integratedPhoto1;
    float integratedPhoto2;
    float integratedPhoto3;
} amdlibROW_SCIENCE_DATA;

typedef struct 
{   
    /* Flux Ratio */
    float fluxRatio[amdlibNB_PHOTO_CHANNELS];

    /* Read-out Noise */
    float ron;

    /* read time in modified julian days */
    /* double time; */
    
    /* Arrays containing data of all rows defined in frame */
    amdlibROW_SCIENCE_DATA row[amdlibMAX_NB_ROWS];
} amdlibFRAME_SCIENCE_DATA;

/* Spectral dispersion. Since the SPECTRAL CALIBRATION Changes with AMBER setup
 * and Time, it is necessary to TAG this table with an unique identifier,
 * because the OI-FITS format will need to refer to this identifier*/
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of wavelengths */
    int nbWlen;
    /* Id of this wavelength once calibrated. Just a copy of the P2VM which
     * can be used to reduce this data */
    int id;
    float *wlen;
    float *bandwidth;
} amdlibWAVELENGTH;

typedef struct 
{   
    /* Arrays contening spectral dispersion of all rows defined in frame */
    amdlibWAVELENGTH row[amdlibMAX_NB_ROWS];
} amdlibSPEC_DISPERSION;

/* Whole science data */
typedef struct 
{   
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;

    /* Id of the P2VM which can be used to reduce this data */
    int p2vmId;

    /* Array of keywords corresponding to the instrument configuration at the
     * time of the data acquisition for P2VM calibration */
    int           nbKeywords;
    amdlibKEYWORD insCfg[amdlibNB_INS_CFG_KEYW];

    /* Integration Time per frame (supposedly globally constant) */ 
    float expTime;    
    
    /* Time each Frame obtained (given by DCS in IMAGING_DATA */
    double *timeTag;

    /* Number of columns (3 or 4) */ 
    int nbCols;      

    /* Dimension and position of columns (shielded area not in science data) */
    amdlibCOL_INFO      col[amdlibMAX_NB_COLS-1];

    /* Number of spectral windows */ 
    int nbRows;      

    /* Dimension and position of rows */
    amdlibROW_INFO      row[amdlibMAX_NB_ROWS];

    /* Arrays contening the spectral dispersion data */
    amdlibSPEC_DISPERSION    specDispersion;

    /* Number of frames */ 
    int nbFrames;    

    /* Arrays contening the frames */
    amdlibFRAME_SCIENCE_DATA *frame;

} amdlibSCIENCE_DATA;

/** Data structures for the Spectral calibration computation */
#define amdlibMAX_NB_FRAMES_SC 3
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Science data */
    amdlibBOOLEAN      dataLoaded[amdlibMAX_NB_FRAMES_SC];
    amdlibSCIENCE_DATA scienceData[amdlibMAX_NB_FRAMES_SC];
} amdlibSC_INPUT_DATA;
/* P2VM type */
typedef enum 
{
    amdlibSC_2T = 1,
    amdlibSC_3T
} amdlibSC_TYPE;


/** Data structures for the P2VM computation */
/* Input data used to compute the P2VM */
#define amdlibMAX_NB_FRAMES_P2VM 9
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    
    /* Id of the P2VM to be computed */
    int p2vmId;

    /* Array containing the visibility of the calibrator */
    double visibilityC[amdlibNB_SPECTRAL_CHANNELS];

    /* Science data */
    amdlibBOOLEAN      dataLoaded[amdlibMAX_NB_FRAMES_P2VM];
    amdlibSCIENCE_DATA scienceData[amdlibMAX_NB_FRAMES_P2VM];
} amdlibP2VM_INPUT_DATA;

/* P2VM type */
typedef enum 
{
    amdlibP2VM_2T = 1,
    amdlibP2VM_3T
} amdlibP2VM_TYPE;

/* P2VM accuracy */
typedef enum 
{
    amdlibP2VM_UNKNOWN_ACC = 0,
    amdlibP2VM_STD_ACC,
    amdlibP2VM_HIGH_ACC
} amdlibP2VM_ACCURACY;

/* P2VM */
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
 
    /* Unique P2VM Id */
    int id;

    /* Array of keywords corresponding to the instrument configuration at the
     * time of the data acquisition for P2VM calibration */
    amdlibKEYWORD insCfg[amdlibNB_INS_CFG_KEYW];

    /* P2VM type (amdlibP2VM_2T or amdlibP2VM_3T) */
    amdlibP2VM_TYPE type;

    /* P2VM accuracy (amdlibP2VM_STD_ACC or amdlibP2VM_HIGH_ACC) */
    amdlibP2VM_ACCURACY accuracy;

    /* Start pixel (in Y direction) of the row corresponding to the P2VM */
    int    startPixel;
    /* Number of pixels in column */
    int    nx;
    /* Number of spectral channels */
    int    nbSpecChannels;  
    /* Array containing the wavelengths */
    float  *wlen;       /* [nbSpecChannels] */
    /* Array containing the ck and dk coefficients */
    double *matrix;     /* [nbSpecChannels][nx][2*nbBase]*/
    /* Arrays containing the vk coefficients */
    double *vk;         /* [nbTel][nbSpecChannels][nx] */
    double *sumVk;      /* [nbBase][nbSpecChannels] */
    /* Array containing the bad pixel flags corresponding to the
     * interferometric channel */
    unsigned char *badPixels;   /* [nbSpecChannels][nx] */
    /* Array containing the Flat field corresponding to the 
     * interferometric channel*/
    float *flatField;   
    /* Arrays containing the photometric data */
    double *photometry; /* [2*nbBase+1][3][nbSpecChannels] */
    /* Arrays containing the validity flag */
    unsigned char *flag;        /* [nbSpecChannels]*/
    /* Array containing the phase shift */
    double *phase;      /* [nbBase][nbSpecChannels] */
    /* Arrays containing the instrument visibilities */
    float insVis[amdlibNBASELINE];
    float insVisErr[amdlibNBASELINE];
} amdlibP2VM_MATRIX;

/* Complex */
typedef struct
{
    double re;
    double im;
} amdlibCOMPLEX;

typedef struct
{
    char telescopeName[16]; /*Telescope name*/
    char stationName[16]; /*Station name*/
    int stationIndex; /*Station number*/
    float elementDiameter; /*Element diameter (meters)*/
    double stationCoordinates[3]; /*Station coordinates relative to array center (meters)*/
} amdlibOI_ARRAY_ELEMENT;

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    int  nbStations; /*Number of stations defined*/
    char arrayName[16]; /*Array name, for cross-referencing*/
    char coordinateFrame[12]; /*Coordinate frame*/
    double arrayCenterCoordinates[3]; /*Array center coordinates (meters)*/
    amdlibOI_ARRAY_ELEMENT *element;
} amdlibOI_ARRAY;

typedef struct
{
    int targetId;
    char targetName[32];
    double raEp0;
    double decEp0;
    float equinox;
    double raErr;
    double decErr;
    double sysVel;
    char velTyp[32];
    char velDef[32];
    double pmRa;
    double pmDec;
    double pmRaErr;
    double pmDecErr;
    double parallax;
    float paraErr;
    char specTyp[32];

} amdlibOI_TARGET_ELEMENT;
typedef struct
{
    void *thisPtr;
    int  nbTargets; /*Number of stations defined*/
    amdlibOI_TARGET_ELEMENT *element;
} amdlibOI_TARGET;

typedef struct
{
    /** Target ID */
    int targetId;
    /** UTC time of observation */
    double time;
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Integration Time per frame (supposedly globally constant) */ 
    float expTime;    
    /** U coord of this visibility measurement */
    double uCoord;
    /** V coord of this visibility measurement */
    double vCoord;
    /** Stations Number contributing to the Data */
    int stationIndex[2];
  
    /** Array[nbWlen] containing the Visibility Expressed as a Complex */
    amdlibCOMPLEX *vis;
    /** Array[nbWlen] containing the corresponding variance */
    amdlibCOMPLEX *sigma2Vis;
    /** Array[nbWlen] containing the covariance
     ** between Real and Imaginary part of the visibilities */
    double *visCovRI;
  
    /** Criterion so say if there are fringes or not */
    double frgContrastSnr;
  
} amdlibVIS_TABLE_ENTRY;

typedef struct
{
    /** Target ID */
    int targetId;
    /** UTC time of observation */
    double time;
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Integration Time per frame (supposedly globally constant) */ 
    float expTime;   
    /** Array[nbWlen] containing the number of electrons collected
     ** in the spectral channel */
    double *fluxSumPiPj;
    double *sigma2FluxSumPiPj;
    /** Array[nbWlen] containing the flux ratio
     ** in the spectral channel */
    double *fluxRatPiPj;
    double *sigma2FluxRatPiPj;
    /** Array[nbWlen] containing the photometric correction on the mks
     ** in the spectral channel */
    double *PiMultPj; 
    /** U coord of this visibility measurement */
    double uCoord;
    /** V coord of this visibility measurement */
    double vCoord;
    /** Stations Number contributing to the Data */
    int stationIndex[2];

} amdlibPHOTOMETRY_TABLE_ENTRY;

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of frames */
    int nbFrames;
    /* Number of Baselines */
    int nbBases;
    /* Number of wavelengths */
    int nbWlen;
    
    amdlibPHOTOMETRY_TABLE_ENTRY *table;
    
} amdlibPHOTOMETRY;
  
typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of frames */
    int nbFrames;
    /* Number of Baselines */
    int nbBases;
    /* Number of wavelengths */
    int nbWlen;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /* IAU_VIS structure with nbSamples components */
    amdlibVIS_TABLE_ENTRY *table;
} amdlibVIS;

/* IAU_VIS2 structure. According to last IAU Draft, 
 * that should update the ESO one about OI_VIS2 
 */
typedef struct
{
    /** Target ID */
    int targetId;
    /** UTC time of observation */
    double time;
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Integration Time per frame (supposedly globally constant) */ 
    float expTime;    
    /** Array containing the Squared Visibility */
    double  *vis2;
    /** Array containing the corresponding error */
    double *vis2Error;
    /** U coord of this visibility measurement */
    double uCoord;
    /** V coord of this visibility measurement */
    double vCoord;
    /** Stations Number contributing to the Data */
    int stationIndex[2];
} amdlibVIS2_TABLE_ENTRY;

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of frames */
    int nbFrames;
    /* Number of Baselines */
    int nbBases;
    /* Number of wavelengths */
    int nbWlen;
    /** Mean visibility with corresponding error */
    float vis12;
    float vis23;
    float vis31;
    float sigmaVis12;
    float sigmaVis23;
    float sigmaVis31;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /* IAU_VIS2 structure with nbWlen components */
    amdlibVIS2_TABLE_ENTRY *table;
} amdlibVIS2;  

/* Bias computation type */ 
typedef enum
{ 
    /* Phasors algorythm, very slow and maybe inaccurate */
    amdlibSTATISTICAL_ERROR = 0,  
    /* Fit of the unwrapped phase */
    amdlibTHEORETICAL_ERROR
} amdlibERROR_TYPE;

/* IAU_VIS3 structure. According to last IAU Draft, 
 * that should update the ESO one about OI_VIS3 
 */
typedef struct
{
    /** Target ID */
    int targetId;
    /** UTC time of observation */
    double time;
    /** Observation date (Modified Julian Date) */
    double dateObsMJD;
    /** Integration Time per frame (supposedly globally constant) */ 
    float expTime;    
    /** Array[nbWlen] containing the Triple Product Amplitude */
    double  *vis3Amplitude;
    /** Array[nbWlen] containing the corresponding error */
    double *vis3AmplitudeError;
    /** Array[nbWlen] containing the Triple Product Phase (degrees)*/
    double  *vis3Phi;
    /** Array[nbWlen] containing the corresponding error (degrees)*/
    double *vis3PhiError;
    /** U1 coord of baseline AB of the triangle */
    double u1Coord;
    /** V1 coord of baseline AB of the triangle */
    double v1Coord;
    /** U2 coord of baseline BC of the triangle */
    double u2Coord;
    /** V2 coord of baseline BC of the triangle */
    double v2Coord;
    /** Stations Number contributing to the triple product (3 values) */
    int stationIndex[3];
} amdlibVIS3_TABLE_ENTRY;

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of frames */
    int nbFrames;
    /* Number of phase closures */
    int nbClosures;
    /* Number of wavelengths */
    int nbWlen;
    /* Returns the closure averaged over all frames and wavelength */
    float averageClosure;
    float averageClosureError;
    /** Observation date (YYY-MM-DDThh:mm:ss UTC) */
    char dateObs[32];
    /* IAU_VIS3 structure with nbWlen components */
    amdlibVIS3_TABLE_ENTRY *table;
} amdlibVIS3;  

typedef struct
{
    /* The following pointer is used to know if the data structure is
     * initialized or not. When initialized, it contains pointer to itself */
    void *thisPtr;
    /* Number of frames */
    int nbFrames;
    /* Number of Baselines */
    int nbBases;
    /* Piston values for each baseline in same unit as lambda */
    float *pistonOPD;
    /* Error on Piston same unit as above */
    float *sigmaPiston;
} amdlibPISTON;

/* Piston algorythm */ 
typedef enum
{ 
    /* Phasors algorythm, very slow and maybe inaccurate */
    amdlibITERATIVE_PHASOR = 0,  
    /* Fit of the unwrapped phase */
    amdlibUNWRAPPED_PHASE
} amdlibPISTON_ALGORITHM;

/** Error message  */
typedef char amdlibERROR_MSG[256];

/**  Definition of the routine completion status */
typedef enum
{
    amdlibFAILURE = 1,
    amdlibSUCCESS
} amdlibCOMPL_STAT;


/** Function prototypes */
void amdlibStripBlanks(char *str);
void amdlibStripQuotes(char *str);
amdlibCOMPL_STAT amdlibSplitFits(amdlibKEYW_LINE keywLine, 
                                 amdlibKEYWORD *keyword);
/* Loading raw data functions */
void amdlibInitRawData(amdlibRAW_DATA *rawData);
amdlibCOMPL_STAT amdlibGetWaveData (fitsfile             *filePtr, 
                                    amdlibAMBER_WAVEDATA *waveData, 
                                    amdlibERROR_MSG      errMsg);
void amdlibReleaseWaveData(amdlibAMBER_WAVEDATA *waveData);
amdlibCOMPL_STAT amdlibSaveWaveData(const char           *filename,
                                    amdlibAMBER_WAVEDATA *waveData,
                                    amdlibERROR_MSG      errMsg);

amdlibCOMPL_STAT amdlibGetImagingDetectorHdr
(fitsfile               *filePtr, 
 amdlibIMAGING_DETECTOR *imagingDetector, 
 amdlibERROR_MSG        errMsg);

amdlibCOMPL_STAT amdlibGetImagingDataHdr
(fitsfile           *filePtr, 
 amdlibIMAGING_DATA *imagingData, 
 amdlibERROR_MSG    errMsg);

amdlibCOMPL_STAT amdlibAllocateRegions(amdlibREGION **regions,
                                       int          nbRegions);
void amdlibFreeRegions(amdlibREGION **regions, int  nbRegions );
amdlibCOMPL_STAT amdlibGetRegionInfo (fitsfile        *filePtr, 
                                      amdlibREGION    **regions, 
                                      int             nbRegions,
                                      int             nbTel,
                                      amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibGetRegionTimeTag (fitsfile        *filePtr, 
                                         amdlibREGION    *regions, 
                                         double          *time,
                                         int             firstFrame,
                                         int             nbFrames,
                                         amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibGetRegionData (fitsfile        *filePtr, 
                                      amdlibREGION    *regions, 
                                      int             nbRegions,
                                      int             firstFrame,
                                      int             nbFrames,
                                      amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibSetRegionData (fitsfile        *filePtr, 
                                      amdlibREGION    *regions, 
                                      int             nbRegions,
                                      amdlibERROR_MSG  errMsg);
amdlibCOMPL_STAT amdlibSetRegionTimeTag (fitsfile        *filePtr, 
                                         double          *time,
                                         int             nbFrames,
                                         amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibCopyRegionInfo (amdlibREGION    *srcRegions, 
                                       amdlibREGION    *dstRegions, 
                                       int             nbRegions,
                                       amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibCopyRegionData (amdlibREGION    *srcRegions, 
                                       amdlibREGION    *dstRegions, 
                                       int             nbRegions,
                                       amdlibERROR_MSG errMsg);

void amdlibReleaseRawData(amdlibRAW_DATA  *rawData);
amdlibCOMPL_STAT amdlibLoadRawData(const char      *filename,
                                   amdlibRAW_DATA  *rawData,
                                   amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibLoadRawDataHdr(const char      *filename,
                                      amdlibRAW_DATA  *rawData,
                                      amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibLoadRawFrames(const char      *filename,
                                     amdlibRAW_DATA  *rawData,
                                     int             firstFrame,
                                     int             nbFrames,
                                     amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibCopyRawDataFile(const char      *srcFilename,
                                       const char      *dstFilename,
                                       amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibStoreRawData(const char      *filename,
                                    amdlibRAW_DATA  *rawData,
                                    amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibExtractColPos(amdlibRAW_DATA *rawData, 
                                     float colPos[amdlibMAX_NB_COLS],
                                     float maxVal[amdlibMAX_NB_COLS]);
amdlibCOMPL_STAT amdlibExtractSpectPos(amdlibRAW_DATA *rawData, 
                                       float spectPos[amdlibMAX_NB_COLS][amdlibMAX_NB_ROWS]);
amdlibCOMPL_STAT amdlibGluedImage2RawData(float *gluedImage,
                                          int   nbRows,
                                          int   nbCols,
                                          int   colWidth[amdlibMAX_NB_COLS],
                                          int   rowHeight[amdlibMAX_NB_ROWS],
                                          amdlibRAW_DATA  *rawData,
                                          amdlibERROR_MSG errMsg);
amdlibBOOLEAN amdlibIsSameDetCfg(amdlibRAW_DATA *rawData1, 
                                 amdlibRAW_DATA *rawData2);
amdlibCOMPL_STAT amdlibSaveRawDataToFits(const char      *filename,
                                         amdlibRAW_DATA  *rawData,
                                         amdlibERROR_MSG errMsg);

/* Calibration data functions */
amdlibCOMPL_STAT amdlibLoadBadPixelMap(const char           *filename,
                                       amdlibBAD_PIXEL_MAP  *badPixels,
                                       amdlibERROR_MSG      errMsg);
amdlibCOMPL_STAT amdlibSetBadPixelMap(amdlibBAD_PIXEL_MAP *badPixels,
                                      amdlibBOOLEAN       goodPixel);

amdlibCOMPL_STAT amdlibLoadFlatFieldMap(const char           *filename,
                                        amdlibFLAT_FIELD_MAP *flatField,
                                        amdlibERROR_MSG      errMsg);
amdlibCOMPL_STAT amdlibSetFlatFieldMap(amdlibFLAT_FIELD_MAP  *flatField,
                                       double value);
amdlibCOMPL_STAT amdlibCheckFlatFieldMap(amdlibFLAT_FIELD_MAP *flatField,
                                         amdlibBAD_PIXEL_MAP  *badPixels,
                                         float                minGain,
                                         float                maxGain,
                                         amdlibBOOLEAN        correct,
                                         amdlibERROR_MSG      errMsg);

void amdlibReleasePixelBiasData(amdlibPIXEL_BIAS_DATA *pixelBias);
amdlibCOMPL_STAT amdlibLoadPixelBiasData(const char             *filename,
                                         amdlibPIXEL_BIAS_DATA  *pixelBias,
                                         amdlibERROR_MSG        errMsg);
amdlibCOMPL_STAT amdlibGeneratePixelBiasData(amdlibRAW_DATA        *rawData,
                                             amdlibPIXEL_BIAS_DATA *pixelBias,
                                             amdlibERROR_MSG       errMsg);
amdlibCOMPL_STAT amdlibSetPixelBiasData (amdlibRAW_DATA        *rawData,
                                         amdlibPIXEL_BIAS_DATA *pixelBias,
                                         float                 value,
                                         amdlibERROR_MSG       errMsg);

amdlibCOMPL_STAT amdlibCalibrateRawData(amdlibBAD_PIXEL_MAP   *badPixels,
                                        amdlibFLAT_FIELD_MAP  *flatField,
                                        amdlibPIXEL_BIAS_DATA *pixelBias,
                                        amdlibRAW_DATA        *rawData,
                                        int                   gaussianWidth,
                                        float                 gaussianFwhm,
                                        amdlibERROR_MSG       errMsg);
/* Science data functions */
void amdlibFreeScienceData(amdlibSCIENCE_DATA *scienceData);
void amdlibReleaseScienceData(amdlibSCIENCE_DATA  *scienceData);
amdlibCOMPL_STAT amdlibRawData2ScienceData(amdlibRAW_DATA     *rawData,
                                           amdlibSCIENCE_DATA *scienceData,
                                           amdlibBAD_PIXEL_MAP   *badPixels,
                                           amdlibFLAT_FIELD_MAP  *flatField,
                                           amdlibBOOLEAN      sumData,
                                           amdlibERROR_MSG    errMsg);
amdlibCOMPL_STAT amdlibCopyScienceData (amdlibSCIENCE_DATA *srcScienceData, 
                                        amdlibSCIENCE_DATA *dstScienceData, 
                                        amdlibERROR_MSG    errMsg);
amdlibCOMPL_STAT amdlibExplodeJHKScienceData (amdlibSCIENCE_DATA *srcScienceData, 
                                              amdlibSCIENCE_DATA *dstScienceData, 
                                              amdlibP2VM_MATRIX *p2vm,
                                              amdlibERROR_MSG    errMsg);

/* Spectral calibration data functions */
void amdlibReleaseSpectralCalibrationData
(amdlibSC_INPUT_DATA *SpectralCalibrationData);
amdlibCOMPL_STAT amdlibAddToSpectralCalibrationData 
(amdlibRAW_DATA      *rawData,
 amdlibSC_INPUT_DATA *SpectralCalibrationData,
 amdlibERROR_MSG       errMsg);

/* Spectral calibration function */
amdlibCOMPL_STAT amdlibPerformSpectralCalibration
(amdlibSC_INPUT_DATA    *scData,
 amdlibSC_TYPE          scType,
 amdlibAMBER_WAVEDATA   *calWaveData,
 amdlibERROR_MSG        errMsg);
/* P2VM data functions */
void amdlibFreeP2vmData(amdlibP2VM_INPUT_DATA *p2vmData);
void amdlibReleaseP2vmData(amdlibP2VM_INPUT_DATA *p2vmData);
amdlibCOMPL_STAT amdlibAddToP2vmData (amdlibRAW_DATA        *rawData,
                                      amdlibP2VM_INPUT_DATA *p2vmData,
                                      float *newSpectralOffsets,
                                      amdlibERROR_MSG       errMsg);
/* P2VM computation functions */
void amdlibReleaseP2VM(amdlibP2VM_MATRIX *p2vm);
amdlibCOMPL_STAT amdlibComputeP2VM(amdlibP2VM_INPUT_DATA *p2vmData,
                                   amdlibBAD_PIXEL_MAP   *badPixel,
                                   amdlibFLAT_FIELD_MAP  *flatField,
                                   amdlibP2VM_TYPE       p2vmType,
                                   amdlibP2VM_MATRIX     *p2vm,
                                   amdlibERROR_MSG       errMsg);
amdlibBOOLEAN amdlibIsP2VMUsable(amdlibSCIENCE_DATA *scienceData,
                                 amdlibP2VM_MATRIX  *p2vm,
                                 float              *percentage);
amdlibCOMPL_STAT amdlibSaveP2VM(const char          *filename,
                                amdlibP2VM_MATRIX   *p2vm,
                                float *newSpectralOffsets,
                                amdlibP2VM_ACCURACY accuracy,
                                amdlibERROR_MSG     errMsg);
amdlibCOMPL_STAT amdlibLoadP2VM(const char        *filename,
                                amdlibP2VM_MATRIX *p2vm,
                                float *newSpectralOffsets,
                                amdlibERROR_MSG   errMsg);
amdlibCOMPL_STAT amdlibCopyP2VM(amdlibP2VM_MATRIX *srcP2vm, 
                                amdlibP2VM_MATRIX *dstP2vm, 
                                amdlibERROR_MSG   errMsg);

/* Matrix utilities */
amdlibCOMPL_STAT amdlibInvertMatrix(double *matrix, 
                                    int dim);
void amdlibTransposeMatrix(double *matrix, 
                           double *tmatrix, 
                           int nx, 
                           int ny);
amdlibCOMPL_STAT amdlibProductMatrix(double *matrix1, 
                                     double *matrix2, 
                                     double *matprod, 
                                     int dim1, 
                                     int dim2, 
                                     int dim3);
void amdlibComputeCov(double *x, 
                      double *y, 
                      int lambda, 
                      int nbWlen, 
                      int nbFrames, 
                      int nbBase,
                      double *cov_xy);
int  amdlibFindIndexOfMaximum(float *data, int N);

/*Allocate, Deallocate and release Vis-related structures*/
/**** Declaration of local functions */
/* 
 * The function amdlibFreeVis() frees the memory allocated within an amdlibVIS
 * structure.
 */
void amdlibFreeVis(amdlibVIS *vis);

/*
 * The function amdlibAllocateVis() allocates all the amdlibVIS structure for
 * further use by amdlibExtractVisibilities.
 */
amdlibCOMPL_STAT amdlibAllocateVis(amdlibVIS *vis, const int nbFrames,
                                   const int nbBases, const int nbWlen);
/*
 * The function amdlibFreeVis2() frees the memory allocated within an
 * amdlibVIS2 structure.
 */
void amdlibFreeVis2(amdlibVIS2 *vis2);

/*
 * The function amdlibAllocateVis2() allocates all the amdlibVIS2 structure
 * for further use by amdlibExtractisibilities.
 */
amdlibCOMPL_STAT amdlibAllocateVis2(amdlibVIS2 *vis2, const int nbFrames,
                                    const int nbBases, const int nbWlen);
/* 
 * The function amdlibFreeVis3() frees the memory allocated within an
 * amdlibVIS3 structure.
 */
void amdlibFreeVis3(amdlibVIS3 *vis3);

/*     
 * The function amdlibAllocateVis3() allocates all the amdlibVIS3 structure
 * for further use by amdlibExtractVisibilities.
 */
amdlibCOMPL_STAT amdlibAllocateVis3(amdlibVIS3 *vis3, const int nbFrames,
                                    const int nbClosures,
                                    const int nbWlen);

/*
 * The function amdlibFreeWavelength() frees the memory allocated within an
 * amdlibWAVELENGTH structure.
 */
void amdlibFreeWavelength(amdlibWAVELENGTH *wave);

/*
 * The function amdlibAllocateWavelength() allocates all the amdlibWAVELENGTH
 * structure for further use by amdlibExtractVisibilities.
 */
amdlibCOMPL_STAT amdlibAllocateWavelength(amdlibWAVELENGTH *wave,
                                          const int nbWlen);

/*
 * The function amdlibFreePiston() frees the memory allocated within an
 * amdlibPISTON structure.
 */
void amdlibFreePiston(amdlibPISTON *piston);

/*
 * The function amdlibAllocatePiston() allocates all the amdlibPISTON
 * structure for further use by amdlibExtractVisibilities.
 */
amdlibCOMPL_STAT amdlibAllocatePiston(amdlibPISTON *piston,
                                      const int nbFrames,
                                      const int nbBases);

amdlibCOMPL_STAT amdlibAllocatePhotometry(amdlibPHOTOMETRY *photometry,
                                          const int nbFrames,
                                          const int nbBases,
                                          const int nbWlen);

void amdlibFreeOiTarget(amdlibOI_TARGET *target);
amdlibCOMPL_STAT amdlibAllocateOiTarget(amdlibOI_TARGET *target,
                                        const int nbElements);
void amdlibFreeOiArray(amdlibOI_ARRAY *array);
amdlibCOMPL_STAT amdlibAllocateOiArray(amdlibOI_ARRAY *array,
                                       const int nbElements);

void amdlibReleasePhotometry(amdlibPHOTOMETRY *photometry);
void amdlibReleaseVis(amdlibVIS *vis);
void amdlibReleaseVis2(amdlibVIS2 *vis2);
void amdlibReleaseVis3(amdlibVIS3 *vis3);
void amdlibReleaseWavelength(amdlibWAVELENGTH *wave);
void amdlibReleasePiston(amdlibPISTON *opd);
void amdlibReleaseOiArray(amdlibOI_ARRAY *array);
void amdlibReleaseOiTarget(amdlibOI_TARGET *target);

amdlibCOMPL_STAT amdlibCheckInsConfig(/* Input */
                                      amdlibSCIENCE_DATA     *data,
                                      amdlibSCIENCE_DATA     *sky,
                                      amdlibP2VM_MATRIX      *mv2p,
                                      amdlibBOOLEAN          noCheckP2vmId,
                                      int                    iRow, 
                                      /* Output */
                                      amdlibERROR_MSG        errMsg);

amdlibCOMPL_STAT amdlibExtractVisibilities(/* Input */
                                           amdlibSCIENCE_DATA     *data,
                                           amdlibSCIENCE_DATA     *sky,
                                           amdlibP2VM_MATRIX      *p2vm,
                                           int                    iRow,
                                           int                    nbBinning,
                                           amdlibERROR_TYPE       errorType,
                                           amdlibPISTON_ALGORITHM pistonType,
                                           amdlibBOOLEAN         noCheckP2vmId,
                                           /* Output */
                                           amdlibPHOTOMETRY       *photometry,
                                           amdlibVIS              *vis,
                                           amdlibVIS2             *vis2,
                                           amdlibVIS3             *vis3,
                                           amdlibWAVELENGTH       *wave,
                                           amdlibPISTON           *opd,
                                           amdlibERROR_MSG        errMsg);

amdlibCOMPL_STAT amdlibCopyPhotometry (/* Input */
                                       amdlibPHOTOMETRY *srcPhotometry, 
                                       /* Output */
                                       amdlibPHOTOMETRY *dstPhotometry, 
                                       amdlibERROR_MSG   errMsg);

amdlibCOMPL_STAT amdlibCopyVis(/* Input */
                               amdlibVIS       *srcVis, 
                               /* Output */
                               amdlibVIS       *dstVis, 
                               amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibCopyVis2(/* Input */
                                amdlibVIS2      *srcVis2, 
                                /* Output */
                                amdlibVIS2      *dstVis2, 
                                amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibCopyVis3(/* Input */
                                amdlibVIS3      *srcVis3, 
                                /* Output */
                                amdlibVIS3      *dstVis3, 
                                amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibCopyWavelength (/* Input */
                                       amdlibWAVELENGTH *srcWlen, 
                                       /* Output */
                                       amdlibWAVELENGTH *dstWlen, 
                                       amdlibERROR_MSG   errMsg);

amdlibCOMPL_STAT amdlibCopyPiston(/* Input */
                                  amdlibPISTON    *srcOpd, 
                                  /* Output */
                                  amdlibPISTON    *dstOpd, 
                                  amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteOiFile(const char           *filename,
                                   amdlibOI_ARRAY       *array,
                                   amdlibOI_TARGET      *target,
                                   amdlibPHOTOMETRY     *photometry,
                                   amdlibVIS            *vis,
                                   amdlibVIS2           *vis2,
                                   amdlibVIS3           *vis3,
                                   amdlibWAVELENGTH     *wave, 
                                   amdlibPISTON         *pst,
                                   amdlibERROR_MSG      errMsg);

amdlibCOMPL_STAT amdlibWriteOiArray(fitsfile        *filePtr,
                                    amdlibOI_ARRAY   *array,
                                    amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteOiTarget(fitsfile        *filePtr,
                                     amdlibOI_TARGET *target,
                                     amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteOiWavelength (fitsfile             *filePtr,
                                          char                 *insName,
                                          amdlibWAVELENGTH     *wave, 
                                          amdlibERROR_MSG      errMsg);

amdlibCOMPL_STAT amdlibWriteOiVis(fitsfile        *filePtr, 
                                  char            *insName,
                                  char            *arrName,
                                  amdlibVIS       *vis, 
                                  amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteOiVis2(fitsfile         *filePtr, 
                                   char            *insName,
                                   char            *arrName,
                                   amdlibVIS2       *vis2, 
                                   amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteOiVis3(fitsfile        *filePtr, 
                                   char            *insName,
                                   char            *arrName,
                                   amdlibVIS3      *vis3, 
                                   amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibWriteAmberData(fitsfile        *filePtr,
                                      char            *insName,
                                      amdlibPHOTOMETRY  *photometry,
                                      amdlibVIS       *vis,
                                      amdlibPISTON    *pst,
                                      amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadOiFile(const char           *filename,
                                  amdlibOI_ARRAY       *array,
                                  amdlibOI_TARGET      *target,
                                  amdlibPHOTOMETRY     *photometry,
                                  amdlibVIS            *vis,
                                  amdlibVIS2           *vis2,
                                  amdlibVIS3           *vis3,
                                  amdlibWAVELENGTH     *wave,
                                  amdlibPISTON         *pst,
                                  amdlibERROR_MSG      errMsg);

amdlibCOMPL_STAT amdlibReadOiWavelength (fitsfile             *filePtr,
                                         amdlibWAVELENGTH     *wave,
                                         amdlibERROR_MSG      errMsg);

amdlibCOMPL_STAT amdlibReadOiVis (fitsfile        *filePtr,
                                  amdlibVIS       *vis,
                                  int             nbBases,
                                  amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadOiVis2 (fitsfile        *filePtr,
                                   amdlibVIS2      *vis2,
                                   int             nbBases,
                                   amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadOiVis3 (fitsfile        *filePtr,
                                   amdlibVIS3      *vis3,
                                   amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadAmberData (fitsfile        *filePtr,
                                      amdlibPHOTOMETRY  *photometry,
                                      amdlibVIS       *vis,
                                      amdlibPISTON    *pst,
                                      int             nbBases,
                                      amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadOiTarget( fitsfile *filePtr, 
                                     amdlibOI_TARGET *target, 
                                     amdlibERROR_MSG errMsg);

amdlibCOMPL_STAT amdlibReadOiArray(fitsfile *filePtr,
                                   amdlibOI_ARRAY *array,
                                   amdlibERROR_MSG errMsg );

amdlibCOMPL_STAT amdlibReadEsoArrayGeometry(const char     *filename,
                                            amdlibOI_ARRAY *array,
                                            amdlibERROR_MSG errMsg );

/* High level functions */
amdlibCOMPL_STAT amdlibGenerateCalImage(const char *badPixelFile,
                                        const char *flatFieldFile,
                                        const char *darkFile,
                                        const int  oneOverNoiseCorrection,
                                        const char *inputFile,
                                        const char *outputFile);
amdlibCOMPL_STAT amdlibBtbl2Fits(const char *inputFile,
                                 const char *outputFile);

amdlibCOMPL_STAT  amdlibComputeP2vm2T(const char *badPixelFile,
                                      const char *flatFieldFile,
                                      const char *biasFile,
                                      const char *inputFile1,
                                      const char *inputFile2,
                                      const char *inputFile3,
                                      const char *inputFile4,
                                      const char *p2vmFile,
                                      float *newSpectralOffsets);

amdlibCOMPL_STAT amdlibComputeP2vm3T(const char *badPixelFile,
                                     const char *flatFieldFile,
                                     const char *biasFile,
                                     const char *inputFile1,
                                     const char *inputFile2,
                                     const char *inputFile3,
                                     const char *inputFile4,
                                     const char *inputFile5,
                                     const char *inputFile6,
                                     const char *inputFile7,
                                     const char *inputFile8,
                                     const char *inputFile9,
                                     const char *p2vmFile,
                                     float *newSpectralOffsets);
amdlibCOMPL_STAT amdlibComputeSpectralCalibration2T(const char *badPixelFile,
                                                    const char *flatFieldFile,
                                                    const char *biasFile,
                                                    const char *inputFile1,
                                                    const char *inputFile2,
                                                    const char *SpectralCalibrationFile);
amdlibCOMPL_STAT amdlibComputeSpectralCalibration3T(const char *badPixelFile,
                                                    const char *flatFieldFile,
                                                    const char *biasFile,
                                                    const char *inputFile1,
                                                    const char *inputFile2,
                                                    const char *inputFile3,
                                                    const char *SpectralCalibrationFile);
amdlibCOMPL_STAT amdlibExtractVis(const char    *badPixelFile,
                                  const char    *flatFieldFile,
                                  const char    *p2vmFile,
                                  const char    *darkFile,
                                  const char    *skyFile,
                                  const char    *inputFile, 
                                  const char    *outputFile,
                                  const int           nbBinning,
                                  const amdlibERROR_TYPE errorType,
                                  const amdlibPISTON_ALGORITHM pistonType,
                                  const amdlibBOOLEAN noCheckP2vmId);

void amdlibVersion(amdlibERROR_MSG versionStr);

int amdlibGetBand (double l);

amdlibCOMPL_STAT amdlibSetPixShift(/* INPUT */
                       int     nbPix,
                       fftw_real *tab_in,
                       fftw_real  shift,
                       /* OUTPUT */
                       fftw_real *tab_out,
		       amdlibERROR_MSG errMsg);
amdlibCOMPL_STAT amdlibGetPixShift(/* INPUT */
                       int     nbPix,
                       fftw_real *tab1,
                       fftw_real *tab2,
                       /* OUTPUT */
                       fftw_real *shift,
                       fftw_real *sigma2_shift,
                       amdlibERROR_MSG errMsg);


amdlibCOMPL_STAT amdlibFillOiTargetTableEntry(/* Input */
                                               amdlibSCIENCE_DATA *data,
                                               /* Output */
                                               amdlibOI_TARGET    *target,
                                               int                pos,
                                               int                targetId);

amdlibCOMPL_STAT amdlibFillInVisTableHeader(/* Input */
                                               amdlibSCIENCE_DATA     *data,
                                               /* Output */
                                               amdlibVIS              *vis,
                                               amdlibERROR_MSG        errMsg);

amdlibCOMPL_STAT amdlibFillInVis2TableHeader(/* Input */
                                               amdlibSCIENCE_DATA     *data,
                                               /* Output */
                                               amdlibVIS2             *vis2,
                                               amdlibERROR_MSG        errMsg);

amdlibCOMPL_STAT amdlibFillInVis3TableHeader(/* Input */
                                               amdlibSCIENCE_DATA     *data,
                                               /* Output */
                                               amdlibVIS3             *vis3,
                                               amdlibERROR_MSG        errMsg);


#ifdef __cplusplus
}
#endif


#endif /*!amdlib_H*/
