/************************************************************************
 * E.S.O. - VLT project
 *
 * who       when      what
 * --------  --------  ----------------------------------------------
 * ealtariba 2006-12-04 - completed result files headers, by appending 
 *                      ESO PRO keywords containing information relative to
 *                      history.
 *                      - fixed bug (Stack Overflow when using GASGANO).
 *                      (corrected by Tom Licha)
 * gluck     2006-07-04 completed display functions associated with verbose
 *                      parameter in amdlibExtractVis function
 * ealtariba 2006-07-04 updated amdlibComputeVisibilities call.
 * ealtariba 2006-06-06 - fixed bug related to instrument cfg keyword list  
 * gzins     2006-05-31 - fixed bug related to version 
 * ealtariba 2006-05-24 - added frame selection.
 * gzins     2006-05-09 - added treatment for current amdlib library version.
 * gluck     2006-05-05 - added verbose parameter in amdlibComputeP2vm2T 
 *                        amdlibComputeP2vm3T and amdlibExtractVis functions
 * altariba  2006-04-04 - added merge option when extracting visibilities
 * gzins     2006-02-12 - printed out pistons and fringe cronstrasts
 *                        in amdlibExtractVis 
 * gzins     30/01/06  - fixed potential bug related to static waveDataLoaded
 *                       variable 
 * gzins     26/01/06  - completed P2VM merging 
 * gduvert   30/08/05  - supressed reading of produced OI-FITS file (saves
 *                       time). Better layout for summary.
 * gduvert   23/08/05  - added support for OI_TARGET and OI_ARRAY.
 * gduvert   30/05/05  - define and use of amdlibExplodeJHKScienceData and
 *                       amdlibIsJHKMode to separate at ScienceData level frames
 *                       taken in JHK spectral mode into 3 rows.
 * gduvert   22/04/05  - use of newSpectralOffsets to enable different 
 *                       spectral offsets in P2VM computation and Visibility
 *                       extraction. Implied changes of argument list for 
 *                       functions, and an 'on the spot' replacement
 *                       of waveData.photoOffset in calibrated raw data prior
 *                       to science data conversion.
 * gduvert   12/02/05  - structures are declared static to prevent use of
 *                       uninitialised values when checking ->thisPtr value
 *                       in all structure-creating routines.
 * gduvert   10/12/04  - changed logic of releasing structures.
 * gduvert   26/10/04  - modify pixel saturation check
 *                     - change parameters for amdlibPerformSpectralCalibration
 *                       call
 * gzins     24/04/03  created
 */

/**
 * @file
 * High-level function for AMBER data reduction processing. 
 *     
 * This file includes high-level functions which provide interface to AMBER data
 * reduction functions with no internal data structure as input or output
 * paremeters.
 *
 * These functions very often uses : 
 * \li a bad pixel map file which contains for each pixel of the detector
 * (512x512 pixels) a flag indicating if pixel is good or not. This file is
 * generated by the detector Maintenance Software (see amdmsServer(1)) through
 * the AMBER Maintenance Templates. If the file name is an empty string all
 * pixels are marked as good.
 *
 * \li a flat field map file which contains for each pixel of the detector
 * (512x512 pixels) a factor which gives the gain of that pixel. This file is
 * generated by the detector Maintenance Software (see amdmsServer(1)) through
 * the AMBER Maintenance Templates. If the file name is an empty string the gain
 * of all pixels is set to 1.
 *
 * \li a dark file which is used to compensate for detector effects. It
 * corresponds to a series of sky images or to a series of images with all
 * shutters closed. The detector subwindow setup and exposure time must be the
 * same than the ones corresponding to the raw data to be calibrated. If the
 * file name is an empty string the bias for each pixel is set to 0.
 */

#define _POSIX_SOURCE 1

 static char *rcsId="@(#) $Id: amdlibUtils.c,v 1.6 2007/03/16 16:53:38 altariba Exp $"; 
static void *use_rcsId = (0 ? &use_rcsId : (void*)&rcsId);

 /*
  * System Headers 
  */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

 /*
  * Local Headers 
  */
#include "amdlib.h"
#include "amdlibProtected.h"

/*
 * Local function definition
 */

static amdlibPIXEL_BIAS_DATA pixelBias = {NULL};
static amdlibRAW_DATA        rawData = {NULL};
static amdlibSC_INPUT_DATA   SpectralCalibrationData = {NULL};
static amdlibWAVEDATA        calWaveData;

#ifndef ESO_CPL_PIPELINE_VARIANT
static amdlibCOMPL_STAT amdlibAppendKeywordListToP2VM(
                                                const char *p2vmFile,
                                                const char *badPixelFile,
                                                const char *flatFieldFile,
                                                const char **inputFiles,
                                                int nbInputFiles,
                                                float newSpectralOffsets[],
                                                amdlibBOOLEAN addOffsets,
                                                amdlibERROR_MSG errMsg);
static amdlibCOMPL_STAT amdlibAppendKeywordListToOIFITS(
                                const char *oifitsFile,
                                const char *badPixelFile,
                                const char *flatFieldFile,
                                const char *darkFile,
                                const char *skyFile,
                                const char *inputFile, 
                                const int   nbBinning,
                                const amdlibERROR_TYPE errorType,
                                const amdlibPISTON_ALGORITHM pistonType,
                                const amdlibBOOLEAN noCheckP2vmId,
                                const amdlibBOOLEAN mergeOutputFiles,
                                const amdlibFRAME_SELECTION selectionType,
                                const double selectionRatio,
                                amdlibERROR_MSG errMsg);

#endif
static amdlibCOMPL_STAT amdlibAppendSelectionKeywords(
                                 const char *oifitsFile,
                                 const char *selFile,
                                 const char *inOiFits,
                                 amdlibINS_CFG *selKewList,
                                 amdlibBOOLEAN saveInOiFits,
                                 const amdlibFRAME_SELECTION selectionType,
                                 const double selectionRatio,
                                 amdlibERROR_MSG errMsg);
                                
/*
 * Public function
 */
/** 
 * Return version of AMBER Data reduction library.
 *
 * @param version string where software version is copied in
 */
void amdlibGetVersion(char version[32])
{
#ifdef CCS_LIGHT
    char *versionStr = "$Revision: 1.6 $";

    sscanf(versionStr, "%*s %s", version);
#else
    strcpy(version, amdlibVERSION);
#endif
}

/** 
 * Print version of AMBER Data reduction library.
 */
void amdlibPrintVersion()
{
    char version[32];

    amdlibGetVersion(version);

    printf("amdlib %s version\n",version);
}

/*
 * Protected functions
 */
/** 
 * Calibrate raw image.
 *
 * This function converts raw image into calibrated image, by correcting
 * detector effects, and stores calibrated image in the ouput file (if given) or
 * the input file otherwise. 
 *
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param darkFile name of file containing data for pixel bias estimation
 * @param inputFile name of file containing data to calibrate
 * @param outputFile name of file where calibrated data will be stored 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
  */
amdlibCOMPL_STAT amdlibGenerateCalImage(const char *badPixelFile,
                                        const char *flatFieldFile,
                                        const char *darkFile,
                                        const char *inputFile,
                                        const char *outputFile)
{
    amdlibERROR_MSG       errMsg;

    amdlibTrace("amdlibGenerateCalImage()\n");
    
    /* If a bad pixel file has been specified */
    if ((badPixelFile != NULL) && (strlen(badPixelFile) != 0))
    {
        /* Load it */
        if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bad pixel map '%s'\n", badPixelFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* Else */
    else
    {
        /* Set it with all pixels marked as good */
        if (amdlibSetBadPixelMap(amdlibTRUE) != amdlibSUCCESS)
        {
            printf ("Could not set bad pixel map\n");
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* End if */

    /* If a flat field file has been specified */
    if ((flatFieldFile != NULL) && (strlen(flatFieldFile) != 0))
    {
        /* Load it */
        if (amdlibLoadFlatFieldMap(flatFieldFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load flat field map '%s'\n", flatFieldFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* Else */
    else
    {
        /* Set it with gain set to 1.0 */
        if (amdlibSetFlatFieldMap(1.0) != amdlibSUCCESS)
        {
            printf ("Could not set flat field map\n");
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* End if */

    /* If a dark file has been specified */
    if ((darkFile != NULL) && (strlen(darkFile) != 0))
    {
        /* Load it */
        if (amdlibLoadRawData(darkFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", darkFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }   

        /* Compute pixel bias map */
        if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                        errMsg) != amdlibSUCCESS)
        {
            printf ("Could not generate pixel bias map\n");
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* Else */
    else
    {
        /* Load input data file */
        if (amdlibLoadRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }  

        /* Set pixel bias map to 0.0 */
        if (amdlibSetPixelBiasData(&rawData, &pixelBias, 
                                   0.0, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not set bias map\n");
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    /* End if */

    /* Load input file */
    if ((inputFile == NULL) || (strlen(inputFile) == 0))
    {
        printf ("Invalid name for input file\n");
        amdlibReleaseRawData(&rawData);
        amdlibReleasePixelBiasData(&pixelBias);
        return (amdlibFAILURE);
    }
    if (amdlibLoadRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
    {
        printf ("Could not load raw data file '%s'\n", inputFile);
        printf ("%s\n", errMsg);
        amdlibReleaseRawData(&rawData);
        amdlibReleasePixelBiasData(&pixelBias);
        return (amdlibFAILURE);
    }

    /* Equalize raw data */
    if (amdlibCalibrateRawData(&pixelBias, &rawData, errMsg) !=amdlibSUCCESS)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseRawData(&rawData);
        amdlibReleasePixelBiasData(&pixelBias);
        return (amdlibFAILURE);
    }

    /* If output file has been specified */
    if ((outputFile != NULL) && (strlen(outputFile) != 0))
    {
        /* Copy input file to output file */
        if (amdlibCopyRawDataFile(inputFile, 
                                  outputFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not create raw data file '%s'\n", "toto");
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }        

        /* Store calibrated data in specified output file or in original file */
        if (amdlibStoreRawData(outputFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not store raw data file '%s'\n", outputFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }
    else
    {
        if (amdlibStoreRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not store raw data file '%s'\n", inputFile);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            amdlibReleasePixelBiasData(&pixelBias);
            return (amdlibFAILURE);
        }
    }

    amdlibReleaseRawData(&rawData);
    amdlibReleasePixelBiasData(&pixelBias);
    return (amdlibSUCCESS);
}

/**
 * Convert AMBER data file into FITS file.
 *
 * This function creates FITS files from data files produced by the AMBER
 * instrument which are stored as binary tables. This consists to load the
 * original data file, to create glued images from data of detector regions and
 * save them into FITS file.
 *
 * @param inputFile name of file containing AMBER data
 * @param outputFile name of file where 2D images will be stored 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibBtbl2Fits(const char *inputFile,
                                 const char *outputFile)
{
    amdlibERROR_MSG  errMsg;

    amdlibTrace("amdlibBtbl2Fits()\n");

    /* Load input file */
    if ((inputFile == NULL) || (strlen(inputFile) == 0))
    {
        printf ("Invalid name for input file\n");
        amdlibReleaseRawData(&rawData);
        return (amdlibFAILURE);
    }
    if (amdlibLoadRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
    {
        printf ("Could not load raw data file '%s'\n", inputFile);
        printf ("%s\n", errMsg);
        amdlibReleaseRawData(&rawData);
        return (amdlibFAILURE);
    }       

    /* Store image into FITS file */
    if ((outputFile == NULL) || (strlen(outputFile) == 0))
    {
        printf ("Invalid name for output file\n");
        amdlibReleaseRawData(&rawData);
        return (amdlibFAILURE);
    }
    if (amdlibSaveRawDataToFits(outputFile, &rawData, errMsg) != amdlibSUCCESS)
    {
        printf ("Could not create FITS file '%s'\n", outputFile);
        amdlibReleaseRawData(&rawData);
        return (amdlibFAILURE);
    }
    amdlibReleaseRawData(&rawData);
    return (amdlibSUCCESS);
}

/**
 * Compute the P2VM (Photometry to Visibility Matrix) for the 2 telescope
 * configuration.
 *
 * This function computes the P2VM (Photometry to Visibility Matrix) for the 2
 * telescope configuration by processing the input data files. The resulting
 * P2VM is saved into the file given as parameter. If this file exists, it is
 * overwritten.
 *
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param biasFile name of file containing data for pixel bias estimation
 * @param inputFile1 name of the 1st file containing AMBER data
 * @param inputFile2 name of the 2nd file containing AMBER data
 * @param inputFile3 name of the 3rd file containing AMBER data
 * @param inputFile4 name of the 4th file containing AMBER data
 * @param p2vmFile name of file where P2VM will be stored 
 * @param newSpectralOffsets offset to be applied on photometric channels 
 * @param verbose indicates if information have to be displayed. 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibComputeP2vm2T(const char *badPixelFile,
                                     const char *flatFieldFile,
                                     const char *biasFile,
                                     const char *inputFile1,
                                     const char *inputFile2,
                                     const char *inputFile3,
                                     const char *inputFile4,
                                     const char *p2vmFile,
                                     float      newSpectralOffsets[],
                                     const amdlibBOOLEAN verbose)
{
    amdlibBOOLEAN         newSpecOffset;
    const char            *inputFiles[5];
    int                   i;
    amdlibERROR_MSG       errMsg;
    amdlibP2VM_INPUT_DATA p2vmData = {NULL};
    amdlibP2VM_MATRIX     p2vm = {NULL};
    amdlibWAVEDATA        waveData;
    amdlibBOOLEAN         waveDataLoaded;

    amdlibTrace("amdlibComputeP2vm2T()\n");
    
    /* Init list of input files */
    inputFiles[0] = biasFile;
    inputFiles[1] = inputFile1;
    inputFiles[2] = inputFile2;
    inputFiles[3] = inputFile3;
    inputFiles[4] = inputFile4;

    /* If a bad pixel file has been specified */
    if ((badPixelFile != NULL) && (strlen(badPixelFile) != 0))
    {
        /* Load it */
        if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bad pixel map '%s'\n", badPixelFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Load flat field map */
    if ((flatFieldFile != NULL) && (strlen(flatFieldFile) != 0))
    {
        if(amdlibLoadFlatFieldMap(flatFieldFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load Flat Field map '%s'\n", flatFieldFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Check P2VM file name */
    if ((p2vmFile == NULL) || (strlen(p2vmFile) == 0))
    {
        printf ("Invalid name for P2VM file\n");
        return (amdlibFAILURE);
    }

    /* For each input files */
    waveDataLoaded = amdlibFALSE;
    newSpecOffset = amdlibFALSE;
    for (i = 0; i < 5; i++)
    {
        if ((inputFiles[i] == NULL) || (strlen(inputFiles[i]) == 0))
        {
            printf ("Invalid name for %dth input file\n", i+1);
            return (amdlibFAILURE);
        }

        /* Load raw data */
        printf ("Loading %s ...\n", inputFiles[i]);

        if (amdlibLoadRawData(inputFiles[i], &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFiles[i]);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }   

        if (rawData.frameType == amdlibUNKNOWN_FRAME)
        {
            printf ("Invalid frame type '%d'\n", amdlibUNKNOWN_FRAME);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }
        else if (rawData.frameType == amdlibDARK_FRAME)
        {
            /* Compute pixel bias map */
            if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                            errMsg) != amdlibSUCCESS)
            {
                printf ("Could not generate pixel bias map\n");
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }
        }
        else
        {
            /* Equalize raw data */
            if (amdlibCalibrateRawData(&pixelBias, 
                                       &rawData, errMsg) !=amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }

            /* Test if some pixel is saturated */
            if (rawData.dataIsSaturated)
            {
                printf ("Data in file number %s is saturated!\n", 
                        inputFiles[i]);
            }

            /* Get wave data from the first given file */
            if (waveDataLoaded == amdlibFALSE)
            {
                int p;

                if (amdlibGetWaveDataFromRawData(&rawData, &waveData, 
                                                 errMsg) == amdlibFAILURE)
                {
                    printf ("%s\n", errMsg);
                    amdlibReleasePixelBiasData(&pixelBias);
                    amdlibReleaseRawData(&rawData);
                    amdlibReleaseP2vmData(&p2vmData);
                    return (amdlibFAILURE);
                }

                /* And set new offsets (if given) */
                for (p = 0; p < 3; p++)
                {
                    if (newSpectralOffsets[p] != amdlibOFFSETY_NOT_CALIBRATED)
                    {
                        newSpecOffset = amdlibTRUE;
                        waveData.photoOffset[p] = newSpectralOffsets[p];
                    }
                }

                waveDataLoaded = amdlibTRUE;
            }

            /* Store calibrated data into P2VM data structure */ 
            if (amdlibAddToP2vmData(&rawData, &waveData, &p2vmData, 
                                    errMsg) != amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                amdlibReleaseP2vmData(&p2vmData);
                return (amdlibFAILURE);
            }
        }

        amdlibReleaseRawData(&rawData);
    }
    /* End for */

    amdlibReleasePixelBiasData(&pixelBias);

    /* Compute P2VM */
    printf ("Computing P2VM ...\n");
    if (amdlibComputeP2VM(&p2vmData, amdlibP2VM_2T, &waveData, &p2vm, errMsg) 
            == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseP2vmData(&p2vmData);
        amdlibReleaseP2VM(&p2vm);
        return (amdlibFAILURE);
    }

    if (verbose == amdlibTRUE)
    {
        amdlibDisplayP2vm(&p2vm);
    }

    amdlibReleaseP2vmData(&p2vmData);

    printf ("Saving P2VM file %s ...\n", p2vmFile);
    if (amdlibSaveP2VM(p2vmFile, &p2vm, 
                       amdlibP2VM_UNKNOWN_ACC,
                       errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseP2VM(&p2vm);
        return (amdlibFAILURE);
    }

    amdlibReleaseP2VM(&p2vm);

#ifndef ESO_CPL_PIPELINE_VARIANT
    if (amdlibAppendKeywordListToP2VM(p2vmFile, badPixelFile, flatFieldFile,
                                     inputFiles, 5, newSpectralOffsets,
                                     newSpecOffset, errMsg) != amdlibSUCCESS)
    {
        printf("Could not had PRO keywords - %s", errMsg);
        return amdlibFAILURE;
    }
#endif

    return (amdlibSUCCESS);
}

/**
 * Compute the P2VM (Photometry to Visibility Matrix) for the 3 telescope
 * configuration.
 *
 * Ditto amdlibComputeP2vm2T for a 3 telescope configuration.
 *
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param biasFile name of file containing data for pixel bias estimation
 * @param inputFile1 name of the 1st file containing AMBER data
 * @param inputFile2 name of the 2nd file containing AMBER data
 * @param inputFile3 name of the 3rd file containing AMBER data
 * @param inputFile4 name of the 4th file containing AMBER data
 * @param inputFile5 name of the 5th file containing AMBER data
 * @param inputFile6 name of the 6th file containing AMBER data
 * @param inputFile7 name of the 7th file containing AMBER data
 * @param inputFile8 name of the 8th file containing AMBER data
 * @param inputFile9 name of the 9th file containing AMBER data
 * @param p2vmFile name of file where P2VM will be stored 
 * @param newSpectralOffsets offset to be applied on photometric channels 
 * @param verbose indicates if information have to be displayed. 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibComputeP2vm3T(const char *badPixelFile,
                                     const char *flatFieldFile,
                                     const char *biasFile,
                                     const char *inputFile1,
                                     const char *inputFile2,
                                     const char *inputFile3,
                                     const char *inputFile4,
                                     const char *inputFile5,
                                     const char *inputFile6,
                                     const char *inputFile7,
                                     const char *inputFile8,
                                     const char *inputFile9,
                                     const char *p2vmFile,
                                     float *newSpectralOffsets,
                                     const amdlibBOOLEAN verbose)
{
    amdlibBOOLEAN         newSpecOffset;
    const char            *inputFiles[10];
    int                   i;
    amdlibERROR_MSG       errMsg;
    amdlibP2VM_INPUT_DATA p2vmData = {NULL};
    amdlibP2VM_MATRIX     p2vm = {NULL};
    amdlibWAVEDATA        waveData;
    amdlibBOOLEAN         waveDataLoaded;


    amdlibTrace("amdlibComputeP2vm3T()\n");
    
    /* Init list of input files */
    inputFiles[0] = biasFile;
    inputFiles[1] = inputFile1;
    inputFiles[2] = inputFile2;
    inputFiles[3] = inputFile3;
    inputFiles[4] = inputFile4;
    inputFiles[5] = inputFile5;
    inputFiles[6] = inputFile6;
    inputFiles[7] = inputFile7;
    inputFiles[8] = inputFile8;
    inputFiles[9] = inputFile9;

    /* If a bad pixel file has been specified */
    if ((badPixelFile != NULL) && (strlen(badPixelFile) != 0))
    {
        /* Load it */
        if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bad pixel map '%s'\n", badPixelFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    } 


    /* Load flat field map */
    if ((flatFieldFile != NULL) && (strlen(flatFieldFile) != 0))
    {
        if(amdlibLoadFlatFieldMap(flatFieldFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load Flat Field map '%s'\n", flatFieldFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Check P2VM file name */
    if ((p2vmFile == NULL) || (strlen(p2vmFile) == 0))
    {
        printf ("Invalid name for P2VM file\n");
        return (amdlibFAILURE);
    }

    /* For each input files */
    waveDataLoaded = amdlibFALSE;
    newSpecOffset = amdlibFALSE;
    for (i = 0; i < 10; i++)
    {
        if ((inputFiles[i] == NULL) || (strlen(inputFiles[i]) == 0))
        {
            printf ("Invalid name for %dth input file\n", i+1);
            return (amdlibFAILURE);
        }

        /* Load raw data */
        printf ("Loading %s ...\n", inputFiles[i]);
        if (amdlibLoadRawData(inputFiles[i], &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFiles[i]);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }   

        if (rawData.frameType == amdlibUNKNOWN_FRAME)
        {
            printf ("Invalid frame type '%d'\n", amdlibUNKNOWN_FRAME);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }
        else if (rawData.frameType == amdlibDARK_FRAME)
        {
            /* Compute pixel bias map */
            if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                            errMsg) != amdlibSUCCESS)
            {
                printf ("Could not generate pixel bias map\n");
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }
        }
        else
        {
            int p;

            /* Equalize raw data */
            if (amdlibCalibrateRawData(&pixelBias,
                                       &rawData, errMsg) != amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }
           
            /* Test if some pixel is saturated */
            if (rawData.dataIsSaturated)
            {
                printf ("Data in file number %s is saturated!\n", 
                        inputFiles[i]);
            }

            /* Get wave data form the first given file */
            if (waveDataLoaded == amdlibFALSE)
            {
                if (amdlibGetWaveDataFromRawData(&rawData, &waveData, 
                                                 errMsg) == amdlibFAILURE)
                {
                    printf ("%s\n", errMsg);
                    amdlibReleasePixelBiasData(&pixelBias);
                    amdlibReleaseRawData(&rawData);
                    amdlibReleaseP2vmData(&p2vmData);
                    return (amdlibFAILURE);
                }

                /* And set new offsets (if given) */
                for (p = 0; p < 3; p++)
                {
                    if (newSpectralOffsets[p] != amdlibOFFSETY_NOT_CALIBRATED)
                    {
                        newSpecOffset = amdlibTRUE;
                        waveData.photoOffset[p] = newSpectralOffsets[p];
                    }
                }

                waveDataLoaded = amdlibTRUE;
            }
              
            /* Store calibrated data into P2VM data structure */ 
            if (amdlibAddToP2vmData(&rawData, &waveData, &p2vmData,
                                    errMsg) != amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                amdlibReleaseP2vmData(&p2vmData);
                return (amdlibFAILURE);
            }

        }
        amdlibReleaseRawData(&rawData);
    }
    /* End for */

    amdlibReleasePixelBiasData(&pixelBias);


    /* Compute P2VM */
    printf ("Computing P2VM ...\n");
    if (amdlibComputeP2VM(&p2vmData, amdlibP2VM_3T, &waveData, &p2vm, errMsg) 
            == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseP2vmData(&p2vmData);
        amdlibReleaseP2VM(&p2vm);
        return (amdlibFAILURE);
    }

    amdlibReleaseP2vmData(&p2vmData);

    if (verbose == amdlibTRUE)
    {
        amdlibDisplayP2vm(&p2vm);
    }
    
    printf ("Saving P2VM file %s ...\n", p2vmFile);
    if (amdlibSaveP2VM(p2vmFile, &p2vm, 
                       amdlibP2VM_UNKNOWN_ACC,
                       errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseP2VM(&p2vm);
        return (amdlibFAILURE);
    }
    amdlibReleaseP2VM(&p2vm);

#ifndef ESO_CPL_PIPELINE_VARIANT
    if (amdlibAppendKeywordListToP2VM(p2vmFile, badPixelFile, flatFieldFile,
                                     inputFiles, 10, newSpectralOffsets,
                                     newSpecOffset, errMsg) != amdlibSUCCESS)
    {
        printf("Could not had PRO keywords - %s", errMsg);
        return amdlibFAILURE;
    }
#endif


    return (amdlibSUCCESS);
}

/** 
 * Compute spectral calibration for the 2 telescopes configuration.
 *
 * This function computes the spectral calibration  for the 2 telescope 
 * configuration. The resulting calculated offsets are saved into the file 
 * given as parameter. If this file exists, it is overwritten.
 *
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param biasFile name of file containing data for pixel bias estimation
 * @param inputFile1 name of file containing data on specific shutter
 * configuration useful to perform spectral calibration
 * @param inputFile2 name of file containing data on specific shutter
 * configuration useful to perform spectral calibration
 * @param spectralCalibrationFile name of file where calibration offsets will 
 * be stored 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
  */
amdlibCOMPL_STAT amdlibComputeSpectralCalibration2T(
                                        const char *badPixelFile,
                                        const char *flatFieldFile,
                                        const char *biasFile,
                                        const char *inputFile1,
                                        const char *inputFile2,
                                        const char *spectralCalibrationFile)
{
    const char            *inputFiles[3];
    int                   i;
    amdlibERROR_MSG       errMsg;

    amdlibTrace("amdlibComputeSpectralCalibration2T()\n");

    /* Init list of input files */
    inputFiles[0] = biasFile;
    inputFiles[1] = inputFile1;
    inputFiles[2] = inputFile2;

    /* If a bad pixel file has been specified */
    if ((badPixelFile != NULL) && (strlen(badPixelFile) != 0))
    {
        /* Load it */
        if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bad pixel map '%s'\n", badPixelFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Load flat field map */
    if ((flatFieldFile != NULL) && (strlen(flatFieldFile) != 0))
    {
        if(amdlibLoadFlatFieldMap(flatFieldFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load Flat Field map '%s'\n", flatFieldFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }



    /* Check Spectral Calibration file name */
    if ((spectralCalibrationFile == NULL) || 
        (strlen(spectralCalibrationFile) == 0))
    {
        printf ("Invalid name for Spectral Calibration file\n");
        return (amdlibFAILURE);
    }

    /* For each input files */
    for (i = 0; i < 3; i++)
    {
        if ((inputFiles[i] == NULL) || (strlen(inputFiles[i]) == 0))
        {
            printf ("Invalid name for %dth input file\n", i+1);
            return (amdlibFAILURE);
        }

        /* Load raw data */
        printf ("Loading %s ...\n", inputFiles[i]);

        if (amdlibLoadRawData(inputFiles[i], &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFiles[i]);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }   

        
        if (rawData.frameType == amdlibUNKNOWN_FRAME)
        {
            printf ("Invalid frame type '%d'\n", amdlibUNKNOWN_FRAME);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }
        else if (rawData.frameType == amdlibDARK_FRAME)
        {
            /* Compute pixel bias map */
            if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                            errMsg) != amdlibSUCCESS)
            {
                printf ("Could not generate pixel bias map\n");
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }
        }
        else
        {
            /* Equalize raw data */
            if (amdlibCalibrateRawData(&pixelBias, &rawData, errMsg) 
                !=amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }

            /* Store calibrated data into Spectral Calibration data structure */ 
            if (amdlibAddToSpectralCalibrationData(&rawData, 
                                                   &SpectralCalibrationData,
                                                   errMsg) != amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);
                return (amdlibFAILURE);
            }
        }
        /* copy first rawData->waveData structure in calWaveData to fill all 
         * fields*/
        if (i) 
        {
            memcpy(&calWaveData, &rawData.waveData, 
                   sizeof(amdlibWAVEDATA));
        }
        amdlibReleaseRawData(&rawData);
    }
    /* End for */

    amdlibReleasePixelBiasData(&pixelBias);

    /* Compute Spectral Calibration */
    printf ("Computing Spectral Calibration ...\n");
    if (amdlibPerformSpectralCalibration(&SpectralCalibrationData, amdlibSC_2T,
                                         &calWaveData, errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);
        return (amdlibFAILURE);
    }

    amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);

    printf ("Saving Spectral Calibration file %s ...\n", 
            spectralCalibrationFile);
    if (amdlibSaveWaveData(spectralCalibrationFile, &calWaveData, 
                           errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    return (amdlibSUCCESS);
}

/** 
 * Compute spectral calibration for the 3 telescopes configuration.
 *
 * Ditto amdlibComputeSpectralCalibration2T for the 3 telescopes configuration
 * 
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param biasFile name of file containing data for pixel bias estimation
 * @param inputFile1 name of file containing data on specific shutter
 * configuration useful to perform spectral calibration
 * @param inputFile2 name of file containing data on specific shutter
 * configuration useful to perform spectral calibration
 * @param inputFile3 name of file containing data on specific shutter
 * configuration useful to perform spectral calibration
 * @param spectralCalibrationFile name of file where calibration offsets will 
 * be stored 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibComputeSpectralCalibration3T(const char *badPixelFile,
                                                    const char *flatFieldFile,
                                                    const char *biasFile,
                                                    const char *inputFile1,
                                                    const char *inputFile2,
                                                    const char *inputFile3,
                                                    const char *spectralCalibrationFile)
{
    const char            *inputFiles[4];
    int                   i;
    amdlibERROR_MSG       errMsg;

    amdlibTrace("amdlibComputeSpectralCalibration3T()\n");

    /* Init list of input files */
    inputFiles[0] = biasFile;
    inputFiles[1] = inputFile1;
    inputFiles[2] = inputFile2;
    inputFiles[3] = inputFile3;

    /* If a bad pixel file has been specified */
    if ((badPixelFile != NULL) && (strlen(badPixelFile) != 0))
    {
        /* Load it */
        if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bad pixel map '%s'\n", badPixelFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Load flat field map */
    if ((flatFieldFile != NULL) && (strlen(flatFieldFile) != 0))
    {
        if(amdlibLoadFlatFieldMap(flatFieldFile, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load Flat Field map '%s'\n", flatFieldFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE);
        }
    }

    /* Check Spectral Calibration file name */
    if ((spectralCalibrationFile == NULL) || 
        (strlen(spectralCalibrationFile) == 0))
    {
        printf ("Invalid name for Spectral Calibration file\n");
        return (amdlibFAILURE);
    }

    /* For each input files */
    for (i = 0; i < 4; i++)
    {
        if ((inputFiles[i] == NULL) || (strlen(inputFiles[i]) == 0))
        {
            printf ("Invalid name for %dth input file\n", i+1);
            return (amdlibFAILURE);
        }

        /* Load raw data */
        printf ("Loading %s ...\n", inputFiles[i]);

        if (amdlibLoadRawData(inputFiles[i], &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFiles[i]);
            printf ("%s\n", errMsg);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }   

        if (rawData.frameType == amdlibUNKNOWN_FRAME)
        {
            printf ("Invalid frame type '%d'\n", amdlibUNKNOWN_FRAME);
            amdlibReleaseRawData(&rawData);
            return (amdlibFAILURE);
        }
        else if (rawData.frameType == amdlibDARK_FRAME)
        {
            /* Compute pixel bias map */
            if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                            errMsg) != amdlibSUCCESS)
            {
                printf ("Could not generate pixel bias map\n");
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }
        }
        else
        {
            /* Equalize raw data */
            if (amdlibCalibrateRawData(&pixelBias, &rawData, errMsg) 
                !=amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                return (amdlibFAILURE);
            }

            /* Store calibrated data into Spectral Calibration data structure */
            if (amdlibAddToSpectralCalibrationData(&rawData, 
                                                   &SpectralCalibrationData,
                                                   errMsg) != amdlibSUCCESS)
            {
                printf ("%s\n", errMsg);
                amdlibReleasePixelBiasData(&pixelBias);
                amdlibReleaseRawData(&rawData);
                amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);
                return (amdlibFAILURE);
            }
        }
        /* copy first rawData->waveData structure in calWaveData to fill 
         * all fields*/
        if (i)
        {
            memcpy(&calWaveData, &rawData.waveData, sizeof(amdlibWAVEDATA));
        }
        amdlibReleaseRawData(&rawData);
    }
    /* End for */

    amdlibReleasePixelBiasData(&pixelBias);

    /* Compute Spectral Calibration */
    printf ("Computing Spectral Calibration ...\n");
    if (amdlibPerformSpectralCalibration(&SpectralCalibrationData, amdlibSC_3T,
                                         &calWaveData, errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);
        return (amdlibFAILURE);
    }

    amdlibReleaseSpectralCalibrationData(&SpectralCalibrationData);

    printf ("Saving Spectral Calibration file %s ...\n", 
            spectralCalibrationFile);
    if (amdlibSaveWaveData(spectralCalibrationFile, 
                           &calWaveData, errMsg) == amdlibFAILURE)
    {
        printf ("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    return (amdlibSUCCESS);
}

/**
 * Compute visibilities.
 * 
 * This function extracts the visibilities and piston from the input file, using
 * the given p2vm, and save result into the output file. If dark file and/or sky
 * file are not specified (i.e set to ""), there are simply ignored. In the same
 * way, if file name for output is "", it is not created.
 *
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param p2vmFile name of file containing P2VM
 * @param darkFile name of file containing data for pixel bias estimation
 * @param skyFile name of file containing data on sky
 * @param inputFile name of the file containing AMBER data
 * @param outputFile name of the resulting file
 * @param nbBinning number of binnings 
 * @param errorType indicates wether the noise figures are estimated 
 * statistically from the sequence of photometries during the bin 
 * (amdlibSTATISTICAL_ERROR) or theoretically by the poisson error on N photons
 * (amdlibTHEORETICAL_ERROR). The latter is forced obviously when the binsize
 * is 1. 
 * @param pistonType indicates wether the piston is to be measured by fitting
 * a slope in the phasors amdlibITERATIVE_PHASOR or in the differential phases
 * after dewrapping (amdlibUNWRAPPED_PHASE). 
 * @param noCheckP2vmId forces amdlib to use without wuestion the passed p2vm,
 * even if its magic number is not OK. Can happen if the P2VM has been 
 * observed AFTER the science observations. 
 * @param mergeOutputFiles indicates if many files will be produced (a single 
 * one if TRUE or as much as there are bands otherwise).
 * @param selectionType name of the chosen selection criteria 
 * (amdlibNO_FRAME_SEL if no selection).
 * @param selectionRatio rato or threshold associated to the selection criteria.
 * @param verbose indicates if information have to be displayed. 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibExtractVis(const char *badPixelFile,
                                  const char *flatFieldFile,
                                  const char *p2vmFile,
                                  const char *darkFile,
                                  const char *skyFile,
                                  const char *inputFile, 
                                  const char *outputFile,
                                  const int   nbBinning,
                                  const amdlibERROR_TYPE errorType,
                                  const amdlibPISTON_ALGORITHM pistonType,
                                  const amdlibBOOLEAN noCheckP2vmId,
                                  const amdlibBOOLEAN mergeOutputFiles,
                                  const amdlibFRAME_SELECTION selectionType,
                                  const double selectionRatio,
                                  const amdlibBOOLEAN verbose)
{
    amdlibERROR_MSG       errMsg;
    amdlibSCIENCE_DATA    sky = {NULL};
    amdlibSCIENCE_DATA    scienceData = {NULL};
    amdlibP2VM_MATRIX     p2vm = {NULL};
    amdlibPHOTOMETRY      photometry = {NULL}, imdPhot = {NULL};
    amdlibWAVELENGTH      wave = {NULL}, imdWave = {NULL};
    amdlibPISTON          opd = {NULL}, imdOpd = {NULL};
    amdlibOI_TARGET       target = {NULL};
    amdlibOI_ARRAY        array = {NULL};
    amdlibVIS             vis = {NULL}, imdVis = {NULL};
    amdlibVIS2            vis2 = {NULL}, imdVis2 = {NULL};
    amdlibVIS3            vis3 = {NULL}, imdVis3 = {NULL};
    amdlibSCIENCE_DATA    *skyPtr;
    amdlibSCIENCE_DATA    *sciencePtr;
    amdlibWAVEDATA        waveData;
    amdlibCPT_VIS_OPTIONS visOptions = {nbBinning, errorType, pistonType, 
        noCheckP2vmId, selectionType, selectionRatio};
    char localfile[512];
    int band;

    amdlibTrace("amdlibExtractVis()\n");

    /* Load bad pixel map */
    printf ("Loading %s ...\n", badPixelFile);
    if (amdlibLoadBadPixelMap(badPixelFile, errMsg) != amdlibSUCCESS)
    {
        printf ("Could not load bad pixel map '%s'\n", badPixelFile);
        printf ("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    /* Load flat field map */
    printf ("Loading %s ...\n", flatFieldFile);
    if (amdlibLoadFlatFieldMap(flatFieldFile, errMsg)!=amdlibSUCCESS)
    {
        printf ("Could not load Flat Field file '%s'\n", flatFieldFile);
        printf ("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    /* Load P2VM */
    printf ("Loading P2VM file %s ...\n", p2vmFile);
    if (amdlibLoadP2VM(p2vmFile,  &p2vm, errMsg) == amdlibFAILURE)
    {
        printf ("Could not load P2VM file '%s'\n", p2vmFile);
        printf ("%s\n", errMsg);
        return (amdlibFAILURE); 
    }

    /* Retrieve wavedata */
    if (amdlibGetWaveDataFromP2vm(&p2vm, &waveData, errMsg) != amdlibSUCCESS)
    { 
        printf ("Could not get wave data\n");
        printf ("%s\n", errMsg);
        return (amdlibFAILURE); 
    }

    /* Load bias file */
    if (strlen(darkFile) != 0)
    {
        printf ("Loading bias file %s ...\n", darkFile);
        if (amdlibLoadRawData(darkFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load bias file '%s'\n", darkFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }
        /* Compute pixel bias map */
        if (amdlibGeneratePixelBiasData(&rawData, &pixelBias,
                                        errMsg) != amdlibSUCCESS)
        {
            printf ("Could not generate pixel bias map\n");
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }
        amdlibReleaseRawData(&rawData);
    }
    else
    {
        printf ("No pixel bias used...\n");
        
        /* Load data file so that the false bias mimics its structure */
        printf ("Getting bias header information from %s ...\n", inputFile);
        if (amdlibLoadRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load raw data file '%s'\n", inputFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }
        amdlibSetPixelBiasData( &rawData,  &pixelBias, 0,  errMsg);
        amdlibReleaseRawData(&rawData);
    }
    
    /* Load sky file */
    if (strlen(skyFile) != 0)
    {
        printf ("Loading sky file %s ...\n", skyFile);
        if (amdlibLoadRawData(skyFile, &rawData, errMsg) != amdlibSUCCESS)
        {
            printf ("Could not load sky file '%s'\n", skyFile);
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }
        /* Equalize raw data */
        if (amdlibCalibrateRawData(&pixelBias,&rawData, errMsg) 
            !=amdlibSUCCESS)
        {
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }

        /* Equalize raw map */
        if (amdlibRawData2ScienceData(&rawData, &waveData, &sky, amdlibTRUE,
                                      errMsg) != amdlibSUCCESS)
        {
            printf ("Could not get science data\n");
            printf ("%s\n", errMsg);
            return (amdlibFAILURE); 
        }
        skyPtr = &sky;
        amdlibReleaseRawData(&rawData);
    }
    else
    {
        printf ("No sky used...\n");
        skyPtr = NULL;
    }

    /* Load data file */
    printf ("Loading %s data file ...\n", inputFile);
    if (amdlibLoadRawData(inputFile, &rawData, errMsg) != amdlibSUCCESS)
    {
        printf ("Could not load raw data file '%s'\n", inputFile);
        printf ("%s\n", errMsg);
        return (amdlibFAILURE); 
    }
    
    /* Equalize raw data */
    if (amdlibCalibrateRawData(&pixelBias, &rawData, errMsg) 
        !=amdlibSUCCESS)
    {
        printf ("%s\n", errMsg);
        return (amdlibFAILURE); 
    }

    /* Retrieve array information from raw data*/
    if (amdlibGetOiArrayFromRawData(&rawData, &array, errMsg) != amdlibSUCCESS)
    {
        printf ("Warning -- Unable to retrieve OI_ARRAY information in input "
                "file\n");
        amdlibReleaseOiArray(&array);
    }

    /* retrieve target information */
    if (amdlibAllocateOiTarget(&target, 1) != amdlibSUCCESS)
    {
        printf ("Could not Allocate Target Structure\n");
        return (amdlibFAILURE); 
    }
    if (amdlibGetOiTargetFromRawData(&rawData,&target) != amdlibSUCCESS)
    {
        printf ("Could not Add to Target Structure\n");
        return (amdlibFAILURE);
    }

    /* Produce science data */
    if (amdlibRawData2ScienceData(&rawData, &waveData, &scienceData, 
                                  amdlibFALSE,
                                  errMsg) != amdlibSUCCESS)
    { 
        printf ("Could not get science data\n");
        printf ("%s\n", errMsg);
        return (amdlibFAILURE); 
    }

    sciencePtr = &scienceData;
    amdlibReleaseRawData(&rawData);
    amdlibReleasePixelBiasData(&pixelBias);
    
    for (band = amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        int nbChannels;
        /* Display for test */
        if (verbose == amdlibTRUE)
        {
            switch (band)
            {
                case amdlibJ_BAND :
                    printf("\nJ band\n");
                    printf("------\n");
                    break;
                case amdlibH_BAND :
                    printf("\nH band\n");
                    printf("------\n");
                    break;
                case amdlibK_BAND :
                    printf("\nK band\n");
                    printf("------\n");
                    break;
                default :
                    break;
            } 
        }
        /* Compute visibilities */
        nbChannels = amdlibComputeVisibilities
            (/* Input */
             sciencePtr, skyPtr, &p2vm, &waveData, band, &visOptions,
             /* Output */
             &imdPhot, &imdVis, &imdVis2, &imdVis3, &imdWave, &imdOpd,
             errMsg);
        if (nbChannels == -1)
        {
            printf ("Could NOT extract visibilities for '%c' band !\n", 
                    amdlibBandNumToStr(band));
            printf ("%s\n", errMsg);
	    continue; /*CAH*/
            return (amdlibFAILURE); 
        }
        else if (nbChannels == 0)
        {
            printf ("There is no channel for '%c' band\n", 
                    amdlibBandNumToStr(band));
        }
        else
        {
#if 1
            /* Print results */
            printf("band: %d\n",band);
            printf("======\n");
            printf ("There is %d channel(s) for '%c' band\n", 
                    nbChannels, amdlibBandNumToStr(band));
            printf ("Global Averaged Visibilities Squared :\n");
            printf ("--------------\n");
            if (imdVis2.nbBases == 1)
            {
                printf("         VIS (err)\n");
                printf("%12.5f (%8.05g)\n", imdVis2.vis12, imdVis2.sigmaVis12);
            }
            else
            {
                printf("      VIS 12 (err)           VIS 23 (err)           "
                       "VIS 31 (err)\n");
                printf("%12.5f (%8.05g)", imdVis2.vis12, imdVis2.sigmaVis12);
                printf("%12.5f (%8.05g)", imdVis2.vis23, imdVis2.sigmaVis23);
                printf("%12.5f (%8.05g)\n\n", imdVis2.vis31, 
                       imdVis2.sigmaVis31);
                printf("\n\nAverageClosurePhase (deg)= %8.05g (%8.05g)\n",
                       imdVis3.averageClosure,imdVis3.averageClosureError);
            }
#endif

            if (mergeOutputFiles == amdlibTRUE)
            {
                if (amdlibMergeOiStructures(&wave, &imdWave,
                                            &photometry, &imdPhot,
                                            &vis, &imdVis,
                                            &vis2, &imdVis2,
                                            &vis3, &imdVis3,
                                            &opd, &imdOpd,
                                            errMsg) != amdlibSUCCESS)
                {
                    printf("Could not merge OI structures \n");
                    printf("%s\n", errMsg);
                    return (amdlibFAILURE);
                }
            }
            else
            {
                /* Save OI file */
                if (strlen(outputFile) != 0)
                {
                    strcpy(localfile, outputFile);
                    char *p;
                    strcpy(localfile, outputFile);
                    p = strstr(localfile, ".fits");
                    if (p != NULL)
                    {
                        sprintf(p, "_%c.%s", amdlibBandNumToStr(band), "fits");
                    }
                    else
                    {
                        sprintf(localfile, "%s_%c", outputFile, 
                                amdlibBandNumToStr(band));
                    }
                    printf ("Writing additional OI file %s \n",localfile);

                    if (amdlibSaveOiFile(localfile, &sciencePtr->insCfg,
                                         &array, &target,
                                         &imdPhot, &imdVis, &imdVis2, 
                                         &imdVis3, &imdWave, 
                                         &imdOpd, errMsg) != amdlibSUCCESS)
                    { 
                        printf ("Could not save OI file\n");
                        printf ("%s\n", errMsg);
                        return (amdlibFAILURE); 
                    }
                    else
                    {
                        printf ("File saved successfully\n");
                    }
#ifndef ESO_CPL_PIPELINE_VARIANT
                    if (amdlibAppendKeywordListToOIFITS(
                                    localfile, badPixelFile, flatFieldFile,
                                    darkFile, skyFile, inputFile, nbBinning,
                                    errorType, pistonType, noCheckP2vmId,
                                    mergeOutputFiles, selectionType,
                                    selectionRatio, errMsg)!= amdlibSUCCESS)
                    {
                        printf("Could not add PRO keywords - %s", errMsg);
                        return amdlibFAILURE;
                    }
#endif
                }

                /* Display for test */
                if (verbose == amdlibTRUE)
                {
                    printf ("\nDisplay spectral dispersion structure ...\n");
                    amdlibDisplayWavelength(&imdWave);

                    printf ("\nDisplay photometry structure ...\n");
                    amdlibDisplayPhotometry(&imdPhot);

                    printf ("\nDisplay visibility structure ...\n");
                    amdlibDisplayVis(&imdVis);
                    
                    printf ("\nDisplay squared visibility structure ...\n");
                    amdlibDisplayVis2(&imdVis2);
                    
                    printf ("\nDisplay closure phases structure ...\n");
                    amdlibDisplayVis3(&imdVis3);
                    
                    printf ("\nDisplay piston structure ...\n");
                    amdlibDisplayPiston(&imdOpd);
                }
            }
            
            amdlibReleasePhotometry(&imdPhot);
            amdlibReleaseVis(&imdVis);
            amdlibReleaseVis2(&imdVis2);
            amdlibReleaseVis3(&imdVis3);
            amdlibReleaseWavelength(&imdWave);
            amdlibReleasePiston(&imdOpd);
            
        }
    }
    
    if (mergeOutputFiles == amdlibTRUE)
    {
        /* Save OI file */
        if (strlen(outputFile) != 0)
        {
            printf ("Writing  OI file %s \n", outputFile);

            if (amdlibSaveOiFile(outputFile,  &sciencePtr->insCfg,
                                 &array, &target,
                                 &photometry, &vis, &vis2, &vis3, &wave, 
                                 &opd, errMsg) != amdlibSUCCESS)
            { 
                printf ("Could not save OI file\n");
                printf ("%s\n", errMsg);
                return (amdlibFAILURE); 
            }
            else
            {
                printf ("File saved successfully\n");
            }
#ifndef ESO_CPL_PIPELINE_VARIANT
            if (amdlibAppendKeywordListToOIFITS(
                                      outputFile, badPixelFile, flatFieldFile,
                                      darkFile, skyFile, inputFile, nbBinning,
                                      errorType, pistonType, noCheckP2vmId,
                                      mergeOutputFiles, selectionType,
                                      selectionRatio, errMsg)!= amdlibSUCCESS)
            {
                printf("Could not add PRO keywords - %s", errMsg);
                return amdlibFAILURE;
            }
#endif
        }

        /* Display for test */
        if (verbose == amdlibTRUE)
        {
            printf ("\nDisplay spectral dispersion structure ...\n");
            amdlibDisplayWavelength(&wave);

            printf ("\nDisplay photometry structure ...\n");
            amdlibDisplayPhotometry(&photometry);

            printf ("\nDisplay visibility structure ...\n");
            amdlibDisplayVis(&vis);

            printf ("\nDisplay squared visibility structure ...\n");
            amdlibDisplayVis2(&vis2);

            printf ("\nDisplay closure phases structure ...\n");
            amdlibDisplayVis3(&vis3);

            printf ("\nDisplay piston structure ...\n");
            amdlibDisplayPiston(&opd);
        }
    }

    amdlibReleaseScienceData(sciencePtr);
    if (skyPtr != NULL) 
    {
        amdlibReleaseScienceData(skyPtr);
    }
    amdlibReleaseP2VM(&p2vm);
    amdlibReleaseVis(&vis);
    amdlibReleaseVis2(&vis2);
    amdlibReleaseVis3(&vis3);
    amdlibReleaseOiArray(&array);
    amdlibReleaseWavelength(&wave);
    amdlibReleasePiston(&opd);
    amdlibReleasePhotometry(&photometry);
    amdlibReleaseOiTarget(&target);
    
    return (amdlibSUCCESS);
}

/**
 * Perform a frame selection on input file.
 * 
 * This function loads the input file, performs a selection on its frames to
 * keep only good ones using the given criterion and ratio, and/or the
 * selection input file. Then it can, if specified,  
 *      - average all data according to good frames kept and then save result 
 *      into the 'outputFile' file.
 *      - save selection information into the 'outputSelFileName' file.
 *
 * @param inputFile name of the file containing AMBER data
 * @param outputFile name of the resulting file
 * @param selectionType name of the chosen selection criteria 
 * (amdlibNO_FRAME_SEL if no selection).
 * @param selectionRatio rato or threshold associated to the selection criteria.
 * @param verbose indicates if information have to be displayed. 
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibPerformSelection(const char *inputFile, 
                                    const char *inputSelFileName,
                                    const char *outputFile,
                                    const char *outputSelFileName,
                                    const amdlibFRAME_SELECTION selectionType,
                                    const double selectionRatio,
                                    const amdlibBOOLEAN useSelFile,
                                    const amdlibBOOLEAN saveSelFile,
                                    const amdlibBOOLEAN averageData)
{
    amdlibERROR_MSG  errMsg;
    amdlibPHOTOMETRY photometry = {NULL}, dstPhot = {NULL};
    amdlibPHOTOMETRY imdPhot[amdlibNB_BANDS];
    amdlibWAVELENGTH wave = {NULL}, dstWave = {NULL};
    amdlibWAVELENGTH imdWave[amdlibNB_BANDS];
    amdlibPISTON     opd = {NULL}, dstOpd = {NULL};
    amdlibPISTON     imdOpd[amdlibNB_BANDS];
    amdlibOI_TARGET  target = {NULL};
    amdlibOI_ARRAY   array = {NULL};
    amdlibVIS        vis = {NULL}, dstVis = {NULL};
    amdlibVIS        imdVis[amdlibNB_BANDS];
    amdlibVIS2       vis2 = {NULL}, dstVis2 = {NULL};
    amdlibVIS2       imdVis2[amdlibNB_BANDS];
    amdlibVIS3       vis3 = {NULL}, dstVis3 = {NULL};
    amdlibVIS3       imdVis3[amdlibNB_BANDS];
    amdlibINS_CFG    insCfg;
    amdlibINS_CFG    selKewList;
    amdlibSELECTION  selectedFrames[3];
    int band;

    amdlibTrace("amdlibPerformSelection()\n");

    /* Load input OI-FITS file */
    printf ("Loading OI-FITS file %s ...\n", inputFile);
    if (amdlibLoadOiFile(inputFile, &insCfg, &array, &target, &photometry, 
                         &vis, &vis2, &vis3, &wave, &opd, 
                         errMsg) != amdlibSUCCESS)
    {
        printf("amdlibPerformSelection(): Could not load OiFits file '%s'\n", 
               inputFile);
        printf("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    if (useSelFile == amdlibTRUE)
    {
        if (amdlibReadSelectionFile(inputSelFileName, &selKewList,
                                    selectedFrames, errMsg) == amdlibFAILURE)
        {
            printf("amdlibPerformSelection(): %s\n", errMsg);
            return amdlibFAILURE;
        }
    }
    else
    {
        for (band=amdlibJ_BAND; band <= amdlibK_BAND; band++)
        {
            amdlibAllocateSelection(&selectedFrames[band], vis.nbFrames);
        }
    }

    if (amdlibSplitOiStructures(&wave, imdWave, &photometry, imdPhot,
                                &vis, imdVis, &vis2, imdVis2, &vis3, imdVis3,
                                &opd, imdOpd, errMsg) != amdlibSUCCESS)
    {
        printf("amdlibPerformSelection(): %s\n", errMsg);        
        return amdlibFAILURE;
    }
    
    for (band = amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        /* If required, perform frame selection */
        if (imdOpd[band].thisPtr == NULL)
        {
            printf("amdlibPerformSelection(): Warning - no channel for band "
                   "%d. No selection for that band\n", band);
            if (saveSelFile == amdlibTRUE)
            {
                /* release selection structure for that band */
                amdlibReleaseSelection(&selectedFrames[band]);
                /* Allocate a new structure indicating all frames are not
                 * selected for that band */
                amdlibAllocateEmptySelection(&selectedFrames[band], 
                                             vis.nbFrames);
            }
        }
        else
        {
            /* Select 'good' frames depending on selection criterion. If no
             * criterion is specified, this function do nothing. */
            if (amdlibSelectFrames(&vis, &photometry, selectionType, 
                                   selectionRatio, &selectedFrames[band], 
                                   band, errMsg) != amdlibSUCCESS)
            {
                printf("amdlibPerformSelection(): Could not select good "
                       "frames\n");
                printf("%s\n", errMsg);
                return amdlibFAILURE;
            }

            if (averageData == amdlibTRUE)
            {
                /* Average visibilities, photometries and pistons on good 
                 * frames */
                if (amdlibAverageVisibilities(&imdPhot[band], &imdVis[band], 
                                              &imdVis2[band], &imdVis3[band], 
                                              &imdOpd[band], band, 
                                              &imdWave[band], 
                                              &selectedFrames[band], 
                                              errMsg) != amdlibSUCCESS)
                {
                    printf("amdlibPerformSelection(): Could not average data "
                           "after frame selection\n");
                    printf("%s\n", errMsg);
                    return amdlibFAILURE;
                }
            }
            
            if (amdlibMergeOiStructures(&dstWave, &imdWave[band],
                                        &dstPhot, &imdPhot[band],
                                        &dstVis, &imdVis[band],
                                        &dstVis2, &imdVis2[band],
                                        &dstVis3, &imdVis3[band],
                                        &dstOpd, &imdOpd[band],
                                        errMsg) != amdlibSUCCESS)
            {
                printf("amdlibPerformSelection(): Could not merge OI "
                       "structures\n");
                printf("%s\n", errMsg);
                return (amdlibFAILURE);
            }
        }
    }
    
   
    if (saveSelFile == amdlibTRUE)
    {
        if (amdlibWriteSelectionFile(outputSelFileName, vis.nbBases, 
                                     selectedFrames, errMsg) == amdlibFAILURE)
        {
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        if (amdlibAppendSelectionKeywords(outputFile, outputSelFileName,
                                          inputFile, &selKewList, amdlibFALSE, 
                                          selectionType,
                                          selectionRatio, 
                                          errMsg) != amdlibSUCCESS)
        {
            printf("amdlibPerformSelection(): Could not add PRO keywords -"
                   " %s", errMsg);
            return amdlibFAILURE;
        }
    }

    if (averageData == amdlibTRUE)
    {
        printf ("Saving OI-FITS file %s ...\n", outputFile);
        if (amdlibSaveOiFile(outputFile, &insCfg, &array, &target, &dstPhot, 
                             &dstVis, &dstVis2, &dstVis3, &dstWave, &dstOpd, 
                             errMsg) != amdlibSUCCESS)
        {
            printf("amdlibPerformSelection(): Could not save result file\n");
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        
        if (amdlibAppendSelectionKeywords(outputFile, outputSelFileName,
                                          inputFile, &selKewList, amdlibTRUE, 
                                          selectionType,
                                          selectionRatio, 
                                          errMsg) != amdlibSUCCESS)
        {
            printf("amdlibPerformSelection(): Could not add PRO keywords -"
                   " %s", errMsg);
            return amdlibFAILURE;
        }
    }

    /* Release all sata structures */
    for (band=amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        amdlibReleaseSelection(&selectedFrames[band]);
        amdlibReleasePhotometry(&imdPhot[band]);
        amdlibReleaseWavelength(&imdWave[band]);
        amdlibReleasePiston(&imdOpd[band]);
        amdlibReleaseVis(&imdVis[band]);
        amdlibReleaseVis2(&imdVis2[band]);
        amdlibReleaseVis3(&imdVis3[band]);
    }
    amdlibReleasePhotometry(&photometry);
    amdlibReleasePhotometry(&dstPhot);
    amdlibReleaseWavelength(&wave);
    amdlibReleaseWavelength(&dstWave);
    amdlibReleasePiston(&opd);
    amdlibReleasePiston(&dstOpd);
    amdlibReleaseVis(&vis);
    amdlibReleaseVis(&dstVis);
    amdlibReleaseVis2(&vis2);
    amdlibReleaseVis2(&dstVis2);
    amdlibReleaseVis3(&vis3);
    amdlibReleaseVis3(&dstVis3);
    amdlibReleaseOiArray(&array);
    amdlibReleaseOiTarget(&target);

    return amdlibSUCCESS;
}

/**
 * Allocate memory for structure containing necessary information for empty 
 * frame selection, ie if no frame is selected for concerned spectral band.
 *
 * @param selection structure to be allocated.
 * @param nbFrames total number of frames in current observation.
 *
 * @return 
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibAllocateEmptySelection(amdlibSELECTION *selection,
                                              int nbFrames)
{
    int base;
    int i;

    amdlibTrace("amdlibAllocateEmptySelection()\n");

    selection->nbFrames = nbFrames;
    for (base = 0; base < amdlibNBASELINE; base++)
    {
        selection->nbSelectedFrames[base] = 0;
        selection->isSelected[base] = calloc(nbFrames, sizeof(char));
        if (selection->isSelected[base] == NULL)
        {
            return amdlibFAILURE;
        }
        memset(selection->isSelected[base], 0, nbFrames * sizeof(char));
    }

    /* Allocate memory for  array */
    selection->framesOkForClosure = calloc(nbFrames, sizeof(int));
    for (i = 0; i < nbFrames; i++)
    {
        selection->framesOkForClosure[i] = 0;
    }
    selection->nbFramesOkForClosure = 0;

    return amdlibSUCCESS;
}

/**
 * This function reads given fits file and extracts selection information.
 * 
 * @param fileName name of the file to read.
 * @param selection array of selection structures, one per spectral band.
 * @param errMsg error description message returned if function fails.
 *
 * @return 
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibReadSelectionFile(const char      *fileName,
                                         amdlibINS_CFG   *selKeywList,
                                         amdlibSELECTION *selection,
                                         amdlibERROR_MSG errMsg)
{
    fitsfile *filePtr;
    int status = 0;
    char fitsioMsg[256];
    int band, base, i;
    int nbFrames;
    long nbRows;
    char keyName[36];
    char keyVal[36];
    char comment[amdlibKEYW_CMT_LEN+1];
    char nullLogical = amdlibFALSE;
    char *boolTab;   
    int anynull;
    int keysExist = 0;
    int moreKeys = 0;
    amdlibKEYW_LINE record;
    
    /* Check a file name has been specified */
    if (strlen(fileName) == 0)
    {
        sprintf(errMsg, "amdlibReadSelectionFile() : "
                "no input file name specified\n");
        return amdlibFAILURE;
    }
    
    /* Open FITS file */
    if (fits_open_file(&filePtr, fileName, READONLY, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibReadSelectionFile() : "
                "impossible to read fits file %s - %s\n", fileName,
                fitsioMsg);
        return amdlibFAILURE;
    }    

    /* Get primary header's keywords */
    keysExist = 0;
    moreKeys = 0;
    if (fits_get_hdrspace(filePtr, &keysExist, &moreKeys, &status) != 0)
    {
        status = 0;
    }
    for (i = 1; i <= keysExist; i++)
    {
        /* Read keyword line #i */
        if (fits_read_record(filePtr, i, record, &status) != 0)
        {
            status = 0;
        }
        else
        {
            if (strstr(record, "COMMENT ") == NULL)
            {
                /* Store keyword */
                if (amdlibAddInsCfgKeyword(selKeywList, 
                                           record, errMsg) == amdlibFAILURE)
                {
                    return amdlibFAILURE;
                }
            }
        }
    }
    
    /* Get all rows data */
    if (fits_movnam_hdu(filePtr, BINARY_TBL, "FRAME_SELECTION", 0, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibReadSelectionFile() : "
                "No FRAME_SELECTION table - %s\n", fitsioMsg);
        return amdlibSUCCESS;
    }  
    
    if (fits_get_num_rows(filePtr, &nbRows, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibReadSelectionFile() : %s\n", fitsioMsg);
        return amdlibSUCCESS;
    }
    
    for (band=amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        sprintf(keyName, "TFORM%d", band+1);
        if (fits_read_key(filePtr, TSTRING, keyName, keyVal, 
                          comment, &status))
        {
            fits_get_errstatus(status, (char*)fitsioMsg);
            sprintf(errMsg, "amdlibReadSelectionFile() : %s\n", fitsioMsg);
            return amdlibSUCCESS;
        }
        sscanf(keyVal, "%dL", &nbFrames);
        amdlibAllocateSelection(&selection[band], nbFrames);

        for (base=1; base <= nbRows; base++)
        {
            boolTab = calloc(nbFrames, sizeof(char));
            if (boolTab == NULL)
            {
                sprintf(errMsg, "amdlibReadSelectionFile() : "
                        "could not allocate memory for temporary array\n");
                return amdlibFAILURE;
            }
            memset(boolTab, 1, nbFrames * sizeof(char));

            if (fits_read_col(filePtr, TLOGICAL, band+1, base, 1, nbFrames,
                              &nullLogical, boolTab,
                              &anynull, &status))
            {
                fits_get_errstatus(status, (char*)fitsioMsg);
                sprintf(errMsg, "amdlibReadSelectionFile() : %s\n", fitsioMsg);
                return amdlibSUCCESS;
            }
            for (i=0; i < nbFrames; i++)
            {
                if (boolTab[i] == amdlibFALSE)
                {
                    selection[band].isSelected[base-1][i] = boolTab[i];
                    selection[band].nbSelectedFrames[base-1]--;
                }
            }
        }
        amdlibCompleteSelection(&selection[band]);
    }
    

    /* Close file */
    if (fits_close_file(filePtr, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibReadSelectionFile() : "
                "impossible to close file %s - %s\n", fileName, fitsioMsg);
        return amdlibFAILURE;
    }
    
    return amdlibSUCCESS;
}

/**
 * This function stores given selection structures into a binary table.
 * 
 * @param fileName name of the file to be written.
 * @param nbBases number of baselines.
 * @param selection array of selection structures, one per spectral band.
 * @param errMsg error description message returned if function fails.
 *
 * @return 
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibWriteSelectionFile(const char      *fileName,
                                          int             nbBases,
                                          amdlibSELECTION *selection,
                                          amdlibERROR_MSG errMsg)
{
    struct stat statBuf;
    fitsfile *filePtr;
    int status = 0;
    char fitsioMsg[256];
    int  tfields = 3; 
    char *ttype[] = {"J", "H", "K"};
    char *tform[3];    
    char *tunit[] = {"", "", ""};
    int band, base, i;
    char *boolTab;
    int index;
    int nbFrames;
    time_t      timeSecs;
    struct tm   *timeNow;
    char        strTime[amdlibKEYW_VAL_LEN+1];
   
    /* Check a file name has been specified */
    if (strlen(fileName) == 0)
    {
        sprintf(errMsg, "amdlibWriteSelectionFile() : "
                "no input file name specified\n");
        return amdlibFAILURE;
    }
    
    /* First remove previous IO file (if any) */
    if (stat(fileName, &statBuf) == 0)
    {
        if (remove(fileName) != 0)
        {
            sprintf(errMsg, "amdlibWriteSelectionFile(): "
                    "could not overwrite file %s", fileName); 
            return (amdlibFAILURE);
        }
    }

    /* Create file */
    if (fits_create_file(&filePtr, fileName, &status) != 0)
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibWriteSelectionFile() : "
                "impossible to create fits file %s - %s\n", fileName,
                fitsioMsg);
        return amdlibFAILURE;
    }

    for (band=amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        tform[band] = calloc(36, sizeof(char));
        sprintf(tform[band], "%dL", selection[band].nbFrames);
    }
    
    /* Create binary table */
    if (fits_create_tbl(filePtr, BINARY_TBL, 1, tfields, ttype, tform, tunit, 
                        "FRAME_SELECTION", &status) != 0)
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibWriteSelectionFile() : "
                "impossible to create binary table - %s\n", fitsioMsg);
        fits_close_file(filePtr, &status);
        return amdlibFAILURE;
    }

    /* Write columns */
    for (band=amdlibJ_BAND; band <= amdlibK_BAND; band++)
    {
        nbFrames = selection[band].nbFrames;
        boolTab = calloc(nbBases * nbFrames, sizeof(char));
        if (boolTab == NULL)
        {
            sprintf(errMsg, "amdlibWriteSelectionFile() : "
                    "could not allocate memory for temporary array\n");
            fits_close_file(filePtr, &status);
            return amdlibFAILURE;
        }
        index = 0;
        for (base=0; base < nbBases; base++)
        {
            for (i=0; i < selection[band].nbFrames; i++)
            {
                boolTab[index] = selection[band].isSelected[base][i];
                index++;
            }
        }
        if (fits_write_col(filePtr, TLOGICAL, band+1, 1, 1, 
                           nbBases*selection[band].nbFrames, boolTab, 
                           &status) != 0)
        {
            fits_get_errstatus(status, (char*)fitsioMsg);
            sprintf(errMsg, "amdlibWriteSelectionFile() : "
                    "impossible to write col %d - %s\n", band+1, fitsioMsg);
            free(boolTab);
            fits_close_file(filePtr, &status);
            return amdlibFAILURE;
        }
        free(boolTab);
    }

    /* Write keywords in primary header */
    if (fits_movabs_hdu(filePtr, 1,0, &status) != 0)
    {
        fits_close_file(filePtr, &status);
    }
    timeSecs = time(NULL);
    timeNow = gmtime(&timeSecs);
    strftime(strTime, sizeof(strTime), "%Y-%m-%dT%H:%M:%S", timeNow);
    if (fits_write_key(filePtr, TSTRING, "DATE", strTime, 
                       "Date this file was written", &status) != 0)
    {
        fits_close_file(filePtr, &status);
        sprintf(errMsg, "Could not complete main header");
        return amdlibFAILURE;
    }

    /* Close file */
    if (fits_close_file(filePtr, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibWriteSelectionFile() : "
                "impossible to close file %s - %s\n", fileName, fitsioMsg);
        return amdlibFAILURE;
    }
    
    return amdlibSUCCESS;
}

/**
 * Merge p2vms
 * 
 * Not yet implemented
 * 
 * @param nbFiles number of P2VM to merge
 * @param p2vmFile list of names of files containing P2VM 
 * @param outputFile name of file containing resulting P2VM
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibMergeP2vmFromFiles(const int  nbFiles,
                                          const char **p2vmFile,
                                          const char *outputFile)
{
    
    static amdlibP2VM_MATRIX p2vm1;
    static amdlibP2VM_MATRIX p2vm2;
    static amdlibP2VM_MATRIX p2vmRes;
    amdlibERROR_MSG          errMsg;
    int                      i;

    amdlibTrace("amdlibMergeP2vmFromFiles()\n");

    if (nbFiles < 2)
    {
        printf("amdlibMergeP2vmFromFiles(): Needs at least 2 input files");
        return (amdlibFAILURE);
    }

    if (amdlibLoadP2VM(p2vmFile[0], &p2vmRes, errMsg) != amdlibSUCCESS)
    {
        printf("amdlibMergeP2vmFromFiles(): Could not load p2vm file '%s'\n", 
               p2vmFile[0]);
        printf("%s\n", errMsg);
        return (amdlibFAILURE);
    }
    for (i = 1; i < nbFiles; i++)
    {
        if (amdlibDuplicateP2VM(&p2vmRes, &p2vm1, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibMergeP2vmFromFiles(): Could not duplicate p2vm\n");
            printf("%s\n", errMsg);
            amdlibReleaseP2VM(&p2vm2);
            amdlibReleaseP2VM(&p2vmRes);
            return (amdlibFAILURE);   
        }
        amdlibReleaseP2VM(&p2vmRes);
        if (amdlibLoadP2VM(p2vmFile[i], &p2vm2, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibMergeP2vmFromFiles(): Could not load p2vm file "
                   "'%s'\n", p2vmFile[i]);
            printf("%s\n", errMsg);
            amdlibReleaseP2VM(&p2vm1);
            amdlibReleaseP2VM(&p2vmRes);
            return (amdlibFAILURE);
        }
        if (amdlibMergeP2VM(&p2vm1, &p2vm2, &p2vmRes, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibMergeP2vmFromFiles(): Could not merge p2vm\n");
            printf("%s\n", errMsg);
            amdlibReleaseP2VM(&p2vm1);
            amdlibReleaseP2VM(&p2vm2);
            amdlibReleaseP2VM(&p2vmRes);
            return (amdlibFAILURE);            
        }
    }

    
    if (amdlibSaveP2VM(outputFile, &p2vmRes, amdlibP2VM_UNKNOWN_ACC,
                       errMsg) != amdlibSUCCESS)
    {
        printf ("%s\n", errMsg);
        amdlibReleaseP2VM(&p2vm1);
        amdlibReleaseP2VM(&p2vm2);
        amdlibReleaseP2VM(&p2vmRes);
        return (amdlibFAILURE);
    }

    amdlibReleaseP2VM(&p2vm1);
    amdlibReleaseP2VM(&p2vm2);
    amdlibReleaseP2VM(&p2vmRes);
    return (amdlibSUCCESS);
}

/**
 * Append Oi-Fits Files into a single one
 *  
 * @param nbFiles number of OI-FITS files to append
 * @param oiFitsFile list of names of OI-FITS files to be append
 * @param outputFile name of resulting file
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibAppendOiFitsFiles(const int  nbFiles,
                                         const char **oiFitsFile,
                                         const char *outputFile)
{
    amdlibPHOTOMETRY srcPhot   = {NULL};
    amdlibPHOTOMETRY dstPhot   = {NULL};
    amdlibWAVELENGTH srcWave   = {NULL};
    amdlibWAVELENGTH dstWave   = {NULL};
    amdlibPISTON     srcPst    = {NULL};
    amdlibPISTON     dstPst    = {NULL};
    amdlibOI_TARGET  target    = {NULL};
    amdlibOI_ARRAY   array     = {NULL};
    amdlibVIS        srcVis    = {NULL};
    amdlibVIS        dstVis    = {NULL};
    amdlibVIS2       srcVis2   = {NULL};
    amdlibVIS2       dstVis2   = {NULL};
    amdlibVIS3       srcVis3   = {NULL};
    amdlibVIS3       dstVis3   = {NULL};
    amdlibINS_CFG    insCfg;
    amdlibERROR_MSG  errMsg;
    int  i;
    int  nbFrames = 0;
    int  nbBases = 0;
    int  nbWlen = 0;
    char arrayName[16];
    
    amdlibTrace("amdlibAppendOiFitsFiles()\n");

    if (nbFiles < 2)
    {
        printf("amdlibAppendOiFitsFiles(): Needs at least 2 input files");
        return (amdlibFAILURE);
    }

    /* Load first file */
    amdlibClearInsCfg(&insCfg);
    if (amdlibLoadOiFile(oiFitsFile[0], &insCfg, &array, &target, &dstPhot, 
                         &dstVis, &dstVis2, &dstVis3, 
                         &dstWave, &dstPst, errMsg) != amdlibSUCCESS)
    {
        printf("amdlibAppendOiFitsFiles(): Could not load OiFits file '%s'\n", 
               oiFitsFile[0]);
        printf("%s\n", errMsg);
        return (amdlibFAILURE);
    }
    strcpy(arrayName, array.arrayName);

    nbFrames = dstVis.nbFrames;
    nbBases = dstVis.nbBases;
    nbWlen = dstVis.nbWlen;

    /* Loop on files and append their data */
    for (i = 1; i < nbFiles; i++)
    {
        /* Load file */
        if (amdlibLoadOiFile(oiFitsFile[i], NULL, &array, &target, &srcPhot, 
                             &srcVis, &srcVis2, &srcVis3, 
                             &srcWave, &srcPst, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibAppendOiFitsFiles(): Could not load OiFits file "
                   "'%s'\n", oiFitsFile[i]);
            printf("%s\n", errMsg);
            return (amdlibFAILURE);
        }
        /* Perform simple checks */
        if (strcmp(arrayName, array.arrayName) != 0)
        {
            printf("amdlibAppendOiFitsFiles(): Different ARRNAME in file "
                   "'%s' (%s) and in previous files(s) (%s)\n", 
                   oiFitsFile[i], arrayName, array.arrayName);
            return (amdlibFAILURE);
        }
        if ((nbBases != srcVis.nbBases) || (nbWlen != srcVis.nbWlen))
        {
            printf("amdlibAppendOiFitsFiles(): Different number of "
                   "Bases (%d vs. %d) or Wlen (%d vs. %d) in file '%s' "
                   "and in previous files(s)\n", nbBases, srcVis.nbBases,
                   nbWlen, srcVis.nbWlen, oiFitsFile[i]);
            return (amdlibFAILURE);
        }

        /* Check amdlibWAVELENGTH structures are identical */
        if (amdlibCompareWavelengths(&srcWave, &dstWave, errMsg) != amdlibTRUE)
        {
            printf("amdlibAppendOiFitsFiles(): Could not append files with "
                   "different wavelength data structures ('%s' and '%s')\n",
                   oiFitsFile[i-1], oiFitsFile[i]);
            printf("%s\n", errMsg);
            return amdlibFAILURE; 
        }
        
        /* Append photometry */
        if (amdlibAppendPhotometry(&dstPhot, &srcPhot, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibAppendOiFitsFiles(): Could not append photometry from"
                   "file '%s'\n", oiFitsFile[i]);
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        /* Append vis */
        if (amdlibAppendVis(&dstVis, &srcVis, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibAppendOiFitsFiles(): Could not append vis from file "
                   "'%s'\n", oiFitsFile[i]);
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        /* Append vis2 */
        if (amdlibAppendVis2(&dstVis2, &srcVis2, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibAppendOiFitsFiles(): Could not append vis2 from file "
                   "'%s'\n", oiFitsFile[i]);
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        /* Append vis3 */
        if (nbBases > 1)
        {
            if (amdlibAppendVis3(&dstVis3, &srcVis3, errMsg) != amdlibSUCCESS)
            {
                printf("amdlibAppendOiFitsFiles(): Could not append vis3 from "
                       "file '%s'\n", oiFitsFile[i]);
                printf("%s\n", errMsg);
                return amdlibFAILURE;
            }
        }
        /* Append piston */
        if (amdlibAppendPiston(&dstPst, &srcPst, errMsg) != amdlibSUCCESS)
        {
            printf("amdlibAppendOiFitsFiles(): Could not append piston from "
                   "file '%s'\n", oiFitsFile[i]);
            printf("%s\n", errMsg);
            return amdlibFAILURE;
        }
        
    }
    
    /* Save output file */
    if(amdlibSaveOiFile(outputFile, &insCfg, &array, &target, &dstPhot,
                        &dstVis, &dstVis2, &dstVis3, &dstWave, &dstPst, 
                        errMsg) != amdlibSUCCESS)
    {
        printf("amdlibAppendOiFitsFiles(): Error writing output file '%s'\n",
               outputFile );
        printf("%s\n", errMsg);
        return (amdlibFAILURE);
    }

    amdlibReleaseVis(&srcVis);
    amdlibReleaseVis(&dstVis);
    amdlibReleaseVis2(&srcVis2);
    amdlibReleaseVis2(&dstVis2);
    amdlibReleaseVis3(&srcVis3);
    amdlibReleaseVis3(&dstVis3);
    amdlibReleaseOiArray(&array);
    amdlibReleaseWavelength(&srcWave);
    amdlibReleaseWavelength(&dstWave);
    amdlibReleasePiston(&srcPst);
    amdlibReleasePiston(&dstPst);
    amdlibReleasePhotometry(&srcPhot);
    amdlibReleasePhotometry(&dstPhot);
    amdlibReleaseOiTarget(&target);


    return (amdlibSUCCESS);
}
/*
 * Local functions
 */

/** Usefull macro to error when writing PRO keywords */ 
#define amdlibPROReturnError(routine)                    \
    fits_close_file(filePtr, &status);                   \
    fits_get_errstatus(status, (char*)fitsioMsg);        \
    sprintf(errMsg, "%s - %s\n", routine, fitsioMsg);    \
    return (amdlibFAILURE);
#ifndef ESO_CPL_PIPELINE_VARIANT
/**
 * Add PRO keywords for P2VM 
 *
 * @param p2vmFile name of file containing P2VM
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param inputFiles names of the files containing AMBER data
 * @param nbInputFiles number of files containing AMBER data
 * @param newSpectralOffsets offset to be applied on photometric channels 
 * @param addOffsets true iif new spectral offsets are considered.
 * @param errMsg error description message returned if function fails
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibAppendKeywordListToP2VM(const char *p2vmFile,
                                               const char *badPixelFile,
                                               const char *flatFieldFile,
                                               const char **inputFiles,
                                               int nbInputFiles,
                                               float newSpectralOffsets[],
                                               amdlibBOOLEAN addOffsets,
                                               amdlibERROR_MSG errMsg)
{
    char     keyName[36];
    char     keyVal[80], *keyValBis;
    int      i;
    fitsfile *filePtr;
    int      status = 0;
    char     fitsioMsg[256];
    int      nbTel;
    int      nbRaws;
    
    amdlibTrace("amdlibAppendKeywordListToFile()\n");
    
    if(fits_open_file(&filePtr, p2vmFile, READWRITE, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibAppendKeywordListToP2VM - "
                "%s file cannot be opened in append mode : %s\n", p2vmFile,
                fitsioMsg);
        return amdlibFAILURE;
    }

    if (fits_movabs_hdu(filePtr, 1, 0, &status) != 0)
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibAppendKeywordListToP2VM - %s\n", fitsioMsg);
        return amdlibFAILURE;
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO DID", "AMBER",
                       "Data dictionnary for PRO", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO CATG", 
                       "P2VM_REDUCED",
                       "Category of product frames", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO TYPE", "REDUCED",
                       "Product type", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 ID", 
                       "amdlibComputeP2vm",
                       "Program identifier", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }
            
    /* Add information relative to bad pixel file */
    strcpy(keyVal, badPixelFile);
    keyValBis = basename(keyVal);
    sprintf(keyVal, "%.40s", keyValBis);
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW1 NAME", 
                       keyVal, "File name of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW1 CATG", 
                       "AMBER_BADPIX",
                       "Category of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }

    /* Add information relative to flat field file */
    strcpy(keyVal, flatFieldFile);
    keyValBis = basename(keyVal);
    sprintf(keyVal, "%.40s", keyValBis);
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW2 NAME", 
                       keyVal, "File name of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW2 CATG", 
                       "AMBER_FLATFIELD",
                       "Category of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }

    if (nbInputFiles == 5)
    {
        nbTel = 2;
    }
    else if (nbInputFiles == 10)
    {
        nbTel = 3;
    }
    else
    {
        sprintf(errMsg, "amdlibAppendKeywordListToP2VM - invalid parameter "
                "nbInputFiles (%d) should be 5 or 10", nbInputFiles);
        return amdlibFAILURE;
    }
    for (i = 0; i < nbInputFiles; i++)
    {
        /* Add information relative to input files name (PRO keywords) */
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d NAME", i+3);
        strcpy(keyVal, inputFiles[i]);
        keyValBis = basename(keyVal);
        sprintf(keyVal, "%.40s", keyValBis);
        if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                           "File name of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d CATG", i+3);
        if (fits_write_key(filePtr, TSTRING, keyName, "AMBER_2P2V",
                           "Category of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
        }
    }
    
    nbRaws = i+2;
    if (fits_write_key(filePtr, TINT, "HIERARCH ESO PRO DATANCOM", &nbRaws,
                       "Number of combined frames", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }
    
    if (addOffsets == amdlibTRUE)
    {
        if (fits_write_key(filePtr, TSTRING, 
                           "HIERARCH ESO PRO REC1 PARAM1 NAME", "-s",
                           "", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
        }
        if (nbTel == 2)
        {
            sprintf(keyVal, "%f,%f", newSpectralOffsets[0], 
                    newSpectralOffsets[1]);

        }
        else
        {
            sprintf(keyVal, "%f,%f,%f", newSpectralOffsets[0], 
                    newSpectralOffsets[1], newSpectralOffsets[2]);
        }
        if (fits_write_key(filePtr, TSTRING, 
                           "HIERARCH ESO PRO REC1 PARAM1 VALUE", keyVal,
                           "", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
        }
    }

    /* Close FITS file */
    if (fits_close_file(filePtr, &status)) 
    {
        amdlibPROReturnError("amdlibAppendKeywordListToP2VM ");
    }


    return amdlibSUCCESS;
}

/**
 * Add PRO keywords in insCfg structure for OI-FITS result file
 * 
 * @param oifitsFile name of the OI-FITS result file.
 * @param badPixelFile name of file containing bad pixel map
 * @param flatFieldFile name of file containing flat-field map
 * @param darkFile name of file containing dark (if any)
 * @param skyFile name of file containing sky data (if any)
 * @param inputFile name of the input science file 
 * @param nbBinning number of binnings 
 * @param errorType indicates wether the noise figures are estimated 
 * statistically from the sequence of photometries during the bin 
 * (amdlibSTATISTICAL_ERROR) or theoretically by the poisson error on N photons
 * (amdlibTHEORETICAL_ERROR). The latter is forced obviously when the binsize
 * is 1. 
 * @param pistonType indicates wether the piston is to be measured by fitting
 * a slope in the phasors amdlibITERATIVE_PHASOR or in the differential phases
 * after dewrapping (amdlibUNWRAPPED_PHASE). 
 * @param noCheckP2vmId forces amdlib to use without wuestion the passed p2vm,
 * even if its magic number is not OK. Can happen if the P2VM has been 
 * observed AFTER the science observations. 
 * @param mergeOutputFiles indicates if many files will be produced (a single 
 * one if TRUE or as much as there are bands otherwise).
 * @param selectionType name of the chosen selection criteria 
 * (amdlibNO_FRAME_SEL if no selection).
 * @param selectionRatio rato or threshold associated to the selection criteria.
 * @param errMsg error description message returned if function fails
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibAppendKeywordListToOIFITS(
                                const char *oifitsFile,
                                const char *badPixelFile,
                                const char *flatFieldFile,
                                const char *darkFile,
                                const char *skyFile,
                                const char *inputFile, 
                                const int   nbBinning,
                                const amdlibERROR_TYPE errorType,
                                const amdlibPISTON_ALGORITHM pistonType,
                                const amdlibBOOLEAN noCheckP2vmId,
                                const amdlibBOOLEAN mergeOutputFiles,
                                const amdlibFRAME_SELECTION selectionType,
                                const double selectionRatio,
                                amdlibERROR_MSG errMsg)
{
    fitsfile *filePtr;
    int      status = 0;
    char     fitsioMsg[256];
    char     keyName[36];
    char     keyVal[80], *keyValBis;
    int      index;
    
    if(fits_open_file(&filePtr, oifitsFile, READWRITE, &status))
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibAppendKeywordListToOIFITS - "
                "%s file cannot be opened in append mode : %s\n", oifitsFile,
                fitsioMsg);
        return amdlibFAILURE;
    }

    if (fits_movabs_hdu(filePtr, 1, 0, &status) != 0)
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibAppendKeywordListToOIFITS - %s\n", fitsioMsg);
        return amdlibFAILURE;
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO DID", "AMBER",
                       "Data dictionnary for PRO", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO CATG", 
                       "SCIENCE_REDUCED",
                       "Category of product frames", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO TYPE", "REDUCED",
                       "Product type", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 ID", 
                       "amdlibExtractVis",
                       "Program identifier", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
            
    /* Add information relative to bad pixel file */
    strcpy(keyVal, badPixelFile);
    keyValBis = basename(keyVal);
    sprintf(keyVal, "%.40s", keyValBis);
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW1 NAME", 
                       keyVal, "File name of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW1 CATG", 
                       "AMBER_BADPIX",
                       "Category of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    /* Add information relative to flat field file */
    strcpy(keyVal, flatFieldFile);
    keyValBis = basename(keyVal);
    sprintf(keyVal, "%.40s", keyValBis);
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW2 NAME", 
                       keyVal, "File name of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    if (fits_write_key(filePtr, TSTRING, "HIERARCH ESO PRO REC1 RAW2 CATG", 
                       "AMBER_FLATFIELD",
                       "Category of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    index = 2;
    /* Add information relative to dark file (if any) (PRO keywords) */
    if (strlen(darkFile) != 0)
    {
        index += 1;
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d NAME", index);
        strcpy(keyVal, darkFile);
        keyValBis = basename(keyVal);
        sprintf(keyVal, "%.40s", keyValBis);
        if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                           "File name of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d CATG", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "AMBER_DARK",
                           "Category of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
    }
    
    /* Add information relative to sky file (if any) (PRO keywords) */
    if (strlen(skyFile) != 0)
    {
        index += 1;
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d NAME", index);
        strcpy(keyVal, skyFile);
        keyValBis = basename(keyVal);
        sprintf(keyVal, "%.40s", keyValBis);
        if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                           "File name of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d CATG", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "AMBER_SKY",
                           "Category of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
    }
    
    /* Add information relative to input science file (PRO keywords) */
    index += 1;
    sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d NAME", index);
    strcpy(keyVal, inputFile);
    keyValBis = basename(keyVal);
    sprintf(keyVal, "%.40s", keyValBis);
    if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                       "File name of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    sprintf(keyName, "HIERARCH ESO PRO REC1 RAW%d CATG", index);
    if (fits_write_key(filePtr, TSTRING, keyName, "AMBER_SCIENCE",
                       "Category of raw frame", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    if (fits_write_key(filePtr, TINT, "HIERARCH ESO PRO DATANCOM", &index,
                       "Number of combined frames", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }

    /* Add information relative to parameters */
    index = 1;
    sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
    if (fits_write_key(filePtr, TSTRING, keyName, "-e",
                       "Parameter name", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
    if (errorType == amdlibSTATISTICAL_ERROR)
    {
        strcpy(keyVal, "STATISTIC");
    }
    else if (errorType == amdlibTHEORETICAL_ERROR)
    {
        strcpy(keyVal, "THEORETIC");
    }
    if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                       "Parameter value", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    index ++;

    sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
    if (fits_write_key(filePtr, TSTRING, keyName, "-p",
                       "Parameter name", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
    if (pistonType == amdlibITERATIVE_PHASOR)
    {
        strcpy(keyVal, "PHASOR");
    }
    else if (pistonType == amdlibUNWRAPPED_PHASE)
    {
        strcpy(keyVal, "PHASE");    
    }
    if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                       "Parameter value", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }
    index ++;

    sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
    if (mergeOutputFiles == amdlibFALSE)
    {
        if (fits_write_key(filePtr, TSTRING, keyName, "-s",
                           "Parameter name", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "",
                           "Parameter value", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        index ++;
    }

    if (noCheckP2vmId == amdlibTRUE)
    {
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "-f",
                           "Parameter name", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "",
                           "Parameter value", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        index ++;
    }
    
    if (nbBinning != 999999)
    {
        int binning;
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "-b",
                           "Parameter name", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
        binning = nbBinning;
        if (fits_write_key(filePtr, TINT, keyName, &binning,
                           "Parameter value", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        index ++;
    }

    if (selectionType != amdlibNO_FRAME_SEL)
    {
        float ratio;
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "-c",
                           "Parameter name", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
        if ((selectionType == amdlibFLUX_SEL_PCG) || 
            (selectionType == amdlibFLUX_SEL_THR))
        {
            strcpy(keyVal, "FLUX");
        }
        else if ((selectionType == amdlibFRG_CONTRAST_SEL_PCG) ||
                 (selectionType == amdlibFRG_CONTRAST_SEL_THR))
        {
            strcpy(keyVal, "FRG");
        }
        if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                           "Parameter value", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        index ++;
    
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d NAME", index);
        if (fits_write_key(filePtr, TSTRING, keyName, "-r",
                           "Parameter name", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC1 PARAM%d VALUE", index);
        ratio = selectionRatio;
        if (fits_write_key(filePtr, TFLOAT, keyName, &ratio,
                           "Parameter value", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
        }
        index ++;
    }

    /* Close FITS file */
    if (fits_close_file(filePtr, &status)) 
    {
        amdlibPROReturnError("amdlibAppendKeywordListToOIFITS ");
    }


    return amdlibSUCCESS;
}
#endif

/**
 * Add PRO keywords in insCfg structure for OI-FITS result file after having
 * performed a frame selection.
 * 
 * @param oifitsFile name of the OI-FITS result file.
 * @param selectionType name of the chosen selection criteria 
 * (amdlibNO_FRAME_SEL if no selection).
 * @param selectionRatio rato or threshold associated to the selection criteria.
 * @param errMsg error description message returned if function fails
 *
 * @return
 * amdlibSUCCESS on successful completion. Otherwise amdlibFAILURE is returned.
 */
amdlibCOMPL_STAT amdlibAppendSelectionKeywords(
                                 const char *oifitsFile,
                                 const char *selFile,
                                 const char *inOiFits,
                                 amdlibINS_CFG *selKeywList,
                                 amdlibBOOLEAN saveInOiFits,
                                 const amdlibFRAME_SELECTION selectionType,
                                 const double selectionRatio,
                                 amdlibERROR_MSG errMsg)
{
    fitsfile *filePtr;
    int      status = 0;
    char     fitsioMsg[256];
    char     keyName[36];
    char     name[36], suffix[36];
    char     keyVal[80], *keyValBis;
    char     keyCmt[amdlibKEYW_CMT_LEN+1];
    int      index;
    int      ind1, ind2, i;
    int      recNb;
    float    ratio;
    amdlibKEYW_LINE keywLine;
    
    recNb = 0;
    if (saveInOiFits == amdlibTRUE)
    {
        if (fits_open_file(&filePtr, oifitsFile, READWRITE, &status))
        {
            fits_get_errstatus(status, (char*)fitsioMsg);
            sprintf(errMsg, "amdlibAppendSelectionKeywords - "
                    "%s file cannot be opened in append mode : %s\n", 
                    oifitsFile, fitsioMsg);
            return amdlibFAILURE;
        }
        /* Search for recipe number */
        index = 0;
        while (recNb == 0)
        {
            index ++;
            sprintf(keyName, "HIERARCH ESO PRO REC%d ID", index);
            if (fits_read_key(filePtr, TSTRING, keyName, keyVal, keyCmt, 
                              &status) != 0)
            {
                recNb = index;
                status = 0;
            }
        }
    }
    else
    {
        if (fits_open_file(&filePtr, selFile, READWRITE, &status))
        {
            fits_get_errstatus(status, (char*)fitsioMsg);
            sprintf(errMsg, "amdlibAppendSelectionKeywords - "
                    "%s file cannot be opened in append mode : %s\n", 
                    selFile, fitsioMsg);
            return amdlibFAILURE;
        }
        recNb = 1;
    }

    
    /* Move to primary header */
    if (fits_movabs_hdu(filePtr, 1, 0, &status) != 0)
    {
        fits_get_errstatus(status, (char*)fitsioMsg);
        sprintf(errMsg, "amdlibAppendSelectionKeywords - %s\n", 
                fitsioMsg);
        return amdlibFAILURE;
    }
    
    /* Append keywords stored in selKeywList (if any) */
    if (selKeywList != NULL)
    {
        if (saveInOiFits == amdlibFALSE)
        {
            for (i = 0; i < selKeywList->nbKeywords; i++)
            {
                if (strstr(selKeywList->keywords[i].name, "DATE   ") == NULL)
                {
                    if ((strstr(selKeywList->keywords[i].name, 
                               "HIERARCH ESO PRO REC") != NULL) &&
                        (strstr(selKeywList->keywords[i].name, "ID") != NULL))
                    {
                        recNb++;
                    }
                    sprintf((char*)keywLine, "%s=%s/%s", 
                            selKeywList->keywords[i].name,
                            selKeywList->keywords[i].value, 
                            selKeywList->keywords[i].comment);
                    if (fits_write_record(filePtr, keywLine, &status) != 0)
                    {
                        fits_close_file(filePtr, &status);
                        sprintf(errMsg, "Could not complete main header");
                        return amdlibFAILURE;
                    }
                }
            }
        }
        else
        {
            ind2 = 0;
            for (i = 0; i < selKeywList->nbKeywords; i++)
            {
                strcpy(name, selKeywList->keywords[i].name);
                if (strstr(name, "HIERARCH") != NULL)
                {
                    if (sscanf(name, "HIERARCH ESO PRO REC%d %[^^]", &ind1, 
                               suffix) != 0)
                    {
                        if ((ind2 != 0) && (ind1 != ind2))
                        {
                            recNb++;
                            ind2 = ind1;
                        }
                        sprintf(keyName, "HIERARCH ESO PRO REC%d %s", recNb, 
                                suffix);
                        amdlibStripQuotes(selKeywList->keywords[i].value);
                        if (fits_write_key(filePtr, TSTRING, keyName, 
                                           selKeywList->keywords[i].value,
                                           selKeywList->keywords[i].comment,
                                           &status) != 0)
                        {
                            fits_close_file(filePtr, &status);
                            sprintf(errMsg, "Could not complete main header");
                            return amdlibFAILURE;
                        }
                    }
                }
            }
            recNb++;
        }
    }

    /* Add information on the reciepe itself */
    sprintf(keyVal, "HIERARCH ESO PRO REC%d ID", recNb);
    if (fits_write_key(filePtr, TSTRING, keyVal, 
                       "amdlibSelectFrames",
                       "Program identifier", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }
    
    /* Add information relative to parameters */
    if (saveInOiFits == amdlibFALSE)
    {
        sprintf(keyName, "HIERARCH ESO PRO REC%d RAW1 NAME", recNb);
        strcpy(keyVal, inOiFits);
        keyValBis = basename(keyVal);
        sprintf(keyVal, "%.40s", keyValBis);
        if (fits_write_key(filePtr, TSTRING, keyName, keyVal,
                           "File name of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendSelectionKeywords ");
        }
        sprintf(keyName, "HIERARCH ESO PRO REC%d RAW1 CATG", recNb);
        if (fits_write_key(filePtr, TSTRING, keyName, "SCIENCE_REDUCED",
                           "Category of raw frame", &status) != 0)
        {
            amdlibPROReturnError("amdlibAppendSelectionKeywords ");
        }
    }
    sprintf(keyName, "HIERARCH ESO PRO REC%d PARAM1 NAME", recNb);
    if (fits_write_key(filePtr, TSTRING, keyName, "-c",
                       "Parameter name", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }
    sprintf(keyName, "HIERARCH ESO PRO REC%d PARAM1 VALUE", recNb);
    if ((selectionType == amdlibFLUX_SEL_PCG) || 
        (selectionType == amdlibFLUX_SEL_THR))
    {
        strcpy(keyVal, "FLUX");
    }
    else if ((selectionType == amdlibFRG_CONTRAST_SEL_PCG) || 
             (selectionType == amdlibFRG_CONTRAST_SEL_THR))
    {
        strcpy(keyVal, "FRG");
    }
    if (fits_write_key(filePtr, TSTRING, keyName, keyVal, 
                       "Parameter value", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }
 
    sprintf(keyName, "HIERARCH ESO PRO REC%d PARAM2 NAME", recNb);    
    if (fits_write_key(filePtr, TSTRING, keyName, "-r",
                       "Parameter name", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }
    sprintf(keyName, "HIERARCH ESO PRO REC%d PARAM2 VALUE", recNb);
    ratio = selectionRatio;
    if (fits_write_key(filePtr, TFLOAT, keyName, &ratio,
                       "Parameter value", &status) != 0)
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }

    /* Close FITS file */
    if (fits_close_file(filePtr, &status)) 
    {
        amdlibPROReturnError("amdlibAppendSelectionKeywords ");
    }

    return amdlibSUCCESS;
}


/*___oOo___*/
