;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************************
;#class#
; FitsImage
;#inheritance#
; FitsFile
;#description#
; This class consists of simple FITS images.
; Access is either by the whole image or a subset of 2D planes.
;#end_class#
;********************************************************************
PRO FitsImage__DEFINE
;********************************************************************
;#structure#
;FitsImage
;#inheritance#
;fitsFile
;#description#
;Internal Variables for FitsImage Object
;#structure_text#
a3 = {FitsImage,    $  ;#class members#
INHERITS FITSFILE,  $  ;#class inheritance#
bitpix:0,           $  ;int  & FITS bits/pixel indicator
bytesPlane:0L,      $  ;long & bytes per image plane
naxis:PTR_NEW(),    $  ;ptr->long array & image dimensions
bzero:0.0d,         $
bscale:1.0d,        $
diskIdlType:0B,     $  ;byte & IDL internal equivalent of bitpix
dataStart:0L,       $  ;long & position of 1st data point (bytes)
dataEnd:0L,         $  ;long & position after last data point
dataCurrent:0L,     $  ;long & current reading position
planeCurrent:0L,    $  ;long & current plane being read/written
planeLast:0L,       $  ;long & number of last plane known
dataBuff:PTR_NEW(), $  ;ptr  & read/write buffer
buffPlanes:0L,      $  ;long & number of planes that fit in buffer
headerStatus:'',    $  ;string & indicates whether header is in \cr
                       ;local storage, on disk...
dataStatus:''       $  ;string & where is data
}
;#end_structure#
;********************************************************************
END

PRO fitsImage::cleanup
;********************************************************************
;#procedure#
; cleanup
;#call#
; OBJ_DESTROY,fitsImage
;#description#
; Class destructor.
;#end_procedure#
;********************************************************************
   PTR_FREE, self.naxis
   PTR_FREE, self.dataBuff
   self->FITSFILE::cleanup
RETURN
END

FUNCTION fitsImage::init, filename, status, naxis=naxis, iErr=iErr
;********************************************************************
;#function#
; init
;#call#
; fitsImageObj=OBJ_NEW(filename, status, naxis=naxis, iErr=iErr)
;#description#
; Constructor for fitsImage class.
;#inputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ filename }{ string }{ filename of disk file}
;\anArg{ status }  { string }{ 'READ' or 'WRITE' }
;\anArg{ naxis }  { int(?) }{ Dimensionality array describing image }
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{optional: Return code, 0 => OK, else failed}
;#return#
; \anArg{--}{FitsImageObj}{the object created}
;#end_function#
;********************************************************************
;Constructor for fitsImage class
;
;INPUTS: see fitsImage::open
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
   errMsg = ''
;default status
   if (N_PARAMS() EQ 1) then status = 'READ'
   st     = strUpCase(StrTrim(status, 2))
;on WRITE, naxis must be given
   if (st EQ 'WRITE') and (NOT KEYWORD_SET(naxis)) then $
      midiSetError, 'FITSIMAGE:NAXIS must be specified at opening new file'
;open file as FITSFILE: for READ this stores header internally
;                       for WRITE this makes a default minimal header
   if (0 EQ self->FITSFILE::init(filename, st, iErr=iErr)) then $
      midiSetError,'',/notInitial
   if (st  EQ 'WRITE') then begin
;copy naxis into local storage but also update in primary header
;extend to 3-d if less than 3-d
      nA = naxis
      nNaxis = n_elements(naxis)
      if (nNaxis eq 1) then nA = [nA, 1, 1] else if (nNaxis eq 2) then nA=[nA,1]
      nNaxis = nNaxis > 3
      self.naxis = PTR_NEW(nA)
      self.priHead->addPar, 'NAXIS', nNaxis, iErr=iErr
      if (iErr NE 0) then midiSetError,'',/notInitial
      for iAxis = 1, nNaxis do begin
         self.priHead->addPar,'NAXIS'+strTrim(string(iAxis),2), $
	    nA(iAxis-1), iErr=iErr
         if (iErr NE 0) then midiSetError,'',/notInitial
      endfor
      self.headerStatus = 'INTERNAL'
   endif                     ; status is WRITE
   if (st EQ 'READ') then begin
;copy naxis into local storage from primary header
      nNaxis = self.priHead->getPar( 'NAXIS', Match )
      if (Match NE 1) then midiSetError, 'NAXIS not found in header'
      self.naxis = PTR_NEW(lonarr(nNaxis))
      for iAxis = 1, nNaxis do begin
         (*self.naxis)[iAxis-1] =self.priHead->getPar('NAXIS' +$
	    strTrim(string(iAxis),2), Match)
         if (Match NE 1) then $
            midiSetError,'NAXIS'+strTrim(string(iAxis),2)+' not found in header'
      endfor
;set data pointers
      point_lun, -self.unit, pointLun
      self.dataStart   = pointLun
      self.dataCurrent = pointLun
      self.planeCurrent = 0
      if (nNaxis le 2) then self.planeLast = 1 else $
         self.planeLast = (*self.naxis)[2]
;figure out size
      byteSize = self.priHead->getPar('BITPIX', Match)
      if (Match NE 1) then midiSetError, 'BITPIX not found in header'
      self.bitpix = byteSize
      byteSize = long(abs(byteSize)/8)
      for iAxis = 0, nNaxis-1 do byteSize = byteSize * (*self.naxis)[iAxis]
      self.dataEnd = pointLun + byteSize
      self.headerStatus = 'DISK'
      self.dataStatus = 'DISK'
   endif                     ; status is READ
RETURN,1
END

PRO fitsImage::headerToDisk, iErr=iErr
;********************************************************************
;#procedure#
; headerToDisk
;#call#
; fitsImageObj->headerToDisk, iErr=iErr
;#description#
; Write the stored header onto disk.
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;write the stored header onto disk
   self.priHead->writeToDisk, self.Unit, iErr=iErr
   if (iErr eq 0) then self.headerStatus = 'DISK'
RETURN
END

FUNCTION fitsImage::readImage, raw=raw, iErr=iErr
;********************************************************************
;#function#
; readImage
;#call#
; readImage (raw=raw, iErr=iErr)
;#description#
; Read an entire image as a block 
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ raw }{ int    }{if set (/raw) return data in raw disk format  }
; otherwise (default) apply bscale and bzero and return as float.
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#return#
; \anArg{--}{float?}{requested image}
;#end_function#
;********************************************************************
;read an entire image as a block and return to user
;INPUTS
;   raw   boolean    if specified and !=0 , return data
;                      in same format as on disk
;                      if not given use bscale and bzero
;                      and return float version
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;check status
   if ((self.state le 0) OR (self.unit le 0) OR (self.openStatus ne 'READ') $
       OR (NOT OBJ_VALID(self. priHead))) then $
      midiSetError, 'input file not correctly opened'
;should we look up and apply bScale and bZero
   reScale = 1
   if (KEYWORD_SET(raw)) then if (raw NE 0) then rescale = 0
   if (rescale EQ 1) then begin
      self.bZero = self.priHead->getPar('BZERO', matches)
      if (matches EQ 0) then self.bZero = 0.
      self.bScale = self.priHead->getPar('BSCALE', matches)
      if (matches EQ 0) then self.bScale = 1.
   endif
;get data type and size
   byteSize = abs(self.bitpix)/8
   nNaxis = n_elements(*self.naxis)
   for iaxis = 0, nNaxis -1 do $
      byteSize = byteSize * (*self.naxis)[iaxis]
   if (byteSize le 0) then midiSetError, 'input image has zero elements'
   CASE self.bitpix OF
       8: begin     ; byte
         idlType = 1
      END
       16: begin     ; int
         idlType = 2
      END
       32: begin     ; long
         idlType = 3
      END
      -32: begin     ; float
         idlType = 4
      END
      -64: begin     ; double
         idlType = 5
      END
   ENDCASE
   if ((idlType le 0) or (idlType gt 6)) then $
      midiSetError, 'BITPIX unrecognized or unimplemented'
;encode naxis as idl size array
   idlSize = [nNaxis, *self.naxis, idlType, byteSize/8]
   image = MAKE_ARRAY(size=idlSize)
;read from disk
   point_lun, self.unit, self.dataStart
   readu, self.unit, image
   self.dataCurrent = self.dataStart + byteSize
   self.dataEnd     = self.dataCurrent
;convert to local 
  IEEE_TO_Host, Image
   iErr = 0
   if (reScale EQ 0) then RETURN, image $
      else RETURN, float(self.bZero) + float(self.bScale)*float(image)
END

FUNCTION fitsImage::readPlanes, plane, nPlane=nPlane, scale=scale, iErr=iErr
;********************************************************************
;#function#
; readPlane
;#call#
; imagePlanes = ImageObj->readPlanes (plane, nPlane=nPlane, 
;    scale=scale,iErr=iErr)
;#description#
; Read one or more  planes of image into data cube.
;#inputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ plane } { int }{first plane in cube to be returned }
;\anArg{ nPlane }{ int }{number of planes to read.  If not specified = 1 }
;\anArg{ scale }{ int }{perform scaling? if 0 or not specified}
; return data in raw disk format; if specified (/scale) apply
; bscale and bzero and return as float
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{optional: Return code, 0 => OK, else failed}
;#return#
;\anArg{--}{?}{2 or 3-D array containing requested planes}
;#end_function#
;********************************************************************
;read one or more  planes of image into data cube
;INPUTS
;   plane    int   first plane in cube to be returned
;   nPlane   int   number of planes to read.  If not specified = 1
;   scale   boolean    if 0 or not specified, return data
;                      in same format as on disk
;                      if given and != 0, use bscale and bzero
;                      and return float version
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;check status
   if ((self.state le 0) OR (self.unit le 0) OR (self.openStatus ne 'READ') $
       OR (NOT OBJ_VALID(self. priHead))) then $
      midiSetError, 'input file not correctly opened'
;should we look up and apply bScale and bZero
   reScale = 0
   if (KEYWORD_SET(scale)) then if (scale NE 0) then begin
      reScale = 1
      self.bZero = self.priHead->getPar('BZERO', matches)
      if (matches EQ 0) then self.bZero = 0.
      self.bScale = self.priHead->getPar('BSCALE', matches)
      if (matches EQ 0) then self.bScale = 1.0
   endif
;check plane number and self defaults; Plane le 0 -> next plane
   if (NOT KEYWORD_SET(nPlane)) then nPlane = 1
;nPlane must be >= 1
   nP = nPlane > 1
   isNew = (self.PlaneCurrent eq 0)
   if (N_PARAMS() LT 1) then iPlane = self.planeCurrent + 1 $
   else if (plane le 0) then iPlane = self.planeCurrent + 1 $
   else iPlane = plane
;is this first call
   if (isNew) then begin
      nNaxis = N_ELEMENTS(*self.naxis)
;figure out how many planes in whole image
      if (nNaxis lt 2) then midiSetError, 'image has less than 2 dimensions'$
      else self.PlaneLast = 1 
      for i=2, nNaxis-1 do self.PlaneLast = self.PlaneLast * (*self.naxis)[i]
;get data type and size; bytes per plane
      byteSize = abs(self.bitpix)/8
      self.bytesPlane = byteSize * (*self.naxis)[0] * (*self.naxis)[1]
      if (self.bytesPlane le 0) then midiSetError, $
         'input planes have zero elements'
      CASE self.bitpix OF
         8: begin     ; byte
           idlType = 1
         END
         16: begin     ; int
           idlType = 2
         END
         32: begin     ; long
           idlType = 3
         END
         -32: begin     ; float
            idlType = 4
         END
         -64: begin     ; double
            idlType = 5
         END
      ENDCASE
      self.diskIdlType = idlType
      if ((idlType le 0) or (idlType gt 6)) then $
         midiSetError, 'BITPIX unrecognized or unimplemented'
   endif      ; end of IsNew
;is requested plane legal
   if (iPlane+nP-1 gt self.planeLast) then begin
      errMsg = 'requested planes '+strtrim(string(iPlane),2)+$
        ':'+strtrim(string(iPlane+nP-1),2)+ $
	' greater than image plane size = '+strtrim(string(self.planeLast))
      midiSetError, errMsg
   endif
;encode image size as idl size array
   if (nP EQ 1) then $
      idlSize = [2, (*self.naxis)[0:1], self.diskIdlType, self.bytesPlane/8]$
   else idlSize=[3, (*self.naxis)[0:1], nP, self.diskIdlType, $
      self.bytesPlane/8]
;allocate buffer if necessary
   if (nP NE self.buffPlanes) then begin
      PTR_FREE, self.dataBuff
      self.dataBuff = PTR_NEW( MAKE_ARRAY(size=idlSize))
      self.buffPlanes = nP
   endif
;point to correct disk location
   pointLun = self.dataStart + (iPlane - 1) * self.bytesPlane
   point_lun, self.unit, pointLun
;read from disk
   READU, self.unit, *self.dataBuff
;update pointers
   self.planeCurrent = iPlane + nP -1
   self.dataCurrent  = pointLun + self.bytesPlane * nP
;convert to local 
  IEEE_TO_Host, *self.dataBuff
   iErr = 0
   if (reScale EQ 0) then RETURN, *self.dataBuff $
   else RETURN, float(self.bZero) + float(self.bScale)*(*self.dataBuff)
END

PRO fitsImage::writeImage, image, iErr=iErr
;********************************************************************
;#procedure#
; writeImage
;#call#
; imageObj->writeImage, image, iErr=iErr
;#description#
; Write an entire image as a block after the header.
;#inputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ image }{ numeric array }{ Data to be written to file}
;#outputs# 
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;write an entire image as a block after the header
;   INPUTS  image
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;check status
   if (self.openStatus NE 'WRITE') then $
      midiSetError,  'Output file not opened with WRITE status'
;determine size and type of input array
   sImage = SIZE(image)
   idlType = sImage(1+sImage(0))
   self.DiskIdlType = idlType
   CASE idlType OF
      1: begin     ; byte
         bitpix = 8
      END
      2: begin     ; short int
         bitpix = 16
      END
      3: begin     ; long int
         bitpix = 32
      END
      4: begin     ; float
         bitpix = -32
      END
      5: begin     ; double
         bitpix = -64
      END
   ENDCASE
;write FITS dimension keywords into local store
   byteSize = abs(bitpix)/8
   self.bitpix = bitpix
   self.priHead->addPar,'BITPIX', bitpix, iErr=iErr
   PTR_FREE, self.naxis
   self.naxis = PTR_NEW(sImage(1:sImage(0)))
   for iAxis = 0, sImage(0)-1 do byteSize = byteSize * (*self.naxis)[iAxis]
   self.priHead->addPar,'NAXIS', sImage(0), iErr=iErr
   for iAxis = 1, sImage(0) do self.priHead->addPar, $
      'NAXIS'+strTrim(string(iAxis),2), SImage(iAxis), iErr=iErr
;dump header to disk.  If you have already done this, overwrite it
   pointLun = 0L
   if (self.headerStatus NE '') then point_lun, self.unit, pointLun
   self.PriHead->WriteToDisk, self.unit, iErr=iErr
   if (iErr NE 0) then midiSetError, 'writeImage: writing header failed', $
      /notInitial
   point_lun, -self.unit, pointLun
   self.headerStatus = 'DISK'
   self.dataStart = pointLun
;write data to disk, with conversion to IEEE.  This may blow core if
;image is very big
   IF (idlType EQ 1) then WRITEU, self.unit, image else begin 
      newImage = image
      HOST_TO_IEEE, newImage
      WRITEU, self.unit, newImage
   endelse
   self.dataEnd   = self.dataStart + byteSize
   self.dataCurrent = self.dataEnd
   self.eof = self.dataEnd
   self->pad2880, rest, iErr=iErr
   if (iErr NE 0) then midiSetError, 'writeImage: padding data failed', $
      /notInitial
   self.dataStatus = 'DISK'
   iErr = 0
RETURN
END

PRO fitsImage::writePlanes, data, plane=plane, iErr=iErr
;********************************************************************
;#procedure#
; writePlanes
;#call#
; imageObj->writePlanes, data, plane=plane, iErr=iErr
;#description#
; Write one or more 2-D planes to disk.
;#inputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ data }{ numeric array }{ Data to be written to file}
;\anArg{ plane }{ int }
;   { which plane 1-relative. If not given, append to end of image.}
;#outputs# 
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;write one or more 2-D planes to disk
;   INPUTS  data   the data
;           plane  which plane 1-relative. 
;                   If not given, append to end of image 
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;check status
   if (self.openStatus NE 'WRITE') then $
      midiSetError, 'Output file not opened with WRITE status'
   sPlane = SIZE(data)
   if ((sPlane[0] NE 2) AND (sPlane[0] NE 3))then $
      midiSetError, 'writePlanes only works on 2 or 3-D planes'
   isNew = self.planeLast eq 0
   idlType = sPlane(1+sPlane[0])
   if (isNew) then begin
      CASE idlType OF
         1: begin     ; byte
            bitpix = 8
         END
         2: begin     ; short int
            bitpix = 16
         END
         3: begin     ; long int
            bitpix = 32
         END
         4: begin     ; float
            bitpix = -32
         END
         5: begin     ; double
            bitpix = -64
         END
      ENDCASE
      self.DiskIdlType = idlType
      self.bitpix = bitpix
      self.priHead->addPar,'BITPIX', bitpix, iErr=iErr
      byteSize = abs(bitpix)/8
      PTR_FREE,self.naxis
      self.naxis = PTR_NEW(lonarr(3))
      (*self.naxis) = [sPlane(1), sPlane(2), 0]
      self.priHead->addPar, 'NAXIS',3, iErr=iErr
      self.priHead->addPar, 'NAXIS1',sPlane(1), iErr=iErr
      self.priHead->addPar, 'NAXIS2',sPlane(2), iErr=iErr
      self.priHead->addPar, 'NAXIS3',1, iErr=iErr
      self.bytesPlane = byteSize * sPlane(1) * sPlane(2)
;reset file position to head
      pointLun = 0L
      point_lun, self.unit, pointLun
;(re)write header
      bottom = 0
      self.priHead->writeToDisk, self.unit, iErr=iErr
      self.headerStatus = 'DISK'
      point_lun, -self.unit, pointLun
      self.dataStart = pointLun
      self.dataCurrent = self.dataStart
      self.planeCurrent = 1
   endif else begin               ; isNew data
      if (self.DiskIdlType NE idlType) then begin
         errMsg = 'Data type of last plane changed from '+ $
	    string(self.DiskIdlType) 
         midiSetError, errMsg
      endif
      if ((*self.naxis)[0] NE sPlane(1)) OR $
         ((*self.naxis)[1] NE sPlane(2)) then $
         midiSetError, 'Dimensions of Plane do not match earlier ones '
   endelse
;check whether specified Planes are in range.  If not specified set to
;end of image
   if (KEYWORD_SET(Plane)) then if (Plane LT 1 OR $
      (Plane GT self.planeLast + 1)) then begin
         errMsg = 'Plane of '+strTrim(string(Plane),2)+$
	    ' not in valid range (1,'+strtrim(string(self.planeLast+1))+')'
         midiSetError, errMsg
   endif 
   if (NOT KEYWORD_SET(Plane)) then Plane = self.planeLast + 1
;set lun pointer 
   if (Plane NE self.planeCurrent) then $
      point_lun, self.unit, self.dataStart + (Plane - 1)*self.bytesPlane
;write data
   if (idlType EQ 1) then WRITEU, self.unit, data else begin
      newPlanes = data
      HOST_TO_IEEE, newPlanes
      WRITEU, self.unit, newPlanes
   endelse
;update pointers
   if (sPlane[0] EQ 2) then nPlane = 1 else nPlane = sPlane[3]
   self.planeLast   = (self.planeCurrent+nPlane-1) > self.planeLast
   self.dataEnd    = self.dataStart + nPlane*self.planeLast*self.bytesPlane
   self.eof    = self.dataStart + self.planeLast * self.bytesPlane
;position of next read/write
   self.dataCurrent = (Plane+nPlane-1)*self.bytesPlane  
   self.planeCurrent = Plane +  nPlane ; position of next read/write
;note planes are counted 1-relative and databytes are 0-relative
   iErr = 0
RETURN
END

PRO fitsImage::writeFinish, iErr=iErr
;********************************************************************
;#procedure#
; writeFinish
;#call#
; imageObj->writeFinish, iErr=iErr
;#description#
; Correctly terminate image on disk
; If we have been writing planes, reset NAXIS3 to the right number
; and pad data out to 2880 byte blocks.
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional: Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;if we have been writing planes, reset NAXIS3 to the right number
;and pad data out to 2880 byte blocks
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level 
      catch, /cancel 
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
   if(self.openStatus NE 'WRITE') then begin
      iErr = 0
      RETURN
   endif
;overwrite NAXIS3 if self.planeLast set (i.e. written by writePlanes and not writeImage)
;first in stored header
   if (self.planeLast GT 0) then begin
      self.priHead->addPar,'NAXIS3', self.planeLast, iErr=iErr
;now on disk
      pointer = long(5*80)
      n3 = strTrim(string(self.planeLast), 2)
      l3 = strLen(n3)
      newLine = 'NAXIS3  ='+STRING(REPLICATE(32B,21-L3))+N3+' /Number of rows'+$
         STRING(REPLICATE(32B,34))
      POINT_LUN, self.unit, pointer
      WRITEU, self.unit, newLine
   endif
   self->pad2880, iErr=iErr
   if (iErr eq 1) then midiSetError,'',/notInitial
   self.dataStatus = 'DISK'
   iErr = 0
RETURN
END

FUNCTION fitsImage::naxis
;********************************************************************
;#function#
; naxis
;#call#
; nAxis = imageObj->naxis()
;#description#
; Return the stored value of naxis.
;#return#
; \anArg{--}{int(*)}{= axis dimensionality array}
;#end_function#
;********************************************************************
;return the currently stored value of naxis
RETURN, *self.naxis
END

PRO fitsImage::setNaxis, naxis
;********************************************************************
;#procedure#
; setNaxis
;#call#
; imageObj->setNaxis, naxis
;#description#
; Update stored values of naxis.
;#inputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ naxis }{ int(*) }{axis dimensionality array }
;#end_procedure#
;********************************************************************
   PTR_FREE,self.naxis
   self.naxis = PTR_NEW(naxis)
RETURN
END

PRO fitsImage::addPar, name, value, iErr=iErr, comment=comment, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;********************************************************************
;#procedure#
; addPar
;#call#
; imageObj->addPar, name, value, iErr=iErr, comment=comment, $
;  BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;#description#
; Add/modify keywords in the header.
; This version works only before the header has been committed to disk.
;#inputs# 
; <name> }{ <type/dim> }{ <description>}
;\anArg{ name }{ string }
;   { String name of the parameter to set.
;     If NAME is already in the header the value and possibly comment 
;     fields are modified.  Otherwise, a new record is added to the header.  
;     If NAME is equal to either "COMMENT" or "HISTORY" then the value 
;     will be added to the record  without replacement. 
;     In this case the comment parameter is ignored}
;\anArg{ value}{ ?      }
;   { Value for parameter.  The value expression must be of the correct 
;     type, e.g. integer, floating or string. 
;     String values of 'T'	or 'F' are considered logical values.
;     logical values.}
;\anArg{ comment }{     }
;   { String field.  The '/' is added by this routine. 
;     Added starting in position 31. If not supplied, or set equal 
;     to ''  (the null string), then any previous comment field in 
;     the header for that keyword is retained (when found).}
;\anArg{ before }{ string }
;   { Keyword string name.  The parameter will be placed
;     before the location of this keyword.  For example, 
;     if BEFORE='HISTORY' then the parameter will be placed 
;     the first history  location.  This applies only when 
;     adding a new keyword;  keywords already in the header
;     are kept in the same position.}
;\anArg{ after }{ string }
;   { Same as BEFORE, but the parameter will be placed after the 
;     location of this keyword.  This keyword takes precedence over 
;     BEFORE.}
;\anArg{ format }{ string }
;   { Specifies FORTRAN-like format for parameter, e.g. 
;     "F7.3".  A scalar string should be used. 
;      For complex numbers the format should be defined so 
;      that it can be applied separately to the real and 
;      imaginary parts.}
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{optional: Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;Pass through interface to add/modify keyword records to
;the primary header
;
;this version works only before the header has been committed to disk
;
   if (self.HeaderStatus ne 'INTERNAL') then midiSetError,$
     'FITSEXTENSION: cant add new keywords if status ne LOCAL'
   self.priHead->addPar,name, value, iErr=iErr, comment=COMMENT,$
      BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
RETURN
END

FUNCTION fitsImage::getPar, name, matches, comment=comments
;********************************************************************
;#function#
; getPar
;#call#
; parValues = imageObj->getPar (name, matches, comment=comments)
;#description#
; Get keyword values from the header.
; 
; If the parameter is complex, double precision, floating point, long or
; string, then the result is of that type.  Apostrophes are stripped from
; strings.  If the parameter is logical, 1 is returned for T, and 0 is
; returned for F.
; If NAME was of form 'keyword*' then a vector of values are returned.
;
; The system variable !err is set to -1 if parameter not found, 0 for a
; scalar value returned.  If a vector is returned it is set to the number
; of keyword matches found.
; 
; If a keyword occurs more than once in a header, a warning is given,
; and the first occurence is used.  However, if the keyword is "HISTORY",
; "COMMENT", or "        " (blank), then multiple values are returned.
;#inputs# 
;\anArg{ name }{ string }
;   { String name of the parameter to return.
;     If NAME is of the form 'keyword*' then an array 
;     is returned containing values of keywordN where 
;     N is an integer. The value of keywordN will be 
;     placed in RESULT(N-1).  The data type of RESULT will
;     be the type of the first valid match of keywordN found.}
;#outputs# 
;\anArg{ matches  }{int     }{ Number of matches found}
;\anArg{ comments }{ strarr }
;   { Array of comments associated with the returned values.}
;#return#
;\anArg{--}{ array}{ the value(s) associated with the requested keyword in the 
;   header array.}
;#end_function#
;********************************************************************
;  Pass through interface to retrieve keyword records from
;  the internally stored extension header
RETURN,self.prihead->getPar(name, matches, comment=comments)
END

PRO fitsImage::close, iErr=iErr
;********************************************************************
;#procedure#
; close
;#call#
; imageObj->close, iErr=iErr
;#description#
; Close FITS image file and object.
;#outputs# 
;\anArg{ <name> }{ <type/dim> }{ <description>}
;\anArg{ iErr }{ int    }{ optional:Return code, 0 => OK, else failed}
;#end_procedure#
;********************************************************************
;make sure that image is correctly terminated, then close actual file
   self->writeFinish, iErr=iErr
   self->FITSFILE::close, iErr=iErr
RETURN
END
