#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "oirSupport.h"
#include "fitsio.h"
#include "imaging_data.h"
#include "imaging_fdata.h"
#define NBIN 5
#define PRISMTOP 25
#define GRISMTOP 29
#define PRISMBOTTOM 9
#define GRISMBOTTOM 6
#define CURVEDY     8
#define maxshift    4
#define DSKYUT     10
#define DSKYAT      6
/*#define DEBUG 1 */

/**********************************************************
 * Usage:
 *    oirChopPhotoImages -in "inFiles" -out outFile -nDrop nDrop -dSky dSky -ref refCurveFile -sky0
 *
 *    Output is a IMAGING_DATA table in a fits file with
 *    NBINS+1 rows, each with 2 or 4 DATA arrays for detector windows 1 and 2
 *    in HIGH_SENS mode and 1-4 in SCI_PHOT mode
 *
 *    The first 1 row is  the grand average of all data.  The
 *    next rows are NBINS subaverages, useful in estimate RMS errors.
 *    All images are in instrumental units: ADUs/s/pixel
 *
 *    nDrop is the number of frames to drop after a shift from Sky to Target
 *    dSky represents the shift of the sky subtraction zones from their nominal position
 *       These nominal positions are specfied by
 *       the #include statements for PRISMTOP, GRISMTOP, GRISMBOTTOM and CURVEDY 
 *
 *    refCurveFile is a file containing IMAGING_DETECTOR data describing the curveature
 *    of the spectra on the detector; if given used to guess best areas for sky subtract
 *    -sky0 is a special flag for SCI_PHOT AOPEN/BOPEN images.  In this case
 *       an attempt is made to debias the sky images to an approximate of true zero
 ************************************************************* */

/***************************************************************
 * read through all raw files associated with one shutter setting
 * separate target and sky frames and accumulate them in
 * given arrays; divide by number of frames
 * INPUTS:
 *    char **string     array of file names
 *    int    nFiles     number of file names
 *    int    chan1      0-based number of 1st interferometric channel
 *    int    chan2      0-based number of 2nd interferometric channel
 *    int    nDrop      drop this many points after change of chop phase
 *    int    npix1      no. pixels in array in 1st channel
 *    int    npix2      no. pixels in array in 2nd channel
 *    float* target1    array to receive target1 data
 *    float* target2    array to receive target2 data
 *    float* sky1       array to receive target1 data
 *    float* sky2       array to receive target2 data
 *    int  * nTarget    number of target frames
 *    int  * nSky       number of sky frames
 **************************************************************** */
float minrtsQPk(int np, float *x, float *y, float *f, int d);
void unChopData(char **fileList, int nFiles, int nChan, 
   int nDrop, long npix1, long npix2, float **target, 
   float **sky, int *nTarget, int *nSky, int *nCycle, int *ierr) {

   fitsfile *inFile;
   imaging_data_table_def *idata;
   imaging_data_row *inData;
   int iFile;

   long iRow, nRow, tRow;
   int i, iChan;
   int TN, SN;
   long iPix;

   char Sky = 'S';
   char Target = 'T';
   char targ, oldTarg;
   int sinceChange;
   float mjd;
   int old;

   *ierr    = 0;
   *nCycle = 0;
   TN = 0;
   SN = 0;
   oldTarg = ' ';
   tRow    = 0;

/*
      loop over files, read rows, read target types, add data
      into appropriate accumulators 
*/

   for (iFile =0; iFile <nFiles; iFile++) {
/*
      open file, open table, create table row
*/
      i = fits_open_file(&inFile, fileList[iFile], READONLY, ierr);
      if (*ierr !=0) {
         printf("ERROR %i opening file %s\n", *ierr, fileList[iFile]);
         printf("%s \n",strerror(errno));
         return;
      }
/*
      get observing data to check chopping flag coding
*/
      i = fits_read_key(inFile, TFLOAT, "MJD-OBS", &mjd,0,ierr);
      old = mjd < 52960.;
      if (*ierr !=0) {
         printf("ERROR %i reading mjd file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      idata = open_imaging_data_table(inFile, ierr);
      if (*ierr !=0) {
         printf("ERROR %i opening imaging data in file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      inData = make_imaging_data_row(idata);
      i = fits_get_num_rows (inFile,  &nRow, ierr);
      /* suppress autoscaling to avoid overflow */
      for (i=0; i<idata->nregion; i++)
         fits_set_tscale(inFile, 1+idata->data_col[i], 1.0, 0.0, ierr);

/*
         read lines, check target type and decide whether to accumulate
*/
      for (iRow =1; iRow<=nRow; iRow++) {
         i = read_imaging_data_table(inFile, idata, inData, iRow, ierr);
         if (*ierr !=0) {
            printf("ERROR %i reading imaging data line %i in file %s\n", 
               *ierr, iRow, fileList[iFile]);
            return;
         }
         tRow++;
         if (tRow == (10 * (int)(tRow/10))) fprintf(stderr,"frame: %d\r",tRow);
         if (old) targ = inData->targtype[0]; 
         else targ = inData->targtype[1]; 
         if (targ != oldTarg ) {
            sinceChange = 0;
            oldTarg     = targ;
            if ((targ == Target) || (targ == Sky)) (*nCycle)++;
         } else {
            ++sinceChange;
         }
/*
            if sinceChange > nDrop decide what type and accumulate data
*/
         if (sinceChange > nDrop && (targ == Target)) {
            for (iChan=0; iChan<nChan; iChan++) 
               for (iPix=0;iPix<npix1;iPix++) target[iChan][iPix] =
               target[iChan][iPix] + 32768. + (short)inData->data[iChan][iPix];
            ++TN;
         } else if (sinceChange > nDrop && (targ == Sky)) {
            for (iChan=0; iChan<nChan; iChan++) 
               for (iPix=0;iPix<npix1;iPix++) sky[iChan][iPix] =
                  sky[iChan][iPix] + 32768. + (short)inData->data[iChan][iPix];
            ++SN;
         }
      } /* read rows in one file */
      
/*
         close file and table, destroy row
*/
      i= close_imaging_data_table(inFile, idata, ierr);
      kill_imaging_data_row(inData);
   } /* read all files in list */
   fprintf(stderr,"\n");
   *nTarget = TN;
   *nSky    = SN;
   (*nCycle) = (*nCycle)/2;
   return;
} /* end of unChopData */

void binChopData(char **fileList, int nFiles, int nChan, int nDrop, 
   long npix1, long npix2, int nCycle, float ***a1, int *ierr) {

   fitsfile *inFile;
   imaging_data_table_def *idata;
   imaging_data_row *inData;
   int iFile;

   long iRow, nRow;
   int i, iChan;
   int TN, SN;
   long iPix;

   char Sky = 'S';
   char Target = 'T';
   char targ, oldTarg;
   int sinceChange;
   float mjd;
   int old;

   float *aTarg[4], *aSky[4] ; /* chop accumulators */
   int iBin, nBin, maxBin;
   int iCycle = 0;
   *ierr = 0;
   TN = 0;
   SN = 0;
   oldTarg = ' ';



/*
      how many cycles per bin?
*/
   nBin = nCycle/NBIN;
   maxBin = NBIN;
   if (nBin < 1) {
      printf("   %i cycles is not enough to create %i bins",
         nCycle, NBIN);
      nBin = 1;
      maxBin = nCycle;
   }
/*
      allocate single cycle accumlators
*/
   for (iChan=0;iChan<nChan;iChan++) {
      aTarg[iChan] = (float *)malloc(npix1 * sizeof(float));
      aSky[iChan]  = (float *)malloc(npix1 * sizeof(float));
   }
/*
      loop over files, read rows, read target types, add data
      into appropriate accumulators 
*/

   for (iFile =0; iFile <nFiles; iFile++) {
/*
      open file, open table, create table row
*/
      i = fits_open_file(&inFile, fileList[iFile], READONLY, ierr);
      if (*ierr !=0) {
         printf("ERROR %i opening file %s\n", *ierr, fileList[iFile]);
         printf("%s \n",strerror(errno));
         return;
      }
      i = fits_read_key(inFile, TFLOAT, "MJD-OBS", &mjd,0,ierr);
      old = mjd < 52960.;
      if (*ierr !=0) {
         printf("ERROR %i reading mjd file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      idata = open_imaging_data_table(inFile, ierr);
      if (*ierr !=0) {
         printf("ERROR %i opening imaging data in file %s\n", 
            *ierr, fileList[iFile]);
         return;
      }
      inData = make_imaging_data_row(idata);
      i = fits_get_num_rows (inFile,  &nRow, ierr);
      /* suppress autoscaling to avoid overflow */
      for (i=0; i<idata->nregion; i++)
         fits_set_tscale(inFile, 1+idata->data_col[i], 1.0, 0.0, ierr);

/*
         read lines, check target type and decide whether to accumulate
*/
      for (iRow =1; iRow<=nRow; iRow++) {
         i = read_imaging_data_table(inFile, idata, inData, iRow, ierr);
         if (*ierr !=0) {
            printf("ERROR %i reading imaging data line %i in file %s\n", 
               *ierr, iRow, fileList[iFile]);
            return;
         }
         if (old) targ = inData->targtype[0]; 
         else targ = inData->targtype[1]; 
         if (targ != oldTarg ) {
            sinceChange = 0;
            oldTarg     = targ;
            if (targ == Target) {
               TN = 0; 
               for (iChan=0;iChan<nChan;iChan++) 
                  memset(aTarg[iChan], 0, npix1*sizeof(float));
            } else if (targ == Sky) {
/*
       normalize accumulated target and sky
*/
               if (TN > 0) {
                  for (iChan=0;iChan<nChan;iChan++) 
                     for (iPix=0;iPix<npix1;iPix++) aTarg[iChan][iPix] =
                        aTarg[iChan][iPix]/TN;
               }
               if(SN > 0) {
                  for (iChan=0;iChan<nChan;iChan++) 
                     for (iPix=0;iPix<npix1;iPix++) aSky[iChan][iPix] =
                        aSky[iChan][iPix]/SN;
/*
                 if previous targ exists, accumulate difference
                 into bin
*/
                 iBin = iCycle/nBin;
                 if ((TN > 0) && (iBin < NBIN)) {
                    for (iChan=0;iChan<nChan;iChan++) 
                       for (iPix=0;iPix<npix1;iPix++) a1[iBin][iChan][iPix] =
                          a1[iBin][iChan][iPix] + aTarg[iChan][iPix] - aSky[iChan][iPix];
                  }
                  ++iCycle;
               }
/*
                     clear sky accumulators
*/
               SN = 0;
               for (iChan=0;iChan<nChan;iChan++) memset(aSky[iChan], 0, npix1*sizeof(float));
            }
         } else {
            ++sinceChange;
         }
/*
            if sinceChange > nDrop decide what type and accumulate data
*/
         if (sinceChange > nDrop && (targ == Target)) {
            for (iChan=0;iChan<nChan;iChan++) 
               for (iPix=0;iPix<npix1;iPix++) aTarg[iChan][iPix] =
               aTarg[iChan][iPix] + 32768. + (short)inData->data[iChan][iPix];
            ++TN;
         } else if (sinceChange > nDrop && (targ == Sky)) {
            for (iChan=0;iChan<nChan;iChan++) 
               for (iPix=0;iPix<npix1;iPix++) aSky[iChan][iPix] =
                  aSky[iChan][iPix] + 32768. + (short)inData->data[iChan][iPix];
            ++SN;
         }
      } /* read rows in one file */
      
/*
         close file and table, destroy row
*/
      i= close_imaging_data_table(inFile, idata, ierr);
      kill_imaging_data_row(inData);
   } /* read all files in list */
/*
        normalize bins
*/
   for (iBin =0; iBin<maxBin; iBin++) 
      for (iChan=0; iChan<nChan; iChan++) 
         for (iPix=0;iPix<npix1;iPix++) a1[iBin][iChan][iPix] = 
            a1[iBin][iChan][iPix]/nBin;
    /* iBin */
   for (iChan=0;iChan<nChan;iChan++) {
      free(aTarg[iChan]);
      free(aSky[iChan]);
   }
   return;
} /* end of binChopData */

void normalizeExposure (float *data, long nPix, int nFrames, float dit) {
   float norm;
   long iPix;

   if (nFrames <= 0) return;
   norm = 1./(nFrames * dit);
   for (iPix =0; iPix<nPix; iPix++) data[iPix] = data[iPix]*norm;
   return;
}
float *getSlitTrace(int iChan, imaging_detector_table_def *DetectorDef){
   
   float *refCoordData;
   int nx, ny, ix;
   long npix;
   int y0;
   float *trace;

   nx = DetectorDef->naxis[iChan][0];
   ny = DetectorDef->naxis[iChan][1];
   npix = nx*(long)ny;
   trace = (float *)malloc(nx*sizeof(float));

   refCoordData = imaging_detector_wcs(DetectorDef, 1+iChan);
   y0 = (int)DetectorDef->crpix[iChan][1];
   for (ix=0;ix<nx;ix++) trace[ix] = y0 - refCoordData[npix + nx * y0 + ix];
   free(refCoordData);
   return trace;
} /*  end of getSlitTrace */

/************************************************************
 *     determine best shift of mask by finding maximum signal
 *     empirically
 *******************************************************************/
   float findMaskShift(float *mask, float *data, int nx, int ny) {
      float corr[2*maxshift+1];
      float shift[2*maxshift+1];
      float x, yShift;
      int i,ix, iy, iShift, is;
      long xy, sy;
      for (iShift=(-maxshift);iShift<=maxshift;iShift++) {
         is = iShift + maxshift;
         shift[is] = iShift;
         corr[is] = 0.;
         for (iy=maxshift;iy<(ny-maxshift);iy++) {
            xy = iy*nx;
            sy = (iy + iShift)*nx;
            for (ix=0;ix<nx;ix++) {
               corr[is] = corr[is] 
                  + data[ix+xy]*mask[ix+sy];
            }
         } /* yloop */
      } /* iShift loop */
      yShift = minrtsQPk(2*maxshift+1, shift, corr, &x, 0);
      return yShift ;
   }  /* end findMaskShift */
/************************************************************
 * make weight mask for fitting sky by calculating trace
 * shifting it up by yShift, and then splitting it into
 * two traces separated by 2*dSky
 *************************************************************/
void makeWeightMask(float *mask, float *trace, int nx, int ny, float yShift, 
   int dSky, float sigma) {
   int ix, iy;
   float dy;
   for (ix=0;ix<nx;ix++) for (iy=0; iy<ny; iy++)  {
      dy = (trace[ix] +yShift +dSky - iy -1)/sigma;
      mask[ix+nx*iy] = exp(-0.5*dy*dy);
      dy = dy - 2* dSky/sigma;
      mask[ix+nx*iy] = exp(-0.5*dy*dy) + mask[ix+nx*iy];
   }
   return;
}

float *getWeightArray(float *data, float *trace, 
   int nx, int ny, int dSky, int UT) {

#ifdef DEBUG
   FILE *debugFile;
#endif
   float *mask;
   float sigma, dy;
   float yShift;

   int ix, iy;

   sigma = 3.5;
   mask = (float *)malloc(nx*(long)ny*sizeof(float));

   for (ix=0;ix<nx;ix++) for (iy=0; iy<ny; iy++)  {
      dy = (trace[ix] - iy -1)/sigma;
      mask[ix+nx*iy] = exp(-0.5*dy*dy);
   }
#ifdef DEBUG
      debugFile = fopen("/tmp/maskarray.dat","a");
      for (iy=0;iy<ny;iy++) for (ix=0;ix<nx;ix++) 
         fprintf(debugFile, " %f \n",mask[ix+nx*iy]);
      fclose(debugFile);
#endif
   yShift = findMaskShift(data, mask, nx, ny);
   if ((yShift > 5) || (yShift < (-5))) yShift = 0;
   printf("   yShift is %f \n",yShift);
   if (UT) sigma = 2.5; else sigma = 1.;
   makeWeightMask(mask, trace, nx, ny, yShift, dSky, sigma);
   return mask;
} /* end of getWeightArray */

void deSky(int iChan, float *data, int nx, int ny, 
   char *gris, int dSky, int UT, char *refFileName, 
   float *weightMask, int polyOrder, int *ierr) {

   int ix, iy;
   int y1, y2, ny2;
   float dy, x1, x2, yc ;
   float topSky, bottomSky;
   float skyOffset;
   int   *topSkyY, *bottomSkyY;
   float *yLine;
   int doCurve;  /* use refFile to find trace of spectrum */
   float *trace;  /* nominal slit position */
   float *yCoord, *yData, *yWeight, *weightArray;
   float sumWeight;
   float polyCoeff[3];
   int isGrism;  

   FILE *debugFile;

   int i;
   fitsfile *refFile;
   imaging_detector_table_def *DetectorDef;

   int dnsky = 2;
   weightArray = NULL;

   doCurve = (refFileName != NULL);
   isGrism = (gris[0] == 'G');
   *ierr   = 0;
   if (polyOrder == (-1)) if (UT) polyOrder = 1; else polyOrder = 0;
   ny2 = ny/2;
/*
 *    if either weightmask or curve file specified; use these
 *    to set the weightArray, which is a set of weights to
 *    use when fitting a polynomial to the sky baseline
*/
   if (weightMask != NULL) 
      weightArray = weightMask;
   else if (doCurve) {
/*
      if taking trace of spectrum from ref file:
      open file; get detector section; get wcs data
*/
      i = fits_open_file(&refFile, refFileName, READONLY, ierr);
      if (*ierr !=0) {
         printf("ERROR %i opening file %s\n", *ierr, refFileName);
         printf("%s \n",strerror(errno));
         return;
      }
      DetectorDef = read_imaging_detector_table(refFile, ierr);
      trace       = getSlitTrace(iChan, DetectorDef); 
#ifdef DEBUG
      debugFile = fopen("/tmp/trace.dat","a");
      for (ix=0;ix<nx;ix++)
         fprintf(debugFile, " %f \n",trace[ix]);
      fclose(debugFile);
#endif
      if (UT) skyOffset = dSky+DSKYUT; else skyOffset = dSky+DSKYAT;
      weightArray = getWeightArray(data, trace, nx, ny, skyOffset, UT);
#ifdef DEBUG
      debugFile = fopen("/tmp/weightarray.dat","a");
      for (iy=0;iy<ny;iy++) for (ix=0;ix<nx;ix++) 
         fprintf(debugFile, " %f \n",weightArray[ix+nx*iy]);
      fclose(debugFile);
#endif
   }  /*    got the weight array   */
      /*    now use it to remove the background */
   if (weightArray != NULL) {
      yCoord      = (float *)malloc(ny * sizeof(float));
      yData       = (float *)malloc(ny * sizeof(float));
      yWeight     = (float *)malloc(ny * sizeof(float));
      for (iy=0;iy<ny;iy++) yCoord[iy] = (float)iy;
/*
        find trace at this x; add and subtract offset, convert to int and
        sub a few points in the y direction into sky arrays;
*/
      for (ix=0; ix<nx; ix++) {
         sumWeight = 0.;
         for (iy=0;iy<ny;iy++) {
            yData[iy]   = data[ix + nx*iy];
            yWeight[iy] = weightArray[ix + nx*iy];
            sumWeight   = sumWeight + yWeight[iy];
         }
/* fit baseline and subtract */
         if (sumWeight > 0.1) 
            minrtsPolyFit(polyOrder, ny, yCoord, yData, yWeight, polyCoeff);
         else {
            polyCoeff[0] = 0.;
            polyCoeff[1] = 0.;
            polyCoeff[2] = 0.;
         }
         for (iy=0;iy<ny;iy++) data[ix+nx*iy]= data[ix+nx*iy]- polyCoeff[0];
         if (polyOrder > 0) for (iy=0;iy<ny;iy++) data[ix+nx*iy]= data[ix+nx*iy]
            - polyCoeff[1]*yCoord[iy];
         if (polyOrder > 1) for (iy=0;iy<ny;iy++) data[ix+nx*iy]= data[ix+nx*iy]
            - polyCoeff[2]*yCoord[iy]*yCoord[iy];
      }
      if (weightMask == NULL) {
         free(weightArray);
         free(trace);
         i = fits_close_file(refFile, ierr);
         kill_imaging_detector_table_def (DetectorDef); 
      }
      free(yCoord);
      free(yData);
      free(yWeight);
   } else {
/*
 *    no complicated polynomial fits 
 *    just use fixed positions to estimate sky; no doCurve
      arrays to hold average of a few rows at top and bottom
*/
      topSkyY    = (int *)malloc(nx * sizeof(int));
      bottomSkyY = (int *)malloc(nx * sizeof(int));

      if (isGrism) y1 = GRISMBOTTOM - dSky; else y1 = PRISMBOTTOM - dSky;
      if (y1 < dnsky) y1 = dnsky;
      for (ix=0;ix<nx;ix++) bottomSkyY[ix] = y1;
      if (isGrism) y1 = GRISMTOP    + dSky; else y1 = PRISMTOP + dSky;
      if (y1 > (ny - dnsky-1)) y1 = ny - dnsky - 1; 
      for (ix=0;ix<nx;ix++) topSkyY[ix] = y1;
/*
      average a few rows
*/
      for (ix=0; ix<nx; ix++) {
         y1 = bottomSkyY[ix];
         y2 = topSkyY[ix];
         topSky    = 0.;
         bottomSky = 0.;
         for (iy =-dnsky; iy<=dnsky; iy++) {
            topSky    += data[ix + (iy+y2)*(long)nx];
            bottomSky += data[ix + (iy+y1)*(long)nx];
         }
         topSky    = topSky/(2*dnsky+1);
         bottomSky = bottomSky/(2*dnsky+1);
         dy = y2 - y1;

/*
      fit and subtraction
*/
         for (iy=0; iy<ny; iy++) {
            if (polyOrder == 0) {
               data[ix + iy*(long)nx] -= .5*(topSky + bottomSky);
            } else {   /* linear */
               x1 = ((float)iy - y1)/dy   ;
               x2 = ((float)iy - y2)/dy;
               data[ix + iy*(long)nx] +=  (-topSky*x1 + bottomSky*x2);
            }
         }
      }
/*
      free arrays
*/
      free(topSkyY);
      free(bottomSkyY);
      }  /* not doCurve */

}  /* end deSky */
/*********************************************************************************************
 *      Assume that data was taken in SCI_PHOT mode with one shutter closed.
 *      Try to determine and subtract an accurate zero level, based on fits to
 *      closed shutter channel and offslit regions of onshutter channel
 ***************************************************************************************/
   void zeroSky(float **sky, int nx, int ny, char *shutter, imaging_detector_table_def *detector) {

   float yy[4*ny], raw[4*ny];
   int ix, iy, zeroChan, iChan;
   float z;
/*     which channel was looking at closed shutter */
   if (strncmp(shutter, "AOPEN", 5)==0) zeroChan = 0; else zeroChan = 3;
/*     determine mean value per x pixel of zero chan and subtract from all similar x pixels */
   for (ix=0;ix<nx;ix++) {
      z = 0.;
      for (iy=0;iy<ny;iy++) z = z + sky[zeroChan][ix + nx*iy];
      z = z/ny;
      for (iChan=0;iChan<4;iChan++) for (iy=0;iy<ny;iy++) 
         sky[iChan][ix + nx*iy] =  sky[iChan][ix + nx*iy] - z;
   }
   return;
} /* end zeroSky */
void Usage() {
   printf("Usage: oirChopPhotoImages -in \"inFiles\" -out outFile [-nDrop nDrop] [-dSky dSky] [-ref curveRefFile] [-skymask skymaskfile] [-order order] [-sky0]\n");
   printf("   inFiles: blank separated list of input files; if more than one file enclose in \"\n");
   printf("   outFile: output file name\n");
   printf("   nDrop: number of frames to drop after a shift from Sky to Target\n");
   printf("   dSky: displacement (in pixels) of sky determination zones from their nominal positions\n");
   printf("   curveRefFile: file containing accurate spectral curvature data; ignored if not present\n");
   printf("   skymaskfile:  file containing weight image specifying where to determine sky for background subtraction\n");
   printf("      overrides dSky and curveRefFile if specified\n");
   printf("   order:  polynomial order of background fit; default = 1 for UTs, 0 for ATs. max = 2\n");
   printf("   sky0: for SCI_PHOT AOPEN/BOPEN data try to determine true zero point of sky data\n");
   return ;
}   /* end of Usage */

int main (int argc, char** argv) {
   char opt1[12];
   char gris[12];
   char shutter[12];
   char station1[12];
   char station2[12];
   int  UT;
   float dit;

   char **fileList;
   int iFile,nFiles;
   int iShutter;
   int i,ierr;
   int nx, ny;
   long npix;
   long iPix;
   int  ix, iy, iBin;
   float xx, mm;
   long  ii;
   int iKey;
   int nCycle;

   int nChan, iChan;
   int iarg;

   char *inFileName;
   char outFileName[130];
   char *refFileName;
   char *skymaskFileName;
   char history[80];

   fitsfile *inFile;
   fitsfile *outFile;
   FILE  *debugFile;

   imaging_fdata_table_def *debugTable;
   imaging_fdata_row       *debugRow;

   imaging_fdata_table_def *odata;
   imaging_detector_table_def *DetectorDef;
   imaging_fdata_row *outData;
   float *weightMask;
   minrtsFrameData *maskFrameData;
   int nxList[12], nyList[12];
   int nRegion;

   int  nShutter;

   float *Target[4], *Sky[4];
   float ***Bin;
   int    TargetN, SkyN;

   int nDrop;
   int dSky;
   int nbin;
   int skyZero;
   int polyOrder;

   char *Ashutter = "AOPEN";
   char *Bshutter = "BOPEN";
   char *ABshutter = "ABOPEN";
   char *highSens = "HIGH_SENS";
   char *bang="!"; 

   ierr    = 0;
   skyZero = 0;
   polyOrder = -1;
   outFile        = NULL;
   debugFile      = NULL;
   outFileName[0] = 0;
   inFileName     = NULL;
   refFileName    = NULL;
   skymaskFileName = NULL;
   weightMask      = NULL;
   nbin           = NBIN;
   printf("\n      ****** oirChopPhotoImages ******\n");
/*
      see if nDrop, dSky are specified externally
*/
   nDrop = 2;
   dSky  = 0;
   iarg  = 1;
   if (argc <2 ){
      Usage();
      return 1;
   }
   while(iarg < argc ) {
      if(!strcmp("-in", argv[iarg])) inFileName = argv[++iarg];
      if(!strcmp("-out", argv[iarg])) { 
         strcpy(outFileName, bang);
         strcat(outFileName, argv[++iarg]);
      }
      if(!strcmp("-nDrop", argv[iarg])) sscanf(argv[++iarg], "%i", &nDrop);
      if(!strcmp("-dSky", argv[iarg]))  sscanf(argv[++iarg], "%i", &dSky);
      if(!strcmp("-order", argv[iarg]))  sscanf(argv[++iarg], "%i", &polyOrder);
      if(!strcmp("-ref",  argv[iarg]))  refFileName = argv[++iarg];
      if(!strcmp("-skymask",  argv[iarg]))  skymaskFileName = argv[++iarg];
      if(!strcmp("-sky0", argv[iarg]))  skyZero = 1;
      ++iarg;
   }
   printf("   nDrop = %i\n", nDrop);
   printf("   dSky  = %i\n", dSky);
   if (refFileName != NULL) printf("   Curvature reference file = %s\n", refFileName);
      else printf("   Not using a curvature reference file\n");
   if (inFileName==NULL) {
      printf("\nYou must specify an input file!\n");
      Usage();
      return 1;
   }
   if (outFileName[0]==0) {
      printf("\nYou must specify an output file!\n");
      Usage();
      return 1;
   }
   if ((polyOrder < (-1)) || (polyOrder > 2)) {
      printf("\n   Polynomial fitting order of %i not allowed\n",polyOrder);
      Usage();
      return 1;
   }
   if (polyOrder == (-1)) 
      printf("   Using default polynomial order for background subtraction.\n");
   else
      printf("   Using polynomial order %i for background subtraction.\n",polyOrder);
   if (skymaskFileName==0) {
      printf("\nNo background skymask specified\n");
      maskFrameData = NULL;
   } else {
      printf("   Using skymask %s\n", skymaskFileName);
      maskFrameData = minrtsFrameDataReadFits(skymaskFileName, &ierr);
      if (ierr !=0) {
         fprintf (stderr, "ERROR opening mask input %s %i\n",skymaskFileName, ierr);
         printf("%s \n",strerror(errno));
         return 1;
      }
   }
   
/*        
       break up file list and print what you find 
       then read and unchop files
*/
   fileList = oirStrSplit(inFileName, &nFiles);
   i = fits_open_file(&inFile, fileList[0], READONLY, &ierr) ;
   if (ierr !=0) {
      fprintf (stderr, "ERROR opening fits input %s %i\n",fileList[0], ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
/*
       find out which shutter is really here
*/
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS SHUT NAME", 
         shutter, 0, &ierr);
   printf("   shutter = %s\n", shutter);
   nShutter = 0;
   if (strncmp(shutter, Ashutter, 5)==0) nShutter=1;
   if (strncmp(shutter, Bshutter, 5)==0) nShutter=2; 
/*
        is this AT or UT ?
*/ 
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO ISS CONF STATION1", 
         station1, 0, &ierr);
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO ISS CONF STATION2", 
         station2, 0, &ierr);
   printf("   stations = %s %s\n", station1, station2);
   UT = (station1[0] == 'U');
/*
      prepare output file
*/
   fits_create_file (&outFile, outFileName, &ierr);
   if (ierr !=0) {
      fprintf (stderr, "ERROR opening fits output %s %i\n",outFileName, ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
/*
              copy header
*/
   i = fits_copy_header(inFile, outFile, &ierr); 
   i = fits_read_key(inFile, TFLOAT, "HIERARCH ESO DET DIT", &dit, 
      0, &ierr);
   printf("   dit = %f\n", dit);
   ierr = 0;
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS OPT1 NAME", opt1,0, &ierr);
   printf("   beam combiner = %s\n", opt1);

   if (strncmp(opt1, highSens, 9) == 0) nChan = 2; else nChan = 4;
   i = fits_read_key(inFile, TSTRING, "HIERARCH ESO INS GRIS NAME", gris,0,&ierr);
   printf("   disperser is = %s\n", gris);
   if (skyZero == 1) {
      printf("   skyZero requested. \n");
      if ( (strncmp(opt1,highSens, 9) == 0) ||
           (strncmp(shutter,ABshutter,6) == 0)) {
         printf("   Cannot do this for given beam combiner and shutter.\n");
         skyZero = 0;
      }
   }
/*
      some history records
*/
   for (iFile=0; iFile<nFiles; iFile++) {
      sprintf(history,"oirChopPhotoImages INPUT %i %s", iFile+1, 
         oirFileRoot(fileList[iFile]));
      i=fits_write_history(outFile, history, &ierr);
   }
/*
      move to detector section
*/
   DetectorDef = read_imaging_detector_table(inFile, &ierr);
   if (ierr !=0) {
      printf("ERROR %i opening detector table in %s\n", 
         ierr, fileList[0]);
      return 1;
   }
   nRegion = DetectorDef->nregion;
   for (i=0;i<nRegion;i++) {
      nxList[i] = DetectorDef->naxis[i][0];
      nyList[i] = DetectorDef->naxis[i][1];
   }
   nx = DetectorDef->naxis[0][0];
   ny = DetectorDef->naxis[0][1];

   i = fits_write_key (outFile, TINT, "NBIN", (void*)(&nbin), "# bins used to calculate RMS", &ierr);
/*
      allocate accumulator arrays
*/
  
   npix = ny * (long)nx;
   Bin = (float ***)malloc(NBIN * sizeof(float **));
   for (iBin=0;iBin<NBIN;iBin++) Bin[iBin] = (float **) malloc(nChan * sizeof(float *));
   for (iChan=0;iChan<nChan;iChan++) {
      Target[iChan] = (float *)malloc(npix * sizeof(float));
      memset (Target[iChan], 0, npix*sizeof(float));
      Sky[iChan]    = (float *)malloc(npix * sizeof(float));
      memset (Sky[iChan], 0, npix*sizeof(float));

      for (iBin=0; iBin<NBIN; iBin++) {
         Bin[iBin][iChan] = (float *)malloc(npix * sizeof(float));
         memset (Bin[iBin][iChan], 0, npix*sizeof(float));
      }
   }
   i = fits_close_file(inFile, &ierr);
/*
          now process the real data
*/
   if (nShutter == 1) printf("   Found %i A-photometry Files:\n", nFiles);
   if (nShutter == 2) printf("   Found %i B-photometry Files:\n", nFiles);
   if (nShutter == 0) printf("   Found %i AB-data Files:\n", nFiles);
   for (i=0; i<nFiles; i++) printf("      %s \n", fileList[i]);
   unChopData(fileList, nFiles, nChan, nDrop, npix, npix, 
      Target, Sky, &TargetN, &SkyN, &nCycle, &ierr);
   if (ierr != 0) {
      printf("\nFailed to process Data: ERROR: %i\n", ierr);
      return 1;
   }
/*
       normalize data by dividing by #frames
*/
   for (iChan=0;iChan<nChan;iChan++) {
      normalizeExposure(Target[iChan], npix, TargetN, dit);
      normalizeExposure(Sky[iChan], npix, SkyN, dit);
/*
       subtract sky
*/
      for (iPix = 0; iPix< npix; iPix++) 
         Target[iChan][iPix] = Target[iChan][iPix] - Sky[iChan][iPix];
   }
#ifdef DEBUG
   debugFile = fopen("/tmp/rawphot.dat","w");
   for (iChan=0;iChan<nChan;iChan++) for (iPix=0;iPix<npix;iPix++)
      fprintf(debugFile, " %f \n",Target[iChan][iPix]);
   fclose(debugFile);
#endif
   /* go back through data and accumlate data in bins */
         printf("   Processing sub-exposures\n");
   binChopData(fileList, nFiles, nChan, nDrop, npix, npix, nCycle, Bin, &ierr); 
   if (ierr != 0) {
      printf("Failed to process RMS-Data: ERROR: %i\n", ierr);
      return 1;
   }
/*
              normalize 
*/
   for (iBin=0; iBin<NBIN; iBin++) for (iChan=0;iChan<nChan;iChan++) 
      normalizeExposure(Bin[iBin][iChan], npix, 1., dit);
   
      oirKillFileList(fileList, nFiles);
/*
      deSky; remove sky using pixels along top and bottom
      leave the details to the deSky subroutine.
      if dSky is larger than ny/2 take that as an indication that we
      dont want to do any columnwise sky subtraction
*/
      if (dSky >= (ny/2)) printf("   NOT subtracting columnwise sky estimate\n");
      if (dSky < (ny/2)) {
         printf("   subtracting columnwise sky estimate\n");
         for (iChan=0; iChan<nChan; iChan++) {
            if (maskFrameData != NULL) weightMask = maskFrameData->data[iChan];
            deSky(iChan, Target[iChan], nx, ny, gris, dSky, UT,
               refFileName, weightMask, polyOrder, &ierr);
            if (ierr != 0) return 1;
         }
         for (iChan=0; iChan<nChan;iChan++) for (iBin=0; iBin<NBIN; iBin++) {
            if (maskFrameData != NULL) weightMask = maskFrameData->data[iChan];
            deSky(iChan, Bin[iBin][iChan], nx, ny, gris, dSky, UT,
               refFileName, weightMask, polyOrder, &ierr);
            if (ierr != 0) return 1;
         }
      }
   odata = make_imaging_fdata_table_def (outFile, DetectorDef, 1, &ierr);
   if (ierr !=0) {
      fprintf (stderr, "ERROR creating data definition\n");
      return 1;
   }
   outData = make_imaging_fdata_row(odata);

   ii = 1;
   for (iChan=0; iChan<nChan; iChan++) 
      for (ix=0;ix<npix;ix++) outData->data[iChan][ix] = Target[iChan][ix];
   i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);

   for (iBin=0;iBin<NBIN;iBin++) {
      for (iChan=0; iChan<nChan; iChan++) 
         for (ix=0;ix<npix;ix++) outData->data[iChan][ix] = Bin[iBin][iChan][ix];
      i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);
   }
   if (skyZero == 1) zeroSky(Sky, nx, ny, shutter, DetectorDef);
   for (iChan=0; iChan<nChan; iChan++) 
      for (ix=0;ix<npix;ix++) outData->data[iChan][ix] = Sky[iChan][ix];
   i = write_imaging_fdata_table(outFile, odata, outData, ii++, &ierr);

   i = close_imaging_fdata_table(outFile, odata, &ierr);
   kill_imaging_fdata_row(outData);

   for (iChan=0; iChan<nChan; iChan++) {
      free(Target[iChan]);
      free(Sky[iChan]);
   }
   for (iBin=0;iBin<NBIN;iBin++) {
      for (iChan=0; iChan<nChan; iChan++) {
         free(Bin[iBin][iChan]);
      }
      free(Bin[iBin]);
   }
   free(Bin);
   return 0;
}
