;
; Copyright 2005 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
; $Id: mask_utilities.pro,v 1.10 2005/12/14 13:30:48 koehler Exp $
;
; mask_utilities.pro
; Created:     Tue Apr 19 16:12:49 2005 by Koehler@rommie
; Last change: Wed Dec 14 14:27:50 2005
;
; PURPOSE:
;	make the world a better place by masking the ugly parts
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; the official way to compute the trace_fit.
; guaranteed free of Chesneau-code!
; NOTE: it does not do the actual fitting,
;	it only creates the fitted trace!
;
; CALLING SEQUENCE:
;	trace_fit = make_trace(x_size,trace_coeff,sigma_coeff)
;
; INPUT:
;	x_size		size in x-direction (length of trace)
;	trace_coeff	coefficients of trace-fit
;	sigma_coeff	coefficients of sigma-fit
;
; OUTPUT:
;	2D-array:
;	trace_fit[0,*]	trace
;	trace_fit[1,*]	trace-FWHM
;	trace_fit[2,*]	trace+FWHM
;
Function make_trace, x_size, trace_coeff, sigma_coeff

  tf = fltarr(3,x_size)
  tf[0,*]= poly(findgen(x_size),trace_coeff)
  FWHM   = poly(findgen(x_size),sigma_coeff) * sqrt(2.*alog(2))
  tf[1,*]= tf[0,*] - FWHM
  tf[2,*]= tf[0,*] + FWHM
  return, tf
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO plot_trace, iFile, x_size, mtrace, COLORS=colors

	;    Fringes PhotA PhotB
  symbols= [ 2,	     1,    7	]

  rng = mtrace.range[0:mtrace.nrange]
  tr = make_trace(x_size,mtrace.trace_coeff,mtrace.sigma_coeff)

  if keyword_set(colors) then begin
      defcol = !P.color
      !P.color= colors[iFile]
      oploterr, rng, mtrace.trace[rng], mtrace.FWHM[rng], symbols[iFile]
      oplot, tr[0,*]
      oplot, tr[1,*]
      oplot, tr[2,*]
      !P.color= defcol
  endif else begin
      oplot, rng, mtrace.trace[rng], psym=symbols[iFile]
      oplot, tr[0,*], linestyle=iFile
      oplot, tr[1,*], linestyle=iFile
      oplot, tr[2,*], linestyle=iFile
  endelse
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO plot_traces, mtraces, TITLE=title, COLORS=colors, POSITION=position, YRANGE=yrange

  nTraces= N_elements(mtraces)
  x_size = mtraces[0].xsize

  if not keyword_set(yrange) then begin
      mint = 999
      maxt = 0
      for i=0,nTraces-1 do begin
          if mtraces[i].trace_coeff[0] ne 0 then begin
              tr= make_trace(x_size, mtraces[i].trace_coeff, mtraces[i].sigma_coeff)
              mint = min([ mint, reform(tr[1,*]) ])
              maxt = max([ maxt, reform(tr[2,*]) ])
          endif
      endfor
      yrange = [mint,maxt]
  endif

  if keyword_set(position) then begin
      ;print,"Position is ",position
      plot, [0,x_size], yrange, TITLE=title, /nodata,$
        POSITION=position, /device, /noerase
  endif else begin
      plot, [0,x_size], yrange, TITLE=title, /nodata
  endelse

  XYOuts, x_size*0.03,yrange[0]+(yrange[1]-yrange[0])*0.95, $
    "Center = " + strtrim(string(mtraces[0].trace_coeff[0]),2) + $
      " + x * " + strtrim(string(mtraces[0].trace_coeff[1]),2) + $
    " + x^2 * " + strtrim(string(mtraces[0].trace_coeff[2]),2)

  XYOuts, x_size*0.03,yrange[0]+(yrange[1]-yrange[0])*0.90, $
    " FWHM = " + strtrim(string(mtraces[0].sigma_coeff[0]),2) + $
      " + x * " + strtrim(string(mtraces[0].sigma_coeff[1]),2)

  for i=0,nTraces-1 do $
    if mtraces[i].trace_coeff[0] ne 0 then begin
      ;if keyword_set(colors) then print,"trace",i,", color=",colors[i]
      plot_trace, i, x_size, mtraces[i], COLORS=colors
    endif
end
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; check_mask_position
; displays RMS of cube multiplied by mask and asks user if it is ok
; This is only used by the ancient routines midivisi and xmidivisi
;
; returns name of input maskfile,
;	or the mask selected by the user,
;	or a null string if user cancelled selection
;
function check_mask_position, fringefile, maskfile

    rmsdata = Cached_MeanRMS(fringefile)
    maskdata= oirgetdata(maskfile)

    screensz= get_screen_size()
    datasz  = size(rmsdata.data1)
    factor  = floor( screensz[0]/datasz[1] < screensz[1]/4/datasz[2] < 5)
    print,"screen size: ", screensz
    print,"data size:   ", datasz[[1,2]]
    print,"enlarging windows by ",factor
    sz= size(rmsdata.data1) * factor
    window,0, xsize=sz[1], ysize=sz[2]*4.2
    loadct,0
    tvscl, rebin(sqrt(rmsdata[1].data2), sz[1], sz[2], /sample), 0,sz[2]*3.2
    tvscl, rebin(sqrt(rmsdata[1].data1), sz[1], sz[2], /sample), 0,sz[2]*1.0

    tvscl, rebin(sqrt(rmsdata[1].data2*maskdata[0].data2),sz[1],sz[2],/sample),0,sz[2]*2.2
    tvscl, rebin(sqrt(rmsdata[1].data1*maskdata[0].data1),sz[1],sz[2],/sample),0,0

    testOk= "string"
    print,'Is the position of the mask satisfactory? (y/n)'
    read, '(Type n to select another maskfile) >> ', testOK

    if (strupcase(testOK) eq "Q") then maskfile=""
    if (strupcase(testOK) eq "N") then begin
        maskfile = dialog_pickfile(Default_Extension=".fits",$
                                   Filter = "Mask_*.fits", $
                                   File = maskfile, /Must_Exist, $
                                   Title= "Select a Fits-file containing a MIDI-mask")
    endif
    return,maskfile
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function make_mask_from_trace, trace, WIDTH=width

  if not keyword_set(width) then width=1.

  mask= fltarr( trace.xsize, trace.ysize)
  trace_fit =  poly( findgen(trace.xsize), trace.trace_coeff)
  sigm2_fit = (poly( findgen(trace.xsize), trace.sigma_coeff) * width)^2 * 2.
  y = findgen(trace.ysize)
  FOR i= 0, trace.xsize-1 DO BEGIN
      mask[i,*]= exp( -(y-trace_fit[i])^2. / sigm2_fit[i])
  endfor
  return, mask
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask__Define
xx = { miamask, $
       fringefile: '', photfiles: ['',''], maskfile: '', $
       trace: replicate({midisptrace},2,4),$	 ;;; 1.index=file, 2.index=window
       traceFr: replicate({midisptrace},4),$	 ;;; 4 windows for SCI_PHOT
       traceFr0: replicate({midisptrace},4),$	 ;;; initial values
       fringemean: ptr_new(), photAmean: ptr_new(), photBmean: ptr_new(),$
       factor: 1., color: lonarr(6), formID: -1, drawID: -1, $
       BgImg: 2, Window: 1, Lines: [1,1,1], $
       maskfrom: 0, TraceFix: [0,0,0], FWHMFix: [0,0,0] $
}
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
Function miamask::init, fringefile, photfiles, WIDTH=width, dSky=dSky, $
                	MASKFILE=maskfile, ERRORLOG=errorlog, NOEWSPHOT=noewsphot

common MidiOptions,midi_options,maskfile_mymidi,kappafile,la_bins	; CAH

  if not keyword_set(width) then width=1. else print,"Maskwidth is",width
  if not keyword_set(dsky) then dsky=0
  ;print,"miamask: errorlog is:_"+errorlog+"_ = "+string(N_elements(errorlog))

  nWin = 0	; 1.file tells us if we have HAI-Sense or SY-Phot
  mtrace= {midisptrace}
  mean_tcoeff= fltarr(4,3)	;; 1.idx = window, 2.idx = polynom-coeff
  mean_scoeff= fltarr(4,2)
  nFiles = intarr(4)		;; counts the files per window we get coeffs from

  ;print,"Fringes: ",fringefile
  self.fringefile= fringefile
  self.fringemean= ptr_new(Cached_MeanRMS(self.fringefile), /no_copy)

  ;;; SCI MODE by FP (under development):
  if N_params() lt 2 then begin    ;;; modified for sci_mode (=no photfiles)
      ;;print,'No photometry files given, entering SCI MODE (under development)'
      self.photfiles[0] = self.fringefile
      self.photfiles[1] = self.fringefile
      nWin = 4                  ; puke if files are not SCI_photic enough
  endif else begin
      self.photfiles = photfiles
  endelse

  if N_elements(self.photfiles) lt 2 then begin
      if N_Elements(errorlog) gt 0 then begin
          errorlog = errorlog + "miamask needs two photometry files, but got only one"
      endif else begin
          print,"miamask needs two photometry files, but got only one!"+string(7B)
      endelse
      return,0
  endif
  if N_elements(self.photfiles) gt 2 then $
    print,"miamask uses only two photometry files"

  if keyword_set(maskfile) then begin
      self.maskfrom= 1  &  self.maskfile = maskfile
      return,1	; CAH
  endif else begin
      self.maskfrom= 0
      self.maskfile= 'PHOTOMETRY/Mask' + $
        "_" + MIDIname_datime(self.photfiles[0]) + $
        "+" + MIDIname_time(self.photfiles[1]) + ".fits"
  endelse

  datatag = intarr(4)

  if not keyword_set(noewsphot) then begin
      print, ""
      print, "========== Reading photometry (EWS-style) ==========="

      chopimgfile = "PHOTOMETRY/ChIm_" + $
        MIDIname_datime(self.photfiles[0]) + "+" + $
        MIDIname_time(self.photfiles[1]) + "_dsky" + strtrim(dSky,2) +".fits"

      if not file_test(chopimgfile) then begin
          print, "Cached chop-frame does not exist, let's create it!"

          cmd = getenv("drsRoot") + '/c/bin/oirChopJitterPhotoImages'
          if file_test(cmd,/EXECUTABLE) then begin
              if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"
              cmd = cmd + ' "' + self.photfiles[0] + '" "' +$
                self.photfiles[1] + '" ' + chopimgfile
	      if keyword_set(dSky) then cmd = cmd + " -dSky " + strtrim(dSky,2)
              print,cmd
              spawn,cmd
          endif else begin
              print,'The program'
              print,cmd
              print,'does not exist or is not executable.'
              print,"Please make sure MIA+EWS was installed properly."
              exit
          endelse
      endif
      chop_images = oirgetdata(chopimgfile)

      tagnms= Tag_Names(chop_images)
      ;for iTag= 0, N_Tags(chop_images)-1 do print,"Tag",iTag,"= ",tagnms[iTag]

      datatag[0] = where(tagnms eq "DATA1")
      datatag[1] = where(tagnms eq "DATA2")
      datatag[2] = where(tagnms eq "DATA3")
      datatag[3] = where(tagnms eq "DATA4")
      if datatag[3] eq -1 then nWin=2 else nWin=4
; CAH Use sub-exposure files
      if midi_options.j then begin
	      chop_images(0).data1=0
	      chop_images(0).data2=0
	      chop_images(1).data1=0
	      chop_images(1).data2=0
	      for j=0,n_elements(chop_images)/2-2 do begin
		      chop_images(0).data1=chop_images(0).data1+chop_images(2+j*2).data1
		      chop_images(0).data2=chop_images(0).data2+chop_images(2+j*2).data2
		      chop_images(1).data1=chop_images(1).data1+chop_images(3+j*2).data1
		      chop_images(1).data2=chop_images(1).data2+chop_images(3+j*2).data2
	      endfor
      endif
      self.photAmean= ptr_new(chop_images[0],/no_copy)
      self.photBmean= ptr_new(chop_images[1],/no_copy)
  endif

  for iFile= 0, 1 do begin
      if keyword_set(noewsphot) then begin
          print, ""
          print, "========== Reading photometry",iFile," =========="
          print, self.photfiles[iFile]

          chop_image= MIDISkySubImage(self.photfiles[iFile], before=1, after=1, ERRORLOG=errorlog)

          if nWin eq 0 then nWin= N_Tags(chop_image) $
          else if nWin ne N_Tags(chop_image) then begin
              print,"Wrong number if windows, can't work with this"
              return,0
          endif
          tagnms= Tag_Names(chop_image)
          ;;for iTag= 0, nWin-1 do print,"Tag",iTag,"= ",tagnms[iTag]

          datatag[0] = where(tagnms eq "DATA1")
          datatag[1] = where(tagnms eq "DATA2")
          datatag[2] = where(tagnms eq "DATA3")
          datatag[3] = where(tagnms eq "DATA4")
      endif else begin
          print, "===== Photometry file",iFile," = ", self.photfiles[iFile]
      endelse

      print,nWin," Windows at tags",datatag
      print,""

      if (datatag[0] lt 0) or (datatag[1] lt 0) or (nWin ge 4 and datatag[2] lt 0) then begin
          print,"Can't find enough windows, I'm confused!"
          return,0
      endif

      im = obj_new('imagedata',(oirFileList(self.photfiles[iFile]))[0])
      he = im->prihead()
      shutter= strcompress(he->getpar('INS SHUT NAME'),/remove_all)
      obj_destroy,im

      for iWin= 0,nWin-1 do begin
          print,"----- doing window",iWin," = ",tagnms[datatag[iWin]],"-----"

          if (nWin ge 4) and $
            (iWin eq 0  and  shutter eq 'AOPEN') or $
            (iWin eq 3  and  shutter eq 'BOPEN') then begin
              print,"No light in SCI_PHOT mode"
          endif else begin
      	      fitmask,/init,iFile=iFile,iWin=iWin	; CAH
              if keyword_set(noewsphot) then begin
                  chop_nod_disp_one_window, chop_image.(datatag[iWin]),1.,2,1,mtrace,fx,fxc,junkmask,/silent
              endif else begin
                  chop_nod_disp_one_window, chop_images[iFile].(datatag[iWin]),1.,2,1,mtrace,fx,fxc,junkmask,/silent
              endelse
              self.trace[iFile,iWin] = mtrace

              answer= ''
              trace= poly(findgen(mtrace.xsize),mtrace.trace_coeff)
              if (min(trace) lt 0) or (max(trace) ge mtrace.ysize) then begin
                  if N_Elements(errorlog) gt 0 then begin
                      errorlog = errorlog + self.photfiles[iFile] + $
                        ", Window " + strtrim(string(iWin),2) +$
                        ": spectrum outside of window" + string(10B)
                  endif else begin
                      read,"TROUBLE: spectrum outside of window! [Hit Return]"+string(7B),answer
                  endelse
              endif
              if (mtrace.sigma_coeff[0] le 0.) or $
                (mtrace.sigma_coeff[0] + mtrace.sigma_coeff[1]*mtrace.xsize le 0.) then begin
                  if N_Elements(errorlog) gt 0 then begin
                      errorlog = errorlog + self.photfiles[iFile] + $
                        ", Window " + strtrim(string(iWin),2) +$
                        ": negative FWHM of Spectrum" + string(10B)
                  endif else begin
                      read,"TROUBLE: negative FWHM of spectrum! [Hit Return]"+string(7B),answer
                  endelse
              endif
              mean_tcoeff[iWin,*] = mean_tcoeff[iWin,*] + mtrace.trace_coeff
              mean_scoeff[iWin,*] = mean_scoeff[iWin,*] + mtrace.sigma_coeff
              nFiles[iWin] = nFiles[iWin]+1
          endelse
      endfor	;; iWin
      if keyword_set(noewsphot) then begin
          ;; this destroys chop_image and muste therefore be at the end
          if iFile eq 0 then $
            self.photAmean= ptr_new(chop_image,/no_copy) $
          else if iFile eq 1 then $
            self.photBmean= ptr_new(chop_image,/no_copy)
      endif
  endfor	;; iFile

  print,""
  print,"========== averaging traces and widths ========="
  iFile= N_elements(self.photfiles)
  sz = size( (*self.photAmean).data1)	;; if the size of the windows are not identical, we have a problem
  for iWin= 0,nWin-1 do begin
      print,"----- window",iWin," = ",tagnms[datatag(iWin)],"-----"
      self.traceFr[iWin].trace_coeff = mean_tcoeff[iWin,*] / nFiles[iWin]
      self.traceFr[iWin].sigma_coeff = mean_scoeff[iWin,*] / nFiles[iWin] * width

      print,"trace coeff: ",self.traceFr[iWin].trace_coeff
      print,"sigma coeff: ",self.traceFr[iWin].sigma_coeff

      szi= size( (*self.photAmean).(datatag[iWin]))
      ;;print,"size of ",tagnms[iWin]," =",szi
      if (sz[1] ne szi[1]) or (sz[2] ne szi[2]) then begin
          print,"Sizes of windows not equal, I can't handle this!"
          return,0
      endif
      self.traceFr[iWin].xsize= sz[1]
      self.traceFr[iWin].ysize= sz[2]
      self.traceFr[iWin].trace= poly(findgen(sz[1]),self.traceFr[iWin].trace_coeff)
      self.traceFr[iWin].FWHM = poly(findgen(sz[1]),self.traceFr[iWin].sigma_coeff)*sqrt(2.*alog(2.))
      self.traceFr[iWin].nRange= sz[1]
      self.traceFr[iWin].range = findgen(sz[1])
  endfor
  self.traceFr0= self.traceFr
  return,1
end

function miamask, files, WIDTH=width, MASKFILE=maskfile
      return, obj_new('miamask',files[0],files[1:2],WIDTH=width,MASKFILE=maskfile)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::gui, TRUECOLOR=truecolor, NO_BLOCK=no_block

  self.color = mia_std_colortable(truecolor)

  screensz= get_screen_size()
  datasz  = size( (*self.fringemean).data1)
  self.factor= floor( (screensz[0]-50)/datasz[1] < (screensz[1]-100)/(datasz[2]*2) < 7)
  ; controls are >300pix on rommie
  print,FORMAT='("screen size: ", I4," x ",I4,", data size: ",I3," x ",I2," => enlarging by factor",I2)',$
    screensz, datasz[[1,2]], self.factor
  sz= size( (*self.fringemean)[1].data1) * self.factor
  sz[2] = sz[2]*2

  !X.style=1	; make sure the plots are exactly the size we want!
  !Y.style=1

  self.BgImg = 0	; set defaults to same values as in cw_form!
  self.Window= 0
  self.Lines = [1,1,1]
  self.TraceFix= [0,0,0]
  self.FWHMFix = [0,0,0]

  base = Widget_Base(Title="Maskedit",/COLUMN)
  desc = [ $
        '1, BASE, ,COLUMN',$
         '1, BASE, ,ROW',$
         '0, BUTTON, Fringes|Photometry A|Photometry B, Column, Frame,'+$
	        'LABEL_TOP=Background Img, SET_VALUE=0,EXCLUSIVE,TAG=BgImg',$
         '2, BUTTON, Fringes|Photometry A|Photometry B, Column, Frame,'+$
		'LABEL_TOP=Lines plotted, SET_VALUE=[1\,1\,1], TAG=Lines',$
         '2, BUTTON, Window 1|Window 2, Row, Frame,'+$
		'LABEL_Top=Detector Window,SET_VALUE=0,EXCLUSIVE,TAG=Window',$
         '0, BUTTON, Coefficients|File, Column, Frame,'+$
                'LABEL_TOP=Mask from:, SET_VALUE='+string(self.maskfrom)+',EXCLUSIVE,TAG=MaskFrom',$
        '1, BASE, ,COLUMN',$
         '1, BASE, ,ROW, Frame',$
          '1, BASE, ,COLUMN',$
           '0, LABEL, Trace Coefficients', $
            $;'1, BASE, ,COLUMN, FRAME',$
            '1, BASE, ,ROW, FRAME', $
             '0, FLOAT,'+string(self.traceFr[0].trace_coeff[0])+',LABEL_LEFT=Trace=,Width= 8,TAG=Trace0', $
             '0, FLOAT,'+string(self.traceFr[0].trace_coeff[1])+',LABEL_LEFT=+ x*,  Width=10,TAG=Trace1', $
             '2, FLOAT,'+string(self.traceFr[0].trace_coeff[2])+',LABEL_LEFT=+ x^2*,Width=12,TAG=Trace2',$
            $;'2, BUTTON, Fixed        |Fixed            |Fixed, ROW, RIGHT, LABEL_LEFT= Fit :,TAG=TraceFix',$
           '2,BASE',$
          '1, BASE, ,COLUMN',$
           '0, LABEL, Width Coefficients',$
            $;'1, BASE, ,COLUMN, FRAME',$
            '1, BASE, ,ROW, FRAME', $
             '0, FLOAT,'+string(self.traceFr[0].sigma_coeff[0])+',LABEL_LEFT=FWHM=,Width= 9,TAG=FWHM0',$
             '2, FLOAT,'+string(self.traceFr[0].sigma_coeff[1])+',LABEL_LEFT=+ x*, Width=11,TAG=FWHM1',$
            $;'2, BUTTON, Fixed         |Fixed, ROW, RIGHT, LABEL_LEFT= Fit:,TAG=FWHMFix',$
           '2,BASE',$
          $;'1, BASE, ,COLUMN',$
           $;'0, BUTTON, Fit Parameters, TAG=FIT',$
           '2, BUTTON, Reset, TAG=RESET',$
	  '1, BASE, ,ROW, Frame',$
           '0, TEXT, '+self.maskfile+', Width=93, LABEL_LEFT=Filename:, TAG=Filename',$
           '2, BUTTON,  Browse , TAG=BROWSE',$
;         '2, BASE',$
          '0, BUTTON, Dismiss GUI, TAG=CREATE' ]

  if where( Tag_Names(*self.photAmean) eq "DATA4") ge 0 then begin
      desc[4] = '2, BUTTON, Window 1|Window 2|Window 3|Window 4, Row, Frame,'+$
		  'LABEL_Top=Detector Window,SET_VALUE=1,EXCLUSIVE,TAG=Window'
      self.Window= 1
  endif

  ;left of draw: form = cw_form(base,desc,/column)
  draw = Widget_Draw(base, XSIZE=sz[1]+50, YSIZE=sz[2]+50)
  self.formID= cw_form(base,desc);,/column)

  WIDGET_CONTROL, base, /REALIZE
  WIDGET_CONTROL, base, SET_UVALUE=self
  WIDGET_CONTROL, draw, GET_VALUE=drawID & self.drawID= drawID

  self->draw

  xmanager, 'miamask', base, NO_BLOCK=no_block
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask_event, ev
  ;print,"event1!"
  WIDGET_CONTROL, ev.TOP, GET_UVALUE=m
  m->event, ev
end

PRO miamask::event, ev

  if ev.id eq self.formID then begin
      ;print,"form event, tag: ",ev.tag,", value: ",ev.value

      case (ev.tag) of
          "BGIMG": if self.BgImg ne ev.value then begin
              self.BgImg= ev.value
              self->draw
          end
          "WINDOW": if self.Window ne ev.value then begin
              self.Window= ev.value
              ; they changed the window, reload all the coeffs
              Widget_Control, ev.ID, Get_value=v
              v.Trace0= self.traceFr[self.window].trace_coeff[0]
              v.Trace1= self.traceFr[self.window].trace_coeff[1]
              v.Trace2= self.traceFr[self.window].trace_coeff[2]
              v.FWHM0 = self.traceFr[self.window].sigma_coeff[0]
              v.FWHM1 = self.traceFr[self.window].sigma_coeff[1]
              Widget_Control, ev.ID, Set_value=v
              self->draw
          end
          "LINES": $
            if (self.Lines[0] ne ev.value[0]) $
            or (self.Lines[1] ne ev.value[1]) $
            or (self.Lines[2] ne ev.value[2]) then begin
              self.Lines= ev.value
              self->draw
          end
          "MASKFROM": if self.maskfrom ne ev.value then begin
              if file_test(self.maskfile,/read,/regular) then begin
                  self.maskfrom= ev.value
                  self->draw
              endif else begin
                  self.maskfrom= 0
                  Widget_Control, ev.ID, Get_value=v
                  v.Maskfrom = 0
                  Widget_Control, ev.ID, Set_value=v
                  msg= dialog_message(["A file named",$
                                       self.maskfile,$
                                       "does not exist, is not a regular file,",$
                                       "or you don't have permission to read it.",$
                                       "You can't use that file as mask."],$
                                      /ERROR, Dialog_Parent=ev.top)
              endelse
          end
          ;;;;;;;;;;;;;;;;;;;; Coefficients ;;;;;;;;;;;;;;;;;;;;
          "TRACE0": begin
              ;; perverted logic: if the user types or deletes a
              ;; digit, the value changes, but if he hits return, the
              ;; value does NOT change.  So we draw on non-changes...
              if self.traceFr[self.window].trace_coeff[0] eq ev.value then $
                self->draw
              self.traceFr[self.window].trace_coeff[0]= ev.value
          end
          "TRACE1": begin
              if self.traceFr[self.window].trace_coeff[1] eq ev.value then $
                self->draw
              self.traceFr[self.window].trace_coeff[1]= ev.value
          end
          "TRACE2": begin
              if self.traceFr[self.window].trace_coeff[2] eq ev.value then $
                self->draw
              self.traceFr[self.window].trace_coeff[2]= ev.value
          end
          "TRACEFIX":
          "FWHM0": begin
              if self.traceFr[self.window].sigma_coeff[0] eq ev.value then $
                self->draw
              self.traceFr[self.window].sigma_coeff[0]= ev.value
          end
          "FWHM1": begin
              if self.traceFr[self.window].sigma_coeff[1] eq ev.value then $
                self->draw
              self.traceFr[self.window].sigma_coeff[1]= ev.value
          end
          "FWHMFIX":
          "RESET": begin
              self.traceFr= self.traceFr0
              Widget_Control, ev.ID, Get_value=v
              v.Trace0= self.traceFr[self.window].trace_coeff[0]
              v.Trace1= self.traceFr[self.window].trace_coeff[1]
              v.Trace2= self.traceFr[self.window].trace_coeff[2]
              v.FWHM0 = self.traceFr[self.window].sigma_coeff[0]
              v.FWHM1 = self.traceFr[self.window].sigma_coeff[1]
              Widget_Control, ev.ID, Set_value=v
              self->draw
          end
          ;;;;;;;;;;;;;;;;;;;;;; file ;;;;;;;;;;;;;;;;;;;;;;
          "FILENAME": self.maskfile= ev.value
          "BROWSE": begin
              maskfile = dialog_pickfile(Dialog_Parent=ev.top, /Must_Exist,$
                                         Default_Extension= ".fits",$
                                         Filter = "*Mask_*.fits", $
                                         File = file_basename(self.maskfile),$
                                         Path = file_dirname(self.maskfile),$
                                         Title= "Select a Fits-file containing a MIDI-mask")
              if maskfile ne "" then begin
                  self.maskfile= maskfile
                  self.maskfrom= 1
                  Widget_Control, ev.ID, Get_value=v
                  v.Filename= self.maskfile
                  v.MaskFrom= self.maskfrom
                  Widget_Control, ev.ID, Set_value=v
                  self->draw
              endif
          end
          "FIT": msg= dialog_message(["Not yet implemented."], Title="Sorry")
          ;; self->fit_trace_and_fwhm
          "CREATE": Widget_control, ev.top, /Destroy	; self->destruct
      endcase
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::draw

  ;;print,"BgImg",self.BgImg,", Window",self.window,", Lines",self.lines
  case self.BgImg of
      0: begin & img = (*self.fringemean)[1]& print,"showing fringe"& end
      1: begin & img = (*self.photAmean)[0] & print,"showing photA" & end
      2: begin & img = (*self.photBmean)[0] & print,"showing photB" & end
  endcase
  if self.maskfrom eq 1 and file_test(self.maskfile,/read,/regular) then begin
      mask = oirgetdata(self.maskfile)
      help,mask,/str
      img.data1 = img.data1 * mask.data1
      img.data2 = img.data2 * mask.data2
  endif
  case self.Window of
      0: img = img.data1
      1: img = img.data2
      2: img = img.data3
      3: img = img.data4
  endcase
  if self.BgImg eq 0 then img= sqrt(img)
  sz= size( img) * self.factor
  sz[2] = sz[2]*2

  wset,self.drawID
  tvscl, rebin(img, sz[1], sz[2], /sample), 30,30
  position= [30, 30, 30+sz[1], 30+sz[2]]
  yrange = [-0.5, sz[2]/self.factor/2-0.5]

  if self.maskfrom eq 0 then begin
      traces= [ self.traceFr[self.window], self.trace[0,self.window], self.trace[1,self.window] ]
      index=where(self.Lines ne 0,count)        ; CAH, this line and the follow.
      if count gt 0 then $
      plot_traces, traces[where(self.Lines ne 0)], POSITION=position, $
        COLORS=self.color[where(self.Lines ne 0)], YRANGE= yrange
  endif else begin
      plot, [0,sz[1]/self.factor], yrange, POSITION=position,/nodata,/device,/noerase
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::fit_trace_and_fwhm

  self.traceFr[self.window].trace_coeff = [ 0.,0.,0. ]
  self.traceFr[self.window].sigma_coeff = [ 0.,0. ]

  for iFile=0,1 do begin
      x = indgen(self.trace[iFile,self.window].xsize)
      tracemeas = self.trace[iFile,self.window].trace
      order = 3
  endfor
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pro miamask::show_traces, fname, NOWIN=nowin, COLORS=colors

  if (N_Params() gt 0) then begin
      set_plot,"ps"
      device,file=fname,/landscape,/color
      nowin= 1
  endif
  if not keyword_set(nowin) then window,3,title="Traces"
  if not keyword_set(colors) then colors= self.color

  nWin= N_Tags(*self.photAmean)
  sz  = size((*self.photAmean).data1)
  !P.multi=[0,1,nWin]

  for i= 0,nWin-1 do begin
    plot_traces, [self.traceFr[i],self.trace[0,i],self.trace[1,i]],$
	title="Window "+strtrim(string(i+1),2), COLORS=colors, yrange=[-0.5,sz[2]-0.5]
    rng = self.traceFr[i].range[0:self.traceFr[i].nrange]
    oplot, rng, self.traceFr[i].trace[rng] + self.traceFr[i].FWHM[rng], psym=3
    oplot, rng, self.traceFr[i].trace[rng] - self.traceFr[i].FWHM[rng], psym=3
  endfor

  if (N_params() gt 0) then begin
      device,/close
      set_plot,"X"
  endif
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Function miamask::get_tracePh
  return, self.trace
end

Function miamask::get_traceFr
  return, self.traceFr
end

Function miamask::get_mask_name
  return, self.maskfile
end

Function miamask::get_mask_data

  if self.maskfrom eq 1 then begin
      print,"Reading mask from file ",self.maskfile
      return, oirgetdata(self.maskfile)
  endif else begin
      print,"Creating mask from fitted spectra"
      if (where(Tag_Names(*self.photAmean) eq "DATA4") lt 0) then nWin=2 else nWin=4

      mask= fltarr( nWin, self.traceFr[0].xsize, self.traceFr[0].ysize)
      for iWin= 0,nWin-1 do begin
          print,"trace_coeffs No.",iWin,": ",self.traceFr[iWin].trace_coeff
          mask[iWin,*,*]= make_mask_from_trace(self.traceFr[iWin])
      endfor

      if (nWin ge 4) then begin
          print,nWin," windows => Sci-Phot"
          return, { data1: reform(mask[0,*,*]), $
                    data2: reform(mask[1,*,*]), $
                    data3: reform(mask[2,*,*]), $
                    data4: reform(mask[3,*,*]) }
      endif else begin
          print,nWin," windows => Hi-Sense"
          return, { data1: reform(mask[0,*,*]), data2: reform(mask[1,*,*]) }
      endelse
  endelse
end

Pro miamask::write_mask
  if self.maskfrom eq 1 then begin
      print,"Mask loaded from file, not overwriting it"
  endif else begin
      print,"Writing mask ",self.maskfile
      dirname = file_dirname(self.maskfile)
      if not file_test(dirname) then file_mkdir,dirname
      oirNewData, (oirFilelist(self.photfiles[0]))[0], self.maskfile, self->get_mask_data()
  endelse
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PRO miamask::cleanup
  print,"miamask cleaning up..."
  ptr_free, self.fringemean
  ptr_free, self.photAmean
  ptr_free, self.photBmean
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; end of mask_utilities
