PRO FITSIMAGE::cleanup
   PTR_FREE, self.naxis
   self->FITSFILE::cleanup
RETURN
END

FUNCTION FITSIMAGE::init, filename, status, naxis=naxis, iErr=iErr
;Constructor for FITSIMAGE class
;
;INPUTS: see FITSIMAGE::open
;
   errMsg = ''
   bottom = 0          ; 0 means no call stack on err return
;default status
   if (N_PARAMS() EQ 1) then status = 'READ'
   st     = strUpCase(StrTrim(status, 2))
;on WRITE, naxis must be given
   if (st EQ 'WRITE') and (NOT KEYWORD_SET(naxis)) then begin
      errMsg = 'FITSIMAGE:NAXIS must be specified at opening new file'
      GOTO, ERR_RET_1
   endif
;open file as FITSFILE: for READ this stores header internally
;                       for WRITE this makes a default minimal header
   if (0 EQ self->FITSFILE::init(filename, st, iErr=iErr)) then GOTO, ERR_RET
   if (st  EQ 'WRITE') then begin
;copy naxis into local storage but also update in primary header
      self.naxis = PTR_NEW(naxis)
      nNaxis = n_elements(naxis)
      self.priHead->addPar, 'NAXIS', nNaxis, iErr=iErr
      if (iErr NE 0) then GOTO, ERR_RET
      for iAxis = 1, nNaxis do begin
         self.priHead->addPar,'NAXIS'+strTrim(string(iAxis),2), $
	    naxis(iAxis-1), iErr=iErr
         if (iErr NE 0) then GOTO, ERR_RET
      endfor
      self.headerStatus = 'INTERNAL'
   endif                     ; status is WRITE
   if (st EQ 'READ') then begin
;copy naxis into local storage from primary header
      nNaxis = self.priHead->getPar( 'NAXIS', Match )
      if (Match NE 1) then begin
         errMsg = 'NAXIS not found in header'
	 GOTO, ERR_RET_1
      endif
      self.naxis = PTR_NEW(lonarr(nNaxis))
      for iAxis = 1, nNaxis do begin
         (*self.naxis)[iAxis-1] =self.priHead->getPar('NAXIS' +$
	    strTrim(string(iAxis),2), Match)
         if (Match NE 1) then begin
            errMsg = 'NAXIS'+strTrim(string(iAxis),2)+' not found in header'
	    GOTO, ERR_RET_1
	 endif
      endfor
;set data pointers
      point_lun, -self.unit, pointLun
      self.dataStart   = pointLun
      self.dataCurrent = pointLun
      self.planeCurrent = 0
      if (nNaxis le 2) then self.planeLast = 1 else $
         self.planeLast = (*self.naxis)[2]
;figure out size
      byteSize = self.priHead->getPar('BITPIX', Match)
      if (Match NE 1) then begin
         errMsg = 'BITPIX not found in header'
	 GOTO, ERR_RET_1
      endif
      self.bitpix = byteSize
      byteSize = long(abs(byteSize)/8)
      for iAxis = 0, nNaxis-1 do byteSize = byteSize * (*self.naxis)[iAxis]
      self.dataEnd = pointLun + byteSize
      self.headerStatus = 'DISK'
      self.dataStatus = 'DISK'
   endif                     ; status is READ
   RETURN,1
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (bottom eq 1) then midiAddErrMsg, errMsg, /trace else $
      midiAddErrMsg, errMsg
   midiPrintErrMain
   iErr = 1
   RETURN,0
END

PRO FITSIMAGE::headerToDisk, iErr=iErr
;write the stored header onto disk
   self.priHead->writeToDisk, self.Unit, iErr=iErr
   if (iErr eq 0) then self.headerStatus = 'DISK'
RETURN
END

FUNCTION FITSIMAGE::readImage, iErr=iErr
;read an entire image as a block and return to user
   bottom = 0          ; on err_ret dump no call stack
   errMsg = ''
   ON_IOERROR, ERR_RET
;check status
   if ((self.state le 0) OR (self.unit le 0) OR (self.openStatus ne 'READ') $
       OR (NOT OBJ_VALID(self. priHead))) then begin
      errMsg = 'input file not correctly opened'
      GOTO, ERR_RET_1
   endif
;get data type and size
   byteSize = abs(self.bitpix)/8
   nNaxis = n_elements(*self.naxis)
   for iaxis = 0, nNaxis -1 do $
      byteSize = byteSize * (*self.naxis)[iaxis]
   if (byteSize le 0) then begin
      errMsg = 'input image has zero elements'
      GOTO, ERR_RET_1
   endif
   CASE self.bitpix OF
       8: begin     ; byte
         idlType = 1
      END
       16: begin     ; int
         idlType = 2
      END
       32: begin     ; long
         idlType = 3
      END
      -32: begin     ; float
         idlType = 4
      END
      -64: begin     ; double
         idlType = 5
      END
   ENDCASE
   if ((idlType le 0) or (idlType gt 6)) then begin
      errMsg = 'BITPIX unrecognized or unimplemented'
      GOTO, ERR_RET_1
   endif
;encode naxis as idl size array
   idlSize = [nNaxis, *self.naxis, idlType, byteSize/8]
   image = MAKE_ARRAY(size=idlSize)
;read from disk
   bottom = 1
   point_lun, self.unit, self.dataStart
   readu, self.unit, image
   self.dataCurrent = self.dataStart + byteSize
   self.dataEnd     = self.dataCurrent
;convert to local 
  IEEE_TO_Host, Image
   iErr = 0
RETURN, image
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (errMsg EQ '') then errMsg = !ERR_STRING+!SYSERR_STRING
   if (bottom EQ 1) then midiAddErrMsg,errMsg,/trace $
      else midiAddErrMsg
   midiPrintErrMain
   iErr = 1
RETURN, 0
END

FUNCTION FITSIMAGE::readPlanes, plane, nPlane=nPlane, iErr=iErr
;read one or more  planes of image into data cube
;INPUTS
;   plane    int   first plane in cube to be returned
;   nPlane   int   number of planes to read.  If not specified = 1
   bottom = 1          ; on err_ret dump no call stack
   errMsg = ''
   ON_IOERROR, ERR_RET
;check status
   if ((self.state le 0) OR (self.unit le 0) OR (self.openStatus ne 'READ') $
       OR (NOT OBJ_VALID(self. priHead))) then begin
      errMsg = 'input file not correctly opened'
      GOTO, ERR_RET_1
   endif
;check plane number and self defaults; Plane le 0 -> next plane
   if (NOT KEYWORD_SET(nPlane)) then nPlane = 1
;nPlane must be >= 1
   nP = nPlane > 1
   isNew = (self.PlaneCurrent eq 0)
   if (N_PARAMS() LT 1) then iPlane = self.planeCurrent + 1 $
   else if (plane le 0) then iPlane = self.planeCurrent + 1 $
   else iPlane = plane
;is this first call
   if (isNew) then begin
      nNaxis = N_ELEMENTS(*self.naxis)
;figure out how many planes in whole image
      if (nNaxis lt 2) then begin
         errMsg = 'image has less than 2 dimensions'
	 GOTO, ERR_RET_1
      endif else self.PlaneLast = 1 
      for i=2, nNaxis-1 do self.PlaneLast = self.PlaneLast * (*self.naxis)[i]
;get data type and size; bytes per plane
      byteSize = abs(self.bitpix)/8
      self.bytesPlane = byteSize * (*self.naxis)[0] * (*self.naxis)[1]
      if (self.bytesPlane le 0) then begin
         errMsg = 'input planes have zero elements'
         GOTO, ERR_RET_1
      endif
      CASE self.bitpix OF
         8: begin     ; byte
           idlType = 1
         END
         16: begin     ; int
           idlType = 2
         END
         32: begin     ; long
           idlType = 3
         END
         -32: begin     ; float
            idlType = 4
         END
         -64: begin     ; double
            idlType = 5
         END
      ENDCASE
      self.diskIdlType = idlType
      if ((idlType le 0) or (idlType gt 6)) then begin
         errMsg = 'BITPIX unrecognized or unimplemented'
         GOTO, ERR_RET_1
      endif
   endif      ; end of IsNew
;is requested plane legal
   if (iPlane+nP-1 gt self.planeLast) then begin
      errMsg = 'requested planes '+strtrim(string(iPlane),2)+$
        ':'+strtrim(string(iPlane+nP-1),2)+ $
	' greater than image plane size = '+strtrim(string(self.planeLast))
      GOTO, ERR_RET_1
   endif
;encode image size as idl size array
   if (nP EQ 1) then $
      idlSize = [2, (*self.naxis)[0:1], self.diskIdlType, self.bytesPlane/8]$
   else idlSize=[3, (*self.naxis)[0:1], nP, self.diskIdlType, $
      self.bytesPlane/8]
   planes = MAKE_ARRAY(size=idlSize)
;point to correct disk location
   pointLun = self.dataStart + (iPlane - 1) * self.bytesPlane
   point_lun, self.unit, pointLun
;read from disk
   READU, self.unit, planes
;update pointers
   self.planeCurrent = iPlane + nP -1
   self.dataCurrent  = pointLun + self.bytesPlane * nP
;convert to local 
  IEEE_TO_Host, planes
   iErr = 0
RETURN, planes
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (errMsg EQ '') then errMsg = !ERR_STRING+!SYSERR_STRING
   if (bottom EQ 1) then midiAddErrMsg,errMsg,/trace $
      else midiAddErrMsg
   midiPrintErrMain
   iErr = 1
RETURN, 0
END

PRO FITSIMAGE::writeImage, image, iErr=iErr
;write an entire image as a block after the header
;   INPUTS  image
;
   bottom = 1
   errMsg = ''
   ON_IOERROR, ERR_RET
;check status
   if (self.openStatus NE 'WRITE') then begin
      errMsg = 'Output file not opened with WRITE status'
      GOTO, ERR_RET_1
   endif
;determine size and type of input array
   sImage = SIZE(image)
   idlType = sImage(1+sImage(0))
   self.DiskIdlType = idlType
   CASE idlType OF
      1: begin     ; byte
         bitpix = 8
      END
      2: begin     ; short int
         bitpix = 16
      END
      3: begin     ; long int
         bitpix = 32
      END
      4: begin     ; float
         bitpix = -32
      END
      5: begin     ; double
         bitpix = -64
      END
   ENDCASE
;write FITS dimension keywords into local store
   byteSize = abs(bitpix)/8
   self.bitpix = bitpix
   self.priHead->addPar,'BITPIX', bitpix, iErr=iErr
   PTR_FREE, self.naxis
   self.naxis = PTR_NEW(sImage(1:sImage(0)))
   for iAxis = 0, sImage(0)-1 do byteSize = byteSize * (*self.naxis)[iAxis]
   self.priHead->addPar,'NAXIS', sImage(0), iErr=iErr
   for iAxis = 1, sImage(0) do self.priHead->addPar, $
      'NAXIS'+strTrim(string(iAxis),2), SImage(iAxis), iErr=iErr
;dump header to disk.  If you have already done this, overwrite it
   pointLun = 0L
   if (self.headerStatus NE '') then point_lun, self.unit, pointLun
   self.PriHead->WriteToDisk, self.unit, iErr=iErr
   if (iErr NE 0) then begin
      errMSG = 'writeImage: writing header failed'
      bottom = 0
      GOTO, ERR_RET
   endif
   point_lun, -self.unit, pointLun
   self.headerStatus = 'DISK'
   self.dataStart = pointLun
;write data to disk, with conversion to IEEE.  This may blow core if
;image is very big
   IF (idlType EQ 1) then WRITEU, self.unit, image else begin 
      newImage = image
      HOST_TO_IEEE, newImage
      WRITEU, self.unit, newImage
   endelse
   self.dataEnd   = self.dataStart + byteSize
   self.dataCurrent = self.dataEnd
   self.eof = self.dataEnd
   self->pad2880, rest, iErr=iErr
   if (iErr NE 0) then begin
      errMsg = 'writeImage: padding data failed'
      bottom = 0
      GOTO, ERR_RET
   endif
   self.dataStatus = 'DISK'
   iErr = 0
RETURN
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (errMsg EQ '') then errMsg = !ERR_STRING+!SYSERR_STRING
   if (bottom EQ 1) then midiAddErrMsg,errMsg,/trace else $
      midiAddErrMsg,errMsg
   midiPrintErrMain
   iErr = 1
RETURN
END

PRO FITSIMAGE::writePlanes, data, plane=plane, iErr=iErr
;write one or more 2-D planes to disk
;   INPUTS  data   the data
;           plane  which plane 1-relative. 
;                   If not given, append to end of image 
;
   bottom = 1
   errMsg = ''
   ON_IOERROR, ERR_RET
;check status
   if (self.openStatus NE 'WRITE') then begin
      errMsg = 'Output file not opened with WRITE status'
      GOTO, ERR_RET_1
   endif
   sPlane = SIZE(data)
   if ((sPlane[0] NE 2) AND (sPlane[0] NE 3))then begin
      errMsg = 'writePlanes only works on 2 or 3-D planes'
      GOTO, ERR_RET_1
   endif
   isNew = self.planeLast eq 0
   idlType = sPlane(1+sPlane[0])
   if (isNew) then begin
      CASE idlType OF
         1: begin     ; byte
            bitpix = 8
         END
         2: begin     ; short int
            bitpix = 16
         END
         3: begin     ; long int
            bitpix = 32
         END
         4: begin     ; float
            bitpix = -32
         END
         5: begin     ; double
            bitpix = -64
         END
      ENDCASE
      self.DiskIdlType = idlType
      self.bitpix = bitpix
      self.priHead->addPar,'BITPIX', bitpix, iErr=iErr
      byteSize = abs(bitpix)/8
      PTR_FREE,self.naxis
      self.naxis = PTR_NEW(lonarr(3))
      (*self.naxis) = [sPlane(1), sPlane(2), 0]
      self.priHead->addPar, 'NAXIS',3, iErr=iErr
      self.priHead->addPar, 'NAXIS1',sPlane(1), iErr=iErr
      self.priHead->addPar, 'NAXIS2',sPlane(2), iErr=iErr
      self.priHead->addPar, 'NAXIS3',1, iErr=iErr
      self.bytesPlane = byteSize * sPlane(1) * sPlane(2)
;reset file position to head
      pointLun = 0L
      point_lun, self.unit, pointLun
;(re)write header
      bottom = 0
      self.priHead->writeToDisk, self.unit, iErr=iErr
      self.headerStatus = 'DISK'
      point_lun, -self.unit, pointLun
      self.dataStart = pointLun
      self.dataCurrent = self.dataStart
      self.planeCurrent = 1
   endif else begin               ; isNew data
      if (self.DiskIdlType NE idlType) then begin
         errMsg = 'Data type of last plane changed from '+ $
	    string(self.DiskIdlType) 
         GOTO, ERR_RET_1
      endif
      if ((*self.naxis)[0] NE sPlane(1)) OR $
         ((*self.naxis)[1] NE sPlane(2)) then begin
         errMsg = 'Dimensions of Plane do not match earlier ones '
	 GOTO, ERR_RET_1
      endif
   endelse
;check whether specified Planes are in range.  If not specified set to
;end of image
   if (KEYWORD_SET(Plane)) then if (Plane LT 1 OR $
      (Plane GT self.planeLast + 1)) then begin
         errMsg = 'Plane of '+strTrim(string(Plane),2)+$
	    ' not in valid range (1,'+strtrim(string(self.planeLast+1))+')'
         GOTO, ERR_RET_1
   endif 
   if (NOT KEYWORD_SET(Plane)) then Plane = self.planeLast + 1
;set lun pointer 
   if (Plane NE self.planeCurrent) then $
      point_lun, self.unit, self.dataStart + (Plane - 1)*self.bytesPlane
;write data
   if (idlType EQ 1) then WRITEU, self.unit, data else begin
      newPlanes = data
      HOST_TO_IEEE, newPlanes
      WRITEU, self.unit, newPlanes
   endelse
;update pointers
   if (sPlane[0] EQ 2) then nPlane = 1 else nPlane = sPlane[3]
   self.planeLast   = (self.planeCurrent+nPlane-1) > self.planeLast
   self.dataEnd    = self.dataStart + nPlane*self.planeLast*self.bytesPlane
   self.eof    = self.dataStart + self.planeLast * self.bytesPlane
;position of next read/write
   self.dataCurrent = (Plane+nPlane-1)*self.bytesPlane  
   self.planeCurrent = Plane +  nPlane ; position of next read/write
;note planes are counted 1-relative and databytes are 0-relative
   iErr = 0
RETURN
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (errMsg EQ '') then errMsg = !ERR_STRING+!SYSERR_STRING
   if (bottom EQ 1) then midiAddErrMsg,errMsg,/trace else $
      midiAddErrMsg, errMsg
   midiPrintErrMain
   iErr = 1
RETURN
END

PRO FITSIMAGE::writeFinish, iErr=iErr
;if we have been writing planes, reset NAXIS3 to the right number
;and pad data out to 2880 byte blocks
   ON_IOERROR, ERR_RET
   if(self.openStatus NE 'WRITE') then begin
      iErr = 0
      RETURN
   endif
;overwrite NAXIS3
;first in stored header
   self.priHead->addPar,'NAXIS3', self.planeLast, iErr=iErr
;now on disk
   pointer = long(5*80)
   n3 = strTrim(string(self.planeLast), 2)
   l3 = strLen(n3)
   newLine = 'NAXIS3  ='+STRING(REPLICATE(32B,21-L3))+N3+' /Number of rows'+$
      STRING(REPLICATE(32B,34))
   POINT_LUN, self.unit, pointer
   WRITEU, self.unit, newLine
   self->pad2880, iErr=iErr
   if (iErr eq 1) then GOTO, ERR_RET
   self.dataStatus = 'DISK'
   iErr = 0
RETURN
ERR_RET:
   errMsg = !ERR_STRING+!SYSERR_STRING
   midiAddErrMsg,errMsg,/trace
   midiPrintErrMain
   iErr = 1
RETURN
END

FUNCTION FITSIMAGE::naxis
;return the currently stored value of naxis
RETURN, *self.naxis
END

PRO FITSIMAGE::setNaxis, naxis
;update stored values of naxis
RETURN
END

PRO FITSIMAGE::addPar, name, value, iErr=iErr, comment=comment, $
   BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
;Pass through interface to add/modify keyword records to
;the extension header which is a member of the current extension
;
;add or modify a parameter in a fits extension header.
;this version works only before the header has been committed to disk
;
   if (self.HeaderStatus ne 'INTERNAL') then begin
      iErr = 1
      midiAddErrMsg, 'FITSEXTENSION: cant add new keywords if status ne LOCAL',/trace
      midiPrintErrMain
      RETURN
   endif
   (*self.head)->addPar,name, value, iErr=iErr, comment=COMMENT,$
      BEFORE=BEFORE, AFTER=AFTER, FORMAT=FORMAT
RETURN
END

FUNCTION FITSIMAGE::getPar, name, matches, comment=comments
;  Pass through interface to retrieve keyword records from
;  the internally stored extension header
RETURN,(*self.prihead)->getPar(name, matches, comment=comments)
END

PRO FITSIMAGE::close, iErr=iErr
;
   self->writeFinish, iErr=iErr
   self->FITSFILE::close, iErr=iErr
RETURN
END

