;*******************************************************************************
; File: mymidigui.pro
; Front-end for Heidelberg and Leiden MIDI data reduction
; software in order to support VLTI/MIDI science operations.
; Part of OYSTER (Christian Hummel)
;
; Compatibility: MIA+EWS-1.3 (midi_options.v=1), MIA+EWS-1.5
;
; Some MIDI info
; --------------
;
; Detector: 320 (along x) x 240 (along y) pixels
; Default read-out windows: GRISM 261,40, PRISM 171,41
;
; Issues:
; -------
; 
; From Olivier Chesneau: MIA does not edit bad photometry (e.g. due to
; clouds) yet, only the fringe data are edited.
;
; Differences between multiplythenadd and addthenmultiply: tradeoff
; between noise and systematics.
;
; Widths of masks different for good and bad photometry?
;
; Location of important utilities:
; --------------------------------
;
; midiChopImage: in fringeUtilties.pro (EWS)
; MIDIChopImageC: in utilities.pro (MIA), calls midiChopimg.c
; MIDISkySubImage: in utilities.pro (MIA)
;
; Change log for MIA+EWS routines (current for 1.4):
; --------------------------------------------------
;
; xmidispvisi_gui.pro
; pro xmidispvisi::gui, BLOCK=block, TRUECOLOR=truecolor, CHARSIZE=charsize, base ; CAH
; 
; chop_nod_disp.pro:
; common MidiOptions,midi_options,maskfile_mymidi,kappafile,la_bins ; CAH
; win=win-median(total(win,1)/x_size); added by CAH: baseline correction
; if midi_options.m then begin  ; CAH
;         fitmask,win,mtrace
;         sigma=mtrace.FWHM(0:x_size-1)/sqrt(2.*alog(2))
; endif else begin
; endelse	; CAH
; CHA: In the following line, we multiply 4.2 by 2, because this didn't work with GRISM
; bad_trace= where( abs(mtrace.trace-median(mtrace.trace[range])) ge 4.2*2, bad_count)
;
; mask_utilities.pro: 
; common MidiOptions,midi_options,maskfile_mymidi,kappafile,la_bins       ; CAH
; fitmask,/init,iFile=iFile,iWin=iWin       ; CAH
;
; idl/fbt/fbtTableGui.pro : 
; number of rows in gorgonzola (maxrow)
;
; Change log for MIA+EWS routines (current for 1.5):
; --------------------------------------------------
;
; xmidispvisi_gui.pro:
; pro xmidispvisi::gui, BLOCK=block, TRUECOLOR=truecolor, CHARSIZE=charsize, base ; CAH
;
; xmidispvisi.pro:
; xmidispvisi::photometry:
; if n_elements(phot) eq 1 and phot[0] le 0. then begin ; CAH
; 
; mask_utilities.pro: 
; miamask::init:
; return,1 ; CAH
; chop_image=chop_image(0)      ; CAH
; miamask::fit_window:
; common MidiOptions,midi_options,maskfile_mymidi,kappafile,la_bins       ; CAH
; win=win-median(total(win,1)/x_size); added by CAH: baseline correction
; if midi_options.m then begin  ; CAH
;         fitmidimask,win,mtrace
;         sigma=mtrace.FWHM(0:x_size-1)/sqrt(2.*alog(2))
; endif else begin
; endelse	; CAH
; CAH: In the following line, we multiply 4.2 by 2, because this didn't work with GRISM
; miamask::fit_spectra:
; fitmidimask,/init,iFile=iFile,iWin=iWin       ; CAH
;
; idl/fbt/fbtTableGui.pro:
; number of rows in gorgonzola (maxrow)
;
; chop_nod_disp.pro is no longer part of MIA, we copied the one from v1.4
; In chop_nod_disp.pro, the main program had been renamed "_obsolete", we
; added the same suffix to the other subroutines, i.e. chop_nod_disp_one_window_obsolete,
; to distinguish it from chop_nod_disp in source/proprietary/olivier.
; Another change to chop_nod_disp main program.
;
; oirGroupDelay.c: two changes to compute mean delay if FINITO is used.
; oirFormFringes.c: Walter's new version to enable no high-pass filter (if piezo not used)
;
; Block directory:
; ----------------
;
; Block 1: chop_images,do_photometry
; Block 2: gaussian2d,flatbg,xmdvavg,pix2lambda,
;	   alloc_midi_options,vlti_stationid,midi_magadu,
;	   read_frames,chop_cycles,chop_frames,
;	   analyze_acqframes,align_acqframes,
;	   analyze_phoframes,align_phoframes,
;	   midiacqimage,midiphoframe,
;	   photometry_jitter,jitter_power,jitter_binning,
;	   winmask,fitmidimask
; Block 3: cleanindex,cleanmidirawdir,analayzemidiconfigids,midiconfigids,
;	   fixobht,midirawdirchecksum,selectmidifiles,
;	   midiupdateconfig,midigenconfig,
;	   midistarids,midiscan,minrtsmask,miosetup,miokappa
;	   defaultrecipes
; Block 4: midigui_event,midigui_configid,midigui_options,files_event,mymidigui,
;	   midiacqgui_event,midiacqgui,midiacqgui2_event,
;	   midimaskfiles_event,midimaskgui_event,midimaskgui,
;	   midirecipes_event,midirecipesgui_event,midirecipesgui,
;	   midiphogui_event,midiphogui,midiphogui2_event,
;	   miditrackgui_event,miditrackgui,
;	   initializepipe,mymidipipe,
;	   leidenpipe,mykappapipe,myacqpipe,myphopipe
; Block 5: photratios,fluxratios,photmovie,framemovie,displaychoppedspectrum
;
;************************************************************************Block 1
; In this first block, we copied code from MIA+EWS to make it usable
; without having to create the objects that go with them.
; For copyright terms, see the MIA+EWS software.
;
function chop_images,fname
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
; make once and for all sure that fname is a scalar
  fname = fname[0]
  hdrname= (oirFileList(fname))[0]

chop_image=MIDIChopImageC(fname,before=midi_options.before,after=midi_options.after)
;
; Code extracted from chop_nod_disp
;
      im = obj_new('imagedata', hdrname)
      he = im->prihead()
      beam = he->getpar('INS OPT1')
      telescope = he->getpar('INS SHUT')
      disp = he->getpar('INS GRIS')
      obj_destroy,im

      if (beam EQ 'HIGH_SENS') or (beam EQ 'OPEN    ') then begin
          print,"high_sens, using data1 & 2"
          on_source_win1 = chop_image[0].data1
          on_source_win2 = chop_image[0].data2
          off_source_win1= chop_image[1].data1
          off_source_win2= chop_image[1].data2
      endif else if (beam EQ 'SCI_PHOT') then begin
          print,"sci_phot, using data2 & 3"
          on_source_win1 = chop_image[0].data2
          on_source_win2 = chop_image[0].data3
          on_source_win3 = chop_image[0].data1
          on_source_win4 = chop_image[0].data4
          off_source_win1= chop_image[1].data2
          off_source_win2= chop_image[1].data3
          off_source_win3= chop_image[1].data1
          off_source_win4= chop_image[1].data4
          if (telescope EQ 'AOPEN   ') then begin
          print,'telescope A'
                on_source_winphot = chop_image[0].data4
                off_source_winphot = chop_image[1].data4
          endif
          if (telescope EQ 'BOPEN   ') then begin
          print,'telescope  B'
                on_source_winphot = chop_image[0].data1
                off_source_winphot = chop_image[1].data1
          endif

      endif else begin
          print,"INS OPT1 is",beam
          print,"Neither HIGH_SENS, OPEN, nor SCI_PHOT, don't know where my data are!"
          return,-1
      endelse
      if keyword_set(reverse) then begin
          win1 = off_source_win1-on_source_win1
          win2 = off_source_win2-on_source_win2
 if (beam EQ 'SCI_PHOT') then winphot = off_source_winphot-on_source_winphot
      endif else begin
          win1 = on_source_win1-off_source_win1
          win2 = on_source_win2-off_source_win2
  if (beam EQ 'SCI_PHOT') then winphot = on_source_winphot-off_source_winphot
      endelse
;
if (beam EQ 'SCI_PHOT') then $
return,{on1:on_source_win1,off1:off_source_win1, $
        on2:on_source_win2,off2:off_source_win2, $
        on3:on_source_win3,off3:off_source_win3, $
        on4:on_source_win4,off4:off_source_win4}
return,{on1:on_source_win1,off1:off_source_win1, $
        on2:on_source_win2,off2:off_source_win2}
;
end
;-------------------------------------------------------------------------------
PRO do_photometry, photfile, maskdata, PhFilt, spec1,spec2,specPh,$
        	PHOTFILEB=photfileB, SCI_MODE=sci_mode, $
		BEFORE=before, AFTER=after, ERRORLOG=errorlog

  if not keyword_set(before)  then before  = 1
  if not keyword_set(after)   then after   = 1
  if not keyword_set(sci_mode) then sci_mode= 0

  im = obj_new('fitsfile',(oirFileList(photfile))[0])
  he = im->prihead()
  shutter= strcompress(he->getpar('INS SHUT NAME'),/remove_all)
  PhFilt = strcompress(he->getpar('INS FILT NAME'),/remove_all)
  Grism  = strcompress(he->getpar('INS GRIS NAME'),/remove_all)
  DIT    = strcompress(he->getpar('DET DIT'),/remove_all)
  obj_destroy,im

  if Grism eq 'OPEN' then $
    print,"This is undispersed data.  I'm not sure I can handle that..."

  if not file_test("PHOTOMETRY") then file_mkdir,"PHOTOMETRY"

  print, 'Reading Chopping data...'
  print, photfile
  chop_image= MIDISkySubImage(photfile, before=before, after=after, ERRORLOG=errorlog)

  if sci_mode then begin   ;;; by FP
          win1 = chop_image.data1
          win2 = chop_image.data4
	  nWin = 4	; CAH
  endif else begin
      win1 = chop_image.data1
      win2 = chop_image.data2
      nWin = 2
      if (where(Tag_Names(chop_image) eq "DATA4") ne -1) then begin
          nWin = 4
          win3 = chop_image.data3
          win4 = chop_image.data4
      endif
  endelse

  if keyword_set(photfileB) then begin
      print, photfileB
      chop_image= MIDISkySubImage(photfileB, before=before, after=after, ERRORLOG=errorlog)
      win1 = sqrt(win1 * chop_image.data1 > 0.)
      win2 = sqrt(win2 * chop_image.data2 > 0.)
      if nWin ge 4 then begin
          ;; NOTE: win1 and win4 are not useful in this case!
          win3 = sqrt(win3 * chop_image.data3 > 0.)
          win4 = sqrt(win4 * chop_image.data4 > 0.)
      endif
  endif

  if sci_mode then begin   ;;; by FP
      spec1 = total(win1 * maskdata.data1, 2) / DIT
      spec2 = total(win2 * maskdata.data4, 2) / DIT
  endif else begin
      if nWin eq 2 then begin
          spec1 = total(win1 * maskdata.data1, 2) / DIT
          spec2 = total(win2 * maskdata.data2, 2) / DIT
      endif else begin
          spec1 = total(win2 * maskdata.data2, 2) / DIT
          spec2 = total(win3 * maskdata.data3, 2) / DIT
          if (shutter eq 'AOPEN') then begin
              specPh= total(win4 * maskdata.data4, 2) / DIT
          endif else begin
              specPh= total(win1 * maskdata.data1, 2) / DIT
          endelse
      endelse
  endelse
; CAH  print,"total flux 1:",total(spec1),",  total flux 2:",total(spec2)
; CAH  if nWin eq 4 then print,"sciphot flux:",total(specPh)
end
;************************************************************************Block 2
function tracefit,x,win,i,p
;
; Expected position of spectrum
wlp=total(win,1)
x_p=where(wlp eq max(wlp))
p=[max(wlp),x_p,2.4]
r=curvefit(x,wlp,wlp*0+1,p,function_name='funct_gauss',/noderiv)
x_p=p(1)
w_p=p(2)
;
p=[max(win(i,*)),x_p,w_p]
r=curvefit(x,win(i,*),wlp*0+1,p,function_name='funct_gauss', $
	fita=[1,1,0],/noderiv)
p(2)=p(2)/(2.0*sqrt(2.0*alog(2.0)))
;
return,r
;
end
;-------------------------------------------------------------------------------
function gaussian2d,nx,ny,a,b,theta
;
; Return image of two-dim. Gaussian with FWHM axes a,b, and position angle theta.
;
RAD=180/!pi
if n_elements(theta) eq 0 then theta=0.0
;
image=fltarr(nx,ny)
x=(findgen(nx*ny) mod nx) - (nx/2)
y=(findgen(nx*ny) / nx) - (ny/2)
r=reform(sqrt(x^2+y^2),nx,ny)
t=reform(atan(y,x)-(theta+90)/RAD,nx,ny)
return,exp(-((r*cos(t))/(a/2))^2)*exp(-((r*sin(t))/(b/2))^2)
;
end
;-------------------------------------------------------------------------------
function flatbg,image0,x0,y0,width
;
image=image0
;
imsize=[n_elements(image(*,0)),n_elements(image(0,*))]
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0))
y=(lindgen(imsize(0)*imsize(1)) / imsize(0))
r=sqrt((x-x0)^2+(y-y0)^2)
offset=median(image(where(r gt width and r lt width*2)))
image(where(r gt width))=offset
;
return,image
;
end
;-------------------------------------------------------------------------------
function xmdvavg,x,bins,z
;
; Executes an average according to the Heidelberg binning system
; Bins are defined as follows: start,end,step,width, where start=1 for
; the first pixel. The number of bins is computed so that the last pixel,
; end, is in the last bin, and no higher pixel values. That means this
; last bin can be an average over less than <width> pixels.
;
n=fix((bins(1)-bins(0)+1)/bins(2))
if n*bins(2)+bins(0)-1+(bins(3)-bins(2)) lt bins(1) then n=n+1
y=x
y=y(0:n-1)
z=y
for l=0,n-1 do begin
	lolam=bins(0)+bins(2)*l-1
	hilam=lolam+bins(3)-1 < bins(1)
	y(l)=total(x(lolam:hilam))/(hilam-lolam+1)
	z(l)=abs(x(hilam)-x(lolam))
endfor
return,y
;
end
;-------------------------------------------------------------------------------
function pix2lambda,x,dispersion
;
; Converts pixel numbers to wavelength for PRISM and GRISM configurations.
; First pixel is numbered 1.
;
; lambda=C2 X^2 + C1 X + C0
; Numbers from MIDI User Manual
;
grism=reverse([-1.21122e-6,0.0231223,7.54423])
grism=reverse([-1.28959e-6,0.0229380,6.68654]) ; From fit to oirGetWaveLength
prism=reverse([-0.0001539,0.009491,15.451905])
prism=reverse([-0.0001812,0.021967,13.976937]) ; From fit to oirGetWaveLength
;
case strupcase(dispersion) of
	'PRISM':c=prism
	'GRISM':c=grism
endcase
;
return,poly(x,c)
;
end
;-------------------------------------------------------------------------------
function alloc_midi_options
;
return,{s:99,c:0,r:0,j:0,m:0,v:0,p:0,f:0, $
        width:1.0,t_order:2,w_order:1,before:0,after:2}
;
end
;-------------------------------------------------------------------------------
function vlti_stationid,stations
;
stations_1=strtrim(stations,2)
;
for i=0,n_elements(stations)-1 do begin
if strmid(stations(i),0,1) eq 'U' $
	then stations_1(i)='UT'+strmid(stations(i),1,1) $
	else stations_1(i)='A'+strmid(stations(i),0,2)
endfor
;
return,stations_1
;
end
;-------------------------------------------------------------------------------
function midi_magadu,station,filter
;
; Given counts from the acquisition image, convert to Jy. The zeropoints given
; below are always derived using the IRAS 12 micron fluxes. Therefore, with
; the MIDI filters, some could produce systematic errors due to the varying color
; of the targets.
;
; Only return approximate values.
;
; First used for atv photometry, where zeropoint for UTs with N8.7 is 11.0.
;					 	 for ATs with N8.7 is 7.4	
; Now should be used only with chop_nod_phot photometry.
;
; Historical records for y (N = y - 2.5*alog10(chop_nod_phot counts))
;
; UTs:
; 2005-05-30: UT2: N8.7 16.3+/-0.6,                  UT4: N8.7 16.9+/-0.1
; 2005-07-21: UT1: N8.7 16.3+/-0.5, Nband 17.1+/-0.1 UT3: N8.7 16.8+/-0.2, Nband 17.5+/-0.2
; 2005-07-23: UT2: N8.7 16.4+/-0.2, Nband 16.8+/-0.3 UT3: N8.7 17.0+/-0.1, Nband 17.6+/-0.2
; 2005-11-13: UT2: N8.7 15.1+/-0.2, SiC   14.3+/-0.0 UT3: N8.7 15.6+/-0.3, SiC   14.8+/-0.0
; 2006-06-10: UT3: N8.7 16.4+/-0.3, SIC   15.8, OPEN 17.0+/-0.3
; 2006-06-10: UT4: N8.7 16.1+/-0.5, SIC   16.7, OPEN 16.6+/-0.4
; 2006-06-12: UT3: N8.7 16.1+/-0.2
; 2006-06-12: UT2: N8.7 16.6+/-0.2
;
; ATs:
; 2006-02-22: AE0: N8.7 13.0+/-0.2 AG0: N8.7 12.8+/-0.1
; 2006-06-16: N8.7 13.6 OPEN 14.1
;
case filter of
	'N8.7'	: magadu=16.3
	'Nband'	: magadu=17.5
	'SiC'	: magadu=15.5	; w/HD152820 on 2006-05-14
	else	: magadu=11.0
endcase
;
if strmid(station,0,1) eq 'A' then magadu=magadu-3.0
;
return,magadu
;
end
;-------------------------------------------------------------------------------
function read_frames,filename,channel,targets,t_int
;
; Read all data frames from a MIDI file.
;
fitsfiles=nameparse(filename)
nfiles=n_elements(fitsfiles)
;
; See how many frames in total we need to allocate space for
nrows=intarr(nfiles)
mrows=10
for ifile=0,nfiles-1 do begin
	imageObj=obj_new('imagedata', fitsfiles(ifile), ierr=ierr)
	naxis=imageObj->naxis()
	nrows(ifile)=naxis(1)
	if nrows(ifile) mod mrows ne 0 then begin
		print,'***Error(READ_FRAMES): must be multiple of', mrows,' rows!'
		return,0
	endif
	obj_destroy,imageObj
endfor
nrows_all=long(total(nrows))
;
for ifile=0,nfiles-1 do begin
	print,'Reading file: ',fitsfiles(ifile)
	imageObj=obj_new('imagedata', fitsfiles(ifile), ierr=ierr)
	dataColumns=imageObj->dataColumns()
	targetColumns=imageObj->targetColumns()
	readColumns=[dataColumns,targetColumns]
	if n_elements(frames) eq 0 then begin
		d=imageObj->readRows(1,col=readColumns,ierr=ierr)
		f=d.data1
		r=size(f)
		nx=r(1)
		ny=r(2)
		tags=tag_names(d)
		frames=fltarr(nx,ny,nrows_all,/nozero)
		targets=strarr(nrows_all)
	endif
	row2=fix(total(nrows(0:ifile)))-1
	row1=row2-nrows(ifile)+1
	for i=0,nrows(ifile)/mrows-1 do begin
		d=imageObj->readRows(indgen(mrows)+1+i*mrows,col=readColumns,ierr=ierr)
		case channel of
			1:f=d.data1
			2:f=d.data2
			3:f=d.data3
			4:f=d.data4
		endcase
		f=float(f)+32768.
		frames(*,*,row1+i*mrows:row1+(i+1)*mrows-1)=f
		targets(row1+i*mrows:row1+(i+1)*mrows-1)=d.tartyp2
	endfor
	obj_destroy,imageObj
;
endfor
;
print,'Number of frames read: ',nrows_all
;
; Read chopping frequency and compute sampling interval
fitsfile=obj_new('fitsfile',fitsfiles(0))
prihead=fitsfile->prihead()
chopfreq=strtrim(prihead->getpar('ISS CHOP FREQ'))
print,'Chopping frequency [Hz]: ',chopfreq
obj_destroy,fitsfile
index=where(targets eq 'T ')
n=n_elements(index)
gindex=index(1:n-1)-index(0:n-2)
on=where(gindex gt 1,n_on)
n_on=n_on+1
if chopfreq ne 0 then t_on=0.5/chopfreq else t_on=0
t_int=n_on*2*t_on/n_elements(targets)
;
return,frames
;
end
;-------------------------------------------------------------------------------
function chop_cycles,t
;
; Using U,T,S tags in "t", analyze how many complete chopping cycles were
; done, how long they are, and where they start. Return cycle ID for each
; tag "t".
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
n=n_elements(t)
;
; Do Fourier analysis
y=fltarr(n)
y(where(t eq 'T '))=+1
y(where(t eq 'S '))=-1
rms=powerspectrum(poweroftwo(y),1.,f,p,f_avg,p_avg)
index=where(p eq max(p))
lc0=1/f(index(0))
lc=lc0
;
; Estimate of cycle length by wrapping
s=fltarr(n)
s(where(t eq 'T '))=1
imin=fix(0.9*lc)
imax=fix(1.1*lc)
c=fltarr(imax+1)
for i=imin,imax do begin
	m=reform(s(0:(n/i)*i-1),i,n/i)
	f=fltarr(i)
	for j=0,i-1 do f(j)=produkt(m(j,0:(n/i-1)<(fix(lc)/2)))
	c(i)=avg(f)
endfor
lc1=where(c eq max(c)) & lc1=float(lc1(0))
lc=lc1
;
; We use the average, because we have seen either being worse
lc=(lc0+lc1)/2
;
; Adjust cycle length by "hitting" U frames
nc=fix(n/lc)
ni=100
d=abs(lc1-lc0)*2.0/ni
c=fltarr(ni+1)
j=intarr(ni+1)
lci=fltarr(ni+1)
index=where(t eq 'U ')
for i=0,ni do begin
	lci(i)=lc+(i-ni/2)*d
	tags0=t((nint(indgen(nc)*lci(i))+index(0)-1) > 0)
	tags1=t((nint(indgen(nc)*lci(i))+index(0)+0) > 0)
	tags2=t((nint(indgen(nc)*lci(i))+index(0)+1) > 0)
	index0=where(tags0 eq 'U ',count0)
	index1=where(tags1 eq 'U ',count1)
	index2=where(tags2 eq 'U ',count2)
	counts=[count0,count1,count2]
	c(i)=max(counts)
	k=where(counts eq c(i)) & k=k(0)
	j(i)=k-1
endfor
k=where(c eq max(c)) & k=k(0)
lc=lci(k)
nc=fix(n/lc)
i0=j(k)
print,'Chopping "Strehl": ',max(c)/nc
;
n_t=0
n_s=0
for i=0,nc-1 do begin
	index_t=where(t(i0+nint(i*lc)>0:i0+nint((i+1)*lc)-1<(n-1)) eq 'T ',count_t)
	index_s=where(t(i0+nint(i*lc)>0:i0+nint((i+1)*lc)-1<(n-1)) eq 'S ',count_s)
	n_t=n_t+count_t
	n_s=n_s+count_s
endfor
n_t=n_t/nc
n_s=n_s/nc
print,'Average target frames per cycle: ',n_t
print,'Average sky frames per cycle: ',n_s
print,'Number of complete cycles: ',nc
print,'First cycle starts at: ',i0
;
cycle_id=intarr(n)
for i=0,nc-1 do cycle_id(i0+nint(i*lc)+midi_options.after:i0+nint((i+1)*lc<(n-1))-1)=i+1
return,cycle_id
;
end
;-------------------------------------------------------------------------------
function chop_frames,frames,targets,cycles
;
nc=n_elements(unique(cycles(where(cycles ne 0))))
;
frames_c=frames
;
for i=0,nc-1 do begin
	index_t=where(cycles eq i+1 and targets eq 'T ',count_t)
	index_s=where(cycles eq i+1 and targets eq 'S ',count_s)
;	Determine mean sky for this cycle
        if count_s eq 1 then sky=frames(*,*,index_s(0)) $
                        else sky=total(frames(*,*,index_s),3)/count_s
;	Subtract mean sky from all frames of this cycle
	index=where(cycles eq i+1,count)
	for j=0,count-1 do frames_c(*,*,index(j))=frames(*,*,index(j))-sky
;
endfor
;
return,frames_c
;
end
;-------------------------------------------------------------------------------
function analyze_acqframes,frames,targets,cycles,a,b,x,y,xmed,ymed,xerr,yerr
;
n=n_elements(targets)
f=fltarr(n)
a=fltarr(n)
b=fltarr(n)
x=fltarr(n)
y=fltarr(n)
;
index_t=where(cycles ne 0 and targets eq 'T ',count_t)
;
image=total(frames(*,*,index_t),3)/count_t
parms0=[0,max(image),1.2,1.2,30,30,0]
;
for i=0,count_t-1 do begin
	parms=parms0
	parms(1)=max(frames(*,*,index_t(i)))
	r=gauss2dfit(frames(*,*,index_t(i)),parms)
	f(index_t(i))=parms(1)
	a(index_t(i))=parms(2)
	b(index_t(i))=parms(3)
	x(index_t(i))=parms(4)
	y(index_t(i))=parms(5)
endfor
;
xmed=medianve(x(index_t),xerr)
ymed=medianve(y(index_t),yerr)
print,'Median position at: ',xmed,ymed
;
return,f
;
end
;-------------------------------------------------------------------------------
function align_acqframes,frames,targets,cycles,x,y,xmed,ymed
;
n=n_elements(targets)
;
frames_c=frames
;
index_t=where(cycles ne 0 and targets eq 'T ',count_t)
;
for i=0,count_t-1 do begin
	dx=xmed-x(index_t(i))
	dy=ymed-y(index_t(i))
	frames_c(*,*,index_t(i))=shift(frames(*,*,index_t(i)),nint(dx),nint(dy))
endfor
;
return,frames_c
;
end
;-------------------------------------------------------------------------------
function analyze_phoframes,frames,targets,cycles,x,a,xmed
;
r=size(frames)
nx=r(1)
ny=r(2)
n=r(3)
ncol=20
x0=120
column=findgen(ny)
p=fltarr(3)
;
f=fltarr(n)
x=fltarr(n)
a=fltarr(n)
;
index_t=where(cycles ne 0 and targets eq 'T ',count_t)
;
; Expected position of spectrum
wlp=total(total(frames(*,*,index_t),1),2)
x_p=where(wlp eq max(wlp))
p=[max(wlp),x_p,2.4]
r=curvefit(column,wlp,wlp*0+1,p,function_name='funct_gauss',/noderiv)
x_p=p(1)
w_p=p(2)
;
for i=0,count_t-1 do begin
	wlp=reform(total(frames(x0-ncol:x0+ncol,*,index_t(i)),1))/(2*ncol+1)
	base=median(wlp)
	wlpe=wlp-base
	p=[max(wlpe),x_p,w_p]
	r=curvefit(column,wlpe,wlpe*0+1,p,function_name='funct_gauss', $
		fita=[1,1,0],/noderiv)
;	r=gaussfit(column,wlpe,p,nterm=3) ; This one has problems in gaussfit.pro
	if abs(p(1)-x_p) lt 10 then begin
		f(index_t(i))=p(0)+base
		x(index_t(i))=p(1)
		a(index_t(i))=p(2)
	endif
endfor
;
xmed=median(x(index_t))
print,'Median position at: ',xmed
;
return,f
;
end
;-------------------------------------------------------------------------------
function align_phoframes,frames,targets,cycles,x,xmed
;
n=n_elements(targets)
;
frames_c=frames
;
index_t=where(cycles ne 0 and targets eq 'T ',count_t)
;
for i=0,count_t-1 do begin
	dx=xmed-x(index_t(i))
	frames_c(*,*,index_t(i))=shift(frames(*,*,index_t(i)),0,nint(dx))
endfor
;
return,frames_c
;
end
;-------------------------------------------------------------------------------
function midiacqimage,filename,channel,align=align,edit=edit
;
; Replace MIDIChopImageC, and return mean of chopped and aligned 
; acquisition frames.
;
if n_elements(align) eq 0 then align=0
if n_elements(edit) eq 0 then edit=0
;
rf=read_frames(filename,channel,t)
c=chop_cycles(t)
cf=chop_frames(rf,t,c)
f=analyze_acqframes(cf,t,c,a,b,x,y,xmed,ymed,xerr,yerr)
if edit then begin
	index=where(abs(x-xmed) lt xerr and abs(y-ymed) lt yerr,count)
	if count gt 0 then begin
		cf=cf(*,*,index)
		t=t(index)
		c=c(index)
		a=a(index)
		b=b(index)
		x=x(index)
		y=y(index)
	endif
endif
if align then af=align_acqframes(cf,t,c,x,y,xmed,ymed) else af=cf
index=where(t eq 'T ' and c ne 0,count)
return,total(af(*,*,index),3)/count
;
end
;-------------------------------------------------------------------------------
function photometry_jitter,frames
;
r=size(frames)
nx=r(1)
ny=r(2)
nrow=r(3)
wlp=total(frames,1)/nx
refsp=total(wlp,2)/nrow
;
nlag=5
lags=findgen(nlag)-nlag/2
corr=fltarr(nlag)
;
pos=fltarr(nrow)
;
for i=0,nrow-1 do begin
	for j=0,nlag-1 do begin
		lag=j-nlag/2
		corr(j)=total((wlp(10:25,i)-refsp(10+lag:25+lag))^2)
	endfor
	corr=corr/max(corr)
	r=poly_fit(lags,corr,2,yfit)
	pos(i)=-r(1)/(2*r(2))
endfor
;
return,pos
;
end
;-------------------------------------------------------------------------------
pro jitterpower,r,sampling_int
;
series=poweroftwo(r)
tp=powerspectrum(series,sampling_int,f,p,f_avg,p_avg)
plot,f_avg,alog10(p_avg),xrange=[f(1),1/(2*sampling_int)],xstyle=1,/xlog, $
	xtitle='Frequency [Hz]',psym=0
;	xgridstyle=1,xticklen=1,ygridstyle=1,yticklen=1
;
end
;-------------------------------------------------------------------------------
pro jitterbinning,frames,labels,onoff
;
r=size(frames)
nc=r(1)
nr=r(2)
nframe=r(3)
nsegmnt=5
nperseg=nframe/nsegmnt
nem=8
;
r=photometry_jitter(frames)
;
binsize=0.00
;
repeat begin
;
binsize=binsize+0.01
;
onoff0=fltarr(nc,nr)
on0=onoff0
off0=onoff0
num_onoff=0
num_on=0
num_off=0
;
for i=0,nsegmnt-1 do begin
	i1=i*nperseg
	i2=i1+nperseg-1
	rs=r(i1:i2)
	ls=labels(i1:i2)
	fs=frames(*,*,i1:i2)
;
	v=medianve(rs,e)
	nbin=fix(nem*e/binsize) > 1
;
	for j=0,nbin-1 do begin
		minv=v-(nem/2)*e+j*binsize
		maxv=minv+binsize
		index=where(rs ge minv and rs lt maxv and rs ne 0,count)
		if count gt 0 then begin
			lsb=ls(index)
			fsb=fs(*,*,index)
			on=where(lsb eq 'T ',count_on)
			off=where(lsb eq 'S ',count_off)
			if count_on gt 10 and count_off gt 10 then begin
				on_total=total(fsb(*,*,on),3)
				off_total=total(fsb(*,*,off),3)
				onoff0=onoff0 $
				      +(on_total/count_on-off_total/count_off)
				num_onoff=num_onoff+1
				on0=on0+on_total
				off0=off0+off_total
				num_on=num_on+count_on
				num_off=num_off+count_off
			endif
		endif
	endfor
endfor
;
endrep until num_onoff gt 1 or binsize gt 0.1
;
print,'Final binsize: ',binsize
print,'Number of ON-OFF frames averaged: ',num_onoff
;
onoff=onoff0/num_onoff
end
;-------------------------------------------------------------------------------
function winmask,win,a,lambda,tophat=tophat
;
; Given the parameters of the mask fit to the spectrum, return a mask
; a=mask parameters:
; a(0:2): Polynomial coeffs for centerline
; a(3): width
; a(4): reference pixel along dispersion direction
;
; lambda: wavelength along dispersion direction (first index in win) 
;
if n_elements(tophat) eq 0 then tophat=0
;
winm=win*0
;
r=size(win)
nx=r(1)
ny=r(2)
x=findgen(nx)
y=findgen(ny)
x0=a(4)
y0=poly(x-x0,a(0:2))
;
for j=0,nx-1 do begin
	p=[1,y0(j),a(3)*lambda(j)/lambda(fix(x0))]
	funct_gauss,y,p,f
	if tophat gt 0 then begin
		index=where(f/max(f) gt tophat,count)
		if count gt 0 then winm(j,index)=1
	endif else winm(j,*)=f
endfor
;
return,winm
;
end
;-------------------------------------------------------------------------------
pro fitmidimask,win,mtrace,init=init,iFile=iFile,iWin=iWin
;
common FitMask,lambda,x0,phot_file,det_window,coeffs,num_bad_col,num_bad_row
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MarquardtFit,fit_options
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
if n_elements(draw_wid) eq 0 then draw_wid=0l
if n_elements(midimask_wid) eq 0 then midimask_wid=0l
;
do_plot=0
if widget_info(midimask_wid,/valid) and widget_info(draw_wid,/valid) then do_plot=1
;
if n_elements(init) eq 0 then init=0
if init then begin
	if widget_info(midimask_wid,/valid) and widget_info(draw_wid,/valid) then begin
		widget_control,draw_wid,get_value=id
		wset,id
		insopt1=strtrim(obht(obsindex(0)).insopt1)
		if !p.multi(0) eq 0 then begin
		if insopt1 eq 'HIGH_SENS' then !p.multi=[0,2,2]
		if insopt1 eq 'SCI_PHOT' then !p.multi=[0,4,2]
		if insopt1 eq 'SCI_PHOT' then !p.charsize=1.5 else !p.charsize=1.0
		endif
	endif
	if n_elements(midi_options) eq 0 then midi_options=alloc_midi_options()
	if n_elements(obsindex) ne 0 then begin
	phot_file=iFile
	det_window=iWin
	dispersion=strtrim(obht(obsindex(0)).insgris)
	insopt1=strtrim(obht(obsindex(0)).insopt1)
	win1nx=detwin(0)
	win1strx=detwin(1)
	case dispersion of
	'PRISM':        begin
                	x0=120.	; Typical peak of emission
                	lambda=pix2lambda(findgen(win1nx)+win1strx,dispersion)
;			UTs
			coeffs=[[16.8,-0.0091,-0.000178, 3.4,x0], $
				[17.9, 0.011,  0.0000600,3.6,x0]]
			coeffs=[[17.4,-0.0040,-0.0000550,3.8,x0], $
				[18.4, 0.011,  0.0000624,3.7,x0]]
;			ATs
			coeffs=[[15.2,-0.0154, 0.0000000,2.7,x0], $
				[14.8,-0.0042, 0.0000000,2.7,x0], $
				[15.9, 0.005,  0.0000000,2.6,x0], $
				[16.1, 0.022,  0.0000000,2.6,x0]]
;			if insopt1 eq 'HIGH_SENS' then coeffs=coeffs(*,1:2)
;			Columns are oriented along the slit
			num_bad_col=[1,31]
			num_bad_row=[3,11]
                	end
	'GRISM':        begin
                	x0=50.	; Typical peak of emission
                	lambda=pix2lambda(findgen(win1nx)+win1strx,dispersion)
;			Typical coefficients, windows 1 & 2
			coeffs=[[14.4, 0.062,-0.000188, 3.1,x0], $
				[17.0,-0.004, 0.0000975,2.5,x0]]
			coeffs=[[15.2, 0.043,-0.0000773,4.2,x0], $
				[17.5,-0.005, 0.0000937,3.6,x0]]
                	x0=150.	; This works better since more central
			coeffs=[[12.8, 0.059,-0.0001263,4.2,x0], $
				[17.8,-0.014, 0.0000957,3.6,x0]]
			coeffs=[[ 9.5, 0.175,-0.00047,2.7,x0], $
				[11.4, 0.073,-0.00017,2.7,x0], $
				[18.1,-0.027, 0.00014,2.6,x0], $
				[23.7,-0.133, 0.00049,2.6,x0]]
;			Typical maskgui coefficients
			trace_coeffs=[[11.8, 0.066,-0.000133], $
				      [17.7,-0.014, 0.0000956]]
;			Columns are oriented along the slit
			num_bad_col=[2,2]
			num_bad_row=[2,11]
                	end
	endcase
	endif
	return
endif
;
r=size(win)
nx=r(1)
ny=r(2)
;
; Sanity check
if max(win) eq 0 then begin
	print,'***Error(FITMASK): zero frame!'
	return
endif
;
; Prepare zero weight mask
win_weight=win*0+1
win_weight(0:num_bad_col(0)-1,*)=0
win_weight(nx-1-num_bad_col(1)+1:nx-1,*)=0
win_weight(*,0:num_bad_row(0)-1)=0
win_weight(*,ny-1-num_bad_row(1)+1:ny-1)=0
;
; Find target in white light column, x0 +/- ncol
column=findgen(ny)
ncol=20
wlp=reform(total(win(x0-ncol:x0+ncol,*),1))/(2*ncol+1)
if do_plot then  plot,wlp,ytitle='White light profile', $
	title='File '+string(phot_file+1,format='(i1)') $
	     +', Detector window '+string(det_window+1,format='(i1)')
gp=fltarr(3)
gp(0)=max(wlp)
gp(1)=where(wlp eq max(wlp))
; FWHM @ 8.7 microns = 3.1 pixels (UT)
; FWHM=2*sqrt(2*alog(2))*gp(2)
FWHM=3.1
gp(2)=FWHM/(2*sqrt(2*alog(2)))
wlpe=wlp
wlpe(0:fix(gp(1)-FWHM*2))=0
wlpe(fix(gp(1)+FWHM*2):ny-1)=0
r=gaussfit(column,wlpe,gp,nterm=3)
index=where(r/max(r) lt 0.01)
win_masked=win
for i=0,nx-1 do win_masked(i,index)=0
;
; Prepare data arrays
ndata=n_elements(win)
x=findgen(ndata)
y=reform(win_masked,ndata)
;
; Estimate a level for noise and systematic error
coeffs_e=coeffs
a=coeffs_e(*,det_window)
a(0)=gp(1)
a(1)=gp(2)*2*sqrt(2*alog(2))
win_bcgnd=win*(1-winmask(win,a,lambda,tophat=0.0001))
bcgnd=reform(win_bcgnd,ndata)
weight=reform(win_weight,ndata)
index=where(weight gt 0)
bias=mean(bcgnd(index))
noise=stddev(bcgnd(index))
print,'--------------------------------------------------'
print,'Mean background=',bias,', noise=',noise
print,'--------------------------------------------------'
y=y-bias
;
; Locate significant flux
a=coeffs_e(*,det_window)
a(0)=gp(1)
r=mask2pix(y,lambda,a,flux=flux)
; if do_plot then plot,lambda,flux,title='Detector window '+string(det_window+1), $
; 	xtitle='Wavelength'
;
; Do some windowing
; win=win*winmask(win,a,lambda*0+1,tophat=0.1)
; y=reform(win,ndata)
;
; Prepare sig array
sig=sqrt(y>0) > noise
index=where(flux lt 2*noise)
win_weight(index,*)=0
weight=reform(win_weight,ndata)
index=where(weight eq 0)
sig(index)=100*noise
;
; Prepare mask parameter vector a
a=[gp(1),0,0,gp(2)*2*sqrt(2*alog(2)),x0]
ma=n_elements(a)
ia=intarr(ma)+1 & ia(ma-1)=0	; We don't fit x0
dispersion=strtrim(obht(obsindex(0)).insgris)
if dispersion eq 'PRISM' then ia(2)=0	; Fit only linear terms
;
; Prepare call to marquardt
r=init_marquardtfit()
fit_options.tolerance=1e-8
if dispersion eq 'GRISM' then fit_options.tolerance=1e-9
fit_options.chifr=0.001
marquardt,'maskfuncs',x,y,sig,ndata,a,ia,ma,chisq,covar
print,a
r=mask2pix(y,lambda,a,flux=flux)
;
; Store the results in structure provided by chop_nod_disp_one_window
; mask=winmask(win,a,lambda,lambda)
; flux=total(win*mask,2)
;
x=findgen(nx)
x0=a(4)
y0=poly(x-x0,a(0:2))
mtrace.trace(0:nx-1)=y0
mtrace.FWHM(0:nx-1)=(a(3)*lambda/lambda(fix(x0)))/2
mtrace.fluxmax(0:nx-1)=flux
;
end
;-------------------------------------------------------------------------------
pro writemidimask,detectorfile,mask
;
; e.g. mask={data1:fltarr(171,41),data2:fltarr(171,41)}
; The detectorfile is used to copy the detector info from.
;
common FitMask,lambda,x0,phot_file,det_window,coeffs,num_bad_col,num_bad_row
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
maskfile=detectorfile
strput,maskfile,'Mask_'
maskfile='PHOTOMETRY/'+maskfile
;
oirNewData,detectorfile,maskfile,mask
;
end
;************************************************************************Block 3
function cleanindex,obs
;
; This function will return the indices of files corresponding
; to data of the first full observation in the observation 
; array given to it.
;
; Identify the first block containing a fringe track record
;
n=n_elements(obs)
;
i=where(obs.m eq 'OBS_FRINGE_TRACK_FOURIER' $
     or obs.m eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
     or obs.m eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
     or obs.m eq 'OBS_FRINGE_TRACK_DISPERSED',c)
if c eq 0 then return,-2	; No more data
; This is the target we will consider
target=obs(i(0)).t
insopt=obs(i(0)).o
i0=0
if c gt 1 then begin
	i1=i(1)-1 
	while i(1)-i(0) eq 1 and insopt ne 'SCI_PHOT' do begin
;		We have one or more repeated fringe tracks here
;		If they are on the same target, take the second one,
;		if the targets are different, the first observation
;		is broken.
		target2=obs(i(1)).t
		if target2 ne target then begin
			obs=obs(i(1)-1:n-1)
			return,-1
		endif
		if i(0) gt i0 then obs=[obs(i0:i(0)-1),obs(i(1):n-1)] $
			      else obs=obs(i(1):n-1)
		n=n_elements(obs)
		i=where(obs.m eq 'OBS_FRINGE_TRACK_FOURIER' $
		     or obs.m eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
     		     or obs.m eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
     		     or obs.m eq 'OBS_FRINGE_TRACK_DISPERSED',c)
		if c eq 1 then begin
			       i=[i,i] 
			       i1=n_elements(obs)-1
			       endif else i1=i(1)-1
	endwhile
endif else i1=n_elements(obs)-1
n=n_elements(obs)
obs0=obs(i0:i1)				; This part contains our observation
; To be processed in the next call
n0=n_elements(obs0)
i=0
while obs0(n0-1-i).m eq 'ACQ_UT_COARSE_CHOP' $
   or obs0(n0-1-i).m eq 'OBS_FRINGE_SEARCH_DISPERSED' $
   or obs0(n0-1-i).m eq 'OBS_FRINGE_SEARCH_DISPERSED_FAST' $
   or obs0(n0-1-i).m eq 'OBS_FRINGE_SEARCH_FOURIER' do i=i+1
i1=i1+1-i
if n-1 ge i1 then obs=obs(i1:n-1) $
	     else begin
		  obs=obs(0)
		  obs.m=''
	     endelse
; Only select observations of this target (the ACQ at the end is of the next OB)
obs0=obs0(0:n0-1-i)
; Now determine the locations of the templates we need
index=where(obs0.m eq 'OBS_FRINGE_TRACK_FOURIER' $
         or obs0.m eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
         or obs0.m eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
         or obs0.m eq 'OBS_FRINGE_TRACK_DISPERSED')
index=index(0)
; Select the last acquisition template
j=where(obs0.m eq 'ACQ_UT_COARSE_CHOP',c)
if c gt 0 then index=[index,j(c-1)]
; For the photometry, select the last ones for both A and B
k=where(obs0.m eq 'DEFAULT_CHOP' and obs0.s eq 'AOPEN' and obs0.t eq target,c)
if c eq 0 and insopt ne 'SCI_PHOT' then return,-1
if c gt 0 then index=[index,k(c-1)]
l=where(obs0.m eq 'DEFAULT_CHOP' and obs0.s eq 'BOPEN' and obs0.t eq target,c)
if c eq 0 and insopt ne 'SCI_PHOT' then return,-1
if c gt 0 then index=[index,l(c-1)]
return,obs0(index).i
;
end
;-------------------------------------------------------------------------------
pro cleanmidirawdir,files
;
if n_elements(files) eq 0 then files=''
if strlen(files(0)) gt 0 then $
ob=obj_new('fitsfilelist',files,/midiCompress) else $
ob=obj_new('fitsfilelist',searchString='MIDI*.fits',/midiCompress)
hk=ob->getHeaderKeys(['NRTS MODE','OBS TARG NAME','INS SHUT','INS OPT1'])
n=n_elements(hk)
i=indgen(n)
mindex=intarr(n)
f=ob->files()
obj_destroy,ob
m=hk.nrtsmode
s=strcompress(hk.insshut,/remove_all)
t=strcompress(hk.obstargname,/remove_all)
o=strcompress(hk.insopt1,/remove_all)
obs={i:0,f:'',m:'',s:'',t:'',o:''}
obs=replicate(obs,n)
obs.i=i
obs.f=f
obs.m=m
obs.s=s
obs.t=t
obs.o=o
obs0=obs
obs=obs(where(strpos(obs.m,'SEARCH') eq -1))
;
repeat begin
	r=cleanindex(obs)
	if r(0) ge 0 then begin
		if n_elements(index) eq 0 then index=r $
					  else index=[index,r]
	endif
endrep until r(0) eq -2
;
mindex(index)=1
spawn,'mkdir -p moremididata'
for i=0,n-1 do begin
	if mindex(i) eq 0 then spawn,'mv '+obs0(i).f+' moremididata'
endfor
;
end
;-------------------------------------------------------------------------------
function analyzemidiconfigids
;
; Determine how many different configurations are represented in the raw
; data of the current directory. Configurations are different for
; PRISM and GRISM, as well as for different baselines. Returns configurations
; in the format, e.g., PRISM-UT2-UT3.
;
files=findfile('MIDI*.fits')
;
if strlen(files(0)) eq 0 then return,'No files'
;
n=n_elements(files)
configurations=strarr(n)
;
for i=0,n-1 do begin
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
;
	nrtsmode=strtrim(prihead->getpar('DET NRTS MODE'))
	index=where(nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
		 or nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
		 or nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
		 or nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED',count)
;
	if count ne 0 then begin
	dispersion=strtrim(prihead->getpar('INS GRIS ID'))
	station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
	station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
	configurations(i)=dispersion+'-'+station_1+'-'+station_2
	endif
;
	obj_destroy,fitsfile
;
endfor
;
index=where(strlen(configurations) ne 0,count)
if count gt 0 then configurations=unique(configurations(index)) $
	      else configurations='Any'
return,configurations
;
end
;-------------------------------------------------------------------------------
function midiconfigids,obht
;
; Return a unique set of configids currently loaded.
;
dispersion=strtrim(obht.insgris)
station_1=strtrim(obht.ISSCONFSTATION1,2)
station_2=strtrim(obht.ISSCONFSTATION2,2)
configurations=dispersion+'-'+station_1+'-'+station_2
;
return,configurations
;
end
;-------------------------------------------------------------------------------
function fixobht,obht
;
	index=where(obht.nrtsmode eq 'ACQ_UT_COARSE_CHOP',count)
	for i=0,count-1 do begin
		if index(i)+1 lt n_elements(obht) then begin
		if (obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED') $
		 and obht(index(i)+1).obstargname eq obht(index(i)).obstargname $
		 then obht(index(i)).insgris=obht(index(i)+1).insgris
		endif
	endfor
	obht.issconfstation1=vlti_stationid(strtrim(obht.issconfstation1,2))
	obht.issconfstation2=vlti_stationid(strtrim(obht.issconfstation2,2))
;
return,obht
;
end
;-------------------------------------------------------------------------------
function midirawdirchecksum
;
; Returns a checksum of the MIDI files in the current directory.
;
f=findfile('MIDI*.fits')
n=n_elements(f)
if n eq 0 then return,0
r=file_info(f(0))
r=replicate(r,n)
for i=0,n-1 do r(i)=file_info(f(i))
;
return,strjoin(string(r.name)+string(r.ctime)+string(r.mtime)+string(r.size))
;
end
;-------------------------------------------------------------------------------
pro selectmidifiles,all=all
;
; Displays the gorgonzola gui to select files. Stores the results in
; common block. If all are requested, return only those corresponding
; to the currently selected configuration.
;
common MidiGuiWids,midi_wid,files_wid,window_slide_wid,pipeline_wid
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
if total(strlen(findfile('rawdirchecksum.sav'))) ne 0 $
	then restore,'rawdirchecksum.sav'
;
; This is used to detect changes to the raw directory
if n_elements(RawDirCheckSum) eq 0 then RawDirCheckSum=''
;
checksum=midirawdirchecksum()
if checksum ne RawDirCheckSum or $
	(keyword_set(all) and total(strlen(findfile('obht.sav'))) eq 0) then begin
	RawDirCheckSum=checksum
	keys=['OBS TARG NAME','NRTS MODE','INS GRIS','INS OPT1','INS SHUT', $
		'INS FILT NAME','ISS CONF STATION1','ISS CONF STATION2']
	print,'Please wait, reading files...'
	if keyword_set(all) then begin
		ob=obj_new('fitsfilelist',searchString='MIDI*.fits',/midiCompress)
		obht=ob->getHeaderKeys(keys)
		nobht=n_elements(obht)
		ob->setHeader,obht,keys
		obsfiles=ob->files()
		obht.filename=obsfiles
	endif else begin
		ob=obj_new('fitsfilelist',dir=dir, $
			guikey=keys,guisearch='MIDI*.fits',/midiCompress)
; 		The header table initially contains all files
		obht=ob->headertable()
		nobht=n_elements(obht)
		obsfiles=ob->files()
		files=obht.filename
		n=n_elements(obsfiles)
		if strlen(obsfiles(0)) eq 0 then n=0
		if n gt 0 then begin
			index=intarr(n)
			for i=0,n-1 do begin
				words=nameparse(obsfiles(i))
				index(i)=where(files eq words(0))
			endfor
			obht=obht(index)
			obht.filename=obsfiles
		endif
	endelse
	ob->save,'gorgonzola.sav'
	obj_destroy,ob
;
;	The acquisition files list INSGRIS as OPEN which is always the
;	default. But we need to set it to what's in the FRINGE file
;	so that we later can select all files for a specific configuration.
	index=where(obht.nrtsmode eq 'ACQ_UT_COARSE_CHOP',count)
	for i=0,count-1 do begin
		if index(i)+1 lt n_elements(obht) then begin
		if (obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
		 or obht(index(i)+1).nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED') $
		 and obht(index(i)+1).obstargname eq obht(index(i)).obstargname $
		 then obht(index(i)).insgris=obht(index(i)+1).insgris
		endif
	endfor
	issconfstation1=strtrim(obht.issconfstation1,2)
	issconfstation2=strtrim(obht.issconfstation2,2)
	if strlen(issconfstation1(0)) ge 2 then obht.issconfstation1=vlti_stationid(issconfstation1)
	if strlen(issconfstation2(0)) ge 2 then obht.issconfstation2=vlti_stationid(issconfstation2)
	if nobht eq n_elements(obsfiles) then save,obht,filename='obht.sav'
	save,RawDirCheckSum,filename='rawdirchecksum.sav'
endif else begin
	if keyword_set(all) then begin
		print,'Reading file list from disk file obht.sav...'
		restore,'obht.sav'
	endif else begin
		print,'Reading file list from disk file gorgonzola.sav...'
		ob=obj_new('fitsfilelist',restore='gorgonzola.sav')
		ob->guiSelect
; 		The header table initially contains all files, reduce to the ones selected
		obht=ob->headertable()
		obsfiles=ob->files()
		files=obht.filename
		n=n_elements(obsfiles)
		if strlen(obsfiles(0)) eq 0 then n=0
		if n gt 0 then begin
			index=intarr(n)
			for i=0,n-1 do begin
				words=nameparse(obsfiles(i))
				index(i)=where(files eq words(0))
			endfor
			obht=obht(index)
			obht.filename=obsfiles
		endif
	endelse
endelse
;
; For option all, we do however select according to the current configid
if keyword_set(all) then begin
	configurations=midiconfigids(obht)
	index=where(configurations eq configid,count)
	if count gt 0 then obht=obht(index)
endif
;
; The obsindex shall refer exactly to those files needed for a visibility
obsindex=where(obht.nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
	    or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
	    or obht.nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
	    or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED' $
	    or obht.nrtsmode eq 'DEFAULT_CHOP')
;
; If a single observation was selected, sort it
if n_elements(obsindex) eq 3 then begin
	si=sort(obht.insshut+obht.nrtsmode)
	obht=obht(si)
	obsindex=where(obht.nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
		    or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
		    or obht.nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
		    or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED' $
		    or obht.nrtsmode eq 'DEFAULT_CHOP')
;	Keep record of which combination was selected
	if n_elements(recipes) gt 0 then begin
		i=where(recipes.track_file eq obht(obsindex(0)).filename,count)
		if count eq 1 then begin
			recipes(i).phot_a_file=obht(obsindex(1)).filename
			recipes(i).phot_b_file=obht(obsindex(2)).filename
			if obht(0).nrtsmode eq 'ACQ_UT_COARSE_CHOP' then $
				recipes(i).acq_file=obht(0).filename else $
				recipes(i).acq_file=''
		endif
	endif
endif
;
; Display selected files in list widget
if n_elements(files_wid) eq 0 then files_wid=0l
if widget_info(files_wid,/valid) then $
widget_control,files_wid,set_value=obht.filename,set_uvalue=obht.filename
;
end
;-------------------------------------------------------------------------------
pro midiupdateconfig,genconfig
;
; The wavelength section of GenConfig is not known until la_bins have been
; defined, either by default, or using lambda_gui. This procedure is called
; once this info is available to finish the full definition of GenConfig.
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
dispersion=strtrim(unique(obht.insgris),2)
;
win1nx=detwin(0)
win1strx=detwin(1)
wavelengths=xmdvavg(pix2lambda(dindgen(win1nx)+win1strx,dispersion), $
			la_bins)*1e-6
genconfig.numspecchan(*)=n_elements(wavelengths)
genconfig.wavelength(0:genconfig.numspecchan(0)-1,0)=wavelengths
genconfig.wavelength(*,1)=genconfig.wavelength(*,0)
genconfig.wavelengtherr=genconfig.wavelength*0+0.1e-6
;
end
;-------------------------------------------------------------------------------
pro midigenconfig,filename,genconfig
;
; Open specified file, and extract information into given genconfig.
; This does not include channel wavelengths and  widths.
;
; Open file
fitsfile=obj_new('fitsfile',filename)
prihead=fitsfile->prihead()
;
; Read configuration
dispersion=strtrim(prihead->getpar('INS GRIS ID'))
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
stations=[station_1,station_2]
delayline_1=strtrim(prihead->getpar('ISS CONF DL1'))
delayline_2=strtrim(prihead->getpar('ISS CONF DL2'))
delaylines=[delayline_1,delayline_2]
delaylineids=strmid(delaylines,2,1)
refname=prihead->getpar('DEL REF NAME')
bcinput_1=strtrim(prihead->getpar('ISS CONF INPUT1'))
bcinput_2=strtrim(prihead->getpar('ISS CONF INPUT2'))
bcinputs=[bcinput_1,bcinput_2]
beam_a=stations(where(bcinputs eq 3)) & beam_a=beam_a(0)
beam_b=stations(where(bcinputs eq 1)) & beam_b=beam_b(0)
;
; Close file
obj_destroy,prihead
obj_destroy,fitsfile
;
; Store configuration in genconfig
genconfig.stationid(0)=stations(0)
genconfig.stationid(1)=stations(1)
genconfig.baselineid(0,*)=genconfig.stationid(0)+'-'+genconfig.stationid(1)
genconfig.delaylineid=delaylineids
ref_count=0
if isnumeric(refname) then $
ref_index=where(delaylineids eq long(strmid(refname,2,1)),ref_count)
if ref_count eq 0 then ref_index=0
genconfig.refstation=ref_index+1
genconfig.bcinputid=bcinputs
genconfig.numoutbeam=2
genconfig.numbaseline=1
case dispersion of
	'PRISM': genconfig.numspecchan=n_elements(genconfig.wavelength(*,0))
	'GRISM': genconfig.numspecchan=n_elements(genconfig.wavelength(*,0))
endcase
genconfig.configid=dispersion+'-'+genconfig.stationid(0)+'-'+genconfig.stationid(1)
;
end
;-------------------------------------------------------------------------------
function midistarids,scans
;
; Adopt a single starid (the first) for stars of the same name.
; This is important for the OBJ designation in case it was derived
; from apparent coordinates.
;
stars=unique(scans.star)
for i=0,n_elements(stars)-1 do begin
	index=where(scans.star eq stars(i))
	scans(index).starid=scans(index(0)).starid
endfor
;
return,scans
;
end
;-------------------------------------------------------------------------------
pro midiscan,filename,scan_i
;
; Open specified file, and extract information into given OYSTER scan
;
; Open file
fitsfile=obj_new('fitsfile',filename)
prihead=fitsfile->prihead()
;
; Read data
airmass=float(strtrim(prihead->getpar('ISS AIRM END')))
seeing=float(strtrim(prihead->getpar('ISS AMBI FWHM')))
dispersion=strtrim(prihead->getpar('INS GRIS ID'))
dateobs=strtrim(prihead->getpar('DATE-OBS'))
time0=hms2h(strmid(dateobs,11,12))*3600	; [s]
obstarg=strtrim(prihead->getpar('OBS TARG NAME'))
obsra=prihead->getpar('RA')/15.
obsdec=prihead->getpar('DEC')
dlt1=prihead->getpar('DEL DLT1 OPL START')
dlt2=prihead->getpar('DEL DLT2 OPL START')
refopl=prihead->getpar('DEL REF OPL')
categ=prihead->getpar('DPR CATG')
;
; Close file
obj_destroy,prihead
obj_destroy,fitsfile
;
; Get starid, for which we try the HD and HIP catalogues first
starid=cri_vlti(obstarg,obsra,obsdec)
;
; Store data in scan
scan_i.time=time0
scan_i.star=obstarg
scan_i.starid=starid
scan_i.ra=obsra
scan_i.dec=obsdec
scan_i.r0=seeing
scan_i.za=acos(1/airmass)*(180/!pi)
;
end
;-------------------------------------------------------------------------------
function minrtsmask,obht
;
case strtrim(obht.insgris) of
'PRISM':begin
	case strtrim(obht.insopt1) of
	'SCI_PHOT':	mask='minrtsMask_FIELD_PRISM_SCI_PHOT_SLIT.fits'
	'HIGH_SENS': 	mask='minrtsMask_FIELD_PRISM_HIGH_SENS_SLIT.fits'
	endcase
	end
'GRISM':begin
	case strtrim(obht.insopt1) of
	'SCI_PHOT':	mask='minrtsMask_SPECTRAL_GRISM_SCI_PHOT_SLIT.fits'
	'HIGH_SENS': 	mask='minrtsMask_SPECTRAL_GRISM_HIGH_SENS_SLIT.fits'
	endcase
	end
endcase
return,getenv('drsRoot')+'/maskfiles/'+mask
;
end
;-------------------------------------------------------------------------------
function miosetup,obht
;
case strtrim(obht.insgris) of
'PRISM': sfile='mioSetup_FIELD_PRISM_SCI_PHOT_SLIT.tmp'
'GRISM': sfile='mioSetup_SPECTRAL_GRISM_SCI_PHOT_SLIT.tmp'
endcase
return,getenv('drsRoot')+'/mio/'+sfile
;
end
;-------------------------------------------------------------------------------
function miokappa,obht
;
case strtrim(obht.insgris) of
'PRISM': kfile='mioKappa_FIELD_PRISM_SCI_PHOT.fits'
'GRISM': kfile='mioKappa_SPECTRAL_GRISM_SCI_PHOT.fits'
endcase
return,getenv('drsRoot')+'/mio/'+kfile
;
end
;-------------------------------------------------------------------------------
function defaultrecipes
;
; For the currently selected MIDI files, establish default recipes.
;
; Rules: if "Coherent" option selected, method will always be "Leiden"
; If there is SCI_PHOT without photometry, "Leiden" will be used.
; If there is SCI_PHOT with photometry, "Heidelberg will be used",
; and a kappa matrix computed (anyway).
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
index_obs=where(obht.nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
	     or obht.nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED',count_obs)
if count_obs ne n_elements(recipes) then return,-1
;
IF midi_options.v THEN BEGIN
;
for i=0,count_obs-1 do begin
	recipes(i).track_file=obht(index_obs(i)).filename
	if midi_options.c then recipes(i).method='Leiden' $
			  else recipes(i).method='Heidelberg'
	if obht(index_obs(i)).insopt1 eq 'HIGH_SENS' and index_obs(i)+2 ge n_elements(obht) then begin
		print,'***Error(DEFAULTRECIPES): HIGH_SENS photometry missing!'
		return,-1
	endif
	if obht(index_obs(i)).insopt1 eq 'SCI_PHOT' then begin
		kappafile=findfile('kappa.'+strtrim(obht(index_obs(i)).insgris,2)+'.fits')
		if strlen(kappafile(0)) eq 0 $
		then recipes(i).kappa_matrix=miokappa(obht(index_obs(i))) $
		else recipes(i).kappa_matrix=kappafile
	endif
	if index_obs(i)+2 lt n_elements(obht) then begin
	if obht(index_obs(i)+1).nrtsmode eq 'DEFAULT_CHOP' and $
	   obht(index_obs(i)+2).nrtsmode eq 'DEFAULT_CHOP' then begin
	   for j=1,2 do begin
	   	case strtrim(obht(index_obs(i)+j).insshut,2) of
			'AOPEN':recipes(i).phot_a_file=obht(index_obs(i)+j).filename
			'BOPEN':recipes(i).phot_b_file=obht(index_obs(i)+j).filename
		endcase
	   endfor
	   if recipes(i).method eq 'Leiden' $
		then recipes(i).mask_file=minrtsmask(obht(index_obs(i))) $
		else recipes(i).mask_file=''
	   if obht(index_obs(i)).insopt1 eq 'SCI_PHOT' then begin
			if strlen(kappafile(0)) eq 0 then recipes(i).kappa_matrix='Compute'
	   endif
	endif else recipes(i).method='Leiden'
	endif else recipes(i).method='Leiden'
	if index_obs(i) gt 0 then begin
	if obht(index_obs(i)-1).nrtsmode eq 'ACQ_UT_COARSE_CHOP' then begin
		recipes(i).acq_file=obht(index_obs(i)-1).filename
	endif
	endif
endfor
;
ENDIF ELSE BEGIN
;
for i=0,count_obs-1 do begin
	recipes(i).track_file=obht(index_obs(i)).filename
	recipes(i).tag=strtrim(obht(index_obs(i)).obstargname,2)+' ' $
		      +strmid(obht(index_obs(i)).filename,16,8)
	if midi_options.c then recipes(i).method='Leiden' $
			  else recipes(i).method='Heidelberg'
	recipes(i).photometry='SciPhot'
	if obht(index_obs(i)).insopt1 eq 'HIGH_SENS' $
	    and index_obs(i)+2 ge n_elements(obht) then begin
		print,'***Error(DEFAULTRECIPES): HIGH_SENS photometry missing!'
		return,-1
	endif
	if obht(index_obs(i)).insopt1 eq 'SCI_PHOT' then begin
		kappafile=findfile('kappa.'+strtrim(obht(index_obs(i)).insgris,2) $
				+'.crossCoeff.fits')
		if strlen(kappafile(0)) eq 0 $
		then recipes(i).kappa_matrix=miokappa(obht(index_obs(i))) $
		else recipes(i).kappa_matrix=kappafile
		photofile.file=obht(index_obs(i)).obstargname+'.photometry.fits'
	endif
	if index_obs(i)+2 lt n_elements(obht) then begin
	if obht(index_obs(i)+1).nrtsmode eq 'DEFAULT_CHOP' and $
	   obht(index_obs(i)+2).nrtsmode eq 'DEFAULT_CHOP' then begin
	   for j=1,2 do begin
	   	case strtrim(obht(index_obs(i)+j).insshut,2) of
			'AOPEN':recipes(i).phot_a_file=obht(index_obs(i)+j).filename
			'BOPEN':recipes(i).phot_b_file=obht(index_obs(i)+j).filename
		endcase
	   endfor
	   if recipes(i).method eq 'Leiden' $
		then recipes(i).mask_file=minrtsmask(obht(index_obs(i))) $
		else recipes(i).mask_file=''
	   if obht(index_obs(i)).insopt1 eq 'SCI_PHOT' then begin
			recipes(i).kappa_matrix='Compute'
	   endif
	endif
	endif
	if index_obs(i) gt 0 then begin
	if obht(index_obs(i)-1).nrtsmode eq 'ACQ_UT_COARSE_CHOP' then begin
		recipes(i).acq_file=obht(index_obs(i)-1).filename
	endif
	endif
endfor
;
ENDELSE
;
save,recipes,filename='Recipes_'+configid+'.xdr'
;
return,0
;
end
;************************************************************************Block 4
function midigui_event,event
;
; This is the call back for buttons on the main MyMidiGui.
;
common MidiGuiWids,midi_wid,files_wid,window_slide_wid,pipeline_wid
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiXmdv,xmdvobj,xmdvgui_wid
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
widget_control,event.id,get_uvalue=dir
;
case event.value of
;
'Observation':	begin
;		Clean desktop a little
		wdall
		if widget_info(midiacq_wid,/valid) then widget_control,midiacq_wid,/destroy
		if widget_info(midipho_wid,/valid) then widget_control,midipho_wid,/destroy
		if widget_info(miditrack_wid,/valid) then widget_control,miditrack_wid,/destroy
		if widget_info(xmdvgui_wid,/valid) then widget_control,xmdvgui_wid,/destroy
;		Destroy objects we know about
		if obj_valid(xmdvobj) then obj_destroy,xmdvobj
;		Clear mask selection
		if widget_info(midimask_wid,/valid) then midimaskgui,/update
;		Clear slide window
		if widget_info(window_slide_wid,/valid) then widget_control,window_slide_wid,/destroy
;		Clear kappa file
		kappafile=''
;		Clear chop data
		phodata1=0
		phodata2=0
		trackdata1=0
		trackdata2=0
		set_screen
		!x.range=0
		!y.range=0
		!p.psym=0
;		Store current OYSTER observation
		if n_elements(geninfo) ne 0 then begin
			storenight,11
			save,recipes,filename='Recipes_'+genconfig.configid+'.xdr'
		endif
;
		selectmidifiles
;
;		If a complete observation was selected, get additional info
		beamcombiner=strtrim(obht(obsindex(0)>0).insopt1)
		if n_elements(obsindex) eq 3 $
		or (n_elements(obsindex) eq 1 and beamcombiner eq 'SCI_PHOT') then begin
			if beamcombiner eq 'HIGH_SENS' then iobs=1 else iobs=0
			fitsfiles=nameparse(strjoin(obht(obsindex(iobs)).filename,' '))
			fitsfile=obj_new('fitsfile',fitsfiles(0))
			prihead=fitsfile->prihead()
        		chopthrow=strtrim(prihead->getpar('ISS CHOP THROW'))
			airmass=strtrim(prihead->getpar('ISS AIRM END'))
			seeing=strtrim(prihead->getpar('ISS AMBI FWHM'))
			dispersion=strtrim(prihead->getpar('INS GRIS ID'))
			obj_destroy,prihead
			obj_destroy,fitsfile
			print,'Target: ',obht(obsindex(0)).obstargname
			print,'Airmass:',airmass,', seeing:',seeing
			print,'Photometry chop throw:',chopthrow
			if midi_options.v then begin
			if dispersion eq 'GRISM' then kappafile='kappa.GRISM.fits' $
						 else kappafile='kappa.PRISM.fits'
			endif else begin
			if dispersion eq 'GRISM' then kappafile='kappa.GRISM.crossCoeff.fits' $
						 else kappafile='kappa.PRISM.crossCoeff.fits'
			endelse
;			Load corresponding OYSTER scans structure
			obht=fixobht(obht)
			if n_elements(geninfo) ne 0 then begin
				currentconfig=unique(midiconfigids(obht(obsindex)))
				index=where(geninfo.configid eq currentconfig,count)
				if count eq 1 then begin
				datum=geninfo(index).date
				array=geoinfo(index).systemid
				loadnight,datum,array,currentconfig
				restore,'Recipes_'+currentconfig+'.xdr'
				endif
			endif
		endif
		end
'Kappa matrix': begin
		print,'Please select two photometry files, shutter A and B!'
		selectmidifiles
		if n_elements(obht) ne 2 then return,-1
		if midi_options.v then begin
			kappamatrix,obht
		endif else begin
			kappafile='kappa.'+strtrim(obht(0).insgris)
			midiCrossCoeff,kappafile,obht.filename
		endelse
		end
' Photo file ':	begin
		if not midi_options.v then begin
			print,'Please select pairs of photometry files for shutter A and B!'
			selectmidifiles
			nf=n_elements(obht)
			if (nf mod 2) ne 0 then return,-1
			target=obht(0).obstargname
			fa=obht(0).filename
			for i=2,nf-2,2 do fa=fa+' '+obht(i).filename
			fb=obht(1).filename
			for i=3,nf-1,2 do fb=fb+' '+obht(i).filename
			photofile_stub=obht(0).obstargname+'.'+strtrim(obht(0).insgris)
			midiPhotoPipe,photofile_stub,[fa,fb]
			photofile.file=photofile_stub+'.photometry.fits'
			photofile.files=[fa,fb]
		endif
		end
'Mask Tracker': begin
		midimaskgui
		end
'OYSTER': 	begin
		initializepipe
		print,'OYSTER data and recipes initialized.'
		widget_control,pipeline_wid,/sensitive
		oyster
		end
'Pipeline':	begin
		mymidipipe
		end
'Recipes': 	begin
		midirecipesgui
		end
'SmartMove':    begin
		print,'Reading files...'
                if n_elements(obht) gt 0 then begin
                        cleanmidirawdir,obht.filename
                endif else cleanmidirawdir
		print,'Moved files to moremididata.'
                end
'Move':		begin
		for i=0,n_elements(obht.filename)-1 do $
			print,obht(i).filename
		answer=' '
		read,prompt='Move these files to "moremididata"? (y/n): ',answer
		if answer eq 'y' then begin
			spawn,'mkdir -p moremididata'
			for i=0,n_elements(obht.filename)-1 do begin
				spawn,'mv '+obht(i).filename+' moremididata'
			endfor
		endif
		end
'Delete':	begin
		for i=0,n_elements(obht.filename)-1 do $
			print,obht(i).filename
		answer=' '
		read,prompt='Delete these files? (yes/no): ',answer
		if answer eq 'yes' then spawn,'rm -f '+strjoin(obht.filename,' ')
		if answer ne 'yes' then print,'Files not removed.'
		end
'Reset':	begin
		spawn,'rm -f gorgonzola.sav'
		spawn,'rm -f obht.sav'
		spawn,'rm -f rawdirchecksum.sav'
		RawDirCheckSum=''
		if n_elements(obht) ne 0 then begin
			obht=obht(0)
			obht.filename=''
			obht.nrtsmode=''
			widget_control,files_wid, $
				set_value=obht.filename,set_uvalue=obht.filename
		endif
		print,'Done.'
		end
'Acquisition':	begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		acqindex=where(obht.nrtsmode eq 'ACQ_UT_COARSE_CHOP',count)
		if count eq 1 then midiacqgui
		end
'Fringe search': begin
		index=where(obht.nrtsmode eq 'OBS_FRINGE_SEARCH_FOURIER',count)
		if count eq 1 then midisearchgui,index(0)
		end
'Fringe track': begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		if obsindex(0) lt 0 then return,-1
		if n_elements(obsindex) eq 3 then miditrackgui
		if n_elements(obsindex) eq 1 and strtrim(obht(obsindex(0)).insopt1) eq 'SCI_PHOT' $
					     then miditrackgui
		end
'Photometry A':	begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		if widget_info(midipho_wid,/valid) then widget_control,midipho_wid,/destroy
		phoindex=where(strpos(obht.nrtsmode,'DEFAULT') ge 0 $
			and strtrim(obht.insshut) eq 'AOPEN',count)
		if count gt 0 and count le 2 then midiphogui
		end
'Photometry B':	begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		if widget_info(midipho_wid,/valid) then widget_control,midipho_wid,/destroy
		phoindex=where(strpos(obht.nrtsmode,'DEFAULT') ge 0 $
			and strtrim(obht.insshut) eq 'BOPEN',count)
		if count gt 0 and count le 2 then midiphogui
		end
'Form':		spawn,'lpr -Plaser1 midi.ps'
;
endcase
;
end
;-------------------------------------------------------------------------------
function midigui_configid,event
;
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
forward_function midirecipesgui_event
;
; Set configid
r=size(event)
if r(n_elements(r)-2) eq 8 then begin
        widget_control,event.id,get_uvalue=configids
        configid=configids(event.index)
endif else configid=event
;
; Load buffered OYSTER observation
if n_elements(geninfo) eq 0 or n_elements(recipes) eq 0 then begin
	print,'Warning: pipeline not initialized'
	return,-1
endif
index=where(geninfo.configid eq configid)
datum=geninfo(index).date
array=geoinfo(index).systemid
loadnight,datum,array,configid
;
; Restore recipes
restore,'Recipes_'+configid+'.xdr'
;
; Update recipes GUI
r=midirecipesgui_event({value:'Update'})
;
end
;-------------------------------------------------------------------------------
function midigui_options,event
;
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
forward_function midirecipesgui_event
;
if midi_options.v then begin
	case event.value of
		0: midi_options.j=event.select
		1: midi_options.m=event.select
		2: midi_options.c=event.select
	endcase
endif else begin
	widget_control,event.id,get_uvalue=value
	if value eq 'Photometry' then begin
		midi_options.p=event.value
	endif else begin
		case event.value of
		0: midi_options.m=event.select
		1: midi_options.s=(event.select+99) mod 100
		2: midi_options.c=event.select
		3: midi_options.f=event.select
		endcase
	endelse
endelse
;
if n_elements(recipes) ne 0 then begin
	if midi_options.c then recipes.method='Leiden' $
			  else recipes.method='Heidelberg'
	case midi_options.p of
		0:recipes.photometry='HighSens'
		1:recipes.photometry='Scaled'
		2:recipes.photometry='SciPhot'
	endcase
	r=midirecipesgui_event({value:'Update'})
endif
;
end
;-------------------------------------------------------------------------------
function files_event,event
;
; When clicking on a file, start fv on it.
;
widget_control,event.id,get_uvalue=files
if n_elements(files) ne 0 then $
spawn,'fv -cmap 2 '+files(event.index)+' &'
;
end
;-------------------------------------------------------------------------------
pro mymidigui,dir
;
; Creates the MyMidiGui main widget.
;
common MidiGuiWids,midi_wid,files_wid,window_slide_wid,pipeline_wid
common MidiXmdv,xmdvobj,xmdvgui_wid
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
;
; Initialization and cleanup
if n_elements(midi_wid) eq 0 then midi_wid=0L
if n_elements(midiacq_wid) eq 0 then midiacq_wid=0L
if n_elements(midipho_wid) eq 0 then midipho_wid=0L
if n_elements(miditrack_wid) eq 0 then miditrack_wid=0L
if n_elements(midimask_wid) eq 0 then midimask_wid=0L
if n_elements(midirecipes_wid) eq 0 then midirecipes_wid=0L
if n_elements(window_slide_wid) eq 0 then window_slide_wid=0L
if n_elements(xmdvgui_wid) eq 0 then xmdvgui_wid=0L
if widget_info(midi_wid,/valid) then widget_control,midi_wid,/destroy
if widget_info(midiacq_wid,/valid) then widget_control,midiacq_wid,/destroy
if widget_info(midipho_wid,/valid) then widget_control,midipho_wid,/destroy
if widget_info(miditrack_wid,/valid) then widget_control,miditrack_wid,/destroy
if widget_info(midimask_wid,/valid) then widget_control,midimask_wid,/destroy
if widget_info(midirecipes_wid,/valid) then widget_control,midirecipes_wid,/destroy
if widget_info(window_slide_wid,/valid) then widget_control,window_slide_wid,/destroy
if widget_info(xmdvgui_wid,/valid) then widget_control,xmdvgui_wid,/destroy
maskfile=''
kappafile=''
photofile={file:'',files:strarr(2)}
phodata1=0
phodata2=0
trackdata1=0
trackdata2=0
;
if total(strlen(findfile('rawdirchecksum'))) ne 0 then restore,'rawdirchecksum'
;
RawDir=''
if n_elements(dir) eq 0 then spawn,'pwd',RawDir else RawDir=dir
RawDir=RawDir(0)+'/'
if total(strlen(findfile(RawDir))) eq 0 then begin
	print,'***Error(MYMIDIGUI): raw file directory non-existent!'
	return
endif
;
; This is used to detect changes to the raw directory
if n_elements(RawDirCheckSum) eq 0 then RawDirCheckSum=''
;
; dispSciPhot cannot create this directory
if not file_test("PHOTOMETRY") then spawn,'mkdir "PHOTOMETRY"'
;
; Allocate options; now done through call to function in structure.pro
; m: Use old mask algorithm (0), or new (1)
; c: Coherent integration
; s: Sky subtraction
; r: Use recipes
; j: Remove jitter in photometry (experimental)
; v: Use mymidigui code for MIA+EWS-1.3
; p: Pseudo-HIGH_SENS
; midi_options={m:0,s:99,c:0,r:0,j:0,v:0,p:0,width:1.0,t_order:2,w_order:1,before:0,after:2}
midi_options=alloc_midi_options()
;
; Initialize lambda binning parameters
la_start=1
la_end=9999
la_step=3
la_width=la_step
la_bins=[la_start,la_end,la_step,la_width]
;
midi_wid=widget_base(/column,title=RawDir(0))
;
templates=['Observation', $
	   'Kappa matrix', $
	   ' Photo file ', $
	   'Mask Tracker']
button_wid=cw_bgroup(midi_wid,/row,templates, $
	event_func='midigui_event',/return_name)
;
templates=['Acquisition', $
	   'Fringe track', $
	   'Photometry A', $
	   'Photometry B']
button_wid=cw_bgroup(midi_wid,/row,templates, $
	event_func='midigui_event',/return_name)
;
files_wid=widget_list(midi_wid,value=strarr(10)+' ', $
	event_func='files_event', $
	xsize=15,ysize=10)
;
templates=['Reset','SmartMove','Move','Delete']
button_wid=cw_bgroup(midi_wid,/row,templates, $
	event_func='midigui_event',/return_name)
;
col_wid=widget_base(midi_wid,/col,/frame)
row_wid=widget_base(col_wid,/row)
;
configids=analyzemidiconfigids()
opmenuc_wid=widget_droplist(row_wid,event_func='midigui_configid', $
	uvalue=configids,value=configids)
configid=configids(0)
;
templates=['Pipeline','Recipes']
pipeline_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='midigui_event',/return_name)
widget_control,pipeline_wid,sensitive=0
templates=['OYSTER']
pipeline2_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='midigui_event',/return_name)
;
if midi_options.v then options=['Binned Avg','MarquardtMask','Coherent'] $
		  else options=['MarquardtMask','SubtractSky','Coherent','Finito']
options_wid=cw_bgroup(col_wid,options,/row,/nonexclusive, $
	event_func='midigui_options',uvalue='Global')
if not midi_options.v then begin
	options=['HighSens','Scaled','SciPhot']
	options_wid=cw_bgroup(col_wid,options,/row,/exclusive, $
		event_func='midigui_options',uvalue='Photometry')
	widget_control,options_wid,set_value=2
	midi_options.p=2
endif
;
widget_control,midi_wid,/realize
xmanager,'mymidigui',midi_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function midiacqgui_event,event
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
common MidiAcqGuiWids,draw_wid
common MidiAcqSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
;
; OYSTER common blocks
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; We take into account slightly different reference pixels:
; WIN1 (i.e. beam A) has lower left corner of STRX,STRY=(160,46)
; WIN2 (i.e. beam B) has lower left corner of STRX,STRY=(160,131)
; Reference pixel is (190,78) in WIN1 and (189,164) in WIN2,
; therefore it is, in the window (62x69) read, (30,32) and (29,33), rsp.
; We have to move WIN2 to make the reference pixel coincide with the one
; in WIN1, i.e. we have to move it by (1,-1).
;
; midichopframes has a bug, actually in EWS/idl/ews/midi/utilities/fringeUtilities.pro,
; line 554, which should read if(blockTarg[row] GT 0) then begin, instead of GE 0.
;
o=[1,-1]	; + moves B up/right
;
widget_control,event.id,get_uvalue=tools
if n_elements(tools) eq 1 then value=event.value else value=tools(event.index)
event={value:value}
;
widget_control,draw_wid,get_value=id
wset,id
erase
;
case event.value of
;
'Beam A ON':		image=z(0).data1
'Beam A OFF':		image=z(1).data1
'Beam A ON-OFF':	image=z(0).data1-z(1).data1
'Beam B ON':		image=shift(z(0).data2,o(0),o(1))
'Beam B OFF':		image=shift(z(1).data2,o(0),o(1))
'Beam B ON-OFF':	image=shift(z(0).data2-z(1).data2,o(0),o(1))
'Beam A - Beam B':	image=(z(0).data1)/max(z(0).data1) $
		       -shift((z(0).data2)/max(z(0).data2),o(0),o(1))
'Beam A Chopped':begin
		index=where(acqdata1.t eq 'T ' and acqdata1.c ne 0,count)
		image=total(acqdata1.cf(*,*,index),3)/count
		end
'Beam B Chopped':begin
		index=where(acqdata2.t eq 'T ' and acqdata2.c ne 0,count)
		image=total(acqdata2.cf(*,*,index),3)/count
		end
'Beam A Offset':begin
		index=where(acqdata1.c ne 0 and acqdata1.t eq 'T ')
                x=acqdata1.x(index)-acqdata1.xmed
                y=acqdata1.y(index)-acqdata1.ymed
                plot,x,y,psym=3
                fluxes=acqdata1.f(index)
                v=medianve(fluxes,e)
                j=where(abs(fluxes-v) gt e/2)
                oplot,x(j),y(j),psym=3,color=2
		return,0
		end
'Beam B Offset':begin
		index=where(acqdata2.c ne 0 and acqdata2.t eq 'T ')
                x=acqdata2.x(index)-acqdata2.xmed
                y=acqdata2.y(index)-acqdata2.ymed
                plot,x,y,psym=3
                fluxes=acqdata2.f(index)
                v=medianve(fluxes,e)
                j=where(abs(fluxes-v) gt e/2)
                oplot,x(j),y(j),psym=3,color=2
		return,0
		end
'Beam A Flux':	begin
		index=where(acqdata1.c ne 0 and acqdata1.t eq 'T ')
		plot,acqdata1.f(index),psym=3,ytitle='Flux (A)'
		return,0
		end
'Beam B Flux':	begin
		index=where(acqdata2.c ne 0 and acqdata2.t eq 'T ')
		plot,acqdata2.f(index),psym=3,ytitle='Flux (B)'
		return,0
		end
'Frames ON-OFF':begin
	frames=acqdata1.cf
	labels=acqdata1.t
	widget_control,i_wid,/sensitive
	widget_control,j_wid,/sensitive
	midiacqgui2_event,'Init'
	return,0
	end
'Movie (chopped)':begin
	framemovie,acqdata1.t,acqdata1.c,acqdata1.cf,acqdata2.cf
	print,'Done.'
	return,0
	zz=midichopframes(obht(acqindex(0)).filename)
	widget_control,draw_wid,get_value=id
	wset,id
	for i=0,n_elements(zz)-1 do begin
		tvsclm,zz(i).data1,2
		tvsclm,zz(i).data2,2,61*2,0
		wait,0.05
	endfor
	print,'Done.'
	return,0
	end
'Movie (aligned)':begin
	framemovie,acqdata1.t,acqdata1.c,acqdata1.af,acqdata2.af
	print,'Done.'
	return,0
	end
'Store':begin
	if n_elements(scans) eq 0 then begin
		print,'You must define OYSTER data set first!'
		return,-1
	endif
	iobs=where(recipes.acq_file eq obht(0).filename)
	words=nameparse(obht(acqindex(0)).filename)
	fitsfile=obj_new('fitsfile',words(0))
	prihead=fitsfile->prihead()
	scans(iobs).AcqFilter=strtrim(prihead->getpar('INS FILT NAME'))
	obj_destroy,prihead
	obj_destroy,fitsfile
;
	image_a=z(0).data1-z(1).data1
	image_b=shift(z(0).data2-z(1).data2,o(0),o(1))
;	Apply a mask
	imsize=[n_elements(image_a(*,0)),n_elements(image_a(0,*))]
	x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
	y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
	offset_a=median(image_a(where(sqrt(x^2+y^2) lt imsize(0)/3)))
	offset_b=median(image_b(where(sqrt(x^2+y^2) lt imsize(0)/3)))
	image_a(where(sqrt(x^2+y^2) gt imsize(0)/4))=offset_a
	image_b(where(sqrt(x^2+y^2) gt imsize(0)/4))=offset_b
;	Fit 2-dim. Gaussian
	aparm=fltarr(7)
	bparm=fltarr(7)
	r_a=gauss2dfit(image_a,aparm,/tilt)
	r_b=gauss2dfit(image_b,bparm,/tilt)
	RAD=180/!pi
	g2d2fwhm=sqrt(2)*2
	madu=midi_magadu(genconfig.stationid(0),scans(iobs).AcqFilter)
	scans(iobs).acqpsf(0,*)=[aparm(2)*g2d2fwhm,aparm(3)*g2d2fwhm, $
				 aparm(6)*RAD,aparm(4),aparm(5)]
	scans(iobs).acqpsf(1,*)=[bparm(2)*g2d2fwhm,bparm(3)*g2d2fwhm, $
				 bparm(6)*RAD,bparm(4),bparm(5)]
	scans(iobs).acqflux=n2jy(madu-2.5*alog10(!pi*2*[aparm(1)*aparm(2)*aparm(3), $
							bparm(1)*bparm(2)*bparm(3)]))
;	Do analysis using Olivier's chop_nod_phot, which writes output into a text file
;	Note that aperture A and B are exchanged in chop_nod_phot!
	chop_nod_phot,recipes(iobs).acq_file,/silent
	nc=strlen(recipes(iobs).acq_file)
	lines=''
	status=dc_read_fixed('PHOTOMETRY/Phot_'+strmid(recipes(iobs).acq_file,5,nc-9)+'txt', $
		lines,/col,format='(a80)')
	index=where(strpos(lines,'Aperture for Beam B in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		scans(iobs).acqflux(0)=n2jy(midi_magadu(genconfig.stationid(0),scans(iobs).AcqFilter) $
						-2.5*alog10(float(words(0))))
	endif
	index=where(strpos(lines,'Aperture for Beam A in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		scans(iobs).acqflux(1)=n2jy(midi_magadu(genconfig.stationid(1),scans(iobs).AcqFilter) $
						-2.5*alog10(float(words(0))))
	endif
	return,0
	end
else:
endcase
;
!x.range=0
!y.range=0
;
; Display in GUI
loadct,0
tvsclm,image,2
;
; Display using atv
atv,image
;
; Write image to disk
; writefits,'image.fits',image
;
end
;-------------------------------------------------------------------------------
pro midiacqgui
;
; GUI for acquisition analysis.
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiAcqGuiWids,draw_wid
common MidiAcqSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
; Remove any existing widget
if widget_info(midiacq_wid,/valid) then widget_control,midiacq_wid,/destroy
;
; Obtain some relevant information
acqimages=nameparse(obht(acqindex).filename)
fitsfile=obj_new('fitsfile',acqimages(0))
prihead=fitsfile->prihead()
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
input_1=strtrim(prihead->getpar('ISS CONF INPUT1'))
input_2=strtrim(prihead->getpar('ISS CONF INPUT2'))
inputs=[input_1,input_2]
stations=[station_1,station_2]
beam_a=stations(where(inputs eq 3)) & beam_a=beam_a(0)
beam_b=stations(where(inputs eq 1)) & beam_b=beam_b(0)
obj_destroy,prihead
obj_destroy,fitsfile
;
; Run chop_nod_phot first for a quick look summary
loadct,0
chop_nod_phot,obht(acqindex).filename,/silent
;
; Run classical MIA chopping
; z=midiChopImage(obht(acqindex).filename,before=3,after=2)
z=MIDIChopImageC(obht(acqindex).filename,before=midi_options.before,after=midi_options.after)
;
; Analysis using mymidigui functions/procedures
print,'Reading and analyzing beam A frames...'
rf=read_frames(obht(acqindex(0)).filename,1,t)
c=chop_cycles(t)
cf=chop_frames(rf,t,c)
f=analyze_acqframes(cf,t,c,a,b,x,y,xmed,ymed)
af=align_acqframes(cf,t,c,x,y,xmed,ymed)
acqdata1={rf:rf,t:t,c:c,cf:cf,f:f,a:a,b:b,x:x,y:y,xmed:xmed,ymed:ymed,af:af}
print,'Reading and analyzing beam B frames...'
rf=read_frames(obht(acqindex(0)).filename,2,t)
c=chop_cycles(t)
cf=chop_frames(rf,t,c)
f=analyze_acqframes(cf,t,c,a,b,x,y,xmed,ymed)
af=align_acqframes(cf,t,c,x,y,xmed,ymed)
acqdata2={rf:rf,t:t,c:c,cf:cf,f:f,a:a,b:b,x:x,y:y,xmed:xmed,ymed:ymed,af:af}
;
; Now create GUI
midiacq_wid=widget_base(/col, $
	title=strtrim(obht(acqindex).obstargname) $
		+'  A: '+beam_a+' '+'  B: '+beam_b)
row_wid=widget_base(midiacq_wid,/row)
col_wid=widget_base(row_wid,/col)
;
tools1=['Beam A ON', $
       	'Beam A OFF', $
       	'Beam A ON-OFF', $
       	'Beam A Chopped', $
	'Beam A Offset', $
	'Beam A Flux']
tools2=['Beam B ON', $
       	'Beam B OFF', $
       	'Beam B ON-OFF', $
       	'Beam B Chopped', $
	'Beam B Offset', $
	'Beam B Flux']
tools3=['Frames ON-OFF','Movie (chopped)','Movie (aligned)','Store']
; button_wid=cw_bgroup(col_wid,/col,tools1, $
; 	event_funct='midiacqgui_event',/return_name)
opmenu_wid=widget_droplist(col_wid,event_func='midiacqgui_event', $
	uvalue=tools1,value=tools1)
; button_wid=cw_bgroup(col_wid,/col,tools2, $
; 	event_funct='midiacqgui_event',/return_name)
opmenu_wid=widget_droplist(col_wid,event_func='midiacqgui_event', $
	uvalue=tools2,value=tools2)
button_wid=cw_bgroup(row_wid,/col,tools3, $
	event_funct='midiacqgui_event',/return_name)
;
draw_wid=widget_draw(midiacq_wid,event_pro='acqdraw_event', $
	scr_xsize=62*4,scr_ysize=69*2)
;
count=1000
i_wid=cw_fslider(midiacq_wid,minimum=1,maximum=count,value=1,/drag, $
	title='Index I',xsize=256,format='(i4)',uvalue='i')
j_wid=cw_fslider(midiacq_wid,minimum=0,maximum=100,value=0,/drag, $
	title='Index J',xsize=256,format='(i3)',uvalue='j')
;
widget_control,midiacq_wid,/realize
xmanager,'midiacqgui2',midiacq_wid,/no_block
	;
widget_control,i_wid,sensitive=0
widget_control,j_wid,sensitive=0
;
end
;-------------------------------------------------------------------------------
pro midiacqgui2_event,event
;
common MidiAcqGuiWids,draw_wid
common MidiAcqSlider,i_wid,j_wid,i_value,j_value,frames,labels
;
result=size(event)
if result(n_elements(result)-2) ne 8 then begin
	value=1
	i_value=1
	j_value=0
	widget_control,draw_wid,get_value=id
	erase
endif else begin
	widget_control,event.id,get_uvalue=slider,get_value=value
	case slider of
		'i':i_value=long(value)
		'j':j_value=long(value)
	endcase
endelse
;
index=i_value+j_value-1
if index ge n_elements(labels) then return
image=frames(*,*,index)
;
widget_control,draw_wid,get_value=id
wset,id
tvsclm,image,2
xyouts,20,20,labels(index),/device
;
end
;-------------------------------------------------------------------------------
function midimaskfiles_event,event
;
; Call-back for clicks on mask files in the midimaskgui.
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
;
widget_control,event.id,get_uvalue=files
maskfile=files(event.index) & maskfile=maskfile(0)
;
if event.id ne ab_files_wid then begin
	widget_control,ab_files_wid,get_uvalue=f
	widget_control,ab_files_wid,set_value=f
endif
if event.id ne a_files_wid then begin
	widget_control,a_files_wid,get_uvalue=f
	widget_control,a_files_wid,set_value=f
endif
if event.id ne b_files_wid then begin
	widget_control,b_files_wid,get_uvalue=f
	widget_control,b_files_wid,set_value=f
endif
;
; Remove object label
words=nameparse(maskfile)
maskfile=words(0)
;
; Add path info
mask=maskfile
if strpos(mask,'minrtsMask') ge 0 then $
	mask=getenv('drsRoot')+'/maskfiles/'+mask else $
	mask='PHOTOMETRY/'+mask
maskfile=mask
;
print,'Selected mask: ',maskfile
;
mask=oirgetdata(maskfile)
widget_control,draw_wid,get_value=id
wset,id
loadct,0,/silent
tv,bytarr(!d.x_size,!d.y_size)
r=size(mask.data1)
nx=r(1) & ny=r(2)
if n_elements(where(strpos(tag_names(mask),'DATA') ge 0)) eq 4 then SCI_PHOT=1 $
							       else SCI_PHOT=0
if SCI_PHOT then labels=['PB','I1','I2','PA'] else labels=['I1','I2']
mask1=mask.data1 & max1=max(mask1)
mask1(*,0)=max1 & mask1(0,*)=max1 & mask1(nx-1,*)=max1 & mask1(*,ny-1)=max1
tvsclm,mask1,2 & xyouts,nx*2+20,ny,labels(0),charsize=2,/device
mask2=mask.data2 & max2=max(mask2)
mask2(*,0)=max2 & mask2(0,*)=max2 & mask2(nx-1,*)=max2 & mask2(*,ny-1)=max2
tvsclm,mask2,2,0,ny*2 & xyouts,nx*2+20,ny*3,labels(1),charsize=2,/device
if SCI_PHOT then begin
mask3=mask.data3 & max3=max(mask3)
mask3(*,0)=max3 & mask3(0,*)=max3 & mask3(nx-1,*)=max3 & mask3(*,ny-1)=max3
tvsclm,mask3,2,0,ny*4 & xyouts,nx*2+20,ny*5,labels(2),charsize=2,/device
mask4=mask.data4 & max4=max(mask4)
mask4(*,0)=max4 & mask4(0,*)=max4 & mask4(nx-1,*)=max4 & mask4(*,ny-1)=max4
tvsclm,mask4,2,0,ny*6 & xyouts,nx*2+20,ny*7,labels(3),charsize=2,/device
endif
;
end
;-------------------------------------------------------------------------------
function midimaskgui_event,event
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
common MidiRecipesGuiWids,recipes_wid,recipetext_wid,recipe_tag
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
;
widget_control,draw_wid,get_value=id
wset,id
;
case event.value of
;
'Update/Clear':	midimaskgui,/update
'Gui':		begin
		if n_elements(obht) eq 0 then begin
			print,'Please select observation!'
			selectmidifiles
		endif
		if n_elements(obsindex) ne 3 $
		   and obht(obsindex(0)).insopt1 eq 'HIGH_SENS' then begin
			print,'Please select one complete observation!'
			return,-1
		endif
;
		fitsfiles=nameparse(strjoin(obht(obsindex(0)).filename,' '))
		fitsfile=obj_new('fitsfile',fitsfiles(0))
		prihead=fitsfile->prihead()
		win1nx=prihead->getpar('DET WIN1 NX')
		win1strx=prihead->getpar('DET WIN1 STRX')
		detwin=[win1nx,win1strx]
		obj_destroy,prihead
		obj_destroy,fitsfile
;
		loadct,0
		set_screen
		!p.multi=[0,2,2]
		!p.charsize=1
		if strtrim(obht(obsindex(0)).insopt1) eq 'SCI_PHOT' then begin
			!p.multi=[0,4,2]
			!p.charsize=2.0
		endif
		!p.psym=0
		if obht(obsindex(0)).insopt1 eq 'HIGH_SENS' $
		then m=obj_new('miamask',obht(obsindex(0)).filename, $
		    	obht(obsindex(1:2)).filename,noewsphot=0,fitmask=2, $
			dsky=midi_options.s) $
		else m=obj_new('miamask',obht(obsindex(0)).filename, $
			noewsphot=0,fitmask=2,dsky=midi_options.s)
		m->gui
		m->write_mask
		midimaskgui,/update
		maskfile=m->get_mask_name()
		localfile=file_basename(maskfile)
		widget_control,ab_files_wid,get_uvalue=masks
		for i=0,n_elements(masks)-1 do begin
			words=nameparse(masks(i))
			masks(i)=words(0)
		endfor
		widget_control,ab_files_wid,set_list_select=where(masks eq localfile)
		if n_elements(recipes) gt 0 then begin
			i=where(recipes.track_file eq obht(obsindex(0)).filename,count)
			if count eq 1 then recipes(i).mask_file=maskfile
		endif
		end
'Store':	begin
;		i=strpos(maskfile,'Mask_')
;		phot_a_tag=strmid(maskfile,i+5,23)
;		phot_b_tag=strmid(maskfile,i+29,12)
;		j=where(strpos(recipes.phot_a_file,phot_a_tag) ge 0 $
;		    and strpos(recipes.phot_b_file,phot_b_tag) ge 0,count)
		if n_elements(recipe_tag) eq 0 then return,-1
		j=where(recipes.tag eq recipe_tag,count)
		if count eq 1 then begin
			recipes(j).mask_file=maskfile
			print,'Stored maskfile info in recipe '+recipes(j).tag
		endif
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro midimaskgui,update=update
;
; GUI to select and manage masks.
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
;
if n_elements(update) eq 0 then update=0
;
maskfile=''
spawn,'pwd',currentdir
;
; First, find all mask files in directory PHOTOMETRY
mask_files=findfile('PHOTOMETRY/Mask_*.fits')
n=n_elements(mask_files)
if strlen(mask_files(0)) eq 0 then begin
	print,'No mask files found in '+currentdir+'/PHOTOMETRY!'
	shutters=''
	nrtsmode=''
endif else begin
;
; Derive the file names of their respective raw data files
raw_files=strarr(n)
i=strpos(mask_files,'Mask_')
shutters=strarr(n)
objects=strarr(n)
nrtsmode=strarr(n)
for j=0,n-1 do begin
	mask_files(j)=strmid(mask_files(j),i(j),strlen(mask_files(j))-i(j))
	raw_files(j)='MIDI.'+strmid(mask_files(j),5,strlen(mask_files(j))-5)
	if strpos(raw_files(j),'+') eq -1 then begin
		if strlen(findfile(raw_files(j))) gt 0 then begin
      			im = obj_new('imagedata',raw_files(j))
      			he = im->prihead()
      			shutters(j) = strcompress(he->getpar('INS SHUT NAME'),/remove_all)
      			objects(j) = strcompress(he->getpar('OBS TARG NAME'),/remove_all)
      			nrtsmode(j) = strcompress(he->getpar('DET NRTS MODE'),/remove_all)
			obj_destroy,im
		endif
	endif else begin
		words=nameparse(raw_files(j),'+')
		ipos=strpos(words(0),'.Aphotometry')
		if ipos ge 0 then words(0)=strmid(words(0),0,ipos)
		if strlen(findfile(words(0)+'.fits')) gt 0 then begin
			im = obj_new('imagedata',words(0)+'.fits')
      			he = im->prihead()
			shutters(j)='ABOPEN'
      			objects(j) = strcompress(he->getpar('OBS TARG NAME'),/remove_all)
      			nrtsmode(j) = strcompress(he->getpar('DET NRTS MODE'),/remove_all)
			obj_destroy,im
		endif
	endelse
endfor
endelse
index=where(shutters eq 'ABOPEN' and nrtsmode eq 'DEFAULT_CHOP',count)
index=where(shutters eq 'ABOPEN' and nrtsmode ne 'ACQ_UT_COARSE_CHOP',count)
if count gt 0 then abmaskfiles=mask_files(index)+' ('+objects(index)+')' else abmaskfiles=strarr(n)+' '
abmaskfiles=['minrtsMask_FIELD_PRISM_HIGH_SENS_SLIT.fits', $
	     'minrtsMask_FIELD_PRISM_SCI_PHOT_SLIT.fits', $
	     'minrtsMask_SPECTRAL_GRISM_HIGH_SENS_SLIT.fits', $
	     'minrtsMask_SPECTRAL_GRISM_SCI_PHOT_SLIT.fits', $
	    abmaskfiles]
index=where(shutters eq 'AOPEN',count)
if count gt 0 then amaskfiles=mask_files(index)+' ('+objects(index)+')' else amaskfiles=strarr(n)+' '
index=where(shutters eq 'BOPEN',count)
if count gt 0 then bmaskfiles=mask_files(index)+' ('+objects(index)+')' else bmaskfiles=strarr(n)+' '
;
if update and widget_info(midimask_wid,/valid) then begin
	widget_control,ab_files_wid,set_value=abmaskfiles,set_uvalue=abmaskfiles
	widget_control,a_files_wid,set_value=amaskfiles,set_uvalue=amaskfiles
	widget_control,b_files_wid,set_value=bmaskfiles,set_uvalue=bmaskfiles
	maskfile=''
	print,'Maskfile selection cleared.'
endif else begin
;
; Remove any existing widget
if widget_info(midimask_wid,/valid) then widget_control,midimask_wid,/destroy
;
midimask_wid=widget_base(/col,title='Mask Tracker '+currentdir+'/PHOTOMETRY')
;
draw_wid=widget_draw(midimask_wid,event_pro='maskdraw_event', $
	scr_xsize=600,scr_ysize=200*2)
;
ab_files_wid=widget_list(midimask_wid,value=abmaskfiles,uvalue=abmaskfiles, $
        event_func='midimaskfiles_event',xsize=25,ysize=10)
;
midimaskrow_wid=widget_base(midimask_wid,/row)
;
midimaskcol_wid=widget_base(midimaskrow_wid,/col)
label_wid=widget_label(midimaskcol_wid,value='Photometry A')
a_files_wid=widget_list(midimaskcol_wid,value=amaskfiles,uvalue=amaskfiles, $
	event_func='midimaskfiles_event',xsize=40,ysize=5)
;
midimaskcol_wid=widget_base(midimaskrow_wid,/col)
label_wid=widget_label(midimaskcol_wid,value='Photometry B')
b_files_wid=widget_list(midimaskcol_wid,value=bmaskfiles,uvalue=bmaskfiles, $
	event_func='midimaskfiles_event',xsize=40,ysize=5)
;
tools=['Update/Clear','Gui','Store']
button_wid=cw_bgroup(midimask_wid,/row,tools, $
	event_funct='midimaskgui_event',/return_name)
;
widget_control,midimask_wid,/realize
xmanager,'midimaskgui',midimask_wid,/no_block
;
endelse
;
end
;-------------------------------------------------------------------------------
function midirecipes_event,event
;
; Call-back for clicks on recipes files in the midirecipesgui.
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiRecipesGuiWids,recipes_wid,recipetext_wid,recipe_tag
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
;
r=size(event)
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=tags
	if event.index lt n_elements(tags) then $
	recipe_tag=tags(event.index) & recipe_tag=recipe_tag(0)
endif
;
index=where(recipes.tag eq recipe_tag,count)
if count gt 0 then begin
	fields=tag_names(recipes)
	for i=0,n_elements(fields)-1 do fields(i)=fields(i)+'='+recipes(index).(i)
	widget_control,recipetext_wid,set_value=fields
endif
;
end
;-------------------------------------------------------------------------------
function midirecipesgui_event,event
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiRecipesGuiWids,recipes_wid,recipetext_wid,recipe_tag
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
;
if n_elements(recipes_wid) eq 0 then recipes_wid=0L
recipes_file='Recipes_'+configid+'.xdr'
;
case event.value of
;
'Update':	begin
		tags=recipes.tag
		if widget_info(recipes_wid,/valid) ne 0 then begin
			widget_control,recipes_wid,set_value=tags,set_uvalue=tags
			widget_control,recipetext_wid,set_value='Click to see recipe'
		endif
		recipe_tag=''
		save,recipes,filename=recipes_file
		print,'Current recipes saved to: '+recipes_file
		end
'Default':	begin
		status=defaultrecipes()
		r=midirecipesgui_event({value:'Update'})
		end
'Get masks':	begin
;		Given the names of the photometry files, find mask file
		if n_elements(ab_files_wid) eq 0 then ab_files_wid=0
		if widget_info(ab_files_wid,/valid) eq 0 then begin
			print,'***Error: Please start the mask tracker first!'
			return,-1
		endif
		widget_control,ab_files_wid,get_uvalue=masks
		for i=0,n_elements(recipes)-1 do begin
			mask='Mask_'+strmid(recipes(i).phot_a_file,5,23)+'+' $
				+strmid(recipes(i).phot_b_file,16,12)
			index=where(strpos(masks,mask) ge 0,count)
			if count eq 1 then begin
				words=nameparse(masks(index(0)))
				recipes(i).mask_file=words(0)
			endif
		endfor
		print,'Stored maskfile info in recipes.'
		r=midirecipesgui_event({value:'Update'})
		end
'Clear masks':	begin
		recipes.mask_file=''
		print,'Maskfiles cleared in recipes.'
		r=midirecipesgui_event({value:'Update'})
		end
'HighSens':	begin
		recipes(where(recipes.tag eq recipe_tag)).photometry='HighSens'
		r=midirecipesgui_event({value:'Update'})
		end
'Scaled':	begin
		i=where(recipes.tag eq recipe_tag)
		recipes(i).photometry='Scaled'
		recipes(i).photo_file=photofile.file
		recipes(i).phot_a_file=photofile.files(0)
		recipes(i).phot_b_file=photofile.files(1)
		r=midirecipesgui_event({value:'Update'})
		end
'SciPhot':	begin
		recipes(where(recipes.tag eq recipe_tag)).photometry='SciPhot'
		r=midirecipesgui_event({value:'Update'})
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro midirecipesgui
;
; GUI to select and manage recipes.
;
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiMaskGuiWids,ab_files_wid,a_files_wid,b_files_wid,draw_wid
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiRecipesGuiWids,recipes_wid,recipetext_wid,recipe_tag
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
;
; Remove any existing widget
if widget_info(midirecipes_wid,/valid) then widget_control,midirecipes_wid,/destroy
;
recipe_tag=''
;
midirecipes_wid=widget_base(/col,title='Recipes')
;
midirow_wid=widget_base(midirecipes_wid,/row)
;
tags=recipes.tag
recipes_wid=widget_list(midirow_wid,value=tags,uvalue=tags, $
        event_func='midirecipes_event',xsize=25,ysize=10)
;
tools=['Update','Default','Get masks','Clear masks','HighSens','Scaled','SciPhot']
button_wid=cw_bgroup(midirow_wid,/col,tools, $
	event_funct='midirecipesgui_event',/return_name)
;
recipetext_wid=widget_text(midirecipes_wid,value='Click to see recipe', $
	/scroll,ysize=10)
;
widget_control,midirecipes_wid,/realize
xmanager,'midirecipesgui',midirecipes_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function midiphogui_event,event
;
; Callback for midiphogui. Note that it gets called both by a button
; compound widget and a drop list, which have different event structures.
;
; Derives images from two sources: MIA routines and mymidigui, the former 
; are related to "ON-OFF" requests, the latter to "chopped" requests.
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiPhoGuiWids,draw_wid
common MidiPhoSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
widget_control,draw_wid,get_value=id
wset,id
set_screen
;
widget_control,i_wid,sensitive=0
widget_control,j_wid,sensitive=0
;
widget_control,event.id,get_uvalue=tools
if n_elements(tools) eq 1 then value=event.value else value=tools(event.index)
event={value:value}
;
; These recycle code but we need to know the channel selected
beamcombiner=obht(phoindex(0)).insopt1
if strpos(value,'BeamJitter') ge 0 then begin
	value='BeamJitter'
	if strpos(event.value,'A') ge 0 then data_id=4
	if strpos(event.value,'B') ge 0 then data_id=1
	if beamcombiner eq 'SCI_PHOT' then begin
		if strpos(event.value,'1') ge 0 then data_id=2
		if strpos(event.value,'2') ge 0 then data_id=3
	endif else begin
		if strpos(event.value,'1') ge 0 then data_id=1
		if strpos(event.value,'2') ge 0 then data_id=2
	endelse
endif
if strpos(value,'BinnedAvg') ge 0 then begin
	value='BinnedAvg'
	if strpos(event.value,'A') ge 0 then data_id=4
	if strpos(event.value,'B') ge 0 then data_id=1
	if beamcombiner eq 'SCI_PHOT' then begin
		if strpos(event.value,'1') ge 0 then data_id=2
		if strpos(event.value,'2') ge 0 then data_id=3
	endif else begin
		if strpos(event.value,'1') ge 0 then data_id=1
		if strpos(event.value,'2') ge 0 then data_id=2
	endelse
endif
if strpos(value,'Frames') ge 0 then begin
	if strpos(value,'aligned') lt 0 then value='Frames'
	if strpos(event.value,'A') ge 0 then data_id=4
	if strpos(event.value,'B') ge 0 then data_id=1
	if beamcombiner eq 'SCI_PHOT' then begin
		if strpos(event.value,'1') ge 0 then data_id=2
		if strpos(event.value,'2') ge 0 then data_id=3
	endif else begin
		if strpos(event.value,'1') ge 0 then data_id=1
		if strpos(event.value,'2') ge 0 then data_id=2
	endelse
endif
;
; This section deals with frames and plots
set_screen
case value of
'Beam I1 Offset':begin
	index=where(phodata1.c ne 0 and phodata1.t eq 'T ')
	plot,phodata1.x(index),psym=3,xtitle='Target frame ID',ytitle='Offset (I1)'
	return,0
	end
'Beam I2 Offset':begin
	index=where(phodata2.c ne 0 and phodata2.t eq 'T ')
	plot,phodata2.x(index),psym=3,xtitle='Target frame ID',ytitle='Offset (I2)'
	return,0
	end
'Beam I1 Flux':	begin
	index=where(phodata1.c ne 0 and phodata1.t eq 'T ')
	plot,phodata1.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (I1)'
	v=medianve(phodata1.f(index),e)
	print,'Median: ',v,', median error: ',e
	return,0
	end
'Beam I2 Flux':	begin
	index=where(phodata2.c ne 0 and phodata2.t eq 'T ')
	plot,phodata2.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (I2)'
	v=medianve(phodata2.f(index),e)
	print,'Median: ',v,', median error: ',e
	return,0
	end
'BeamJitter':begin
	frames=read_frames(obht(phoindex(0)).filename,data_id)
	r=photometry_jitter(frames)
	v=medianve(r,e)
	plot,r,psym=3,yrange=[v-10*e,v+10*e], $
		xtitle='Frame',ytitle='Offset [pixels]'
	return,0
	end
'Frames':begin
	if n_elements(phoindex) eq 2 then return,-1
	frames=read_frames(obht(phoindex(0)).filename,data_id,labels)
	r=size(frames)
	nrows=r(3)
	index=where(labels eq 'S ')
	sky=total(frames(*,*,index),3)/n_elements(index)
	for i=0,nrows-1 do frames(*,*,i)=frames(*,*,i)-sky
	widget_control,i_wid,/sensitive
	widget_control,j_wid,/sensitive
	midiphogui2_event,'Init'
	return,0
	end
'Frames I1 (aligned)':begin
	index=where(phodata1.c ne 0 and phodata1.t eq 'T ')
	frames=phodata1.af(*,*,index)
	labels=phodata1.t(index)
	widget_control,i_wid,/sensitive
	widget_control,j_wid,/sensitive
	midiphogui2_event,'Init'
	return,0
	end
'Frames I2 (aligned)':begin
	index=where(phodata2.c ne 0 and phodata2.t eq 'T ')
	frames=phodata2.af(*,*,index)
	labels=phodata2.t(index)
	widget_control,i_wid,/sensitive
	widget_control,j_wid,/sensitive
	midiphogui2_event,'Init'
	return,0
	end
'Compute masks':begin
	loadct,0
	if n_elements(phoindex) eq 2 then $
	chop_nod_disp_obsolete,/silent, $
		obht(phoindex(1)).filename,obht(phoindex(0)).filename, $
		width=width,trace_order=t_order,fwhm_order=w_order else $
	chop_nod_disp_obsolete,/silent, $
		obht(phoindex(0)).filename, $
		width=width,trace_order=t_order,fwhm_order=w_order
	return,0
	r=chop_images(obht(phoindex).filename)
	chop_nod_disp_one_window_obsolete,r.on1-r.off1,width,t_order,w_order, $
		traceI1, Flux_I1, Flux_I1c, weighting_win1, DEBUG=debug, SILENT=silent
	return,0
	end
'Display spectra':begin
	if n_elements(phoindex) eq 2 then return,-1
	mask=findfile(maskfile)
	if n_elements(mask) ne 1 then mask='' else mask=mask(0)
	if strlen(mask) eq 0 then begin
		print,'Please select a mask first!'
		return,-1
	endif
	mask=oirgetdata(mask)
	if n_elements(where(strpos(tag_names(mask),'DATA') ge 0)) eq 4 then SCI_PHOT=1 $
								       else SCI_PHOT=0
	do_photometry,obht(phoindex).filename,mask,PhFilt,spec1,spec2,specph
	dispersion=strtrim(obht(obsindex(0)).insgris)
	win1nx=detwin(0)
	win1strx=detwin(1)
	wavelengths=pix2lambda(dindgen(win1nx)+win1strx,dispersion)
	if SCI_PHOT then !p.multi=[0,1,2]
	title='Interferometric channel 1 (r) /2 (g)'
	plot,wavelengths,spec1,/nodata,title=title,xtitle='Wavelength [microns]', $
		xrange=[7,13],xstyle=1,yrange=[min([spec1,spec2]),max([spec1,spec2])]
	oplot,wavelengths,spec1,color=2
	oplot,wavelengths,spec2,color=3
	oplot,wavelengths,wavelengths*0
	if SCI_PHOT then begin
		title='Photometric channel'
		plot,wavelengths,specph,title=title,xtitle='Wavelength [microns]', $
			xrange=[7,13],xstyle=1
		oplot,wavelengths,wavelengths*0
	endif
	return,0
	end
'k1A k2A':begin
	mask=findfile(maskfile)
	if n_elements(mask) ne 1 then mask='' else mask=mask(0)
	if strlen(mask) eq 0 then begin
		print,'Please select a mask first!'
		return,-1
	endif
	mask=oirgetdata(mask)
	if n_elements(where(strpos(tag_names(mask),'DATA') ge 0)) eq 4 then SCI_PHOT=1 $
								       else SCI_PHOT=0
	do_photometry,obht(phoindex).filename,mask,PhFilt,spec1,spec2,specph
	dispersion=strtrim(obht(obsindex(0)).insgris)
	win1nx=detwin(0)
	win1strx=detwin(1)
	wavelengths=pix2lambda(dindgen(win1nx)+win1strx,dispersion)
	!p.multi=0
	plot,wavelengths,spec1,/nodata,title='k1A (r) k2A (g)', $
		xtitle="Wavelength [microns]",ytitle='kappa matrix', $
		xrange=[7,13],yrange=[0,2],xstyle=1
	oplot,wavelengths,spec1/specph > 0,color=2
	oplot,wavelengths,spec2/specph > 0,color=3
	return,0
	end
'k1B k2B':begin
	mask=findfile(maskfile)
	if n_elements(mask) ne 1 then mask='' else mask=mask(0)
	if strlen(mask) eq 0 then begin
		print,'Please select a mask first!'
		return,-1
	endif
	mask=oirgetdata(mask)
	if n_elements(where(strpos(tag_names(mask),'DATA') ge 0)) eq 4 then SCI_PHOT=1 $
								       else SCI_PHOT=0
	do_photometry,obht(phoindex).filename,mask,PhFilt,spec1,spec2,specph
	dispersion=strtrim(obht(obsindex(0)).insgris)
	win1nx=detwin(0)
	win1strx=detwin(1)
	wavelengths=pix2lambda(dindgen(win1nx)+win1strx,dispersion)
	!p.multi=0
	plot,wavelengths,spec1,/nodata,title='k1B (r) k2B (g)', $
		xtitle="Wavelength [microns]",ytitle='kappa matrix', $
		xrange=[7,13],yrange=[0,2],xstyle=1
	oplot,wavelengths,spec1/specph > 0,color=2
	oplot,wavelengths,spec2/specph > 0,color=3
	return,0
	end
'Store':begin
	if n_elements(scans) eq 0 then begin
		print,'You must define data set first!'
		return,-1
	endif
	mask=findfile(maskfile)
	if n_elements(mask) ne 1 then mask='' else mask=mask(0)
	if strlen(mask) eq 0 then begin
		print,'Please select a mask first!'
		return,-1
	endif
	mask=oirgetdata(mask)
	do_photometry,obht(phoindex).filename,mask,PhFilt,spec1,spec2
;
;	Find out which scan corresponds to this observation
	iobs=where(recipes.track_file eq obht(obsindex(0)).filename)
	midiupdateconfig,GenConfig
;
	spec1=xmdvavg(spec1,la_bins)
	spec2=xmdvavg(spec2,la_bins)
	case strtrim(obht(phoindex).insshut) of
		'AOPEN':ibeam=0
		'BOPEN':ibeam=1
	endcase
	scans(iobs).photometry(ibeam,0,0:n_elements(spec1)-1)=spec1
	scans(iobs).photometry(ibeam,1,0:n_elements(spec2)-1)=spec2
	scans(iobs).photometryerr(ibeam,0,0:n_elements(spec1)-1)=1
	scans(iobs).photometryerr(ibeam,1,0:n_elements(spec2)-1)=1
	return,0
	end
'Movie I1 ON':begin
	tv,bytarr(!d.x_size,!d.y_size)
	framesfile=file_basename(obht(phoindex).filename)
	framesfile='PHOTOMETRY/ChopFrames_'+strmid(framesfile,5,strlen(framesfile)-10)+'.xdr'
	ffile=findfile(framesfile)
	if strlen(ffile) gt 0 then restore,ffile else begin
		print,'Please wait, chopping frames...'
		images=midichopframes(obht(phoindex).filename)
		save,images,filename=framesfile
	endelse
	index=where(images.tartyp2 eq 'T ')
	for i=0,n_elements(index)-1 do begin
		tvscl,images(index(i)).data1
		wait,0.05
	endfor
	images=0
	return,0
	end
'Movie I1 OFF':begin
	tv,bytarr(!d.x_size,!d.y_size)
	framesfile=file_basename(obht(phoindex).filename)
	framesfile='PHOTOMETRY/ChopFrames_'+strmid(framesfile,5,strlen(framesfile)-10)+'.xdr'
	ffile=findfile(framesfile)
	if strlen(ffile) gt 0 then restore,ffile else begin
		print,'Please wait, chopping frames...'
		images=midichopframes(obht(phoindex).filename)
		save,images,filename=framesfile
	endelse
	index=where(images.tartyp2 eq 'S ' or images.tartyp2 eq 'U ')
	for i=0,n_elements(index)-1 do begin
		tvscl,images(index(i)).data1
		wait,0.05
	endfor
	images=0
	return,0
	end
'Movie I2 ON':begin
	tv,bytarr(!d.x_size,!d.y_size)
	framesfile=file_basename(obht(phoindex).filename)
	framesfile='PHOTOMETRY/ChopFrames_'+strmid(framesfile,5,strlen(framesfile)-10)+'.xdr'
	ffile=findfile(framesfile)
	if strlen(ffile) gt 0 then restore,ffile else begin
		print,'Please wait, chopping frames...'
		images=midichopframes(obht(phoindex).filename)
		save,images,filename=framesfile
	endelse
	index=where(images.tartyp2 eq 'T ')
	for i=0,n_elements(index)-1 do begin
		tvscl,images(index(i)).data2
		wait,0.05
	endfor
	images=0
	return,0
	end
'Movie I2 OFF':begin
	tv,bytarr(!d.x_size,!d.y_size)
	framesfile=file_basename(obht(phoindex).filename)
	framesfile='PHOTOMETRY/ChopFrames_'+strmid(framesfile,5,strlen(framesfile)-10)+'.xdr'
	ffile=findfile(framesfile)
	if strlen(ffile) gt 0 then restore,ffile else begin
		print,'Please wait, chopping frames...'
		images=midichopframes(obht(phoindex).filename)
		save,images,filename=framesfile
	endelse
	index=where(images.tartyp2 eq 'S ' or images.tartyp2 eq 'U ')
	for i=0,n_elements(index)-1 do begin
		tvscl,images(index(i)).data2
		wait,0.05
	endfor
	images=0
	return,0
	end
else:
endcase
;
if 0 then begin
if strpos(event.value,'RMS') ge 0 then begin
	rmsfile=findfile('PHOTOMETRY/'+obht(phoindex).filename+'.ON-OFF')
	if strlen(rmsfile) gt 0 then restore,rmsfile else begin
		print,'Please wait, chopping frames...'
		zz=midichopframes(obht(phoindex).filename)
		save,zz,filename=rmsfile
	endelse
endif else if strpos(event.value,'Beam') ge 0 then begin
	chopfile=findfile('PHOTOMETRY/'+obht(phoindex).filename+'.ONOFF')
	if strlen(chopfile) gt 0 then restore,chopfile else begin
		z=midiChopImage(obht(phoindex).filename,before=3,after=2)
		save,z,filename=chopfile
	endelse
endif
endif
;
; This section deals with all images
if strpos(event.value,'Chopped') lt 0 then begin
if n_elements(phoindex) eq 2 then begin
	tindex=where(phodata1.t eq 'T ',tcount)
	sindex=where(phodata1.t eq 'S ',scount)
	on1=total(phodata1.cf(*,*,tindex),3)/tcount
	off1=total(phodata1.cf(*,*,sindex),3)/scount
	tindex=where(phodata2.t eq 'T ',tcount)
	sindex=where(phodata2.t eq 'S ',scount)
	on2=total(phodata2.cf(*,*,tindex),3)/tcount
	off2=total(phodata2.cf(*,*,sindex),3)/scount
	images={on1:on1,off1:off1,on2:on2,off2:off2}
endif else begin
	images=chop_images(obht(phoindex).filename)
endelse
endif
;
case value of
'Beam I1 ON':	 	image=images.on1
'Beam I1 OFF':		image=images.off1
'Beam I1 ON-OFF':	begin
			image=images.on1-images.off1
			print,'Total mean flux I1 = ',total(image)
			end
'PA ON-OFF':		image=images.on4-images.off4
'Beam I2 ON':		image=shift(images.on2,-1,1)
'Beam I2 OFF':		image=shift(images.off2,-1,1)
'Beam I2 ON-OFF':	begin
			image=shift(images.on2-images.off2,-1,1)
			print,'Total mean flux I2 = ',total(image)
			end
'PB ON-OFF':		image=images.on3-images.off3
'Beam I1 Chopped':	begin
			index=where(phodata1.t eq 'T ' and phodata1.c ne 0,count)
			image=total(phodata1.cf(*,*,index),3)/count
			end
'Beam I2 Chopped':	begin
			index=where(phodata2.t eq 'T ' and phodata2.c ne 0,count)
			image=total(phodata2.cf(*,*,index),3)/count
			end
'BinnedAvg':		begin
			frames=read_frames(obht(phoindex(0)).filename,data_id,targets)
			jitterbinning,frames,targets,image
			end
'I1 ON-OFF RMS':	begin
			index=where(zz.tartyp2 eq 'T ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1,nj)
			mask_on=intarr(n)+1
			for j=-2,3 do mask_on(jndex+j)=0
			mask_on(0:2)=0 & mask_on(n-3:n-1)=0
			r=size(zz.data1)
			on_frames=fltarr(r(1),r(2),nj+1)
			k=0
			nsum=0
			for i=0,n-2 do begin
			if mask_on(i) eq 1 then begin
				on_frames(*,*,k)=on_frames(*,*,k)+zz(index(i)).data1
				nsum=nsum+1
			endif
			if index(i+1)-index(i) gt 1 then begin
				on_frames(*,*,k)=on_frames(*,*,k)/nsum
				k=k+1
				nsum=0
			endif
			endfor
			on_frames=on_frames(*,*,0:k-2)
			index=where(zz.tartyp2 eq 'S ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1,nj)
			mask_off=intarr(n)+1
			for j=-2,3 do mask_off(jndex+j)=0
			mask_off(0:2)=0 & mask_off(n-3:n-1)=0
			r=size(zz.data1)
			off_frames=fltarr(r(1),r(2),nj+1)
			k=0
			nsum=0
			for i=0,n-2 do begin
			if mask_off(i) eq 1 then begin
				off_frames(*,*,k)=off_frames(*,*,k)+zz(index(i)).data1
				nsum=nsum+1
			endif
			if index(i+1)-index(i) gt 1 then begin
				off_frames(*,*,k)=off_frames(*,*,k)/nsum
				k=k+1
				nsum=0
			endif
			endfor
			target=summe(on_frames,2,sdev=image)
			print,'Total ON-OFF RMS I1 = ',sqrt(total(image^2))
			end
'I1 Sky RMS':		begin
			index=where(zz.tartyp2 eq 'S ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1)
			mask=intarr(n)+1
			for j=-2,3 do mask(jndex+j)=0
			index=index(where(mask eq 1))
			sky=summe(zz(index).data1,2,sdev=image)
			print,'Mean Sky RMS I1 = ',mean(image)
			end
'I2 ON-OFF RMS':	begin
			index=where(zz.tartyp2 eq 'T ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1,nj)
			mask_on=intarr(n)+1
			for j=-2,3 do mask_on(jndex+j)=0
			mask_on(0:2)=0 & mask_on(n-3:n-1)=0
			r=size(zz.data2)
			on_frames=fltarr(r(1),r(2),nj+1)
			k=0
			nsum=0
			for i=0,n-2 do begin
			if mask_on(i) eq 1 then begin
				on_frames(*,*,k)=on_frames(*,*,k)+zz(index(i)).data2
				nsum=nsum+1
			endif
			if index(i+1)-index(i) gt 1 then begin
				on_frames(*,*,k)=on_frames(*,*,k)/nsum
				k=k+1
				nsum=0
			endif
			endfor
			on_frames=on_frames(*,*,0:k-2)
			index=where(zz.tartyp2 eq 'S ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1,nj)
			mask_off=intarr(n)+1
			for j=-2,3 do mask_off(jndex+j)=0
			mask_off(0:2)=0 & mask_off(n-3:n-1)=0
			r=size(zz.data2)
			off_frames=fltarr(r(1),r(2),nj+1)
			k=0
			nsum=0
			for i=0,n-2 do begin
			if mask_off(i) eq 1 then begin
				off_frames(*,*,k)=off_frames(*,*,k)+zz(index(i)).data2
				nsum=nsum+1
			endif
			if index(i+1)-index(i) gt 1 then begin
				off_frames(*,*,k)=off_frames(*,*,k)/nsum
				k=k+1
				nsum=0
			endif
			endfor
			target=summe(on_frames,2,sdev=image)
			print,'Total ON-OFF RMS I2 = ',sqrt(total(image^2))
			end
'I2 Sky RMS':		begin
			index=where(zz.tartyp2 eq 'S ',n)
			jndex=where(index(1:n-1)-index(0:n-2) gt 1)
			mask=intarr(n)+1
			for j=-2,3 do mask(jndex+j)=0
			index=index(where(mask eq 1))
			sky=summe(zz(index).data2,2,sdev=image)
			print,'Mean Sky RMS I2 = ',mean(image)
			end
else:
endcase
;
; Now we display the image
if n_elements(image) ne 0 then $
displaychoppedspectrum,image,strtrim(obht(phoindex(0)).insgris,2)
;
end
;-------------------------------------------------------------------------------
pro midiphogui
;
; GUI for photometry analysis. Does only one beam at a time.
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiPhoGuiWids,draw_wid
common MidiPhoSlider,i_wid,j_wid,i_value,j_value,frames,labels
;
; Remove any existing widget
if widget_info(midipho_wid,/valid) then widget_control,midipho_wid,/destroy
;
fitsfiles=nameparse(strjoin(obht(phoindex(0)).filename,' '))
fitsfile=obj_new('fitsfile',fitsfiles(0))
prihead=fitsfile->prihead()
win1nx=prihead->getpar('DET WIN1 NX')
win1strx=prihead->getpar('DET WIN1 STRX')
detwin=[win1nx,win1strx]
obj_destroy,prihead
obj_destroy,fitsfile
la_bins(1) = la_bins(1) < win1nx
;
; Analysis using mymidigui functions/procedures for I1/I2 channels
beamcombiner=strtrim(obht(phoindex(0)).insopt1)
if beamcombiner eq 'SCI_PHOT' then dataid=[2,3] else dataid=[1,2]
print,'Reading and analyzing photometry I1 frames...'
if n_elements(phoindex) eq 2 then begin
	rfs=read_frames(obht(phoindex(0)).filename,dataid(0),ts)
	rft=read_frames(obht(phoindex(1)).filename,dataid(0),tt)
	nf=n_elements(ts)+n_elements(tt)
	onoff=total(rft-rfs,3)
	if max(onoff) lt abs(min(onoff)) then begin
		ts(*)='T '
		tt(*)='S '
		phoindex=reverse(phoindex)
	endif else begin
		ts(*)='S '
		tt(*)='T '
	endelse
	t=[tt,ts]
	rf=[[[rft]],[[rfs]]]
	c=intarr(nf)+1
endif else begin
	phodata1=0	; Release memory
	rf=read_frames(obht(phoindex(0)).filename,dataid(0),t)
	c=chop_cycles(t)
endelse
cf=chop_frames(rf,t,c)
f=analyze_phoframes(cf,t,c,x,a,xmed)
af=align_phoframes(cf,t,c,x,xmed)
phodata1={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
print,'Reading and analyzing photometry I2 frames...'
if n_elements(phoindex) eq 2 then begin
	rfs=read_frames(obht(phoindex(0)).filename,dataid(1),ts)
	rft=read_frames(obht(phoindex(1)).filename,dataid(1),tt)
	nf=n_elements(ts)+n_elements(tt)
	onoff=total(rft-rfs,3)
	if max(onoff) lt abs(min(onoff)) then begin
		ts(*)='T '
		tt(*)='S '
	endif else begin
		ts(*)='S '
		tt(*)='T '
	endelse
	t=[tt,ts]
	rf=[[[rft]],[[rfs]]]
	c=intarr(nf)+1
endif else begin
	phodata2=0	; Release memory
	rf=read_frames(obht(phoindex(0)).filename,dataid(1),t)
	c=chop_cycles(t)
endelse
cf=chop_frames(rf,t,c)
f=analyze_phoframes(cf,t,c,x,a,xmed)
af=align_phoframes(cf,t,c,x,xmed)
phodata2={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
;
midipho_wid=widget_base(/col, $
	   title=string(obht(phoindex(0)).obstargname)+' ' $
		+obht(phoindex(0)).insshut)
;
midiphorow_wid=widget_base(midipho_wid,/row)
midiphocol_wid=widget_base(midiphorow_wid,/col)
tools1=['Beam I1 ON', $
       	'Beam I1 OFF', $
	'BeamJitter I1', $
       	'Beam I1 ON-OFF', $
	'Beam I1 Chopped', $
	'Beam I1 Flux', $
	'Beam I1 Offset', $
	'BinnedAvg I1', $
	'Frames I1', $
	'Frames I1 (aligned)']
;button_wid=cw_bgroup(midiphorow_wid,/col,tools1, $
;	event_funct='midiphogui_event',/return_name)
opmenu_wid=widget_droplist(midiphocol_wid,event_func='midiphogui_event', $
	uvalue=tools1,value=tools1)
tools2=['Beam I2 ON', $
       	'Beam I2 OFF', $
	'BeamJitter I2', $
       	'Beam I2 ON-OFF', $
	'Beam I2 Chopped', $
	'Beam I2 Flux', $
	'Beam I2 Offset', $
	'BinnedAvg I2', $
	'Frames I2', $
	'Frames I2 (aligned)']
;button_wid=cw_bgroup(midiphorow_wid,/col,tools2, $
;	event_funct='midiphogui_event',/return_name)
opmenu_wid=widget_droplist(midiphocol_wid,event_func='midiphogui_event', $
	uvalue=tools2,value=tools2)
if obht(phoindex(0)).insopt1 eq 'SCI_PHOT' then begin
	if strtrim(obht(phoindex(0)).insshut,2) eq 'AOPEN' then $
		tools=['PA ON-OFF','Frames PA','BinnedAvg PA']
	if strtrim(obht(phoindex(0)).insshut,2) eq 'BOPEN' then $
		tools=['PB ON-OFF','Frames PB','BinnedAvg PB']
	opmenu_wid=widget_droplist(midiphocol_wid,event_func='midiphogui_event', $
	        uvalue=tools,value=tools)
endif
if obht(phoindex(0)).insopt1 eq 'SCI_PHOT' then begin
	if strtrim(obht(phoindex(0)).insshut,2) eq 'AOPEN' then button='k1A k2A'
	if strtrim(obht(phoindex(0)).insshut,2) eq 'BOPEN' then button='k1B k2B'
endif else button=''
tools3=['Compute masks', $
	'Display spectra', $
	button, $
	'Store']
tools3=tools3(where(strlen(tools3) gt 0))
button_wid=cw_bgroup(midiphorow_wid,/col,tools3, $
	event_funct='midiphogui_event',/return_name)
;
draw_wid=widget_draw(midipho_wid,event_pro='phodraw_event', $
	scr_xsize=300,scr_ysize=200)
count=5000
i_wid=cw_fslider(midipho_wid,minimum=1,maximum=count,value=1,/drag, $
	title='Index I',xsize=256,format='(i4)',uvalue='i')
j_wid=cw_fslider(midipho_wid,minimum=0,maximum=100,value=0,/drag, $
	title='Index J',xsize=256,format='(i3)',uvalue='j')
;
widget_control,midipho_wid,/realize
xmanager,'midiphogui2',midipho_wid,/no_block
	;
widget_control,i_wid,sensitive=0
widget_control,j_wid,sensitive=0
;
end
;-------------------------------------------------------------------------------
pro midiphogui2_event,event
;
common MidiPhoSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiPhoGuiWids,draw_wid
;
result=size(event)
if result(n_elements(result)-2) ne 8 then begin
	value=1
	i_value=1
	j_value=0
	widget_control,draw_wid,get_value=id
	erase
	loadct,0
endif else begin
	widget_control,event.id,get_uvalue=slider,get_value=value
	case slider of
		'i':i_value=long(value)
		'j':j_value=long(value)
	endcase
endelse
;
index=i_value+j_value-1
if index ge n_elements(labels) then return
image=frames(*,*,index)
;
widget_control,draw_wid,get_value=id
wset,id
tvscl,image
xyouts,20,20,labels(index),/device
;
end
;-------------------------------------------------------------------------------
function miditrackgui_event,event
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiXmdv,xmdvobj,xmdvgui_wid
common MidiTrackGuiLocal,window_slide_wid
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiTrackGuiWids,draw_wid,button2h_wid,button2l_wid
common MidiTrackSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(maskfile) eq 0 then maskfile=''
;
widget_control,event.id,get_uvalue=method
widget_control,event.id,get_uvalue=plots
if n_elements(plots) eq 1 then value=event.value else value=plots(event.index)
event={value:value}
;
target=strtrim(obht(obsindex(0)).obstargname,2)
target=target+'.'+strmid(obht(obsindex(0)).filename,0,24)
filenm=obht(obsindex(0)).filename
time=strmid(filenm,strpos(filenm,'MIDI')+16,8)
dispersion=strtrim(obht(obsindex(0)).insgris)
beamcombiner=strtrim(obht(obsindex(0)).insopt1)
;
; How many frames do we have ?
nrow=0
words=nameparse(obht(obsindex(0)).filename)
for i=0,n_elements(words)-1 do begin
	fitsfile=obj_new('fitsfile',words(i))
	prihead=fitsfile->prihead()
	nrow=nrow+prihead->getpar('DET NDIT')
	obj_destroy,prihead
	obj_destroy,fitsfile
endfor
if midi_options.f then gsmooth=nrow else gsmooth=4
;
widget_control,draw_wid,get_value=id
wset,id
set_screen
!p.charsize=1
;
value=event.value
if strpos(value,'Frames') ge 0 then begin
	value='Frames'
	if strpos(event.value,'A') ge 0 then data_id=4
	if strpos(event.value,'B') ge 0 then data_id=1
	if strpos(event.value,'1') ge 0 then data_id=1
	if strpos(event.value,'2') ge 0 then data_id=2
endif
;
if strpos(value,'PA') ge 0 then begin
	r=size(trackdata1)
	if r(2) ne 8 then begin
		print,'Reading and analyzing PA frames...'
		rf=read_frames(obht(obsindex(0)).filename,1,t)
		c=chop_cycles(t)
		cf=chop_frames(rf,t,c)
		f=analyze_phoframes(cf,t,c,x,a,xmed)
		af=align_phoframes(cf,t,c,x,xmed)
		trackdata1={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
	endif
endif
if strpos(value,'PB') ge 0 then begin
	r=size(trackdata2)
	if r(2) ne 8 then begin
		print,'Reading and analyzing PB frames...'
		rf=read_frames(obht(obsindex(0)).filename,4,t)
		c=chop_cycles(t)
		cf=chop_frames(rf,t,c)
		f=analyze_phoframes(cf,t,c,x,a,xmed)
		af=align_phoframes(cf,t,c,x,xmed)
		trackdata2={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
	endif
endif
;
case value of
'Heidelberg':	begin
		IF midi_options.v THEN BEGIN
		if n_elements(obsindex) eq 1 then begin
			print,"Sorry: Heidelberg can't do SCI_PHOT w/out photometry!"
		endif else begin
;		New EWS style photometry does not work for SCI_PHOT
		if obht(obsindex(0)).insopt1 eq 'SCI_PHOT' then noews=1 else noews=0
		mask=findfile(maskfile)
		if n_elements(mask) ne 1 then mask='' else mask=mask(0)
		if strlen(mask) eq 0 then begin
			xmdvobj=xmdv(obht(obsindex).filename,noews=noews,/dwim)
		endif else begin
			print,'Using mask file: ',mask
			xmdvobj=xmdv(obht(obsindex).filename,noews=noews,maskfile=mask,/dwim)
		endelse
		endelse
		ENDIF ELSE BEGIN
		mask=findfile(maskfile)
		if n_elements(mask) ne 1 then mask='' else mask=mask(0)
		if beamcombiner eq 'SCI_PHOT' and midi_options.p ne 0 then begin
			case dispersion of
			'PRISM':kappafile='kappa.PRISM.crossCoeff.fits'
			'GRISM':kappafile='kappa.GRISM.crossCoeff.fits'
			endcase
			if n_elements(obsindex) eq 3 then begin
;				SCI_PHOT computing new kappa matrix
				midiCrossCoeff,target,obht(obsindex([1,2])).filename
				kappafile1=target+'.crossCoeff.fits'
				spawn,'cp -f '+kappafile1+' '+kappafile
			endif
		endif
		if n_elements(obsindex) eq 1 $
		or (midi_options.p eq 2 and beamcombiner eq 'SCI_PHOT') then begin
;			SCI_PHOT using previously computed kappa matrix, mode A
			r=findfile(kappafile)
			if strlen(r(0)) eq 0 then begin
				print,'***Error: kappa matrix not found: ',kappafile
				return,-1
			endif
			if strlen(mask) eq 0 $
			then xmdvobj=obj_new("xmidispvisi",obht(obsindex(0)).filename, $
				dsky=midi_options.s,cross=kappafile,/dwim,fitmask=2) $
			else xmdvobj=obj_new("xmidispvisi",obht(obsindex(0)).filename, $
				dsky=midi_options.s,cross=kappafile,/dwim,maskfile=mask)
		endif else if n_elements(obsindex) eq 3 then begin
			if beamcombiner eq 'SCI_PHOT' and midi_options.p ne 0 then begin
;				Scaled photometry, mode B
				if strlen(mask) eq 0 $
				then xmdvobj=xmdv(obht(obsindex).filename, $
					dsky=midi_options.s,cross=kappafile,/dwim,fitmask=2) $
				else xmdvobj=xmdv(obht(obsindex).filename, $
					dsky=midi_options.s,cross=kappafile,/dwim,maskfile=mask)
			endif else begin
;				HIGH_SENS or SCI_PHOT in HighSens mode, mode D
				if strlen(mask) eq 0 $
				then xmdvobj=xmdv(obht(obsindex).filename,/dwim, $
					dsky=midi_options.s,fitmask=2) $
				else xmdvobj=xmdv(obht(obsindex).filename,/dwim, $
					dsky=midi_options.s,maskfile=mask)
			endelse
		endif
		ENDELSE
		widget_control,button2h_wid,/sensitive
		end
'Lambda_gui':	begin
		!p.psym=0
		!x.style=1
		!y.style=1
		xmdvobj->lambda_gui
		la_bins=xmdvobj->get_lambda_bins()
		set_screen
		end
'Vis_gui':	begin
		!p.psym=0
		xmdvobj->gui,xmdvgui_wid
		la_bins=xmdvobj->get_lambda_bins()
		end
'Slide_gui':	begin
		!p.charsize=2
		window_slide,xsize=10000
		!p.multi=[0,1,4]
		fringes=xmdvobj->fringes()
		v=medianve(fringes(0,*,*),e)
		xmdvobj->set_opd_range,[v-3*e,v+3*e]*1e3
		xmdvobj->plot_static
		xmdvobj->set_opd_range,[0,0]
		set_screen
		!p.multi=0
		end
'Leiden':	begin
		if midi_options.v then leidenpipe $
		else begin
		mask=findfile(maskfile)
		if n_elements(mask) ne 1 then mask='' else mask=mask(0)
		if beamcombiner eq 'SCI_PHOT' and midi_options.p ne 0 then begin
			case dispersion of
			'PRISM':kappafile='kappa.PRISM.crossCoeff.fits'
			'GRISM':kappafile='kappa.GRISM.crossCoeff.fits'
			endcase
			if n_elements(obsindex) eq 3 then begin
;				SCI_PHOT computing new kappa matrix
				midiCrossCoeff,target,obht(obsindex([1,2])).filename
				kappafile1=target+'.crossCoeff.fits'
				spawn,'cp -f '+kappafile1+' '+kappafile
			endif
		endif
		if n_elements(obsindex) eq 1 $
		or (midi_options.p eq 2 and beamcombiner eq 'SCI_PHOT') then begin
			if strlen(mask) eq 0 $
			then midiSPipe,target,obht(obsindex(0)).filename,gsmooth=gsmooth,cross=kappafile $
			else midiSPipe,target,obht(obsindex(0)).filename,gsmooth=gsmooth,cross=kappafile,mask=mask
		endif else if n_elements(obsindex) eq 3 then begin
			if beamcombiner eq 'SCI_PHOT' and midi_options.p ne 0 then begin
				if strlen(mask) eq 0 $
				then midiSPipe,target,obht(obsindex).filename,gsmooth=gsmooth,cross=kappafile $
				else midiSPipe,target,obht(obsindex).filename,gsmooth=gsmooth,cross=kappafile,mask=mask
			endif else begin
				if strlen(mask) eq 0 $
				then midiPipe,target,obht(obsindex).filename,gsmooth=gsmooth $
				else midiPipe,target,obht(obsindex).filename,gsmooth=gsmooth,mask=mask
			endelse
		endif
		endelse
;		dummy.fits was created by oirFormFringes
		spawn,'rm -f /tmp/dummy.fits'
		widget_control,button2l_wid,/sensitive
		print,'Done.'
		end
'Vis_plot':	begin
		!p.charsize=1.5
		visdata=oirgetvis(target+'.redcal.fits')
		visamp=xmdvavg(visdata.visamp,la_bins)
		visampe=xmdvavg(visdata.visamperr,la_bins)
		win1nx=detwin(0)
		win1strx=detwin(1)
		wavelengths=xmdvavg(pix2lambda(dindgen(win1nx)+win1strx,dispersion), $
					la_bins)
		wset,0
		plot,wavelengths,visamp,title=target,xrange=[8,12], $
			xtitle='Wavelength [microns]',ytitle='Visibility amplitude'
		oploterr,wavelengths,visamp,visampe,3
		end
'Delay_plot':	begin
		!p.charsize=1.5
		print,'Please wait a moment...'
   		delay = oirgetdelay(target+'.groupdelay.fits')
   		delaytime=REFORM(delay.time)&delaytime=86400*(delaytime-delaytime[0])
   		opd = oirgetopd(target+'.fringes.fits',opdtime)
   		delay = reform(delay.delay)
   		if (TOTAL(opd EQ 0.) GT 0 ) then begin
      			delay(where(opd EQ 0.0)) = median(reform(delay)); bad points
      			opd(where(opd EQ 0.0)) = median(opd)  ; bad points
   		endif
		mve=medianve(opd,e)
		!p.title=target
;		if widget_info(window_slide_wid,/valid) then widget_control,window_slide_wid,/destroy
		window_slide,xsize=5000,wid=window_slide_wid
		plot,opdtime,opd,yrange=[mve-3*e,mve+6*e], $
			xtitle='Time [s]',ytitle='Delay [microns]'
		oplot,delaytime,delay,color=3
		end
'Flux_plot':	begin
		!p.charsize=1.0
		loadct,0
		photdata=oirgetdata(target+'.photometry.fits')
		spec1=xmdvavg(photdata(3).data1,la_bins)
		spec2=xmdvavg(photdata(4).data1,la_bins)
		win1nx=detwin(0)
		win1strx=detwin(1)
		wavelengths=xmdvavg(pix2lambda(dindgen(win1nx)+win1strx,dispersion), $
					la_bins)
		wset,0
		!p.multi=[0,1,2]
		plot,wavelengths,spec1,title=target,xrange=[8,12], $
			xtitle='Wavelength [microns]',ytitle='ADU Beam A'
		if midi_options.v then begin
		chopdata=oirgetdata(target+'.photo.fits')
		tvscl,chopdata(0).data1+chopdata(0).data2,0.65,0.85,/normal
		endif
		plot,wavelengths,spec2,title=target,xrange=[8,12], $
			xtitle='Wavelength [microns]',ytitle='ADU Beam B'
		if midi_options.v then begin
		chopdata=oirgetdata(target+'.photo.fits')
		tvscl,chopdata(1).data1+chopdata(1).data2,0.65,0.35,/normal
		endif
		!p.multi=0
		end
'Print group delay': begin
		print,'Please wait...'
		set_ps
		plotfile=target+'.'+time+'.gd.ps'
		device,filename=plotfile
		mididelayplot,target,RawDir(0)
		device,/close
;		spawn,'lpr '+plotfile
		set_screen
		print,'Done. Plot sent to printer.'
		end
'Print visibility':begin
		print,'Please wait...'
		set_ps
		plotfile=target+'.'+time+'.v2.ps'
		device,filename=plotfile
		midiinsvisplot,target,RawDir(0)
		device,/close
;		spawn,'lpr '+plotfile
		set_screen
		print,'Done. Plot sent to printer.'
		end
'Save':		begin
		xmdvobj->save_visibility,target+'.oifits'
		end
'Store':	begin
		if n_elements(scans) eq 0 then begin
			print,'You must define data set first!'
			return,-1
		endif
		if method eq 'Heidelberg' then begin
			la_start=la_bins(0)-1
			la_end=la_bins(1)-1
			la_step=la_bins(2)
			la_width=la_bins(3)
			xmdvobj->set_lambda_bins,la_start,la_end,la_step,la_width
			visi=xmdvobj->visibility()
			phot=xmdvobj->get_photometry(/ab)
			photerr=sqrt(phot)
			index=where(finite(visi) eq 0,count)
			if count gt 0 then visi(index)=-1
			vissq=visi(2,*)^2
			vissqe=((vissq*0.1) > 0.001)*signof(visi(2,*))
		endif else if method eq 'Leiden' then begin
			visdata=oirgetvis(target+'.redcal.fits')
			visamp=xmdvavg(visdata.visamp,la_bins)
			visampe=xmdvavg(visdata.visamperr,la_bins)
			vissq=visamp^2
			vissqe=2*visamp*visampe
			d=oirgetdata(target+'.photometry.fits')
			phot=xmdvavg(d(5).data1,la_bins)
			photerr=xmdvavg(d(10).data1,la_bins)
		endif
;
;		Find out which scan corresponds to this observation
		iobs=where(recipes.track_file eq obht(obsindex(0)).filename) & iobs=iobs(0)
		midiupdateconfig,GenConfig
;
		scans(iobs).vissq(0,0:genconfig.numspecchan(0)-1,0)=vissq
		scans(iobs).vissqerr(0,0:genconfig.numspecchan(0)-1,0)=vissqe
		scans(iobs).vissqc(0,0:genconfig.numspecchan(0)-1,0)=vissq
		scans(iobs).vissqcerr(0,0:genconfig.numspecchan(0)-1,0)=vissqe
;		We store into Photonrate the normalizing spectrum
;		This should be the geometric mean of the masked spectra for A and B
		scans(iobs).photonrate(0,0:genconfig.numspecchan(0)-1)=phot
		scans(iobs).photonrateerr(0,0:genconfig.numspecchan(0)-1)=photerr
;
;		Update recipes
		recipes(iobs).method=method
;		recipes(iobs).mask_file=xmdvobj->get_filename(/mask)
		recipes(iobs).phot_a_file=obht(obsindex(1)).filename
		recipes(iobs).phot_b_file=obht(obsindex(2)).filename
;
		end
'Frames':	begin
		frames=read_frames(obht(obsindex(0)).filename,data_id,labels)
		r=size(frames)
		nrows=r(3)
		index=where(labels eq 'S ',count)
		if count gt 0 then begin
			sky=total(frames(*,*,index),3)/n_elements(index)
			for i=0,nrows-1 do frames(*,*,i)=frames(*,*,i)-sky
		endif
		widget_control,i_wid,/sensitive
		widget_control,j_wid,/sensitive
		miditrackgui2_event,'Init'
		return,0
		end
'PA chopped':	begin
		index=where(trackdata1.t eq 'T ' and trackdata1.c ne 0,count)
		image=total(trackdata1.cf(*,*,index),3)/count
		displaychoppedspectrum,image,strtrim(obht(obsindex(0)).insgris,2)
		end
'PB chopped':	begin
		index=where(trackdata2.t eq 'T ' and trackdata2.c ne 0,count)
		image=total(trackdata2.cf(*,*,index),3)/count
		displaychoppedspectrum,image,strtrim(obht(obsindex(0)).insgris,2)
		end
'PA Offset':	begin
		index=where(trackdata1.c ne 0 and trackdata1.t eq 'T ')
		plot,trackdata1.x(index),psym=3,xtitle='Target frame ID',ytitle='Offset (PA)'
		return,0
		end
'PB Offset':	begin
		index=where(trackdata2.c ne 0 and trackdata2.t eq 'T ')
		plot,trackdata2.x(index),psym=3,xtitle='Target frame ID',ytitle='Offset (PB)'
		return,0
		end
'PA Flux':	begin
		index=where(trackdata1.c ne 0 and trackdata1.t eq 'T ')
		plot,trackdata1.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (PA)'
		v=medianve(trackdata1.f(index),e)
		print,'Median: ',v,', median error: ',e
		return,0
		end
'PB Flux':	begin
		index=where(trackdata2.c ne 0 and trackdata2.t eq 'T ')
		plot,trackdata2.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (PB)'
		v=medianve(trackdata2.f(index),e)
		print,'Median: ',v,', median error: ',e
		return,0
		end
'Display spectra':begin
	mask=findfile(maskfile)
	if n_elements(mask) ne 1 then mask='' else mask=mask(0)
	if strlen(mask) eq 0 then begin
		print,'Please select a mask first!'
		return,-1
	endif
	mask=oirgetdata(mask)
	if n_elements(where(strpos(tag_names(mask),'DATA') ge 0)) ne 4 then begin
		print,'***Error: please specify SCI_PHOT mask!'
		return,-1
	endif
	do_photometry,obht(obsindex(0)).filename,mask,PhFilt,spec1,spec2,sci_mode=1
	dispersion=strtrim(obht(obsindex(0)).insgris)
	win1nx=detwin(0)
	win1strx=detwin(1)
	wavelengths=pix2lambda(dindgen(win1nx)+win1strx,dispersion)
	title='Photometric channel A (g) and B (r)'
	plot,wavelengths,spec1,/nodata,title=title,xtitle='Wavelength [microns]', $
		xrange=[7,13],xstyle=1, $
		yrange=[min([spec1,spec2]),max([spec1,spec2])]
	oplot,wavelengths,spec1,color=2
	oplot,wavelengths,spec2,color=3
	oplot,wavelengths,wavelengths*0
	return,0
	end
else:
endcase
;
end
;-------------------------------------------------------------------------------
pro miditrackgui
;
; Display GUI for software which computes visibilities.
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
common MidiTrackSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiTrackGuiWids,draw_wid,button2h_wid,button2l_wid
common MidiChopData,z,acqdata1,acqdata2,phodata1,phodata2,trackdata1,trackdata2
;
; Remove any existing widget
if widget_info(miditrack_wid,/valid) then widget_control,miditrack_wid,/destroy
;
fitsfiles=nameparse(strjoin(obht(obsindex(0)).filename,' '))
fitsfile=obj_new('fitsfile',fitsfiles(0))
prihead=fitsfile->prihead()
win1nx=prihead->getpar('DET WIN1 NX')
win1strx=prihead->getpar('DET WIN1 STRX')
detwin=[win1nx,win1strx]
obj_destroy,prihead
obj_destroy,fitsfile
la_bins(1) = la_bins(1) < win1nx
;
miditrack_wid=widget_base(/col, $
	title=obht(obsindex(0)).obstargname+' ' $
	     +obht(obsindex(0)).insgris $
	     +obht(obsindex(0)).insopt1)
;
reduction_wid=widget_base(miditrack_wid,/col,/frame)
;
; Heidelberg
reductionrow_wid=widget_base(reduction_wid,/row)
guis1=['Heidelberg']
guis2=['Vis_gui','Lambda_gui','Slide_gui','Save']
button_wid=cw_bgroup(reductionrow_wid,/row,guis1, $
	event_funct='miditrackgui_event',/return_name)
button2h_wid=cw_bgroup(reductionrow_wid,/row,guis2, $
	event_funct='miditrackgui_event',/return_name)
widget_control,button_wid,set_uvalue='Heidelberg'
widget_control,button2h_wid,sensitive=0
;
; Leiden
; widget_control,files_wid,set_uvalue=obfileobj->files()
reductionrow_wid=widget_base(reduction_wid,/row)
guis1=['Leiden']
guis2=['Vis_plot','Delay_plot','Flux_plot','Store']
button_wid=cw_bgroup(reductionrow_wid,/row,guis1, $
	event_funct='miditrackgui_event',/return_name)
button2l_wid=cw_bgroup(reductionrow_wid,/row,guis2, $
	event_funct='miditrackgui_event',/return_name)
widget_control,button_wid,set_uvalue='Leiden'
widget_control,button2l_wid,sensitive=0
;
plots=['Frames I1','Frames I2']
if obht(obsindex(0)).insopt1 eq 'SCI_PHOT' then begin
	plots=[plots,'Frames PA','Frames PB', $
		     'PA chopped','PA Offset','PA Flux', $
		     'PB chopped','PB Offset','PB Flux']
endif
; button_wid=cw_bgroup(miditrack_wid,/row,guis, $
;	event_funct='miditrackgui_event',/return_name)
; widget_control,button_wid,set_uvalue='Frames'
miditrackrow_wid=widget_base(miditrack_wid,/row)
opmenu_wid=widget_droplist(miditrackrow_wid,event_func='miditrackgui_event', $
	uvalue=plots,value=plots)
if obht(obsindex(0)).insopt1 eq 'SCI_PHOT' then begin
	plots='Display spectra'
	button_wid=cw_bgroup(miditrackrow_wid,plots, $
		event_funct='miditrackgui_event',/return_name)
endif
;
draw_wid=widget_draw(miditrack_wid,event_pro='trackdraw_event', $
	scr_xsize=300,scr_ysize=200)
count=5000
i_wid=cw_fslider(miditrack_wid,minimum=1,maximum=count,value=1,/drag, $
	title='Index I',xsize=256,format='(i4)',uvalue='i')
j_wid=cw_fslider(miditrack_wid,minimum=0,maximum=100,value=0,/drag, $
	title='Index J',xsize=256,format='(i3)',uvalue='j')
;
widget_control,miditrack_wid,/realize
xmanager,'miditrackgui2',miditrack_wid,/no_block
;
widget_control,i_wid,sensitive=0
widget_control,j_wid,sensitive=0
;
end
;-------------------------------------------------------------------------------
pro miditrackgui2_event,event
;
common MidiTrackSlider,i_wid,j_wid,i_value,j_value,frames,labels
common MidiTrackGuiWids,draw_wid,button2h_wid,button2l_wid
;
result=size(event)
if result(n_elements(result)-2) ne 8 then begin
	value=1
	i_value=1
	j_value=0
	widget_control,draw_wid,get_value=id
	loadct,0
	erase
endif else begin
	widget_control,event.id,get_uvalue=slider,get_value=value
	case slider of
		'i':i_value=long(value)
		'j':j_value=long(value)
	endcase
endelse
;
index=i_value+j_value-1
if index ge n_elements(labels) then return
image=frames(*,*,index)
;
widget_control,draw_wid,get_value=id
wset,id
tvscl,image
xyouts,20,20,labels(index),/device
;
end
;-------------------------------------------------------------------------------
pro midisearchgui,index
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiTaskWids,midiacq_wid,midipho_wid,miditrack_wid,midimask_wid,midirecipes_wid
;
; Remove any existing widget
if widget_info(midisrch_wid,/valid) then widget_control,midisrch_wid,/destroy
;
objindex=index
;
print,'Reading data, please wait...'
filelist=oirfilelist(obht(objindex).filename)
nframes=0l
naxes=intarr(n_elements(filelist))
for i=0,n_elements(filelist)-1 do begin
	im=obj_new('imagedata',filelist(i))
	naxis=im->naxis()
	naxes(i)=naxis(1)
	nframes=nframes+naxis(1)
	obj_destroy,im
endfor
opd=fltarr(nframes)
di=fltarr(nframes)
if nframes mod 10 ne 0 then begin
	print,'nframes mod 10 ne 0!'
	return
endif
k=0
j=0
im=obj_new('imagedata',filelist(k))
for i=0L,nframes/10-1 do begin
	if j*10 eq naxes(k) then begin
		obj_destroy,im
		k=k+1
		j=0
		im=obj_new('imagedata',filelist(k))
	endif
	j=j+1
	off=0
	if k gt 0 then off=long(total(naxes(0:k-1)))
	z=im->readrows(indgen(10)+1+i*10-off)
	dimg=z.data1-z.data2
	r=size(dimg)
	dv=total(dimg(10:r(1)-20,10:r(2)-10,*),2)
	di(i*10+indgen(10))=total(dv,1)
;	i1=total(z.data1,2)
;	i2=total(z.data2,2)
;	i1w=total(i1,1)
;	i2w=total(i2,1)
;	di(i*10+indgen(10))=i1w-i2w
	opd(i*10+indgen(10))=(z.opd(1)-z.localopd(0))*1e6
endfor	
obj_destroy,im
di=di-mean(di)
;
; Some editing is necessary
fndex=where(finite(opd) eq 1,n)
di=di(fndex)
opd=opd(fndex)
;
; Look for jump from bias position
dopd=opd(1:n-1)-opd(0:n-2)
index=where(dopd gt 1000)
opd=opd(index(0)+1:n-1)
di=di(index(0)+1:n-1)
n=n_elements(opd)
;
; Determine scan boundaries
dopd=opd(1:n-1)-opd(0:n-2)
index=where(dopd lt 0,n1)
n2=median(index(1:n1-1)-index(0:n1-2))
n1=n1+1
index=[index,n-1]
y=fltarr(n2,n1)
for i=0,n1-1 do begin
	if index(i)-n2+1 ge 0 and $
  	   index(i) lt n_elements(di) then y(*,i)=di(index(i)-n2+1:index(i))
endfor
x=y
for i=0,n1-1 do begin
	if index(i)-n2+1 ge 0 and $
	   index(i) lt n_elements(opd) then x(*,i)=opd(index(i)-n2+1:index(i))
endfor
set_screen
!x.range=[min(opd),max(opd)]
!y.range=[min(di),max(di)]
!x.title='Delay [microns]'
plot_slide,x,y,wid=midisrch_wid
;
end
;-------------------------------------------------------------------------------
pro kappamatrix,obht,ofile
;
words=nameparse(obht(0).filename)
afile=words(0)
words=nameparse(obht(1).filename)
bfile=words(0)
ofile='Kappa_'+afile+'+'+bfile
;
case strtrim(obht(0).insgris) of
	'GRISM':begin
		sfile='mioSetup_FIELD_GRISM_SCI_PHOT_SLIT.tmp '
		dfile='kappa.GRISM.fits'
		end
	'PRISM':begin
		sfile='mioSetup_FIELD_PRISM_SCI_PHOT_SLIT.tmp '
		dfile='kappa.PRISM.fits'
		end
endcase
sfile=getenv('drsRoot')+'/mio/'+sfile
command=getenv('drsRoot')+'/c/bin/oirCrossCoeff '
spawn,command+afile+' '+bfile+' '+sfile+ofile
spawn,'cp -f '+ofile+' '+dfile
;
end
;-------------------------------------------------------------------------------
pro initializepipe
;
; This procedure will initialize the OYSTER scans structure for each fringe
; track file of all configurations found in the current directory.
;
; OYSTER configurations are defined by the specific selection of baseline
; and spectrometer (GRISM or PRISM).
;
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
; OYSTER common blocks
common Starbase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Save configid
configid_save=configid
;
initialized=0
configids=analyzemidiconfigids()
num_config=n_elements(configids)
;
FOR iconfig=0,num_config-1 DO BEGIN
;
configid=configids(iconfig)
selectmidifiles,/all
dispersion=unique(strtrim(obht.insgris,2))
if n_elements(dispersion) ne 1 then begin
	print,'***Error: more than one configuration selected!'
	return
endif
files=nameparse(obht(obsindex(0)).filename)
fitsfile=obj_new('fitsfile',files(0))
prihead=fitsfile->prihead()
dateobs=prihead->getpar('date-obs')
Date=strmid(dateobs,0,10)
time=hms2h(strmid(dateobs,11,12))
; if time gt 12 then Date=nextdate(Date); This date is stored in geninfo.date
obj_destroy,prihead
obj_destroy,fitsfile
case dispersion of
	'PRISM': get_sysconfig,sysid='VLTI/MIDI-N-PRISM'
	'GRISM': get_sysconfig,sysid='VLTI/MIDI-N-GRISM'
endcase
index_obs=where(obht.nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
	     or obht.nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED',count_obs)
;
; The recipe table will hold all reduction info for the pipeline
recipe={tag:'',acq_file:'',track_file:'',phot_a_file:'',phot_b_file:'', $
	mask_file:'',method:'',photometry:'', $
	kappa_matrix:'',photo_file:''}
recipes=replicate(recipe,count_obs)
status=defaultrecipes()
;
; Open first file and read configuration information
midigenconfig,files(0),genconfig
; Allocate scans structure
scans=replicate(scan(),count_obs)
scans.acqfilter='nband'
; Read data for each file
for i=0,count_obs-1 do begin
	s=scans(i)
	files=nameparse(obht(index_obs(i)).filename)
	midiscan,files(0),s
	s.iscan=i+1
	scans(i)=s
endfor
scans=midistarids(scans)
get_scantable
get_startable,/names
;
; Initialization for AMOEBA
if not initialized then begin
	freememory
	bufferinfo=replicate(nightinfo(),num_config)
	geoinfo=replicate(geoparms,num_config)
	geninfo=replicate(allocgenconfig(/geninfo),num_config)
	geninfo.configid=configids
	table=startable
	initialized=1
endif else begin
        table=merge_startable(table,startable)
endelse
confignum=where(geninfo.configid eq configid)
g=geninfo(confignum)
struct_assign,genconfig,g
geninfo(confignum)=g
geoinfo(confignum)=geoparms
storenight
save,recipes,filename='Recipes_'+configid+'.xdr'
;
ENDFOR
;
; Make sure startable has entries for all stars observed this night
startable=table
;
; Restore selection of configid
configid=configid_save
r=midigui_configid(configid)
;
end
;-------------------------------------------------------------------------------
pro mymidipipe,dir
;
; Processes all MIDI files returned by selectmidifiles and stores results 
; in OYSTER scans structure. Data should be of same configuration.
; The obsindex, normally defined for an individual observation, is initialized 
; here only for the fitmask routine.
;
; Mymidipipe uses the instructions contained in the recipes structure for 
; the selection of the reduction method. The default recipes are created
; and stored when initializing the pipeline.
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiGuiWids,midi_wid,files_wid,window_slide_wid,pipeline_wid
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
common MidiPipeline,rawdir,rawdirchecksum,configid,recipes
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ModelFit,parameters,ds_options
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
RAD=180/!pi
;
selectmidifiles,/all
;
; Determine how many observations there are
index_obs=where(obht.nrtsmode eq 'OBS_FRINGE_TRACK_FOURIER' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED_OFF' $
	     or obht.nrtsmode eq 'OBS_FRINGE_NOTRACK_DISPERSED' $
	     or obht.nrtsmode eq 'OBS_FRINGE_TRACK_DISPERSED',count_obs)
;
; We process using the recipes, make sure they correspond
count_rec=n_elements(recipes)
if count_rec gt count_obs then begin
	print,'***Error(MYMIDIPIPE): Recipes inconsistent with selected files!'
	return
endif
index_obs=intarr(count_rec)
for i=0,count_rec-1 do begin
	j=where(obht.filename eq recipes(i).track_file,count)
	if count eq 0 then begin
		print,'***Error(MYMIDIPIPE): not found: '+recipes(i).track_file+'!'
		return
	endif
	index_obs(i)=j
endfor	
;
; Obtain some general information on this night from the first file
;
words=nameparse(recipes(0).track_file)
fitsfile=obj_new('fitsfile',words(0))
prihead=fitsfile->prihead()
;
; Baseline
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
stations=[station_1,station_2]
delayline_1=strtrim(prihead->getpar('ISS CONF DL1'))
delayline_2=strtrim(prihead->getpar('ISS CONF DL2'))
delaylines=[delayline_1,delayline_2]
delaylineids=strmid(delaylines,2,1)
bcinput_1=strtrim(prihead->getpar('ISS CONF INPUT1'))
bcinput_2=strtrim(prihead->getpar('ISS CONF INPUT2'))
bcinputs=[bcinput_1,bcinput_2]
beam_a=stations(where(bcinputs eq 3)) & beam_a=beam_a(0)
beam_b=stations(where(bcinputs eq 1)) & beam_b=beam_b(0)
;
; UT Date of the first observation
Date=strmid(prihead->getpar('date-obs'),0,10)
parsedate,Date,y,m,d
jd0=julian(y,m,d)
;
; Cleanup
obj_destroy,prihead
obj_destroy,fitsfile
;
; Star names
stars=strarr(count_obs)
starids=strarr(count_obs)
categrs=strarr(count_obs)
;
; Image shift, for explanation see midigui_acq
o=[1,-1]	; + moves B up/right
;
; Loop over all observations and process acquisition, track, and photometry
;
FOR iobs=0,count_rec-1 DO BEGIN
;
set_screen
;
; Get some general info on this observation from the fringe track file
;
words=nameparse(recipes(iobs).track_file)
nfile=n_elements(words)
dlt1start=fltarr(nfile)
dlt1end=fltarr(nfile)
delay1=fltarr(nfile+1)
dlt2start=fltarr(nfile)
dlt2end=fltarr(nfile)
delay2=fltarr(nfile+1)
utc=fltarr(nfile)
dutc=fltarr(nfile+1)
;
; Open first fringe track file and get some general information
fitsfile=obj_new('fitsfile',words(0))
prihead=fitsfile->prihead()
airmass=strtrim(prihead->getpar('ISS AIRM END'))
seeing=strtrim(prihead->getpar('ISS AMBI FWHM'))
dispersion=strtrim(prihead->getpar('INS GRIS ID'))
dateobs=strtrim(prihead->getpar('DATE-OBS'))
obstarg=strtrim(prihead->getpar('OBS TARG NAME'))
obsra=prihead->getpar('RA')/15.
obsdec=prihead->getpar('DEC')
win1nx=prihead->getpar('DET WIN1 NX')
win1ny=prihead->getpar('DET WIN1 NY')
win1strx=prihead->getpar('DET WIN1 STRX')
win1stry=prihead->getpar('DET WIN1 STRY')
win2nx=prihead->getpar('DET WIN2 NX')
win2ny=prihead->getpar('DET WIN2 NY')
win2strx=prihead->getpar('DET WIN2 STRX')
win2stry=prihead->getpar('DET WIN2 STRY')
dlt1=prihead->getpar('DEL DLT1 OPL START')
dlt2=prihead->getpar('DEL DLT2 OPL START')
refopl=prihead->getpar('DEL REF OPL')
refname=prihead->getpar('DEL REF NAME')
categ=prihead->getpar('DPR CATG')
obj_destroy,prihead
obj_destroy,fitsfile
detwin=[win1nx,win1strx]
;
; Fill in wavelength info assuming all others are like the first obs.
if iobs eq 0 then begin
	la_bins(1)=la_bins(1) < win1nx
	wavelengths=xmdvavg(pix2lambda(dindgen(win1nx)+win1strx,dispersion),la_bins,chanwidths)*1e-6
	chanwidths=chanwidths*1e-6
	nwave=n_elements(wavelengths)
	genconfig.stationid(0)=stations(0)
	genconfig.stationid(1)=stations(1)
	genconfig.baselineid(0,*)=genconfig.stationid(0)+'-'+genconfig.stationid(1)
	genconfig.delaylineid=delaylineids
	ref_count=0
	if isnumeric(refname) then $
	ref_index=where(delaylineids eq long(strmid(refname,2,1)),ref_count)
	if ref_count eq 0 then ref_index=0
	genconfig.refstation=ref_index+1
	genconfig.bcinputid=bcinputs
	genconfig.numoutbeam=2
	genconfig.numbaseline=1
	genconfig.numspecchan(*)=nwave
	genconfig.wavelength(0:genconfig.numspecchan(0)-1,0)=wavelengths
	genconfig.wavelength(*,1)=genconfig.wavelength(*,0)
	genconfig.wavelengtherr=0.1e-6
	genconfig.chanwidth(0:genconfig.numspecchan(0)-1,0)=chanwidths
	genconfig.chanwidth(*,1)=genconfig.chanwidth(*,0)
	genconfig.chanwidtherr=0.1e-6
	genconfig.configid=dispersion+'-'+genconfig.stationid(0)+'-'+genconfig.stationid(1)
endif
;
if 0 then begin	; obsolete
; For the OPD data, open all fringe track (sub) files
for i=0,nfile-1 do begin
	fitsfile=obj_new('fitsfile',words(i))
	prihead=fitsfile->prihead()
	dlt1start(i)=prihead->getpar('DEL DLT1 OPL START')
	dlt2start(i)=prihead->getpar('DEL DLT2 OPL START')
	dlt1end(i)=prihead->getpar('DEL DLT1 OPL END')
	dlt2end(i)=prihead->getpar('DEL DLT2 OPL END')
	utc(i)=prihead->getpar('UTC')
	obj_destroy,prihead
	obj_destroy,fitsfile
endfor
utc=utc-utc(0)	; [s]
delay1(0)=dlt1start(0)
delay1(1:nfile)=dlt1end
delay2(0)=dlt2start(0)
delay2(1:nfile)=dlt2end
if strtrim(obht(index_obs(iobs)).insopt1) eq 'HIGH_SENS' then fmask='hmask'
if strtrim(obht(index_obs(iobs)).insopt1) eq 'SCI_PHOT'  then fmask='smask'
spawn,getenv('vltiCbin')+'/oir1dCompressData "'+obht(index_obs(iobs)).filename+'" ' $
     +getenv(strlowcase(dispersion)+fmask)+' '+obstarg+'.compressed.fits'
data=oirgetdata(obstarg+'.compressed.fits')
time=reform(data.time)
i=where(finite(time) eq 0)
if i(0) ge 0 then begin
	time(i)=0
	time=double(median(time,5))
endif
time=(time-time(0))*86400		; [s]
dutc(0:nfile-1)=utc
dutc(nfile)=max(time)
opd=data.opd+data.localopd
opd1=opd(0,*)+interpol(delay1,dutc,time)
opd2=opd(1,*)+interpol(delay2,dutc,time)
endif	; end of obsolete code
;
; Compute the squared visibilities using either Heidelberg or Leiden procedures
;
mask_file='PHOTOMETRY/'+recipes(iobs).mask_file
track_file=recipes(iobs).track_file
phot_a_file=recipes(iobs).phot_a_file
phot_b_file=recipes(iobs).phot_b_file
phot_files=[phot_a_file,phot_b_file]
xmdv_in=strarr(3)
xmdv_in(0)=track_file
xmdv_in(1)=phot_a_file
xmdv_in(2)=phot_b_file
target=obstarg+'.'+strmid(recipes(iobs).track_file,0,24)
beamcombiner=strtrim(obht(index_obs(iobs)).insopt1)
;
mask=findfile(mask_file)
if n_elements(mask) ne 1 then mask='' else mask=mask(0)
;
if recipes(iobs).method eq 'Leiden' then begin
	IF midi_options.v THEN BEGIN
;	Leiden does not compute masks
	obsindex=index_obs(iobs)
	if obht(index_obs(iobs)).insopt1 eq 'SCI_PHOT' then begin
		if recipes(iobs).kappa_matrix eq 'Compute' then begin
			kappamatrix,obht(index_obs(iobs)+1:index_obs(iobs)+2),kappafile
		endif else kappafile=recipes(iobs).kappa_matrix
		leidenpipe
	endif else begin
		if strlen(mask) eq 0 then begin
			midipipe,obstarg,files=[obht(index_obs(iobs)).filename, $
					       recipes(iobs).phot_a_file, $
					       recipes(iobs).phot_b_file]
		endif else begin
			print,'Using mask file: ',mask
			midipipe,obstarg,files=[obht(index_obs(iobs)).filename, $
					       recipes(iobs).phot_a_file, $
					       recipes(iobs).phot_b_file],mask=mask
		endelse
	endelse
	ENDIF ELSE BEGIN
	if beamcombiner eq 'SCI_PHOT' and recipes(iobs).photometry ne 'HighSens' then begin
;		SCI_PHOT Modes A and B
		if recipes(iobs).kappa_matrix eq 'Compute' then begin
			print,'Calling midiCrossCoeff...'
			midiCrossCoeff,target,phot_files
			kappafile=target+'.crossCoeff.fits'
		endif else kappafile=recipes(iobs).kappa_matrix
		r=findfile(kappafile)
		if strlen(r(0)) eq 0 then begin
			print,'***Error: kappa matrix not found: ',kappafile
			return
		endif
		if total(strlen(phot_files)) eq 0 $
		or recipes(iobs).photometry eq 'SciPhot' then begin
;			Mode A
			if strlen(mask) eq 0 $
			then midiSPipe,target,track_file,cross=kappafile $
			else midiSPipe,target,track_file,cross=kappafile,mask=mask
		endif else begin
;			Mode B
			if strlen(mask) eq 0 $
			then midiSPipe,target,[track_file,phot_files],cross=kappafile $
			else midiSPipe,target,[track_file,phot_files],cross=kappafile,mask=mask
		endelse
	endif else begin
;		HIGH_SENS and SCI_PHOT mode D
                if strlen(mask) eq 0 $
                then midiPipe,target,[track_file,phot_files] $
                else midiPipe,target,[track_file,phot_files],mask=mask
	endelse
;	dummy.fits was created by oirFormFringes
	spawn,'rm -f /tmp/dummy.fits'
	ENDELSE
	target=obstarg+'.'+strmid(track_file,0,24)
	visdata=oirgetvis(target+'.redcal.fits')
	visamp=xmdvavg(visdata.visamp,la_bins)
	visampe=xmdvavg(visdata.visamperr,la_bins)
	vissq=visamp^2
	vissqe=2*visamp*visampe
;	Geometric mean of A and B masked spectra
	d=oirgetdata(target+'.photometry.fits')
	phot=xmdvavg(d(5).data1,la_bins)
	photerr=xmdvavg(d(10).data1,la_bins)
	photA=phot
	photB=phot
   	delay=oirgetdelay(target+'.groupdelay.fits')
	d=reform(delay.delay) 
	d=d(n_elements(d)/10:n_elements(d)-1)
	d=d-median(d,n_elements(d)/10)
	v=medianve(d,delayrms)
endif else begin
	errorlog=''
	IF midi_options.v THEN BEGIN
;	Heidelberg can currently only be called if photometry was taken
	obsindex=index_obs(iobs)+indgen(3)
;	New EWS style photometry does not work for SCI_PHOT
	if obht(index_obs(iobs)).insopt1 eq 'SCI_PHOT' then noews=1 else noews=0
	if strlen(mask) eq 0 then begin
		xmdvobj=xmdv(xmdv_in,noews=noews,/NOMASKCHECK,errorlog=errorlog)
	endif else begin
		xmdvobj=xmdv(xmdv_in,noews=noews,/NOMASKCHECK,errorlog=errorlog,maskfile=mask)
	endelse
;	recipes(iobs).mask_file=xmdvobj->get_filename(/mask)
	la_start=la_bins(0)-1
	la_end=la_bins(1)-1
	la_step=la_bins(2)
	la_width=la_bins(3)
	xmdvobj->set_lambda_bins,la_start,la_end,la_step,la_width
	visi=xmdvobj->visibility()
	index=where(finite(visi) eq 0,count)
	if count gt 0 then visi(index)=-1
	vissq=visi(2,*)^2
	vissqe=((vissq*0.1) > 0.001)*signof(visi(2,*))
;	Geometric mean of A and B masked spectra
	phot=xmdvobj->get_photometry(/ab)
	photerr=sqrt(phot)
	photA=phot
	photB=phot
	ENDIF ELSE BEGIN
	obsindex=index_obs(iobs)+indgen(3)
	if beamcombiner eq 'SCI_PHOT' and recipes(iobs).photometry ne 'HighSens' then begin
;		SCI_PHOT Modes A and B
		if recipes(iobs).kappa_matrix eq 'Compute' then begin
			print,'Calling midiCrossCoeff...'
			midiCrossCoeff,target,xmdv_in(1:2)
			kappafile=target+'.crossCoeff.fits'
		endif else kappafile=recipes(iobs).kappa_matrix
		r=findfile(kappafile)
		if strlen(r(0)) eq 0 then begin
			print,'***Error: kappa matrix not found: ',kappafile
			return
		endif
		if total(strlen(xmdv_in(1:2))) eq 0 $
		or recipes(iobs).photometry eq 'SciPhot' then begin
;			Mode A
			if strlen(mask) eq 0 $
			then xmdvobj=obj_new("xmidispvisi",xmdv_in(0),/DWIM, $
				dsky=midi_options.s,cross=kappafile,errorlog=errorlog) $
			else xmdvobj=obj_new("xmidispvisi",xmdv_in(0),/DWIM, $
				dsky=midi_options.s,cross=kappafile,errorlog=errorlog, $
				maskfile=mask)
		endif else begin
;			Mode B
			if strlen(mask) eq 0 $
			then xmdvobj=xmdv(xmdv_in,/DWIM,cross=kappafile, $
					dsky=midi_options.s,errorlog=errorlog,fitmask=2) $
			else xmdvobj=xmdv(xmdv_in,/DWIM,cross=kappafile, $
					dsky=midi_options.s,errorlog=errorlog,maskfile=mask)
		endelse
	endif else begin
;		SCI_PHOT Mode D and HIGH_SENS
		if strlen(mask) eq 0 $
		then xmdvobj=xmdv(xmdv_in,/DWIM,errorlog=errorlog, $
			dsky=midi_options.s,fitmask=2) $
		else xmdvobj=xmdv(xmdv_in,/DWIM,errorlog=errorlog, $
			dsky=midi_options.s,maskfile=mask)
	endelse
	recipes(iobs).mask_file=xmdvobj->get_filename(/mask)
	la_start=la_bins(0)-1
	la_end=la_bins(1)-1
	la_step=la_bins(2)
	la_width=la_bins(3)
	xmdvobj->set_lambda_bins,la_start,la_end,la_step,la_width
	visi=xmdvobj->visibility()
	index=where(finite(visi) eq 0,count)
	if count gt 0 then visi(index)=-1
	vissq=visi(2,*)^2
	vissqe=((vissq*0.1) > 0.001)*signof(visi(2,*))
;	Geometric mean of A and B masked spectra
	phot=xmdvobj->get_photometry(/ab)
	photA=xmdvobj->get_photometry(/atotal)
	photB=xmdvobj->get_photometry(/btotal)
	photerr=sqrt(phot)
	delayrms=0.0
	ENDELSE
endelse
;
; Store visibility
scans(iobs).vissq(0,0:nwave-1,0)=vissq
scans(iobs).vissqerr(0,0:nwave-1,0)=vissqe
scans(iobs).vissqc(0,0:nwave-1,0)=vissq
scans(iobs).vissqcerr(0,0:nwave-1,0)=vissqe
;
; Store normalizing spectrum in Photonrate
scans(iobs).photonrate(0,0:nwave-1)=phot
scans(iobs).photonrateerr(0,0:nwave-1)=photerr
;
; Store delay data
if genconfig.refstation eq 1 then dlt2=dlt2-dlt1
if genconfig.refstation eq 2 then dlt1=dlt1-dlt2
scans(iobs).fdlpos(0)=dlt1
scans(iobs).fdlpos(1)=dlt2
scans(iobs).fdlposerr=10e-6
;
; Store delay rms
scans(iobs).delayrms(0,0)=delayrms
;
; Store time
obsdate=strmid(dateobs,0,10)
parsedate,obsdate,y,m,d
jdobs=julian(y,m,d)
time0=hms2h(strmid(dateobs,11,12))*3600	; [s]
scans(iobs).time=time0+(jdobs-jd0)*86400
;
if 0 then begin
; For a better delay estimate, do this...
if genconfig.refstation eq 1 then opd2=opd2-opd1
if genconfig.refstation eq 2 then opd1=opd1-opd2
nf=n_elements(time)
scans(iobs).fdlpos(0)=mean(opd1(nf/2-nf/10:nf/2+nf/10))
scans(iobs).fdlpos(1)=mean(opd2(nf/2-nf/10:nf/2+nf/10))
scans(iobs).time=time0+(jdobs-jd0)*86400+mean(time(nf/2-nf/10:nf/2+nf/10))
endif
;
; Information for StarTable, see if we can find HD or HIP numbers
categrs(iobs)=strtrim(categ,2)
stars(iobs)=obht(index_obs(iobs)).obstargname
starid=cri_vlti(stars(iobs),obsra,obsdec)
starids(iobs)=starid
scans(iobs).starid=starid
scans(iobs).iscan=iobs+1
scans(iobs).star=stars(iobs)
scans(iobs).ra=obsra
scans(iobs).dec=obsdec
scans(iobs).r0=seeing
scans(iobs).za=acos(1/airmass)*RAD
;
; Analyze acquisition images
;
if strlen(recipes(iobs).acq_file) gt 0 then begin
	words=nameparse(recipes(iobs).acq_file)
	fitsfile=obj_new('fitsfile',words(0))
	prihead=fitsfile->prihead()
	scans(iobs).AcqFilter=strtrim(prihead->getpar('INS FILT NAME'))
	obj_destroy,prihead
	obj_destroy,fitsfile
;	Chop images
;	z=midiChopImage(obht(index_obs(iobs)-1).filename,before=3,after=2)
	z=MIDIChopImageC(recipes(iobs).acq_file,before=midi_options.before,after=midi_options.after)
	image_a=z(0).data1-z(1).data1
	image_b=shift(z(0).data2-z(1).data2,o(0),o(1))
;	Apply a mask
	imsize=[n_elements(image_a(*,0)),n_elements(image_a(0,*))]
	x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
	y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
	offset_a=median(image_a(where(sqrt(x^2+y^2) lt imsize(0)/3)))
	offset_b=median(image_b(where(sqrt(x^2+y^2) lt imsize(0)/3)))
	image_a(where(sqrt(x^2+y^2) gt imsize(0)/3))=offset_a
	image_b(where(sqrt(x^2+y^2) gt imsize(0)/3))=offset_b
;	Fit 2-dim. Gaussian
	aparm=fltarr(7)
	bparm=fltarr(7)
	r_a=gauss2dfit(image_a,aparm,/tilt)
	r_b=gauss2dfit(image_b,bparm,/tilt)
	g2d2fwhm=sqrt(2)*2
	madu=midi_magadu(genconfig.stationid(0),scans(iobs).AcqFilter)
	scans(iobs).acqpsf(0,*)=[aparm(2)*g2d2fwhm,aparm(3)*g2d2fwhm, $
				 aparm(6)*RAD,aparm(4),aparm(5)]
	scans(iobs).acqpsf(1,*)=[bparm(2)*g2d2fwhm,bparm(3)*g2d2fwhm, $
				 bparm(6)*RAD,bparm(4),bparm(5)]
	scans(iobs).acqflux=n2jy(madu-2.5*alog10(!pi*2*[aparm(1)*aparm(2)*aparm(3), $
							bparm(1)*bparm(2)*bparm(3)]))
;	Do analysis using chop_nod_phot, which writes output into a text file
	chop_nod_phot,recipes(iobs).acq_file,/silent
	nc=strlen(recipes(iobs).acq_file)
	lines=''
	status=dc_read_fixed('PHOTOMETRY/Phot_'+strmid(recipes(iobs).acq_file,5,nc-9)+'txt', $
		lines,/col,format='(a80)')
	index=where(strpos(lines,'Aperture for Beam B in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		scans(iobs).acqflux(0)=n2jy(midi_magadu(genconfig.stationid(0),scans(iobs).AcqFilter) $
						-2.5*alog10(float(words(0))))
	endif
	index=where(strpos(lines,'Aperture for Beam A in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		scans(iobs).acqflux(1)=n2jy(midi_magadu(genconfig.stationid(1),scans(iobs).AcqFilter) $
						-2.5*alog10(float(words(0))))
	endif
endif
;
; Analyze photometry
phot_files=[recipes(iobs).phot_a_file,recipes(iobs).phot_b_file]
for ibeam=0,1 do begin
	if obht(index_obs(iobs)).insopt1 eq 'SCI_PHOT' then chopped_file=recipes(iobs).track_file $
						       else chopped_file=phot_files(ibeam)
	words=nameparse(chopped_file)
        fitsfile=obj_new('fitsfile',words(0))
        prihead=fitsfile->prihead()
        scans(iobs).ChopThrow(ibeam)=strtrim(prihead->getpar('ISS CHOP THROW'))
        scans(iobs).ChopAngle(ibeam)=strtrim(prihead->getpar('ISS CHOP POSANG'))
        obj_destroy,prihead
        obj_destroy,fitsfile
;
;	if midi_options.c or obht(index_obs(iobs).insopt1 eq 'SCI_PHOT' then begin
	if recipes(iobs).method eq 'Leiden' then begin
		d=oirgetdata(target+'.photometry.fits')
		scans(iobs).Photometry(ibeam,0,0:nwave-1)=xmdvavg(d(3).data1,la_bins)
		scans(iobs).Photometry(ibeam,1,0:nwave-1)=xmdvavg(d(3).data1,la_bins)
	endif else begin
;		We now prefer to run Olivier's original chop_nod_disp
		loadct,0
		chop_nod_disp,phot_files(ibeam), $
			width=width,trace_order=t_order,fwhm_order=w_order,/silent
		readphot,phot_files(ibeam),spec12
		spec1=spec12(*,1)
		spec2=spec12(*,2)
;
;		mask=xmdvobj->get_filename(/mask)
;		mask=oirgetdata(mask)
;		do_photometry,chopped_file,mask,PhFilt,spec1,spec2
;
		scans(iobs).Photometry(ibeam,0,0:nwave-1)=xmdvavg(spec1,la_bins)
		scans(iobs).PhotometryErr(ibeam,0,0:nwave-1)=1
		scans(iobs).Photometry(ibeam,1,0:nwave-1)=xmdvavg(spec2,la_bins)
		scans(iobs).PhotometryErr(ibeam,1,0:nwave-1)=1
		if ibeam eq 1 then obj_destroy,xmdvobj
	endelse
;
endfor
;
ENDFOR
;
genconfig.date=checkdate()
confignum=where(geninfo.configid eq configid)
g=geninfo(confignum)
struct_assign,genconfig,g
geninfo(confignum)=g
geoinfo(confignum)=geoparms
;
; Make sure we have only one star ID per star
scans=midistarids(scans)
get_scantable
;
; Now allocate and fill the startable
get_startable,unique(starids)
for i=0,n_elements(startable)-1 do begin
	j=where(starids eq startable(i).starid) & j=j(0)
	startable(i).name=stars(j)
	if categrs(j) eq 'CALIB' then startable(i).bflag='C'
	startable(i).ra=scans(j).ra
	startable(i).dec=scans(j).dec
endfor
get_stationtable
calcastrom
calcvis
storenight,11
;
; Write the data to disk
filestub=genconfig.date+'-'+genconfig.configid
put_xdr,filestub+'.cha.xdr'
save,recipes,filename='Recipes_'+genconfig.configid+'.xdr'
;
; Also save in OIFITS
put_oifits,filestub+'.fits'
;
if n_elements(errorlog) ne 0 then ConstrictorLog=errorlog
;
end
;-------------------------------------------------------------------------------
pro leidenpipe
;
; Process a SCI_PHOT observation without photometry. If photometry is avaliable,
; this can be processed beforehand to produce a Kappa matrix stored in a file.
;
; This routine is now obsolete starting with Version 1.4
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
fringesmooth=' 50'
delaysmooth=' 2'
dSky=' 0'
nDrop=' 1'
cbin=getenv('drsRoot')+'/c/bin/'
;
target=strtrim(obht(obsindex(0)).obstargname,2)
target=target+'.'+strmid(obht(obsindex(0)).filename,0,24)
dispersion=strtrim(obht(obsindex(0)).insgris)
beamcombiner=strtrim(obht(obsindex(0)).insopt1)
;
mask=findfile(maskfile)
if n_elements(mask) ne 1 then mask='' else mask=mask(0)
if strlen(mask) eq 0 then mask=minrtsmask(obht(obsindex(0)))
;
kappa=findfile(kappafile)
if n_elements(kappa) ne 1 then kappa=miokappa(obht(obsindex(0))) $
			  else kappa=kappa(0)
;
words=nameparse(obht(obsindex(0)).filename)
setup=words(0)
;
; First, do the photometry
if beamcombiner eq 'HIGH_SENS' or n_elements(obsindex) eq 3 then begin
	command=cbin+'oirChopPhotometry "'+obht(obsindex(1)).filename+'" "'+obht(obsindex(2)).filename+'" '+mask+' '+target+'.photometry.fits -dSky'+dsky
	spawn,command
	command=cbin+'oirChopPhotoImages "'+obht(obsindex(1)).filename+'" "'+obht(obsindex(2)).filename+'" '+target+'.photo.fits -dSky'+dsky
	spawn,command
endif else if beamcombiner eq 'SCI_PHOT' and n_elements(obsindex) eq 1 then begin
	command=cbin+'oirSciPhotometry -data "'+obht(obsindex(0)).filename+'" -ref '+setup+' -cross '+kappa+' -out '+target+'.photometry.fits -mask sciphot.mask'
	spawn,command
endif
;
; Then compute the visibilities
command=cbin+'oir1dCompressData "'+obht(obsindex(0)).filename+'" '+mask+' '+target+'.compressed.fits'
spawn,command
command=cbin+'oirFormFringes '+target+'.compressed.fits '+target+'.fringes.fits'+fringesmooth
spawn,command
command=cbin+'oirRotateInsOpd '+target+'.fringes.fits '+target+'.insopd.fits'
spawn,command
command=cbin+'oirGroupDelay '+target+'.insopd.fits '+target+'.groupdelay.fits -smooth'+delaysmooth
spawn,command
command=cbin+'oirRotateGroupDelay '+target+'.fringes.fits '+target+'.groupdelay.fits '+target+'.ungroupdelay.fits'
spawn,command
command=cbin+'oirAutoFlag '+target+'.fringes.fits '+target+'.groupdelay.fits '+target+'.flag.fits'
spawn,command
command=cbin+'oirAverageVis '+target+'.ungroupdelay.fits '+target+'.flag.fits '+target+'.corr.fits'
spawn,command
command=cbin+'oirRedCal '+target
spawn,command
;
end
;-------------------------------------------------------------------------------
pro mykappapipe
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
if midi_options.v then begin
kappafiles=findfile('Kappa_*.fits')
endif else begin
kappafiles=findfile('*.crossCoeff.fits')
endelse
;
if strlen(kappafiles(0)) eq 0 then begin
;
selectmidifiles,/all
index=where(obht.nrtsmode eq 'DEFAULT_CHOP')
obht=obht(index)
for i=0,n_elements(obht)/2-1 do begin
	if midi_options.v then kappamatrix,obht(i*2:i*2+1) $
			  else begin
			       files=nameparse(obht(i*2).filename)
			       kappafile=strtrim(obht(i*2).obstargname,2)+'.'+files(0)
			       midiCrossCoeff,kappafile,obht(i*2:i*2+1).filename
			  endelse
endfor
;
endif
;
fitsfiles=nameparse(obht(0).filename)
fitsfile=obj_new('fitsfile',fitsfiles(0))
prihead=fitsfile->prihead()
win1nx=prihead->getpar('DET WIN1 NX')
win1strx=prihead->getpar('DET WIN1 STRX')
detwin=[win1nx,win1strx]
obj_destroy,prihead
obj_destroy,fitsfile
;
if midi_options.v then begin
kappafiles=findfile('Kappa_*.fits')
endif else begin
kappafiles=findfile('*.crossCoeff.fits')
endelse
;
dispersion=strtrim(obht(obsindex(0)).insgris)
win1nx=detwin(0)
win1strx=detwin(1)
wavelengths=pix2lambda(dindgen(win1nx)+win1strx,dispersion)
;
index=where(wavelengths gt 8 and wavelengths lt 12,count)
y=fltarr(n_elements(kappafiles),count)
x=y
for i=0,n_elements(kappafiles)-1 do x(i,*)=wavelengths(index)
;
!p.multi=[0,2,2]
!x.range=[8,12]
!y.range=[0.5,1.5]
!x.title='Wavelengths [micron]'
;
for i=0,n_elements(kappafiles)-1 do begin
d=oirgetdata(kappafiles(i))
y(i,*)=d.data1(index)
!y.title='I1/A'
if i eq 0 then plot,wavelengths,d.data1 else oplot,wavelengths,d.data1,color=getcolor(i)
endfor
xx=reform(x,n_elements(x))
yy=reform(y,n_elements(y))
r=poly_fit(xx,yy,4,yfit)
oplot,wavelengths,poly(wavelengths,r),thick=3
print,stddev(yy-yfit)
;
for i=0,n_elements(kappafiles)-1 do begin
d=oirgetdata(kappafiles(i))
y(i,*)=d.data2(index)
!y.title='I1/B'
if i eq 0 then plot,wavelengths,d.data2 else oplot,wavelengths,d.data2,color=getcolor(i)
endfor
xx=reform(x,n_elements(x))
yy=reform(y,n_elements(y))
r=poly_fit(xx,yy,4,yfit)
oplot,wavelengths,poly(wavelengths,r),thick=3
print,stddev(yy-yfit)
;
for i=0,n_elements(kappafiles)-1 do begin
d=oirgetdata(kappafiles(i))
y(i,*)=d.data3(index)
!y.title='I2/A'
if i eq 0 then plot,wavelengths,d.data3 else oplot,wavelengths,d.data3,color=getcolor(i)
endfor
xx=reform(x,n_elements(x))
yy=reform(y,n_elements(y))
r=poly_fit(xx,yy,4,yfit)
oplot,wavelengths,poly(wavelengths,r),thick=3
print,stddev(yy-yfit)
;
for i=0,n_elements(kappafiles)-1 do begin
d=oirgetdata(kappafiles(i))
y(i,*)=d.data4(index)
!y.title='I2/B'
if i eq 0 then plot,wavelengths,d.data4 else oplot,wavelengths,d.data4,color=getcolor(i)
endfor
xx=reform(x,n_elements(x))
yy=reform(y,n_elements(y))
r=poly_fit(xx,yy,4,yfit)
oplot,wavelengths,poly(wavelengths,r),thick=3
print,stddev(yy-yfit)
;
end
;-------------------------------------------------------------------------------
pro myacqpipe
;
; Analyzes all acquisition frames. Assumes data are all from one baseline!
;
common Starbase,StarTable,Notes
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
obht_file=findfile('obht.sav')
if strlen(obht_file(0)) eq 0 then selectmidifiles,/all $
			     else restore,obht_file
;
obht0=obht(where(obht.nrtsmode eq 'ACQ_UT_COARSE_CHOP'))
;
; Get the station info
fitsfile=obj_new('fitsfile',obht0(0).filename)
prihead=fitsfile->prihead()
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
input_1=strtrim(prihead->getpar('ISS CONF INPUT1'))
input_2=strtrim(prihead->getpar('ISS CONF INPUT2'))
inputs=[input_1,input_2]
stations=[station_1,station_2]
beam_a=stations(where(inputs eq 3)) & beam_a=beam_a(0)
beam_b=stations(where(inputs eq 1)) & beam_b=beam_b(0)
obj_destroy,prihead
obj_destroy,fitsfile
;
hdn=cri(obht0.obstargname,'hdn',/lists)
bsc=cri(hdn,'hdn-bsc')
index=where(bsc gt 0)
obht0=obht0(index)
bsc=bsc(index)
get_startable,'BSC'+string(bsc,format='(i4.4)')
get_f12
index=where(startable.f12 gt 0)
startable=startable(index)
obht0=obht0(index)
;
janskys=startable.f12
fluxes=fltarr(2,n_elements(janskys))
acqfilter=strtrim(obht0.insfiltname,2)
;
; Do analysis using chop_nod_phot
for i=0,n_elements(obht0)-1 do begin
	chop_nod_phot,obht0(i).filename,/silent,threshold=0.05
	nc=strlen(obht0(i).filename)
	lines=''
	status=dc_read_fixed('PHOTOMETRY/Phot_'+strmid(obht0(i).filename,5,nc-9)+'txt', $
		lines,/col,format='(a80)')
	index=where(strpos(lines,'Aperture for Beam B in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		fluxes(0,i)=float(words(0))
	endif
	index=where(strpos(lines,'Aperture for Beam A in pixels') ge 0,count)
	if count gt 0 then begin
		words=nameparse(lines(index(0)+1))
		fluxes(1,i)=float(words(0))
	endif
endfor
;
index=where(fluxes(0,*) gt 0 and fluxes(1,*) gt 0)
fluxes=fluxes(*,index)
janskys=janskys(index)
acqfilter=acqfilter(index)
;
janskys0=janskys
fluxes0=fluxes
;
ufilter=unique(acqfilter)
nfilter=n_elements(ufilter)
x=findgen(300)
!p.multi=[0,2,nfilter]
for j=0,nfilter-1 do begin
index=where(acqfilter eq ufilter(j))
janskys=janskys0(index)
fluxes=fluxes0(*,index)
;
y=2.5*alog10(fluxes(0,*))+jy2n(janskys)
plot,janskys,y,psym=1,title=beam_a, $
	ytitle=ufilter(j),xtitle='Flux [Jy]'
oplot,findgen(300),fltarr(300)+mean(y),psym=0
if n_elements(janskys) ge 2 then sdev=stddev(y) else sdev=0
print,'Beam A: '+beam_a+', ',ufilter(j),mean(y),sdev
;
y=2.5*alog10(fluxes(1,*))+jy2n(janskys)
plot,janskys,y,psym=1,title=beam_b, $
	ytitle=ufilter(j),xtitle='Flux [Jy]'
oplot,findgen(300),fltarr(300)+mean(y),psym=0
if n_elements(janskys) ge 2 then sdev=stddev(y) else sdev=0
print,'Beam B: '+beam_b+', ',ufilter(j),mean(y),sdev
endfor
;
return
;
plot,janskys,fluxes(0,*),psym=1,title='Beam A',ytitle=ufilter(j)
if n_elements(janskys) ge 2 then begin
	r=poly_fit(janskys,fluxes(0,*),1)
	oplot,x,poly(x,r),psym=0
endif
plot,janskys,fluxes(1,*),psym=1,title='Beam B',ytitle=ufilter(j)
if n_elements(janskys) ge 2 then begin
	r=poly_fit(janskys,fluxes(1,*),1)
	oplot,x,poly(x,r),psym=0
endif
;
end
;-------------------------------------------------------------------------------
pro myphopipe
;
; Analyzes all photometry frames.
;
common Starbase,StarTable,Notes
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
obht_file=findfile('obht.sav')
if strlen(obht_file(0)) eq 0 then selectmidifiles,/all $
			     else restore,obht_file
;
obht0=obht(where(obht.nrtsmode eq 'DEFAULT_CHOP',count))
;
openw,unit,'myphopipe.txt',/get_lun
;
for i=0,n_elements(obht0)-1 do begin
;
; Get the station info
words=nameparse(obht0(i).filename)
fitsfile=obj_new('fitsfile',words(0))
prihead=fitsfile->prihead()
airmass=float(strtrim(prihead->getpar('ISS AIRM END')))
shutter=strcompress(prihead->getpar('INS SHUT NAME'),/remove_all)
beamcombiner=strtrim(obht0(i).insopt1)
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
input_1=strtrim(prihead->getpar('ISS CONF INPUT1'))
input_2=strtrim(prihead->getpar('ISS CONF INPUT2'))
inputs=[input_1,input_2]
stations=[station_1,station_2]
beam_a=stations(where(inputs eq 3)) & beam_a=beam_a(0)
beam_b=stations(where(inputs eq 1)) & beam_b=beam_b(0)
obj_destroy,prihead
obj_destroy,fitsfile
if shutter eq 'AOPEN' then beam=beam_a else beam=beam_b
;
print,'Airmass: ',airmass,', '+shutter+', '+beam+', '+beamcombiner
printf,unit,'Airmass: ',airmass,', '+shutter+', '+beam+', '+beamcombiner
;
if beamcombiner eq 'SCI_PHOT' then dataid=[2,3] else dataid=[1,2]
;
print,'Reading and analyzing photometry I1 frames...'
rf=read_frames(obht0(i).filename,dataid(0),t)
c=chop_cycles(t)
cf=chop_frames(rf,t,c)
f=analyze_phoframes(cf,t,c,x,a,xmed)
af=align_phoframes(cf,t,c,x,xmed)
phodata1={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
;
index=where(phodata1.c ne 0 and phodata1.t eq 'T ')
; plot,phodata1.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (I1)'
v=medianve(phodata1.f(index),e)
print,'Median: ',v,', median error: ',e
printf,unit,'I1: ','Median: ',v,', median error: ',e
;
print,'Reading and analyzing photometry I2 frames...'
rf=read_frames(obht0(i).filename,dataid(1),t)
c=chop_cycles(t)
cf=chop_frames(rf,t,c)
f=analyze_phoframes(cf,t,c,x,a,xmed)
af=align_phoframes(cf,t,c,x,xmed)
phodata2={t:t,c:c,cf:cf,f:f,a:a,x:x,xmed:xmed,af:af}
;
index=where(phodata2.c ne 0 and phodata2.t eq 'T ')
; plot,phodata2.f(index),psym=3,xtitle='Target frame ID',ytitle='Flux (I2)'
v=medianve(phodata2.f(index),e)
print,'Median: ',v,', median error: ',e
printf,unit,'I2: ','Median: ',v,', median error: ',e
printf,unit,' '
;
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro myprimapipe
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
common MidiOptions,midi_options,maskfile,kappafile,photofile,la_bins
;
fringesmooth=' -50'	; negative value suppresses high-pass filter
delaysmooth=' 2'
cbin=getenv('drsRoot')+'/c/bin/'
;
index=where(strpos(obht.insshut,'ABOPEN') ge 0)
;
target='lamp'
target=target+'.'+strmid(obht(index).filename,0,24)
dispersion=strtrim(obht(index).insgris)
beamcombiner=strtrim(obht(index).insopt1)
;
mask=findfile(maskfile)
if n_elements(mask) ne 1 then mask='' else mask=mask(0)
if strlen(mask) eq 0 then mask=minrtsmask(obht(index))
;
; Then compute the visibilities
command=cbin+'oir1dCompressData "'+obht(index).filename+'" '+mask+' '+target+'.compressed.fits'
spawn,command
command=cbin+'oirFormFringes '+target+'.compressed.fits '+target+'.fringes.fits'+fringesmooth
spawn,command
command=cbin+'oirRotateInsOpd '+target+'.fringes.fits '+target+'.insopd.fits'
spawn,command
command=cbin+'oirGroupDelay '+target+'.insopd.fits '+target+'.groupdelay.fits -smooth'+delaysmooth
spawn,command
;
!p.charsize=1.5
delay = oirgetdelay(target+'.groupdelay.fits')
delaytime=REFORM(delay.time)&delaytime=86400*(delaytime-delaytime[0])
delay = reform(delay.delay)
mve=medianve(delay,e)
!p.title=target
plot,delaytime,delay,yrange=[mve-3*e,mve+6*e], $
	xtitle='Time [s]',ytitle='Delay [microns]',psym=1
set_ps
plot,delaytime,delay,yrange=[mve-3*e,mve+6*e], $
	xtitle='Time [s]',ytitle='Delay [microns]',psym=1
device,/close
set_screen
;
end
;************************************************************************Block 5
pro photratios
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
index=indgen(30)+15
x=genconfig.wavelength(index,0)*1e6
!x.title='Wavelength [microns]'
;
n=n_elements(scans)
;
!p.psym=0
!p.multi=[0,2,2]
;
first=1
ibeam=0
!y.title='Photometry I2/I1'
!p.title=genconfig.stationid(ibeam)
for i=0,n-1 do begin
;
	if total(scans(i).photometry) gt 0 then begin
		y=scans(i).photometry(ibeam,1,index) $
		 /scans(i).photometry(ibeam,0,index)
		if first then plot,x,y else oplot,x,y
		first=0
	endif
endfor
;
ypix_per_unit_char=!d.y_ch_size
ypix_per_char=fix(ypix_per_unit_char*!p.charsize)
xyouts,1.0-float(ypix_per_char)/!d.x_size,!y.window(1), $
        GenConfig.Date+' '+GenConfig.configid,/normal, $
        orientation=-90,size=!p.charsize
;
first=1
ibeam=1
!y.title='Photometry I1/I2'
!p.title=genconfig.stationid(ibeam)
for i=0,n-1 do begin
;
	if total(scans(i).photometry) gt 0 then begin
		y=scans(i).photometry(ibeam,0,index) $
		 /scans(i).photometry(ibeam,1,index)
		if first then plot,x,y else oplot,x,y
		first=0
	endif
endfor
;
!y.range=[0.2,1.8]
first=1
obeam=0
!y.title='Photometry '+genconfig.stationid(1)+'/'+genconfig.stationid(0)
!p.title='I1'
;
p=fltarr(n)
for i=0,n-1 do p(i)=total(scans(i).photometry)
pe=p
jndex=where(pe lt 4e7)
pe=pe(jndex)
bin=(max(pe)-min(pe))/3
;
for i=0,n-1 do begin
;
	if total(scans(i).photometry) gt 0 then begin
		y=scans(i).photometry(1,obeam,index) $
		 /scans(i).photometry(0,obeam,index)
;		icol=7
;		if p(i) lt (min(pe)+bin) then icol=2
;		if p(i) gt (max(pe)-bin) then icol=3
if i eq 1 then icol=3 else icol=1
		if first then plot,x,y,/nodata
		oplot,x,y,color=icol
		first=0
	endif
endfor
;
first=1
obeam=1
!y.title='Photometry '+genconfig.stationid(1)+'/'+genconfig.stationid(0)
!p.title='I2'
for i=0,n-1 do begin
;
	if total(scans(i).photometry) gt 0 then begin
		y=scans(i).photometry(1,obeam,index) $
		 /scans(i).photometry(0,obeam,index)
;		icol=7
;		if p(i) lt (min(pe)+bin) then icol=2
;		if p(i) gt (max(pe)-bin) then icol=3
if i eq 1 then icol=3 else icol=1
		if first then plot,x,y,/nodata
		oplot,x,y,color=icol
		first=0
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro fluxratios
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
!x.title='UT [h]'
;
n=n_elements(scans)
;
!p.psym=1
!p.multi=0
!y.range=[0.4,2.5]
!p.charsize=2.0
;
ibeam=0
!y.title='Acq. flux ratio '+genconfig.stationid(1)+'/'+genconfig.stationid(0)
!p.title=genconfig.date+' '+genconfig.configid
index=where(total(scans.acqflux,1) gt 0)
plot,scans(index).time/3600,scans(index).acqflux(1)/scans(index).acqflux(0)
;
end
;-------------------------------------------------------------------------------
pro photmovie
;
common MidiObservation,obht,acqindex,obsindex,phoindex,detwin
;
words=nameparse(obht(phoindex(0)).filename)
;
imageObj = obj_new('imagedata', words(0), ierr=ierr)
;
dataColumns = imageObj->dataColumns()
targetColumns = imageObj->targetColumns()
readColumns = [dataColumns,targetColumns[0:1]]
;
nrows=imageobj->naxis() & nrows=nrows(1)
for i=1,nrows-1 do begin
	data = imageObj->readRowsf(i, col=readColumns, ierr=ierr)
	tvsclm,data.data2,2
endfor
;
obj_destroy,imageObj
;
end
;-------------------------------------------------------------------------------
pro framemovie,targets,cycles,frames1,frames2
;
for i=0,n_elements(targets)-1 do begin
	if targets(i) eq 'T ' and cycles(i) ne 0 then begin
		tvsclm,frames1(*,*,i),2
		if n_elements(frames2) ne 0 then tvsclm,frames2(*,*,i),2,61*2,0
		wait,0.05
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro displaychoppedspectrum,image,dispersion
;
erase
tek_color
!p.multi=0
!p.charsize=1.0
case dispersion of
	'PRISM':x0=120
	'GRISM':x0=50
endcase
ncol=20
title='White light profile (' $
	+string(x0-ncol,format='(i3.3)')+':' $
	+string(x0+ncol,format='(i3.3)')+')'
plot,total(image(x0-ncol:x0+ncol,*),1)/(2*ncol+1),ymargin=[8,3],title=title
m=max(image)
s=size(image)
image(0,*)=m & image(*,0)=m & image(s(1)-1,*)=m & image(*,s(2)-1)=m
if 0 then begin
if dispersion eq 'PRISM' then begin
	image(*,7)=m/2 & image(*,11)=m/2
	image(*,23)=m/2 & image(*,27)=m/2
endif else begin
	image(*,4)=m/2 & image(*,8)=m/2
	image(*,27)=m/2 & image(*,31)=m/2
endelse
endif
loadct,0
tvscl,image
;
end
;-------------------------------------------------------------------------------
