/*
 * Copyright 2005, 2006 University of Leiden.
 *
 * This file is part of MIA+EWS.
 *
 * MIA+EWS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MIA+EWS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MIA; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include "fitsio.h"
#include "imaging_fdata.h"
#include "imaging_data.h"
#include "oirSupport.h"

/*
   data structure to handle multiple input files in raw formats
*/
typedef struct 
{
   char **fileNames;
   int  iFile, nFiles;
   long iRow, nRows;
   fitsfile *inFile;
   imaging_data_table_def  *rdata;
   imaging_fdata_table_def *fdata;
   imaging_detector_table_def *detector;
   imaging_data_row   *rData;
   int nregion;
   int rawInput;
   int iChan1, iChan2;
   int nx, ny;
   long  npix;
} inFileInfo;
/*
    Usage::
    oirFormFringes infile outfile 
    oirFormFringes infile outfile -smooth nSmooth -removeAverage -no20 -fixopd -noI1 -noI2
*/

void Usage() {
    printf("\noirFormFringes infile outfile [-smooth nSmooth] [-removeAverage] [-no20] [-noI1] [-noI2]\n");
    printf("   infile  = name of file with output from oir1dCompressData \n");
    printf("   outfile = name of output file\n");
    printf("   nSmooth = width (frames) of high-pass filter region (default 50)\n");
    printf("     If nSmooth <=0 no high-pass filter is applied\n");
    printf("   removeAverage if specified: remove average value (over wavelength) of each spectrum\n");
    printf("      default: dont remove\n");
    printf("   noI1 : do not include I1 data in fringes (default yes include)\n");
    printf("   noI2 : do not include I2 data in fringes (default yes include)\n");
    return;
}

/******************************************************************
 *   copy data from a raw (short int) data row to a float data row
 ******************************************************************/
void raw2Float(imaging_data_row *rawData, 
   imaging_fdata_row *inData, int iChan1, int iChan2, int nx, int ny) {

   int ix, iy;
   for (iy=0;iy<ny;iy++) for (ix=0;ix<nx;ix++) {
      inData->data[iChan1][ix+nx*iy] = (float)rawData->data[iChan1][ix+nx*iy]; 
      inData->data[iChan2][ix+nx*iy] = (float)rawData->data[iChan2][ix+nx*iy]; 
      }
/*
            copy nonData info
*/
   inData->time = rawData->time;
   inData->opd[0] = rawData->opd[0];
   inData->opd[1] = rawData->opd[1];
   inData->local_opd[1] = rawData->local_opd[1];
   inData->local_opd[0] = rawData->local_opd[0];
   inData->exp          = rawData->exp;
   inData->offset[0]    = rawData->offset[0];
   inData->offset[1]    = rawData->offset[1];
   inData->rotation     = rawData->rotation;
   inData->frame        = rawData->frame;
   inData->step         = rawData->step;
   inData->reference    = rawData->reference;
   inData->targtype[iChan1]  = rawData->targtype[iChan1];  
   inData->targtype[iChan2]  = rawData->targtype[iChan2];  
}

 void allocateSmoothBuffers(char *fileName, int nSmooth, imaging_fdata_row **buff, 
   int *ierr) {

   fitsfile *inFile, *pFile;
   imaging_fdata_table_def *idata;
   imaging_detector_table_def *detector;

   int i, iSmooth;

    i = fits_open_file(&inFile, fileName, READONLY, ierr);
    if (*ierr != 0) {
       printf("ERROR:fits openfile of %s gave error %i",fileName, *ierr); 
       printf("%s \n",strerror(errno));
       return ;
    }
    detector = read_imaging_detector_table(inFile, ierr); 
    if (*ierr != 0) {
       printf("ERROR:fits open detector of %s gave error %i",fileName, *ierr); 
       printf("%s \n",strerror(errno));
       return ;
    }
    i=fits_close_file(inFile, ierr);
    pFile = NULL; /* dummy file; we just want table definition */
    idata= make_imaging_fdata_table_def(pFile, detector, 40, ierr);
    *ierr = 0;
    for (iSmooth=0;iSmooth<nSmooth;iSmooth++) buff[iSmooth] = 
       make_imaging_fdata_row(idata);
    kill_imaging_fdata_table_def(idata);
} /* end allocate smooth buffers */

int  readInputRow(inFileInfo *fileInfo, imaging_fdata_row *outData, int *ierr){

   int i;
/*
      if iRow >= nRows initialize next file or quit if done
*/
   *ierr = 0;
   if (fileInfo->iRow >= fileInfo->nRows)  { /* end of file */
      if (fileInfo->iFile >=0) {
         if(fileInfo->rawInput) i = close_imaging_data_table( fileInfo->inFile, fileInfo->rdata, ierr);
         else i = close_imaging_fdata_table( fileInfo->inFile, fileInfo->fdata, ierr);
      }
      if (++fileInfo->iFile >= fileInfo->nFiles)  {
         if (fileInfo->rawInput) kill_imaging_data_row(fileInfo->rData);
         return 1; 
      }/* all done */
/*
         open next input file
*/
      fileInfo->iRow = 0;
      fits_open_file(&fileInfo->inFile, fileInfo->fileNames[fileInfo->iFile], 
         READONLY, ierr);
      if (*ierr != 0) {
         printf("ERROR:fits openfile of %s gave error %i",
            fileInfo->fileNames[fileInfo->iFile], *ierr);
         return 2;
      }
/*        open new input table   */
      if (fileInfo->rawInput) {
         fileInfo->rdata = open_imaging_data_table(fileInfo->inFile, ierr);
         for (i=0;i<fileInfo->nregion;i++) fits_set_tscale(fileInfo->inFile, 
            1+fileInfo->rdata->data_col[i], 1.0, 0.0, ierr);
         if (fileInfo->iFile == 0) fileInfo->rData = make_imaging_data_row(fileInfo->rdata);
      } else {
         fileInfo->fdata = open_imaging_fdata_table(fileInfo->inFile, ierr);
      }
      i = fits_get_num_rows(fileInfo->inFile, &fileInfo->nRows, ierr);
      fprintf(stderr,"   Reading %i rows from file %i \r", 
         fileInfo->nRows, 1+fileInfo->iFile);
   }   /* ready for new file */
/*
       read the next row; if raw copy to output row and convert to float
*/
      ++fileInfo->iRow;
      if (fileInfo->rawInput) {
         i = read_imaging_data_table(fileInfo->inFile, fileInfo->rdata, fileInfo->rData, 
            fileInfo->iRow, ierr);
         if (*ierr != 0) {
            printf("ERROR: reading data table row %li error %i",fileInfo->iRow, *ierr);
            return 2;
         } 

         raw2Float(fileInfo->rData, outData, fileInfo->iChan1, fileInfo->iChan2, 
            fileInfo->nx, fileInfo->ny); 

      } else {
         i = read_imaging_fdata_table(fileInfo->inFile, fileInfo->fdata, outData, 
            fileInfo->iRow, ierr);
         if (*ierr != 0) {
         printf("ERROR: reading data table row %li error %i",fileInfo->iRow, *ierr);
            return 2;
         }
      }
   return 0;
} /* end of read input row */

void removeAverage(imaging_fdata_row *inData, int nx, int ny,
   int iChan1, int iChan2) {

   int ix, iy;
   float sum;
   for (iy=0; iy<ny;iy++) {
      sum = 0.;
      for (ix=0;ix<nx;ix++) sum = sum + inData->data[iChan1][ix+iy*nx];
      sum = sum/nx;
      for (ix=0;ix<nx;ix++) inData->data[iChan1][ix+iy*nx] = inData->data[iChan1][ix+iy*nx] - sum;
      sum = 0.;
      for (ix=0;ix<nx;ix++) sum = sum + inData->data[iChan2][ix+iy*nx];
      sum = sum/nx;
      for (ix=0;ix<nx;ix++) inData->data[iChan2][ix+iy*nx] = inData->data[iChan2][ix+iy*nx] - sum;
   }
      return;

}  /* end of removeAverage */


void remove20(imaging_fdata_row *inData, int nx, int ny, int iChan1, int iChan2) {
   int ix, iy;
   int i20, j20, n20;
   float sum;

   n20 = nx/20;
   for (iy=0;iy<ny;iy++) {
      for (i20=0;i20<20;i20++) {
         sum = 0.;
         for (ix=0;ix<n20;ix++) sum = sum + inData->data[iChan1][i20+20*ix+iy*nx];
         sum = sum/n20;
         for (ix=0;ix<n20;ix++) inData->data[iChan1][i20+20*ix+iy*nx] = 
            inData->data[iChan1][i20+20*ix+iy*nx] - sum;
      }
      for (i20=0;i20<20;i20++) {
         sum = 0.;
         for (ix=0;ix<n20;ix++) sum = sum + inData->data[iChan2][i20+20*ix+iy*nx];
         sum = sum/n20;
         for (ix=0;ix<n20;ix++) inData->data[iChan2][i20+20*ix+iy*nx] = 
            inData->data[iChan2][i20+20*ix+iy*nx] - sum;
      }
   }
   return;
}  /* end of remove20 */

void outputRowLessAve(int subtractSmooth, imaging_fdata_row *inData, 
   imaging_fdata_row *outData, long nx, float dit, float* average) {

   long ix;
   float xdit;

   if (dit > 0.) xdit = 1./dit; else xdit = 1.;
/*
            copy nonData info
*/
   outData->time   = inData->time;
   outData->opd[0] = inData->opd[0];
   outData->opd[1] = inData->opd[1];
   outData->local_opd[1] = inData->local_opd[1];
   outData->local_opd[0] = inData->local_opd[0];
   outData->exp = inData->exp;
   outData->offset[0] = inData->offset[0];
   outData->offset[1] = inData->offset[1];
   outData->rotation  = inData->rotation;
   outData->frame     = inData->frame;
   outData->step      = inData->step;
   outData->reference = inData->reference;
   outData->targtype[0] = inData->targtype[0];  
/*
            copy data and remove average multiply by exposure time
*/
   if (subtractSmooth) 
      for (ix=0;ix<nx;ix++) outData->data[0][ix] =
        (inData->data[0][ix] - average[ix])*xdit;
   else 
      for (ix=0;ix<nx;ix++) outData->data[0][ix] = inData->data[0][ix] *xdit;
   return;
} /* end of outputRowLessAve */

int main (int argc, char** argv) {

   int nregion, one;
   int i, j, ierr;
   long nRows, oRow;
   int nx, ny ;
   long npix;
   float *totalBuffer, *averageBuffer, *saveOldBuffer;
   int nSmooth;
   int iSmooth;
   int iChan1, iChan2;
   int nextIn, nextOut;
   int iKey, maxstep;
   int iarg;
   int subtractAverage;
   int subtractSmooth;
   int doUn20;
   int fixOpd;
   int done;

   char outFileName[130];
   char history[80];
   char *bang = "!";

   inFileInfo fileInfo;


   fitsfile *inFile  = NULL;
   fitsfile *outFile = NULL;
   imaging_fdata_table_def *odata;
   imaging_detector_table_def *outDetectorDef;
   imaging_fdata_row **inData;
   imaging_fdata_row *outData;

   float dit;

   int doI1, doI2;
   ierr = 0;
/*    
      set defaults and read optional args
*/
   iarg = 3;
   subtractAverage = 0;
   subtractSmooth  = 1;
   doUn20  = 0;
   nSmooth = 50;
   fixOpd  = 0;
   doI1     = 1;
   doI2     = 1;

   if (argc < 3) {
      Usage();
      return 0;
   }
   printf("      ****** oirFormFringes ******\n");
   while (iarg < argc ) {
      if (!strcmp("-smooth", argv[iarg])) sscanf(argv[++iarg], "%i", &nSmooth);
      if (!strcmp("-removeAverage", argv[iarg]))  subtractAverage = 1;
      if (!strcmp("-no20", argv[iarg]))    doUn20 = 1;
      if (!strcmp("-fixopd", argv[iarg]))  fixOpd = 1;
      if (!strcmp("-noI1", argv[iarg]))    doI1   = 0;
      if (!strcmp("-noI2", argv[iarg]))    doI2   = 0;
      ++iarg;
   }
   if (nSmooth <= 0) {
      subtractSmooth = 0;
      printf("   NOT applying high-pass filter\n");
      nSmooth = 1; /* so rest of program doesn't crash */
   } else 
      printf("   Using high-pass window of %i points\n", nSmooth);
   if (subtractAverage==1) printf("   Subtracting spectrum average\n"); else
      printf("   NOT subtracting spectrum average\n");
   if (doUn20) printf("   Filtering 20 point noise \n");
   if (fixOpd) printf("   fudging opd \n");
/*
      get  input file list
*/
   fileInfo.fileNames = oirStrSplit(argv[1], &fileInfo.nFiles);
   printf("   reading input data from %i files:\n",fileInfo.nFiles);
   for (i=0; i<fileInfo.nFiles; i++) {printf("      %s \n", fileInfo.fileNames[i]);}
   ierr = 0;
/*
      create space for smoothing buffers
*/
   inData = (imaging_fdata_row**)malloc(nSmooth*sizeof(imaging_fdata_row *));
/*   allocateSmoothBuffers(&fileInfo, nSmooth, inData, &ierr); */
   allocateSmoothBuffers(fileInfo.fileNames[0], nSmooth, inData, &ierr); 
   fits_open_file (&inFile, fileInfo.fileNames[0], READONLY, &ierr);
   if (ierr != 0) {
      printf("ERROR:fits openfile of %s gave error %i",fileInfo.fileNames[0], ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
/*
      create output data table 
*/
   strcpy(outFileName, bang);
   strcat(outFileName, argv[2]);
   fits_create_file (&outFile, outFileName, &ierr);

   if (ierr !=0) {
      printf ("ERROR: opening fits output %s %i\n",argv[2], ierr);
      printf("%s \n",strerror(errno));
      return 1;
   }
/*
      copy primary header from input to output
      add some history records
*/
   i = fits_copy_header(inFile, outFile, &ierr);
/*   for (i=0; i<fileInfo.nFiles; i++) {
      sprintf(history,"oirFormFringes INFILE= %s", fileInfo.fileNames[i]);
      i=fits_write_history(outFile, history, &ierr); 
   }
*/
   if (subtractSmooth) sprintf(history,"oirFormFringes NSMOOTH= %i", nSmooth);
      else sprintf(history,"oirFormFringes NO highpass filter");
   i=fits_write_history(outFile, history, &ierr);
   if (subtractAverage==1)
      sprintf(history,"oirFormFringes SUBTRACTING AVERAGE FRINGE");
   else
      sprintf(history,"oirFormFringes NOT SUBTRACTING AVERAGE FRINGE");
   i=fits_write_history(outFile, history, &ierr);
/*
      get useful keywords from input primary header
*/
   i = fits_read_key(inFile, TFLOAT, "HIERARCH ESO DET DIT", &dit, 0, &ierr);
   ierr = 0;

/*
      get detector info then close input
*/
   outDetectorDef = read_imaging_detector_table(inFile, &ierr);
   if (ierr != 0) {
      printf("ERROR: reading detector table %i",ierr);
      return 1;
   }
   nx = outDetectorDef->naxis[0][0];
   ny = outDetectorDef->naxis[0][1];
   nregion = outDetectorDef->nregion;
   fileInfo.nregion = nregion;
   fileInfo.rawInput = (ny > 1); 
   if (fileInfo.rawInput) printf("   input data is RAW\n"); 
   else   printf("   input data is COMPRESSED\n"); 

   i = fits_close_file(inFile, &ierr);
   one            = 1;
   if (nregion == 2) {
      iChan1 = 0;
      iChan2 = 1;
   } else {
      iChan1 = 1;
      iChan2 = 2;
   }
   fileInfo.iChan1 = iChan1;
   fileInfo.iChan2 = iChan2;
/*     
   only one output row   
*/
   outDetectorDef->nregion = 1;

   npix = nx * (long)ny;
   fileInfo.npix = npix;
   fileInfo.nx = nx;
   fileInfo.ny = ny;
   iKey = table_find_key(outDetectorDef->def, "NREGION");
   table_put_keyword (outDetectorDef->def, iKey, (void *)&one, &ierr);
   odata = make_imaging_fdata_table_def (outFile, outDetectorDef, 40, &ierr); 
   if (ierr !=0) {
      printf ("ERROR: creating data definition\n");
      return 1;
      }
   outData = make_imaging_fdata_row(odata);

/*
         create smoothing buffers
*/
   totalBuffer   = (float *)malloc(npix * sizeof(float));
   averageBuffer = (float *)malloc(npix * sizeof(float));
   saveOldBuffer = (float *)malloc(npix * sizeof(float));
   memset(totalBuffer,   0, npix*sizeof(float));
   memset(averageBuffer, 0, npix*sizeof(float));
   memset(saveOldBuffer, 0, npix*sizeof(float));
/*
         initial fill of read buffers
         read a row, difference interferometric channels, accumulate->totalBuffer
*/
   iSmooth = 0;
   fileInfo.iFile = -1;
   fileInfo.iRow  = 0;
   fileInfo.nRows = -100;

   while(iSmooth < nSmooth) {
      done = readInputRow(&fileInfo, inData[iSmooth], &ierr);
      if (done == 2) {
         printf("ERROR: reading data table row %li error %i",fileInfo.iRow, ierr);
         return 1;
      }
      if (fixOpd) {
         inData[iSmooth]->opd[0] = 0.5*inData[iSmooth]->opd[0];
         inData[iSmooth]->opd[1] = 0.5*inData[iSmooth]->opd[1];
      }
      if (inData[iSmooth]->targtype[iChan1] == 'T') {
         if (subtractAverage) removeAverage(inData[iSmooth], nx, ny, iChan1, iChan2);
         if (doUn20)  remove20(inData[iSmooth], nx, ny, iChan1, iChan2);
         for (j=0; j<npix; j++) { 
            inData[iSmooth]->data[0][j] = 
               doI1*inData[iSmooth]->data[iChan1][j] - 
               doI2*inData[iSmooth]->data[iChan2][j];
            totalBuffer[j] = totalBuffer[j]+ inData[iSmooth]->data[0][j];
         }
         iSmooth = iSmooth+1;
      }
   }
   for (j=0; j<npix; j++) averageBuffer[j] = totalBuffer[j]/nSmooth;
/*
         output nSmooth/2 rows with subtraction of average
*/
   oRow = 1;
   for (nextOut =0; nextOut<(nSmooth/2); nextOut++) {
      outputRowLessAve(subtractSmooth, inData[nextOut], outData, npix, 
         dit, averageBuffer);
      i = write_imaging_fdata_table(outFile, odata, outData, 
         oRow++, &ierr);
      if (ierr != 0) {
         printf("ERROR: writing data table row %li error %i",oRow, ierr);
         return 1;
      }
    }
/*       
         now loop over rest of input data, replacing old row with
         new row in data buffer and in accumulator until done
*/
   nextIn = 0;
   memcpy(saveOldBuffer, inData[nextIn]->data[0], npix*sizeof(float));
   while (done == 0) {
/*
        read new row, difference channels and accumulate
*/
      done = readInputRow(&fileInfo, inData[nextIn], &ierr);
      if (done == 2) {
         printf("ERROR: reading data table row %li error %i",fileInfo.iRow, ierr);
         return 1;
      }
      if (fixOpd) {
         inData[nextIn]->opd[0] = 0.5*inData[nextIn]->opd[0];
         inData[nextIn]->opd[1] = 0.5*inData[nextIn]->opd[1];
      }
/*
          check if correct chopping phase
*/
      if (inData[nextIn]->targtype[iChan1] == 'T') {
         if (subtractAverage) removeAverage(inData[nextIn], nx, ny, iChan1, iChan2);
         if (doUn20)  remove20(inData[nextIn], nx, ny, iChan1, iChan2);
         for (j=0; j<npix; j++) {
            inData[nextIn]->data[0][j] = 
               doI1*inData[nextIn]->data[iChan1][j] - 
               doI2*inData[nextIn]->data[iChan2][j];
            totalBuffer[j] = totalBuffer[j] + 
               inData[nextIn]->data[0][j] - saveOldBuffer[j];
         }
/*
        average buffer, output, and update counters 
*/
         for (j=0; j<npix; j++) averageBuffer[j] = totalBuffer[j]/nSmooth;
         outputRowLessAve(subtractSmooth, inData[nextOut], outData, npix, 
            dit, averageBuffer);
         i = write_imaging_fdata_table(outFile, odata, outData, 
            oRow++, &ierr);
         if (ierr != 0) {
            printf("ERROR: writing data table row %li error %i",oRow, ierr);
            return 1;
         }
         if (++nextIn == nSmooth)  nextIn  = 0;
         memcpy(saveOldBuffer, inData[nextIn]->data[0], npix*sizeof(float));
         if (++nextOut == nSmooth) nextOut = 0;
      }
   }  /*  loop over remaining input rows */
/*
            finish output rows
*/
   if (nextOut > nextIn) {
      while (nextOut < nSmooth) {
         outputRowLessAve(subtractSmooth, inData[nextOut++], outData, npix, 
            dit, averageBuffer);
         i = write_imaging_fdata_table(outFile, odata, outData, 
            oRow++, &ierr);
         if (ierr != 0) {
            printf("ERROR: writing data table row %li error %i",oRow, ierr);
            return 1;
         }
      }
   }
/*   while (nextOut <= nextIn) { */
   while (nextOut < nextIn) { 
      outputRowLessAve(subtractSmooth, inData[nextOut++], outData, npix, 
         dit, averageBuffer);
      i = write_imaging_fdata_table(outFile, odata, outData, 
         oRow++, &ierr);
      if (ierr != 0) {
         printf("ERROR: writing data table row %li error %i",oRow, ierr);
         return 1;
      }
   }
   printf("   %i output rows \n",oRow-1);
   i = close_imaging_fdata_table(outFile, odata, &ierr);
   kill_imaging_fdata_row(outData);

   for (i=0;i<nSmooth;i++) kill_imaging_fdata_row(inData[i]); 
   free (inData);
   free (totalBuffer);
   free (averageBuffer);
   free (saveOldBuffer);
   return 0;
}
