function [meas, cmd, SR, psf2, WF_res, DM_shape] = Closed_Loop_PSIM(filename, IMfile, CMfile, PSfile, Turb, wl_im,...
    SHWFS_PSIM_input, fqcy, delay, gain, Niter, Ctrl_meth, modes_ID, amplitudes, Coeffs, cmd_ref, disp_ID, nb_neighb_SA)

% [meas, cmd, SR, psf2, WF_res, DM_shape] = Closed_Loop_PSIM(filename, IMfile, CMfile, PSfile, Turb, wl_im,...
%   SHWFS_PSIM_input, fqcy, delay, gain, Niter, Ctrl_meth, modes_ID, amplitudes, Coeffs, cmd_ref, disp_ID, nb_neighb_SA);
%
% This function simulates iterations of an AO closed loop.
% The inputs are:
% - 'filename' is the generic name one wants to give to the output files of
%   DM commands ([filename]_cmd.fits), measurements ([filename]_meas.fits),
%   Strehl Ratios ([filename]_SR.fits) and Long Exposure PSF
%   ([filename]_psf.fits).
% - The Control Matrix in 'CMfile' (a fits file) is the one used to close
%   the loop, while the Interaction Matrix in 'IMfile' (a fits file) is the
%   actual one of the system. The CM could have been generated from another
%   IM (using 'invIM.m' for example) to study the effect of
%   mis-registration.
% - 'PSfile' (a fits file) contains the turbulent Phase Screen(s). This
%   file is a 3D matrix in which the 3rd dimension contains different
%   layers of turbulence, all normalized to a r0 of 1 meter.
% - 'Turb' is a matrix containing the properties of those turbulent layers:
%    each column correspond to another layer. The first line is the wind
%    speed in meters per second, the second the r0 in the layer in meters
%    and the third one the wind direction (0, 90, 180 or 270 degrees).
% - 'wl_im' is the imaging wavelength (in microns)
% - 'SHWFS_PSIM_input' is the .m file containing the SHWFS parameters
% - 'fqcy' (in Hz) is the loop frequency
% - 'delay' is the delay (in number of frames) between when slopes are
%   measured and the crrecsponding commands are applied
% - 'gain' is the integral gain of the loop
% - 'Niter' is a vector of values, ranging between the first and last loop
%   cycles for which the Long Exposure PSF will be integrated. The loop
%   simulation will always start at the cycle 1, and end at the cycle
%   number which is the largest value in 'Niter'. A simple example is that
%   'Niter' = 15:1024 will generate a simulation of 1024 loop cycles, and
%   the Long Exposure PSF will be integrated between the cycles 15 and
%   1024. The Long Exposure PSF is given in the output 'psf2'.
% - The string 'Ctrl_meth' describes the control law, and can be either:
%   - 'geom': The input WF is measured by the WFS, and the subtraction with
%     the DM shape is done in the WFS space (after DM commands are
%     multiplied by the IM). Then the correction is computed like for the
%     'MVM' method. Works only if the WFS parameter SH_model is set to 1
%     (geometric model)
%   - 'MVM': Matrix Vector Multiplication. The input WF is computed by
%     subtracting the input turbulence to the DM shape. This one has thus
%     to be computed at each loop cycle, which is time-demanding. The
%     correction is computed by directly multiplying the measurements by
%     the CM * the gain (after delay).
%   - 'POLC': Pseudo-Open Loop Control. The POL measurements are
%     reconstructed and the correction is computed on them, also applying
%     priors on the OL turbulence. NOT IMPLEMENTED YET
% - 'modes_ID' is the generic name of the files (.fits) where the Influence
%   Functions of the DM are stored, including the path. Not used if
%   'Ctrl_meth' is set to 'geom'.
% - 'amplitudes' should contain the amplitude of the modes with which the
%   IM was recorded (global or mode by mode).
% - 'Coeffs' contains the mis-registration parameters that have been used
%   to record the IM (they are used to create the pupil shape).
% - 'cmd_ref' is a variable that can contain the initial DM commands to
%   apply. It can be useful for faster loop closure. Set to zero if not
%   used. 
% - If 'disp_ID' is set to 1, some images/plots are displayed at each loop
%   iteration (input WF, DM shape, commands...)
% - When using a Zonal IM, the knowledge of the actuators' position
%   allows restricting the WF computation to the 'nb_neighb_SA' nearest
%   sub-apertures. In the case of Gaussian IFs, the DM shape computation
%   is also restricted to the 'nb_neighb_SA' nearest sub-apertures,
%   further speeding up the computation.
% 
% The outputs are:
% - The concatenation of the measurement vector 'meas' of size [Niter x (2
%   * nb of valid subaps)]
% - The concatenation of the measurement vector 'meas' of size [Niter+delay
%   x nb of actuators]
% - Several estimations of Strehl Ratio are provided in the output
%   variable 'SR', which is a matrix of size [max(Niter), 5]. At each loop
%   cycle, the Strehl values computed are:
%   1) Instantaneous Strehl from Marechal approximation applied to the
%      residual tip-tilt free WF (is the same as #2 in 'geom' mode)
%   2) Instantaneous Strehl from Marechal approximation applied to the
%      reconstructed residual tip-tilt free  WF
%   3) Instantaneous Strehl as the intensity at the center of the PSF
%      divided by the reference intensity (perfect psf)
%   4) Instantaneous Strehl as the peak intensity of the PSF divided
%      by the reference intensity (perfect psf)
%   5) Long-Exposure Strehl as the intensity at the center of the Long
%      Exposure PSF divided by the reference intensity (perfect psf)
% - The Long Exposure PSF is given in the output 'psf2'.
% - The last DM shape 'DM_shape' and WF residual 'WF_res' are also given as
%   outputs. 
% 
% A typical entry could be:
% [meas, cmd, SR, psf, WF_res, DM_shape] = Closed_Loop_PSIM('AOF_test1', 'AOF_IM', 'AOF_CM', 'AOF_PS1', [2.6, 0.1, 0], 2.2,...
%   'SHWFS_input_AOF_geom', 1000, 2, 0.3, 15:1024, 'MVM', 'Act_coords_AOF', 1, [0 0 0 0 0 4 4], cmd0, 1, 20);
%
% Custom sub-routines required:
% 1) First level
% - SHWFS_PSIM.m
% - rf.m
% - write_fits.m
% - dp2phi2.m
% 2) Other levels
% - SHWFS_PSIM_Mainloop.m
% - createZ_PSIM.m
% - bin_PSIM.m
% - Slopes_PSIM.m
% - makepsf_PSIM.m
% - FindNM.m
% - ZPolynomeQuick.m
% - ImSensor_PSIM.m
% - centroid_PSIM.m
% - Corr_PSIM.m
% - crop_PSIM.m
% - imnoise.m (Image Processing Toolbox)
% - poissrnd.m (Statistics Toolbox)
% - nanmean.m (Statistics Toolbox)
% - nansum.m (Statistics Toolbox)
%
% v1.0 J.Kolb 16/12/10
% v1.1 J.Kolb 23/02/11
% - added input of matrix slopes2Zernikes, for loop performance
%   estimation.
% - added computation of Zernike modes for display of the residual phase
%   in 'geom' mode
% - implemented 'MVM' reconstruction, i.e. computation of DM shape at each
%   loop, and WF measurement on residual (closer to reality, but much
%   slower, than the 'geom' simulation).
% - loading of modes or IFs for DM shape computation
% - includes mis-registration in the non-'geom' model
% - added flag for Gaussian WF input
% v1.9 J.Kolb 24/05/12
% - replaced 'readfits' by 'rf' for parameters loading
% - removed pupil rotation commented lines, and intermediate variables IM2
%   and CM2. Removed other old commented lines.
% - improved comments
% - improvement of the mis-registration effects computation, copied from
%   the modifications done to 'MakeIM_PSIM.m'. Thsi includes a better pupil definition.
% - Other copy from 'MakeIM_PSIM.m': addition of the parameter
%   'nb_neighb_SA' that allows speeding up the commputation by reducing the
%   number of sub-apertures in which the DM shape is computed (only in the
%   neighbourhood of the actuators)
% - implemented the reconstruction of WF from slopes, using the Roddier
%   algorithm (sub-routine 'dp2phi2')
% - uniformisation of PSF computation with real WF or reconstructed one, in
%   particular long-exposure PSF computation. Related to this: modification
%   of the input parameter 'Niter' to include the starting iteration at
%   which PSF integration should start
% - improvement of Strehl Ratio computation and correction of mistakes.%
% v2.0 J.Kolb 10/12/12
% - improved HELP

% load input files and parameters
IM = rf(num2str(IMfile)); CM = rf(num2str(CMfile)); PS = rf(num2str(PSfile));
run(num2str(SHWFS_PSIM_input))

% Possible re-orientation of the Phase Screens matrix
if size(PS,2) < size(PS,1)
    PS = permute(PS,[2 1 3]);
end
% Possible resizing of the DM modes amplitudes vector
if length(amplitudes) == 1
    amplitudes = amplitudes*ones(1,size(IM,1));
end

% Initializations
nbpixWF = size(PS,1);
cmd = zeros(max(Niter)+delay,size(IM,1));
meas = zeros(max(Niter),size(IM,2));
WFvar = Inf*zeros(max(Niter),6);
SR = NaN*zeros(max(Niter),5);
modes_path = ['DM_', num2str(modes_ID), '_', num2str(nbpixWF), '/'];
% 'nbpixWF2' is the size the residual WF used for display and for computing
% the PSF
if strcmp(Ctrl_meth, 'geom')
    nbpixWF2 = SHsize;
else
    nbpixWF2 = nbpixWF;
end
psf = zeros(nbpixWF2,nbpixWF2,1); psf2 = psf;
WF_res = zeros(nbpixWF2,nbpixWF2,1);
Z = createZ_PSIM(nbpixWF2,2:3,outmin,outmax);
WF_res_Rod = zeros(SHsize,SHsize,1);
Z_Rod = createZ_PSIM(SHsize,2:3,outmin,outmax);
% Starting commands
if cmd_ref ~= 0
    cmd(1:delay,:) = ones(delay,1) * cmd_ref;
end

% loading of actuator coordinates (if any)
if exist([num2str(modes_ID), '.mat'], 'file') == 2
    % 1) for creation of Gaussian Influence Functions
    load(num2str(modes_ID))
    Gauss_flag = 1;
elseif (exist([num2str(modes_path), 'Act_coords.mat'], 'file') == 2) && (nargin > 17)
    % 2) for sub-selection of sub-apertures in case of pre-loaded Zonal
    % Influence Functions
    load([num2str(modes_path), 'Act_coords.mat']);
    Gauss_flag = 2;
else
    % 3) pre-loaded IFs and no actuators coordinates inputs
    Gauss_flag = 0;
end

% Creation of the perfect pupil mask
[pupmask, ~, out] = createZ_PSIM(nbpixWF,1,outmin,outmax);
pupmask(out) = NaN;
nanpupmask = pupmask;
Pup_flux_loss = nansum(pupmask(:));

% The pupil mask for generating the perfect PSF is based on a pupil not
% mis-registrered.
if strcmp(Ctrl_meth, 'geom')
    WF_mask = createZ_PSIM(SHsize,1, outmin, outmax) ; WF_mask(WF_mask == 0) = NaN;
else
    WF_mask = pupmask;
end
% Perfect PSF Nyquist-sampled, at the imaging wavelength
[psfmax, wl_im, ~] = makepsf_PSIM(WF_res, WF_mask, wl_im, D, -0.5, size(WF_res,1), 0, 1);
% Amplitude of the perfect PSF
SRmax = psfmax(size(psfmax,1)/2+1,size(psfmax,2)/2+1);

% Scaling to account for mis-registration (also for sub-selection of
% sub-apertures)
if (sum(Coeffs(1:5).^2) ~= 0) || (Gauss_flag == 1)
    % Creation of the WF coordinates
    [x,y] = meshgrid(1:nbpixWF,1:nbpixWF);          % perfect coordinates
    
    % Scaling of WF coordinates
    xWF = x - nbpixWF/2-0.5 - Coeffs(1)/100/SHsize*nbpixWF; % x shift
    yWF = y - nbpixWF/2-0.5 - Coeffs(2)/100/SHsize*nbpixWF; % y shift
    [theta, rho] = cart2pol(xWF,yWF); theta = theta - Coeffs(3)/180*pi;
    [xWF, yWF] = pol2cart(theta, rho);
    xWF = (1 + Coeffs(4)/100)*xWF;                          % x stretch
    yWF = (1 + Coeffs(5)/100)*yWF;                          % y stretch
    xWF = xWF + nbpixWF/2+0.5;
    yWF = yWF + nbpixWF/2+0.5;
    
    % Scaling of the actual pupil mask (accounting for a finite DM of size
    % equal to the pupil)
    pupmask = interp2(x,y,pupmask,xWF,yWF,'nearest').*pupmask;
end
WF_mask = pupmask;
if (Gauss_flag ~= 0) && (nargin > 17)
    % Scaling of Actuators coordinates
    Act_coords3(:,1) = Act_coords(:,1) + Coeffs(1)/100/SHsize*nbpixWF;
    Act_coords3(:,2) = Act_coords(:,2) + Coeffs(2)/100/SHsize*nbpixWF;
    c = Coeffs(1)/100/SHsize*nbpixWF+SHsize*nbpixCCD/2+0.5;
    d = Coeffs(2)/100/SHsize*nbpixWF+SHsize*nbpixCCD/2+0.5;             % rotation
    [theta, rho] = cart2pol(Act_coords3(:,1)-c,Act_coords3(:,2)-d); theta = theta + Coeffs(3)/180*pi;
    [Act_coords3(:,1), Act_coords3(:,2)] = pol2cart(theta, rho);
    Act_coords3(:,1) = Act_coords3(:,1) + c ; Act_coords3(:,2) = Act_coords3(:,2) + d;
    Act_coords3(:,1) = (1 + Coeffs(4)/100)*(Act_coords3(:,1)-c)+c;      % x stretch
    Act_coords3(:,2) = (1 + Coeffs(5)/100)*(Act_coords3(:,2)-d)+d;      % y stretch
end
a = nbpixWF/nbpixCCD/SHsize; b = 0.5-a/2;
Pup_flux_loss = (Pup_flux_loss - nansum(pupmask(:))) / Pup_flux_loss;

% Removing NaNs from the pupil mask
% pupmask(isnan(pupmask)) = 0;

% Conversion of PS to microns and to the right r0 per layer
PS = PS * 0.5 / 2 / pi;
for cpt_lay = 1:size(Turb,2)
    PS(:,:,cpt_lay) = PS(:,:,cpt_lay) * ((Turb(2,cpt_lay))^(-5/6));
end
r0 = sum(Turb(2,:).^(-5/3))^(-3/5);

% Error checking
if strcmp(Ctrl_meth, 'geom') && (SH_model ~= 1)
    error('SH_model must be set to 1 to use the ''geom'' Control law')
end

if disp_ID == 1
    % Initialization of the display
    figure('Position',[10,10,1200,800])
end

Tag_params = 0;     % set to zero so that the first SHWFS measurement creates the params
WFdisp = 0;
% initialization of the parameters for the turbulence shift due to wind
[xPS, yPS] = meshgrid(1:size(PS,2),1:size(PS,1));
[xWF2, yWF2] = meshgrid(1:nbpixWF,1:nbpixWF);
wind_shift = Turb(1,:) / fqcy * nbpixWF / D;

% initialization of the parameters for the artificial display of spot in
% the geometric case.
[x_spot, y_spot] = meshgrid(-(nbpixCCD-1)/2:(nbpixCCD-1)/2);

% First SHWFS measurement in order to creates the detailed WF parameters
% file. In particular xyvalid_ref contains the full list of valid
% sub-apertures before sub-selection
[~, ~, wl, C, xyvalid_ref, xyfullill, xyf_of_v, ~] =...
    SHWFS_PSIM(zeros(nbpixWF,nbpixWF), SHWFS_PSIM_input, Coeffs, 0, 0);

% Main loop on the number of iterations
wbID=0;
for cpt = 1:max(Niter);
    wbID = wb(cpt,max(Niter), wbID);
    
    % cut WF
    WF = zeros(nbpixWF,nbpixWF);
    for cpt_lay = 1:size(Turb,2)
        WF = WF + rot90(interp2(xPS, yPS, PS(:,:,cpt_lay),...
            xWF2 + (cpt-1)*wind_shift(cpt_lay), yWF2, 'cubic'),Turb(3,cpt_lay)/90);
    end
    WF = WF.*WF_mask;
    
    if disp_ID == 1
        subplot(2,3,1) ; imagesc(2*pi*WF.*WF_mask/wl) ; colormap('bone') ; axis square ; colorbar ;...
            title(['Turbulent WF (radians ', num2str(wl), ' um)'])
    end
    
    % Computes the DM shape (not if 'Ctrl_meth' is set to 'geom') from the
    % Influence Functions (or modes) and the commands at this loop cycle.
    DM_shape = zeros(nbpixWF2,nbpixWF2,1);
    if ~strcmp(Ctrl_meth, 'geom') && (sum(cmd(cpt,:).^2) ~= 0)
        % loop on the Influence Functions / modes
        for cpt_IF = 1:size(IM,1)
            % mode loading in different cases
            if strcmp(modes_ID,'Z')
                % 1) Zernikes
                DM = createZ_PSIM(nbpixWF,cpt_IF,outmin,outmax); % 1 rad rms WF at wl
                DM = wl * DM / 2 / pi .* pupmask * 10^-6 / 2; % conversion to meters
            elseif (Gauss_flag == 0) || (Gauss_flag == 2)
                % 2) pre-recorded modes
                load([num2str(modes_path), 'Mode_', num2str(cpt_IF), '.mat']);
            elseif Gauss_flag == 1
                % 3) Gaussian IFs
                DM = 0*DM_shape;
                % sub-selection of sub-apertures
                if nargin > 17
                    tmp = sqrt(((xyvalid_ref(:,1)-0.5)*nbpixCCD+0.5-Act_coords3(cpt_IF,2)).^2 +...
                        ((xyvalid_ref(:,2)-0.5)*nbpixCCD+0.5-Act_coords3(cpt_IF,1)).^2);
                    [~, IX] = sort(tmp);
                    % xyvalid_index contain the information about the sub-selection of
                    % sub-apertures (the ones in which to compute the WF)
                    xyvalid_index = IX(1:min(nb_neighb_SA,end));
                    xyvalid = xyvalid_ref(xyvalid_index,:);
                end
                % Sub-selection of DM surface
                x1 = (min(xyvalid(:,1))-1)*nbpixWF/SHsize+1; x2 = max(xyvalid(:,1))*nbpixWF/SHsize;
                y1 = (min(xyvalid(:,2))-1)*nbpixWF/SHsize+1; y2 = max(xyvalid(:,2))*nbpixWF/SHsize;
                DM(x1:x2,y1:y2) = 0.5*(1e-6)*exp(-((yWF(x1:x2,y1:y2)-a*Act_coords(cpt_IF,2)-b).^2)/(2*(Coeffs(7)^2)) -...
                    ((xWF(x1:x2,y1:y2)-a*Act_coords(cpt_IF,1)-b).^2)/(2*(Coeffs(6)^2)));
            end
            DM_shape = DM_shape + amplitudes(cpt_IF)*DM.*cmd(cpt,cpt_IF);
        end
        
        % Interpolation to actual pupil mis-registration characteristics.
        % The factor 10^6 is for the conversion from meters to microns
        if (Gauss_flag ~= 1) && (sum(Coeffs(1:5).^2) ~= 0)
            DM_shape = interp2(x,y,DM_shape,xWF,yWF,'spline').*pupmask * 1e6;
        else
            % if Gaussian IFs, mis-registration already included
            DM_shape = DM_shape .* pupmask * 1e6;
        end
        
        % In non-'geom' mode, DM correction is added here. The factor 2 is
        % for convertion from Surface to WF
        WF = WF - 2 * DM_shape; % in microns
        % In 'geom' mode, the turbulent WF is sampled by the SHWFS, not the
        % residual after DM correction.
    end
    
    % WF goes through SHWFS
    [~, xyslopes, wl, ~, xyvalid, ~, ~, Tag_params, CCDall2] =...
        SHWFS_PSIM(WF, SHWFS_PSIM_input, Coeffs, WFdisp, Tag_params);
    % reshaping of the slopes
    Sl_sig = zeros(1,size(IM,2));
    tmp = xyslopes(:,:,1);
    Sl_sig(1,1:length(xyvalid)) = tmp(xyvalid(:,3));
    tmp = xyslopes(:,:,2);
    Sl_sig(1,(length(xyvalid)+1):2*length(xyvalid)) = tmp(xyvalid(:,3));
    
    if strcmp(Ctrl_meth, 'geom')
        % compute DM shape in the WFS space, including mis-registration
        % (accounted for in the IM)
        DM = cmd(cpt,:) * IM;
        % in 'geom' mode, DM correction is added here
        meas(cpt,:) = Sl_sig - DM;
        % reshaping of the slopes for WF reconstruction
        tmp = NaN*xyslopes(:,:,1);
        tmp(xyvalid(:,3)) = meas(cpt,1:length(xyvalid));
        xyslopes(:,:,1) = tmp;
        tmp(xyvalid(:,3)) = meas(cpt,(length(xyvalid)+1):end);
        xyslopes(:,:,2) = tmp;
    else
        meas(cpt,:) = Sl_sig;
        WF_mask = pupmask ; WF_mask(WF_mask == 0) = NaN;
        WF_res = 2*pi*WF.*WF_mask/wl;
    end
    
    % Reconstruction of WF from slopes based on the Roddier algorithm
    pupille = NaN*xyslopes(:,:,1) ; pupille(xyvalid(:,3)) = 1;
    nanpupille = pupille; pupille(isnan(pupille)) = 0;
    xyslopes(isnan(xyslopes)) = 0;
    WF_res_Rod = dp2phi2(xyslopes(:,:,1)*pix_scale/180/3600*pi*1000,xyslopes(:,:,2)*pix_scale/180/3600*pi*1000,pupille,3,10,D/SHsize,1);
    WF_res_Rod = WF_res_Rod.*pupille*2*pi/wl*1e6;
    
    % In 'geom' mode, the reconstructed WF is the only one available
    if strcmp(Ctrl_meth, 'geom')
        WF_res = WF_res_Rod;
        pupmask = pupille; nanpupmask = nanpupille;
    end
    
    WFvar(cpt,1) = proj(WF_res.*nanpupmask,Z(:,:,1).*nanpupmask);
    WFvar(cpt,2) = proj(WF_res.*nanpupmask,Z(:,:,2).*nanpupmask);
    WFvar(cpt,3) = proj(WF_res_Rod.*nanpupille,Z_Rod(:,:,1).*nanpupille);
    WFvar(cpt,4) = proj(WF_res_Rod.*nanpupille,Z_Rod(:,:,2).*nanpupille);
    WFtmp = WF_res.*nanpupmask - WFvar(cpt,1)*Z(:,:,1) - WFvar(cpt,2)*Z(:,:,2);
    WFvar(cpt,5) = nanvar(WFtmp(:));
    WFtmp = WF_res_Rod.*nanpupille - WFvar(cpt,3)*Z_Rod(:,:,1) - WFvar(cpt,4)*Z_Rod(:,:,2);
    WFvar(cpt,6) = nanvar(WFtmp(:));
    
    SR(cpt,1) = exp(-WFvar(cpt,5)*((wl/wl_im)^2));
    SR(cpt,2) = exp(-WFvar(cpt,6)*((wl/wl_im)^2));

    % In geometric model, the artificial WFS image has to be computed
    % because a) it is not computed by the sub-routine
    % 'SHWFS_PSIM_Mainloop' and b) if it were, it would be for the full
    % WF, not residual
    if (disp_ID == 1) && (SH_model == 1)
        for subap = 1:size(xyvalid,1)
            xWF0 = xyvalid(subap,1)-1 ; yWF0 = xyvalid(subap,2)-1;
            CCDall2(xWF0*nbpixCCD+1:xWF0*nbpixCCD+nbpixCCD,yWF0*nbpixCCD+1:yWF0*nbpixCCD+nbpixCCD) =...
                exp(-((x_spot-meas(cpt,subap)).^2 + (y_spot-meas(cpt,subap + length(xyvalid))).^2)*log(2));
        end
    end
    
    % Compute the correction
    if ~strcmp(Ctrl_meth, 'POLC')
        tmp = meas(cpt,:);
        tmp(isnan(tmp)) = 0;
        Corr = tmp * CM;
        cmd(cpt+delay,:) = cmd(cpt+delay-1,:) + gain * Corr;
    else
        % Place holder for POLC
    end
    
    [psf, wl_im, C] = makepsf_PSIM(WF_res/2/pi*wl, pupmask, wl_im, D, -0.5, size(WF_res,1), 0, 1);
    SR(cpt,3) = psf(size(psfmax,1)/2+1,size(psfmax,2)/2+1)/SRmax;
    SR(cpt,4) = max(psf(:))/SRmax;
    if cpt >= min(Niter)
        psf2 = psf2 + psf;
        SR(cpt,5) = psf2(size(psfmax,1)/2+1,size(psfmax,2)/2+1)/SRmax/(cpt-min(Niter)+1);
    end
    
    if strcmp(Ctrl_meth, 'geom')
        % Addition of the fitting error, only for the Strehl values comnputed
        % from the WF reconstructed from slopes
        SR(cpt,1:5) = SR(cpt,1:5)*exp(-0.26*((D/SHsize/r0)^(5/3))*((wl/wl_im)^2));
        % Addition of the effect of pupil shape, estimated to be equal to
        % the flux loss.
        SR(cpt,1:5) = SR(cpt,1:5) - Pup_flux_loss;
    else
        % Addition of the fitting error, only for the Strehl values comnputed
        % from the WF reconstructed from slopes
        SR(cpt,2) = SR(cpt,2)*exp(-0.26*((D/SHsize/r0)^(5/3))*((wl/wl_im)^2));
        % Addition of the effect of pupil shape
        SR(cpt,1:2) = SR(cpt,1:2) - Pup_flux_loss;
        % Removal of the effect of the flux loss.
        SR(cpt,3:5) = SR(cpt,3:5) + Pup_flux_loss;
    end
    
    
    if disp_ID == 1
        subplot(2,3,3) ; imagesc(WF_res*wl/wl_im) ; axis square ; colorbar ; title(['Residual WF (radians at ', num2str(wl_im), ' um)'])
        subplot(2,3,6) ; imagesc(CCDall2) ; axis square ; title(['WFS image (at ', num2str(wl), ' um)'])
        subplot(2,3,4) ; hold off ; plot(SR) ; grid on ;...
            title(['Strehl Ratio (at ', num2str(wl_im), ' um)']) ;...
            legend('Inst. from WF','Inst. from WF rec. from slopes (Roddier)',...
            'Center of inst. PSF','Max of inst. PSF','From LE PSF','Location','South')
        subplot(2,3,2) ; imagesc(DM_shape) ; axis square ; colorbar ; title('DM shape (Surface in microns)')
        subplot(2,3,5) ; imagesc(log(abs(psf + sum(sum(psf))/100000*randn(size(psf))))) ;...
            axis square ; title(['Simulated PSF (at ', num2str(wl_im), ' um)'])
    end
end

save([num2str(filename), '_data.mat'],'meas','cmd','WFvar','SR');

write_fits([num2str(filename), '_meas.fits'],meas);
write_fits([num2str(filename), '_cmd.fits'],cmd);
write_fits([num2str(filename), '_SR.fits'],SR);
write_fits([num2str(filename), '_psf.fits'],psf2);

% end of function