function [p_res, fval, psf2, FWHM] = MGfit(psf, fit_meth)

% [p_res, fval, psf2, FWHM] = MGfit(psf, fit_meth);
% 
% This function finds the best Gaussian of Moffat fit (circular or
% elliptical) to an input PSF, by minimization of a criterion (rms of the
% difference between the input PSF and a model one) and with free
% parameters the ones describing the model PSF.
% The inputs are:
% - 'psf' the input PSF
% - 'fit_meth' is the fitting method to be used, and should be set to:
%   - 1 for regular Gaussian (requires 5 input parameters)
%   - 2 for elliptical Gaussian (requires 7 input parameters)
%   - 3 for regular Moffat (requires 6 input parameters)
%   - 4 for elliptical Moffat (requires 8 input parameters)
% The outputs are:
% - 'p_res' the best set of parameters
% - 'fval' the value of the minimization criterion for this set of
%   parameters
% - 'psf2' is the model PSF built from this set of parameters
% - 'FWHM' is the FWHM of 'PSF2'. It is a single value in case of a
%   circular function and a vector of 2 values in case of an elliptical
%   function. It is computed from the sigma(s) of the model function.
% 
% A typical entry could be:
% [p_res, fval, psf2, FWHM] = MGfit(psf, 4);
%
% Custom sub-routines required:
% 1) First level
% - MGfit_crit.m
% 1) Other levels
% - crop_PSIM.m
% - nansum.m (Statistics Toolbox)
% 
% v1.0 J.Kolb 05/12/12
% - first release of the fully commented file together with version 2.0 of
%   the PSIM code.

% Generation of the coordinates maps
[x,y] = meshgrid(1:size(psf,1),1:size(psf,2));

% Automatic setting of the starting parameters
switch fit_meth
    case 1 % Circular Gaussian (requires 5 input parameters)
        p0 = [median(psf(:)) max(psf(:))-median(psf(:)) round(size(psf)/2) round(length(psf)/10)];
    case 2 % Elliptical Gaussian (requires 7 input parameters)
        p0 = [median(psf(:)) max(psf(:))-median(psf(:)) round(size(psf)/2) round(size(psf)/10) pi];
    case 3 % Circular Moffat (requires 6 input parameters)
        p0 = [median(psf(:)) max(psf(:))-median(psf(:)) round(size(psf)/2) round(length(psf)/10) 2];
    case 4 % Elliptical Moffat (requires 8 input parameters)
        p0 = [median(psf(:)) max(psf(:))-median(psf(:)) round(size(psf)/2) round(size(psf)/10) pi 2];
end

% Determination of the best set of parameters by minimization of the rms
% between the input PSF and the model one (output of the function
% 'MGfit_crit')
[p_res, fval] = fminsearch(@(p)MGfit_crit(psf, 0, x, y, p, fit_meth), p0,...
optimset('TolX',10^-3,'TolFun',10^-3,'MaxFunEvals',10000,'MaxIter',10000)); 

% Generation of the modeal PSF with the best fitting parameters and
% conmputation of the FWHM
[~, psf2, FWHM] = MGfit_crit(psf, 0, x, y, p_res, fit_meth);

% end of function