function [Crit, PSF2, FWHM] = MGfit_crit(PSF, psf_dl, X, Y, p, fit_meth)

% [Crit, PSF2, FWHM] = MGfit_crit(PSF, psf_dl, X, Y, p, fit_meth);
% 
% This function generates a Gaussian or Moffat PSF from the input
% parameters, and computes a criterion equivalent to the rms of the
% difference between an input PSF and this model one. NaNs are ignored. The
% FWHM of the model PSF is also computed.
% The inputs are:
% - 'PSF' the input PSF (in a rectangular matrix)
% - 'psf_dl' the diffraction limited PSF. It should be set to zero if the
%   source is unresolved. Otherwise it should be an image of the same
%   dimensions as 'PSF'. This input is useful to disentangle between the
%   contributions of the input WF and of the diffraction.
% - 'X' and 'Y' are the coordinate maps of the pixels in the input image,
%   generated by the 'meshgrid' function
% - 'fit_meth' is the fitting method to be used, and should be set to:
%   - 1 for regular Gaussian (requires 5 input parameters)
%   - 2 for elliptical Gaussian (requires 7 input parameters)
%   - 3 for regular Moffat (requires 6 input parameters)
%   - 4 for elliptical Moffat (requires 8 input parameters)
% - 'p' a vector that contains the input parameters
%   - p(1) is the offset
%   - p(2) is the amplitude
%   - p(3:4) are the coordinates of the center of the PSF, in pixels
%   - for a circular Gaussian fit, p(5) is the sigma of the Gaussian, in
%     pixels
%   - for an elliptical Gaussian fit, p(5:6) are the sigmas of the Gaussian
%     along the short and long axes in pixels, and p(7) is the orientation
%     angle in radians
%   - for an circular Moffat fit, p(5) is the sigma of the function in
%     pixels and p(6) the beta coefficient (in the exponent)
%   - for an elliptical Moffat fit, p(5:6) are the sigmas of the function
%     along the short and long axes in pixels, p(7) is the orientation
%     angle in radians and p(8) the beta coefficient (in the exponent).
% The outputs are:
% - 'Crit' is equivalent to the rms of the difference between an input PSF
%   and the model one 'PSF2'
% - 'PSF2' is the model PSF built from the input parameters
% - 'FWHM' is the FWHM of 'PSF2'. It is a single value in case of a
%   circular function and a vector of 2 values in case of an elliptical
%   function. It is computed from the sigma(s) of the model function.
% 
% A typical entry could be:
% [Crit, PSF2, FWHM] = MGfit_crit(PSF, 0, X, Y, [100 10^6 64 64 10 10 pi 2], 4);
%
% Custom sub-routines required:
% 1) First level
% - crop_PSIM.m
% - nansum.m (Statistics Toolbox)
% 
% v1.0 J.Kolb 05/12/12
% - first release of the fully commented file together with version 2.0 of
%   the PSIM code.

% Selection of function
switch fit_meth
    case 1 % Circular Gaussian (requires 5 input parameters)
        PSF2 = p(1) + abs(p(2))*exp(-((X-p(3)).^2+(Y-p(4)).^2)/(2*p(5)*p(5)));
        FWHM = 2*sqrt(2*log(2))*p(5);
        
    case 2 % Elliptical Gaussian (requires 7 input parameters)
        p(5:6) = sort(p(5:6));
        a = 0.5*(cos(p(7))/p(5))^2 + 0.5*(sin(p(7))/p(6))^2;
        b = -0.25*sin(2*p(7))/(p(5)^2) + 0.25*sin(2*p(7))/(p(6)^2);
        c = 0.5*(sin(p(7))/p(5))^2 + 0.5*(cos(p(7))/p(6))^2;
        PSF2 = p(1) + p(2)*exp(-(a*((X-p(3)).^2)+2*b.*(X-p(3)).*(Y-p(4))+c*((Y-p(4))).^2));
        FWHM = 2*sqrt(2*log(2))*p(5:6);
        
    case 3 % Circular Moffat (requires 6 input parameters)
        PSF2 = p(1) + abs(p(2))*(1+(((X-p(3)))/p(5)).^2 + ((-(Y-p(4)))/p(5)).^2).^(-p(6));
        FWHM = 2*sqrt((2^(1/p(6))-1)/((1/p(5))^2));
        
    case 4 % Elliptical Moffat (requires 8 input parameters)
        p(5:6) = sort(p(5:6));
        a = 0.5*(cos(p(7))/p(5))^2 + 0.5*(sin(p(7))/p(6))^2;
        b = -0.25*sin(2*p(7))/(p(5)^2) + 0.25*sin(2*p(7))/(p(6)^2);
        c = 0.5*(sin(p(7))/p(5))^2 + 0.5*(cos(p(7))/p(6))^2;
        PSF2 = p(1) + p(2)*(1+2*(a*((X-p(3)).^2)+2*b.*(X-p(3)).*(Y-p(4))+c*((Y-p(4))).^2)).^(-p(8));
        FWHM = 2*p(5:6)*sqrt((2^(1/p(8)))-1);
end
if psf_dl ~= 0
    PSF3 = conv2(PSF2,psf_dl);
    PSF2 = crop_PSIM(PSF3, size(PSF), size(PSF)+0.5);
end

Crit = sqrt(nansum(nansum((PSF-PSF2).^2)));

% end of function