function [PS, D] = createPS_PSIM(PSsize, enl_fact, L0, Scale, r0, PSdisp, D)

% [PS, D] = createPS_PSIM(PSsize, enl_fact, L0, Scale, r0, PSdisp, D);
% 
% This routine returns a turbulence phase screen in radians computed via
% the Fourier approach: a randomized white noise map is colored in the
% Fourier space by the turbulence power spectral density (PSD) function,
% and the inverse Fourier transform of the outcome correspond to a phase
% screen realization.
% The input parameters are:
% - 'PSsize' the requested size of the output PS in pixels. It is a vector
%   of dimensions [2x1] or [1x2]. If it is a single value then this value
%   is used for both dimensions
% - 'enl_fact' is the size over which the PS should be computed before
%   being truncated to 'PSsize', in units of PSsize. Its smaller possible
%   value is 2. It is always preferable to generate a PS over a (much)
%   larger diameter than the pupil one, in order to avoid aliasing and to
%   reproduce properly the desired Outer Scale
% - 'L0' is the desired Outer Scale [m]
% - 'Scale' is the scaling factor of the array in [Meters/pixels]. To be
%   well sampled it is better to have at least 2 pixels per r0 (even
%   better with 4)
% - 'r0' is the desired Fried parameter [m]
% - if 'PSdisp' is set to 1, the PS will be displayed at the end
% - if the coordinate matrix 'D' is inputted (optional), it will save the
%   time to recompute it. This is useful in the case many Phase Screens are
%   generated in a row using this function.
% The outputs are:
% - 'PS" the computed Phase Screen in radians
% - 'D' is an output of the function so that it can be reused as input the
%   next time the function runs.
% 
% A typical entry could be:
% [PS_1, D] = createPS_PSIM([256 1024], 2, 22, 8/256, 0.1, 1);
% And just after the next call is much faster with 'D' inputted:
% [PS_2, ~] = createPS_PSIM([256 1024], 2, 22, 8/256, 0.1, 0, D);
%
% Custom sub-routines required:
% 1) First level
% - ficac.m
%
% v1.0 J.Kolb 04/05/12
% - first release of the commented function
% v2.0 J.Kolb 04/12/12
% - improved detection of input of the parameter 'D'

% Tests on the size of the Phase screen
if enl_fact < 2
    % 'enl_fact' must be equal to or larger than 2
    error('enl_fact must be larger than two (and as large as possible) to avoid aliasing');
elseif enl_fact > 2048/max(PSsize)
    % The generation of too large Phase Screens (more than 4000 or 6000
    % pixels wide including enlargement factor) might block Matlab for an
    % undefinite amout of time. To avoid this, the user has 5 seconds to
    % abort the routine after seing the warning 
    disp('This enl_fact might require too much memory to process');
    disp('Press "CTRL + C" to stop');
    for cpt = 1:5
        disp(num2str(6-cpt)) ; pause(1);
    end
end

% Initializations
N = max(PSsize) * enl_fact;
if length(PSsize) == 1
    PSsize(2) = PSsize(1);
end

% Phase matrix generation
Phase = randn(N,N);
disp('Phase matrix created')

% Computation of 'D' (if not inputted)
if nargin < 7
    % Generation of the matrix of distance to the array corner
    [x,y] = meshgrid(((-N/2+1):(N/2)));
    Dist = (x.^2 + y.^2);
    disp('Dist matrix created')
    
    % Re-arrangement of the 4 quadrants of the coordinates matrix
    Dist2(1:N/2,1:N/2) = rot90(Dist(1:N/2,1:N/2),2);
    Dist2(N/2+1:2*N/2,1:N/2) = rot90(Dist(N/2+1:2*N/2,1:N/2),2);
    Dist2(1:N/2,N/2+1:2*N/2) = rot90(Dist(1:N/2,N/2+1:2*N/2),2);
    Dist2(N/2+1:2*N/2,N/2+1:2*N/2) = rot90(Dist(N/2+1:2*N/2,N/2+1:2*N/2),2);
    Dist2 = Dist2/((N*Scale)^2);
    disp('Dist2 matrix created')
    clear Dist

    % scaling of the matrix according to the Van Karman spectrum in a power of (-11/6)
    cst = (gamma(11/6)^2/(2*(pi^(11/3))))*((24*gamma(6/5)/5)^(5/6));
    D = cst*(r0^(-5/3))*((Dist2+(1/L0)^2).^(-11/6));
    % For normalization purposes
    D(1,1) = D(1,2);
    disp('D matrix created')
    clear Dist x y
end

% Inverse fft to create the Phase screen
WF = real(fft2(ifft2(Phase)/Scale.*sqrt(D)));
PS = WF(1:PSsize(1),1:PSsize(2));
disp('PS created')

% Display
if PSdisp == 1
    ficac(PS)
end

% end of function