function WF = dp2phi2(slopex, slopey, pupille, precis, nbiter, cste, tilt)

% WF = dp2phi2(slopex, slopey, pupille, precis, nbiter, cste, tilt);
% 
% This function reconstructs a Wavefront map 'WF' in microns out of the
% slopes provided by a Shack-Hartmann wavefront sensor 'slopex' and
% 'slopey', in MILLIRADIANS.
% It has been translated and adapted from an ONERA IDL routine written by
% L. Rousset-Rouviere based on the Roddier algorithm (App. Opt. vol 30, no
% 11,1991, pp1325-1327).
% No acceleration of the convergence (by apodisation) has been implemented
% yet. Usually, the convergence is reached after two or three iterations
% The result obtained is  close from the result obtained by the algortihm
% of Imagine-optic (for the HASO WFS), but there is a typical 10%
% difference observed, mostly in low order modes (where the S-H sensors are
% noisy).
% The inputs are:
% - The WF slopes 'slopex' and 'slopey', matrices of size [SHsize,SHsize]
%   where 'SHsize' is the linear number of sub-apertures in the WFS.
% - 'pupille' has the same size, and has for values 1 when the sub-aperture
%   is inside the pupil (i.e. to be used for the computation) and 0 when
%   outside.
% - 'precis' is the base 10 logarithm of the "accuracy" required, ie the
%   ratio between the variance of the slopes measured and the variance of
%   the slopes estimated after an iteration; barely better than 2.5.
% - 'nbiter' is the maximum number of iterations; usually no gain is
%   observed beyond 5 to 10. 
% - 'cste' is the size of a sub-aperture in meters.
% - 'tilt' = 0: WFE without tilt; 'tilt' = 1: tilt removed before iteration
%   then added.
% 
% A typical entry could be:
% WF = dp2phi2(xyslopes(:,:,1)*pix_scale/180/3600*pi*1000,...
%   xyslopes(:,:,2)*pix_scale/180/3600*pi*1000,pup_mask,3,10,D/SHsize,1); 
%
% Custom sub-routines required:
% - none
%
% v1.0 J.Kolb 24/05/12
% - added comments
% v2.0 J.Kolb 05/12/12
% - improved HELP
% - first release of fully commented file together with version 2.0 of the
%   PSIM code.

% Initializations
nsspupx=size(pupille,1);
nsspupy=size(pupille,2);
pupille_inv=1-pupille;
index=find(~(pupille_inv));

% Creation of the u,v functions for derivation in Fourier plane
rmpu=[-500./cste:1000./cste./(nsspupx-1):500./cste];
rmpv=[-500./cste:1000./cste./(nsspupy-1):500./cste];
fx=2i.*pi.*repmat(rmpu,nsspupy,1);
fy=2i.*pi.*repmat(rmpv',1,nsspupx);
absfx=fx.*conj(fx);
absfy=fy.*conj(fy);
absfxfy=fx.*conj(fy);
absfyfx=fy.*conj(fx);
denom=(absfx+absfy);
denom(denom<1e-3)=1;

% Subtraction of the average slopes to the initial data
mslopex=mean(slopex(index));
mslopey=mean(slopey(index));
slopex=pupille.*(slopex-mslopex);
slopey=pupille.*(slopey-mslopey);
slopex_e=slopex;
slopey_e=slopey;

% Variance of the measured slopes and initialisation of the loop break
% criteria
err_mes=var(slopex_e(index))+var(slopey_e(index));
err_calc=1000;
cpt=0;

% The while loop iterates the Fourier transform to get the estimate of the
% slopes, and breaks the loop after 'nbiter' or when the error is small
% enough between the measured slopes ('slopex', 'slopey') and the estimated
% ones ('slopex_e', 'slopey_e').
while (cpt<nbiter && (err_calc>(err_mes./10.^precis)))

    % Addition of the estimated slopes to the measured ones (outside the measured area)
    slopex_e=slopex+pupille_inv.*slopex_e;
    slopey_e=slopey+pupille_inv.*slopey_e;
    
    % New slopes FT estimation (without reconstructing the phase)
    slopex_tild = fftshift(fft2(slopex_e));
    slopey_tild = fftshift(fft2(slopey_e));
    slopex_tild_e = (absfx.*slopex_tild+absfxfy.*slopey_tild)./denom;
    slopey_tild_e = (absfy.*slopey_tild+absfyfx.*slopex_tild)./denom;
    % Estimated slopes    
    slopex_e = ifft2(ifftshift(slopex_tild_e));
    slopey_e = ifft2(ifftshift(slopey_tild_e));
    
    % Reconstruction error
    err_calc=var(abs(slopex_e(index)-slopex(index)))+var(abs(slopey_e(index)-slopey(index)));
    cpt=cpt+1;
end
% [err_mes err_calc cpt];

% Reconstruction of the phase
phi_tild_e=(conj(fx).*(slopex_tild_e) + conj(fy).*(slopey_tild_e))./denom;
phi_e = ifft2(ifftshift(phi_tild_e));
phi_tilt = [-0.5*(size(slopex,1)-1):0.5*(size(slopex,1)-1)]'*(mslopey.*ones(1,size(slopex,2))) +...
    (mslopex.*ones(1,size(slopex,2)))'*[-0.5*(size(slopex,1)-1):0.5*(size(slopex,1)-1)];

% 'slopex' and 'slopey' being angles in mrad, there is a 1000 factor

phi_e = phi_e + cste./1000.*tilt.*phi_tilt;
WF = real(phi_e -mean(phi_e(index))).*pupille;

% end of function