function y = newVKaiaj4(i0,j0,D,f0,r0,nlim)

% y = newVKaiaj4(i0,j0,D,f0,r0,nlim);
% 
% The goal of this function is to compute the covariance of the Zernike
% polynomials in a turbulent atmosphere with a Van Karman spectrum.
% The inputs are:
% - the orders of the Zernike polynomials 'i0' and 'j0'. They can be single
%   numbers or vectors.
% - the diameter of the aperture 'D'
% - 'f0' the inverse of the outer scale in meters (f0 = 1/L0)
% - 'r0' the Fried parameter in meters
% - 'nlim' is maximum number of orders to use for computing the
%   normalization factor to the full varuiance of the turbulence (optional,
%   set to 50 by default)
% The output is the covariance matrix 'y'. It diagonal contains the
% variance of the Zernike polynomials in a an Karman spectrum defined by
% the input r0 and L0.
% 
% A typical entry could be:
% Noll = diag(newVKaiaj4(1:Zmax,1:Zmax,D,1/L0,r0))';
% 
% Custom sub-routines required:
% - none
% 
% v1.0 J.Kolb 11/12/12
% - function created by R.Conan before 2002

% Initializations
cst = (2.*gamma(11./6)./pi.^(3./2)).*(gamma(6./5)*24./5)^(5/6);

[ni0,mi0]   = FindNM(i0);
[nj0,mj0] = FindNM(j0);

l = length(i0);
if nargin<6
   nlim = 50;
end

k = 0:nlim;
factk = [1 cumprod(1:nlim)];
red = (-1).^k./factk;
piDf0 = pi.*D.*f0;

y = zeros(l);

% Main loops on the modes
for cpt1=1:l
   for cpt2=1:l
      i = i0(cpt1);
      j = j0(cpt2);
      ni = ni0(cpt1);
      nj = nj0(cpt2);
      mi = mi0(cpt1);
      mj = mj0(cpt2);
      
      if ((~rem(abs(i-j),2) | mi==0) & (mi==mj))
         
         G1 = gamma(11./6+k).*gamma(-5./6+(ni+nj)./2-k).*gamma(7./3+k).*gamma(17./6+k)./...
            (gamma(23./6+(ni+nj)./2+k).*gamma(17./6+(ni-nj)./2+k).*gamma(17./6+(nj-ni)./2+k));
         
         G2 = gamma(1+(ni+nj)./2+k).*gamma(5./6-(ni+nj)./2-k).*gamma(1.5+(ni+nj)./2+k).*gamma(2+(ni+nj)./2+k)./...
            (gamma(3+(ni+nj)+k).*gamma(2+nj+k).*gamma(2+ni+k));
         
         y(cpt1,cpt2) = cst.*sqrt((ni+1).*(nj+1)).*(-1).^((ni+nj-(mi+mj))./2).*...
            sum(red.*(G1.*piDf0.^(2.*k)+G2.*piDf0.^(2.*k+ni+nj-5./3)));
         
      end         
   end
end

% Scaling to the inputted r0
if nargin>4
   y = y.*(D./r0).^(5./3);
end

% end of function