function Coeff = proj(A,B,C)

% Coeff = proj(A,B,C);
% 
% This function computes the projection of the matrix 'A' on the matrix
% 'B', in the sense of the least squares. 'A' and 'B' can be 2D matrices or
% vectors, but have to have the same size. NaNs are ignored.
% The output 'Coeff' is a number.
% When 'proj' is used in a loop, it can be time-saving to also provide the
% norm of 'B' in stead of recomputing each at each loop. It can be done in
% the optional input 'C' (a number)
% 
% A typical entry could be:
% Coeff(1,2) = proj(IM_1,IM_2);
%
% Custom sub-routines required:
% 1) First level
% - nanmean.m (Statistics Toolbox)
% - nansum.m (Statistics Toolbox)
%
% v1.0 J.Kolb 02/05/12
% - first release of the commented function
% v2.0 J.Kolb 30/08/12
% - added third input for faster computations

% Error checking
if (numel(A) ~= numel (B)) || (size(A,1) ~= size(B,1))
    error('Input matrices must have the same size')
end

% Offset removal before projection
A = A - nanmean(A(:));
B = B - nanmean(B(:));

% Projection
if nargin > 2
    Coeff = nansum(nansum(A.*B))./C;
else
    Coeff = nansum(nansum(A.*B))./nansum(nansum(B.*B));
end

% end of function