function write_fits(fname, array, datatype)

% write_fits(fname, array, datatype);
%
% Tis function writes 1D, 2D or 3D array into a fitsfile "more or less"
% compliant to "ESO standards", i.e. that can be inputted to the RTC of an
% ESO AO system.
% WRITE_FITS(fname,array)  writes "array" into fitsfile "fname".
% If "datatype" is set, the array is forced to the data type specified
% Allowed data types: 'int8', 'uint8', 'int16', 'uint16', 'int32',
% 'uint32', 'single', 'double'.
% 
% A typical entry could be:
% write_fits('AOF_Interaction_Matrix_1.fits', IM, 'int16');
%
% Custom sub-routines required:
% - none
%
% v1.0 J.Kolb 04/05/12
% - first release of the commented function

% Size of the array
si=size(array);

% Setting of output data type
if ~exist('datatype','var')
    dt=class(array);
else
    dt=datatype;
end

% Header iniitialization
hdr=ones(36,80)*32;
hdr=char(hdr);

% Writing of the header part about array dimansions
lnx = 3;
if length(si)==2,
    hdr(lnx,1:80)=['NAXIS   =                    2 / Number of data axes                            '];
    lnx = lnx+1;
    line =        ['NAXIS1  =                  ' sprintf('%d',si(1))];
    hdr(lnx,1:length(line))=line;
    lnx = lnx+1;
    line =        ['NAXIS2  =                  ' sprintf('%d',si(2))];
    hdr(lnx,1:length(line))=line;
    lnx = lnx+1;
elseif length(si)==3,
    hdr(lnx,1:80)=['NAXIS   =                    3 / Number of data axes                            '];
    lnx = lnx+1;
    line =        ['NAXIS1  = ' sprintf('%d',si(1))];
    hdr(lnx,1:length(line))=line;
    lnx = lnx+1;
    line =        ['NAXIS2  = ' sprintf('%d',si(2))];
    hdr(lnx,1:length(line))=line;
    lnx = lnx+1;
    line =        ['NAXIS3  = ' sprintf('%d',si(3))];
    hdr(lnx,1:length(line))=line;
    lnx = lnx+1;
end

% Writing of the header part about data format
hdr(lnx,1:80)=    ['EXTEND  =                    T / FITS data may contain extensions               '];
lnx = lnx+1;
hdr(1,1:80)=      ['SIMPLE  =                    T / File does conform to FITS standard             '];

if strmatch('int8',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                    8 / Number of bits per data pixel                  '];
elseif strmatch('uint8',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                    8 / Number of bits per data pixel                  '];
    hdr(lnx,1:80)=['BZERO   =                  128 / Data are unsigned byte                         '];
    lnx = lnx+1;
    hdr(lnx,1:80)=['BSCALE  =                    1 / Data scaling for unsigned integers             '];
    lnx = lnx+1;
elseif strmatch('int16',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                   16 / Number of bits per data pixel                  '];
elseif strmatch('uint16',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                   16 / Number of bits per data pixel                  '];
    hdr(lnx,1:80)=['BZERO   =                32768 / Data are unsigned integer                      '];
    lnx = lnx+1;
    hdr(lnx,1:80)=['BSCALE  =                    1 / Data scaling for unsigned integers             '];
    lnx = lnx+1;
elseif strmatch('int32',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                   32 / Number of bits per data pixel                  '];
elseif strmatch('uint32',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                   32 / Number of bits per data pixel                  '];
    hdr(lnx,1:80)=['BZERO   =           2147483648 / Data are unsigned long                         '];
    lnx = lnx+1;
    hdr(lnx,1:80)=['BSCALE  =                    1 / Data scaling for unsigned integers             '];
    lnx = lnx+1;
elseif strmatch('single',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                  -32 / Number of bits per data pixel                  '];
elseif strmatch('double',dt,'exact')==1,
    hdr(2,1:80)=  ['BITPIX  =                  -64 / Number of bits per data pixel                  '];
end

hdr(lnx,1:80)=    ['COMMENT FITS (Flexible Image Transport System) format is defined in Astronomy   '];
lnx = lnx+1;
hdr(lnx,1:80)=    ['COMMENT and Astrophysics, volume 376, page 359; bibcode 2001A&A...376..359H     '];
lnx = lnx+1;
hdr(lnx,1:80)=    ['END                                                                             '];
hdr=hdr';
hdr=hdr(:);

% Opens the fits file to write in
fid=fopen(fname,'w','b');
cnt=fwrite(fid,hdr,'char');
cnt1=fwrite(fid,array(:),dt);

% Padding with bytes to complete the 2880 bytes block
if strmatch('int8',dt,'exact')==1
    cnt=cnt+cnt1;
    nzer=(2880-rem(cnt,2880));
elseif strmatch('uint8',dt,'exact')==1,
    cnt=cnt+cnt1;
    nzer=(2880-rem(cnt,2880));    
elseif strmatch('int16',dt,'exact')==1,
    cnt=cnt+cnt1*2;
    nzer=(2880-rem(cnt,2880))/2;
elseif strmatch('uint16',dt,'exact')==1,
    cnt=cnt+cnt1*2;
    nzer=(2880-rem(cnt,2880))/2;    
elseif strmatch('int32',dt,'exact')==1,
    cnt=cnt+cnt1*4;
    nzer=(2880-rem(cnt,2880))/4;
elseif strmatch('uint32',dt,'exact')==1,
    cnt=cnt+cnt1*4;
    nzer=(2880-rem(cnt,2880))/4;    
elseif  strmatch('single',dt,'exact')==1,
    cnt=cnt+cnt1*4;
    nzer=(2880-rem(cnt,2880))/4;
elseif  strmatch('double',dt,'exact')==1,
    cnt=cnt+cnt1*8;
    nzer=(2880-rem(cnt,2880))/8;
end    

% Writes the fits file
fwrite(fid,zeros(nzer,1),dt);
fclose(fid);

% End of function