function corr_coeff = Corr_PSIM(CCDin, param, barxy)

% corr_coeff = Corr_PSIM(CCDin, param, barxy);
% 
% This function computes 'corr_coeff', opposite of the correlation
% coefficient between the image 'CCDin' and a Gaussian of FWHM 'param'
% centered on the coordinates 'barxy'.
% It can be used as criterion to minimize by a function such as
% 'fminsearch', for example in the method of centroiding determination by
% correlation (function 'centroid_PSIM.m')
% 
% A typical entry could be
% corr_coeff = Corr_PSIM(CCD_subimage, 4, [3.5 3.5]);
% 
% Custom sub-routines required:
% - none
% 
% v1.0 J.Kolb 16/12/10
% v2.0 J.Kolb 13/04/12
% - added error checking
% - corrected error in typical entry example

% initialization
l_psf = size(CCDin);

% error checking
if max(l_psf) == 1
    error('The dimension of the image should be at least [2x2]')
end
if (param <= 0) || (length(param) > 1)
    error('The input parameter should be a positive number')
end
if length(barxy) ~= 2
    error('The input barycenter coordinates should be a [2x1] vector')
end

% Conversion of Gaussian FWHM to sigma
PSF_sig = param / (2*sqrt(2*log(2)));

% Generation of the coordinates maps
[x1,y1] = meshgrid(-(l_psf(1)-1)/2:(l_psf(1)-1)/2,-(l_psf(2)-1)/2:(l_psf(2)-1)/2);

% Generation of the Gaussian function
Srce_PSF = exp(-((x1-barxy(2)).^2 + (y1-barxy(1)).^2)/(2*(PSF_sig^2)));

% Computation of the criterion
corr_coeff = -sum(sum(Srce_PSF.*CCDin));

% end of function