function IMout = ImSensor_PSIM(IMin, QE, PSF, Dark, Bias, CF, IMsat, XS, NL, ...
    RON, Exptime, Digit, Sensor_maps, binfact, IMdisp, IMout_type, noise_quick)

% IMout = ImSensor_PSIM(IMin, QE, PSF, Dark, Bias, CF, IMsat, XS, NL, ...
%     RON, Exptime, Digit, Sensor_maps, binfact, IMdisp, IMout_type, ...
%     noise_quick);
%
% This function simulates an image sensor, i.e. converts the flux from an
% input image ('IMin', array of number of photons > 0) into the signal
% coming out of the image sensor ('IMout'), that can either be Voltages or
% Digital Numbers after digitization (depending if the tag 'IMout_type' is
% set to the character 'V' or 'ADU' respectively).
%
% It takes into account the following properties, defects and noise sources
% of the sensor:
% - QE:         Quantum Efficiency from 0 to 1 [no unit]
% - PSF:        Point Spread Function of the sensor (=0 if all charges
%               collected in one pixel) [pixels] or image of the PSF
% - Dark:       Dark current signal [electrons per pixel per second]
% - Bias:       The overall Bias Voltage, converted into [electrons]
%                   Bias [e] = 10^6 * Bias[V] / CF [microV/e]
% - CF:         Conversion Factor [microV/e]
% - IMsat:      Saturation level [electrons]
% - XS:         Excess Noise [no unit] (1.4 for EMCCD)
% - NL:         Non-Linearity polynomial equation coefficients, starting
%               from the highest power and ending with the offset [no unit]
%               (not used if a single number)
% - RON:        Read-Out Noise [electrons rms per pixel and per frame]
% - Exptime:    Exposure time (for computation of the Dark signal) [s]
% - Digit:      Gain of the ADC [electron/ADU]
%
% 'Sensor_maps' should be the name (character format) of a .mat file
% containing maps (same size as the input image) of defaults or
% characteristics of the sensor, pixel by pixel (not taken into account if
% input to 0):
% - PRNUmap:    Map of Pixel Response Non Uniformity [no unit,
%               multiplicative, around 1]
% - ColBiasMap: Map of Column Bias to be added to the Bias [electrons,
%               additive, around 0]
% - DeadpxMap:  Map of Dead pixels [no unit, multiplicative, map of ones
%               with zeros at the location of dead pixels]
% A help for generation of those maps is provided at the end of the
% function.
% 
% 'binfact' is the factor with which the detector will sample the input
% image.
% 'IMdisp' should be set to 1 for a display of the output signal 'IMout'.
% If any value is inputted in 'noise_quick', the excess noise will be
% computed at the same time as the photon noise (less rigourous, but
% faster). This trick cannot be used if 'binfact' is different from 1.
%
% A typical entry could be:
% IMout = ImSensor_PSIM(WFSimage, 0.85, 0.8, 0.5*700, 500, 120, 10000, 0, [1e-5 0.97 0], 3, 1/700, 0.5, 'Detector1', 4, 1, 'ADU');
%
% Custom sub-routines required:
% 1) First level
% - crop_PSIM.m
% - bin_PSIM.m
% - imnoise.m (Image Processing Toolbox)
% - poissrnd.m (Statistics Toolbox)
% 2) Other levels
% - centroid_PSIM.m
%
% v1.0 J.Kolb 16/12/10
% v2.0 J.Kolb 24/04/12
% - included Non-Linearity
% - improved comments

% Load sensor characteristics
if Sensor_maps ~= 0
    load(Sensor_maps)
end

% Get the size of the input image;
IMxy = size(IMin); % [pixels]

% Add QE
IMout = QE * IMin; % [photons that will generate photo-electrons in the image plane]

% Add Poisson noise (and XS noise if 'quick_noise' variable inputted and 'binfact' is 1)
if (nargin > 16) && (binfact == 1)
    IMout = (XS^2)*(10^12)*imnoise(IMout/(XS^2)*10^-12,'poisson');
    %       [photo-electrons generated in the image plane including photon
    %       and excess noises]
else
    IMout = (10^12)*imnoise(IMout*10^-12,'poisson');
    %       [photo-electrons generated in the image plane including photon
    %       noise]
end

% Add PSF
% First create the PSF image if is not inputted
if (length(PSF) == 1) && (PSF ~= 0)
    PSF_sig = PSF / (2*sqrt(2*log(2))) * binfact;
    PSF_size = round(10*PSF_sig);
    if uint16(PSF_size/2) ~= PSF_size/2
        PSF_size = PSF_size + 1;
    end
    [x,y] = meshgrid(-(PSF_size-1)/2:(PSF_size+1)/2,-(PSF_size-1)/2:(PSF_size+1)/2);
    Srce_PSF = exp(-((x-0.5).^2 + (y-0.5).^2)/(2*(PSF_sig^2)));
    Srce_PSF = Srce_PSF ./ sum(Srce_PSF(:));
elseif (length(PSF) ~= 1)
    % if PSF image is inputted
    Srce_PSF = PSF ./ sum(PSF(:));
else
    Srce_PSF = PSF;
end
% Convolution by the PSF image (if PSF not zero)
if (length(Srce_PSF) ~= 1)
    IMout = conv2(IMout,Srce_PSF);
    IMout = crop_PSIM(IMout, IMxy);
    %                           [Photoelectrons generated and collected in the pixels]
end

% Sampling by the pixels
IMout = bin_PSIM(IMout, binfact) * (binfact^2);

% Get the size of the output image;
IMxy = size(IMout); % [pixels]

% Add Dark current
if Dark ~= 0
    IMout = IMout + poissrnd(Dark * Exptime,IMxy(1),IMxy(2)); % [Electrons generated in the pixel]
end

% Add Saturation
if IMsat ~= 0
    IMout(IMout > IMsat) = IMsat; % [Electrons actually detected]
end

% Add Excess Noise (already done if 'quick_noise' variable inputted)
if (nargin < 17) && (XS > 1)
    IMout = ((XS^2)-1)*(10^12)*imnoise(IMout/((XS^2)-1)*10^-12,'poisson');
    %                   [Electrons detected after avalanche amplification]
end

% Convert to Voltage
IMout = IMout * CF; % [microVolts converted from electrons]

% Add PRNU
if (Sensor_maps ~= 0)
    if exist('PRNUmap', 'var')
        IMout = IMout .* PRNUmap; %[microVolts including PRNU]
    end
end

% Add Linearity
if length(NL) > 1
    IMout_tmp = IMout; IMout = 0*IMout;
    for cpt = 1:length(NL)
        IMout = IMout + NL(cpt)*(IMout_tmp.^(length(NL)-cpt)); % [microVolts tramsmitted]
    end
end

% Add Bias
IMout = IMout + Bias * CF; % [microVolts tramsmitted including Bias]

% Add Column Bias
if (Sensor_maps ~= 0)
    if exist('ColBiasMap', 'var')
        IMout = IMout + Bias * ColBiasMap * CF; % [microVolts tramsmitted including column Bias]
    end
end

% Add RON
IMout = IMout + RON * CF * randn(IMxy); % [microVolts measured]

% Add defect pixels
if (Sensor_maps ~= 0)
    if exist('DeadpxMap', 'var')
        IMout = IMout .* DeadpxMap; % [microVolts measured in non-dead pixels]
    end
end

% Add digitization
if strcmp(IMout_type, 'ADU')
    IMout = floor(IMout / (CF * Digit)); % [ADUs]
end

% Display
if IMdisp == 1
    figure ; imagesc(IMout) ; colorbar ; axis square ; colormap('bone')
end

% end of function

% % To help with the creation of defects maps:
%
% PRNU = 1; %[%]
% PRNUmap = (1 + PRNU / 100 * randn(IMxy));
% ficac(PRNUmap)
%
% NbColBias = 2; %[%]
% ColBias = NbColBias / 100 * randn(1,IMxy(1));
% ColBiasMap = ones(IMxy(2),1) * ColBias;
% ficac(ColBiasMap)
%
% NbDeadpx = 0.2; %[%]
% xDeadpx = ceil(100*rand(1,round(NbDeadpx/100*prod(IMxy))));
% yDeadpx = ceil(100*rand(1,round(NbDeadpx/100*prod(IMxy))));
% DeadpxMap = ones(IMxy);
% for cpt = 1:length(xDeadpx)
%     DeadpxMap(xDeadpx(cpt),yDeadpx(cpt)) = 0;
% end
% ficac(DeadpxMap)
%
% save 'MyDetector_1' PRNUmap ColBiasMap DeadpxMap
%
% % end of help with the creation of defects maps