function [CCDall, xyslopes, wl, C, xyvalid, xyfullill, xyf_of_v, Tag_params, CCDall2] =...
    SHWFS_PSIM(WF, SHWFS_PSIM_input, Coeffs, WFdisp, Tag_params, pup)

% [CCDall, xyslopes, wl, C, xyvalid, xyfullill, xyf_of_v, Tag_params, CCDall2] =...
%     SHWFS_PSIM(WF, SHWFS_PSIM_input, Coeffs, WFdisp, Tag_params, pup);
%
% This function is a simulation of a Shack-Hartmann type Wavefront Sensor,
% i.e. converts an input wavefront into a SHWFS image. The main role of
% this routine is actually parameters computation and saving, and it then
% calls to 'SHWFS_PSIM_Mainloop.m', the core of the WF Sensing (loop on the
% sub-apertures).
% The inputs are:
% - the input 'WF' (in microns)
% - the .m file named 'SHWFS_PSIM_input' (including path) contains the
%   SHWFS input physical parameters
% - The alignment errors (mis-registration DM <-> WFS) are inputted in the
%   vector 'Coeffs', containing  at least 5 values. In this function the
%   mis-registration effect is only on the pupil shape.
%   - x shift in % of a sub-aperture (>0 or <0)
%   - y shift in % of a sub-aperture (>0 or <0)
%   - rotation in degrees (>0 or <0)
%   - x stretch in % (>0 or <0, >0 for shrinking of the DM on the WFS)
%   - y stretch in % (>0 or <0, >0 for shrinking of the DM on the WFS)
% - The variable 'WFdisp' triggers the display of the WF, CCD image +
%   sub-apertures, and slopes. 0 for no, 1 for yes.
% - In case many WFS simulations have to be run, the SH parameters should
%   be saved in a .mat file named as the .m input parameters file, and the
%   variable 'Tag_params' should be the same as 'SHWFS_PSIM_input'.
%   In case of doubt, just create a variable 'Tag_params' equal to zero. At
%   the first iteration, all parameters will be computed and the variable
%   properly changed and outputted. At the following iterations the
%   parameters will be simply loaded (faster).
% - if the pupil is different that the one described solely by the SHWFS
%   parameters and mis-registration coefficients (case of pupil within
%   meta-pupil in tomography), it should be inputted in 'pup' as a matrix
%   of the same size as the input 'WF'
% 
% The outputs are:
% - 'CCDall' the SHWFS image
% - the matrices of slopes (in x and y) measured by the SHWFS are stored in
%   the 3D matrix 'xyslopes' (x in the 1st layer, y in second)
% - 'wl' in output is the same as the input wavelength (um).
% - 'C' is the actual sampling of the image by the detector.
% - The x and y coordinates of the valid subapertures are stored in the 2
%   columns of the table 'xyvalid'. The (Matlab) index are stored in the
%   3rd column.
% - The coordinates of the fully illuminated subapertures are stored in the 2
%   columns of the table 'xyfullill'.
% - 'xyf_of_v' contains the index of the fully illuminated sub-apertures
%   among the valid ones (useful for selecting only those SA in the IM &
%   CM).
% - the output variable 'Tag_params' is the same as the input name
%   'SHWFS_PSIM_input', so that if this variable is used as output in the
%   next use of the function (in a loop), the SHWFS parameters don't need
%   to be recomputed
% - 'CCDall2' the effective image used for slopes computation, i.e.
%   'CCDall' after thresholding, windowing...
%
% A typical entry could be:
% [CCDall, xyslopes, wl, C, xyvalid, xyfullill, xyf_of_v, Tag_params, CCDall2] =...
%     SHWFS_PSIM(Phase, 'SHWFS_input_AOFgeom_240px', [0 0 0 0 0], 1, Tag_params);
%
% Custom sub-routines required:
% 1) First level
% - SHWFS_PSIM_Mainloop.m
% - createZ_PSIM.m
% - bin_PSIM.m
% - nanmean.m (Statistics Toolbox)
% - nansum.m (Statistics Toolbox)
% 2) Other levels
% - Slopes_PSIM.m
% - makepsf_PSIM.m
% - FindNM.m
% - ZPolynomeQuick.m
% - ImSensor_PSIM.m
% - centroid_PSIM.m
% - Corr_PSIM.m
% - crop_PSIM.m
% - imnoise.m (Image Processing Toolbox)
% - poissrnd.m (Statistics Toolbox)
%
% v1.0 J.Kolb 16/12/10
% v1.8 J.Kolb 25/04/12
% - added the variable NL (non-linearity) after modification of the routine
%  'ImSensor_PSIM.m'
% - Re-normalization of the reference tilts 'perf_Tip' and 'perf_Tilt'. The
%   normalization maps 'nansum_perf_Tip' and 'nansum_perf_Tilt' and thus
%   renderred useless and removed from the routine 'SHWFS_PSIM_Mainloop' and
%   the present one
% - added and improved pupil scaling for mis-registration
% - Computation of lenslet imaging parameters removed, and rather done via
%   a call to the function 'makepsf_PSIM.m' on a flat wavefront.
% - Computation of extended source image moved to after computation of
%   lenslet imaging parameters, to avoid duplication of some computations
% - Introduced tilt referencing for the 'G' geometric method, in order to
%   simplify computations in the routine 'SHWFS_PSIM_Mainloop.m'.
% - Modification of he call to 'SHWFS_PSIM_Mainloop' to account for all new
%   input parameters. Also now if the input parameters already existed
%   ('Tag_paramas' set to 1), all of 'SHWFS_PSIM_input' has to be loaded
%   and not only a few variables. thsi replaces having to load it at the
%   beginning of 'SHWFS_PSIM_Mainloop'
% v1.9 J.Kolb 16/10/12
% - corrected missing input 'Coeff' in the help and example.
% v2.0 J.Kolb 06/12/12
% - improved HELP

% Loads and computes SHWFS parameters only if tag variable isn't the same
% as the input parameters file name. Otherwise will load existing variables.
if  ~exist('Tag_params','var') || ~strcmp(Tag_params, SHWFS_PSIM_input)
    % Load the input parameters
    run(num2str(SHWFS_PSIM_input))
    if ~exist('Sensor_maps','var')
        Sensor_maps = 0;
    end
    
    % Computation of some other input parameters
    nbpix = SHsize * nbpixCCD;  % total number of pixels
    nbpixWF = length(WF);       % size in pixels of the input WF
    spup = nbpixWF/SHsize;      % size in pixels of one SA in the input WF
    
    % Error checks on the input image size
    if round(nbpixWF/SHsize) ~= nbpixWF/SHsize
        error('WF size must be multiple of the number of sub-apertures')
    end
    if nbpixWF/SHsize < 2
        error('WF must have at least 2 points per sub-aperture')
    end
    
    % Determination of valid and fully illuminated sub-apertures
    [tmp, ~, ~] = createZ_PSIM(SHsize*10, 1, outmin, outmax);
    tmp = bin_PSIM(tmp,10);
    [xyvalid(:,1), xyvalid(:,2)] = find(tmp >= subap_min_illu);
    xyvalid(:,3) = find(tmp >= subap_min_illu);
    [xyfullill(:,1), xyfullill(:,2)] = find(tmp == max(tmp(:)));
    xyf_of_v = [];
    for cpt = 1:size(xyvalid,1)
        if tmp(xyvalid(cpt,1), xyvalid(cpt,2)) == 1;
            xyf_of_v = [xyf_of_v;cpt];
        end
    end
    
    % Creation of the perfect pupil mask
    if nargin < 6
        [pup, in, out] = createZ_PSIM(nbpixWF,1,outmin,outmax);
        pup(out) = NaN;
    else
        [~, in, out] = createZ_PSIM(nbpixWF,1,outmin,outmax);
    end
    
    % Creation of the actual pupil mask (accounting for a finite DM of size
    % equal to the pupil)
    if sum(Coeffs(1:5).^2) ~= 0
        [x,y] = meshgrid(1:nbpixWF,1:nbpixWF);      % perfect coordinates
        
        % Scaling of WF coordinates
        xWF = x - nbpixWF/2-0.5 - Coeffs(1)/100/SHsize*nbpixWF; % x xhift
        yWF = y - nbpixWF/2-0.5 - Coeffs(2)/100/SHsize*nbpixWF; % y shift
        [theta, rho] = cart2pol(xWF,yWF); theta = theta - Coeffs(3)/180*pi; % rotation
        [xWF, yWF] = pol2cart(theta, rho);
        xWF = (1 + Coeffs(4)/100)*xWF;                          % x stretch
        yWF = (1 + Coeffs(5)/100)*yWF;                          % y stretch
        xWF = xWF + nbpixWF/2+0.5;
        yWF = yWF + nbpixWF/2+0.5;
        
        % Scaling of the actual pupil mask (accounting for a finite DM of size
        % equal to the pupil)
        if sum(Coeffs.^2) ~= 0
            pup = interp2(x,y,pup,xWF,yWF,'nearest').*pup;
        end
    end
    
    % Initialisation for spot display in geom model
    [x_spot, y_spot] = meshgrid(-(nbpixCCD-1)/2:(nbpixCCD-1)/2);
    
    % Initialization of source image
    Srce = source_size/pix_scale; wl_eff = wl;
    
    % Referencing of the Tilt
    if SH_model == 2 % Diffractive
        [~, wl_eff, C, ~, zeropad] = makepsf_PSIM(ones(spup,spup), ones(spup,spup), wl, D/SHsize, pix_scale, nbpixCCD, Srce, Cmin);
        
        if source_size ~= 0
            % Computation of extended source image
            N2 = 2*ceil((zeropad+1)/2)-1;
            [x_srce, y_srce] = meshgrid(-(N2-1)/2:(N2-1)/2);
            Srce = exp(-((x_srce).^2 +(y_srce).^2)*4*log(2)/((Srce*C)^2));
        end
        
        %  This adds a tilt offset to the WF so that the PSFs in the
        %  sub-apertures are centered on the pixels' crossing.
        
        % First create a TT.
        TT = createZ_PSIM(nbpixWF,2,outmin,outmax); TT = TT ./ max(TT(:));
        % Computation of the TT angle in microns PTV
        TT_angle = pix_scale / 3600 / 180 * pi / 2 / 2 / C * D * (nbpixWF-1) / nbpixWF * 1e6;
        % Scales the TT WF
        TT = TT * TT_angle;
        %  Adds TT in second axis
        TT = TT - rot90(TT);
        
        % creates empty variables (for consistency with SH_model = 1).
        perf_Tip = 0; perf_Tilt = 0;
        
    elseif SH_model == 1 % Geometric
        if strcmp(Cen_meth,'Z')
            % Create the reference TT for slope computations
            [perf_Tip_0, perf_Tilt_0] = meshgrid(-1:(2/(spup-1)):1);
            perf_Tip_0 = perf_Tip_0 * D / SHsize * tan(pix_scale/3600/180*pi)*(10^6)*(spup-1)/spup/2;
            perf_Tilt_0 = perf_Tilt_0 * D / SHsize * tan(pix_scale/3600/180*pi)*(10^6)*(spup-1)/spup/2;
            for subap = 1:size(xyvalid,1)
                % Slicing of the pupil mask
                xWF = xyvalid(subap,1)-1 ; yWF = xyvalid(subap,2)-1;
                pupmask = pup(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
                % Modification of the reference TT (important for partially illuminated subaps)
                perf_Tip_tmp = perf_Tip_0 .* pupmask;
                perf_Tilt_tmp = perf_Tilt_0 .* pupmask;
                perf_Tip_tmp = perf_Tip_tmp - nanmean(perf_Tip_tmp,2)*ones(1,spup);
                perf_Tilt_tmp = perf_Tilt_tmp - ones(spup,1)*nanmean(perf_Tilt_tmp);
                
                perf_Tip(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1)) =...
                    perf_Tip_tmp./nansum(nansum(perf_Tip_tmp.*perf_Tip_tmp));
                perf_Tilt(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1)) =...
                    perf_Tilt_tmp./nansum(nansum(perf_Tilt_tmp.*perf_Tilt_tmp));
            end
        elseif strcmp(Cen_meth,'G')
            perf_Tip = NaN*pup; perf_Tilt = NaN*pup;
            for subap = 1:size(xyvalid,1)
                % Slicing of the pupil mask
                xWF = xyvalid(subap,1)-1 ; yWF = xyvalid(subap,2)-1;
                pupmask = pup(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
                % Modification of the reference TT (important for partially illuminated subaps)
                perf_Tip_tmp = NaN*pupmask; perf_Tilt_tmp = NaN*pupmask;
                cpt0=0;
                for cptpx = 1:spup
                    if nansum(pupmask(cptpx,:)) ~= 0
                        h1 = find(pupmask(cptpx,:) == 1,1,'first');
                        h2 = find(pupmask(cptpx,:) == 1,1,'last');
                        if h2 ~= h1
                            perf_Tip_tmp(cptpx,h1) = -1/(h2-h1);
                            perf_Tip_tmp(cptpx,h2) = 1/(h2-h1);
                            cpt0 = cpt0+1;
                        end
                    end
                end
                perf_Tip_tmp = perf_Tip_tmp./cpt0;
                cpt0=0;
                for cptpx = 1:spup
                    if nansum(pupmask(:,cptpx)) ~= 0
                        h1 = find(pupmask(:,cptpx) == 1,1,'first');
                        h2 = find(pupmask(:,cptpx) == 1,1,'last');
                        if h2 ~= h1
                            perf_Tilt_tmp(h1,cptpx) = -1/(h2-h1);
                            perf_Tilt_tmp(h2,cptpx) = 1/(h2-h1);
                            cpt0 = cpt0+1;
                        end
                    end
                end
                perf_Tilt_tmp = perf_Tilt_tmp./cpt0;
                perf_Tip(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1)) = perf_Tip_tmp;
                perf_Tilt(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1)) = perf_Tilt_tmp;
            end
        else
            error('Centroid method must be "G" or "Z"')
        end
        perf_Tip(isnan(perf_Tip)) = 0;
        perf_Tilt(isnan(perf_Tilt)) = 0;
        
        % creates empty variables (for consistency with SH_model = 2).
        TT = 0;
        C = 0;
    end
    pup(out) = 0;
    
    % Creation of the detector PSF image (due to charge diffusion)
    if (noise == 1) && (length(PSF) == 1) && (PSF ~= 0)
        PSF_sig = PSF / (2*sqrt(2*log(2))) * C;
        PSF_size = round(10*PSF_sig);
        if uint16(PSF_size/2) ~= PSF_size/2
            PSF_size = PSF_size + 1;
        end
        [x,y] = meshgrid(-(PSF_size-1)/2:(PSF_size+1)/2,-(PSF_size-1)/2:(PSF_size+1)/2);
        Srce_PSF = exp(-((x-0.5).^2 + (y-0.5).^2)/(2*(PSF_sig^2)));
        PSF = Srce_PSF ./ sum(Srce_PSF(:));
    end
    
    % Creation of the tag variable, so that if further WF sensing are
    % simulated, the time to load parameters is saved.
    Tag_params = SHWFS_PSIM_input;
    
    % saving all parameters in a .mat file with the same name as the basic
    % paramater .m file.
    save(num2str(SHWFS_PSIM_input),...                          % output file name (.mat)
        'SH_model', 'param', 'Cen_meth', 'D', 'SHsize',...
        'nbpixCCD', 'pix_scale', 'noise', 'wl', 'wl_eff', 'source_size', 'pixsize',...
        'outmax', 'outmin', 'subap_min_illu', 'Cmin', ...       % primary SH input parameters
        'Nphot', 'Exptime', 'QE', 'PSF', 'Dark', 'IMsat', 'CF',...
        'Bias', 'XS', 'NL', 'RON', 'Digit', 'Sensor_maps',...   % detector parameters
        'nbpix', 'nbpixWF', 'spup', 'xyvalid', 'xyfullill', 'xyf_of_v', 'pup', 'in', 'out', 'C', 'x_spot',...
        'y_spot', 'Srce', 'TT', 'perf_Tip', 'perf_Tilt');       % Secondary inputs (calculated from the primary)
else
    load(num2str(SHWFS_PSIM_input));
end

% Display of input WF
if WFdisp == 1
    figure('Position',[10, 10, 1200, 800]);
    subplot(2,3,1) ; imagesc(WF) ; colormap('bone') ; axis off ; axis square
    title('Input WF (microns)');
end

% Now that input parameters are saved in any case, one little thig to do is
% to add an offset TT so that the spots fall at the crossing of 4 pixels in
% the diffractive model.
WF = WF - TT;

% 2) run the SH WF sensing routine (loop on the sub-apertures).

[CCDall, xyslopes, wl_eff, C, CCDall2] = SHWFS_PSIM_Mainloop(WF, SH_model, param, Cen_meth, D, SHsize,...
    nbpixCCD, pix_scale, noise, wl, wl_eff, source_size, pixsize, outmax, outmin, subap_min_illu, Cmin,...
    Nphot, Exptime, QE, PSF, Dark, IMsat, CF, Bias, XS, NL, RON, Digit, Sensor_maps, nbpix, nbpixWF, spup,...
    xyvalid, xyfullill, xyf_of_v, pup, in, out, C, x_spot, y_spot, Srce, TT, perf_Tip, perf_Tilt, WFdisp);

% end of function