function [CCDall, xyslopes, wl, C, CCDall2] = SHWFS_PSIM_Mainloop(WF, SH_model, param, Cen_meth, D, SHsize,...
    nbpixCCD, pix_scale, noise, wl, wl_eff, source_size, pixsize, outmax, outmin, subap_min_illu, Cmin,...
    Nphot, Exptime, QE, PSF, Dark, IMsat, CF, Bias, XS, NL, RON, Digit, Sensor_maps, nbpix, nbpixWF, spup,...
    xyvalid, xyfullill, xyf_of_v, pup, in, out, C, x_spot, y_spot, Srce, TT, perf_Tip, perf_Tilt, WFdisp)

% [CCDall, xyslopes, wl, C, CCDall2] = SHWFS_PSIM_Mainloop(WF, SH_model, param, Cen_meth, D, SHsize,...
%    nbpixCCD, pix_scale, noise, wl, wl_eff, source_size, pixsize, outmax, outmin, subap_min_illu, Cmin,...
%    Nphot, Exptime, QE, PSF, Dark, IMsat, CF, Bias, XS, NL, RON, Digit, Sensor_maps, nbpix, nbpixWF, spup,...
%    xyvalid, xyfullill, xyf_of_v, pup, in, out, C, x_spot, y_spot, Srce, TT, perf_Tip, perf_Tilt, WFdisp);
%
% This function is a simulation of a Shack-Hartmann type Wavefront Sensor,
% i.e. converts an input wavefront 'WF' in microns into a SHWFS image. The
% role of this routine is the main loop on the sub-apertures, but it should
% generally be run from the general SHWFS routine 'SHWFS_PSIM.m' which does
% parameters computation and saving.
% The outputs are:
% - 'CCDall' the SHWFS image
% - the matrices of slopes (in x and y) measured by the SHWFS are stored in
%   the 3D matrix 'xyslopes' (x in the 1st layer, y in second)
% - 'wl' in output is the same as the input wavelength.
% - 'C' is the actual sampling of the image by the detector.
% - 'CCDall2' the effective image used for slopes computation, i.e.
%   'CCDall' after thresholding, windowing...
% 
% Given the large number of inputs, a typical entry is not meaningful.
% 
% A detailed description of all input parameters is as follows:
% - WF:                 input WF in microns
% - SH_model:           SH model used: 1=geometric, 2=diffractive
% - Cen_meth:           Centroiding method, if diffractive SH model:
%                            'Bar' = barycenter (in pixels from the corner)
%                            'CoG' = Center of Gravity (in pixels from the center)
%                            'wCoG' = Weighted Center of Gravity (requires weighting map)
%                            'Corr' = Correlation (requires correlation map)
%                            'MF' = Matched filter (requires filter map)
%                       and if geometric SH model:
%                            'Z' = Z-tilt method (projection on zernike Tilt)
%                            'G' = G-tilt methog (angle of arrival)
% - D:                  Telescope diameter in meters
% - SHsize:             Number of sub-apertures across the diameter
% - nbpixCCD:           Number of pixels per sub-aperture
% - pix_scale:          Detector pixel scale in arcsec/pixel
% - noise:              0 if perfect detector, 1 if noise added
% - wl:                 Wavelength (um)
% - source_size:        Source size (FWHM) in " ( = 0 if point source)
% - pixsize:            Detector pixel size in microns
% - outmax:             External diameter of the pupil mask (>0, between outmin and 1)
% - outmin:             Internal diameter of the pupil mask (between 0 and outmin, <1)
% - subap_min_illu:     Min illumination required for validity of the sub-apertures (between 0 and 1)
% - Cmin                Min sampling of the image by the detector
% - Nphot:              Number of incident photons per sub-aperture and per frame
% - Exptime:            Exposure time (in s)
% - QE:                 Quantum efficiency from 0 to 1 [no unit]
% - PSF:                Point Spread Function of the sensor (=0 if all
%                       charges collected in one pixel) [pixels] or image
%                       of the PSF
% - Dark:               Dark current signal [electrons per pixel per second]
% - IMsat:              Saturation level [electrons]
% - CF:                 Conversion Factor [microV/electron]
% - Bias:               The overall Bias Voltage, converted into [electrons]
%                           Bias [e] = 10^6 * Bias[V] / CF [microV/e]
% - XS:                 Excess Noise [no unit] (1.4 for EMCCD)
% - RON:                Read-Out Noise [electrons rms per pixel and per frame]
% - Digit:              Gain of the ADC [electron/ADU]
% - nbpix:              total number of pixels
% - nbpixWF:            size in pixels of the input WF
% - spup:               size in pixels of one SA in the input WF
% - xyvalid:            coordinates of the valid subapertures
% - xyfullill:          coordinates of the fully illuminated subapertures
% - xyf_of_v:           index of the fully illuminated sub-apertures among the valid ones
% - pup:                pupil mask
% - in:                 index of sub-apertures inside the pupil
% - out:                index of sub-apertures outside the pupil
% - x_spot:             x grid for spot display in geom model
% - y_spot:             y grid for spot display in geom model
% - Srce:               Source image or size in arcsec
% - TT:                 Tilt offset to the WF used so that that the PSFs in the
%                       sub-apertures are centered on the pixels' crossing
%                       (diffractive model)
% - perf_Tip:           Reference Tip for geometric slope computation
% - perf_Tilt:          Reference Tilt for geometric slope computation
% - nansum_perf_Tip:    Reference Tip (sum) --> Obsolete
% - nansum_perf_Tilt:   Reference Tilt (sum) --> Obsolete
% - The variable 'WFdisp' triggers the display of the WF, CCD image +
%   sub-apertures, and slopes. 0 for no, 1 for yes.
%
% Custom sub-routines required:
% 1) First level
% - bin_PSIM.m
% - makepsf_PSIM.m
% - ImSensor_PSIM.m
% - Slopes_PSIM.m
% 2) Other levels
% - centroid_PSIM.m
% - Corr_PSIM.m
% - crop_PSIM.m
% - imnoise.m (Image Processing Toolbox)
% - poissrnd.m (Statistics Toolbox)
%
% v1.0 J.Kolb 16/12/10
% v1.1 J.Kolb 21/02/11
%  - added error detection on centroid method for geometric model
% v2.0 J.Kolb 25/04/12
% - added the variable NL (non-linearity) after modification of the routine
% 'ImSensor_PSIM.m'
% - for speed reasons, the input has been modified. It used to be the
%  filename 'SHWFS_PSIM_input' of the WFS data to load each time the routine
%  is used. This was simplifying the command line (only 3 inputs). It has
%  been changed to directly input all the values that are stored in
%  'SHWFS_PSIM_input', that is now 47 (!) inputs. This speeds up the
%  computations, and the large number of inputs is not an issue, as actually
%  this routine is never used in stand-alone, but called by 'SHWFS_PSIM'.
%  This is the high level function that should be called by the user when he
%  wants to simulate a SHWFS .
% - Modification of the Z-tilt computation after re-normalization of the
%  reference tilts 'perf_Tip' and 'perf_Tilt'. The normalization maps
%  'nansum_perf_Tip' and 'nansum_perf_Tilt' and thus renderred useless and
%  removed from the routine 'SHWFS_PSIM' and the present one
% - Great simplification of the G-tilt computation
% - Added an error message if input is neither 'G' nor 'Z'
% - Added clipping of the slopes measurement (Geometric model) if values
%  larger than sub-aperture FoV. More accurately: slopes are set to zero in
%  that case.
% - Improved comments
% - removed useless slicing of pupil mask in geometric model

% Initializations
psf_all = zeros(SHsize*nbpixCCD*C,SHsize*nbpixCCD*C);
CCDall = zeros(nbpix,nbpix);
xyslopes = zeros(SHsize,SHsize,2);
Avg_flux = 0; cpt_flux = 0;
WF(isnan(WF)) = 0;
% Quadratic sum of the source size with the diffraction size of the
% sub-aperture, only for noise computation in geometric case
source_size = sqrt(source_size.^2 + (wl/D*SHsize * 1e-6 * 3600 * 180 / pi)^2);

% Loop on the sub-apertures
for subap = 1:size(xyvalid,1)
    % coordinates of the SA for slicing of the input WF
    xWF = xyvalid(subap,1)-1 ; yWF = xyvalid(subap,2)-1;
    
    % Selection of the SH model: Geometric or Diffractive
    switch SH_model
        case 1  % Geometric
            % Slicing of the input WF
            perf_Tip_msk = perf_Tip(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
            perf_Tilt_msk = perf_Tilt(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
            WFmask = WF(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
            
            % Selection of the centroiding method in the Geometric model
            if strcmp(Cen_meth,'Z')
                % Slopes computation by projection on the ref TT (Z-tilt method)
                xyslopes(xWF+1,yWF+1,1) = sum(perf_Tip_msk(:).*WFmask(:));
                xyslopes(xWF+1,yWF+1,2) = sum(perf_Tilt_msk(:).*WFmask(:));
            elseif strcmp(Cen_meth,'G')
                % Slopes computation by difference of WF at the edges (G-tilt method)
                opd = sum(sum(WFmask.*perf_Tip_msk,2)) * (spup-1);
                xyslopes(xWF+1,yWF+1,1) = opd/(D/SHsize) * spup/(spup-1) * 1e-6 * 3600 * 180 / pi/pix_scale;
                opd = sum(sum(WFmask.*perf_Tilt_msk,1)) * (spup-1);
                xyslopes(xWF+1,yWF+1,2) = opd/(D/SHsize) * spup/(spup-1) * 1e-6 * 3600 * 180 / pi/pix_scale;
            else
                error('Centroid method must be "G" or "Z"')
            end
            
            if noise == 1
                % Photon noise addition to the slopes, in meters of difference of step
                sig_noise = sqrt((source_size/3600/180*pi)^2 * (D/SHsize)^2 / 8 / log(2) / Nphot);
                % converted into rad, then arcsec, then pixels
                sig_noise = sig_noise / (D/SHsize) *3600*180/pi / pix_scale;
                % noise addition
                xyslopes(xWF+1,yWF+1,1) = xyslopes(xWF+1,yWF+1,1) + sig_noise*randn;
                xyslopes(xWF+1,yWF+1,2) = xyslopes(xWF+1,yWF+1,2) + sig_noise*randn;
            end
            
            if WFdisp == 1
                % Artificial display of spots in the geometric case
                CCDall(xWF*nbpixCCD+1:xWF*nbpixCCD+nbpixCCD,yWF*nbpixCCD+1:yWF*nbpixCCD+nbpixCCD) =...
                    exp(-((x_spot-xyslopes(xWF+1,yWF+1,1)).^2 + (y_spot-xyslopes(xWF+1,yWF+1,2)).^2)*log(2));
            end
            CCDall2 = CCDall;
            
        case 2  % Diffractive
            % Slicing of the input pupil
            pupmask = pup(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1));
            
            % PSF computation
            [psf, wl, C] = makepsf_PSIM(WF(spup*xWF+1:spup*(xWF+1),spup*yWF+1:spup*(yWF+1)),...
                pupmask, wl, D/SHsize, pix_scale, nbpixCCD, Srce, Cmin, 1);
            
            % for computation of avg flux (only fullill SA)
            if ~isnan(sum(pupmask(:)))
                Avg_flux = Avg_flux + sum(psf(:)); cpt_flux = cpt_flux + 1;
            end
            
            % Mosaicing of the full detector image
            psf_all((xWF*nbpixCCD*C+1):(xWF+1)*(nbpixCCD*C),(yWF*nbpixCCD*C+1):(yWF+1)*(nbpixCCD*C)) = psf;
    end
end

% Addition of noise and slopes computation in the diffractive case. This
% part is outside of the previous loop because it requires the full
% detector image.
if SH_model == 2 % Diffractive
    
    % Detector noise addition
    if noise == 1
        Avg_flux = Avg_flux / cpt_flux;
        CCDall = ImSensor_PSIM(psf_all/Avg_flux*Nphot, QE, PSF, Dark, Bias, CF,...
            IMsat, XS, NL, RON, Exptime, Digit, Sensor_maps, C, 0, 'ADU', 1);
    else
        % Trick to reduce computing time: WFS detector images are computed with the
        % same number of pixels as the input WF, and are binned only once here,
        % instead of binning at each SA computation ('Tag_bin' parameter set to one
        % as input in the function makepsf_PSIM).
        psf_all = bin_PSIM(psf_all, C);
        CCDall = psf_all;
    end
    
    % Slopes computation in diffractive case
    [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCDall, xyvalid, SHsize, nbpixCCD, Cen_meth, param, 0);
    
elseif SH_model == 1 % Geometric
    % Setting of the slopes measurement to zero if larger than the
    % sub-aperture FoV
    xyslopes(abs(xyslopes) > nbpixCCD/2) = 0;
end

if WFdisp == 1
    % Display of SH slopes
    subplot(2,3,2);
    quiver(0.5:1:(SHsize-0.5),0.5:1:(SHsize-0.5),flipud(xyslopes(:,:,1)),-flipud(xyslopes(:,:,2)),0.5);
    axis square ; axis([-1 SHsize+1 -1 SHsize+1])
    title('Slopes (pixels)');
    xlabel('-->  increasing X slope -->') ; ylabel('<--  increasing Y slope <--')
    
    % Display of Detector
    subplot(2,3,3) ; imagesc(CCDall) ; axis off ; colormap('bone') ; axis square
    title('WFS Detector image (ADUs)');
    % Display of the SH grid and subaps centroids
    for subap = 1:size(xyvalid,1)
        xWF = xyvalid(subap,1)-0.5 ; yWF = xyvalid(subap,2)-0.5;
        line([nbpixCCD*xWF, nbpixCCD*xWF+1], [nbpixCCD*yWF, nbpixCCD*yWF+1],'color','red')
        line([nbpixCCD*xWF+1, nbpixCCD*xWF], [nbpixCCD*yWF, nbpixCCD*yWF+1],'color','red')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF-0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF+0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF-0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF+0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
    end
    
    % Display of the detector after thresholding/windowing/weighting/...
    % thresholding (if any)
    subplot(2,3,6) ; imagesc(CCDall2) ; axis off ; colormap('bone') ; axis square
    title('WFS Detector image (ADUs) after thresholding/windowing/weighting/...');
    
    % Display of the maps of x and y slopes
    subplot(2,3,4) ; imagesc(xyslopes(:,:,1)) ; colorbar ; axis square ; colormap('bone')
    title('X slopes (pixels)');
    subplot(2,3,5) ; imagesc(xyslopes(:,:,2)) ; colorbar ; axis square ; colormap('bone')
    title('Y slopes (pixels)');
    
end

% end of function