function [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCDall, xyvalid, SHsize, nbpixCCD, Cen_meth, param, WFdisp)

% [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCDall, xyvalid, SHsize, nbpixCCD, Cen_meth, param, WFdisp)
% 
% This function computes slopes from a Shack-Hartmann WFS image 'CCDall'.
% It can be used in a full SHWFS simulation, or as stand-alone to analyse
% WFS images.
% The inputs are:
% - 'CCDall' the input image
% - 'xyvalid' contains the list of valid sub-apertures in a matrix of size
%   [number of valid sub-apertures,>2]. The first 2 columns are the x and
%   y coordinates of those valid sub-apertures in the [SHsize x SHsize]
%   matrix of sub-apertures. 'xyvalid' is automatically generated
%   when using this routine as part of 'SHWFS_PSIM.m'
% - 'SHsize' is the number of sub-apertures across the pupil diameter
% - 'nbpixCCD' is the number of pixels per sub-aperture
% - 'Cen_meth' is the centroid method to be used
% - 'param' is a thresholding or weighting parameter (see 'centroid_PSIM.m'
%   about those last two). 
% - 'WFdisp' should be set to 1 for displaying various images (slopes,
%   detector image...).
% Alternatively, the SH parameters (xyvalid, SHsize, nbpixCCD, Cen_meth,
% param) can be recomputed from the SH input physical parameters. The
% function becomes then:
% [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCDall, SHWFS_PSIM_input, WFdisp)
% with 'SHWFS_PSIM_input' (including path) the .m file containing the SHWFS
% input physical parameters.
%
% The outputs are:
% - 'xyslopes' a 3D matrix of size [SHsize,SHsize,2] that contains the x
%   and y slopes in the valid sub-apertures, and NaNs in the others.
% - 'CCDall2' the effective image used for slopes computation, i.e.
%   'CCDall' after thresholding, windowing... 
% - 'CCDint' contains the intensities in the sub-apertures of the 'CCDall2'
%   image.
% 
% A typical entry could be:
% [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCD_image, xyvalid, 24, 22, 'wCoG', weighting_map, 1);
% Alternatively:
% [xyslopes, CCDall2, CCDint] = Slopes_PSIM(CCD_image, 'SHWFS_input_NAOS' , 1);
% 
% Custom sub-routines required:
% 1) First level
% - centroid_PSIM.m
% - createZ_PSIM.m
% - bin_PSIM.m
% 2) Other levels
% - Corr_PSIM.m
% - FindNM.m
% - ZPolynomeQuick.m
% 
% v1.0 J.Kolb 16/12/10
% v1.1 J.Kolb 21/02/11
%  - added computation of 'xyvalid' and loading of other parameters if not inputted
% v1.9 J.Kolb 24/04/12
% - improved comments on the last modification
% v2.0 J.Kolb 05/12/12
% - Improved HELP

% Computation of 'xyvalid' and other parameters if not inputted
if nargin < 4
    WFdisp = SHsize;
    SHWFS_PSIM_input = xyvalid; clear xyvalid;
    run(num2str(SHWFS_PSIM_input));
    
    % Computation of some other input parameters
    nbpix = SHsize * nbpixCCD;  % total number of pixels
    nbpixWF = length(CCDall);   % size in pixels of the input WF
    spup = nbpixWF/SHsize;      % size in pixels of one SA in the input WF

    % Determination of valid and fully illuminated sub-apertures
    [tmp, in, out] = createZ_PSIM(SHsize*10, 1, outmin, outmax);
    tmp = bin_PSIM(tmp,10);
    [xyvalid(:,1), xyvalid(:,2)] = find(tmp >= subap_min_illu);
    xyvalid(:,3) = find(tmp >= subap_min_illu);
end

% Initializations
xyslopes = NaN.*zeros(SHsize,SHsize,2);
CCDall2 = 0.*CCDall;
CCDint = zeros(SHsize,SHsize);
% If the parameter map is identical for all sub-apertures, 'param' has the
% size of a sub-aperture. Otherwise it should have the size of the detector
% image 'CCDall' and will be sliced later on.
if size(param) ~= size(CCDall)
    param_tmp = param;
end

% Main loop
for subap = 1:size(xyvalid,1)
    % slicing of the CCD image
    xWF = xyvalid(subap,1)-1 ; yWF = xyvalid(subap,2)-1;
    CCD = CCDall(xWF*nbpixCCD+1:xWF*nbpixCCD+nbpixCCD,yWF*nbpixCCD+1:yWF*nbpixCCD+nbpixCCD);
    % if required, slicing of the parameter map
    if size(param) == size(CCDall)
        param_tmp = param(xWF*nbpixCCD+1:xWF*nbpixCCD+nbpixCCD,yWF*nbpixCCD+1:yWF*nbpixCCD+nbpixCCD);
    end
    % Slopes computation by centroiding (various methods possible)
    [tmp, CCD2] = centroid_PSIM(CCD, Cen_meth, param_tmp);
    % Computation of output image and intensities
    CCDall2(xWF*nbpixCCD+1:xWF*nbpixCCD+nbpixCCD,yWF*nbpixCCD+1:yWF*nbpixCCD+nbpixCCD) = CCD2;
    CCDint(xWF+1, yWF+1) = sum(CCD2(:));
    % re-ordering of the slopes into the output matrix
    xyslopes(xWF+1,yWF+1,2) = tmp(1); xyslopes(xWF+1,yWF+1,1) = tmp(2);
end

if WFdisp == 1
    figure('Position',[10, 10, 1200, 800]);
    
    % Display of SH slopes
    subplot(2,3,2);
    quiver(0.5:1:(SHsize-0.5),0.5:1:(SHsize-0.5),flipud(xyslopes(:,:,1)),-flipud(xyslopes(:,:,2)),0.5);
    axis square ; axis([-1 SHsize+1 -1 SHsize+1])
    title('Slopes (pixels)');
    xlabel('-->  increasing X slope -->') ; ylabel('<--  increasing Y slope <--')
    
    % Display of Detector
    subplot(2,3,3) ; imagesc(CCDall) ; axis off ; colormap('bone') ; axis square
    title('WFS Detector image (ADUs)');
    % Display of the SH grid and subaps centroids
    for subap = 1:size(xyvalid,1)
        xWF = xyvalid(subap,1)-0.5 ; yWF = xyvalid(subap,2)-0.5;
        line([nbpixCCD*xWF, nbpixCCD*xWF+1], [nbpixCCD*yWF, nbpixCCD*yWF+1],'color','red')
        line([nbpixCCD*xWF+1, nbpixCCD*xWF], [nbpixCCD*yWF, nbpixCCD*yWF+1],'color','red')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF-0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF+0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF-0.5)+0.5, nbpixCCD*(yWF-0.5)+0.5],'color','green')
        line([nbpixCCD*(xWF-0.5)+0.5, nbpixCCD*(xWF+0.5)+0.5],...
            [nbpixCCD*(yWF+0.5)+0.5, nbpixCCD*(yWF+0.5)+0.5],'color','green')
    end
    
    % Display of the detector after thresholding/windowing/weighting/...
    % (if any)
    subplot(2,3,6) ; imagesc(CCDall2) ; axis off ; colormap('bone') ; axis square
    title('WFS Detector image (ADUs) after thresholding/windowing/weighting/...');
    
    % Display of the maps of x and y slopes
    subplot(2,3,4) ; imagesc(xyslopes(:,:,1)) ; colorbar ; axis square ; colormap('bone')
    title('X slopes (pixels)');
    subplot(2,3,5) ; imagesc(xyslopes(:,:,2)) ; colorbar ; axis square ; colormap('bone')
    title('Y slopes (pixels)');
end

% end of function