function binned = bin_PSIM(pict, binfact)

% binned = bin_PSIM(pict, binfact);
%
% This function bins the input image 'pict' by the factor 'binfact'.
% 'binfact' is a [2x1] variable containing the binning factors in the x and y
% dimensions. If it is a single value then the binning is the same in both
% dimensions.
% The output 'binned' is the binned image.
% Nan values in the input image 'pict' are excluded from the binning, i.e.
% the binned values are computed only for the non-NaN input values.
%
% A typical entry could be
% WFbin = bin_PSIM(WF, [2 3]);
%
% Custom sub-routines required:
% - none
%
% v1.0 J.Kolb 16/12/10

% Initialization
[a,b] =  size(pict);

% Error checking
if length(binfact) == 1
    binfact(2) = binfact;
elseif length(binfact) > 2
    error('the input binfactor should contain 1 or 2 integer values')
end

% Initialization
a2 = a/binfact(1) ; b2 = b/binfact(2);
% Error checking, main loop starts only if bin factors valid
if (a2 ~= int16(a2)) || (b2 ~= int16(b2))
    error('impossible to bin with this factor')
else
    % Further initializations. 'binfact_eff' and 'pict2' are used to
    % exclude the NaNs
    binned = zeros(a2,b2); binfact_eff = binned;
    pict2 = ~isnan(pict); pict(~pict2) = 0;
    % Main loop on both dimensions
    for cptx = 1:binfact(1)
        for cpty = 1:binfact(2)
            binned = binned +...
                pict(cptx:binfact(1):a-binfact(1)+cptx,cpty:binfact(2):b-binfact(2)+cpty);
            binfact_eff = binfact_eff +...
                pict2(cptx:binfact(1):a-binfact(1)+cptx,cpty:binfact(2):b-binfact(2)+cpty);
        end
    end
    % re-normalization of the binned image to exclude the NaNs
    binned = binned ./binfact_eff;
end

% end of function