function [barxy, psf2] = centroid_PSIM(psf, Cen_meth, param)

% [barxy, psf2] = centroid_PSIM(psf, Cen_meth, param);
% 
% This function computes the centroid of an input image 'psf' using the
% method 'Cen_meth' (and the parameter 'param') that can either be:
% - 'Bar':  simple Barycenter, in pixels, from the *corner* of the image.
%           In that case 'param' is the threshold value to be applied to
%           the image before computing the CoG (without pedestal, i.e. the
%           threshold value is removed from the image and then all negative
%           values are set to zero). No threshold is applied if 'param' is
%           not inputted.
% - 'CoG':  Center of Gravity, in pixels from the *center* of the image.
%           Same threshold possibility as for 'Bar'
% - 'wCoG': weighted Center of Gravity. The input image is weighted before
%           computing the CoG. 'param' can then be a) the weighting map and
%           should be the same size as the input image, or b) the FWHM (in
%           pixels) of the Gaussian spot to use as weighting function (i.e.
%           a positive number), or c) the opposite of the power to which to
%           take the input image (i.e. a negative number),
%           If 'param' is not inputted, then the weighting map is the image
%           itself, which is equivalent to squaring the image before
%           computing the simple CoG.
% - 'Corr': Correlation. 'param' is then the FWHM (in pixels) of the
%           Gaussian function to be used as correlation map.
% - 'MF':   Matched filter. 'param' is then the filter map and should
%           be the same size as the input image. NOT IMPLEMENTED YET
% The 1x2 output vector 'barxy' contains the x and y centroids (in that order).
% The output 'psf2' is the effective image used for slopes computation,
% i.e. 'psf' after thresholding or weighting.
% 
% A typical entry could be:
% [barxy, psf2] = centroid_PSIM(psf_image, 'CoG', 0);
%
% Custom sub-routines required:
% 1) First level
% - Corr_PSIM.m (only for Correlation)
%
% v1.0 J.Kolb 16/12/10
% v2.0 J.Kolb 13/04/12
% - clarified help on 'wCoG' and on 'psf2'
% - corrected 'psf2' output in 'wCoG' case
% - removed commented line about thresholding with pedestal
% - added Error detecton and thresholding possibility in 'Bar' case

% If no centroid method, then barycenter by default
if nargin == 1
    Cen_meth = 'Bar';
end

% Initializations
l_psf = size(psf);
psf2 = psf;

switch Cen_meth
    case 'Bar'
        % Error detecton and thresholding (if 'param' inputted)
        if (nargin > 2) && (length(param) == 1)
            % thresholding without pedestal
            psf2 = psf - param; psf2(psf2 <= 0) = 0;
            if param < 0;
                psf2 = psf2 + param;
            end
        elseif (nargin > 2) && (length(param) ~= 1)
            error('The size of the input "param" should be [1x1] to be used as thresholding value in the "Bar" computation');
        end
        % simple barycenter computation
        s_psf = sum(psf(:));
        barxy(1) = ((1:l_psf(1))*sum(psf,2))/s_psf;
        barxy(2) = (sum(psf,1)*((1:l_psf(2))'))/s_psf;
    case 'CoG'
        % Error detecton and thresholding (if 'param' inputted)
        if (nargin > 2) && (length(param) == 1)
            % thresholding without pedestal
            psf2 = psf - param; psf2(psf2 <= 0) = 0;
            if param < 0;
                psf2 = psf2 + param;
            end
        elseif (nargin > 2) && (length(param) ~= 1)
            error('The size of the input "param" should be [1x1] to be used as thresholding value in the "CoG" computation');
        end
        % CoG computation
        s_psf = sum(psf2(:));
        barxy(1) = ((0.5:l_psf(1))*sum(psf2,2))/s_psf - l_psf(1)/2;
        barxy(2) = (sum(psf2,1)*((0.5:l_psf(2))'))/s_psf - l_psf(2)/2;
    case 'wCoG'
        % Error detecton and weighting (if 'param' not inputted)
        if nargin < 3
            % simple power 2
            psf = psf.^2;
            % otherwise multiplication by the weighting map
        elseif size(param) == l_psf
            psf = psf.*param;
        elseif (length(param) == 1) && (param < 0)
            % simple power
            psf = psf.^(-param);
        elseif (length(param) == 1) && (param > 0)
            % creation of a Gaussian with FWHM = param
            param = param / (2*sqrt(2*log(2)));
            [x1,y1] = meshgrid(-(l_psf(1)-1)/2:(l_psf(1)-1)/2,-(l_psf(2)-1)/2:(l_psf(2)-1)/2);
            % Multiplication by the Gaussian
            psf = psf .* exp(-(x1.^2 + y1.^2)/(2*(param^2)));
        else
            error('The size of the input "param" should be as the image size, or 1');
        end
        psf2 = psf;
        % CoG computation of the weighted image
        s_psf = sum(psf(:));
        barxy(1) = ((0.5:l_psf(1))*sum(psf,2))/s_psf - l_psf(1)/2;
        barxy(2) = (sum(psf,1)*((0.5:l_psf(2))'))/s_psf - l_psf(2)/2;
    case 'Corr'
        % Error detection
        if nargin < 3
            error('no correlation parameter found')
        elseif length(param) ~= 1
            error('The input "param" should be a single value equal to the FWHM of the Gaussian correlation map to be generated');
        end
        % pre-computation by 'CoG' method to use as starting point for correlation
        psf(psf <= 0) = 0; s_psf = sum(psf(:));
        barxy0(1) = ((0.5:l_psf(1))*sum(psf,2))/s_psf - l_psf(1)/2;
        barxy0(2) = (sum(psf,1)*((0.5:l_psf(2))'))/s_psf - l_psf(2)/2;
        % Maximization of correlation with Gaussian of FWHM 'param'
        barxy = fminsearch(@(x)Corr_PSIM(psf, param, x), barxy0, optimset('TolX',1e-3));
    case 'MF' % placeholder
        error('NOT IMPLEMENTED YET')
end

% end of function