function [Z, in, out] = createZ_PSIM(nbpix, orders, innerD, outerD)

% [Z, in, out] = createZ_PSIM(nbpix, orders, innerD, outerD);
% 
% The goal of this function is to generate Zernike polynomials, computed
% over a circular surface of radius 1, sampled over a square grid of
% pixels and cropped to the inputted inner and outer diameters.
% The inputs are:
% - 'nbpix' the linear size of the grid of pixels over which the Zernike
%   polynomials have to be computed
% - 'orders' is a vector (can be a single value) that contains the order of
%   the N polynomials to be generated (order 1 is piston)
% - 'innerD' and % 'outerD' are the radii (with 0 <= innerD < outerD <= 1)
%   of respectively the central obstruction and external pupil mask.
% The outputs are:
% - 'Z' the matrix of size nbpix*nbpix*N containing the Zernike polynomials
%   inside the pupil defined by 'innerD' and 'outerD', and zero outside.
% - the vectors 'in' and 'out' contain the coordinates of the pixels inside
%   and outside the pupil respectively. 
% 
% Note that because of this cropping if innerD is different from 0 or
% outerD from 1, the output Zernike polynomials are not anymore
% orthonormals (i.e. neither orthogonals nor normalized to a rms of 1).
% 
% A typical entry could be:
% [Z, in, out] = createZ_PSIM(960, 1:10, 0.12, 1);
% 
% Custom sub-routines required:
% 1) First level
% - FindNM.m
% - ZPolynomeQuick.m
% 
% v1.0 J.Kolb 16/12/10
% v2.0 J.Kolb 04/12/12
% - improved HELP

% Build a matrix of coordinates
[X,Y] = meshgrid(-1+1/nbpix:(2-2/nbpix)/(nbpix-1):1-1/nbpix,-1+1/nbpix:(2-2/nbpix)/(nbpix-1):1-1/nbpix);
% Convert into polar
[THETA,RHO] = cart2pol(X,Y);
% Checking the inputs
if nargin < 4
    outerD = 1;
    if nargin < 3
        innerD = 0;
    end
end
% Generating pupil
out = find(RHO > outerD | RHO < innerD);
in = find(RHO <= outerD & RHO >= innerD);

% Initialization
Z = zeros(nbpix, nbpix, length(orders));

% Loop on the polynomials to be computed
for cpt = 1:length(orders)
    % computes radial and orthogonal orders
    [nf,mf] = FindNM(orders(cpt));
    % calculation of the polynomial
    Zcpt = ZPolynomeQuick(orders(cpt),nf,mf,RHO,THETA);
    % nulling outside the pupil
    Zcpt(out) = 0;
    % population of the output matrix
    Z(:,:,cpt) = Zcpt;
end

% end of function