function [psf2, startxy, endxy] = crop_PSIM(psf, sizexy, barxy)

% [psf2, startxy, endxy] = crop_PSIM(psf, sizexy, barxy);
%
% This function crops a window in the input image 'psf'.
% The inputs are:
% - 'psf' the input image to crop
% - 'sizexy' is the size of the window to crop. It is a vector of 2
%   integers, or if only one value it is repeated)
% - 'barxy' contains the coordinates around which the crop has to be done.
%   It is a vector of 2 integers, or if only one value it is repeated. If
%   no value or 0, the center of the image is used. If -1 the barycenter of
%   the image is used).
% The outputs are:
% - 'psf2' the cropped image
% - 'startxy' and 'endxy', the coordinates of the crop rectangle in the
%   original image.
%
% A typical entry could be:
%   [psf_window, startxy, endxy] = crop_PSIM(psf, 120, [512 256]);
%
% Custom sub-routines required:
% 1) First level
% - centroid_PSIM.m
%
% v1.0 J.Kolb 16/12/10
% v2.0 J.Kolb 04/12/12
% - added the output of 'startxy' and 'endxy'
% - improved the HELP

% Initialization
s_psf = size(psf);

% Checking the input barxy
if nargin < 3
    barxy = (s_psf+1)/2;
end
if barxy == 0
    barxy = (s_psf+1)/2;
end
if length(barxy) == 1
    barxy(2) = barxy(1);
end
if barxy == -1
    barxy = centroid_PSIM(psf, 'Bar');
end

% Checking the input sizexy
if sizexy ~= uint16(sizexy)
    error('output image size must be integer')
end
if length(sizexy) == 1
    sizexy(2) = sizexy(1);
end

% startxy are the coordinates of one corner of the cropped image
startxy = barxy - (sizexy-1)/2;
% Checking that startxy is integer
if startxy ~= int16(startxy)
    disp('Non integer barycenter and/or odd input/output size combination. Rounding used.')
    startxy = round(startxy);
end
% endxy are the coordinates of the other corner of the cropped image
endxy = startxy + sizexy - 1;

% Checking that the crop fits into the input image
if (min(startxy) < 1) || (endxy(1) > s_psf(1)) || (endxy(2) > s_psf(2))
    error(['max size of a square window for the crop = [ ',...
        num2str(round(min(2*barxy(1),2*(s_psf(1)-barxy(1))))), ', ',...
        num2str(round(min(2*barxy(2),2*(s_psf(2)-barxy(2))))), ' ] pixels'])
else
    % cropping
    psf2 = psf(startxy(1):endxy(1),startxy(2):endxy(2));
end

% end of function