function [IM, pupmask, wl, C, xyvalid_ref, xyfullill, xyf_of_v] =...
    makeIM_PSIM_safe_131112(SHWFS_PSIM_input, nbpixWF, modes_ID, maxorder, amplitudes, Coeffs, nb_neighb_SA)

% [IM, pupmask, wl, C, xyvalid, xyfullill, xyf_of_v] =...
%       makeIM_PSIM(SHWFS_PSIM_input, nbpixWF, modes_ID, maxorder, amplitudes, Coeffs, nb_neighb_SA);
%
% This function generates an Interaction Matrix (IM) between 'maxorder'
% Deformable Mirror (DM) shapes and a Shack-Hartmann Wavefront Sensor (SHWFS).
% The SHWFS physical parameters are defined in the .m file named
% 'SHWFS_PSIM_input' (including path).
%
% The DM shapes are deformation maps (surface in meters) of nbpixWF*nbpixWF
% pixels. They can be either:
% - Zernike polynomials of 1 radian rms, if 'modes_ID' is set to 'Z'
% - Gaussian Influence Functions around the location of the actuators,
%   stored in the file 'modes_ID.mat' (variable 'Act_coords' of size
%   [nb_IFs x 2]). The location units are pixels in the WFS detector space
% - stored in .mat files (variable 'DM') in the subfolder
%   'DM_[modes_ID]_[nbpixWF]' from the current folder. 'modes_ID' could be
%   'SM' for System Modes, 'IF' for Influence Functions... For speeding up
%   the computations using 'nb_neighb_SA', the actuators coordinates need
%   to be inputed, in the file 'Act_coords.m' (variable 'Act_coords' of
%   size [nb_IFs x 2]) in the subfolder 'DM_[modes_ID]_[nbpixWF]' from the
%   current folder
%
% The DM shapes have a certain normalization, so if one wants to apply
% different amplitudes, those should be inputted in the parameter
% 'amplitudes', which can be either a scalar (same amplitude for all modes)
% or a vector of length 'maxorders'.
%
% This function uses the sub-routine 'SHWFS_PSIM_Mainloop.m' for the
% wavefont sensing, so its main role is more in loading (or computing) the
% DM shapes, and interpolating them (spline method) to account for
% mis-alignments before running them through the WFS model.
% Note that the routine 'SHWFS_PSIM.m' is used only once to generate the
% detailed WF parameters file.
%
% The alignment errors (mis-registration DM <-> WFS) are inputted in the
% vector 'Coeffs', containing 7 values:
% - x shift in % of a sub-aperture (>0 or <0)
% - y shift in % of a sub-aperture (>0 or <0)
% - rotation in degrees (>0 or <0)
% - x stretch in % (>0 or <0, >0 for shrinking of the DM on the WFS)
% - y stretch in % (>0 or <0, >0 for shrinking of the DM on the WFS)
% - x FWHM of the Gaussian IFs, in pixels in the space of the input WF (not
%   used if pre-computed or Zernike IFs)
% - y FWHM of the Gaussian IFs, in pixels in the space of the input WF (not
%   used if pre-computed or Zernike IFs)
%
% When recording a Zonal IM, the knowledge of the actuators' position
% allows restricting the WF computation to the 'nb_neighb_SA' nearest
% sub-apertures. In the case of Gaussian IFs, the DM shape computation is
% also restricted to the 'nb_neighb_SA' nearest sub-apertures, further
% speeding up the computation.
%
% The outputs are the Interaction Matrix 'IM' in pixels (size maxorder x [2
% x the number of valid-subapertures]), the pupil mask 'pupmask'
% (accounting for a finite DM of size equal to the pupil), the effective WF
% sensing wavelength 'wl' (matters only for the spot size on the WFS
% detector), the binning factor 'C' used between the FFT image produced and
% the one with the same size as the detector.
% The x and y coordinates of the valid subapertures are stored in the 2
% columns of the table 'xyvalid'. The (Matlab) index are stored in the
% 3rd column.
% The coordinates of the fully illuminated subapertures are stored in the 2
% columns of the table 'xyfullill'.
% 'xyf_of_v' contains the index of the fully illuminated sub-apertures
% among the valid ones (useful for selecting only those SA in the IM & CM).
%
% The main loop on the IFs is fully parallelisable by replacing 'for' by
% 'parfor' and commenting out the waitbar command.
%
% A typical entry could be:
% [IM, pupmask, wl, C, xyvalid, xyfullill, xyf_of_v] =...
%           makeIM_PSIM('SHWFS_input_AOFgeom_240px', 240, 'SM', 1170, 1, [-5, 1, 0.1, 1, -2]);
% Or for Gaussian IFs:
% [IM, pupmask, wl, C, xyvalid, xyfullill, xyf_of_v] =...
%           makeIM_PSIM('SHWFS_input_AOFgeom_240px', 240, 'Act_AOF_coords', 1170, 1, [-5, 1, 0.1, 1, -2], 30);
%
% Custom sub-routines required:
% 1) First level
% - SHWFS_PSIM.m
% - SHWFS_PSIM_PSIM.m
% - wb.m
% 2) Other levels
% - Slopes_PSIM.m
% - makepsf_PSIM.m
% - FindNM.m
% - ZPolynomeQuick.m
% - ImSensor_PSIM.m
% - centroid_PSIM.m
% - Corr_PSIM.m
% - crop_PSIM.m
% - imnoise.m (Image Processing Toolbox)
% - poissrnd.m (Statistics Toolbox)
% - createZ_PSIM.m
% - bin_PSIM.m
% - nanmean.m (Statistics Toolbox)
% - nansum.m (Statistics Toolbox)
%
% v1.0 J.Kolb 16/12/10
% v1.1 J.Kolb 22/02/11
% - adds possibility to use Zernikes as Influence functions
% - corrects error on conversion from Surface to WF (was done twice)
% - corrects error on conversion from meters to radians before WF sensing
% - creation of the WF coordinates and scaling to account for aberrations
%   only if necessary
% - added flag for Gaussian WF input
% v2.0 J.Kolb 03/05/12
% - added the possibility to input 'nb_neighb_SA', the number of
%   sub-apertures neighbourg to an actuator to use for computing the IM.
%   This greatly speeds up the computing time by running the WFS only on
%   selected sub-apertures. Moreover when the Influence functions are
%   computed as Gaussians, it is done only on a limited portion of the DM,
%   reducing further the computation time
% - moved actuators coordinates loading (if any) to beginning of the
%   function. Also added the possibility to load actuators coordinates even
%   if IFs are pre-recorded to speed up computation by ding it only in the
%   neighbourghood of the actuator
% - Improved scaling for mis-registration, including pupil
% - Added scaling of the actuators position for mis-registration (useful
%   for sub-selection of sub-apertures around actuators)
% - removal of the variable 'Tag_params', not anymore usefull in this
%   function as the WFS is directly called using 'SHWFS_PSIM_Mainloop',
%   after only one call of 'SHWFS_PSIM' to generate the detailed WF
%   parameters file
% - Initialization of the IM moved outside of the main loop
% - Update of the waitbar only every 50 cycles to save somputation time
% - Added commented option for saving influence functions
% - Modified a few variables calls so that the main loop is fully
%   parallelisable by replacing 'for' by 'parfor' and commenting out the
%   waitbar command. In particular the detailed WF parameters are called in
%   a structure.
% - Corrected a mistake in the generation of IFs, that was producing wrong
%   IMs with rotation mis-registration. Actually 'flipud' on 'xWF' removed.
%   Consequently, to keep the matching with real AO systems, the actuator
%   coordinates input files had to be modified.
% - Added check of 'maxorder' vs. the number of modes available in
%   'Act_coords'
% v2.1 J.Kolb 02/10/12
% - Kept NaNs outside of the pupil in the variable 'pupmask' (removed the
%   replacement with zeros)
% - Improved interpolation at edges of the pupil.
% - corrected missing addition of a tilt offset to the WF so that the PSFs
%   in the sub-apertures are centered on the pixels' crossing (in the
%   diffractive case)

% Initializations
modes_path = ['DM_', num2str(modes_ID), '_', num2str(nbpixWF), '/'];
run(num2str(SHWFS_PSIM_input))

% loading of actuator coordinates (if any)
if exist([num2str(modes_ID), '.mat'], 'file') == 2
    % 1) for creation of Gaussian Influence Functions
    load(num2str(modes_ID))
    Gauss_flag = 1;
    if maxorder > length(Act_coords)
        maxorder = length(Act_coords);
        warning(['Number of Modes required is larger than coordinates available. Modes above ', num2str(maxorder), ' ignored.']);
    end
elseif (exist([num2str(modes_path), 'Act_coords.mat'], 'file') == 2) && (nargin > 6)
    % 2) for sub-selection of sub-apertures in case of pre-loaded Zonal
    %    Influence Functions
    load([num2str(modes_path), 'Act_coords.mat']);
    Gauss_flag = 2;
    if maxorder > length(Act_coords)
        maxorder = length(Act_coords);
        warning(['Number of Modes required is larger than coordinates available. Modes above ', num2str(maxorder), ' ignored.']);
    end
else
    % 3) pre-loaded IFs and no actuators coordinates inputs
    Gauss_flag = 0;
end

% check the size of the "amplitudes" vector
if length(amplitudes) == 1
    amplitudes = amplitudes*ones(1,maxorder);
end

% Creation of the perfect pupil mask
[pupmask, ~, out] = createZ_PSIM(nbpixWF,1,outmin,outmax);
pupmask(out) = NaN;

% Scaling to account for mis-registration (also for sub-selection of
% sub-apertures)
if (sum(Coeffs(1:5).^2) ~= 0) || (Gauss_flag == 1)
    % Creation of the WF coordinates
    [x,y] = meshgrid(1:nbpixWF,1:nbpixWF);                          % perfect coordinates
    
    % Scaling of WF coordinates
    xWF = x - nbpixWF/2-0.5 - Coeffs(1)/100/SHsize*nbpixWF; % x xhift
    yWF = y - nbpixWF/2-0.5 - Coeffs(2)/100/SHsize*nbpixWF; % y shift
    [theta, rho] = cart2pol(xWF,yWF); theta = theta - Coeffs(3)/180*pi; % rotation
    [xWF, yWF] = pol2cart(theta, rho);
    xWF = (1 + Coeffs(4)/100)*xWF;                          % x stretch
    yWF = (1 + Coeffs(5)/100)*yWF;                          % y stretch
    xWF = xWF + nbpixWF/2+0.5;
    yWF = yWF + nbpixWF/2+0.5;
    
    % Scaling of the actual pupil mask (accounting for a finite DM of size
    % equal to the pupil)
    pupmask = interp2(x,y,pupmask,xWF,yWF,'nearest').*pupmask;
end

if (Gauss_flag ~= 0) && (nargin > 6)
    Flag1 = 1;
    % Scaling of Actuators coordinates
    Act_coords3(:,1) = Act_coords(:,1) + Coeffs(1)/100/SHsize*nbpixWF;
    Act_coords3(:,2) = Act_coords(:,2) + Coeffs(2)/100/SHsize*nbpixWF;
    c = Coeffs(1)/100/SHsize*nbpixWF+SHsize*nbpixCCD/2+0.5;
    d = Coeffs(2)/100/SHsize*nbpixWF+SHsize*nbpixCCD/2+0.5;             % rotation
    [theta, rho] = cart2pol(Act_coords3(:,1)-c,Act_coords3(:,2)-d); theta = theta + Coeffs(3)/180*pi;
    [Act_coords3(:,1), Act_coords3(:,2)] = pol2cart(theta, rho);
    Act_coords3(:,1) = Act_coords3(:,1) + c ; Act_coords3(:,2) = Act_coords3(:,2) + d;
    Act_coords3(:,1) = (1 + Coeffs(4)/100)*(Act_coords3(:,1)-c)+c;      % x stretch
    Act_coords3(:,2) = (1 + Coeffs(5)/100)*(Act_coords3(:,2)-d)+d;      % y stretch
else
    Flag1 = 0 ; nb_neighb_SA = 0; Act_coords3 = 0;
end
a = nbpixWF/nbpixCCD/SHsize; b = 0.5-a/2;

% First SHWFS measurement in order to creates the detailed WF parameters
% file. In particular xyvalid_ref contains the full list of valid
% sub-apertures before sub-selection
[~, ~, wl, C, xyvalid_ref, xyfullill, xyf_of_v, ~] =...
    SHWFS_PSIM(zeros(nbpixWF,nbpixWF), SHWFS_PSIM_input, Coeffs, 0, 0);

% Initialization of the IM
IMx = zeros(maxorder,length(xyvalid_ref)); IMy = IMx;

% Loads the detailed WF parameters. They have to ne put in the structure
% 'u' so as to be transferred to all cores in case of a 'parfor' loop.
u = load(num2str(SHWFS_PSIM_input));

% Main loop on the modes
wbID = 0;
for cpt = 1:maxorder
    % waitbar
    wbID = wb(cpt, maxorder, wbID);
    
    % Initialization of those variables to avoid warning messages in
    % 'parfor' loop
    xyvalid=[]; xyvalid_index = [];
    
    % Sub-selection of sub-apertures
    switch Flag1
        case 1
            tmp = sqrt(((xyvalid_ref(:,1)-0.5)*u.nbpixCCD+0.5-Act_coords3(cpt,2)).^2 +...
                ((xyvalid_ref(:,2)-0.5)*u.nbpixCCD+0.5-Act_coords3(cpt,1)).^2);
            [~, IX] = sort(tmp);
            xyvalid_index = IX(1:min(nb_neighb_SA,end));
            xyvalid = xyvalid_ref(xyvalid_index,:);
        case 0
            % xyvalid_index contain the information about the sub-selection of
            % sub-apertures (the ones in which to compute the WF)
            xyvalid_index = 1:size(xyvalid_ref,1);
            xyvalid = xyvalid_ref;
    end
    
    % Modes loading
    if strcmp(modes_ID,'Z')
        % 1) Zernikes
        DM = createZ_PSIM(nbpixWF,cpt,outmin,outmax); % 1 rad rms WF at wl
        DM = wl * DM / 2 / pi .* pupmask * 10^-6 / 2; % conversion to surface in meters
    elseif Gauss_flag == 1
        % 2) Gaussian IFs
        DM = zeros(nbpixWF, nbpixWF);
        % Sub-selection of DM surface
        x1 = (min(xyvalid(:,1))-1)*u.nbpixWF/u.SHsize+1; x2 = max(xyvalid(:,1))*u.nbpixWF/u.SHsize;
        y1 = (min(xyvalid(:,2))-1)*u.nbpixWF/u.SHsize+1; y2 = max(xyvalid(:,2))*u.nbpixWF/u.SHsize;
        DM(x1:x2,y1:y2) = 0.5*(1e-6)*exp(-((yWF(x1:x2,y1:y2)-a*Act_coords(cpt,2)-b).^2)/(2*(Coeffs(7)^2)) -...
            ((xWF(x1:x2,y1:y2)-a*Act_coords(cpt,1)-b).^2)/(2*(Coeffs(6)^2)));
    else
        % 3) pre-recorded modes (with or without actuator coordinates)
        if ~exist([num2str(modes_path), 'Mode_', num2str(cpt), '.mat'],'file')
            error(['Mode number ', num2str(cpt),...
                ' cannot be found. Could be solved by setting the input variable ''maxorder'' to ', num2str(cpt-1)]);
        end
        tmp = load([num2str(modes_path), 'Mode_', num2str(cpt), '.mat']);
        DM = tmp.DM;
    end
    DM(isnan(DM)) = 0;
    
    % Option for saving Influence Functions
    % save(['DM_PEACE_IF_480/Mode_', num2str(cpt), '.mat'], 'DM');
    
    % Interpolation to actual mis-registration characteristics. The
    % factor 2 is for the conversion from Surface to WF
    if (Gauss_flag ~= 1) && (sum(Coeffs(1:5).^2) ~= 0)
        DM2 = DM ; DM = 0*DM;
        % Sub-selection of DM surface
        x1 = (min(xyvalid(:,1))-1)*nbpixWF/SHsize+1; x2 = max(xyvalid(:,1))*nbpixWF/SHsize;
        y1 = (min(xyvalid(:,2))-1)*nbpixWF/SHsize+1; y2 = max(xyvalid(:,2))*nbpixWF/SHsize;
        DM(x1:x2,y1:y2) = 2 * interp2(x,y,DM2,xWF(x1:x2,y1:y2),yWF(x1:x2,y1:y2),'cubic').*pupmask(x1:x2,y1:y2);
        tmp = find(~isnan(pupmask) & isnan(DM));
        DM(tmp) = 2 * interp2(x,y,DM2.*pupmask,xWF(tmp),yWF(tmp),'linear').*pupmask(tmp);

        F = TriScatteredInterp(x(tmp),y(tmp),DM2(tmp).*pupmask(tmp),'nearest');
        DM(tmp) = 2*F(xWF(tmp),yWF(tmp));
    else
        % if Gaussian IFs, mis-registration already included
        DM = 2 * DM .* pupmask;
    end
    
    % WF measurement, directly through the function "SHWFS_PSIM_Mainloop"
    % so as not to reload the SHWFs parameters as it would do when using
    % "SHWFS_PSIM". The variables are loaded from the structure u in order
    % for the function to be parallelisable. A tilt offset is added to the
    % WF so that the PSFs in the sub-apertures are centered on the pixels'
    % crossing (in the diffractive case).
    [~, xyslopes, ~, ~, ~] = SHWFS_PSIM_Mainloop(amplitudes(cpt) * DM * 10^6  - u.TT, u.SH_model, u.param, u.Cen_meth, u.D, u.SHsize,...
        u.nbpixCCD, u.pix_scale, u.noise, u.wl, u.wl_eff, u.source_size, u.pixsize, u.outmax,u. outmin, u.subap_min_illu, u.Cmin,...
        u.Nphot, u.Exptime, u.QE, u.PSF, u.Dark, u.IMsat, u.CF, u.Bias, u.XS, u.NL, u.RON, u.Digit, u.Sensor_maps, u.nbpix, u.nbpixWF, u.spup,...
        xyvalid, xyfullill, xyf_of_v, u.pup, u.in, u.out, u.C, u.x_spot, u.y_spot, u.Srce, u.TT, u.perf_Tip, u.perf_Tilt, 0);
    
    % population of the IM
    subIM = zeros(1,length(xyvalid_ref)); tmp = xyslopes(:,:,1); subIM(1,xyvalid_index) = tmp(xyvalid(:,3))';
    IMx(cpt,:) = subIM;
    subIM = 0*subIM; tmp = xyslopes(:,:,2); subIM(1,xyvalid_index) = tmp(xyvalid(:,3))';
    IMy(cpt,:) = subIM;
end

IM = [IMx IMy];

% end of function