function [psf, wl, C, N, zeropad] = makepsf_PSIM(WFmask, pupmask, wl, d, pix_scale, nbpixCCD, Srce, Cmin, Tag_bin)

% [psf, wl, C, N, zeropad] = makepsf_PSIM(WFmask, pupmask, wl, d, pix_scale, nbpixCCD, Srce, Cmin, Tag_bin);
%
% This function computes a PSF image from an incoming WF.
% The inputs are:
% - 'WFmask' is the incomping WF in microns (a 2D square matrix)
% - 'pupmask' is the pupil mask through which the PSF is imaged (1 in the
%   pupil, 0 outside)
% - 'wl' is the wavelength in microns at which the PSF is created
% - 'd' is the pysical diameter of the aperture in meters
% - 'pix_scale' the pixel scale of this image in arcsec/pixel. For an image
%   with a resolution at Nyquist sampling of the diffraction spot of the
%   aperture, 'pix_scale' should be set to 0. For an image with a
%   resolution at a fraction of Nyquist sampling, 'pix_scale' should be set
%   to the opposite of this fraction (for example: -0.25 for an image at 4
%   times Nyquist sampling)
% - 'nbpixCCD' the size the output image required, in pixels
% - The source size can be inputted in 'Srce', either as a value equals to
%   the FWHM of a gaussian-shaped source (0 for diffraction limited), or as
%   the Source image itself (its size has then to be the same as WFmask).
% - 'Cmin' is the minimum required oversampling of the detector w.r.t. the
%   PSF image (should be set to 1 by default) The output 'C' is the actual
%   value used, i.e. how many PSF pixels were binned to produce the value
%   in one pixel of the output image. In that case binning is equivalent to
%   the collection of photons over the area of a CCD pixel.
% - The optionnal input 'Tag_bin' is only used as trick to reduce computing
%   time, in the case of simulating a Shack-Hartmann Wavefront Sensor for
%   example: detector images are computed with the same number of pixels as
%   the input WF, and are binned only once outside of the present routine
%   when the full image is computed, instead of binning at each SA
%   computation. In that case 'Tag_bin' is set to 1. Most of the time it is
%   ommited or set to 0.
% The outputs are:
% - 'psf', the image
% - 'wl' is the same as the input wavelength. Note that the
%   effective wavelength 'wl_eff' at which the image is generated (often
%   different from the input one for sampling reasons) is not outputted.
%   This one impacts only the diffraction limited spot size and not the
%   pixel scale.
% - The output 'C' is the actual value used to oversample the detector
%   w.r.t. the PSF image.
% - 'N' is the number of image pixels before binning by the factor 'C'
% - 'zeropad' is the size of the zero padding to be used in the FFT, i.e.
%   size of the FFT image
% 
% A typical entry could be:
%   [psf, wl, C] = makepsf_PSIM(Phase, pup, 2.2, 8, 0.0281, 500, 0, 4);
% or (when used in a SHWFS model):
%   [psf, wl, C] = makepsf_PSIM(WFmask, pupmask, 0.6, 0.2, 0.83, 6, 2*0.83, 2, 1);
% 
% Custom sub-routines required:
% 1) First level
% - bin_PSIM.m
% - crop_PSIM.m
% 2) Other levels
% - centroid_PSIM.m
% 
% v1.0 J.Kolb 16/12/10
% v1.1 J.Kolb 25/02/11
%  - added possibility to specify pixel scale as fraction of Nyquist
% v2.0 J.Kolb 04/12/12
% - rethink the constrains on the number of input pixels and on the
%   scalings (input WF, wavelength, pixel scale)
% - Improved comments and simplified error message on required number of
%   input pixels.
% - corrected the mistake of the squaring of the image that was done twice
%   at the end.
% - improved the HELP

% Computation of dimensions
L = length(WFmask);

% Removing of NaN values
WFmask(isnan(WFmask)) = 0; pupmask(isnan(pupmask)) = 0;

% Default pixel scale at Nyquist (diffraction limit)
diff_lim = wl/d * 1e-6 * 3600 * 180 / pi;
if pix_scale == 0
   pix_scale = diff_lim/2;
elseif pix_scale < 0
   pix_scale = - pix_scale * diff_lim/2; 
end

% Determination of the optimum zero padding value to match the input
% parameters. This is done by scanning a wide range of possible values.

% Computation of the minimum WF size to image (twice) the full required FoV
Lmin = ceil(2 * pix_scale * nbpixCCD * d / (wl * (10^-6)) / 3600 / 180 * pi);
% possible zeropadding values
nbpix_opt = (L:100*L)';
nbpix_opt = [nbpix_opt zeros(length(nbpix_opt),7)];
% FFT pixel size vs. possible zeropadding values
nbpix_opt(:,2) = diff_lim./nbpix_opt(:,1).*L;
% rounded factor between FFT pixel size and desired pixel scale
nbpix_opt(:,3) = round(pix_scale./nbpix_opt(:,2));
% for given wl, actual pixel scale after binning vs. possible zeropadding values
nbpix_opt(:,4) = nbpix_opt(:,3) .* nbpix_opt(:,2);
% for given pixel scale, actual wavelength at which spot is generated
% (after binning)
nbpix_opt(:,5) = wl*pix_scale./nbpix_opt(:,4);
% relative difference w.r.t. desired wl
nbpix_opt(:,6) = abs(nbpix_opt(:,5)-wl)/wl*100;
% sign of the slope actual wl vs. zeropadding value
nbpix_opt(1:(end-1),7) = sign(nbpix_opt(2:end,6) - nbpix_opt(1:(end-1),6));
% change of slope
nbpix_opt(2:end,8) = nbpix_opt(2:end,7) - nbpix_opt(1:(end-1),7);
% detection of positive change of slope = minimum relative wl difference
opt = find((nbpix_opt(:,8)>=1) & (nbpix_opt(:,3) >= Cmin),1,'first');

% zeropad is the size of the zero padding to be used in the FFT, i.e. size
% of the FFT image
zeropad = nbpix_opt(opt,1);
% C is the corresponding binning
C = nbpix_opt(opt,3);

% error checking
if Lmin > L
    error(['To image the required FoV, the input WF must be at least ',...
        num2str(Lmin), ' pixels wide'])
end

% N is the number of image pixels before binning
N = C * nbpixCCD;
% The actual wavelength value is recomputed, as it can be slightly
% different from the inputted one, for sampling reasons. It impacts only
% the diffraction-limited spot size, and not the pixel scale.
wl_eff = nbpix_opt(opt,5);

% Conversion of the WF into radians at the effective wavelength
WFmask = 2 * pi * WFmask / wl_eff;

% PSF computation by FFT
psf = abs(fftshift(fft2(complex(pupmask.*cos(WFmask),pupmask.*sin(WFmask)),zeropad,zeropad))).^2;
N2 = 2*ceil((zeropad+1)/2)-1;

% Convolution with extended source
if (length(Srce) == 1) && (Srce ~= 0)
    % If only source size inputted, draw source
    [x_srce, y_srce] = meshgrid(-(N2-1)/2:(N2-1)/2);
    Srce = exp(-((x_srce).^2 +(y_srce).^2)*4*log(2)/((Srce*C)^2));
end
if (length(Srce) ~= 1)
    % Convolution with source image
    psf = conv2(psf,Srce,'same');
end

% PSF truncation to the required size
if int16((zeropad - N)/2) == ((zeropad - N)/2)
    psf = crop_PSIM(psf,N);
else
    psf = crop_PSIM(psf(1:(end-1), 1:(end-1)),N);
end

% Binning the PSF to the expected number of pixels (skipped only in the
% special mode where 'Tag_bin' is inputted)
if nargin < 9
    psf = bin_PSIM(psf, C);
end

% end of function