/* cluster.c 
   Simulation of evolution of very young open cluster in the HRD. 
   Author: Mario van den Ancker (mario@astro.uva.nl)
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <math.h>
#include <time.h>

/* gcc on HP-UX contains a wrong value of RAND_MAX, so we have to 
   re-define this... */
#ifndef LONG_MAX
# define RAND_MAX   ((long)(((unsigned long) -1) >> 1))
# define LONG_MAX   ((long)(((unsigned long) -1) >> 1))
# define LONG_MIN   (~LONG_MAX)
#endif

#define NPreMSTracks    19   /* Last pre-main sequence track. */
#define NPostMSTracks   20   /* Last post-main sequence track. */
#define NBLinePoints    26   /* Last point on birthline. */
#define NStars        4999   /* Number of mass bins in simulation - 1. */


typedef struct TStar {
  double Mass;          /* Main-sequence mass in solar masses. */
  double CurMass;      /* Current mass in solar masses. */
  double Age;          /* Age offset in years. */
  double logL, logT;   /* log L [solar luminosities] and log Teff [K]. */
  long StarType;       /* Type of star: 0: Pre-MS, right of birthline.
                                        1: Pre-MS, visible.
                                        2: MS star.
                                        3: Post-MS star.
                                        4: Died. */
} TStar;

typedef TStar TStars[NStars+1];

typedef struct TTrack {
  double logL[150], logT[150], CurMass[150], Age[150];
  double Mass;
  long NPoints;
} TTrack;

typedef struct TBLineElement {
  double Age, Mass, logL, logT;
} TBLineElement;


static TStar Stars[NStars+1];
static TTrack PreMSTracks[26], PostMSTracks[26];
static TBLineElement BirthLine[NBLinePoints+1];


/* This function computes 10^x. */
static double dex(x)
double x;
{
  return exp(x*2.302585093);
}


/* Returns Miller & Scalo (1979) IMF for given mass (in solar masses). */
static double IMF(M)
double M;
{
  double logM;

  logM = log10(M);
  return (dex(1.53 - 0.96*logM - 0.47*logM*logM)/(M*2.302585093));
}


/* This function returns a linear interpolation between the points (x1, y1)
   and (x2, y2). Returns y. */
static double InterPolate(x1, y1, x2, y2, x)
double x1, y1, x2, y2, x;
{
  if (x1 == x2)
    return ((y1 + y2)/2.);
  else
    return (((x1-x)*y2 + (x-x2)*y1)/(x1-x2));
}


/* Get path for data files (same as program path) from first argument 
   to program. */
static void GetPath(argv, PathName)
char *argv, *PathName;
{
  int i, j;

  /* First determine path for data files (same as program). */
  j = -1;
  i = 0;
  do {
    PathName[i] = argv[i];
    if (PathName[i] == '/') j = i;
    i++;
  } while (argv[i] != '\0');
  PathName[j+1] = '\0';
}

/* Read the Pre-MS and Post-MS evolutionary tracks from disk into arrays. */
static void ReadTracks(PathName)
char *PathName;
{
  static int PostMSTracksNPoints[NPostMSTracks+1] =
    {12, 16, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50,
     50, 50, 50, 50, 50, 50};
  static int PreMSTracksNPoints[NPreMSTracks+1] =
    {83, 90, 95, 98, 101, 105, 109, 115, 50, 50, 50, 50, 50, 50, 
     50, 50, 50, 50, 50, 50};

  char FileName[MAXPATHLEN+12];
  FILE *InpFile;
  long i, j;
  TTrack *P;

  /* First read Pre-MS tracks. */
  sprintf(FileName, "%sprems.dat", PathName);
  InpFile = fopen(FileName, "r");
  if (InpFile == NULL)
  {
    printf("Error opening file %s!\n", FileName);
    exit(1);
  }
  for (i = 0; i <= NPreMSTracks; i++) {
    P = &PreMSTracks[i];
    fscanf(InpFile, "%lg%*[^\n]", &P->Mass);
    getc(InpFile);
    for (j = 0; j <= PreMSTracksNPoints[i]; j++)
    {
      fscanf(InpFile, "%lg%lg%lg%lg%*[^\n]", &P->Age[j], &P->logL[j],
	     &P->logT[j], &P->CurMass[j]);
      getc(InpFile);
    };
    P->NPoints = PreMSTracksNPoints[i];
  }
  fclose(InpFile);

  /* And continue with Post-MS tracks. */
  sprintf(FileName, "%spostms.dat", PathName);
  InpFile = fopen(FileName, "r");
  if (InpFile == NULL)
  {
    printf("Error opening file %s!\n", FileName);
    exit(1);
  }
  for (i = 0; i <= NPostMSTracks; i++) {
    P = &PostMSTracks[i];
    fscanf(InpFile, "%lg%*[^\n]", &P->Mass);
    getc(InpFile);
    for (j = 0; j <= PostMSTracksNPoints[i]; j++)
    {
      fscanf(InpFile, "%lg%lg%lg%lg%*[^\n]", &P->Age[j], &P->logL[j],
	     &P->logT[j], &P->CurMass[j]);
      getc(InpFile);
    };
    P->NPoints = PostMSTracksNPoints[i];
  }
  fclose(InpFile);

  /* And also read birthline. */
  sprintf(FileName, "%sbirthlin.dat", PathName);
  InpFile = fopen(FileName, "r");
  if (InpFile == NULL)
  {
    printf("Error opening file %s!\n", FileName);
    exit(1);
  }
  for (i = 0; i <= NBLinePoints; i++) {
    fscanf(InpFile, "%lg%lg%lg%lg%*[^\n]", &BirthLine[i].Mass, &BirthLine[i].Age, &BirthLine[i].logL, &BirthLine[i].logT);
    getc(InpFile);
  }
  fclose(InpFile);
}


/* Get, by interpolation, values of log L, log T and current mass from one 
   pre-main sequence track. */
static void GetPreMSLT(TNr, Age, logL, logT, CurMass, StarType)
long TNr;
double Age, *logL, *logT, *CurMass;
long *StarType;
{
  long i, i1, i2;
  double BLAge;

  if (Age <= PreMSTracks[TNr].Age[0]) {
    *logL = PreMSTracks[TNr].logL[0];
    *logT = PreMSTracks[TNr].logT[0];
    *StarType = 0;   /* Star is embedded YSO. */
  }
  if (Age >= PreMSTracks[TNr].Age[PreMSTracks[TNr].NPoints]) {
    *logL = PreMSTracks[TNr].logL[PreMSTracks[TNr].NPoints];
    *logT = PreMSTracks[TNr].logT[PreMSTracks[TNr].NPoints];
    *StarType = 2;   /* Star has already reached ZAMS. */
  }
  if (Age <= PreMSTracks[TNr].Age[0] ||
      Age >= PreMSTracks[TNr].Age[PreMSTracks[TNr].NPoints])
    return;
  for (i = 0; i <= PreMSTracks[TNr].NPoints; i++) {
    if (Age > PreMSTracks[TNr].Age[i])
      i1 = i;
  }
  i2 = i1+1;
  *logL = InterPolate(PreMSTracks[TNr].Age[i1], PreMSTracks[TNr].logL[i1],
		      PreMSTracks[TNr].Age[i2], PreMSTracks[TNr].logL[i2],
		      Age);
  *logT = InterPolate(PreMSTracks[TNr].Age[i1], PreMSTracks[TNr].logT[i1],
		      PreMSTracks[TNr].Age[i2], PreMSTracks[TNr].logT[i2],
		      Age);
  *CurMass = InterPolate(PreMSTracks[TNr].Age[i1], PreMSTracks[TNr].CurMass[i1],
		         PreMSTracks[TNr].Age[i2], PreMSTracks[TNr].CurMass[i2],
		         Age);

  /* Now determine if this one is already visible. */
  i1 = 0;
  for (i = 0; i <= NBLinePoints; i++) {
    if (PreMSTracks[TNr].Mass > BirthLine[i].Mass)
      i1 = i;
  }
  i2 = i1 + 1;
  if (i2 > NBLinePoints)
    i2 = i1-1;
  BLAge = InterPolate(BirthLine[i1].Mass, BirthLine[i1].Age,
		      BirthLine[i2].Mass, BirthLine[i2].Age,
		      PreMSTracks[TNr].Mass);
  if (BLAge < 0) {
    printf("Internal error in procedure GetPreMSLT!\n");
    exit(1);
  }
  if (Age < BLAge)   /* Optically visible pre-main sequence star. */
    *StarType = 0;   /* Star is embedded YSO. */
  else
    *StarType = 1;
}


/* Get, by interpolation, values of log L, log T and current mass from one 
   post-main sequence track. */
static void GetPostMSLT(TNr, Age, logL, logT, CurMass, StarType)
long TNr;
double Age, *logL, *logT, *CurMass;
long *StarType;
{
  long i, i1, i2;

  if (Age < PostMSTracks[TNr].Age[0]) {
    *logL = PostMSTracks[TNr].logL[0];
    *logT = PostMSTracks[TNr].logT[0];
    *StarType = 2;   /* Star is still on the main sequence. */
  }
  if (Age >= PostMSTracks[TNr].Age[PostMSTracks[TNr].NPoints]) {
    *logL = PostMSTracks[TNr].logL[PostMSTracks[TNr].NPoints];
    *logT = PostMSTracks[TNr].logT[PostMSTracks[TNr].NPoints];
    *StarType = 4;
	/* This star has died. Ceased to be. This is an ex-star. */
  }
  if (Age < PostMSTracks[TNr].Age[0] ||
      Age >= PostMSTracks[TNr].Age[PostMSTracks[TNr].NPoints])
    return;
  for (i = 0; i <= PostMSTracks[TNr].NPoints; i++) {
    if (Age > PostMSTracks[TNr].Age[i])
      i1 = i;
  }
  i2 = i1 + 1;
  *logL = InterPolate(PostMSTracks[TNr].Age[i1], PostMSTracks[TNr].logL[i1],
		      PostMSTracks[TNr].Age[i2], PostMSTracks[TNr].logL[i2],
		      Age);
  *logT = InterPolate(PostMSTracks[TNr].Age[i1], PostMSTracks[TNr].logT[i1],
		      PostMSTracks[TNr].Age[i2], PostMSTracks[TNr].logT[i2],
		      Age);
  *CurMass = InterPolate(PostMSTracks[TNr].Age[i1], PostMSTracks[TNr].CurMass[i1],
		         PostMSTracks[TNr].Age[i2], PostMSTracks[TNr].CurMass[i2],
		         Age);
  *StarType = 3;   /* Post-main sequence star. */
}


/* `Evolve' all stars: Find Teff and L corresponding to age+Cage and find out
   type of star. */
static void EvolveStars(CAge, Stars)
double CAge;
TStar *Stars;
{
  long i, j, j1, j2;
  double logL1, logL2, logT1, logT2, CurMass1, CurMass2, TotAge;
  long StarType1, StarType2;
  TStar *P;

  for (i = 0; i <= NStars; i++) {
    P = &Stars[i];
    TotAge = P->Age + CAge;
    if (TotAge < 0)
      TotAge = 0.0;
    else
      TotAge = log10(TotAge);

    /* Find out which two pre-ms tracks we're gonna use to
       interpolate between. */
    j1 = 0;
    for (j = 0; j <= NPreMSTracks; j++) {
      if (P->Mass > PreMSTracks[j].Mass)
	j1 = j;
    }
    j2 = j1+1;
    if (j2 > NPreMSTracks) {  /* Upper pre-MS mass is 7! */
      j1--;
      j2--;
    }
    GetPreMSLT(j1, TotAge, &logL1, &logT1, &CurMass1, &StarType1);
    GetPreMSLT(j2, TotAge, &logL2, &logT2, &CurMass2, &StarType2);
    P->logL = InterPolate(PreMSTracks[j1].Mass, logL1,
	  		     PreMSTracks[j2].Mass, logL2, P->Mass);
    P->logT = InterPolate(PreMSTracks[j1].Mass, logT1,
			     PreMSTracks[j2].Mass, logT2, P->Mass);
    P->CurMass = InterPolate(PreMSTracks[j1].Mass, CurMass1,
			        PreMSTracks[j2].Mass, CurMass2, P->Mass);
    P->StarType = (StarType1 + StarType2)/2;

    /* Always use post-ms tracks for massive stsrs */
    if (P->Mass > BirthLine[NBLinePoints].Mass) P->StarType = 2;

    /* Star has already reached ZAMS, so could be post-main sequence? */
    if (P->StarType == 2) {
      if (P->Mass >= PostMSTracks[0].Mass)
        /* We can savely assume that low-mass stars are still on MS... */
      { /* Again find out which two post-ms tracks we're gonna use to
	   interpolate between. */
	j1 = 0;
	for (j = 0; j <= NPostMSTracks; j++) {
	  if (P->Mass > PostMSTracks[j].Mass)
	    j1 = j;
	}
	j2 = j1+1;
	GetPostMSLT(j1, TotAge, &logL1, &logT1, &CurMass1, &StarType1);
	GetPostMSLT(j2, TotAge, &logL2, &logT2, &CurMass2, &StarType2);
	P->logL = InterPolate(PostMSTracks[j1].Mass, logL1,
			      PostMSTracks[j2].Mass, logL2, P->Mass);
	P->logT = InterPolate(PostMSTracks[j1].Mass, logT1,
			      PostMSTracks[j2].Mass, logT2, P->Mass);
	P->CurMass = InterPolate(PostMSTracks[j1].Mass, CurMass1,
			   	 PostMSTracks[j2].Mass, CurMass2, P->Mass);
	P->StarType = (StarType1 + StarType2)/2;
      }
    }
    if (P->Mass > BirthLine[NBLinePoints].Mass) P->StarType = 2;
    if ((TotAge < 0) || ((P->Mass > BirthLine[NBLinePoints].Mass) && 
                         (TotAge < BirthLine[NBLinePoints].Age)))
        P->StarType = 0;   /* Unborn/embedded: never visible */
  }
}


/* Save the resulting HRD to a raw data file. */
static void SaveHRD(SeqNr, Stars)
long SeqNr;
TStar *Stars;
{
  FILE *DataFile;
  char FileName[256];
  long i;

  sprintf(FileName, "hrd_%ld.dat", SeqNr);
  DataFile = fopen(FileName, "w+");
  for (i = 0; i <= NStars; i++) {
    if (Stars[i].StarType != 0 && Stars[i].StarType != 4)
      fprintf(DataFile, "  %1.5f  %1.5f\n", Stars[i].logT, Stars[i].logL);
  }
  fclose(DataFile);
}

/* Save luminosity function to a data file. */
static void SaveLF(SeqNr, Stars)
long SeqNr;
TStar *Stars;
{
  FILE *DataFile;
  char FileName[256];
  long i, bin;
  long LF[100];

  /* First rebin the data */
  for (i = 0; i <= 100; i++)
    LF[i] = 0;
  for (i = 0; i <= NStars; i++) {
    if (Stars[i].StarType != 0 && Stars[i].StarType != 4 && 
        Stars[i].logL >= -2.0 && Stars[i].logL <= 8.0) {
      bin = (Stars[i].logL+2)*10.;
      LF[bin]++;
    }  
  }

  /* And save it to a file. */
  sprintf(FileName, "lf_%ld.dat", SeqNr);
  DataFile = fopen(FileName, "w+");
  for (i = 0; i <= 100; i++) {
    fprintf(DataFile, "  %1.4f  %4i\n", (i+0.5)/10.-2., LF[i]);
  }
  fclose(DataFile);
}

/* Save mass function to a data file. */
static void SaveMF(SeqNr, Stars)
long SeqNr;
TStar *Stars;
{
  FILE *DataFile;
  char FileName[256];
  long i, bin;
  long MF[100];

  /* First rebin the data */
  for (i = 0; i <= 100; i++)
    MF[i] = 0;
  for (i = 0; i <= NStars; i++) {
    if (Stars[i].StarType != 0 && Stars[i].StarType != 4) {
      bin = log(Stars[i].CurMass+1)*50.;
      MF[bin]++;
    }
  }

  /* And save it to a file. */
  sprintf(FileName, "mf_%ld.dat", SeqNr);
  DataFile = fopen(FileName, "w+");
  for (i = 0; i <= 100; i++) {
    fprintf(DataFile, "  %1.4f  %4i\n", (i+0.5)/50.-1., MF[i]);
  }
  fclose(DataFile);
}


/* Save HRD plot file for use with wip plotting package. */
static void SaveWip(SeqNr, CAge)
long SeqNr;
double CAge;
{
  FILE *WipFile;
  char FileName[256];

  sprintf(FileName, "hrd_%ld.wip", SeqNr);
  WipFile = fopen(FileName, "w+");
  fprintf(WipFile, "# Set up plot window\n");
  fprintf(WipFile, "viewport 0.15 0.85 0.15 0.85\n"); 
  fprintf(WipFile, "font 2\n");
  fprintf(WipFile, "expand 1.1\n\n");
  fprintf(WipFile, "# set limits and create box\n");
  fprintf(WipFile, "limits 4.65 3.6 0.7 5.4\n");
  fprintf(WipFile, "box bcnst bcnstv\n\n");
  fprintf(WipFile, "# add labels\n");
  fprintf(WipFile, "expand 1.3\n");
  fprintf(WipFile, "mtext B 2.5 0.5 0.5 log T\\deff\\u\n");
  fprintf(WipFile, "mtext L 2.2 0.5 0.5 log L/L\\d\\(2281)\\u\n");
  fprintf(WipFile, "mtext T 1.5 0.5 0.5 t = %4.2f Myr\n\n", CAge/1.0e6);
  fprintf(WipFile, "# and plot data\n");
  fprintf(WipFile, "symbol 17\n");
  fprintf(WipFile, "expand 1.3\n");
  fprintf(WipFile, "data hrd_%ld.dat\n", SeqNr);
  fprintf(WipFile, "xcolumn 1\n");
  fprintf(WipFile, "ycolumn 2\n");
  fprintf(WipFile, "points\n");
  fclose(WipFile);
}


main(argc, argv)
int argc;
char *argv[];
{
  static char SpTypes[7] = {'O', 'B', 'A', 'F', 'G', 'K', 'M'};

  char PathName[MAXPATHLEN];
  long Statistics[8][6];
  double CAge;   /* Cluster age in years. Max. possible age is
                    determined by available post-main sequence
                    evolutionary tracks (about 3E9 yr at present). */
  double Spread; /* Duration of star formation. */
  double Prob, NormFac;
  long i, j, SpType, SeqNr;

  /* Read data files. */
  GetPath(argv[0], PathName);
  ReadTracks(PathName);

  /* Generate cluster population. */
  printf("Now starting generating cluster...\n");

  /* First distribute Masses between 0.15 and 100 solar masses, according
     to Miller-Scale IMF. */
  srandom(time(0));    /* Put a seed in the random generator. */
  NormFac = IMF(0.15);
  i = 0;
  do {
    Stars[i].Mass = 0.15 + 99.85*(double)random()/RAND_MAX;
    if ((double)random()/RAND_MAX < IMF(Stars[i].Mass)/NormFac)
      i++;
  } while (i <= NStars);

  /* And now distribute ages according to star formation scenario. */
  Spread = 1.0e6;  /* Duration of star formation [yr]. */

  /* Scenario 1: Instantaneous star formation. */
/*  for (i = 0; i <= NStars; i++) Stars[i].Age = 0.;  */

  /* Scenario 2: Constant rate of star formation. */
/*  for (i = 0; i <= NStars; i++)
    Stars[i].Age = -(Spread*random()/RAND_MAX);  */

  /* Scenario 3: Increasing rate of star formation. */
  i = 0;
  do {
    Stars[i].Age = -Spread*(double)random()/RAND_MAX;
    if ((double)random()/RAND_MAX < (-Stars[i].Age)/Spread)
      i++;
  } while (i <= NStars);

  /* Scenario 4: Decreasing rate of star formation. */
/*  i = 0;
  do {
    Stars[i].Age = -Spread*(double)random()/RAND_MAX;
    if ((double)random()/RAND_MAX < (Spread+Stars[i].Age)/Spread)
      i++;
  } while (i <= NStars);  */
    
  printf("Cluster generated!\n\n");

  printf("Results of young cluster simulation\n");
  printf("===================================\n\n");
  printf("Total number of stars in simulation: %12ld\n", NStars + 1L);
  printf("Scenario: Uniform star formation\n");
  printf("Total duration of star formation: % .1Eyears\n\n", Spread);

  SeqNr = 100;   /* Start of output file numbering. */
  CAge = 5.0e4;  /* Initial cluster age [yr]. */
  do {
    /* Do evolution. */
    EvolveStars(CAge, Stars);

    /* Generate statistics. */
    for (i = 0; i <= 7; i++) {
      for (j = 0; j <= 5; j++)
	Statistics[i][j] = 0;
    }
    for (i = 0; i <= NStars; i++) 
      if (Stars[i].Age+CAge >= 0)  /* Do not count unborn stars. */
      {
        SpType = 0;   /* O star. */
        if (Stars[i].logT <= 4.486)   /* B star. */
	  SpType = 1;
        if (Stars[i].logT <= 3.978)   /* A star. */
	  SpType = 2;
        if (Stars[i].logT <= 3.857)   /* F star. */
	  SpType = 3;
        if (Stars[i].logT <= 3.780)   /* G star. */
	  SpType = 4;
        if (Stars[i].logT <= 3.720)   /* K star. */
	  SpType = 5;
        if (Stars[i].logT <= 3.585)   /* M star. */
	  SpType = 6;
        Statistics[SpType][Stars[i].StarType]++;
        Statistics[7][Stars[i].StarType]++;
        Statistics[SpType][5]++;
        Statistics[7][5]++;
      }

    /* Summarize results. */
    printf("Results at t =% .3E yr:\n", CAge);
    printf("Sp. Type  Embedded  Pre-ZAMS      MS    Post-MS     Died     Total\n");
    for (i = 0; i <= 6; i++) {
      printf("   %c        ", SpTypes[i]);
      for (j = 0; j <= 5; j++)
	printf("%4ld      ", Statistics[i][j]);
      putchar('\n');
    }
      printf(" Total      ");
      for (j = 0; j <= 5; j++)
	printf("%4ld      ", Statistics[7][j]);
      putchar('\n');
    putchar('\n');

    /* And save the HRD in data file. */
    SaveHRD(SeqNr, Stars);
/*    SaveLF(SeqNr, Stars);
    SaveMF(SeqNr, Stars);  */
    SaveWip(SeqNr, CAge);

    /* And increase cluster age. */
    CAge += 5.0e4;
    SeqNr++;
  } while (CAge <= 3.0e7);

  exit(0);
}

/* End. */
