#!/usr/bin/env python
# PURPOSE:	qc_rawdisp.py: create screenshot-like plots of raw frames from an AB
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- July 2015
#		1.1	-- MAD method for statistics, support of data cubes, 
#			   MAXRAW as list of exposures, option -n, bug fixes (2015-07-22)
#		1.2	-- create log files that list existing raw fits, function check_dir introduced (2015-08-03)
#		1.2.1	-- new option -A for non-standard AB dir, used when re-creating plots (2015-10-15)
#		1.2.2	-- usage of astropy instead of pyfits (2018-03-09)
#		1.2.3	-- force legend to 'upper right' in cut plots (2018-10-17)
#
# PARAMETER:	-a <AB> 			name of AB (mandatory)
# OPTIONS:	-v, --version			show program's version number and exit
#		-h, --help			show help message and exit
#		-n, --no_overwrite		do not overwrite already existing plots
#		-A <DIR>, --ab_dir=<DIR>	use <DIR> instead of <DFO_AB_DIR>
#
# OUTPUT:	GIF files in <DFS_PRODUCT>/RAWDISP/<DATE>
#		rawlog files in <DFO_MON_DIR>/RAWDISP/<DATE>
#
# COMMENTS:	The script runs standalone.
#		It needs config.rawdisp in <DFO_CONF_DIR> and the <AB> in <DFO_AB_DIR>. 
# ===========================================================================================================================
TOOL_VERSION="1.2.3"
_version_ = TOOL_VERSION

# =====================================================================================
# 0. initialization
# =====================================================================================

# import modules
import sys					# interaction with Python interpreter
import os					# operating system services
import argparse					# for parsing command line arguments and options
import logging					# output of logging messages
import numpy					# methods for numerical arrays
#import pyfits					# fits file handling
from astropy.io import fits			# fits file handling
import math					# mathematical functions
import scipy					# numerical functions
import scipy.optimize				# least square optimization
import pylab					# plotting module
import string					# string handling
import time					# to get plot date
import random					# random numbers

# change some default plotting parameters
from matplotlib import rcParams
rcParams['font.size'] = 8
rcParams['lines.linewidth'] = 0.75
rcParams['axes.linewidth'] = 0.75
rcParams['legend.borderpad'] = 0.3
rcParams['legend.borderaxespad'] = 0.3
rcParams['legend.handletextpad'] = 0.4
rcParams['legend.labelspacing'] = 0.0
rcParams['legend.loc'] = 'upper right'
rcParams['patch.linewidth'] = 0.75

# =====================================================================================
# 0.1 helper functions

def set_logging(level='info', format=''):
	"""set logging level and output format (from qclib)"""

	if format == '':
		logFMT = '%(module)15s [%(levelname)7s] %(message)s'
	else:
		logFMT = format
	loglevel = {'debug': logging.DEBUG, 'info': logging.INFO, 'warning': logging.WARNING, 'error': logging.ERROR}
	logging.basicConfig(level=loglevel.get(level.lower(), logging.INFO), format=logFMT, stream=sys.stdout)

class ConfigFile:
	"""representation of ASCII configuration file
	format: <TAG> <VALUE1> <VALUE2> ... 
	(from qclib)"""

	def __init__(self, file_name='', mult_tags=[]):
		conf_file = open(file_name, 'r')
		all_the_file=conf_file.read().splitlines()
		conf_file.close()

		self.content = {}
		for line in all_the_file:
			words = line.split()
			if len(words) == 0:
				continue
			if words[0][0] == '#':
				continue
			tag = words[0]
			if not tag in mult_tags:
				# only first value is taken
				if len(words) == 1:
					self.content[tag] = ''
				else:
					if words[1] == '#':
						self.content[tag] = ''
					else:
						self.content[tag] = os.path.expandvars(words[1])
			else:
				if not tag in self.content:
					self.content[tag] = []
				sublist = []
				for word in words[1:]:
					if word[0] == '#':
						continue
					sublist.append(os.path.expandvars(word))
				self.content[tag].append(sublist)

class AssociationBlock(ConfigFile):
	"""representation of AB contents (from qclib, shortened)"""

	def __init__(self, ab_name='', ab_dir=''):
		if os.path.exists(ab_name):
			full_name = ab_name
		else:
			if ab_dir == '':
				ab_dir = os.environ.get('DFO_AB_DIR')
			full_name = ab_dir + '/' + ab_name
		mult_tag_list = ['RAW_MATCH_KEY', 'RAWFILE', 'RASSOC', 'PRODUCTS', 'MCALIB',
				'MASSOC', 'PARAM', 'WAITFOR', 'FURTHER_PS', 'FURTHER_GIF',
				'FURTHER_PNG', 'AB_STATUS', 'RB_CONTENT', 'SOF_CONTENT' ]
		ConfigFile.__init__(self, file_name=full_name, mult_tags=mult_tag_list)
		for key in self.content:
			attr = key.lower()
			self.__dict__[attr] = self.content[key]

def get_rawimg(HDU, ext=0, prescan=0, overscan=0):
	"""returns a raw image array, chopped from pre and overscan"""

	if len(HDU[ext].data.shape) == 3:
		image = HDU[ext].data[0,:,:]
	else:
		image = HDU[ext].data
	# chop prescan and overscan
	if overscan > 0:
		image = image[:,prescan:-overscan]
	else:
		image = image[:,prescan:]

	return image

def histogram(a, bins):
	"""histogram function (from qclib)"""

	# Note that argument names below are reverse of the
	# searchsorted argument names
	n = numpy.searchsorted(numpy.sort(a), bins)
	n = numpy.concatenate([n, [len(a)]])
	return n[1:]-n[:-1]

def residuals(p, y, x):
	"""residuals in Gaussian fit (from qclib)"""

	A,mu,sigma = p
	err = y-A*scipy.exp(-(x-mu)*(x-mu)/2/sigma/sigma)
	return err

def pval(x, p):
	"""calculate Gauss, based on current parameter vector (from qclib)"""

	return p[0]*scipy.exp(-(x-p[1])*(x-p[1])/2/p[2]/p[2])

def arr_median(arr):
	"""get median of a numpy array, much faster than scipy.median (copied from qclib)"""

	flatarr = arr.ravel()
	indarr = flatarr.argsort()
	index = indarr.shape[0] // 2
	if indarr.shape[0] % 2 != 0:
		return flatarr[indarr[index]]
	else:
		return (flatarr[indarr[index]] + flatarr[indarr[index-1]]) / 2.

def arr_stddev(arr, thres=5.0, median=None):
	"""calculates standard deviation of a numpy array with clipping (copied from qclib)"""

	# calculate median only if it is not given
	if median == None:
		median = arr_median(arr)
	sigma = arr.std()
	if sigma == 0.0:
		return sigma
	flatarr = arr.ravel()
	# clip those values outside median +/- thres*sigma
	flatarr = numpy.compress(numpy.less(flatarr, median+thres*sigma), flatarr)
	flatarr = numpy.compress(numpy.greater(flatarr, median-thres*sigma), flatarr)
	# return standard deviation of clipped array
	sigma = flatarr.std()
	return sigma

def arr_MAD(arr, median=None):
	"""calculate Median Absolute Deviation [WHu]"""

	# calculate median only if it is not given
	if median == None:
		median = arr_median(arr)
	# subtract median from image
	flatarr = arr.ravel() - median
	# get absolute deviations
	absdev = numpy.fabs(flatarr)
	# median of absolute deviations
	mad = arr_median(absdev)
	return mad

# =====================================================================================
# 0.2 plotting routines from qclib

def paper_size(size = 'a4'):
	"""gives paper size of A4, A5, A6 in inch"""

	size_in_inch = {'a4': (11.7,8.3), 'a5': (8.3,5.8), 'a6': (5.8,4.1)}
	return size_in_inch.get(size.lower(), (11.7,8.3))

def get_fontsizes(paper_size = 'a4'):
	"""definition of fontsizes"""

	if paper_size == 'a6':
		title_font = 8
		subtitle_font = 6
		footer_font = 5
	elif paper_size == 'a5':
		title_font = 11
		subtitle_font = 7
		footer_font = 7
	else:
		title_font = 14
		subtitle_font = 10
		footer_font = 8
	return (title_font, subtitle_font, footer_font)

class BasicPlot:
	"""plots 1D data"""

	def __init__(self):
		self.plotdata = []
		return
	def add_data(self, data, name='', lin_col='0.0'):
		self.plotdata.append((data, lin_col, name))
		return
	def draw(self, xrange=(), yrange=(), xarray=[]):
		ymin1, ymin2, ymax1, ymax2 = [], [], [], []
		xmax1 = []
		for lindef in self.plotdata:
			line = lindef[0]
			if len(xarray) == 0:
				pylab.plot(line, color=lindef[1], linestyle='steps', label=lindef[2])
			else:
				pylab.plot(xarray, line, color=lindef[1], linestyle='steps', label=lindef[2])
			# prepare for propper scaling
			mu = arr_median(line)
			sigma = arr_stddev(line, 5.0, mu)
			ymin1.append(mu-5*sigma)
			ymax1.append(mu+5*sigma)
			ymin2.append(line.min())
			ymax2.append(line.max())
			xmax1.append(line.shape[0])

		# scaling in x
		if len(xrange) == 0 and len(xarray) == 0:
			xmin = -10
			xmax = numpy.array(xmax1).max() + 10
		else:
			if len(xrange) != 0:
				xmin = xrange[0]
				xmax = xrange[1]
			else:
				xmin = xarray[0]
				xmax = xarray[-1]
		pylab.xlim(xmin, xmax)

		# scaling in y
		if len(yrange) == 0:
			ymin = numpy.array([numpy.array(ymin1).min(), numpy.array(ymin2).min()]).max()
			ymax = numpy.array([numpy.array(ymax1).max(), numpy.array(ymax2).max()]).min()
			if ymin == ymax:
				ymin = ymin - abs(ymin)
				ymax = ymax + abs(ymax)
			ydiff = ymax - ymin
			ymin = ymin - 0.1 * ydiff
			ymax = ymax + 0.3 * ydiff
		else:
			ymin = yrange[0]
			ymax = yrange[1]
		pylab.ylim(ymin, ymax)

		return xmin, xmax, ymin, ymax

class LinePlot(BasicPlot):
	"""plots several rows/columns of 2D images"""

	def __init__(self):
		self.plotdata = []
		return
	def get_line(self, image, lin, axis):
		if axis == 'row':
			return image[lin,:]
		if axis == 'col':
			return image[:,lin]
	def get_avg(self, image, axis):
		if axis == 'row':
			return numpy.sum(image,axis=0) / float(image.shape[0])
		if axis == 'col':
			return numpy.sum(image,axis=1) / float(image.shape[1])
	def add_line(self, image, axis='row', line='@0', name='', lin_col='0.0'):
		if line[0] == '@':
			data = self.get_line(image, int(line[1:]), axis)
		if line == 'avg':
			data = self.get_avg(image, axis)
		self.plotdata.append((data, lin_col, name))
		return

class HistoPlot:
	"""plots histograms of 2D images and fits to histograms (modified from qclib)"""

	def __init__(self, logplot=False, binrange=5.0, binnum=0):
		self.histo_def = []
		self.logplot = logplot
		self.binrange = binrange
		self.binnum = binnum
		self.xrange = xrange
		return
	def add_histo(self, image, fit=False, name='', lin_col=0.0):
		self.histo_def.append((image, fit, name, lin_col))
		return
	def draw(self):
		ymin, ymax = [], []
		image = self.histo_def[0][0]
		try:
			lenrange = len(self.binrange)
		except:
			lenrange = 0
		if lenrange == 2:
			if self.binnum > 0:
				stepsize = (self.binrange[1] - self.binrange[0]) / self.binnum
				if stepsize < 1.0:
					# avoid empty bins if stepsize is too small
					stepsize = 1.0
				xbins = numpy.arange(self.binrange[0], self.binrange[1], stepsize)
				xplot = xbins + 0.5 * stepsize
			else:
				sigma = 5.0 * arr_stddev(image)
				xbins = numpy.arange(self.binrange[0], self.binrange[1], 0.05*sigma)
				xplot = xbins + 0.5 * 0.05 * sigma
		else:
			mean = arr_median(image)
			sigma = self.binrange * arr_stddev(image, median=mean)
			if self.binnum > 0:
				stepsize = 2 * sigma / self.binnum
				if stepsize < 1.0:
					# avoid empty bins if stepsize is too small
					stepsize = 1.0
				xbins = numpy.arange(mean-sigma, mean+sigma, stepsize)
				xplot = xbins + 0.5 * stepsize
			else:
				xbins = numpy.arange(mean-sigma,mean+sigma,0.05*sigma)
				xplot = xbins + 0.5 * 0.05 * sigma
		for histo in self.histo_def:
			image = histo[0]
			hm = histogram(image.ravel(), xbins)
			if not histo[1]:
				if self.logplot:
					hm = numpy.where(hm == 0.0, 1.0, hm)
					hmlog = numpy.log10(hm)
					ymin.append(min(hmlog))
					ymax.append(max(hmlog))
					pylab.plot(xplot, hmlog, linestyle='steps', label=histo[2], color=histo[3])
				else:
					ymin.append(min(hm))
					ymax.append(max(hm))
					pylab.plot(xplot, hm, linestyle='steps', label=histo[2], color=histo[3])
			else:
				xbinf = xbins[:-1]
				hmf = hm[0:hm.size-1]
				A = max(hmf)
				mu = arr_median(image)
				sigma = arr_stddev(image, median=mu)
				p0 = [A, mu, sigma]
				plsq = scipy.optimize.leastsq(residuals, p0, args=(hmf, xbinf))
				A, mu, sigma = plsq[0]
				if self.logplot:
					valarr = pval(xbins,plsq[0])
					valarr = numpy.where(valarr == 0.0, 1.0, valarr)
					logarr = numpy.log10(valarr)
					pylab.plot(xplot, logarr, label=histo[2], color=histo[3])
				else:
					pylab.plot(xplot, pval(xbins,plsq[0]), label=histo[2], color=histo[3])
		# scaling for plot
		pylab.xlim(xbins.min(), xbins.max())
		diff = max(ymax) - min(ymin)
		pylab.ylim(min(ymin), max(ymax) + 0.05 * diff)
		return xbins.min(), xbins.max(), min(ymin), max(ymax) + 0.05 * diff

# =====================================================================================
# 0.3 plot a single 2D image

def plot_single_image(image, cuts=(1.0,), cmap=pylab.cm.hot, aspect='equal', lines=(-1,-1), name='', figsize='a4'):
	"""displays a single 2D image"""

	if len(cuts) == 1:
		median = arr_median(image)
		sigma = arr_stddev(image, thres=5.0, median=median)
		lcut = median-cuts[0]*sigma
		hcut = median+cuts[0]*sigma
	elif len(cuts) == 2:
		lcut = cuts[0]
		hcut = cuts[1]
	else:
		# use minimum and maximum
		lcut = image.min()
		# force hcut slightly above max value, otherwise pixel with max value becomes black
		# this seems to be a bug in matplotlib
		hcut = image.max()
		hcut = hcut + 0.01 * abs(hcut)
	pylab.imshow(image, cmap, vmin=lcut, vmax=hcut, origin='lower', aspect=aspect, interpolation='nearest')
	if name != '':
		# main plot; name == '' means thumbnail
		if lines[0] > 0:
			#pylab.plot([lines[0]-1, lines[0]-1], [0, image.shape[0]], 'b:')
			#y_ext = image.shape[0] // 40
			#pylab.plot([lines[0]-1, lines[0]-1], [-y_ext, image.shape[0]+y_ext], 'b:', clip_on=False)
			y_ext = image.shape[0] // 100
			pylab.plot([lines[0]-1, lines[0]-1], [-y_ext, image.shape[0]+y_ext], 'b-', linewidth=0.5, clip_on=False)
		if lines[1] > 0:
			#pylab.plot([0, image.shape[1]], [lines[1]-1, lines[1]-1], 'b:')
			#x_ext = image.shape[1] // 40
			#pylab.plot([-x_ext, image.shape[1]+x_ext], [lines[1]-1, lines[1]-1], 'b:', clip_on=False)
			x_ext = image.shape[1] // 100
			pylab.plot([-x_ext, image.shape[1]+x_ext], [lines[1]-1, lines[1]-1], 'b-', linewidth=0.5, clip_on=False)
		pylab.xlim(0, image.shape[1])
		pylab.ylim(0, image.shape[0])
		fs_title, fs_sub, fs_foot = get_fontsizes(figsize)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('Y', size=fs_foot)
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		#title = '%s [cuts=%5.2f,%5.2f]' % (name, lcut, hcut)
		#pylab.title(title, size=fs_sub)
	return

# =====================================================================================
# 0.4 histogram plot from image

def plot_histo(image, histo_cuts=(), figsize='a4'):
	"""histogram plot from image"""

	fs_title, fs_sub, fs_foot = get_fontsizes(figsize)
	if len(histo_cuts) == 0:
		binrange = (image.min(), image.max())
	elif len(histo_cuts) == 1:
		binrange = histo_cuts[0]
	else:
		binrange = histo_cuts
	histoplot = HistoPlot(logplot=True, binrange=binrange, binnum=50)
	histoplot.add_histo(image, fit=False, lin_col='b')
	xmin, xmax, ymin, ymax = histoplot.draw()
	locs, labels = pylab.xticks()
	if len(labels) > 6:
		labels = []
		loc_idx = 0
		for locator in locs:
			if locator < xmin:
				labels.append('%i' % locator)
			elif loc_idx % 2 == 0:
				labels.append('%i' % locator)
				loc_idx = loc_idx + 1
			else:
				labels.append('')
				loc_idx = loc_idx + 1
		pylab.xticks(locs, labels, size=fs_foot)
		pylab.xlim(xmin, xmax)

	pylab.xlabel('counts', size=fs_foot)
	pylab.ylabel('log frequency', size=fs_foot)
	#pylab.title('histogram', size=fs_sub)

	return xmin, xmax

# =====================================================================================
# 0.5 determine cut values for image plotting and histograms

def get_cuts(act_cuts='DEFAULT', median=0.0, sigma=1.0, mad=1.0):
	"""determine cut values for image plotting and histograms"""

	if act_cuts == 'DEFAULT':
		cut_values = (median - sigma, median + sigma)
	else:
		if act_cuts == 'MINMAX':
			cut_values = ()
		elif act_cuts[0:3] == 'SIG':
			mult = float(act_cuts[4:])
			cut_values = (median - mult * sigma, median + mult * sigma)
		elif act_cuts[0:3] == 'MAD':
			mult = float(act_cuts[4:])
			cut_values = (median - mult * mad, median + mult * mad)
		elif act_cuts[0:3] == 'VAL':
			exec 'cut_values = (' + act_cuts[4:] + ')'
		else:
			cut_values = (median - sigma, median + sigma)

	return cut_values

# =====================================================================================
# 0.6 check if raw fits files exist

def check_raw(AB, recreate=False):
	"""check if raw fits files exist,
	return a list with existing files,
	write this list to log file"""

	existing_files = []
	for line in AB.rawfile:
		file = line[0]
		if os.path.exists(file):
			existing_files.append(os.path.basename(file))

	if not recreate:
		logfile = os.environ.get('DFO_MON_DIR') + '/RAWDISP/' + AB.date + '/' + string.rstrip(AB.ab_name, '.ab') + '.rawlog'
		output = open(logfile, 'w')
		for rawfile in existing_files:
			output.write(rawfile + '\n')
		output.close()

	return existing_files

# =====================================================================================
# 0.7 check if directory exists

def check_dir(dir):
	"""check that dir exists, create if not"""

	if not os.path.isdir(dir):
		logging.info('creating directory ' + dir)
		try:
			os.mkdir(dir)
		except:
			if not os.path.isdir(dir):
				logging.error('directory ' + dir + ' could not be created')
				return False
	return True

# =====================================================================================
# 1. function for creating main plot
# =====================================================================================

def create_mainplot(AB, image, hdr, do_class, ext, remove_raw_match=[], 
		act_image_cuts='DEFAULT', act_histo_cuts='DEFAULT', act_full_cuts='DEFAULT', 
		ima_med = 0.0, ima_sig=1.0, ima_mad=1.0, xy_cuts=[0, 0], no_overwrite=False,
		plot_index=0, fig_size='a5'):

	# =============================================================================
	# 1.1 define some variables

	ins_name = AB.instrument
	night = AB.date
	file_name = hdr['ARCFILE']
	if len(hdr['ORIGFILE']) > 34:
		orig_name = string.rstrip(hdr['ORIGFILE'], '.fits')
	else:
		orig_name = hdr['ORIGFILE']
	dpr_catg = hdr['HIERARCH ESO DPR CATG']
	dpr_type = hdr['HIERARCH ESO DPR TYPE']
	dpr_tech = hdr['HIERARCH ESO DPR TECH']
	tpl_id = hdr['HIERARCH ESO TPL ID']

	outpng = '%s_%02i.png' % (string.rstrip(file_name,'.fits'), plot_index)
	outgif = '%s_%02i.gif' % (string.rstrip(file_name,'.fits'), plot_index)

	# in daily operations, do not create plot again if already existing
	if no_overwrite and os.path.exists(outgif):
		logging.warning('%s is already existing and is not created again' % outgif)
		return 0

	# =============================================================================
	# 1.2 figure header

	# create figure
	fig = pylab.figure(10, figsize=paper_size(fig_size))
	fig.clear()
	# smaller fonts than usual
	if fig_size == 'a4':
		font_fig_size = 'a5'
	else:
		font_fig_size = 'a6'
	fs_title, fs_sub, fs_foot = get_fontsizes(font_fig_size)

	#fig.clear()
	# invisible main axes plot used for boxes in light grey underneath title and signature
	main_ax = pylab.axes([0.001,0.001,0.998,0.998])
	pylab.axis('off')
	#pylab.plot([0], [0], 'w-')
	#pylab.plot([1], [1], 'w-')
	main_ax.broken_barh([(0,1)], (0.967,0.033), linewidths=0, facecolors='0.93')
	main_ax.broken_barh([(0,1)], (0,0.023), linewidths=0, facecolors='0.93')
	pylab.xlim(0, 1)
	pylab.ylim(0, 1)

	# header of figure
	# left column: instrument name, origfile, arcfile, extension
	fig.text(0.005, 0.97, '%s: %s' % (night, ins_name), horizontalalignment='left', verticalalignment='bottom', fontsize=fs_title)
	fig.text(0.005, 0.96, 'ORIGFILE:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.005, 0.94, 'ARCFILE:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.005, 0.92, 'raw_type:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.005, 0.90, 'do_class:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.005, 0.88, 'extension:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.060, 0.96, '%-34.34s' % orig_name[0:34], horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.060, 0.94, '%-34.34s' % file_name, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.060, 0.92, AB.raw_type, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.060, 0.90, '%-34.34s' % do_class[0:34], horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.060, 0.88, ext, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	# middle column: dpr keys, template
	fig.text(0.270, 0.97, 'type info', horizontalalignment='left', verticalalignment='bottom', fontsize=fs_sub)
	fig.text(0.270, 0.96, 'DPR.CATG:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.270, 0.94, 'DPR.TYPE:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.270, 0.92, 'DPR.TECH:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.270, 0.90, 'TPL.ID:', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.325, 0.96, dpr_catg, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.325, 0.94, '%-30.30s' % dpr_type, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.325, 0.92, '%-30.30s' % dpr_tech, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	fig.text(0.325, 0.90, '%-30.30s' % tpl_id, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	# right column: set-up keys (RAW_MATCH_KEYs from AB)
	fig.text(0.505, 0.97, 'set-up info', horizontalalignment='left', verticalalignment='bottom', fontsize=fs_sub)
	if 'RAW_MATCH_KEY' in AB.content:
		offset = 0.0
		match_key_list = []
		for raw_match_key in AB.content['RAW_MATCH_KEY']:
			match_key = string.split(string.replace(raw_match_key[0], '=', ' '))[0]
			key_val = string.split(string.replace(raw_match_key[0], '=', ' '))[1]
			if not match_key in remove_raw_match:
				match_key_list.append([match_key, key_val])
		offset = 0.015
		cnt = 0
		for line in match_key_list:
			if cnt < 5:
				fig.text(0.505, 0.96-cnt*offset, '%-.20s:' % line[0], 
						horizontalalignment='left', verticalalignment='top', fontsize=fs_foot)
				fig.text(0.610, 0.96-cnt*offset, '%-.25s' % line[1], 
						horizontalalignment='left', verticalalignment='top', fontsize=fs_foot)
			elif cnt < 10:
				fig.text(0.750, 0.96-(cnt-5)*offset, '%-.20s:' % line[0], 
						horizontalalignment='left', verticalalignment='top', fontsize=fs_foot)
				fig.text(0.855, 0.96-(cnt-5)*offset, '%-.25s' % line[1], 
						horizontalalignment='left', verticalalignment='top', fontsize=fs_foot)
			else:
				break
			cnt = cnt + 1

	# bottom: signature
	fig.text(0.005, 0.005, 'powered by QC: www.eso.org/HC', 
			horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	year, month, day, hour, minute, sec = time.localtime()[0:6]
	today = str('%04i-%02i-%02iT%02i:%02i:%02i' %(year, month, day, hour, minute, sec))
	fig.text(0.995, 0.005, 'created by qc_rawdisp.py v%s on %s' % (_version_, today), 
			horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

	# =============================================================================
	# 1.3 plotting

	# determine cuts in x and y (if default values are used)
	xy_lines = xy_cuts[:]
	if xy_lines[0] == 0:
		xy_lines[0] = image.shape[1] // 2
	if xy_lines[1] == 0:
		xy_lines[1] = image.shape[0] // 2

	# plot image
	ax = pylab.axes([0.085, 0.07, 0.57282609, 0.8])
	plot_cuts = get_cuts(act_image_cuts, ima_med, ima_sig, ima_mad)
	plot_single_image(image, cuts=plot_cuts, cmap=pylab.cm.hot, aspect='equal', lines=xy_lines,
			name='raw image', figsize=font_fig_size)
	rcParams['font.size'] = fs_foot
	pylab.colorbar()

	# plot histogram (detail)
	ax = pylab.axes([0.73152174, 0.73206897, 0.22, 0.13793103])
	histo_cuts = get_cuts(act_histo_cuts, ima_med, ima_sig, ima_mad)
	# re-use binrange for cuts through image later
	lcut, hcut = plot_histo(image, histo_cuts=histo_cuts, figsize=font_fig_size)
	fig.text(0.965, 0.80, 'histogram (detail)', horizontalalignment='right', verticalalignment='center', 
			rotation=270, fontsize=fs_foot)

	# plot histogram (full)
	histo_full = get_cuts(act_full_cuts, ima_med, ima_sig, ima_mad)
	ax = pylab.axes([0.73152174, 0.5341, 0.22, 0.13793103])
	plot_histo(image, histo_cuts=histo_full, figsize=font_fig_size)
	fig.text(0.965, 0.60, 'histogram (full)', horizontalalignment='right', verticalalignment='center', 
			rotation=270, fontsize=fs_foot)

	# plot cuts through image
	if xy_lines[1] > -1:
		ax = pylab.axes([0.73152174, 0.2679, 0.22, 0.13793103])
		plot_xcut = LinePlot()
		plot_xcut.add_line(image, 'row', '@'+str(xy_lines[1]-1), 'y='+str(xy_lines[1]), 'b')
		plot_xcut.draw(yrange=(lcut, hcut))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('counts', size=fs_foot)
		pylab.legend()
		fig.text(0.965, 0.34, 'cut in x', horizontalalignment='right', verticalalignment='center', 
				rotation=270, fontsize=fs_foot)
	else:
		fig.text(0.84, 0.34, 'cut in x not configured', 
				horizontalalignment='center', verticalalignment='center', fontsize=fs_foot)

	if xy_lines[0] > -1:
		ax = pylab.axes([0.73152174, 0.07, 0.22, 0.13793103])
		plot_ycut = LinePlot()
		plot_ycut.add_line(image, 'col', '@'+str(xy_lines[0]-1), 'x='+str(xy_lines[0]), 'b')
		plot_ycut.draw(yrange=(lcut, hcut))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('Y', size=fs_foot)
		pylab.ylabel('counts', size=fs_foot)
		pylab.legend()
		fig.text(0.965, 0.14, 'cut in y', horizontalalignment='right', verticalalignment='center', 
				rotation=270, fontsize=fs_foot)
	else:
		fig.text(0.84, 0.14, 'cut in y not configured', 
				horizontalalignment='center', verticalalignment='center', fontsize=fs_foot)

	# save figure
	pylab.savefig(outpng, dpi=150, orientation='portrait')
	os.system('convert ' + outpng + ' ' + outgif)
	os.remove(outpng)
	logging.info('plot saved as ' + outgif)

	return 1

# =====================================================================================
# 2. function for creating thumbnail image
# =====================================================================================

def create_thumb(image, file_name, act_image_cuts='DEFAULT', ima_med=0.0, ima_sig=1.0, ima_mad=1.0, thumb_frac = 0.1,
		no_overwrite=False, plot_index=0, fig_size='a5'):

	# in daily operations, do not create plot again if already existing
	outpng = '%s_thumb_%02i.png' % (string.rstrip(file_name,'.fits'), plot_index)
	outgif = '%s_thumb_%02i.gif' % (string.rstrip(file_name,'.fits'), plot_index)
	if no_overwrite and os.path.exists(outgif):
		logging.warning('%s is already existing and is not created again' % outgif)
		return

	# for thumbnail: reduce original size
	nx = int(thumb_frac * image.shape[1])
	ny = int(thumb_frac * image.shape[0])
	dpi = 150.0

	fig = pylab.figure(20, figsize=(nx/dpi, ny/dpi))
	fig.clear()

	ax = pylab.axes([0, 0, 1, 1])
	plot_cuts = get_cuts(act_image_cuts, ima_med, ima_sig, ima_mad)
	plot_single_image(image, cuts=plot_cuts, cmap=pylab.cm.hot, aspect='equal', name='')

	# save figure
	pylab.savefig(outpng, dpi=dpi, orientation='portrait')
	os.system('convert ' + outpng + ' ' + outgif)
	os.remove(outpng)
	logging.info('plot saved as ' + outgif)

	return

# =====================================================================================
# 3. loop over raw frames
# =====================================================================================

def create_plots(AB, maxnum=6, ext_list=[], xy_cuts=[0, 0], plot_config=[], def_config=[], 
		remove_raw_match=[], thumb_frac=0.1, no_overwrite=False, recreate=False, fig_size='a5'):

	# check if raw fits exist
	existing_files = check_raw(AB, recreate)
	if len(existing_files) == 0 and not recreate:
		logging.info('no raw files available')
		return 0

	raw_type = AB.raw_type
	max_per_catg = { 'ANY': maxnum }
	# configuration overwrites global value
	if plot_config != []:
		for conf in plot_config:
			if conf[0] == 'ANY' and conf[2] != 'DEFAULT':
				max_per_catg[conf[1]] = conf[2]
			elif conf[0] == raw_type and conf[2] != 'DEFAULT':
				max_per_catg[conf[1]] = conf[2]

	if len(max_per_catg) == 1:
		files_per_catg = { 'ANY': range(len(AB.rawfile)) }
	else:
		# find raw files per do_catg
		files_per_catg = {}
		for idx in range(len(AB.rawfile)):
			line = AB.rawfile[idx]
			do_catg = line[1]
			if not do_catg in files_per_catg:
				files_per_catg[do_catg] = []
			files_per_catg[do_catg].append(idx)

	# display only max_per_catg raw files
	idxlist = []
	for do_catg in files_per_catg:
		num_files = len(files_per_catg[do_catg])
		if do_catg in max_per_catg:
			max_files = max_per_catg[do_catg]
		else:
			max_files = max_per_catg['ANY']
		try:
			#stupid test if maxfiles is list
			dummy = max_files / 2
		except:
			# max_files is list
			idxlist = idxlist + max_files
			continue
		if max_files == 0:
			continue
		elif num_files <= max_files:
			idxlist = idxlist + files_per_catg[do_catg]
		else:
			if max_files == 1:
				idxlist.append(files_per_catg[do_catg][0])
			else:
				idxlist.append(files_per_catg[do_catg][0])
				idxlist.append(files_per_catg[do_catg][-1])
				random.seed(1)
				cnt = 0
				while cnt < max_files - 2:
					idx = files_per_catg[do_catg][random.randint(1, num_files-1)]
					if not idx in idxlist:
						idxlist.append(idx)
						cnt = cnt + 1

	# loop over selected HDUs
	num_plots = 0
	for idx in idxlist:
		if idx >= len(AB.rawfile):
			# this can happen if MAXRAW is user-defined list
			continue
		line = AB.rawfile[idx]
		file = line[0]
		if os.path.basename(file) not in existing_files:
			logging.info('%s not available' % file)
			if recreate:
				logging.info('trying to download using ngasClient ...')
				current_dir = os.getcwd()
				check_dir(os.path.dirname(file))
				os.chdir(os.path.dirname(file))
				os.system('ngasClient -f ' + os.path.basename(file) + ' >/dev/null 2>&1')
				os.chdir(current_dir)
				if not os.path.exists(file):
					logging.warning('... download uncsuccessful')
					continue
				else:
					logging.info('... download successful')
			else:
				continue
		HDU = fits.open(file)
		# print DPR keys on plot
		dpr_str = 'DPR.CATG=' + HDU[0].header['HIERARCH ESO DPR CATG'] + \
				' DPR.TYPE=' + HDU[0].header['HIERARCH ESO DPR TYPE'] + \
				' DPR.TECH=' + HDU[0].header['HIERARCH ESO DPR TECH'] + \
				' TPL.ID=' + HDU[0].header['HIERARCH ESO TPL ID']
		# print set-up on plot (as in AB)
		param_str = ''
		if 'RAW_MATCH_KEY' in AB.content:
			for raw_match_key in AB.content['RAW_MATCH_KEY']:
				match_key = string.split(string.replace(raw_match_key[0], '=', ' '))[0]
				if not match_key in remove_raw_match:
					param_str = param_str + ' ' + raw_match_key[0]

		do_catg = AB.content['RAWFILE'][idx][1]
		# set image/histo cuts to default values, then overwrite if configured for specific RAWTYPE/DO_CATG combination
		act_image_cuts = def_config[3]
		act_histo_cuts = def_config[4]
		act_full_cuts = def_config[5]
		xy_cuts = def_config[6]
		ext_list = def_config[7]
		# RAWTYPE/DO_CATG configuration overwrites global values for image/histo cuts, extensions, cut lines
		if plot_config != []:
			for conf in plot_config:
				# PLOT_CUTS
				if conf[0] == 'ANY' and conf[1] == 'ANY' and conf[3] != 'DEFAULT':
					act_image_cuts = conf[3]
				elif conf[0] == raw_type and conf[1] == 'ANY' and conf[3] != 'DEFAULT':
					act_image_cuts = conf[3]
				elif conf[0] == raw_type and conf[1] == do_catg and conf[3] != 'DEFAULT':
					act_image_cuts = conf[3]
				# HISTO_CUTS
				if conf[0] == 'ANY' and conf[1] == 'ANY' and conf[4] != 'DEFAULT':
					act_histo_cuts = conf[4]
				elif conf[0] == raw_type and conf[1] == 'ANY' and conf[4] != 'DEFAULT':
					act_histo_cuts = conf[4]
				elif conf[0] == raw_type and conf[1] == do_catg and conf[4] != 'DEFAULT':
					act_histo_cuts = conf[4]
				# HISTO_FULL
				if conf[0] == 'ANY' and conf[1] == 'ANY' and conf[5] != 'DEFAULT':
					act_full_cuts = conf[5]
				elif conf[0] == raw_type and conf[1] == 'ANY' and conf[5] != 'DEFAULT':
					act_full_cuts = conf[5]
				elif conf[0] == raw_type and conf[1] == do_catg and conf[5] != 'DEFAULT':
					act_full_cuts = conf[5]
				# LINES_XY
				if conf[0] == 'ANY' and conf[1] == 'ANY':
					xy_cuts = conf[6]
				if conf[0] == raw_type and conf[1] == 'ANY':
					xy_cuts = conf[6]
				if conf[0] == raw_type and conf[1] == do_catg:
					xy_cuts = conf[6]
				# EXT_LIST
				if conf[0] == 'ANY' and conf[1] == 'ANY' and conf[7] != 'DEFAULT':
					ext_list = conf[7]
				elif conf[0] == raw_type and conf[1] == 'ANY' and conf[7] != 'DEFAULT':
					ext_list = conf[7]
				elif conf[0] == raw_type and conf[1] == do_catg and conf[7] != 'DEFAULT':
					ext_list = conf[7]

		# check for statistics methods
		if act_image_cuts[0:3] == 'SIG' or act_histo_cuts[0:3] == 'SIG' or act_full_cuts[0:3] == 'SIG':
			use_sig = True
		else:
			use_sig = False
		if act_image_cuts[0:3] == 'MAD' or act_histo_cuts[0:3] == 'MAD' or act_full_cuts[0:3] == 'MAD':
			use_mad = True
		else:
			use_mad = False

		# determine fits extension for plotting
		if ext_list == []:
			ext_list = range(len(HDU))

		# loop over extensions
		for ext in ext_list:
			if 'XTENSION' in HDU[ext].header and HDU[ext].header['XTENSION'] != 'IMAGE':
				continue
			elif not 'NAXIS' in HDU[ext].header or not 'NAXIS1' in HDU[ext].header or not 'NAXIS2' in HDU[ext].header:
				continue
			elif HDU[ext].header['NAXIS'] != 2 and HDU[ext].header['NAXIS'] != 3:
				continue
			elif HDU[ext].header['NAXIS1'] == 0 or HDU[ext].header['NAXIS2'] == 0:
				continue
			elif HDU[ext].header['NAXIS'] == 3 and not 'NAXIS3' in HDU[ext].header:
				continue
			elif HDU[ext].header['NAXIS'] == 3 and HDU[ext].header['NAXIS3'] == 0:
				continue
			else:
				if ext < 0:
					plot_ext = len(HDU) + ext
				else:
					plot_ext = ext
				image = get_rawimg(HDU, ext)
				if use_sig or use_mad:
					ima_med = arr_median(image)
				else:
					ima_med = 0.0
				if use_sig:
					ima_sig = arr_stddev(image, thres=5.0, median=ima_med)
				else:
					ima_sig = 0.0
				if use_mad:
					ima_mad = arr_MAD(image, median=ima_med)
				else:
					ima_mad = 0.0
				nn = create_mainplot(AB, get_rawimg(HDU, ext), HDU[0].header, do_catg, plot_ext,
						remove_raw_match=remove_raw_match, xy_cuts=xy_cuts,
						act_image_cuts=act_image_cuts, act_histo_cuts=act_histo_cuts, 
						act_full_cuts=act_full_cuts, ima_med=ima_med, ima_sig=ima_sig, ima_mad=ima_mad,
						no_overwrite=no_overwrite, plot_index=plot_ext, fig_size=fig_size)
				create_thumb(get_rawimg(HDU, ext), HDU[0].header['ARCFILE'],
						act_image_cuts=act_image_cuts, ima_med=ima_med, ima_sig=ima_sig, ima_mad=ima_mad,
						thumb_frac=thumb_frac, no_overwrite=no_overwrite,
						plot_index=plot_ext, fig_size=fig_size)
				num_plots = num_plots + nn
		HDU.close()
	return num_plots

# =====================================================================================
# 4. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(description='Create screenshot-like plots of raw frames.')
	parser.add_argument('-v', '--version', action='store_true', default=False,
			help='''show program's version number and exit''')
	parser.add_argument('-a', dest='ab', metavar='AB', help='name of AB', default='')
	parser.add_argument('-n', '--no_overwrite', dest='no_overwrite', action='store_true', default=False,
			help='''do not overwrite already existing plots''')
	parser.add_argument('-A', '--ab_dir', metavar='DIR', dest='ab_dir', default='',
			help='''use DIR as AB directory instead of DFO_AB_DIR,
			to be used when re-creating plots after execution of moveProducts;
			non-existing raw files are downloaded, no bookkeeping in DFO_MON_DIR''')
	#parser.add_argument('-r', '--recreate', dest='recreate', action='store_true', default=False,
	#		help='''recreate plots for usage as reference:
	#		download raw files if necessary, no bookkeeping;
	#		to be used only within rawdisp2reference''')

	# parse arguments/options
	args = parser.parse_args()

	# set logging level
	set_logging()

	if args.version:
		print _version_
		sys.exit(0)
	
	if args.ab == '':
		logging.error('AB not given. Exit.')
		sys.exit(0)
	
	# if non-standard AB dir is given: re-create plots
	if args.ab_dir == '':
		recreate = False
		ab_dir = os.environ.get('DFO_AB_DIR')
	else:
		recreate = True
		ab_dir = args.ab_dir

	# get configuration
	dfo_config_dir = os.environ.get('DFO_CONFIG_DIR')
	config_file = 'config.rawdisp'
	if not os.path.exists(dfo_config_dir + '/' + config_file):
		logging.error('config file ' + config_file + ' not found. Exit.')
		sys.exit(0)
	all_config = ConfigFile(dfo_config_dir + '/' + config_file, 
			mult_tags=['PLOT_CONFIG', 'REMOVE_RAW_MATCH', 'SUPPRESS_RAWTYPE'])

	# convert configuration values where necessary
	plot_config = []
	if 'PLOT_CONFIG' in all_config.content:
		for conf in all_config.content['PLOT_CONFIG']:
			# MAXRAW
			if conf[2] != 'DEFAULT':
				if ',' in conf[2]:
					exec 'conf[2] = ['+conf[2]+']'
					for idx in range(len(conf[2])):
						conf[2][idx] = conf[2][idx] - 1
				else:
					conf[2] = int(conf[2])
			# LINES_XY
			split_conf = conf[6].replace(',', ' ').split()
			conf[6] = [0, 0]
			if split_conf[0] == 'DEFAULT':
				conf[6][0] = 0
			elif split_conf[0] == 'NONE':
				conf[6][0] = -1
			else:
				try:
					conf[6][0] = int(split_conf[0])
				except:
					conf[6][0] = 0
			if split_conf[1] == 'DEFAULT':
				conf[6][1] = 0
			elif split_conf[1] == 'NONE':
				conf[6][1] = -1
			else:
				try:
					conf[6][1] = int(split_conf[1])
				except:
					conf[6][1] = 0
			# EXT_LIST
			if conf[7] != 'DEFAULT':
				try:
					exec 'conf[7] = ['+conf[7]+']'
				except:
					conf[7] = []
			plot_config.append(conf)

	# raw match keys not to be displayed
	remove_raw_match = []
	if 'REMOVE_RAW_MATCH' in all_config.content:
		for item in all_config.content['REMOVE_RAW_MATCH']:
			remove_raw_match.append(item[0])

	# size of thumbnails
	if 'THUMBNAIL_SIZE' in all_config.content:
		thumb_frac = float(all_config.content['THUMBNAIL_SIZE']) / 100.
	else:
		#thumb_frac = 0.1
		thumb_frac = 0.05

	# set default values
	maxnum = 6			# maximum number of raw files to be plotted
	ext_list = []			# list of extentions to be plotted
	xy_cuts = [0, 0]		# cuts through image
	# complete default configuration: [ RAWTYPE, DO_CATG, MAXRAW, IMAGE_CUTS, HISTO_CUTS, HISTO_FULL, LINES_XY, EXT_LIST ]
	def_config = [ 'ANY', 'ANY', maxnum, 'SIG=1', 'SIG=6', 'VAL=-5000,70000', xy_cuts, ext_list ]
	plot_size = 'a5'		# size of plots

	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB

	AB = AssociationBlock(args.ab, ab_dir)
	logging.info(args.ab + ' parsed')

	if 'SUPPRESS_RAWTYPE' in all_config.content:
		if [AB.raw_type] in all_config.content['SUPPRESS_RAWTYPE']:
			logging.info('RAW_TYPE "%s" configfured as SUPPRESS_RAWTYPE' % AB.raw_type)
			logging.info('no plots created')
			logging.info('finished')
			return

	# create monitor directory
	if not recreate:
		rawdisp_mon = os.environ.get('DFO_MON_DIR') + '/RAWDISP'
		mon_dir = rawdisp_mon + '/' + AB.date
		if not check_dir(rawdisp_mon):
			sys.exit(1)
		if not check_dir(mon_dir):
			sys.exit(1)

	# create <DFS_PRODUCT>/RAWDISP/<DATE> if not already existing
	rawdisp_dir = os.environ.get('DFS_PRODUCT') + '/RAWDISP'
	work_dir = rawdisp_dir + '/' + AB.date
	if not check_dir(rawdisp_dir):
		sys.exit(1)
	if not check_dir(work_dir):
		sys.exit(1)

	# always create plots in <DFS_PRODUCT>/RAWDISP/<DATE>
	os.chdir(work_dir)

	# draw plots
	num_plots = create_plots(AB, maxnum=maxnum, ext_list=ext_list, xy_cuts=xy_cuts, thumb_frac=thumb_frac, 
			plot_config=plot_config, def_config=def_config, remove_raw_match=remove_raw_match, 
			no_overwrite=args.no_overwrite, recreate=recreate, fig_size=plot_size)

	if num_plots == 0:
		logging.info('no new plots created')
	else:
		logging.info('plots created in ' + work_dir)
	logging.info('finished')

# =====================================================================================
# 5. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

