#!/usr/bin/env python
# PURPOSE:	qc_genplt.py: instrument-independent basic QC plots
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- edited from qc_mstplt.py (February 2010)
#		1.0.1	-- remove usage of has_key dictionary attribute (2014-03-11)
#		1.0.2	-- usage of argparse module; option --ext_list instead of -e (2014-04-04)
#
# PARAMETERS:	-a <AB>	[ -e <EXTENSION> ]
#
# OPTIONS:	--version | -h | --help
#

_version_ = "1.0.2"

# =====================================================================================
# 0. initialization 
# 0.1 import modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =============================================================================
# 0.2 local functions

def edit_raw(fitsHDU, ext, norm, overscan=0, prescan=0):
	"normalise raw image, chop prescan and overscan"

	edited_image = norm(fitsHDU, ext)
	if overscan > 0:
		edited_image = edited_image[:,prescan:-overscan]
	else:
		edited_image = edited_image[:,prescan:]
	return edited_image

def edit_mst(fitsHDU, ext, norm):
	"scale master frame"

	mst_image = norm(fitsHDU, ext)
	return mst_image

def get_ref(proHDUs, ext, norm, catg_master, ref_match_keys, ref_dir):
	"find reference frame and normalize"

	ref_keys = ['HIERARCH ESO PRO CATG']
	key_values = [catg_master]
	for key in ref_match_keys:
		if key in proHDUs[catg_master][0].header:
			ref_keys.append(key)
			key_values.append(proHDUs[catg_master][0].header[key])

	ref_frame = find_fits(ref_keys, key_values, ref_dir)

	ref_image = []
	if ref_frame == '':
		logging.info('no reference frame found for '+catg_master)
	else:
		ref_HDU = pyfits.open(ref_dir + '/' + ref_frame)
		if proHDUs[catg_master][0].header['PIPEFILE'] == ref_HDU[0].header['PIPEFILE']:
			ref_frame = ''
			logging.warning('reference and master identical for '+catg_master)
		else:
			logging.info('reference frame ('+catg_master+') = ' + ref_frame)
			ref_image = edit_mst(ref_HDU, ext, norm)
	return ref_frame, ref_image
	
# =====================================================================================
# 1. main function 
# =====================================================================================

def draw_plots(AB, rawHDUs, proHDUs, ext_list=(), crow=-1, ccol=-1, 
		config12=[], config45=[], config3=[], config6=[],
		plot_tag='X', plot_index=0, fig_size='a4'):

	# =============================================================================
	# 1.1 import configuration

	# import general configuration file, if present
	try:
		exec 'from config_general import * '
	except:
		pass
	else:
		logging.info('configuration imported from config_gen')

	# import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB
	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.error('configuration file could not be found. Exit.')
		sys.exit(1)
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)

	# =============================================================================
	# 1.2 determine configuration for sub plots

	# decision process:
	# 1) take configuration as set in function arguments
	# 2) if function arguments are empty: take configuration from configuration file
	# 3) if not configured: take default values

	if len(config12) > 0 :
		plot12_config = config12
	else:
		try:
			test_conf = len(plot12_config)
		except:
			test_conf = 0
		if test_conf == 0:
			plot12_config = [('raw', 0, '1st raw', '0.00')]
	
	if len(config45) > 0 :
		plot45_config = config45
	else:
		try:
			test_conf = len(plot45_config)
		except:
			test_conf = 0
		if test_conf == 0:
			plot45_config = [('raw', 0, '1st raw', '0.00')]
	
	if len(config3) > 0 :
		plot3_config = config3
	else:
		try:
			test_conf = len(plot3_config)
		except:
			test_conf = 0
		if test_conf == 0:
			plot3_config = [('raw', 0, 'raw', '0.00')]
	
	if len(config6) > 0 :
		plot6_config = config6
	else:
		try:
			test_conf = len(plot6_config)
		except:
			test_conf = 0
		if test_conf == 0:
			plot6_config = [('raw', 0, 'raw', '0.00')]

	# =============================================================================
	# 1.3 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	masterfile = proHDUs[catg_master][0].header['PIPEFILE']

	# central row, column (first row, column is 0)
	if crow < 0:
		crow = centrow
	if ccol < 0:
		ccol = centcol

	# find reference frame for catg_master file (for plots 7-9)

	ref_frame, ref_image = get_ref(proHDUs, allow_ext[0], norm_mst, catg_master, ref_match_keys, ref_dir)

	if ref_frame == '':
		plot_matrix = 230
	else:
		plot_matrix = 330
	
	# =============================================================================
	# 1.4 create plots per extension

	# Plotting the results using matplotlib
	# Plot display = 3 x 3 subplots
	# 
	# Plot Order   = 1 2 3
	#                4 5 6
	#		 7 8 9
	#
	#
	# Plot 1: Row cut
	# Plot 2: Column cut
	# Plot 3: Histogram (raw/master)
	# Plot 4: Enlargement (rows)
	# Plot 5: Enlargement (columns)
	# Plot 6: Averages of rows and columns
	# Plot 7: comparison to reference (row cut)
	# Plot 8: comparison to reference (col cut)
	# Plot 9: comparison to reference (histogram)

	# which detectors to plot?
	if ext_list == ():
		ext_list = allow_ext

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes(fig_size)
	fig = pylab.figure(20, figsize=paper_size(fig_size))

	for ext in ext_list:
		logging.info('plotting for ext ' + str(ext))

		fig.clear()

		# header of figure
		fig.text(0.01, 0.99, ins_name + ': ' + plot_type, 
				horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
		fig.text(0.5, 0.99, AB.content['AB_NAME'] + '[ext=' + str(ext) + ']', 
				horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
		fig.text(0.99, 0.99, AB.content['DATE'], horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
		fig.text(0.01, 0.96, 'Raw and product frames', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
		paramstr = ''
		if len(parameters) == 0:
			if 'RAW_MATCH_KEY' in AB.content:
				for raw_match_key in AB.content['RAW_MATCH_KEY']:
					paramstr = paramstr + ' ' + string.split(string.replace(raw_match_key[0], '=', ' '))[-1]
		else:
			for param in parameters:
				if param[1] in proHDUs[catg_master][0].header:
					paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])

		if fig_size == 'a6':
			fig.text(0.99, 0.955, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)
		else:
			fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

		# bottom
		if ref_frame == '':
			fig.text(0.01, 0.03, 'Reference frame ('+catg_master+'): none', 
					horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		else:
			fig.text(0.01, 0.03, 'Reference frame ('+catg_master+'): ' + ref_frame, 
					horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		fig.text(0.01, 0.01, 'Product ('+catg_master+'): ' + masterfile + '[ext=' + str(ext) + ']', 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
		# figure date
		year, month, day, hour, min, sec = time.localtime()[0:6]
		today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
		fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

		# Plot 1
		logging.info('plot 1: cut in X')

		pylab.subplot(plot_matrix+1)

		plot1 = LinePlot()
		for pdata in plot12_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				plot1.add_line(raw_image, 'row', '@'+str(crow), pdata[2], pdata[3])
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				plot1.add_line(mst_image, 'row', '@'+str(crow), pdata[2], pdata[3])
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				if ref_frame != '':
					plot1.add_line(ref_image, 'row', '@'+str(crow), pdata[2], pdata[3])
	
		plot1.draw(yrange=plot1_yrange)
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.title('row @ Y = ' + str(crow), size=fs_sub)
		pylab.legend()

		# Plot 2
		pylab.subplot(plot_matrix+2)

		logging.info('plot 2: cut in Y')

		plot2 = LinePlot()
		for pdata in plot12_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				plot2.add_line(raw_image, 'col', '@'+str(ccol), pdata[2], pdata[3])
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				plot2.add_line(mst_image, 'col', '@'+str(ccol), pdata[2], pdata[3])
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				if ref_frame != '':
					plot2.add_line(ref_image, 'col', '@'+str(ccol), pdata[2], pdata[3])
		
		plot2.draw(yrange=plot2_yrange)
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('Y', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.title('column @ X = ' + str(ccol), size=fs_sub)
		pylab.legend()

		# Plot 3
		logging.info('plot 3: histograms')
		pylab.subplot(plot_matrix+3)

		plot3 = HistoPlot(logplot=histo_log, stepsize=histo_step, binrange=histo_range)
		for pdata in plot3_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				raw_hst = raw_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				plot3.add_histo(raw_hst, False, pdata[2], pdata[3])
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				mst_hst = mst_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				plot3.add_histo(mst_hst, False, pdata[2], pdata[3])
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				ref_hst = ref_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				if ref_frame != '':
					plot3.add_histo(ref_hst, False, pdata[2], pdata[3])
			if pdata[0] == 'rawfit':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				raw_hst = raw_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				plot3.add_histo(raw_hst, True, pdata[2], pdata[3])
			if pdata[0] == 'profit':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				mst_hst = mst_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				plot3.add_histo(mst_hst, True, pdata[2], pdata[3])
			if pdata[0] == 'reffit':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				ref_hst = ref_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]
				if ref_frame != '':
					plot3.add_histo(ref_hst, True, pdata[2], pdata[3])

		plot3.draw()

		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('counts',size=fs_foot)
		pylab.ylabel('log frequency',size=fs_foot)
		pylab.title('histograms',size=fs_sub)
		pylab.legend()

		# Plot 4
		logging.info('plot 4: cut in X detail')

		pylab.subplot(plot_matrix+4)
		plot4 = LinePlot()
		for pdata in plot45_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				plot4.add_line(raw_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), pdata[2], pdata[3])
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				plot4.add_line(mst_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), pdata[2], pdata[3])
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				if ref_frame != '':
					plot4.add_line(ref_image[:,xcentrange[0]:xcentrange[1]], 'row', '@'+str(crow), pdata[2], pdata[3])
	
		plot4.draw(xrange=xcentrange, yrange=plot4_yrange, xarray=numpy.arange(xcentrange[0], xcentrange[1]))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.legend()

		# Plot 5
		logging.info('plot 5: cut in Y detail')

		pylab.subplot(plot_matrix+5)
		plot5 = LinePlot()
		for pdata in plot45_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				plot5.add_line(raw_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), pdata[2], pdata[3])
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				plot5.add_line(mst_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), pdata[2], pdata[3])
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				if ref_frame != '':
					plot5.add_line(ref_image[ycentrange[0]:ycentrange[1],:], 'col', '@'+str(ccol), pdata[2], pdata[3])
	
		plot5.draw(xrange=ycentrange, yrange=plot5_yrange, xarray=numpy.arange(ycentrange[0], ycentrange[1]))
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('Y', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.legend()

		logging.info('plot 6: avg of rows/cols')

		pylab.subplot(plot_matrix+6)
		plot6 = LinePlot()
		for pdata in plot6_config:
			if pdata[0] == 'raw':
				raw_image = edit_raw(rawHDUs[pdata[1]], ext, norm_raw, overscan, prescan)
				plot6.add_line(raw_image, 'row', 'avg', pdata[2]+': avg rows', col_row)
				plot6.add_line(raw_image, 'col', 'avg', pdata[2]+': avg cols', col_col)
			if pdata[0] == 'pro':
				mst_image = edit_mst(proHDUs[pdata[1]], ext, norm_mst)
				plot6.add_line(mst_image, 'row', 'avg', pdata[2]+': avg rows', col_row)
				plot6.add_line(mst_image, 'col', 'avg', pdata[2]+': avg cols', col_col)
			if pdata[0] == 'ref':
				ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, pdata[1], ref_match_keys, ref_dir)
				if ref_frame != '':
					plot6.add_line(ref_image, 'row', 'avg', pdata[2]+': avg rows', col_row)
					plot6.add_line(ref_image, 'col', 'avg', pdata[2]+': avg cols', col_col)
		
		plot6.draw()
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		pylab.xlabel('X / Y', size=fs_foot)
		pylab.ylabel('ADU', size=fs_foot)
		pylab.legend()

		if ref_frame == '':
			logging.info('no reference frame available, comparison plots not created')
		else:
			# Plot 7
			logging.info('plot 7: comparison to reference, cut in X')

			mst_image = edit_mst(proHDUs[catg_master], ext, norm_mst)
			ref_frame, ref_image = get_ref(proHDUs, ext, norm_mst, catg_master, ref_match_keys, ref_dir)
			if comp_method == 'sub':
				comp_image = mst_image - ref_image
			else:
				comp_image = im_divide(mst_image, ref_image)

			# chop image for histogram
			comp_hst = comp_image[stat_yrange[0]:stat_yrange[1],stat_xrange[0]:stat_xrange[1]]

			pylab.subplot(337)

			plot7 = LinePlot()
			plot7.add_line(comp_image, 'row', '@'+str(crow), 'mst vs. ref', col_ref)
			plot7.draw()
	
			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('X', size=fs_foot)
			pylab.ylabel('ADU', size=fs_foot)
			pylab.legend()
	
			# Plot 8
			logging.info('plot 8: comparison to reference, cut in Y')

			pylab.subplot(338)

			plot8 = LinePlot()
			plot8.add_line(comp_image, 'col', '@'+str(ccol), 'mst vs. ref', col_ref)
			plot8.draw()
	
			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('Y', size=fs_foot)
			pylab.ylabel('ADU', size=fs_foot)
			pylab.legend()
	
			# Plot 9
			logging.info('plot 9: mst vs. ref histogram')

			pylab.subplot(339)

			plot9 = HistoPlot(logplot=histo_log, stepsize=0, binrange=20.0)
			plot9.add_histo(comp_hst, False, 'mst vs. ref', col_ref)
			plot9.draw()

			pylab.xticks(size=fs_foot)
			pylab.yticks(size=fs_foot)
			pylab.xlabel('counts',size=fs_foot)
			pylab.ylabel('log frequency',size=fs_foot)
			pylab.legend()

		# save figure
		outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
		pylab.savefig(outpng, dpi=150, orientation='portrait')
		logging.info('plot saved as ' + outpng)
		plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Creates instrument-independent basic QC plots on master products.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('--ext_list', metavar='EXT_LIST', dest='ext_list', default='()',
			help='''list of extentions in raw/product frames, e.g. --ext_list="(1,2,3,4)"''')

	# parse arguments/options
	args = parser.parse_args()

	# convert into Python tuples/lists
	exec 'ext_list = ' + args.ext_list

	# set logging level
	set_logging()
	
	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB 

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')
	
	# reading fits files
	# get list with HDUs of raw files

	logging.info('reading raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('reading product frames')
	proHDUs = AB.get_pro()

	# draw plots
	draw_plots(AB, rawHDUs, proHDUs, ext_list=ext_list, plot_tag='X', plot_index=0, fig_size='a4')

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

