#!/usr/bin/env python
# PURPOSE:	qc_growth.py: instrument-independent plots for growth curves from twilight flats
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- May 2014
#
# PARAMETERS:	-a <AB>	[ --raw_exts=<RAW_EXT_LIST> ] [ --pro_exts=<PRO_EXT_LIST> ]
#		[ --not_used=<EXPOSURE_LIST> ]
#
# OPTIONS:	--version | -h | --help
#

_version_ = "1.0"

# =====================================================================================
# 0. initialization: import modules 
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 1. function for drawing plots
# =====================================================================================

def draw_plots(AB, rawHDUs, proHDUs, raw_exts=(), pro_exts=(), not_used=(), plot_map=(), med_pos=(), std_pos=(),
		plot_tag='A', plot_index=0, fig_size='a4'):

	# =============================================================================
	# 1.1 import configuration

	# import general configuration file, if present
	try:
		exec 'from config_gen import * '
	except:
		pass
	else:
		logging.info('configuration imported from config_gen')

	# import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB
	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.error('configuration file could not be found. Exit.')
		sys.exit(1)
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	masterfile = proHDUs[catg_master][0].header['PIPEFILE']
	rawfile1 = AB.content['RAWFILE'][0][0].split('/')[-1]
	rawfile2 = AB.content['RAWFILE'][-1][0].split('/')[-1]

	if plot_type == '':
		raw_type = AB.raw_type
	else:
		raw_type = plot_type

	# which detectors to plot?
	if raw_exts == ():
		det_list = allow_ext
	else:
		det_list = raw_exts

	# map extension in raw frame to extension in product
	# default: it is assumed that detector 1 is in extension 1 of raw fits, detector 2 in extension 2, etc.
	# and that this is the same for pipeline products;
	# this is not true if products contain e.g. error maps in addition
	if pro_exts == ():
		try:
			pro_ext = raw_to_pro_ext
		except:
			pro_ext = allow_ext
	else:
		pro_ext = pro_exts

	# mapping of sub plots
	try:
		plot_map = fgc_plot_map
	except:
		plot_map = ()
	try:
		med_pos = fgc_med_pos
	except:
		med_pos = ()
	try:
		std_pos = fgc_std_pos
	except:
		std_pos = ()
	if len(plot_map) == 0:
		if len(det_list) == 1:
			plot_map = (2, 1)
		elif len(det_list) < 6:
			plot_map = (len(det_list), 2)
		else:
			plot_map = ((2*len(det_list)+3)//4, 4)
	
	if len(med_pos) == 0:
		if plot_map[1] == 1:
			med_pos = (1,)
			std_pos = (2,)
		elif plot_map[1] == 2:
			med_pos = range(1, plot_map[0]+1)
			std_pos = range(plot_map[0]+1, 2*plot_map[0]+1)
		else:
			med_pos = range(1, len(det_list)+1)
			std_pos = range(len(det_list)+1, 2*len(det_list)+1)

	# =============================================================================
	# 1.3 create figure, display setting information, etc.

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes(fig_size)
	fig = pylab.figure(10, figsize=paper_size(fig_size))
	fig.clear()

	# header of figure
	fig.text(0.01, 0.99, ins_name + ': ' + raw_type, 
			horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
	fig.text(0.5, 0.99, AB.ab_name, horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
	fig.text(0.99, 0.99, AB.date, horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
	fig.text(0.01, 0.96, 'Growth curves', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	paramstr = ''
	if len(parameters) == 0:
		if 'RAW_MATCH_KEY' in AB.content:
			for raw_match_key in AB.content['RAW_MATCH_KEY']:
				paramstr = paramstr + ' ' + string.split(string.replace(raw_match_key[0], '=', ' '))[-1]
	else:
		for param in parameters:
			if param[1] in proHDUs[catg_master][0].header:
				paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])
	if fig_size == 'a6':
		fig.text(0.99, 0.955, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)
	else:
		fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

	# bottom
	fig.text(0.01, 0.03, 'First raw file : ' + rawfile1,
			horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	fig.text(0.01, 0.01, 'Last raw file : ' + rawfile2,
			horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	# figure date
	year, month, day, hour, minute, sec = time.localtime()[0:6]
	today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, minute, sec))
	fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)
	
	# =============================================================================
	# 1.4 loop over detectors: two sub plots (median, standard deviation) per detector

	for det_idx in range(len(det_list)):
		det = det_list[det_idx]
		logging.info('growth curve for det ' + str(det))

		med_list, std_list, mjd_list = [], [], []
		med_used, std_used, mjd_used = [], [], []
		raw_idx = 0
		# loop over raw frames
		for HDU in rawHDUs:
			raw_image = HDU[det].data
	
			# chop pre and overscan
			if overscan > 0:
				raw_image = raw_image[:,prescan:-overscan]
			else:
				raw_image = raw_image[:,prescan:]

			# calculate median and standard deviation
			med_list.append(arr_median(raw_image[fgc_stat_yrange[0]:fgc_stat_yrange[1],fgc_stat_xrange[0]:fgc_stat_xrange[1]]))
			std_list.append(arr_stddev(raw_image[fgc_stat_yrange[0]:fgc_stat_yrange[1],fgc_stat_xrange[0]:fgc_stat_xrange[1]]))
			mjd_list.append(HDU[0].header['MJD-OBS'])

			# 2nd lists containing only exposures that have been used for combined flat
			if len(not_used) > 0:
				# user-specified list
				if not raw_idx in not_used:
					med_used.append(med_list[raw_idx])
					std_used.append(std_list[raw_idx])
					mjd_used.append(mjd_list[raw_idx])
			else:
				# try header key that contains this information
				try:
					key = fgc_key_raw_used % (raw_idx + 1)
				except:
					key = ''
				if key != '' and key in proHDUs[catg_master][pro_ext[det_idx]].header:
					if proHDUs[catg_master][pro_ext[det_idx]].header[key]:
						med_used.append(med_list[raw_idx])
						std_used.append(std_list[raw_idx])
						mjd_used.append(mjd_list[raw_idx])
				else:
					med_used.append(med_list[raw_idx])
					std_used.append(std_list[raw_idx])
					mjd_used.append(mjd_list[raw_idx])

			raw_idx = raw_idx + 1

		mjd0 = mjd_list[0]
		for idx in range(len(mjd_list)):
			mjd_list[idx] = mjd_list[idx] - mjd0
		for idx in range(len(mjd_used)):
			mjd_used[idx] = mjd_used[idx] - mjd0

		# create sub plots
		try:
			if len(fgc_key_det_id) > 0:
				if fgc_key_det_id in rawHDUs[0][det].header:
					det_name = rawHDUs[0][det].header[fgc_key_det_id]
				else:
					det_name = det
		except:
			det_name = det

		# median
		ax = pylab.subplot(plot_map[1], plot_map[0], med_pos[det_idx])
		if plot_map[0] > 3:
			majorLocator = pylab.MaxNLocator(5)
			ax.xaxis.set_major_locator(majorLocator)
		
		pylab.plot(mjd_list, med_list, 'rx-', label='all')
		pylab.plot(mjd_used, med_used, 'ro', label='used')

		pylab.xlim(-0.05*mjd_list[-1], mjd_list[-1]+0.05*mjd_list[-1])
		if len(fgc_std_yrange) == 2:
			pylab.ylim(fgc_med_yrange[0], fgc_med_yrange[1])
		
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
	
		if plot_map[1] > 2:
			if med_pos[det-1] > (plot_map[1] - 1) * plot_map[0]:
				pylab.xlabel('MJD - %f' % mjd0, size=fs_foot)
		else:
			pylab.xlabel('MJD - %f' % mjd0, size=fs_foot)
		if plot_map[0] > 3:
			if (med_pos[det-1] -1) % plot_map[0] == 0:
				pylab.ylabel('median / ADU', size=fs_foot)
		else:
			pylab.ylabel('median / ADU', size=fs_foot)
	
		if plot_map[0] > 4:
			title = 'Median, %s' % str(det_name)
		else:
			title = 'Median, det = %s' % str(det_name)
		pylab.title(title, size=fs_foot)
		pylab.legend()

		# standard deviation
		ax = pylab.subplot(plot_map[1], plot_map[0], std_pos[det_idx])
		if plot_map[0] > 3:
			majorLocator = pylab.MaxNLocator(5)
			ax.xaxis.set_major_locator(majorLocator)
		
		pylab.plot(mjd_list, std_list, 'bx-', label='all')
		pylab.plot(mjd_used, std_used, 'bo', label='used')

		pylab.xlim(-0.05*mjd_list[-1], mjd_list[-1]+0.05*mjd_list[-1])
		if len(fgc_std_yrange) == 2:
			pylab.ylim(fgc_std_yrange[0], fgc_std_yrange[1])
		pylab.xticks(size=fs_foot)
		pylab.yticks(size=fs_foot)
		
		if plot_map[1] > 2:
			if std_pos[det-1] > (plot_map[1] - 1) * plot_map[0]:
				pylab.xlabel('MJD - %f' % mjd0, size=fs_foot)
		else:
			pylab.xlabel('MJD - %f' % mjd0, size=fs_foot)
		if plot_map[0] > 3:
			if (std_pos[det-1] -1) % plot_map[0] == 0:
				pylab.ylabel('std. dev. / ADU', size=fs_foot)
		else:
			pylab.ylabel('std. dev. / ADU', size=fs_foot)
	
		if plot_map[0] > 4:
			title = 'Std. dev., %s' % str(det_name)
		else:
			title = 'Std. dev., det = %s' % str(det_name)
		pylab.title(title, size=fs_foot)
		pylab.legend()

	# save figure
	outpng = '%s%s%02i.png' % (string.rstrip(masterfile,'0123456789.fits'), plot_tag, plot_index)
	pylab.savefig(outpng, dpi=150, orientation='portrait')
	logging.info('plot saved as ' + outpng)
	plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Creates flux growth curves for skyflats.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('--raw_exts', metavar='RAW_EXTs', dest='raw_exts', default='()',
			help='''list of extentions in raw frames, e.g. --raw_exts="(1,2,3)"''')
	parser.add_argument('--pro_exts', metavar='PRO_EXTs', dest='pro_exts', default='()',
			help='''map det -> ext of product, e.g. --pro_exts="(1,3,5)"''')
	parser.add_argument('--not_used', metavar='EXPOSURE_LIST', dest='not_used', default='()',
			help='''list of exposures that are not used in combined flat, e.g. --not_used="(0,5,6)";
			useful parameter if pipeline does not write this information into header keys''')

	# parse arguments/options
	args = parser.parse_args()

	# convert into Python list
	exec 'raw_exts = ' + args.raw_exts
	exec 'pro_exts = ' + args.pro_exts
	exec 'not_used = ' + args.not_used
	
	# set logging level
	set_logging()
	
	logging.info('started')

	# parse AB
	# AB.content: dictionary with string content of AB 

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')
	
	# reading fits files
	# get list with HDUs of raw files

	logging.info('reading raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('reading product frames')
	proHDUs = AB.get_pro()

	# draw plots
	draw_plots(AB, rawHDUs, proHDUs, raw_exts=raw_exts, pro_exts=pro_exts, not_used=not_used, 
			plot_tag='A', plot_index=0, fig_size='a4')

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

