#!/usr/bin/env python
# PURPOSE:	qc_imaovw.py: overview of images contained in a multi-extension fits file
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0 	-- February 2014
#		1.0.1	-- check compatibility of extension list with actual HDU (2014-02-19)
#		1.0.2	-- usage of argparse module (2014-04-08)
#		1.0.3	-- make robust against missing keys when determining plot titles (2014-12-02)
#		1.1	-- optional overplot of horizontal/vertical lines (2017-03-20)
#
# PARAMETERS:	-a <AB> [ --type=<TYPE> ] [ --ext_list=<EXT_LIST> ]
#		[ --shape=<SHAPE> ] [ --plot_pos=PLOT_POSITION_LIST ]
#		[ --cuts=<CUTS> ] [ --aspect=equal|auto ]
#		[ --plot_title_fmt=<FMTSTR> ] [ --plot_title_keys=<KEYLIST> ]
#		[ --add_info=<STRING> ] [ --add_setting=<STRING> ]
#		[ --all_labels ] [ --long_titles ]
#		[ --plot_tag=<TAG> ] [ --plot_index=<IDX> ]
#

_version_ = "1.1"

# =====================================================================================
# 0. initialization: import modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 0.1 helper functions

def get_image(HDU, ext=0, prescan=0, overscan=0):
	"""returns a raw image , chopped from pre and overscan in case of a raw frame, NAN replaced by 0"""

	image = HDU[ext].data
	if not 'PIPEFILE' in HDU[0].header:
		# chop prescan and overscan
		if overscan > 0:
			image = image[:,prescan:-overscan]
		else:
			image = image[:,prescan:]
	# replace NAN values
	image[numpy.isnan(image)] = 0.0

	return image

# =====================================================================================
# 1. create overview plot
# =====================================================================================

def create_overview(AB, HDU, ext_list=(), image_map=(), plot_pos=(), act_cuts=(1.0,), aspect='equal', addcol=(), addrow=(),
		default_image=numpy.zeros((14,14), dtype=numpy.float32),
		plot_title_fmt='%s', plot_title_keys=('EXTNAME',), infostr='', paramstr='', 
		all_labels=False, title_length='short', plot_tag='A', plot_index=0):

	# =============================================================================
	# 1.1 import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB

	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.info('configuration file could not be found. Using default.')
		# define some variables
		overscan = 0
		prescan = 0
		parameters = []
		plot_type = ''
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	
	if 'PIPEFILE' in HDU[0].header:
		masterfile = HDU[0].header['PIPEFILE']
		catg = HDU[0].header['HIERARCH ESO PRO CATG']
		if ext_list == () or ext_list == []:
			try:
				ext_list = raw_to_pro_ext
			except:
				if len(HDU) == 1:
					ext_list = [0]
				else:
					ext_list = range(1, len(HDU))
	else:
		masterfile = AB.content['PROD_ROOT_NAME'] + '_0000.fits'
		catg = HDU[0].header['HIERARCH ESO DPR CATG'] + ' ' + HDU[0].header['HIERARCH ESO DPR TYPE'] + ' ' \
				+ HDU[0].header['HIERARCH ESO DPR TECH']
		if ext_list == () or ext_list == []:
			try:
				ext_list = allow_ext
			except:
				if len(HDU) == 1:
					ext_list = [0]
				else:
					ext_list = range(1, len(HDU))

	if plot_type == '':
		raw_type = AB.content['RAW_TYPE']
	else:
		raw_type = plot_type
	
	if infostr == '':
		infostr = catg
	
	if plot_pos == () or plot_pos == []:
		plot_pos = range(1, len(ext_list)+1)
	
	if image_map == () or image_map == []:
		if len(ext_list) == 1:
			image_map = (1, 1)
		elif len(ext_list) < 4:
			image_map = (len(ext_list), 1)
		elif len(ext_list) < 7:
			image_map = ((len(ext_list)+1)//2, 2)
		elif len(ext_list) < 13:
			image_map = ((len(ext_list)+2)//3, 3)
		elif len(ext_list) < 25:
			image_map = ((len(ext_list)+3)//4, 4)
		else:
			image_map = ((len(ext_list)+4)//5, 5)

	# =============================================================================
	# 1.3 create figure, display setting information, etc.

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes('A4')
	fig = pylab.figure(10, figsize=paper_size('A4'))
	fig.clear()

	# header of figure
	fig.text(0.05, 0.99, ins_name + ': ' + raw_type, horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
	fig.text(0.5, 0.99, AB.content['AB_NAME'], horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
	fig.text(0.99, 0.99, AB.content['DATE'], horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
	fig.text(0.05, 0.96, infostr, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	if paramstr == '':
		if len(parameters) == 0:
			if 'RAW_MATCH_KEY' in AB.content:
				for raw_match_key in AB.content['RAW_MATCH_KEY']:
					paramstr = paramstr + ' ' + string.split(string.replace(raw_match_key[0], '=', ' '))[-1]
		for param in parameters:
			if param[1] in HDU[0].header:
				paramstr = paramstr + ' ' + param[0] + ' = ' + str(HDU[0].header[param[1]])
	fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

	# file name
	if 'PIPEFILE' in HDU[0].header:
		fig.text(0.01, 0.01, 'Product: ' + masterfile, 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	elif 'ARCFILE' in HDU[0].header:
		fig.text(0.01, 0.01, 'Raw file: ' + HDU[0].header['ARCFILE'], 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)

	# figure date
	year, month, day, hour, min, sec = time.localtime()[0:6]
	today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
	fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

	# =============================================================================
	# 1.4 create sub plots
	
	logging.info('plot: overview of extensions')
	mainplot = ImagePlot(image_map=image_map)
	
	# loop over extensions
	for idx in range(len(ext_list)):
		ext = ext_list[idx]
		if ext > len(HDU) - 1:
			logging.warning('... ext %i not found' % ext)
			continue
		logging.info('... ext %i' % ext)
		if HDU[ext].header['NAXIS'] == 2:
			image = get_image(HDU, ext, prescan, overscan)
		else:
			image = default_image
		if plot_title_keys == [] and plot_title_keys == ():
			key_values = ('')
		else:
			tmp_values = []
			for key in plot_title_keys:
				if key in HDU[ext].header:
					tmp_values.append(HDU[ext].header[key])
				else:
					tmp_values.append('')
			key_values = tuple(tmp_values)
		try:
			title = plot_title_fmt % key_values
		except:
			title = ''
		mainplot.add_image(image, plot_pos[idx], title, cuts=act_cuts)
	mainplot.draw(aspect=aspect, all_labels=all_labels, title_length=title_length, addcol=addcol, addrow=addrow)

	# save figures
	outpng = '%s%s%02i.png' % (string.rstrip(masterfile,'0123456789.fits'), plot_tag, plot_index)
	pylab.savefig(outpng, dpi=150, orientation='portrait')
	logging.info('plot saved as ' + outpng)
	plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Creates an overview plot of images contained in a multi-extension fits file.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('-t', '--type', metavar='TYPE', dest='type', default='0',
		help='''file to be plotted; 0, 1, 2, -1, etc. for first, second, third, last, etc. raw frame;
		or PRO.CATG of product [default=0]''')
        parser.add_argument('--ext_list', metavar='EXT_LIST', dest='ext_list', default='()',
		help='list of extensions to be plotted, e.g. --ext_list="(1,3,5)"')
	parser.add_argument('--shape', metavar='SHAPE', dest='image_map', default='()',
		help='shape of sub plots on figure, e.g. for 3x2 plots use --shape="(3,2)"')
        parser.add_argument('--plot_pos', metavar='PLOT_POSITIONs', dest='plot_pos', default='()',
		help='position of images in plot, e.g. --plot_pos="(1,2,3)"')
	parser.add_argument('--cuts', metavar='CUTs', dest='cuts', default='(1,)',
		help='''lower and higher cut values for plots, e.g.
		--cuts="()" [set cuts to min/max of image]
		--cuts="(2,)" [set cuts to avg +/- 2 * sigma]
		--cuts="(0,1)" [lower cut = 0, upper = 1]
		default is --cuts="(1,)" ''')
	parser.add_argument('--aspect', metavar='ASPECT', dest='aspect', default='equal',
		help='aspect ratio of plots: equal|auto')
	parser.add_argument('--addcol', metavar='ADDCOL', dest='addcol', default='()',
		help='''optional overplot of vertical line,
		e.g. --addcol=6.5''')
	parser.add_argument('--addrow', metavar='ADDROW', dest='addrow', default='()',
		help='''optional overplot of horizontal line,
		e.g. --addrow=6.5''')
	parser.add_argument('--plot_title_fmt', metavar='FMTSTR', dest='plot_title_fmt', default='%s',
		help='''format string for titles of sub plots,
		e.g. --plot_title_fmt="ext = %%s" ''')
	parser.add_argument('--plot_title_keys', metavar='KEYLIST', dest='plot_title_keys', default='["EXTNAME"]',
		help='''header keys of extensions to appear in sub plot
		titles, e.g. --plot_title_keys="['EXTNAME', 'HIERARCH ESO DET DIT']"''')
	parser.add_argument('--add_info', metavar='STRING', dest='info', default='',
		help='''additional information to be printed on figure,
		appears on the left side of the figure sub title''')
	parser.add_argument('--add_setting', metavar='STRING', dest='setting', default='',
		help='''additional information to be printed on figure,
		appears on the right side of the figure sub title''')
	parser.add_argument('--all_labels', dest='all_labels', action='store_true', default=False,
		help='enforce plotting labels on all sub plots')
	parser.add_argument('--long_titles', dest='long_titles', action='store_true', default=False,
		help='use long titles for sub plots')
	parser.add_argument('--plot_tag', metavar='TAG', dest='plot_tag', default='A',
		help='''plot tag used for png output;
		output is written to <PROD_ROOT_NAME>_<TAG><IDX>.png''')
	parser.add_argument('--plot_index', metavar='IDX', dest='plot_index', type=int, default=0,
		help='''plot index used for png output;
		output is written to <PROD_ROOT_NAME>_<TAG><IDX>.png''')

	# parse arguments/options
	args = parser.parse_args()

	# convert list of extensions into Python list
	exec 'ext_list = ' + args.ext_list
	exec 'image_map = ' + args.image_map
	exec 'plot_pos = ' + args.plot_pos
	exec 'act_cuts = ' + args.cuts
	exec 'plot_title_keys = ' + args.plot_title_keys
	exec 'addcol = ' + args.addcol
	exec 'addrow = ' + args.addrow

	# manage some options values
	if args.aspect == 'auto':
		aspect='auto'
	else:
		aspect='equal'

	if args.long_titles:
		title_length='long'
	else:
		title_length='short'

	# set logging level
	set_logging()

	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')

	# reading fits files
	if args.type[0] in ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-']:
		# get list with HDUs of raw files
		logging.info('reading raw frames')
		rawHDUs = AB.get_raw()
		HDU = rawHDUs[int(args.type)]
	else:
		# get dictionary with HDUs of all product files
		logging.info('reading product frames')
		proHDUs = AB.get_pro()
		HDU = proHDUs[args.type]

	# draw plots
	create_overview(AB, HDU, ext_list=ext_list, image_map=image_map, plot_pos=plot_pos, act_cuts=act_cuts, aspect=aspect,
			addcol=addcol, addrow=addrow,
			plot_title_fmt=args.plot_title_fmt, plot_title_keys=plot_title_keys, title_length=title_length,
			infostr=args.info, paramstr=args.setting, all_labels=args.all_labels,
			plot_tag=args.plot_tag, plot_index=args.plot_index)

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

