#!/usr/bin/env python
# PURPOSE:	qc_mstdisp.py: visualize a 2D master product with plots for raw, master, and comparison to reference
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- August 2008, edited from crires_mstdisp.py
# 		1.1	-- new handling of options and configuration import (2009-03-20)
#		1.1.1	-- optional image compression (2009-05-27)
#		1.1.2	-- crow and ccol as parameters for display2D (2009-08-13)
#		1.2	-- upgrade to numpy V1.3 and matplotlib V0.99 (2009-11-23)
#		1.3	-- allow different extension in product frame than in raw frame (2014-01-17)
#		1.3.1	-- bug fix with outpng name (2014-01-24)
#		1.3.2	-- remove usage of has_key dictionary attribute (2014-03-12)
#		1.3.3	-- usage of argparse module (2014-04-08)
#
# PARAMETERS:	-a <AB> [ -e <EXT> ] [ --pro_ext=<PRO_EXT> ] [ --aspect=auto|equal ]
#
# COMMENTS:

_version_ = "1.3.3"

# =====================================================================================
# 0. initialization: import modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 1. function for displaying 2D images
# =====================================================================================

def display2D(AB, proHDUs, rawHDUs, ext=0, pro_ext=-1, ccol=-1, crow=-1, aspect='auto', 
		imcomp=1, plot_tag='A', plot_index=0, fig_size='a4'):

	# =============================================================================
	# 1.1 import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB

	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.error('configuration file could not be found. Exit.')
		sys.exit(1)
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	masterfile = proHDUs[catg_master][0].header['PIPEFILE']

	# function parameters overwrite configuration
	if ccol > -1:
		centcol = ccol
	if crow > -1:
		centcol = crow
	if pro_ext < 0:
		pro_ext = ext

	# =============================================================================
	# 1.3 find reference frame

	all_files = os.listdir(ref_dir)
	cand_files = []
	for file in all_files:
		if ref_root_name in file and 'fits' in file:
			cand_files.append(file)

	ref_frame = ''
	for file in cand_files:
		full_name = ref_dir + '/' + file
		temp_HDU = pyfits.open(full_name)
		for key in ref_match_keys:
			if key in proHDUs[catg_master][0].header:
				if not key in temp_HDU[0].header:
					break
				if temp_HDU[0].header[key] != proHDUs[catg_master][0].header[key]:
					break
		else:
			ref_frame = file
			ref_HDU = temp_HDU
			break

	if ref_frame == '':
		logging.warning('no reference frame found')
	else:
		if proHDUs[catg_master][0].header['PIPEFILE'] == ref_HDU[0].header['PIPEFILE']:
			ref_frame = ''
			logging.warning('reference and master identical')
		else:
			logging.info('reference frame = ' + ref_frame)

	# =============================================================================
	# 1.3 create two plots: overview of whole frames plus details

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes(fig_size)
	fig = pylab.figure(20, figsize=paper_size(fig_size))

	# header of figure
	fig.clear()
	fig.text(0.01, 0.99, ins_name + ': ' + plot_type, horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
	fig.text(0.5, 0.99, AB.content['AB_NAME'] + '[' + str(ext) + ']', horizontalalignment='center', verticalalignment='top', 
			fontsize=fs_title)
	fig.text(0.99, 0.99, AB.content['DATE'], horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
	fig.text(0.01, 0.96, 'Master and reference frames', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	paramstr = ''
	for param in parameters:
		if param[1] in proHDUs[catg_master][0].header:
			paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])
	fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

	# bottom
	if ref_frame == '':
		fig.text(0.01, 0.03, 'RefFrame: none', horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	else:
		fig.text(0.01, 0.03, 'RefFrame: ' + ref_frame + '[ext=' + str(pro_ext) + ']', 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	fig.text(0.01, 0.01, 'Product: ' + masterfile + '[ext=' + str(pro_ext) + ']', 
			horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	# figure date
	year, month, day, hour, min, sec = time.localtime()[0:6]
	today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
	fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

	# scale master and reference
	compImgs = []
	mstImgs = []
	mst_image = norm_mst(proHDUs[catg_master], pro_ext)
	mstImgs.append(mst_image)
	if ref_frame != '':
		ref_image = norm_mst(ref_HDU, pro_ext)
		# comparison of master with reference
		if comp_method == 'sub':
			compImgs.append(mst_image - ref_image)
		else:
			compImgs.append(im_divide(mst_image, ref_image))
	
	# chop prescan and overscan
	rawImgs = []
	raw_image = rawHDUs[0][ext].data
	if overscan > 0:
		raw_image = raw_image[:,prescan:-overscan]
	else:
		raw_image = raw_image[:,prescan:]
	rawImgs.append(raw_image)

	# Plot raw, master, and master - reference (whole frames)
	logging.info('plot 1: overview whole frames')
	if ref_frame != '':
		mainplot = ImagePlot(image_map=(3,1))
	else:
		mainplot = ImagePlot(image_map=(2,1))
	mainplot.add_image(rawImgs[0], 1, '1st RAW', compress=imcomp, cmap=col_map)
	mainplot.add_image(mstImgs[0], 2, 'MST', compress=imcomp, cmap=col_map)
	if ref_frame != '':
		mainplot.add_image(compImgs[0], 3, 'MST vs REF', compress=imcomp, cmap=col_map)
	mainplot.draw(addcol=(centcol,), addrow=(centrow,), addsquare=(xcentrange[0], ycentrange[0], xcentrange[1], ycentrange[1]),
			aspect=aspect)

	# save figures
	#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
	#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'_0123456789.fits'), plot_tag, plot_index)
	outpng = '%s%s%02i.png' % (string.rstrip(masterfile,'0123456789.fits'), plot_tag, plot_index)
	pylab.savefig(outpng, dpi=150, orientation='portrait')
	logging.info('plot saved as ' + outpng)
	plot_index = plot_index + 1

	# second figure
	fig.clear()
	fig.text(0.01, 0.99, ins_name + ': ' + plot_type, horizontalalignment='left', verticalalignment='top', fontsize=fs_title)
	fig.text(0.5, 0.99, AB.content['AB_NAME'] + '[' + str(ext) + ']', horizontalalignment='center', verticalalignment='top',
			fontsize=fs_title)
	fig.text(0.99, 0.99, AB.content['DATE'], horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
	fig.text(0.01, 0.96, 'Details of centre', horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
	paramstr = ''
	for param in parameters:
		if param[1] in proHDUs[catg_master][0].header:
			paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])
	fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

	# bottom
	if ref_frame == '':
		fig.text(0.01, 0.03, 'RefFrame: none', horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	else:
		fig.text(0.01, 0.03, 'RefFrame: ' + ref_frame + '[ext=' + str(pro_ext) + ']', 
				horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	fig.text(0.01, 0.01, 'Product: ' + masterfile + '[ext=' + str(ext) + ']', 
			horizontalalignment='left', verticalalignment='bottom', fontsize=fs_foot)
	# figure date
	year, month, day, hour, min, sec = time.localtime()[0:6]
	today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
	fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=8)

	# Plot raw, master, and master - reference (centre)
	logging.info('plot 2: overview frame centres')
	if ref_frame != '':
		mainplot = ImagePlot(image_map=(3,1))
	else:
		mainplot = ImagePlot(image_map=(2,1))
	mainplot.add_image(rawImgs[0], 1, '1st RAW', xrange=xcentrange, yrange=ycentrange, cmap=col_map)
	mainplot.add_image(mstImgs[0], 2, 'MST', xrange=xcentrange, yrange=ycentrange, cmap=col_map)
	if ref_frame != '':
		mainplot.add_image(compImgs[0], 3, 'MST vs REF', xrange=xcentrange, yrange=ycentrange, cmap=col_map)
	mainplot.draw(addcol=(centcol,), addrow=(centrow,), aspect=aspect)

	# save figure
	#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
	#outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'_0123456789.fits'), plot_tag, plot_index)
	outpng = '%s%s%02i.png' % (string.rstrip(masterfile,'0123456789.fits'), plot_tag, plot_index)
	pylab.savefig(outpng, dpi=150, orientation='portrait')
	logging.info('plot saved as ' + outpng)
	plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Visualizes a 2D master product with plots for raw, master, and comparison to reference.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('-e', '--ext', metavar='EXT', dest='ext', type=int, default=0, help='fits extension')
	parser.add_argument('--pro_ext', metavar='PRO_EXT', dest='pro_ext', type=int, default=-1,
			help='fits extension of product, if different from raw frame')
	parser.add_argument('--aspect', metavar='ASPECT', dest='aspect', default='auto',
			help='aspect ratio of plots: equal|auto')

	# parse arguments/options
	args = parser.parse_args()

	# set logging level
	set_logging()

	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')

	# reading fits files
	# get list with HDUs of raw files

	logging.info('reading raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('reading product frames')
	proHDUs = AB.get_pro()

	# draw plots
	display2D(AB, proHDUs, rawHDUs, ext=args.ext, pro_ext=args.pro_ext, aspect=args.aspect)

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

