#!/usr/bin/env python
# PURPOSE:	vimos_4Q.py: displays an overview of all four quadrants
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- August 2009
#		1.0.1	-- bug fix for y centre range (2009-09-01)
#		1.0.2	-- supporting numpy V1.3 and matplotlib V0.99 (2009-10-30)
#		1.0.3	-- improved handling of reference frames (2010-02-12)
#		1.0.4	-- bug fix if one quadrant is missing (2010-09-09)
#		1.0.5	-- remove usage of has_key dictionary attribute (2014-03-12)
#		1.0.6	-- usage of argparse module (2014-05-19)
#
# PARAMETERS:	-a <AB> [ -e <EXT> ] [ --type=<TYPE> ] [ --master=<MASTER> ]
#		[ --area=full|centre ] [ --cuts=<SIGMA>|<LOWER_CUT>,<HIGHER_CUT> ]
#		[ --plot_index=<IDX> ] [ --plottag=<TAG> ] [ --plotsize=a4|a5|a6 ]
#		[ --compress=<COMP> ]
#
# COMMENTS:	The script can be used standalone. It needs qclib.py but does not 
#		require any further configuration files.
#		The script can also be imported into other Python scripts; then,
#		the function display_image() can be called directly.

_version_ = "1.0.6"

# =====================================================================================
# 0. initialization: import modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.
from vimos_lib import *				# VIMOS-specific classes, functions, etc.

# general modules
import string					# string handling
import time					# to get plot date
import logging					# for writing info, warning, and error messages

# =====================================================================================
# 1. function for displaying 2D images of all four quadrants
# =====================================================================================

def display_all(AB, rawHDUs, proHDUs, ext=0, master_catg='', plot_config=[('pro','full')],
		plot_cuts=(1,), plot_tag='G', plot_index=0, imcomp=1, fig_size='a4'):

	# =============================================================================
	# 1.1 import general configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB

	module_name = get_confname(config_files, AB)
	if module_name == '':
		logging.info('configuration file could not be found. Using default.')
		# define some variables
		overscan = 0
		prescan = 0
		if master_catg == '':
			catg_master = proHDUs.keys()[0]
		else:
			catg_master = master_catg
		parameters = []
	else:
		exec 'from ' + module_name + ' import * '
		logging.info('configuration imported from ' + module_name)
		if master_catg != '':
			catg_master = master_catg

	# =============================================================================
	# 1.2 define some variables

	ins_name = os.environ.get('DFO_INSTRUMENT')
	masterfile = AB.content['PROD_ROOT_NAME'] + '_0000.fits'

	# =============================================================================
	# 1.3 create plots

	# create figure
	fs_title, fs_sub, fs_foot = get_fontsizes(fig_size)
	fig = pylab.figure(10, figsize=paper_size(fig_size))

	for pdata in plot_config:
		# find frames from all four quadrants
		if pdata[0] == 'pro' or pdata[0] == 'ref':
			tpl_start = proHDUs[catg_master][0].header['HIERARCH ESO TPL START']
			prod_path = AB.content['PROD_PATH']
			framenames, framedata, framehdrs = get_pro4Q(catg_master, tpl_start, prod_path)
		if pdata[0] == 'ref':
			refnames = []
			for quad in range(1,5):
				if framehdrs[quad-1] == 'empty':
					refnames.append('')
					continue
				ref_keys = ['HIERARCH ESO PRO CATG']
				key_values = [catg_master]
				for key in ref_match_keys:
					if key in framehdrs[quad-1] and key != 'HIERARCH ESO OCS CON QUAD':
						ref_keys.append(key)
						key_values.append(framehdrs[quad-1][key])
				ref_keys.append('HIERARCH ESO OCS CON QUAD')
				key_values.append(quad)

				ref_frame = find_fits(ref_keys, key_values, ref_dir)
				refnames.append(ref_frame)
				if ref_frame != '':
					HDU = pyfits.open(ref_dir+'/'+ref_frame)
					refdata = HDU[ext].data
					HDU.close()
					if comp_method == 'sub':
						comp_image = framedata[quad-1] - refdata
					else:
						comp_image = im_divide(framedata[quad-1], refdata)
					framedata[quad-1] = comp_image
			framenames = refnames
		if pdata[0] == 'raw':
			tpl_start = rawHDUs[0][0].header['HIERARCH ESO TPL START']
			raw_path = os.environ.get('DFO_RAW_DIR') + '/' + AB.content['DATE']
			framenames = []
			framedata = []
			for quad in range(1,5):
				filename = find_fits(['HIERARCH ESO TPL EXPNO', 'HIERARCH ESO TPL START', 'HIERARCH ESO OCS CON QUAD'], [1, tpl_start, quad], raw_path)
				framenames.append(filename)
				if filename == '':
					framedata.append(numpy.zeros((2,2), dtype=numpy.float32))
				else:
					HDU = pyfits.open(raw_path+'/'+filename)
					raw_image = HDU[ext].data
					# chop pre / over scan
					if overscan > 0:
						raw_image = raw_image[:,prescan:-overscan]
					else:
						raw_image = raw_image[:,prescan:]
					framedata.append(raw_image)
					HDU.close()

		# header of figure
		fig.clear()
		plot_type = AB.content['RAW_TYPE']
		fig.text(0.01, 0.99, ins_name + ': ' + plot_type, horizontalalignment='left', 
				verticalalignment='top', fontsize=fs_title)
		fig.text(0.5, 0.99, AB.content['AB_NAME'], 
				horizontalalignment='center', verticalalignment='top', fontsize=fs_title)
		fig.text(0.99, 0.99, AB.content['DATE'], 
				horizontalalignment='right', verticalalignment='top', fontsize=fs_title)
		if pdata[0] == 'pro':
			add_info = catg_master
		elif pdata[0] == 'ref':
			if comp_method == 'sub':
				add_info = catg_master + ' - reference'
			else:
				add_info = catg_master + ' / reference'
		elif pdata[0] == 'raw':
			add_info = rawHDUs[0][0].header['HIERARCH ESO DPR CATG'] + ' ' + \
				   rawHDUs[0][0].header['HIERARCH ESO DPR TYPE'] + ' ' + \
				   rawHDUs[0][0].header['HIERARCH ESO DPR TECH']

		else:
			add_info = ''
		if pdata[1] == 'centre':
			add_info = add_info + ' (centre)'
		paramstr = ''
		if len(parameters) == 0:
			if 'RAW_MATCH_KEY' in AB.content:
				for raw_match_key in AB.content['RAW_MATCH_KEY']:
					if string.split(string.replace(raw_match_key[0], '=', ' '))[0] != 'OCS.CON.QUAD':
						paramstr = paramstr + ' ' + string.split(string.replace(raw_match_key[0], '=', ' '))[-1]
		else:
			for param in parameters:
				if param[1] in proHDUs[catg_master][0].header and param[1] != 'HIERARCH ESO OCS CON QUAD':
					paramstr = paramstr + ' ' + param[0] + ' = ' + str(proHDUs[catg_master][0].header[param[1]])
		if fig_size == 'a6':
			fig.text(0.01, 0.955, add_info, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
			fig.text(0.99, 0.955, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)
		else:
			fig.text(0.01, 0.96, add_info, horizontalalignment='left', verticalalignment='top', fontsize=fs_sub)
			fig.text(0.99, 0.96, paramstr, horizontalalignment='right', verticalalignment='top', fontsize=fs_sub)

		# figure date
		year, month, day, hour, min, sec = time.localtime()[0:6]
		today = str('%04i-%02i-%02i %02i:%02i:%02i' %(year, month, day, hour, min, sec))
		fig.text(0.99, 0.01, 'created ' + today, horizontalalignment='right', verticalalignment='bottom', fontsize=fs_foot)

		logging.info('plotting ' + pdata[0])
		mainplot = ImagePlot(image_map=(2,2))
		idx = 0
		for quad in [2, 1, 3, 4]:
			if pdata[0] == 'ref':
				title_str = 'Q' + str(quad) + ': reference ' + framenames[quad-1]
			else:
				title_str = 'Q' + str(quad) + ': ' + framenames[quad-1]
			idx = idx + 1
			if framenames[quad-1] != '':
				if pdata[1] == 'centre':
					cx = framedata[quad-1].shape[1] // 2
					cy = framedata[quad-1].shape[0] // 2
					xcentrange = (cx - 200, cx + 200)
					if cy < 200:
						ycentrange = (0, framedata[quad-1].shape[0])
					else:
						ycentrange = (cy - 200, cy + 200)
					mainplot.add_image(framedata[quad-1], idx, title_str, cuts=plot_cuts, compress=imcomp,
							xrange=xcentrange, yrange=ycentrange)
				else:
					mainplot.add_image(framedata[quad-1], idx, title_str, cuts=plot_cuts, compress=imcomp)

		if pdata[1] == 'centre':
			mainplot.draw(aspect='equal', figsize=fig_size)
		else:
			mainplot.draw(figsize=fig_size)

		# save figures
		outpng = '%s_%s%02i.png' % (string.rstrip(masterfile,'.fits'), plot_tag, plot_index)
		pylab.savefig(outpng, dpi=150, orientation='portrait')
		logging.info('plot saved as ' + outpng)
		plot_index = plot_index + 1

	return plot_index

# =====================================================================================
# 2. function main(), only called in standalone usage
# =====================================================================================

def main():
	# command line parser
	parser = argparse.ArgumentParser(parents=[basic_parser],
			description='Displays an overview of all four quadrants.')

	# additional options
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.add_argument('--type', metavar='TYPE', dest='type', default='pro',
			help='''files to be plotted; can be pro, ref, raw [default=pro]''')
	parser.add_argument('--master', metavar='MASTER', dest='master', default='',
			help='''PRO CATG of master frame, only needed if TYPE=pro or ref''')
	parser.add_argument('--area', metavar='AREA', dest='area', default='full',
			help='''area of frames to be plotted; can be full or centre [default=full]''')
	parser.add_argument('--cuts', metavar='CUTS', dest='cuts', default='1',
			help='''cut values for plotting, either multiple of sigma (e.g. --cuts=2),
			or lower and upper value (e.g. --cuts=100,200) [default=1]''')
	parser.add_argument('--plottag', metavar='TAG', dest='plot_tag', default='G',
			help='''plot tag used for png output; 
			output is written to <PROD_ROOT_NAME>_OOOO_<TAG><IDX>.png''')
	parser.add_argument('--plotindex', metavar='IDX', dest='plot_index', default='0',
			help='''plot index used for png output; 
			output is written to <PROD_ROOT_NAME>_OOOO_<TAG><IDX>.png''')
	parser.add_argument('--plotsize', metavar='SIZE', dest='plot_size', default='a4',
			help='plot size, either a4, a5, or a6')
	parser.add_argument('--compress', metavar='COMP', dest='compress', default='1',
			help='compression factor for images, useful to save execution time for large images')

	# parse arguments/options
	args = parser.parse_args()

	# convert plot type (that is a char string) into Python list
	plot_config = [ (args.type, args.area) ]

	# convert cuts
	exec 'plot_cuts = [' + str(args.cuts) +']'

	# check compression
	compress = int(args.compress)
	if compress < 1:
		compress = 1
	
	# set logging level
	set_logging()

	logging.info('started')

	# parse AB
	# AB.content : dictionary with string content of AB

	AB = AssociationBlock(args.ab)
	logging.info(args.ab + ' parsed')

	# reading fits files
	# get list with HDUs of raw files

	logging.info('opening raw frames')
	rawHDUs = AB.get_raw()

	# get dictionary with HDUs of all product files

	logging.info('opening product frames')
	proHDUs = AB.get_pro()

	# draw plots
	display_all(AB, rawHDUs, proHDUs, ext=0, master_catg=args.master, 
		plot_config=plot_config, plot_cuts=plot_cuts,
		plot_index=int(args.plot_index), plot_tag=str(args.plot_tag), 
		imcomp=int(args.compress), fig_size=args.plot_size)

	logging.info('finished')

# =====================================================================================
# 3. if standalone call procedure main()
# =====================================================================================

if __name__ == '__main__':
	main()
	sys.exit(0)

