#!/usr/bin/env python
# PURPOSE:	vimos_bias.py: wrapper script for QC checks on VIMOS bias frames
# AUTHOR:	Burkhard Wolff, ESO-DMO
# VERSION:	1.0	-- new layout, copied from vimos_mosflat.py (August 2009)
#		1.0.1	-- supporting numpy V1.3 and matplotlib V0.99 (2009-10-30)
#		1.0.2	-- create cover sheet (2011-05-20)
#		1.0.3	-- skip call of scoreQC (2011-05-31)
#		1.0.4	-- changes needed for muc (2013-02-07)
#		1.0.5	-- changes for qc_mstplt v1.4 (2014-01-15)
#		1.0.6	-- changes for qc_mstdisp v1.3 (2014-01-20)
#		1.0.7	-- usage of argparse module (2014-03-31)
#		1.0.8	-- bug fix in call of qc_mstdisp (2014-05-09)
#
# PARAMETERS:	-a <AB>
#
# OPTIONS:	-i	ingest QC1 parameters [optional]
#		--all	create all plots, i.e. also 2D overview for individual quadrant
#
# COMMENTS:
#
# DOCU_NAME vimos_bias.py
#
# DOCU_VERSION 1.0   -- edited from vimos_mosflat.py <br>
# DOCU_VERSION 1.0.1 -- supporting numpy V1.3 and matplotlib V0.99 (2009-10-30)<br>
# DOCU_VERSION 1.0.2 -- create cover sheet (2011-05-20)<br>
# DOCU_VERSION 1.0.3 -- skip call of scoreQC (2011-05-31)<br>
# DOCU_VERSION 1.0.4 -- changes needed for muc (2013-02-07)<br>
# DOCU_VERSION 1.0.5 -- changes for qc_mstplt v1.4 (2014-01-15)<br>
# DOCU_VERSION 1.0.6 -- changes for qc_mstdisp v1.3 (2014-01-20)<br>
# DOCU_VERSION 1.0.7 -- usage of argparse module (2014-03-31)<br>
# DOCU_VERSION 1.0.8 -- bug fix in call of qc_mstdisp (2014-05-09)
#
# DOCU_SYNTAX PYTHON
#
# DOCU_RAWTYPE BIAS
#
# DOCU_CALL processQC <br>
# DOCU_CALL from $DFS_PRODUCT/BIAS/$DATE <br>
# DOCU_CALL /home/vimos/python/vimos_bias.py -i -a $AB <br>
#
# DOCU_PURPOSE QC check on raw and master bias frames
#
# DOCU_PROCINPUT raw bias frames and MASTER_BIAS product
#
# DOCU_QC1TABLE vimos_bias
#
# DOCU_QC1PLOTS NAME bias0.png
# DOCU_QC1PLOTS DESCR bias0.png <b>upper left panel:</b> cuts through first raw frame and master bias in X direction<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>upper middle panel:</b> cuts through first raw frame and master bias in Y direction<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>upper right panel:</b> histrograms of first raw frame and master, Gauss fit to master<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>central left panel:</b> cuts through first raw frame and master in X direction, central area<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>central middle panel:</b> cuts through first raw frame and master in Y direction, central area<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>central right panel:</b> row and column averages of master<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>lower left panel:</b> comparison to reference (current master - reference), cut in X direction<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>lower middle panel:</b> comparison to reference (current master - reference), cut in Y direction<br>
# DOCU_QC1PLOTS DESCR bias0.png <b>lower right panel:</b> comparison to reference, histogram<br>
#
# DOCU_QC1PLOTS NAME bias1.png
# DOCU_QC1PLOTS DESCR bias1.png current master bias - reference master, for all four quadrants<br>
# DOCU_QC1PLOTS DESCR bias1.png (only created together with plot for quadrant 1)
#
# DOCU_QC1PLOTS NAME bias2.png
# DOCU_QC1PLOTS DESCR bias2.png current master bias frames from all four quadrants<br>
# DOCU_QC1PLOTS DESCR bias2.png (only created together with plot for quadrant 1)
#
# DOCU_QC1PLOTS NAME bias3.png
# DOCU_QC1PLOTS DESCR bias3.png central area of first raw bias, for all four quadrants<br>
# DOCU_QC1PLOTS DESCR bias3.png (only created together with plot for quadrant 1)
#
# DOCU_QC1PARAM QC1 parameters written in QC1 database: <br>
# DOCU_QC1PARAM <b>QC1 DB names:</b> median_master | mean_master | rms_master | fpn_master | struc_master | noise_master |
# DOCU_QC1PARAM median_raw | mean_raw | ron_bias | rms_raw | fpn_raw | struc_raw <br>
# DOCU_QC1PARAM All parameters are calculated by the pipeline recipe.
#
# DOCU_ALGORITHM Description of algorithms: <br>
# DOCU_ALGORITHM <b>median_master:</b> median of the 1600x1800 central pixels of master frame <br>
# DOCU_ALGORITHM <b>mean_master:</b> mean of the 1600x1800 central pixels of master frame <br>
# DOCU_ALGORITHM <b>rms_master:</b> standard deviation of the 1600x1800 central pixels of master frame <br>
# DOCU_ALGORITHM <b>fpn_master:</b> fixed pattern noise of master; shift master by 10x10 pixels and subtract,
# DOCU_ALGORITHM calculate standard deviation of 1600x1800 central pixels of difference, subtract noise_master quadratically <br>
# DOCU_ALGORITHM <b>struc_master:</b> subtract quadratically fpn_master and noise_master from rms_master <br>
# DOCU_ALGORITHM <b>noise_master:</b> standard deviation of 1600x1800 central pixels but excluding pixels deviating by
# DOCU_ALGORITHM more than 3 * ron_bias / sqrt(DATANCOM) <br>
# DOCU_ALGORITHM <b>median_raw:</b> median of the 1600x1800 central pixels of first raw frame <br>
# DOCU_ALGORITHM <b>mean_raw:</b> mean of the 1600x1800 central pixels of first raw frame <br>
# DOCU_ALGORITHM <b>ron_bias:</b> standard deviation of the 1600x1800 central pixels of the difference of the first two raw frames,
# DOCU_ALGORITHM divided by sqrt(2) <br>
# DOCU_ALGORITHM <b>rms_raw:</b> standard deviation of the 1600x1800 central pixels of the first raw frame <br>
# DOCU_ALGORITHM <b>fpn_raw:</b> standard deviation of the 1600x1800 central pixels of the difference of the first and second
# DOCU_ALGORITHM raw bias, shifted by 10x10 pixels; ron_bias is quadratically subtracted <br>
# DOCU_ALGORITHM <b>struc_raw:</b> subtract quadratically fpn_raw and ron_bias from rms_raw <br>
#
# DOCU_CERTIF Reasons for rejection:<br>
# DOCU_CERTIF unusual high bias level or read-out noise, change of structure, lower DATANCOM value
#
# DOCU_COMMENTS MIDAS script has been replaced by Python script (2009-08-07)

_version_ = "1.0.8"

# =====================================================================================
# 0. initialization
# 0.1 general modules
# =====================================================================================

# QC configuration and library
from config_cf import *				# names of configuration files
from qclib import *				# qclib classes, functions, etc.
from vimos_lib import *				# VIMOS-specific classes, functions, etc.

# =====================================================================================
# 0.2 parse options and read AB
# =====================================================================================

# command line parser
# some general options are taken from the parent wrapper_parser which is defined in qclib
parser = argparse.ArgumentParser(parents=[wrapper_parser], description='Wrapper script for QC plots on VIMOS bias frames.')

# additional options, specific to this script
parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
parser.add_argument('--all', dest='all', action='store_true', default=False,
		help='''plot 2D overview of individual frames''')

# parse arguments/options
args = parser.parse_args()

# args now contains the command line options:
# args.ab = <AB_NAME>
# args.ingest = False | True
# args.all = False | True

# set logging level
set_logging()

logging.info('started')

# parse AB
AB = AssociationBlock(args.ab)
logging.info(args.ab + ' parsed')

# AB has now the content of the AB, e.g
# instrument: AB.instrument
# civil date: AB.date
# This is equivalent to AB.content['INSTRUMENT'] and AB.content['DATE'].

# =====================================================================================
# 0.3 import script-specific modules and configuration
# =====================================================================================

# specific plotting routines
logging.info('importing qc_mstplt')
import qc_mstplt				# cuts of raw/master frames, histograms
logging.info('importing qc_mstdisp')
import qc_mstdisp				# overview plots (2D)
logging.info('importing vimos_4Q')
import vimos_4Q					# overview of all four quadrants

# import configuration dependend on RAW_TYPE and RAW_MATCH_KEYs in AB

module_name = get_confname(config_files, AB)
if module_name == '':
	logging.error('configuration file could not be found. Exit.')
	sys.exit(1)
else:
	exec 'from ' + module_name + ' import * '
	logging.info('configuration imported from ' + module_name)

# =====================================================================================
# 0.4 reading fits files
# =====================================================================================

# get list with HDUs of raw files

logging.info('reading raw frames')
rawHDUs = AB.get_raw()

# get dictionary with HDUs of all product files

logging.info('reading product frames')
proHDUs = AB.get_pro()

# =====================================================================================
# 1. main part: create plots
# =====================================================================================

# cuts through raw and products frames, histograms
logging.info('starting qc_mstplt')
plot_index = qc_mstplt.draw_plots(AB, rawHDUs, proHDUs, plot_tag='0000_A', plot_index=0, fig_size='a4')

# overview plots (2D images)
if args.all:
	logging.info('starting vimos_mstdisp')
	plot_index = qc_mstdisp.display2D(AB, proHDUs, rawHDUs, ext=0, aspect='equal', imcomp=2, 
			plot_tag='0000_A', plot_index=plot_index)

# overview of all four quadrants, if quad = 1
if rawHDUs[0][0].header['HIERARCH ESO OCS CON QUAD'] == 1:
	logging.info('starting vimos_4Q')
	plot_index = vimos_4Q.display_all(AB, rawHDUs, proHDUs, plot_config=[('ref','full'), ('pro','full')],
			plot_cuts=(1,), plot_index=0, plot_tag='G', imcomp=2, fig_size='a5')
	plot_index = vimos_4Q.display_all(AB, rawHDUs, proHDUs, plot_config=[('raw','centre')],
			plot_index=plot_index, plot_tag='G', imcomp=1, fig_size='a5')

# =====================================================================================
# 2. post-processing
# =====================================================================================

logging.info('creating cover sheet')
# This is a VIMOS-specific function, defined in vimos_lib.
make_cover(AB, det_tags=[('A00','cuts and histograms')], 
		gen_tags=[('G00','mst vs. ref'),('G01','mst (full view)'),('G02','mst (centre)')])

# calculating additional QC1 parameters
# NONE

# store into QC1 database
if not args.ingest:
	logging.info('QC1 parameters not ingested')
else:
	logging.info('ingesting QC1 parameters')

	civil_date = AB.date
	origf = rawHDUs[0][0].header['ORIGFILE']
	detncom = 1

	quad = rawHDUs[0][0].header['HIERARCH ESO OCS CON QUAD']
	temp = rawHDUs[0][0].header['HIERARCH ESO INS FOCU'+str(quad)+' TEMP']

	# ingest QC1 parameters using writeQC
	tmp_dir = os.path.expandvars('$TMP_DIR')
	for ext in allow_ext:
		os.system('writeQC -m PREPARE -c config.writeQC_python -a ' + AB.ab_name + ' -e ' + str(ext))
		execfile(tmp_dir + '/' + AB.ab_name + '_' + str(ext) + '/writeQC.py')
		os.system('writeQC -m INGEST -c config.writeQC_python -a ' + AB.ab_name + ' -e ' + str(ext))

logging.info('finished')

sys.exit(0)

