#!/usr/bin/env python
# NAME:		vimos_lib.py
# PURPOSE:	providing VIMOS-specific classes and functions used for QC scripts
# AUTHOR:       Burkhard Wolff, ESO-DMO
# VERSIONS:     1.0	-- December 2009 / February 2010
#		1.1	-- adding make_cover (2011-05-20/26)
#		1.1.1	-- adding tpl_expno in get_pro4Q (2011-06-15)
#		1.2	-- adding overexp (2012-01-20)
#		1.2.1	-- bug fix in overexp: replace greater operator with greater_equal (2012-08-30)
#		1.2.2	-- function overexp removed, now in qclib (2013-04-29)
#		1.2.3	-- make_cover: fixed font size for links to QC plots (2014-03-26)
#		1.2.4	-- function get_wavearray removed, now in qclib (204-03-27)
#
# CONTENTS:
#
#		def plot_lines():	plots a linelist over a spectrum
#		def get_pro4Q():	finds products of specified pro_catg from all 4 quadrants
#		def make_cover():	creates a cover sheet for QC plots
#

_version_ = "1.2.4"

# =====================================================================================
# 0. import modules
# =====================================================================================

from qclib import *			# qclib classes, functions, etc.
import string				# string handling

# =====================================================================================
# 1. classes, functions
# =====================================================================================

def plot_lines(waverange, yrange, linelist=[], linelabels=[], linecol='r'):
	"plots a linelist over a spectrum"

	xmin = waverange[0]
	xmax = waverange[1]
	ymin = yrange[0]
	ymax = yrange[1]
	ydelta = 0.06 * (ymax-ymin)
	yoff = 0.0

	for idx in range(len(linelist)):
		line = linelist[idx]
		if line >= xmin and line <= xmax:
			pylab.plot([line, line], [ymin, ymax], color=linecol, linewidth=0.75)
			if idx < len(linelabels):
				pylab.text(xmin+0.05*(xmax-xmin), ymin+0.025*(ymax-ymin)+yoff,
						'%s %7.2f' % (linelabels[idx], line), size=8, color='k')
				yoff = yoff + ydelta
	pylab.xlim(xmin, xmax)
	pylab.ylim(ymin, ymax)

	return

def get_pro4Q(pro_catg, tpl_start, prod_path, tpl_expno=0, ext=0, datadefault=numpy.zeros((2,2), dtype=numpy.float32)):
	"""finds products of specified pro_catg from all 4 quadrants;
	returns file names, data for specified extension, headers"""

	framenames, framedata, framehdrs = [], [], []
	for quad in range(1,5):
		if tpl_expno == 0:
			filename = find_fits(['HIERARCH ESO PRO CATG', 'HIERARCH ESO TPL START', 'HIERARCH ESO OCS CON QUAD'],
				[pro_catg, tpl_start, quad], prod_path)
		else:
			filename = find_fits(['HIERARCH ESO PRO CATG', 'HIERARCH ESO TPL START', 'HIERARCH ESO OCS CON QUAD',
				'HIERARCH ESO TPL EXPNO'], [pro_catg, tpl_start, quad, tpl_expno], prod_path)
		framenames.append(filename)
		if filename == '':
			framedata.append(datadefault)
			framehdrs.append('empty')
		else:
			HDU = pyfits.open(prod_path+'/'+filename)
			framedata.append(HDU[ext].data)
			framehdrs.append(HDU[ext].header)
			HDU.close()

	return framenames, framedata, framehdrs

def make_cover(AB, det_tags = [('A00','One quadrant (plot1)'),('A01','One quadrant (plot1)'),('A02','One quadrant (plot2)')],
		add1_tags = [], add2_tags = [],
		gen_tags = [('G00','All: plot 1'),('G01','All: plot 2'),('G02','All: plot 3')]):
	"create a cover sheet html page for all QC plots associated to AB"

	# find plot files associated to AB
	prod_root = AB.content['PROD_ROOT_NAME']
	dir = AB.content['PROD_PATH']
	all_files = os.listdir(dir)
	plot_files = []
	for file in all_files:
		if prod_root in file and file[-3:] == 'png':
			plot_files.append(file)

	# find plots for individual detectors
	det_plots = []
	for item in det_tags:
		for file in plot_files:
			if item[0] in file:
				det_plots.append(file)
				break
		else:
			det_plots.append('NONE')
	
	# find additional plots (which go into separate lines below detector plots)
	add1_plots = []
	for item in add1_tags:
		for file in plot_files:
			if item[0] in file:
				add1_plots.append(file)
				break
		else:
			add1_plots.append('NONE')
	add2_plots = []
	for item in add2_tags:
		for file in plot_files:
			if item[0] in file:
				add2_plots.append(file)
				break
		else:
			add2_plots.append('NONE')
	
	# find general (overview) plots
	gen_plots = []
	for item in gen_tags:
		for file in plot_files:
			if item[0] in file:
				gen_plots.append(file)
				break
		else:
			gen_plots.append('NONE')
	for file in gen_plots:
		if file != 'NONE':
			break
	else:
		gen_plots = []
	
	# output html page
	setup0 = ''
	for item in AB.content['RAW_MATCH_KEY']:
		setup0 = setup0 + '_'+item[0].replace('=', ' ').split()[1]
	setup = setup0.lstrip('_')

	cs_name = '%s_cs.html' % string.rstrip(AB.content['AB_NAME'], '.ab')
	cs = open(cs_name, 'w')
	cs.write('<!DOCTYPE /HTML PUBLIC "-.W3C.DTD /HTML 3.2.EN">\n')
	cs.write('<html><head>\n')
	cs.write('<meta http-equiv="content-type" content="text/html; charset=ISO-8859-1">\n')
	cs.write('<title>Coversheet for '+AB.content['AB_NAME']+'</title>\n')
	cs.write('</head>\n')
	cs.write('<body>\n')
	cs.write('<table width="500" border="0">\n')
	cs.write('<tr bgcolor="#CCCCCC">\n')
	cs.write('<td>\n')
	cs.write('<b><font color=#006A9D size=+1>Coversheet for QC plots</font></b>\n')
	cs.write('<pre>\n')
	cs.write('<b>'+AB.content['AB_NAME']+'</b>\n')
	cs.write('RAW_TYPE:    '+AB.content['RAW_TYPE']+'\n')
	cs.write('setup:       '+setup+'\n')
	cs.write('</pre>\n')

	cs.write('<font size=2>\n')
	cs.write('<b>This quadrant</b><br><!--SCORE_REPORT-->\n')
	if det_tags == []:
		cs.write('no plots<br><!--SCORE_REPORT-->\n')
	else:
		cnt = 0
		for idx in range(len(det_tags)):
			if det_plots[idx] != 'NONE':
				cnt = cnt + 1
				if cnt == 1:
					cs.write('<a href="'+dir+'/'+det_plots[idx]+'">'+det_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
				else:
					cs.write('| <a href="'+dir+'/'+det_plots[idx]+'">'+det_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
		cs.write('<br><!--SCORE_REPORT-->\n')

	if len(add1_tags) > 0:
		cnt = 0
		for idx in range(len(add1_tags)):
			if add1_plots[idx] != 'NONE':
				cnt = cnt + 1
				if cnt == 1:
					cs.write('<a href="'+dir+'/'+add1_plots[idx]+'">'+add1_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
				else:
					cs.write('| <a href="'+dir+'/'+add1_plots[idx]+'">'+add1_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
		cs.write('<br><!--SCORE_REPORT-->\n')

	if len(add2_tags) > 0:
		cnt = 0
		for idx in range(len(add2_tags)):
			if add2_plots[idx] != 'NONE':
				cnt = cnt + 1
				if cnt == 1:
					cs.write('<a href="'+dir+'/'+add2_plots[idx]+'">'+add2_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
				else:
					cs.write('| <a href="'+dir+'/'+add2_plots[idx]+'">'+add2_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
		cs.write('<br><!--SCORE_REPORT-->\n')

	if len(gen_plots) > 0:
		cs.write('<b>Overview plots</b><br><!--SCORE_REPORT-->\n')
		cnt = 0
		for idx in range(len(gen_tags)):
			if gen_plots[idx] != 'NONE':
				cnt = cnt + 1
				if cnt == 1:
					cs.write('<a href="'+dir+'/'+gen_plots[idx]+'">'+gen_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
				else:
					cs.write('| <a href="'+dir+'/'+gen_plots[idx]+'">'+gen_tags[idx][1]+'</a><!--SCORE_REPORT-->\n')
			idx = idx + 1
		cs.write('<br><!--SCORE_REPORT-->\n')

	cs.write('</font></td></tr></table>\n')
	cs.write('</body></html>\n')
	cs.close()

	return

# =====================================================================================
# 2. handle standalone call
# =====================================================================================

if __name__ == '__main__':
	parser = argparse.ArgumentParser(description=
			'Module with definitions, functions, and classes for VIMOS QC scripts. No standalone usage.')
	parser.add_argument('--version', action='version', version='%(prog)s ' + _version_)
	parser.parse_args()
	set_logging(level='info')
	logging.info('Module with definitions, functions, and classes for VIMOS QC scripts. No standalone usage. Exit.')
	sys.exit(0)

