   ; CENTROIDER: sub-pixel centering of an image by iterative
   ; fractional shift. The fractional shift is performed by IMAGE_SHIFT.

   ; ALGORITHM DESCRIPTION: the image is centered by an iterative shift
   ; aimed at minimizing the distance of the image centroid from a
   ; pre-fixed pixel position (adapted from J. Christou). The centroid
   ; is computed over a small box centered on specified pixel.

   ; INPUT
   ;	image: image to be centered
   ;	[XC = , YC = ]: position where the centroiding box must be centered
   ;		(default = image maximum)
   ;	[CENTROID_BOX = ]: scalar or 2-elements vector specifying the size
   ;		of the box to compute the image centroid
   ;		(default = 3)
   ;		If box is a scalar, a square box is assumed.
   ;	[CENTROID_TOL = ]: the iterative shift is stopped when the centroid
   ;		off-centering is <= tol (default 0.05 pixels)
   ;	[CENTROID_IT = ]: maximum number of iterations (default = 20);
   ;		if maxit is exceeded, the input image is returned
   ;	[_EXTRA = ]: see  IMAGE_SHIFT
   ; OUTPUT
   ;	centered image
   ;	[XSHIFT = , YSHIFT = ]: overall x- and y- shifts to center the image

   ; NOTE: This algorithm is well suited to center an image with sub-pixel
   ;	accuracy around a local maximum. It may fail if the local maximum
   ;	is too near a brighter one: in this case the latter may be centered
   ;	instead. Before being shifted, the image is extended along each
   ;	direction to avoid edge effects. If the initial position of the
   ;	centroiding box is located too far away from the local maximum
   ;	being centered, edge effects may occur anyway.


   FUNCTION centroider, image, XC = xc, YC = yc, CENTROID_BOX = box, $
   						CENTROID_TOL = tol, CENTROID_IT = maxit,     $
   						_EXTRA = extra, XSHIFT = xs, YSHIFT = ys

	if  mysize( image, /N_DIM ) ne 2  then  return, image
	s = mysize( image, /DIM )
	if  n_elements( xc ) * n_elements( yc ) eq 0  then begin
	   m = get_max( image )  &  x = m[0]  &  y = m[1]
	endif else begin
	   x = xc  &  y = yc
	endelse
	x = round( x )  &  y = round( y )
	if  n_elements( box ) eq 0  then $
	   b = fwhm(image, X = x, Y = y)  else  b = box
	b = round( b )
	if  n_elements( b ) eq 1  then  b = [b, b]
	b = b < s  &  b = b + 1 - b mod 2  &  b = b > 3
	if  n_elements( tol )   eq 0  then  tol = 0.05
	if  n_elements( maxit ) eq 0  then  maxit = 20
	i = image  &  it = 0  &  xs = 0  &  ys = 0
	repeat begin
	   it = it + 1
	   c = centroid( extract( i, b[0], b[1], X = x, Y = y, $
	   						  LO_X = lx, LO_Y = ly ) )
	   dx = x - lx - c[0]  &  dy = y - ly - c[1]
	   xs = xs + dx   &  ys = ys + dy
	   i = image_shift( i, xs, ys, _EXTRA = extra, /ANY_SHIFT, $
	   					AUX_DATA = aux_data )
	   convergence = abs( dx ) lt tol and abs( dy ) lt tol
	endrep until  convergence or it eq maxit
	if  it eq maxit and not convergence  then begin
	   i = image  &  xs = 0  &  ys = 0
	endif
	return, i
   end
