   ; FINDSTARS: given a stellar field, (try to) detect stars and estimate
   ; their positions and fluxes.

   ; ALGORITHM DESCRIPTION (the essential parameters of the algorithm are
   ; in CAPITAL letters in the following description)

   ; GENERAL DESCRIPTION
   ; The algorithm is iterative. One step consists of 3 phases:
   ;	1) detection of presumed stars
   ;	2) check and analysis of detected objects, sorted by decreasing
   ;	   intensity
   ;	3) re_fitting
   ; Pixel (x,y) is the approximate center of a 'presumed star' if
   ; a) IMAGE(x,y) is a local maximum
   ; b) IMAGE(x,y) > BACKGROUND(x,y) + IMAGE_MODEL(x,y) + MIN_INTENSITY,
   ; where IMAGE is the stellar field, BACKGROUND is an approximation of
   ; the background emission, IMAGE_MODEL is a sum of shifted scaled
   ; replicas of the PSF (one for each detected star) and MIN_INTENSITY
   ; is the minimum central intensity of a detectable star. In practice
   ; the user passes a set of MIN_INTENSITY levels (generally decreasing!)
   ; and the number of these levels fixes the number of times the basic
   ; step (Phase 1 + 2) is repeated. At the first iteration, IMAGE_MODEL
   ; is identically zero, but in later iterations it contains all the stars
   ; which have been detected. Subtraction of these stars allows detection
   ; of fainter objects and a better estimation of the image BACKGROUND.
   ; It should be noticed that stars (IMAGE_MODEL) are subtracted only in
   ; order to detected new presumed stars and refine the background: all
   ; the subsequent astrometric and photometric analysis is performed on
   ; the original IMAGE.

   ; DETAILED DESCRIPTION OF PHASE 2
   ; After the list of presumed stars is created, each object is analyzed
   ; as follows (phase 2):
   ; a) Re-identification
   ; b) Classification as
   ;	1) single object
   ;	2) blend
   ; If the object is single (b.1), we have
   ; c1) correlation with the PSF.
   ; c2) local fitting
   ; If the object is a blend (b.2), deblending is performed as follows:
   ;			.....

   ; If the presumed object passes all the check and operations described
   ; above, it is accepted as a star, its parameters are saved and the
   ; IMAGE_MODEL is updated

   ; CALLING SEQUENCE (essential parameters):
   ; findstars, image, psf, [BACKGROUND = ], [/NO_SLANT],	$
   ;			min_intensity, min_correlation,				$
   ;			[X_BAD = , Y_BAD = ], [SV_PAR = ],			$
   ;			[FILE_NAME = ],	[_EXTRA = extra],			$
   ;			x_pos, y_pos, fluxes, image_model, background





; FINDSTARS auxiliary modules to handle the data structure
; representing the list of stars.

   ; STAR: define a structure containing information on a single star.

   FUNCTION star

	return, { star, x: 0., y: 0., flux: 0., is_a_star: 0B }
   end

   ; CREATE_LIST: create an array of n structured elements 'star'.

   FUNCTION create_list, n

	element = star()
	return, replicate( element, n )
   end

   ; UPDATE_LIST: update the elements of list subscripted by s.
   ; Set the flag is_a_star to 'true' if  /IS_A_STAR is set.

   FUNCTION update_list, list, SUBSCRIPTS = s, x, y, f, IS_STAR = is_star

	l = list  &  n = n_elements( s )
	if  n eq 0  then  s = lindgen( n_elements( x ) )	; update all
	if  s[0] ge 0  then begin
	   l[s].x = x  &  l[s].y = y
	   if  n_elements( f ) ne 0  then  l[s].flux = f
	   l[s].is_a_star = keyword_set( is_star )
	endif
	return, l
   end


   ; CREATE_ELEMENT: create and initialize a new element.

   FUNCTION create_element, x, y, f

	element = star()
	return, update_list( element, x, y, f )
   end

   ; MERGE_LIST: merge two lists.

   FUNCTION merge_list, l1, l2

	n1 = n_elements( l1 )  &  n2 = n_elements( l2 )
	l = create_list( n1 + n2 )
	if  n1 ne 0  then  l[0] = l1
	l[n1] = l2
	return, l
   end

    ; ADD_ELEMENT: add an element to list.

   FUNCTION add_element, list, element

	return, merge_list( list, element )
   end

  ; ADD_SUBSCRIPT: add subscript s to vector of subscripts.

   FUNCTION add_subscript, subscripts, s

	w = add_elements( subscripts, s )
	if  w[0] lt 0  then  w = s
	return, w
   end

   ; DELETE_ELEMENT: delete last element of list.

   FUNCTION delete_element, list

	l = list  &  n = n_elements( l )
	return, l[0:n-2]
   end

   ; EXTRACT_ELEMENTS: extract from list the elements subscripted by s.

   PRO extract_elements, list, SUBSCRIPTS = s, n, x, y, f

	n = n_elements( s )
	if  n eq 0  then begin
	   n = n_elements( list )  &  s = lindgen( n )	; extract all elements
	endif
	if  s[0] lt 0  then  n = 0  else begin $
	   x = list[s].x  &  y = list[s].y  &  f = list[s].flux
	endelse
	return
   end


   ; WHERE_STARS: return subscripts of elements having the field
   ; is_a_star = true and lying into the specified range of coordinates.

   FUNCTION where_stars, list, LO_X = lx, UP_X = ux, $
   						 LO_Y = ly, UP_Y = uy, n

	flag = list.is_a_star
	if  n_elements( lx ) ne 0  then $
	   flag = flag and list.x ge lx and list.x le ux $
	   			   and list.y ge ly and list.y le uy
	return, where( flag and 1B, n )
   end

   ; SUB_LIST: extract list[s].

   FUNCTION sub_list, list, s

	l = list
	return, l[s]
   end

   ; EXTRACT_STARS: extract from list the sub-list of elements
   ; fulfilling the condition is_a_star = 'true'.

   FUNCTION extract_stars, list, n

    s = where_stars( list, n )
    if  n ne 0  then  return, sub_list( list, s )  else  return, -1
   end

   ; PRINT_STARS: print positions and fluxes of currently detected stars.

   PRO print_stars, list, file_name

	extract_elements, list, SUBSCRIPTS = where_stars( list ), $
					  n_stars, x, y, f
	get_lun, unit  &  openw, unit, file_name
	for  n = 0, n_stars - 1  do  printf, unit, x[n], y[n], f[n]
	close, unit
	return
   end

   ; SORT_LIST: sort list of objects in order of decreasing intensity.

   FUNCTION sort_list, list, SUBSCRIPTS = s

	s = reverse( sort( list.flux ) )  &  l = list
	return, l[s]
   end

   ; REVERSE_CLASS: reverse the is_a_star field of the elements
   ; of list subscripted by s.

   FUNCTION reverse_class, list, SUBSCRIPTS = s

	if  n_elements( s ) eq 0  then  s = lindgen( n_elements( list ) )
	l = list
	l[s].is_a_star = ( not l[s].is_a_star ) and 1B
	return, l
   end


; Other FINDSTARS auxiliary modules.

   ; DISPLAY_SIZE: compute x- and y- size for IDL CONGRID when
   ; displaying an image.

   FUNCTION display_size, image, presumed_size, available_size

	return, round(  float( mysize( image, /DIM ) ) / presumed_size * $
					available_size )
   end

   ; DISPLAY: graphic output.

   PRO display, win_par, image, position, boxsize, x, y

	tvscl, congrid( win_par.empty_display, $
					win_par.box_size, win_par.box_size ), position
	if  n_elements( x ) * n_elements( y ) * n_elements( boxsize ) ne 0 $
	then begin
	   box = extract( image, X = x, Y = y, boxsize, boxsize )
	   s = display_size( box, boxsize, win_par.box_size )
	endif else begin
	   box = image  &  s = win_par.full_size
	endelse
	tvscl, congrid( box, s[0], s[1] ), position
	return
   end

   ; TEXT: text output.

   PRO text, x, y

	print, ''
	print, 'examining object in  (', x, y, ' )', FORMAT = '(A, I5, I5, A)'
	return
   end

   ; BAD_HERE: find bad pixels in the sub-image [lx:ux,ly:uy]

   PRO bad_here, bad_pix, lx, ux, ly, uy, x_bad, y_bad

	w = where( bad_pix.x ge lx and bad_pix.x le ux and $
			   bad_pix.y ge ly and bad_pix.y le uy, n )
	if  n ne 0  then begin
	   x_bad = bad_pix.x[w] - lx  &  y_bad = bad_pix.y[w] - ly
	endif
	return
   end

   ; BOXES: extract boxes from image, image model and background.

   PRO boxes, x, y, boxsize, image, image_model, background, lx, ux, ly, uy, $
   			  ima_box, mod_box, bac_box, bad_pix, x_bad, y_bad

	ima_box = extract( image, boxsize, boxsize, X = x, Y = y, $
					   LO_X = lx, UP_X = ux, LO_Y = ly, UP_Y = uy )
	mod_box = image_model[lx:ux,ly:uy]
	bac_box = background[lx:ux,ly:uy]
	if  n_elements( bad_pix ) ne 0  then $
	   bad_here, bad_pix, lx, ux, ly, uy, x_bad, y_bad
	return
   end

   ; LOCAL_PSF: define a local PSF at (x,y), if it is space-variant.

   FUNCTION local_psf, list_of_stars, this_star, _EXTRA = extra, $
   					   ref_psf, sv_par

	extract_elements, list_of_stars, SUB = this_star, num, x, y
	l = sv_par.l  &  w = sv_par.w
	l1 = l[0]  &  if  n_elements( l ) eq 2  then  l2 = l[1]
	w1 = w[0]  &  if  n_elements( w ) eq 2  then  w2 = w[1]
	svpsf = space_variant( ref_psf, sv_par.x0, sv_par.y0, x, y, $
	                       /NO_EXTEND, L1 = l1, L2 = l2, W1 = w1, W2 = w2 )
	return, svpsf
   end

   ; IDENTIFY: re-identification of a presumed star.

   PRO identify, par, image, image_model, background, found, x, y

	boxes, x, y, par.fitting_box, image, image_model, background, $
	   	   lx, ux, ly, uy, ima_box, mod_box, bac_box
	search_max, ima_box - mod_box - bac_box, MINIFICATION = par.minif, $
				par.min_intensity, X0 = x - lx, Y0 = y - ly, /NEAREST, $
				n, x, y
	found = n ne 0
	if  found  then begin
	   x = x + lx  &  y = y + ly
	endif
	return
   end

   ; IS_A_BLEND: check the area of a suspected star to identify blends.

   FUNCTION is_a_blend, par, image, image_model, background, x, y

	boxes, x, y, par.correlation_box, image, image_model, background, $
	   	   lx, ux, ly, uy, ima_box, mod_box, bac_box
	object_area = image_area( ima_box - mod_box - bac_box, $
							  X = x - lx, Y = y - ly,	   $
							  REL_THRESHOLD = par.area_threshold )
	return, object_area gt par.area_tol * par.psf_area
   end

   ; CORRELATE: correlation check.

   PRO correlate, par, image, image_model, background, bad_pix, $
   				  template, _EXTRA = extra, accepted, x, y

	sub_pix_correl = 0.6
	boxsize = par.correlation_box + par.search_box + 1
	boxes, x, y, boxsize, image, image_model, background, lx, ux, ly, uy, $
		   ima_box, mod_box, bac_box, bad_pix, x_bad, y_bad
	sub_box = ima_box - mod_box - bac_box
	max_correlation, sub_box, template, x - lx, y - ly,		   	   $
					 par.correlation_box, par.correlation_box,	   $
					 par.search_box, X_BAD = x_bad, Y_BAD = y_bad, $
					 x, y, correl, _EXTRA = extra, SUB_PIX = 	   $
					 ( par.min_correlation gt sub_pix_correl ) and 1B
	x = x + lx  &  y = y + ly
	accepted = correl ge par.min_correlation and $
			   x ge lx and x le ux and y ge ly and y le uy
	return
   end

   ; CHECK_RESULTS: check outcome of local fitting.

   FUNCTION check_results, list, par, lx, ux, ly, uy, $
   						   x_fit, y_fit, f_fit, fit_error

	; Check convergence of fit, minimum acceptable value of fluxes,
	; range of positions
	check = fit_error ge 0 and min( f_fit ) ge par.min_flux  and $
			min( x_fit ) ge lx  and  max( x_fit ) le ux  and $
			min( y_fit ) ge ly  and  max( y_fit ) le uy
	if  check  then begin
	; Consider the subset of stars in a neighborhood of [lx:ux,ly:uy]
	; and check their reciprocal distances
	s = where_stars( LO_X = lx - par.min_distance, $
					 UP_X = ux + par.min_distance, $
					 LO_Y = ly - par.min_distance, $
					 UP_Y = uy + par.min_distance, list )
	extract_elements, list, SUBSCRIPTS = s, n, x, y, f
	if  n gt 1  then $
	   check = min( reciprocal_distance( x, y ) ) ge par.min_distance
	endif
	return, check
   end

   ; FIT_BOX: local fitting.

   PRO fit_box, list, this_max, par, image, image_model, background, $
   				bad_pix, psf, fitting_psf, NO_SLANT = no_slant,		 $
   				_EXTRA = extra, star_here, RE_FITTING = re_fitting

	; Extract boxes
	extract_elements, list, SUBSCRIPTS = this_max, n, x, y
	boxes, x, y, par.fitting_box, image, image_model, background, $
		   lx, ux, ly, uy, ima_box, mod_box, bac_box, bad_pix, x_bad, y_bad
	if  n_elements( x_bad ) * n_elements( y_bad ) ne 0  then $
	   w_bad = coord_to_subs( x_bad, y_bad, ( mysize( ima_box, /DIM ) )[0] )
	; Select known stars into ima_box and subtract them from the image model
	lx_s = lx + par.psf_fwhm / 2  &  ux_s = ux - par.psf_fwhm / 2
	ly_s = ly + par.psf_fwhm / 2  &  uy_s = uy - par.psf_fwhm / 2
	s = where_stars( LO_X = lx_s, UP_X = ux_s, $
					 LO_Y = ly_s, UP_Y = uy_s, list, n )
	if  keyword_set( re_fitting )  then begin
	   if  n eq 0  then  s = this_max	; edge star
	   s_and_this = s
	endif  else  s_and_this = add_subscript( s, this_max )
	extract_elements, list, SUBSCRIPTS = s, n, x, y, f
	if  n ne 0  then begin
	   stars_to_add = stars( psf, x, y, f, $
	   			 par.imsize[0], par.imsize[1], /NO_NORM )
	   image_model = image_model - stars_to_add
	endif else  stars_to_add = fltarr( par.imsize[0], par.imsize[1] )
	; Fixed contribution for local fitting
	contrib = image_model[lx:ux,ly:uy]	; stars outside fitting box
	if  keyword_set( no_slant )  then begin
	   contrib = contrib + bac_box
	   bac_box = bac_box - bac_box
	endif
	; Local fitting
	if  keyword_set( re_fitting )  then  f0 = f  else $
	   extract_elements, list, SUBSCRIPTS = s_and_this, n, x, y
	if  n_elements( w_bad ) ne 0		then $
	fitstars, ima_box, FIXED = contrib, fitting_psf, $
		  x - lx, y - ly, F0 = f0, BACKGROUND = bac_box, $
		  NO_SLANT = no_slant, _EXTRA = extra, BAD_DATA = w_bad, $
		  x, y, f, b, fit_error		else $
	fitstars, ima_box, FIXED = contrib, fitting_psf, $
		  x - lx, y - ly, F0 = f0, BACKGROUND = bac_box, $
		  NO_SLANT = no_slant, _EXTRA = extra, $
		  x, y, f, b, fit_error
	x = x + lx  &  y = y + ly  &  f = f * par.flux_norm
	; Is the examined max indeed a star? Assume it is, then perform checks
	temp_list = list
	temp_list = update_list( temp_list, SUB = s_and_this, x, y, f, /IS_STAR )
	star_here = check_results( temp_list, par, lx, ux, ly, uy, $
							   x, y, f, fit_error )
	; Update list of stars and image model
	if  star_here  then begin
	   list = temp_list
	   stars_to_add = stars( psf, x, y, f, par.imsize[0], par.imsize[1], $
	   						 /NO_NORM, _EXTRA = extra )
	endif
	image_model = image_model + stars_to_add
	return
   end

   ; CHECK_IT: analyze the object list[this_max].

   PRO check_it, list, this_max, template, fitting_psf, psf, image,		$
	   			 image_model, background, bad_pix, par,	_EXTRA = extra, $
	   			 deblend, win_par, star_here

	extract_elements, list, SUB = this_max, n, x, y
	verbose = n_elements( win_par ) ne 0
	if  verbose  then begin
	   x0 = x  &  y0 = y
	   display, win_par, image, win_par.pos1, par.fitting_box, x0, y0
	   display, win_par, image_model, win_par.pos2, par.fitting_box, x0, y0
	   text, x0, y0
	endif
	; Is the maximum list[this_max] a feature of an already detected star?
	identify, par, image, image_model, background, star_here, x, y
	if  not star_here  then begin	; Yes, it is!
	   if  verbose  then  print, 'no star here'
	   return
	endif
	; Is the presumed object a blend?
	blended = deblend
	if  deblend  then $
	   blended = is_a_blend( par, image, image_model, background, x, y )
	if  not blended  then begin		; Single object
	; Correlation check
	correlate, par, image, image_model, background, bad_pix, $
   			   template, _EXTRA = extra, star_here, x, y
	if  not star_here  then begin	; Unsuccessful correlation check
	   if  verbose  then $
	      print, 'presumed star rejected by correlation check'
	   return
	endif
	; Update estimate of this_max position before fitting
	list = update_list( list, SUB = this_max, x, y )
	; Fit image box
	fit_box, list, this_max, par, image, image_model, background, bad_pix, $
			 psf, fitting_psf, _EXTRA = extra, star_here
	if  not star_here  then $		; Unsuccessful fitting
	   if  verbose  then  print, 'presumed star rejected by fit check'
	endif else begin				; Crowded group
	   if  verbose  then  print, 'blended object'
	   deblend_it, list, this_max, template, fitting_psf, psf,	 $
	   			   image, image_model, background, bad_pix, par, $
	   			   _EXTRA = extra, star_here
	endelse
	if  verbose  then begin
	   display, win_par, image_model, win_par.pos2, par.fitting_box, x0, y0
	   display, win_par, image_model, win_par.pos3
	endif
	return
   end

   ; DEBLEND_IT: (try to) deblend a crowded group of stars.

   PRO deblend_it, list, this_max, template, fitting_psf, psf,	 $
   				   image, image_model, background, bad_pix, par, $
   				   _EXTRA = extra, star_here

	n_blended = 0  &  saved_list = list  &  saved_image_model = image_model
	; Identify blend 'centroid'
	extract_elements, list, SUB = this_max, n, x0, y0
	; Single component fit
	fit_box, list, this_max, par, image, image_model, background, $
   			 bad_pix, psf, fitting_psf, _EXTRA = extra,	star_here
	if  star_here  then  n_blended = 1
	while  star_here  do begin
	   ; Identify other components of the blend
	   boxes, x0, y0, par.fitting_box, image, image_model, background, $
	   		  lx, ux, ly, uy, ima_box, mod_box, bac_box
	   search_max, ima_box - mod_box - bac_box, MINIFICATION = par.minif, $
				   par.min_intensity, /MAXIM, n, x, y
	   found = n ne 0  &  star_here = found
	   if  found  then begin
	      ; Fit the new detected component
	      x = x + lx  &  y = y + ly
	      list = add_element( list, create_element( x, y, 0 ) )
	      fit_box, list, n_elements( list ) - 1, par, image, image_model, $
	         	   background, bad_pix, psf, fitting_psf, _EXTRA = extra, $
	         	   star_here
	      if  star_here  then  n_blended = n_blended + 1
	   endif
	endwhile
	if  n_blended eq 1  then begin
	   ; The object was mis-classified as a blend skipping the correlation
	   ; check, but the fit has produced just one component: to accept it
	   ; as a single star do now the correlation check.
	   x = x0  &  y = y0
	   correlate, par, image, saved_image_model, background, bad_pix, $
	   			  template, _EXTRA = extra, star_here, x, y
	   if  not star_here  then begin
	      list = saved_list
	      image_model = saved_image_model
	   endif
	endif
	return
   end

   ; CHECK_CONVERGENCE: Check convergence of positions and fluxes for Phase 4.

   FUNCTION check_convergence, list0, list, $
   			ASTROMETRIC_TOL = a_tol, PHOTOMETRIC_TOL = ph_tol

	if  n_elements( a_tol )  eq 0  then  a_tol  = 0.01
	if  n_elements( ph_tol ) eq 0  then  ph_tol = 0.01
	s = where_stars( list, n )
	if  n eq 0  then  return, n eq 0
	extract_elements, list0, SUB = s, n, x0, y0, f0
	extract_elements, list,  SUB = s, n, x,  y,  f
	return, convergence( x0, x, a_tol, /ABSOLUTE ) and $
			convergence( y0, y, a_tol, /ABSOLUTE ) and $
			convergence( f0, f, ph_tol )
   end

   ; OPT_THRESHOLD: auxiliary procedure for DEFINE_PAR.

   PRO opt_threshold, psf, dist, threshold, area, ratio, _EXTRA = extra

	dx = 0  &  dy = 0  &  d = dist
	if  randomn( seed ) lt 0  then  d = -d
	if  randomn( seed ) lt 0  then  dx = dx + d  else  dy = dy + d
	template1 = psf + image_shift( psf, dx, dy, /ANY, _EXTRA = extra )
 									; simulate close binary
	template2 = image_shift( psf, 0.5, 0.5, _EXTRA = extra )
									; off-centered PSF
	threshold = [ 0.15, 0.25, 0.5, 0.75 ]  &  n_guess = 4
	area = fltarr( n_guess )
	t_ratio = fltarr( n_guess )  &  p_ratio = fltarr( n_guess )
	for  n = 0, n_guess - 1  do begin
	   area[n] = image_area( psf, REL = threshold[n] )
	   t_ratio[n] = image_area( template1, REL = threshold[n] ) / area[n]
	   p_ratio[n] = image_area( template2, REL = threshold[n] ) / area[n]
	endfor
	w1 = where( t_ratio gt p_ratio )
	m = max( ( t_ratio[w1] - p_ratio[w1] ) / p_ratio[w1], w )
	threshold = threshold[w1[w]]  &  area = area[w1[w]]
	ratio = ( 0.9 * t_ratio[w1[w]] ) > p_ratio[w1[w]]
	return
   end

   ; DEFINE_PAR: define program parameters.

   PRO define_par, min_intensity, minif, min_correlation, $
   				   deblend, imsize, par, psf, template, fitting_psf, $
   				   _EXTRA = extra

	; Parameters concerning search for suspected stars, correlation, fitting.
	core_size = 15	; upper limit to the FWHM of typical PSFs
	fwhm = image_width( extract( psf, core_size, core_size ) )
	correlation_box = round( 1.5 * fwhm ) > 5
	correlation_box = correlation_box + 1 - correlation_box mod 2
	search_box = correlation_box / 2
	search_box = search_box + 1 - search_box mod 2
	fitting_box = round( correlation_box + fwhm )
	fitting_box = fitting_box + 1 - fitting_box mod 2
	min_distance = 0.75 * fwhm
	; Parameters for blends identification
	if  deblend  then begin
	   box = 2 * round( fwhm + min_distance ) + 1
	   opt_threshold, extract( psf, box, box ), min_distance, $
	   				  area_threshold, psf_area, area_tol, _EXTRA = extra
	endif else begin
	   area_threshold = 0  &  psf_area = 0  &  area_tol = 0
	endelse
	; PSF
	template = extract( psf, correlation_box, correlation_box )
	fitting_psf = extract( psf, 2 * fitting_box, 2 * fitting_box )
	flux_norm = total( psf ) / total( fitting_psf )
	fitting_psf = normalize( fitting_psf )  &  psf = normalize( psf )
	min_flux  = min_intensity / max( psf )
	; Store parameters into a single structure
	par = { min_intensity	: min_intensity,	$
			minif			: minif,			$
			psf_fwhm		: fwhm,				$
			correlation_box	: correlation_box,	$
			search_box		: search_box,		$
			min_correlation	: min_correlation,	$
			fitting_box		: fitting_box,		$
			min_distance	: min_distance,		$
			area_threshold	: area_threshold,	$
			psf_area		: psf_area,			$
			area_tol		: area_tol,			$
			flux_norm		: flux_norm,		$
			min_flux		: min_flux,			$
			imsize			: imsize,			$
			psfsize			: ( mysize( psf, /DIM ) )[0] }
	return
   end

   ; UPDATE_THRESHOLD: update detection threshold and minimum detectable flux.

   PRO update_threshold, par, min_intensity, n, psf

	par.min_intensity = $
		min_intensity[n > 0 < ( n_elements( min_intensity ) - 1 )]
	par.min_flux = par.min_intensity / max( psf )
	return
   end

   ; DEFINE_WINDOW: open a new window for graphic output. It is assumed
   ; that the program must display three images on the window. Store the
   ; set of related parameters into a single structure.

   FUNCTION define_window, imsize

	window
	x_size = !D.x_size  &  box_size = x_size / 3
	full_size = round( float( imsize ) / max( imsize ) * box_size )
	window_par = { box_size: box_size, full_size: full_size, $
				   empty_display: fltarr( box_size, box_size ), $
				   pos1: 0, pos2: 1, pos3: 2 }
	return, window_par
   end


   ; FIT: fit stars given approximate positions and fluxes.

   PRO fit, image, psf, x_in, y_in, _EXTRA = extra,			 $
   			min_intensity, min_correlation, N_ITER = n_iter, $
   			x, y, f, image_model, background

;	on_error, 1
	if  n_elements( x_in ) * n_elements( y_in ) eq 0  then $
	   message, 'missing parameters'
	half_corr_box = 0		; dummy value
	if  n_elements( n_iter ) eq 0  then  n_iter = 5
	findstars, X_IN = x_in, Y_IN = y_in, image, psf, _EXTRA = extra, $
			   min_intensity, min_correlation, N_ITER = n_iter,		 $
			   x, y, f, image_model, background
	return
   end


   PRO findstars, image, psf_in, min_intensity, min_correlation, 		$
   				  DEBLEND = deblend, N_ITER = n_iter, N_FWHM = n_fwhm,	$
   				  BACKGROUND = background_in, NO_BACKGROUND =			$
   				  no_background, SV_PAR = sv_par,						$
   				  X_BAD = x_bad, Y_BAD = y_bad,							$
   				  MINIFICATION = minif, _EXTRA = extra,					$
   				  SHOW = show, FILE_NAME = file_name,		 			$
   				  x_pos, y_pos, fluxes, image_model, background,		$
   				  X_IN = x_in, Y_IN = y_in

	; Preliminary operations
	imsize = mysize( image, /DIM )  &  psf = psf_in
	image_model = fltarr( imsize[0], imsize[1] )	; initialize the image
								; model to add a PSF for each detected star
	sv = n_elements( sv_par ) ne 0
	if  sv  then  ref_psf = extend_pow2( psf_in )
	if  n_elements( x_bad ) * n_elements( y_bad ) ne 0  then $
	   bad_pix = { x: x_bad, y: y_bad }
	if  n_elements( minif ) eq 0  then  minif = 1
	if  n_elements( n_fwhm ) eq 0  then  n_fwhm = 5
	deblend = keyword_set( deblend ) ;and minif eq 1
	if  n_elements( n_iter ) eq 0  then  n_iter = 2
	define_par, min_intensity[0], minif, min_correlation, deblend, $
				imsize, par, psf, template, fitting_psf, _EXTRA = extra
	if  keyword_set( show )  then  win_par = define_window( imsize )
	print_to_file = n_elements( file_name ) ne 0

 	; Create a list of stars for pure fitting of a set of presumed objects.
	fit_only = n_elements( x_in ) * n_elements( y_in ) ne 0
	if  fit_only  then $
	   list_of_stars = update_list( create_list( n_stars ), x_in, y_in )

    ; Preliminary operations on background
	if  keyword_set( no_background )				then $
	   background = fltarr( imsize[0], imsize[1] )  else $
	if  n_elements( background_in ) ne 0			then $
	   background = background_in					else $
	   background = estimate_background( image, _EXTRA = extra, $
	   									 n_fwhm * round( par.psf_fwhm ) )

	; Iterative search for suspected stars and subsequent analysis.
	; The number of iterations is defined by the threshold levels
	; passed with the input parameter min_intensity.
	if  fit_only  then begin
	   n_levels = 1
	   n_stars = n_elements( x_in )  &  n_suspected = n_stars
	endif else begin
	   n_levels = n_elements( min_intensity )
	   n_stars = 0L  &  n_suspected = 0L
	endelse

	for  n_lev = 0L, n_levels - 1  do begin

	if  not fit_only  then begin

	; Update threshold
	update_threshold, par, min_intensity, n_lev, psf
	; Phase 1: find all local maxima having
	;		   central intensity > known stars + background + min_intensity
	print, 'FINDSTARS: search for suspected stars'
	search_max, image - image_model - background, MINIFICATION = minif, $
				par.min_intensity, n_max, x0, y0, i0
	n_suspected = n_suspected + n_max
	; Phase 2: analyze presumed stars with correlation and fitting,
	;		   estimating positions and fluxes.
	if  n_max ne 0  then begin
	   print, 'FINDSTARS: analysis of suspected stars'
	   list_of_max = update_list( create_list( n_max ), x0, y0, i0 )
	   list_of_stars = merge_list( list_of_stars, sort_list( list_of_max ) )
	   for  n = n_stars, n_stars - 1 + n_max  do begin
	      if  sv  then begin
	         psf = local_psf( list_of_stars, n, _EXTRA = extra, $
	         				  ref_psf, sv_par )
	         define_par, par.min_intensity, minif, min_correlation, $
   					     deblend, imsize, par, psf, template, fitting_psf
   		  endif
	      check_it, list_of_stars, n, template, fitting_psf, psf, image,   $
	      			image_model, background, bad_pix, par, _EXTRA = extra, $
	      			deblend, win_par, star_here
	      if  star_here and print_to_file  then $
	         print_stars, list_of_stars, file_name
	   endfor
	   list_of_stars = extract_stars( list_of_stars, n_stars )
	   ; Update background estimate
	   if  not keyword_set( no_background ) then $
	      background = estimate_background( image - image_model, _EXTRA = $
	      							extra, n_fwhm * round( par.psf_fwhm ) )
	endif	; n_max ne 0
	endif	; not fit_only
	; Phase 3: re-determination of positions and fluxes by iterative
	;		  fitting of detected stars.
	if  n_lev lt n_levels - 1  then  maxit = 1  else  maxit = n_iter
	if  n_stars ne 0  then  list0 = list_of_stars
	iter = 0L  &  converging = n_stars eq 0
	while  iter lt maxit and not converging  do begin
	   print, 'FINDSTARS: re-fitting iteration', iter + 1
	   first_fit = fit_only and iter eq 0
	   if  first_fit  then  s = lindgen( n_stars )  else $
	      s = where_stars( list_of_stars, n_stars )
	   for  n = 0L, n_stars - 1  do begin
	      if  sv  then begin
	         psf = local_psf( list_of_stars, s[n], _EXTRA = extra, $
	         				  ref_psf, sv_par )
	         define_par, par.min_intensity, minif, min_correlation, $
   					     deblend, imsize, par, psf, template, fitting_psf
   		  endif
	      fit_box, list_of_stars, s[n], par, image, image_model,   $
	   		       background, bad_pix, psf, fitting_psf, _EXTRA = $
	      		   extra, RE_FITTING = ( not first_fit ) and 1B
	   endfor
	   if  maxit gt 1  then $
	      converging = check_convergence( list0, list_of_stars, $
	       								  _EXTRA = extra )
	   iter = iter + 1  &  list0 = list_of_stars
	endwhile

	endfor	; level no. n_lev

	; Save results
	if  n_stars ne 0  then begin
	   list_of_stars = extract_stars( list_of_stars, n_stars )
	   list_of_stars = sort_list( list_of_stars )
	   if  print_to_file  then  print_stars, list_of_stars, file_name
	   extract_elements, list_of_stars, n_stars, x_pos, y_pos, fluxes
	endif
	print, 'FINDSTARS', n_suspected,   '  : presumed stars'
	print, 'FINDSTARS', n_stars, '  : detected stars '
	return
   end
