   ; FIT_KERNEL: fit a (background subtracted) stellar image with a
   ; model given by the convolution of a reference PSF (guide star)
   ; and a 'blurring kernel', represented by an elliptical gaussian
   ; with the major axis oriented towards the reference position.
   ; The optimization is performed by means of NEWTON_GAUSS.

   ; ALGORITHM DESCRIPTION: the input stellar image and on-axis PSF are
   ; centered with sub-pixel accuracy, to eliminate any off-centering
   ; which may bias the width of the best fit gaussian. The data are then
   ; slightly reduced in size (by default 2 pixels on each side) to reject
   ; possible edge effects due to centering. Then an iterative fit is
   ; started, by means of a Newton-Gauss scheme. The fitting region is
   ; the intersection of the stellar image array and the model array (i.e.
   ; the convolution of the on-axis PSF and an elliptical gaussian). The
   ; intersection is determined after ideally superposing the two peaks.

   ; INPUT
   ;	image: stellar image
   ;	psf: reference (on axis) PSF
   ;	pos_angle: position angle of stellar image
   ;	[EDGE = ]: width of outer border rejected after centering
   ;		(default = 2)
   ;	[/WEIGHED]: weighed least squares fit
   ;	[INTERP = ]: select interpolation technique (for details see the
   ;		module IMAGE_SHIFT in the 'image_shift.pro' file)

   ; OUTPUT
   ;	l, w: elongation and width of the best fit gaussian
   ;	scale_factor: model scaling factor; = 1 if /SCALED is not used
   ;	fit_error: normalized least square error of fit, as defined in
   ;		the function FK_ERROR. A negative output value represents an
   ;		error condition
   ;	model: best fit model
   ;	centered_data: centered stellar image, to be compared with model
   ; NOTE 1: the elliptical gaussian is defined as in ELLIPT_GAUSSIAN
   ; NOTE 2: weights for weighed fit (see /WEIGHED) are defined by the
   ; 		 function FK_WEIGHTS



   ; FK_MODEL: compute model.

   FUNCTION fk_model, p

	common fk_ref, psf, psf_max, psf_size, angle
	common fk_bounds, lx, ly, ux, uy
	common fk_kermod, kernel, model

	fk_deparametrize, p, l, w, s
	kernel = gaussian_kernel( angle, l, w, MAX_AT = psf_max, $
							  FIXED_SIZE = psf_size ) ; tot. = 1
	model = convolution( psf, kernel, /NO_NORM ) ; kernel already normalized
	model = model[lx:ux,ly:uy]	; this is stored in common fk_kermod
	return, s * model
   end

   ; FK_IACOBI: compute Iacobi matrix of model.

   FUNCTION fk_iacobi, p

	common fk_bounds, lx, ly, ux, uy
	common fk_iac, npix
	common fk_kermod, kernel, model
	common fk_ref, psf
	common fk_rot_coord, rx2, ry2

	iacobi = fltarr( npix, 3 )
	fk_deparametrize, p, l, w, s
	deriv = s * convolution(  -(1/l)*rx2*kernel, psf, /NO_NORM )
	iacobi[*,0] = reform( deriv[lx:ux,ly:uy], npix )
	deriv = s * convolution(  -(1/w)*ry2*kernel, psf, /NO_NORM )
	iacobi[*,1] = reform( deriv[lx:ux,ly:uy], npix )
	deriv = model
	iacobi[*,2] = reform( deriv, npix )
	return, transpose( iacobi )
   end

   ; FK_CONVERGE: check convergence of parameters.

   FUNCTION fk_converge, p0, p

	common fk_tol, lw_tol, s_tol

	fk_deparametrize, p0, l0, w0, s0
	fk_deparametrize, p, l, w, s
	check = convergence( l0, l, lw_tol, /ABSOLUTE ) and $
			convergence( w0, w, lw_tol, /ABSOLUTE ) and $
			convergence( s0, s, s_tol )
	return, check
   end

   ; FK_PARAMETRIZE: convert user parameters to single vector.

   FUNCTION fk_parametrize, l, w, s

	return, [ 1/l, 1/w, s ]
   end

   ; FK_DEPARAMETRIZE: convert program parameters to user parameters
   ;	(semi-axes of gaussian)

   PRO fk_deparametrize, p, l, w, s

	l = 1 / p[0]  &  w = 1 / p[1]  &  s = p[2]
	return
   end

   ; FK_WEIGHTS: compute weights for weighted fit.

   FUNCTION fk_weights, data

	return, ( data > 0 ) ^2
					; the least squares error computed by FK_ERROR
					; is normalized: there is no need to normalize weights
   end

   ; FK_ERROR: compute (normalized, weighted) least square error
   ; between model and data.

   FUNCTION fk_error, model, data, weights

	return, sqrt(  total( weights * ( model - data )^2 ) / $
				   total( weights * data^2 )  )
   end



   PRO fit_kernel, image, psf, pos_angle, EDGE = edge, _EXTRA = extra,  $
   				   WEIGHED = weighed, WIDTH_TOL = wtol, SCALE_TOL = stol, $
   				   l, w, scale_factor, fit_error, best_fit_model, centered_data

	common fk_bounds, rlx, rly, rux, ruy
	common fk_iac, npix
	common fk_ref, reference, ref_max, ref_size, angle
	common fk_rot_coord, rx2, ry2
	common fk_tol, lw_tol, s_tol

	fit_error = -1	; error condition in case of return before completion

	; Sub-pixel centering of image and PSF
	image_fwhm = fwhm( image )  &  min_size = 3 * round( image_fwhm )
	box = round( image_fwhm )  &  box = box + 1 - box mod 2
	data = centroider( image, CENTROID_BOX = box, _EXTRA = extra )
	psf_fwhm = fwhm( psf )
	box = round( psf_fwhm )  &  box = box + 1 - box mod 2
	reference = centroider( psf, CENTROID_BOX = box, _EXTRA = extra )
	; Size reduction, to eliminate possible edge effects due to centering
	if  n_elements( edge ) eq 0  then  edge = 1  &  edge = edge > 1
	s = mysize( data, /DIM )
	if  min( s - 2*edge ) lt min_size  then  return
	data = data[ edge:s[0]-edge-1, edge:s[1]-edge-1 ]
	s = mysize( reference, /DIM )
	if  min( s - 2*edge ) lt min_size  then  return
	reference = reference[ edge:s[0]-edge-1, edge:s[1]-edge-1 ]

	; Find intersection between data array and reference array,
	; to define the fitting region.
	; NOTE: the data array is reduced and normalized now; the reference
	; array (PSF) will be cut after convolution with the gaussian, i.e.
	; after the computation of the fitting model. The reference array is
	; normalized now so that the total flux within the fitting region is
	; 1, i.e. equal to the total flux of the data being fitted. Any flux
	; differences between the fitted data and the model (due to the larger
	; size of the PSF or to the spread or a small fraction of light after
	; convolution) should be accounted for by the scaling factor included
	; as a 3rd parameter in the model.
	array_intersection, data, reference, dlx, dux, dly, duy, rlx, rux, rly, ruy
	data = data[dlx:dux,dly:duy]  &  data = normalize( data )
	reference = reference / total( reference[rlx:rux,rly:ruy] )
		; this normalization of the reference array (PSF) will slightly
		; change after convolution with the gaussian: the total flux of
		; the fitting region extracted from the model should be slightly
		; smaller than 1. The proper normalization will be assured by the
		; model scaling factor (which should result to be slightly larger
		; than 1)

	; Define weights
	if  keyword_set( weighed )  then $
	   weights = fk_weights( data )  else  weights = 1
	; Define other common block variables
	angle = pos_angle  &  npix = n_elements( data )
	ref_max = get_max( reference );[rlx:rux,rly:ruy] )
	ref_size = mysize( reference, /DIM );[rlx:rux,rly:ruy], /DIM )
	rot_coord, reference, angle, rx2, ry2  &  rx2 = rx2^2  &  ry2 = ry2^2
	if  n_elements( wtol ) ne 0  then  lw_tol = wtol  else  lw_tol = 0.01
	if  n_elements( stol ) ne 0  then  s_tol = stol  else  s_tol = 0.01

	; Initial estimates of parameters
	sigma = sqrt( ( image_fwhm^2 - psf_fwhm^2 ) > 0.01 ) / sqrt( 8*alog(2) )
	p0 = fk_parametrize( sigma, sigma, 1. )
	; Iterative fit
	newton_gauss, 'fk_model', 'fk_iacobi', 'fk_converge', p0, data, $
				  /SCALE, WEIGHTS = weights, p, sigma_p, model, converged, IT = it
	if  converged  then begin
	   fk_deparametrize, p, l, w, scale_factor
	   best_fit_model = fk_model( p )  &  centered_data = data
	   fit_error = fk_error( best_fit_model, centered_data, weights )
	endif
	return
   end
