   ; FIT_KERNEL: given a stack of stellar images, fit each image with a
   ;	model given by the convolution of a reference PSF and a blurring
   ;	kernel, represented by an elliptical gaussian with the major axis
   ;	oriented towards a reference position.
   ;	The optimization is performed by means of IDL AMOEBA.

   ; INPUT
   ;	stack: stack of stellar images (or single image)
   ;	x, y: vectors of coordinates, such that (x[i],y[i]) is the position
   ;		of i-th star
   ;	psf0: reference PSF
   ;	x0, y0: position of reference PSF, in the same reference frame as (x,y)
   ;	[INTERP_TYPE = , CENTROID_BOX = ]: CENTROIDER optional keywords
   ;	[TOLERANCE = ]: relative tolerance for the error between the best fit
   ;		model and stellar image, used as a stopping criterion by AMOEBA
   ;		(default 1e-4)
   ;	[SCALE = ]: AMOEBA input keyword (default = 2 pixel)
   ;	[P0 = ]: AMOEBA input keyword (default = [0,0])
   ; OUTPUT
   ;	r: vector of radial distances of stars from the reference position
   ;	l, w: vectors containing major and minor semi-axis length for the
   ;		elliptical gaussian corresponding to each star in the input
   ;		stack
   ;	stack_model: stack of best fit models. A default model = 0 is assumed
   ;		when the fit is not acceptable
   ; NOTE 1: the elliptical gaussian is defined as in ELLIPT_GAUSSIAN
   ; NOTE 2: each frame in the input stack is centered and normalized before
   ;	fitting. So the input images need only be background subtracted



   ; FK_MODEL: compute model = gaussian * psf

   FUNCTION fk_model, par_width

	common	fk_data, sx, sy, pos_angle, image, psf, interp, cbox

	fk_deparam, par_width, l, w
	c = abs( cos( pos_angle ) )  &  s = abs( sin( pos_angle ) )
	ker_size = [ max( [l*c, w*s] ), max( [l*s, w*c] ) ]  &  n_sigma = 3
	ker_size = ( 2 * n_sigma * round( ker_size ) + 1 ) > 3
	kernel = ellipt_gaussian( ker_size[0], ker_size[1], $
	   	    ker_size[0]/2, ker_size[1]/2, pos_angle, l, w )
	model = convolution( psf, kernel, /NO_NORM )
	s = mysize( model, /DIM )
	model = centroider( model, INTERP = interp, CENTROID_BOX = cbox, $
			   XC = s[0] / 2, YC = s[1] / 2 )
	model = extract( model, sx, sy, X = s[0] / 2, Y = s[1] / 2 )
	model = normalize( model )
	return, model
   end

   ; FK_ERROR: compute error to be minimized

   FUNCTION fk_error, par_width

	common	fk_data, sx, sy, pos_angle, image, psf, interp, cbox
;	weigths = image ;weighed fit
	weigths = 1	;unweighed fit
w = where( image gt 0.5 * max( image ) )
model = fk_model( par_width )
i = image[w]  &  model = model[w]
	residual =( model - i ) * weigths
	error = sqrt(total( residual^2 )/total(image^2))
	fk_deparam, par_width, l, w
	if  l lt 0 or w lt 0  then  error = 10 * error
	return, error
   end

   ; FK_DEPARAM: convert program parameters to user parameters
   ;	(semi-axes of gaussian)

   PRO fk_deparam, par_width, l, w

	l = par_width[0]  &  w = par_width[1]
	return
   end


   PRO fit_kernel, stack, x, y, psf0, x0, y0, INTERP_TYPE = interp_type, $
   		   TOLERANCE = tol, SCALE = scale, P0 = p0, $
   		   x_out, y_out, r, l, w, models, centered_stack, fit_error

	common	fk_data, sx, sy, pos_angle, image, psf, interp, centroid_box

	s = mysize( stack, /DIM )  &  sx = s[0]  &  sy = s[1]
	if  mysize( stack, /N_DIM ) eq 2  then $
	   nframes = 1  else  nframes = s[2]
	psf = psf0  &  s = mysize( psf, /DIM )
	s = [ max( [s[0], sx] ), max( [s[1], sy] ) ]
	psf = center_image( extend_array( psf, s[0], s[1] ), $
			    s[0] / 2, s[1] / 2 )
	sx = sx - 2  &  sy = sy - 2
	x_out = x  &  y_out = y
	r = fltarr( nframes ) -1  &  l = r  &  w = r  &  fit_error = r
	models = fltarr( sx, sy, nframes )  &  centered_stack = models
	if  n_elements( interp_type )  ne 0  then $
	   interp = interp_type  else  interp = ''
	if  n_elements( tol )   eq 0  then  tol = 1.e-4
	if  n_elements( scale ) eq 0  then  scale = 2.
	if  n_elements( p0 )    eq 0  then  p0 = [0.,0.]
	for  n = 0, nframes - 1  do begin
	   centroid_box = image_width( stack[*,*,n] )
	   centroid_box = centroid_box + 1 - centroid_box mod 2
	   image = normalize( centroider( stack[*,*,n], $
	   	INTERP = interp, CENTROID_BOX = centroid_box ) )
	   image = extract( image, sx, sy )
	   r_distance = distance( x[n], y[n], x0, y0 )
	   pos_angle = angle( x0, y0, x[n], y[n] )
	   width = amoeba( FUNCTION_NAME = 'fk_error', tol, $
	   		   SCALE = scale, P0 = p0 )
	   if  n_elements( width ) ne 1  then begin
	      fk_deparam, width, ln, wn
	      if  ln ge 0 and wn ge 0  then begin
   		 r[n] = r_distance  &  l[n] = ln  &  w[n] = wn
		 fit_error[n] = fk_error( width )
	         models[*,*,n] = fk_model( width )
		 centered_stack[*,*,n] = image
	      endif
	   endif
	endfor
	s = where( r ge 0, n )
	if  n ne 0  then begin
	   x_out = x_out[s]  &  y_out = y_out[s]
	   r = r[s]  &  l = l[s]  &  w = w[s]  &  fit_error = fit_error[s]
	   models = models[*,*,s]
   	   centered_stack = centered_stack[*,*,s]
	endif else  message, 'no successful fit!'
	return
   end
