   ; FIT_KERNEL_AMOEBA: fit a (background subtracted) stellar image
   ; with a model given by the convolution of a reference PSF (guide
   ; star) and a 'blurring kernel', represented by an elliptical
   ; gaussian with the major axis oriented towards the reference
   ; position. The optimization is performed by means of IDL AMOEBA.

   ; ALGORITHM DESCRIPTION: the input stellar image and on-axis PSF are
   ; centered with sub-pixel accuracy, to eliminate any off-centering
   ; which may bias the width of the best fit gaussian. The data are then
   ; slightly reduced in size (by default 2 pixels on each side) to reject
   ; possible edge effects due to centering. Then an iterative fit is
   ; started, by means of AMOEBA (downhill simplex). The fitting region is
   ; the intersection of the stellar image array and the model array (i.e.
   ; the convolution of the on-axis PSF and an elliptical gaussian). The
   ; intersection is determined after ideally superposing the two peaks.

   ; INPUT
   ;	image: stellar image
   ;	psf: reference (on axis) PSF
   ;	pos_angle: position angle of stellar image
   ;	[EDGE = ]: width of outer border rejected after centering
   ;		(default = 2)
   ;	[/WEIGHED]: weighed least squares fit
   ;	[/SCALED]: estimate scale factor of fitting model
   ;		(default: scaling is defined by normalization)
   ;	[TOLERANCE = ]: convergence criterion for AMOEBA (default = 1e-4)
   ;	[INTERP = ]: select interpolation technique (for details see the
   ;		module IMAGE_SHIFT in the 'image_shift.pro' file)

   ; OUTPUT
   ;	l, w: elongation and width of the best fit gaussian
   ;	scale_factor: model scaling factor; = 1 if /SCALED is not used
   ;	fit_error: normalized least square error of fit, as defined in
   ;		the function FK_ERROR. A negative output value represents an
   ;		error condition
   ;	model: best fit model
   ;	centered_data: centered stellar image, to be compared with model
   ; NOTE 1: the elliptical gaussian is defined as in ELLIPT_GAUSSIAN
   ; NOTE 2: weights for weighed fit (see /WEIGHED) are defined by the
   ; 		 function FK_WEIGHTS



   ; FK_MODEL: compute fitting model.

   FUNCTION fk_model, par

	common fk_amoeba, data, weights, angle, psf, lx, ux, ly, uy
	; Compute model
	fk_deparam, par, l, w, scale
	kernel = gaussian_kernel( angle, l, w, /NO_NORM )	;1 kernel = gaussian_kernel( angle, l, w )
	model = convolution( psf, kernel, /NO_NORM )
	; Extract fitting region from model
	model = scale * normalize( model[lx:ux,ly:uy] )		;2 model = scale * model[lx:ux,ly:uy]
	return, model
   end

   ; FK_CONSTRAINTS: verify constraints on the solution.

   FUNCTION fk_constraints, par

	fk_deparam, par, l, w, scale
	check = l ge 0 and w ge 0 and scale gt 0
	if  check  then  check = l ge w
	return, check
   end

   ; FK_ERROR: compute error to be minimized.

   FUNCTION fk_error, par

	common fk_amoeba, data, weights
	; Define model and data to be fit
	model = fk_model( par )
	; Compute weighed normalized least square error between model and data
	error = sqrt(  total( weights * ( model - data )^2 ) / $
				   total( weights * data^2 )  )
	; Verify constraints on the solution: if not fulfilled, increase error
	if  not fk_constraints( par )  then  error = 10 * error
	return, error
   end

   ; FK_DEPARAM: convert program parameters to user parameters
   ;	(semi-axes of gaussian).

   PRO fk_deparam, par, l, w, scale

	l = par[0]  &  w = par[1]
	if  n_elements( par ) eq 3  then  scale = par[2]  else  scale = 1
	return
   end

   ; FK_WEIGHTS: compute weights for weighed fit.

   FUNCTION fk_weights, data

	return, ( data > 0 ) ^2
					; the least squares error computed by FK_ERROR
					; is normalized: there is no need to normalize weights
   end





   PRO fit_kernel_amoeba, image, psf, pos_angle, EDGE = edge, _EXTRA = extra, $
   				   WEIGHED = weighed, SCALED = scaled, TOLERANCE = tol, $
   				   l, w, scale_factor, fit_error, model, centered_data

	common fk_amoeba, data, weights, angle, reference, rlx, rux, rly, ruy

	fit_error = -1	; error condition in case of return before completion

	; Sub-pixel centering of image and PSF
	image_fwhm = fwhm( image )  &  min_size = 3 * round( image_fwhm )
	box = round( image_fwhm )  &  box = box + 1 - box mod 2
	data = centroider( image, CENTROID_BOX = box, _EXTRA = extra )
	psf_fwhm = fwhm( psf )
	box = round( psf_fwhm )  &  box = box + 1 - box mod 2
	reference = centroider( psf, CENTROID_BOX = box, _EXTRA = extra )
	; Size reduction, to eliminate possible edge effects due to centering
	if  n_elements( edge ) eq 0  then  edge = 2  &  edge = edge > 1
	s = mysize( data, /DIM )
	if  min( s - 2*edge ) lt min_size  then  return
	data = data[ edge:s[0]-edge-1, edge:s[1]-edge-1 ]
	s = mysize( reference, /DIM )
	if  min( s - 2*edge ) lt min_size  then  return
	reference = reference[ edge:s[0]-edge-1, edge:s[1]-edge-1 ]

	; Find intersection between data array and reference array,
	; to define the fitting region.
	; NOTE: the data array is reduced and normalized now, whereas the
	; reference array will be cut and normalized after convolution with
	; the gaussian, i.e. after computing the fitting model
	array_intersection, data, reference, dlx, dux, dly, duy, rlx, rux, rly, ruy
	if  min( [ dux-dlx+1, duy-dly+1 ] ) lt min_size  then  return
	data = data[dlx:dux,dly:duy]  &  data = normalize( data )
														;3 reference = normalize( reference )

	; Define weights
	if  keyword_set( weighed )  then $
	   weights = fk_weights( data )  else  weights = 1
	; Define other common block variables
	angle = pos_angle

	; AMOEBA input parameters
	if  n_elements( tol )  eq 0  then  tol = 1.e-4
	sigma = sqrt( ( image_fwhm^2 - psf_fwhm^2 ) > 0.01 ) / sqrt( 8*alog(2) )
	p0 = [ sqrt(3)*sigma, sigma ]  &  scale = p0
	if  keyword_set( scaled )  then begin
	   p0 = add_elements( p0, 1. )
	   scale = add_elements( scale, 0.01 )
	endif
	; AMOEBA fit
	par = amoeba( FUNCTION_NAME = 'fk_error', tol, SCALE = scale, $
				  P0 = p0, FUNCTION_VALUE = fk_error_val,		  $
				  NCALLS = ncalls, SIMPLEX = simplex )

	; If fit is acceptable ...
	if  n_elements( par ) ne 1  then  fit_error = fk_error_val[0]
	if  fit_error ge 0  then begin
	   fk_deparam, par, l, w, scale_factor
	   ; ... check fitted parameters
	   if  fk_constraints( par )  then begin
	      model = fk_model( par )
	      centered_data = data
	   endif
	endif
	return
   end
