   ; FITSTARS:	fit an image with a model made of a sum of shifted scaled
   ;	replicas of a PSF (stars) + a slanting plane as background.
   ; ALGORITHM DESCRIPTION: the fitting model parameters (stars positions and
   ;	intensities, background mean, x-gradient and y-gradient) are optimized
   ;	by minimizing iteratively a least squares error function.
   ;	It is possible to mask (i.e. exclude from fitting) bad data in one of
   ;	the following ways:
   ;	1) supplying on input an array of subscripts for the points to be
   ;	   masked (see keyword BAD_DATA)
   ;	2) supplying on input an estimate of the background noise standard
   ;	   deviation, used by the algorithm to identify the pixels where the
   ;	   error between the data and the model (computed at a pre-fixed
   ;	   iteration of the iterative scheme) is too large.

   ; INPUT
   ;	image: image to be fit
   ;	[FIXED = fixed_image]: fixed additive contribution to the fitting model
   ;	psf: Point Spread Function (single object template)
   ;	NOTE: the psf should be twice as large as the image being fit
   ;	x0, y0: estimates of objects positions (in pixels)
   ;	[/NO_SLANT]: the fitting model has no slanting plane as background
   ;	[POS_TOL = pos_tol_in]: absolute convergence criterion for positions
   ;		(default = 0.01 pixels)
   ;	[PHOT_TOL = phot_tol_in]: relative convergence criterion for intensities
   ;		and background parameters (default = 0.01 = 1%)
   ;	[/VERBOSE]: print current estimates of parameters at each iteration
   ;	[_EXTRA = extra]: optional parameters required by the auxiliary
   ;		functions of FITSTARS
   ;	Important keyword parameters are:
   ;	INTERP_TYPE = {'FT' (default), 'SPLINE1', 'SPLINE3', ...},
   ;		which specifies the interpolating technique to shift the PSF
   ;		(see IMAGE_SHIFT for details)
   ;	BAD_DATA = bad_data: array of subscripts for the data points to be
   ;		masked (data masking MODE 1, described above)
   ;	/MASK: if set, bad data masking is performed as described above at
   ;		point 2) (data masking MODE 2). Using this data masking mode, the
   ;		following parameters may be supplied by the user:
   ;		SIGMA_NOISE = sigma, NOISE_TOL = tol, WHEN = mask_now (see below)
   ;	SIGMA_NOISE = sigma: standard deviation of noise;
   ;		to be supplied only if /MASK is set (default = estimate of the
   ;		background noise standard deviation, derived by BACKGROUND_NOISE)
   ;	WHEN = mask_now: iteration no. to search for bad data to be masked
   ;		(i.e. bad data are identified after (mask_now-1) iterations);
   ;		to be supplied only if /MASK is set (default = at the 2nd iteration)
   ;	NOISE_TOL = tol: bad data masking is active on the data points
   ;		where the absolute error between the data and the the model at
   ;		iteration "mask_now" is > tol * sigma;
   ;		to be supplied only if /MASK is set (default = 5)

   ; OUTPUT
   ;	x, y: estimated positions of stars
   ;	i: estimated intensities of stars
   ;	b: array of background parameters
   ;		(b[0] = mean; b[1], b[2] = x- and y- gradient)
   ;	[MODEL = model]: best fit image model
   ;	[FIT_ERROR = fit_error]: least squares fitting error. Return a negative
   ;		number if the fit does not converge
   ;	[IT = it]: number of iterations to achieve convergence
   ;	[W_BAD = w_bad]: array subscript for masked pixels
   ;		(only if /MASK is set)





   ; MODEL_CAL: compute the image model for the current estimates
   ;	of the parameters.

   FUNCTION model_cal, parameters, DATA = data, _REF_EXTRA = extra

	deparametrize, parameters, x, y, i, b, DATA = data
	unit_flux = 1  &  model = data.image - data.image
	for  n = 0, data.nstar - 1  do begin
	   data.psf_stack[*,*,n] = stars( data.psf, x[n], y[n], unit_flux, $
	   							  data.sx, data.sy, /NO_NORM, _EXTRA = extra )
	   model = model + i[n] * data.psf_stack[*,*,n]
	endfor
	bplane = 0
	if  data.b_nterms ne 0  then  $
	   bplane = plane( b[0], b[1], b[2], data.sx, data.sy )
	model = model + data.fixed_image + bplane
	return, model
   end



   ; PARAMETRIZE: store the model parameters (i.e. user parameters x, y, etc.)
   ;	into a single vector (program parameters).

   FUNCTION parametrize, x, y, i, b, DATA = data

	parameters = fltarr( 3*data.nstar + data.b_nterms )
	parameters[0] = i
	subs = data.nstar + 2*lindgen( data.nstar )
	parameters[subs] = x  &  parameters[subs+1] = y
	if  data.b_nterms ne 0  then  parameters[3*data.nstar] = b
	return, parameters
   end



   ; DEPARAMETRIZE: convert program parameters vector to user parameters.

   PRO deparametrize, parameters, x, y, i, b, DATA = data

	i = parameters[0:data.nstar-1]
	subs = data.nstar + 2*lindgen( data.nstar )
	x = parameters[subs]  &  y = parameters[subs+1]
	if  data.b_nterms ne 0  then $
	   b = parameters[3*data.nstar:3*data.nstar+data.b_nterms-1]
	return
   end



   ; CENTER_DERIVATIVE: compute the partial derivative of an image
   ;	with respect to the x- or y- coordinate of its center (maximum).

   FUNCTION center_derivative, image, coord, DATA = data

	if  coord eq 'X'  then begin
	   f = frequency( data.sx ) # ( fltarr( data.sy ) + 1 )  &  s = data.sx
	endif else begin
	   f = ( fltarr( data.sx ) + 1 ) # frequency( data.sy )  &  s = data.sy
	endelse
	f = 2*!pi/s * fft( image ) * f
	deriv = complex( imaginary( f ), -float( f ) )
	deriv = float( fft( deriv, /INVERSE ) )
	return, deriv
   end



   ; IACOBIAN: compute the Iacobi matrix of the parametric model.

   FUNCTION iacobian, parameters, DATA = data, _REF_EXTRA = no_extra

	n_pix = data.sx * data.sy
	iacobi = fltarr( n_pix, 3*data.nstar+data.b_nterms )
	for  n = 0, data.nstar - 1  do begin
	   iacobi[*,n] = reform( data.psf_stack[*,*,n], n_pix )
	   deriv = center_derivative( data.psf_stack[*,*,n], 'X', DATA = data )
	   iacobi[*,data.nstar+2*n]   = reform( parameters[n] * deriv, n_pix )
	   deriv = center_derivative( data.psf_stack[*,*,n], 'Y', DATA = data )
	   iacobi[*,data.nstar+2*n+1] = reform( parameters[n] * deriv, n_pix )
	endfor
	if  data.b_nterms ne 0  then  iacobi[*,3*data.nstar] = 1
	if  data.b_nterms eq 3  then begin
	   iacobi[*,3*data.nstar+1] = $
	   		reform( plane( 0, 1, 0, data.sx, data.sy ), n_pix )
	   iacobi[*,3*data.nstar+2] = $
	   		reform( plane( 0, 0, 1, data.sx, data.sy ), n_pix )
	endif
	return, transpose( iacobi )
   end



   ; OUTPUT_PRO: print iteration no., fitting error, current values
   ;	of parameters.

   PRO output_pro, it, fit_error, parameters, DATA = data

	print, ''
	print, 'iteration     ', it
	print, 'fitting error ', fit_error
	print, 'parameters: '
	deparametrize, parameters, x, y, i, b, DATA = data
	print, 'x coordinates: ', x
	print, 'y coordinates: ', y
	print, 'fluxes	     : ', i
	if  data.b_nterms ne 0  then  print, 'background   : ', b
   	return
   end



   ; LS_ERROR: compute the least squares error between the model and the image.

   FUNCTION ls_error, model, DATA = data, EXCLUDE = exclude

	residuals = mask_pixels( model - data.image, exclude )
	return, total( residuals^2 )
   end



   ; FIT_CONVERG: test the convergence condition for the iterative estimation.

   FUNCTION fit_converg, p0, p, DATA = data

	deparametrize, p0, x0, y0, i0, b0, DATA = data
	deparametrize, p,  x,  y,  i,  b,  DATA = data
	check = convergence( x0, x, data.pos_tol, /ABSOLUTE ) and $
			convergence( y0, y, data.pos_tol, /ABSOLUTE ) and $
			convergence( i0, i, data.phot_tol )
	if  data.b_nterms ne 0  then $
	   check = check and convergence( b0, b, data.phot_tol )
	return, check
   end





   PRO fitstars, image_in, FIXED = fixed_image_in, psf_in, x0, y0,		  $
		 NO_SLANT = no_slant, POS_TOL = pos_tol_in, PHOT_TOL = phot_tol_in, $
		 VERBOSE = verbose, _EXTRA = extra, x, y, i, b, MODEL = model,	  $
		 fit_error, IT = it, W_BAD = w_bad

	image = image_in  &  psf = psf_in
	s = mysize( image, /DIM )  &  sx = s[0]  &  sy = s[1]
	if  n_elements( fixed_image_in ) eq 0  then $
	   fixed_image = image - image		   else $
	   fixed_image = fixed_image_in
	nstar = n_elements( x0 )
	psf_stack = fltarr( sx, sy, nstar )
	if  n_elements( pos_tol_in )  eq 0  then $
	   pos_tol  = 0.01  else  pos_tol = pos_tol_in
	if  n_elements( phot_tol_in) eq 0  then $
	   phot_tol = 0.01  else  phot_tol = phot_tol_in
	if  keyword_set( verbose )  then  out_pro = 'output_pro'

	; Initial estimates of parameters
	x = x0  &  y = y0
	i = image[x,y] / total( image ) * ( total( image ) - total( fixed_image ) )
	i = i / max( psf )
	if  not keyword_set( no_slant )  then begin
	   b_nterms = 3  &  b = fltarr( b_nterms )
	endif else begin
	   b_nterms = 0  &  b = 0
	endelse
	data = { image: image, fixed_image: fixed_image, psf: psf, $
			 psf_stack: psf_stack, nstar: nstar, sx: sx, sy: sy, $
			 b_nterms: b_nterms, pos_tol: pos_tol, phot_tol: phot_tol }
	p0 = parametrize( x, y, i, b, DATA = data )

	; Iterative estimation of parameters
	newton_gauss, 'model_cal', 'iacobian', 'ls_error', 'fit_converg', $
		      p0, image, VERBOSE = verbose, OUT_PRO = out_pro,	  $
		      _EXTRA = extra, DATA = data, $
		      p, fit_error, model, IT = it, W_BAD = w_bad

	if  fit_error ge 0  then begin
	   deparametrize, p, x, y, i, b, DATA = data
	   if  ( mysize( x0 ) )[0] eq 0  then begin
	      x = x[0]  &  y = y[0]  &  i = i[0]
	   endif
	endif
	return
   end


