   ; FITSTARS: fit an image with a model made of a sum of shifted scaled
   ;	replicas of a PSF (stars) + a slanting plane as background.
   ; ALGORITHM DESCRIPTION: the fitting model parameters (stars positions
   ;	and fluxes, slanting plane coefficients) are optimized by
   ;	minimization of the least square error between the data and the
   ;	model.
   ;	It is possible to mask (i.e. exclude from fitting) bad data in one
   ;	( or both) of the following ways:
   ;	1) supplying on input an array of subscripts for the points to be
   ;	   masked (see keyword BAD_DATA)
   ;	2) identifying bad pixels as those points where the error between
   ;	   the data and the model (computed at a pre-fixed iteration of the
   ;	   iterative scheme) is larger than a threshold proportional to the
   ;		noise standard deviation (mean or per pixel)
   ;	It is possible to perform a weighted fit: in this case the image is
   ;	supposed to be contaminated by gaussian noise and/or photon noise.

   ; INPUT
   ;	image: image to be fit
   ;	[FIXED = ]: fixed additive contribution to the fitting model
   ;	psf: Point Spread Function (single object template)
   ;	NOTE: the psf should be at least twice as large as the image
   ;	x0, y0: estimates of objects positions (in pixels)
   ;	[F0 = ]: initial estimates of fluxes
   ;	[BACKGROUND = ]: image background estimate (2-D array), useful to
   ;		determine initial estimates of fluxes when not supplied on input
   ;	[/NO_SLANT]: the fitting model has no slanting plane as background
   ;	[POS_TOL = ]: absolute convergence criterion for positions
   ;		(default = 0.01 pixels)
   ;	[PHOT_TOL = ]: relative convergence criterion for intensities
   ;		and slanting plane coefficients (default = 0.01 = 1%)
   ;	[/VERBOSE]: print current estimates of parameters at every
   ;		iteration
   ;	[INTERP_TYPE = {'FT' (default), 'SPLINE1', 'SPLINE3', ...}]:
   ;		to specify the interpolating technique to shift the PSF
   ;		(see IMAGE_SHIFT for details)
   ;	[GAUSSIAN_NOISE = ]: standard deviation of gaussian noise (ADU/pix)
   ;	[PHOTON_COUNTS = ]: photon counts (ADU/pix)
   ;	[_EXTRA = ]: optional parameters required by the auxiliary
   ;		functions of FITSTARS
   ;	Important keyword parameters are:
   ;	[BAD_DATA = ]: array of subscripts for the data points to be
   ;		masked (data masking MODE 1, described above)
   ;	[/MASK]: if set, bad data masking is performed as described above
   ;		at point 2) (data masking MODE 2). Using this data masking mode,
   ;		the following parameters may be supplied by the user:
   ;		[NOISE_TOL = ], [WHEN = ] (see below)
   ;	[WHEN = ]: iteration no. to search for bad data to be masked;
   ;		to be supplied only if /MASK is set (default is at the 2nd
   ;		iteration)
   ;	[NOISE_TOL = ]: bad data masking is active on the data points where
   ;		the absolute error between the data and the the model at the
   ;		iteration specified by the keyword WHEN is > tol * sigma;
   ;		to be supplied only if /MASK is set (default = 5)

   ; OUTPUT
   ;	x, y: estimated positions of stars
   ;	f: estimated fluxes of stars
   ;	b: slanting plane coefficients. The plane is defined as follows:
   ;		plane(x,y) = b[0] + b[1] * x + b[2] * y
   ;	sigma_x, sigma_y, sigma_f, sigma_b: errors (1 sigma) on estimated
   ;		parameters (only if info on gaussian and/or photon noise have
   ;		been input)
   ;	[MODEL = ]: best fit image model
   ;	fit_error: least squares fitting error. Return a negative number if
   ;		the fit does not converge
   ;	[IT = ]: number of iterations to achieve convergence
   ;	[W_BAD = ]: array of subscripts for masked pixels
   ;		(only if /MASK is set)





   ; FS_MODEL: compute the image model for the current estimates
   ;	of the parameters.

   FUNCTION fs_model, parameters

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	common	image_shift_data, shift_data, interp
	fs_deparam, parameters, x, y, f, b
	unit_flux = 1  &  model = image - image
	for  n = 0, nstar - 1  do begin
	   if  n_elements( interp ) ne 0   then $
	   psf_stack[*,*,n] = stars( psf, x[n], y[n], unit_flux, sx, sy, $
	   			 /NO_NORM, INTERP_TYPE = interp, $
				    AUX_DATA = shift_data )	else $
	   psf_stack[*,*,n] = stars( psf, x[n], y[n], unit_flux, sx, sy, $
	   			 /NO_NORM, AUX_DATA = shift_data )
	   model = model + f[n] * psf_stack[*,*,n]
	endfor
	bplane = 0
	if  b_nterms ne 0  then  bplane = plane( b[0], b[1], b[2], sx, sy )
	model = model + fixed_image + bplane
	return, model
   end



   ; FS_PARAM: store the model parameters (i.e. user parameters x, y, etc.)
   ;	into a single vector (program parameters).

   FUNCTION fs_param, x, y, f, b

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	parameters = fltarr( 3*nstar + b_nterms )
	parameters[0] = f
	subs = nstar + 2*lindgen( nstar )
	parameters[subs] = x  &  parameters[subs+1] = y
	if  b_nterms ne 0  then  parameters[3*nstar] = b
	return, parameters
   end



   ; FS_DEPARAM: convert program parameters vector to user parameters.

   PRO fs_deparam, parameters, x, y, f, b

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	f = parameters[0:nstar-1]
	subs = nstar + 2*lindgen( nstar )
	x = parameters[subs]  &  y = parameters[subs+1]
	if  b_nterms ne 0  then  b = parameters[3*nstar : 3*nstar+b_nterms-1]
	return
   end



   ; CENTER_DERIVATIVE: compute the partial derivative of an image
   ;	with respect to the x- or y- coordinate of its center (maximum).

   FUNCTION center_derivative, image, sx, sy, coord

	if  coord eq 'X'  then begin
	   f = frequency( sx ) # ( fltarr( sy ) + 1 )  &  s = sx
	endif else begin
	   f = ( fltarr( sx ) + 1 ) # frequency( sy )  &  s = sy
	endelse
	f = 2*!pi/s * fft( image ) * f
	deriv = complex( imaginary( f ), -float( f ) )
	deriv = float( fft( deriv, /INVERSE ) )
	return, deriv
   end



   ; FS_IACOBI: compute the Iacobi matrix of the parametric model.

   FUNCTION fs_iacobi, parameters

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	n_pix = sx * sy
	iacobi = fltarr( n_pix, 3*nstar+b_nterms )
	for  n = 0, nstar - 1  do begin
	   iacobi[*,n] = reform( psf_stack[*,*,n], n_pix )
	   deriv = center_derivative( psf_stack[*,*,n], sx, sy, 'X' )
	   iacobi[*,nstar+2*n]   = reform( parameters[n] * deriv, n_pix )
	   deriv = center_derivative( psf_stack[*,*,n], sx, sy, 'Y' )
	   iacobi[*,nstar+2*n+1] = reform( parameters[n] * deriv, n_pix )
	endfor
	if  b_nterms ne 0  then  iacobi[*,3*nstar] = 1
	if  b_nterms eq 3  then begin
	   iacobi[*,3*nstar+1] = reform( plane( 0, 1, 0, sx, sy ), n_pix )
	   iacobi[*,3*nstar+2] = reform( plane( 0, 0, 1, sx, sy ), n_pix )
	endif
	return, transpose( iacobi )
   end



   ; FS_OUT: print iteration no., fitting error, current values
   ;	of parameters.

   PRO fs_out, it, fit_error, parameters

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	print, ''
	print, 'iteration     ', it
	print, 'fitting error ', fit_error
	print, 'parameters: '
	fs_deparam, parameters, x, y, f, b
	print, 'x coordinates: ', x
	print, 'y coordinates: ', y
	print, 'fluxes	     : ', f ;* scale
	if  b_nterms ne 0  then  print, 'background   : ', b
   	return
   end


   ; FS_WEIGHTS: compute weights for weighted least squares fit.

   FUNCTION fs_weights, var

	min_var = min( var )
	if  min_var le 0.  then  min_var = max( var ) * 1e-15
	return, 1 / ( var > min_var )
   end



   ; FS_ERROR: compute the least squares error between the model and the image.

   FUNCTION fs_error, model, data, WEIGHTS = weights

	w = 1
	if  n_elements( weights ) ne 0  then  w = weights
	return, total( w * ( model - data )^2 )
   end



   ; FS_CONVERGENCE: test the convergence condition for the iterative estimation.

   FUNCTION fs_convergence, p0, p

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	fs_deparam, p0, x0, y0, f0, b0
	fs_deparam, p,  x,  y,  f,  b
	check = convergence( x0, x, pos_tol, /ABSOLUTE ) and $
			convergence( y0, y, pos_tol, /ABSOLUTE ) and $
			convergence( f0, f, phot_tol )
	if  b_nterms ne 0  then $
	   check = check and convergence( b0, b, phot_tol )
	return, check
   end





   PRO fitstars, image_in, FIXED = fixed_image_in, psf_in, x0, y0, F0 = f0,  $
   				 BACKGROUND = background, NO_SLANT = no_slant, POS_TOL =	 $
   				 pos_tol_in, PHOT_TOL = phot_tol_in, VERBOSE = verbose,		 $
   				 _EXTRA = extra, INTERP_TYPE = interp_type,	GAUSSIAN_NOISE = $
   				 gaussian_noise_std, PHOTON_COUNTS = photon_counts,			 $
   				 x, y, f, b, sigma_x, sigma_y, sigma_f, sigma_b, fit_error,	 $
   				 MODEL = model, IT = it, W_BAD = w_bad

	common	fitstars_data, image, fixed_image, psf, psf_stack, $
			sx, sy, nstar, b_nterms, pos_tol, phot_tol
	common	image_shift_data, shift_data, interp

	image = image_in  &  psf = psf_in
	s = mysize( image, /DIM )  &  sx = s[0]  &  sy = s[1]
	if  n_elements( fixed_image_in ) eq 0  then $
	   fixed_image = image - image		   else $
	   fixed_image = fixed_image_in
	if  n_elements( background ) eq 0  then  if  n_elements( f0 ) eq 0  then $
	   background = image - image
	if  n_elements( interp_type ) ne 0  then  interp = interp_type
	nstar = n_elements( x0 )
	psf_stack = fltarr( sx, sy, nstar )
	if  n_elements( pos_tol_in )  eq 0  then $
	   pos_tol  = 0.01  else  pos_tol = pos_tol_in
	if  n_elements( phot_tol_in ) eq 0  then $
	   phot_tol = 0.01  else  phot_tol = phot_tol_in
	if  keyword_set( verbose )  then  out_pro = 'fs_out'

	; Initial estimates of parameters
	x = x0  &  y = y0
	if  n_elements( f0 ) eq 0  then begin
	   sub = image - fixed_image - background
	   f = sub[round( x ) > 0 < ( sx - 1 ), round( y ) > 0 < ( sy - 1 )]
	   										; central intensity of sources
	   f = f / total( f ) * total( sub ) / total( psf )
	   										; conversion to fluxes
	endif else  f = f0
	if  not keyword_set( no_slant )  then begin
	   b_nterms = 3  &  b = fltarr( b_nterms )
	endif else begin
	   b_nterms = 0  &  b = 0
	endelse
	;scale = n_elements( image ) / total( psf )
	;psf = psf * scale  &  f = f / scale
	p0 = fs_param( x, y, f, b )

	; Define noise standard deviation and weights
	if  n_elements( gaussian_noise_std ) ne 0 or $
		n_elements( photon_counts ) ne 0  then begin
	   if  n_elements( gaussian_noise_std ) eq 0  then $
	      gaussian_noise_std = 0
	   if  n_elements( photon_counts ) eq 0  then $
	      photon_counts = 0
	   noise = noise_var( gaussian_noise_std, photon_counts )
	   weights = fs_weights( noise )  &  noise = sqrt( noise )
	endif

	; Iterative estimation of parameters
	newton_gauss, 'fs_model', 'fs_iacobi', 'fs_error', 'fs_convergence', $
		p0, image, VERBOSE = out_pro, _EXTRA = extra, NOISE_STD = noise, $
;		WEIGHTS = weights, /SCALE,										 $
		WEIGHTS = weights, SCALE = n_elements( weights ) ne 0,			 $
		p, sigma_p, fit_error, model, IT = it, W_BAD = w_bad

	if  fit_error ge 0  then begin
	   fs_deparam, p, x, y, f, b
	   if  n_elements( sigma_p ) ne 0  then $
	   fs_deparam, sigma_p, sigma_x, sigma_y, sigma_f, sigma_b
	   if  ( mysize( x0 ) )[0] eq 0  then begin
	      x = x[0]  &  y = y[0]  &  f = f[0]
	   endif
;	   f = f * scale  &  sigma_f = sigma_f * scale
	endif
	interp = ''		; to avoid overriding the default value in future calls
	shift_data = 0	; IMAGE_SHIFT will recognize that the input PSF may have
					; changed
	return
   end
