   ; GAUSSIAN_KERNEL: compute a blurring kernel which, when convolved
   ; with a reference (on-axis) PSF, gives the space-variant PSF
   ; corresponding to a given radial distance from a reference position.
   ; The blurring kernel is represented by an elliptical gaussian having
   ; its major axis pointing at the reference position.

   ; INPUT
   ;	pos_angle: azimuthal angle around reference position
   ;	l, w: dispersion parameters of the gaussian kernel (radial
   ;		elongation and transverse width)
   ;	[FIXED_SIZE = ]: scalar or 2-components vector specifying
   ;		kernel size
   ;	[MAX_AT = ]: 2-components vector specifying the position of
   ;		the kernel peak (only if FIXED_SIZE is used)
   ;	[/NO_NORM]: skip kernel normalization
   ;	[_EXTRA = ]: optional input parameters of GAUSSIAN_KER_SIZE

   ; OUTPUT
   ;	Return a rectangular array containing the blurring kernel,
   ;	normalized to a total volume of 1
   ;	ker_size: 2-elements integer vector [sx, sy], containing the
   ;		x- and y- sizes of the kernel

   ; NOTE: the elliptical gaussian is defined as in ELLIPT_GAUSSIAN


   FUNCTION gaussian_kernel, pos_angle, l, w, $
   							 FIXED_SIZE = s, MAX_AT = m, $
   							 NO_NORM = no_norm, _EXTRA = extra, ker_size

	if  n_elements( s ) ne 0  then  ker_size = s  else $
	   ker_size = gaussian_ker_size( l, w, pos_angle, _EXTRA = extra )
	if  n_elements( ker_size ) eq 1  then $
	   ker_size = replicate( ker_size, 2 )
	ker_size = ker_size[0:( n_elements(ker_size)< 2 )-1]
	if  n_elements( s ) * n_elements( m ) eq 0  then  m = ker_size / 2
	if  n_elements( m ) eq 1  then $
	   m = replicate( m, 2 )
	m = m[0:( n_elements(m)< 2 )-1]
	if  l * w eq 0  then begin
	   kernel = fltarr( ker_size[0], ker_size[1] )  &  kernel[m[0],m[1]] = 1
	endif else $
	   kernel = ellipt_gaussian( ker_size[0], ker_size[1], $
	   							 m[0], m[1], pos_angle, l, w )
	if  not keyword_set( no_norm )  then  kernel = normalize( kernel )
	return, kernel
   end
